/* 
 * tkCanv3dUtil.c --
 *
 *	This procedure contains a collection of utility procedures
 *	used by the implementations of various canvas3d item types.
 *
 */

#include "tkCanvas3dInt.h"

/*
 *----------------------------------------------------------------------
 *
 * Tk_Canvas3dInterp --
 *
 *	Given a token for a canvas3d, this procedure returns the
 *	Tcl_Interp that represents the canvas3d.
 *
 * Results:
 *	The return value is a Tcl_Interp for the widget.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tcl_Interp *
Tk_Canvas3dInterp(canvas3d)
    Tk_Canvas3d canvas3d;			/* Token for the canvas3d. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) canvas3d;
    return canvas3dPtr->interp;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_Canvas3dTkwin --
 *
 *	Given a token for a canvas3d, this procedure returns the
 *	widget that represents the canvas3d.
 *
 * Results:
 *	The return value is a handle for the widget.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tk_Window
Tk_Canvas3dTkwin(canvas3d)
    Tk_Canvas3d canvas3d;			/* Token for the canvas3d. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) canvas3d;
    return canvas3dPtr->tkwin;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_Canvas3dFace --
 *
 *	Given a token for a canvas3d, this procedure returns the
 *	geInterface that represents the canvas3d.
 *
 * Results:
 *	The return value is a geInterface pointer for the canvas3d.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

geInterface*
Tk_Canvas3dFace(canvas3d)
    Tk_Canvas3d canvas3d;			/* Token for the canvas3d. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) canvas3d;
    return canvas3dPtr->facePtr;
}



/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dTagsParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	"-tags" options for canvas3d items.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	The tags for a given item get replaced by those indicated
 *	in the value argument.
 *
 *--------------------------------------------------------------
 */

int
Tk_Canvas3dTagsParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    register Tk_Item3d *itemPtr = (Tk_Item3d *) widgRec;
    int argc, i;
    char **argv;
    Tk_Uid *newPtr;

    /*
     * Break the value up into the individual tag names.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Make sure that there's enough space in the item to hold the
     * tag names.
     */

    if (itemPtr->tagSpace < argc) {
	newPtr = (Tk_Uid *) ckalloc((unsigned) (argc * sizeof(Tk_Uid)));
	for (i = itemPtr->numTags-1; i >= 0; i--) {
	    newPtr[i] = itemPtr->tagPtr[i];
	}
	if (itemPtr->tagPtr != itemPtr->staticTagSpace) {
	    ckfree((char *) itemPtr->tagPtr);
	}
	itemPtr->tagPtr = newPtr;
	itemPtr->tagSpace = argc;
    }
    itemPtr->numTags = argc;
    for (i = 0; i < argc; i++) {
	itemPtr->tagPtr[i] = Tk_GetUid(argv[i]);
    }
    ckfree((char *) argv);
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dTagsPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-tags" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing all the tags for
 *	the item referred to by "widgRec".  In addition, *freeProcPtr
 *	is filled in with the address of a procedure to call to free
 *	the result string when it's no longer needed (or NULL to
 *	indicate that the string doesn't need to be freed).
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

char *
Tk_Canvas3dTagsPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    register Tk_Item3d *itemPtr = (Tk_Item3d *) widgRec;

    if (itemPtr->numTags == 0) {
	*freeProcPtr = (Tcl_FreeProc *) NULL;
	return "";
    }
    if (itemPtr->numTags == 1) {
	*freeProcPtr = (Tcl_FreeProc *) NULL;
	return (char *) itemPtr->tagPtr[0];
    }
    *freeProcPtr = TCL_DYNAMIC;
    return Tcl_Merge(itemPtr->numTags, (char **) itemPtr->tagPtr);
}

