/* 
 * tkCanv3dSurf.c --
 *
 *	This file implements surf items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"


/*
 * Prototypes for procedures defined in this file:
 */

static int		ConfigureSurf _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateSurf _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteSurf _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));

static int		Tk_Canvas3dColorParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dColorPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateSurf.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};

static Tk_CustomOption colorOption = {Tk_Canvas3dColorParseProc,
    Tk_Canvas3dColorPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_DOUBLE, "-ambient", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(SurfItem, ambientIntensity), 0},
    {TK_CONFIG_DOUBLE, "-shine", (char *) NULL, (char *) NULL,
	"20.0", Tk_Offset(SurfItem, shininess), 0},
    {TK_CONFIG_DOUBLE, "-trans", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(SurfItem, transparency), 0},
    {TK_CONFIG_CUSTOM, "-diffuse", (char *) NULL, (char *) NULL,
	"1.0 1.0 1.0 1.0", Tk_Offset(SurfItem, diffuseColor), TK_CONFIG_NULL_OK, &colorOption},
    {TK_CONFIG_CUSTOM, "-emissive", (char *) NULL, (char *) NULL,
	"1.0 1.0 1.0 1.0", Tk_Offset(SurfItem, emissiveColor), TK_CONFIG_NULL_OK, &colorOption},
    {TK_CONFIG_CUSTOM, "-specular", (char *) NULL, (char *) NULL,
	"1.0 1.0 1.0 1.0", Tk_Offset(SurfItem, specularColor), TK_CONFIG_NULL_OK, &colorOption},
    {TK_CONFIG_INT, "-ramp", (char *) NULL, (char *) NULL,
	"16", Tk_Offset(SurfItem, ramp), 0},
    {TK_CONFIG_STRING, "-texture", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(SurfItem, textureString), TK_CONFIG_NULL_OK},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}

};

/*
 * The structures below defines the surf item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkSurfType = {
    "surface",				/* name */
    sizeof(SurfItem),			/* itemSize */
    CreateSurf,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureSurf,			/* configureProc */
    DeleteSurf,				/* deleteProc */
    (Tk_Item3dDisplayProc *)NULL,	/* displayProc */
    (Tk_Item3dExBufProc *)NULL,		/* exbufProc */
    (Tk_Item3dRotateProc *)NULL,	/* rotateProc */
    (Tk_Item3dScaleProc *)NULL,		/* scaleProc */
    (Tk_Item3dTransformProc *)NULL,	/* transformProc */
    (Tk_Item3dTranslateProc *)NULL,	/* translateProc */
    (Tk_Item3dFillProc *)NULL,		/* fillProc */
    (Tk_Item3dWrapProc *)NULL,		/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateSurf --
 *
 *	This procedure is invoked to create a new surf item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new surf item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateSurf(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing surf. */
{
    SurfItem *surfPtr = (SurfItem *) itemPtr;
   
    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    surfPtr->canvas3d = canvas3d;
    surfPtr->itemCode = ITEM_SURFACE;
    itemPtr->itemCode = ITEM_SURFACE;
    surfPtr->surfData = (SurfData *) ckalloc(sizeof(SurfData));
    surfPtr->surfData->matObject = NULL;
    surfPtr->ambientIntensity = 1.0;
    surfPtr->shininess = 20.0;
    surfPtr->transparency = 1.0;
    surfPtr->diffuseColor.r = 1.0;
    surfPtr->diffuseColor.g = 1.0;
    surfPtr->diffuseColor.b = 1.0;
    surfPtr->diffuseColor.a = 1.0;
    surfPtr->emissiveColor.r = 1.0;
    surfPtr->emissiveColor.g = 1.0;
    surfPtr->emissiveColor.b = 1.0;
    surfPtr->emissiveColor.a = 1.0;
    surfPtr->specularColor.r = 1.0;
    surfPtr->specularColor.g = 1.0;
    surfPtr->specularColor.b = 1.0;
    surfPtr->specularColor.a = 1.0;
    surfPtr->ramp = 16;
    surfPtr->textureString = NULL;

    if (ConfigureSurf(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteSurf(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
 
 
/*
 *--------------------------------------------------------------
 *
 * ConfigureSurf --
 *
 *	This procedure is invoked to configure various aspects
 *	of a surf item such as its diffuseColor.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureSurf(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Surf item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    SurfItem *surfPtr = (SurfItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    int result;

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) surfPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * Call the function in Graphics Engine to create surface. If this 
     * function is called the first time while creating, a new material
     * object is created. And if called for itemconfigure, then the 
     * earlier material object is deleted and new one created with new
     * configuration values!
     */
    result = GE_CreateSurf(*facePtr, surfPtr);
    if(result == -1) {
	Tcl_AppendResult(interp, "Texture File not found!", (char *) NULL);	
	return TCL_ERROR;
    }
    if (result != 1) {
	Tcl_AppendResult(interp, "Error in creating surface", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteSurf --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a surf item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteSurf(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    SurfItem *surfPtr = (SurfItem *) itemPtr;

    /* 
     * Release the information held in the SurfData.
     */
    GE_DestroySurf(surfPtr);
}
 
/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dColorParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	canv3dColor options (-diffuse, -emissive and -specular) for 
 *	surface item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dColorParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    SurfItem *surfPtr = (SurfItem *) widgRec;
    double r, g, b, a;
    int argc;
    char **argv = NULL;

    if ((offset != Tk_Offset(SurfItem, diffuseColor))
	    && (offset != Tk_Offset(SurfItem, emissiveColor))
	    && (offset != Tk_Offset(SurfItem, specularColor))) {
	panic("Canvas3dColorParseProc received bogus offset");
    }
    
    /*
     * Break the value up into the individual color values.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad color value \"", value,
		"\": must be list with four numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 4) {
	goto syntaxError;
    }
    if ((Tcl_GetDouble(interp, argv[0], &r) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &g) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &b) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[3], &a) != TCL_OK)) {
	goto syntaxError;
    }

    /* 
     * If parsing is being done for -diffuse option.
     */
    if (offset == Tk_Offset(SurfItem, diffuseColor)) {
	surfPtr->diffuseColor.r = r;
	surfPtr->diffuseColor.g = g;
	surfPtr->diffuseColor.b = b;
	surfPtr->diffuseColor.a = a;
    }

    /*
     * If parsing is being done for -emissive option.
     */
    if (offset == Tk_Offset(SurfItem, emissiveColor)) {
	surfPtr->emissiveColor.r = r;
	surfPtr->emissiveColor.g = g;
	surfPtr->emissiveColor.b = b;
	surfPtr->emissiveColor.a = a;
    }

    /*
     * If parsing is being done for -specular option.
     */
    if (offset == Tk_Offset(SurfItem, specularColor)) {
	surfPtr->specularColor.r = r;
	surfPtr->specularColor.g = g;
	surfPtr->specularColor.b = b;
	surfPtr->specularColor.a = a;
    }

    ckfree((char *) argv);
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dColorPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the -diffuse, -specular
 *	or -emissive configuration options for surface items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	-diffuse, -specular or -emissive option value for 
 *	the surface item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
Tk_Canvas3dColorPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    SurfItem *surfPtr = (SurfItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);

    if (offset == Tk_Offset(SurfItem, diffuseColor)) {
	sprintf(buffer, "%.1f %.1f %.1f %.1f", 
		surfPtr->diffuseColor.r, surfPtr->diffuseColor.g, 
		surfPtr->diffuseColor.b, surfPtr->diffuseColor.a);	
    } else if (offset == Tk_Offset(SurfItem, emissiveColor)) {
	sprintf(buffer, "%.1f %.1f %.1f %.1f", 
		surfPtr->emissiveColor.r, surfPtr->emissiveColor.g, 
		surfPtr->emissiveColor.b, surfPtr->emissiveColor.a);	
    } else if (offset == Tk_Offset(SurfItem, specularColor)) {
	sprintf(buffer, "%.1f %.1f %.1f %.1f", 
		surfPtr->specularColor.r, surfPtr->specularColor.g, 
		surfPtr->specularColor.b, surfPtr->specularColor.a);	
    }
    
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}


