/* 
 * tkCanv3dSphr.c --
 *
 *	This file implements sphere items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"

/*
 * Prototypes for procedures defined in this file. 
 */

static int		ConfigureSphr _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateSphr _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteSphr _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		DisplaySphr _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		ExBufSphr _ANSI_ARGS_((Tk_Canvas3d canvas3d, 
			    Tk_Item3d *itemPtr));
static void		RotateSphr(Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int axis, double angle);
static void		ScaleSphr _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xScale,
			    double yScale, double zScale));
static void		TransformSphr _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, int matrixId));
static void		TranslateSphr _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xTrans, 
			    double yTrans, double zTrans));
static void		FillSphr _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int fillType));
static void		WrapSphr _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int wrapStyle));

static int		Tk_Canvas3dCenterParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dCenterPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dSurfParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dSurfPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateSphr.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption centerOption = {Tk_Canvas3dCenterParseProc,
    Tk_Canvas3dCenterPrintProc, (ClientData) NULL
};
static Tk_CustomOption surfOption = {Tk_Canvas3dSurfParseProc,
    Tk_Canvas3dSurfPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-center", (char *) NULL, (char *) NULL,
	"0.0 0.0 0.0", Tk_Offset(SphrItem, center), TK_CONFIG_NULL_OK, &centerOption},
    {TK_CONFIG_DOUBLE, "-radius", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(SphrItem, radius), 0},
    {TK_CONFIG_INT, "-rings", (char *) NULL, (char *) NULL,
	"9", Tk_Offset(SphrItem, numRings), 0},
    {TK_CONFIG_INT, "-sections", (char *) NULL, (char *) NULL,
	"15", Tk_Offset(SphrItem, numSections), 0},
    {TK_CONFIG_CUSTOM, "-surface", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(SphrItem, surface), TK_CONFIG_NULL_OK, &surfOption},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the sphr item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkSphrType = {
    "sphere",				/* name */
    sizeof(SphrItem),			/* itemSize */
    CreateSphr,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureSphr,			/* configureProc */
    DeleteSphr,				/* deleteProc */
    DisplaySphr,			/* displayProc */
    ExBufSphr,				/* exbufProc */
    RotateSphr,				/* rotateProc */
    ScaleSphr,				/* scaleProc */
    TransformSphr,			/* transformProc */
    TranslateSphr,			/* translateProc */
    FillSphr,				/* fillProc */
    WrapSphr,				/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateSphr --
 *
 *	This procedure is invoked to create a new sphr item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new sphr item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateSphr(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;			/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing sphr. */
{
    SphrItem *sphrPtr = (SphrItem *) itemPtr;

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    sphrPtr->canvas3d = canvas3d;
    sphrPtr->itemCode = ITEM_SPHERE;
    itemPtr->itemCode = ITEM_SPHERE;
    sphrPtr->shapeData = (ShapeData *)ckalloc(sizeof(ShapeData));
    sphrPtr->center.x = 0.0;
    sphrPtr->center.y = 0.0;
    sphrPtr->center.z = 0.0;
    sphrPtr->radius = 1.0;		
    sphrPtr->numRings = 9;		
    sphrPtr->numSections = 15;			
    sphrPtr->surface = (SurfItem *)ckalloc(sizeof(SurfItem));
    sphrPtr->surface = NULL;

    /*
     * Initialize the ShapeData associated with the SphrItem. ShapeData
     * is a graphics engine structure that holds all the internal details
     * about shapes.
     */
    GE_InitShape(sphrPtr->shapeData);
    
    if (ConfigureSphr(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteSphr(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
 

/*
 *--------------------------------------------------------------
 *
 * ConfigureSphr --
 *
 *	This procedure is invoked to configure various aspects
 *	of a sphr item such as its radius.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Execute Buffers for the shape are created too.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureSphr(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Sphr item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    SphrItem *sphrPtr = (SphrItem *) itemPtr;
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    
    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) sphrPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /* 
     * Create the sphere according to configure options.
     */
    if (GE_CreateSphere(sphrPtr) != GE_OK){
	Tcl_AppendResult(interp, "cannot create sphere", NULL);
	return TCL_ERROR;
    }

    /*
     * Create execute buffers for the shape. 
     */
    ExBufSphr(canvas3d, (Tk_Item3d *)sphrPtr);

    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteSphr --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a sphr item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteSphr(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    SphrItem *sphrPtr = (SphrItem *) itemPtr;
   
    /* 
     * Release the information held in the ShapeData.
     */
    GE_DestroyShape(sphrPtr->shapeData);
}
 
 
/*
 *--------------------------------------------------------------
 *
 * DisplaySphr --
 *
 *	This procedure is invoked to draw a sphr item in a given
 *	canvas3d viewport.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in viewport using the transformation
 *	information in canvas3d.
 *
 *--------------------------------------------------------------
 */

static void
DisplaySphr(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d that contains item. */
    Tk_Item3d *itemPtr;			/* Item to be displayed. */
{
    SphrItem *sphrPtr = (SphrItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    
    /* 
     * Render the shape execute buffers.
     */
    GE_RenderShape(*facePtr, *sphrPtr->shapeData);
}

/*
 *--------------------------------------------------------------
 *
 * ExBufSphr --
 *
 *	This procedure is invoked to create execute buffers for
 *	the shape. Whenever the shape configuration, or any other
 *	of its property is changed, execute buffers are recreated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */ 

static void
ExBufSphr(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing sphr. */
    Tk_Item3d *itemPtr;			/* Sphr to be scaled. */
 {
    SphrItem *sphrPtr = (SphrItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (sphrPtr->surface != NULL) {
	GE_CreateShapeExBuf(facePtr, sphrPtr->shapeData, 
				sphrPtr->surface->surfData);
    } else {
	GE_CreateShapeExBuf(facePtr, sphrPtr->shapeData, NULL);
    }
  }
 

/*
 *--------------------------------------------------------------
 *
 * RotateSphr --
 *
 *	This procedure is invoked to rotate a sphere item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
RotateSphr(canvas3d, itemPtr, axis, angle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing sphr. */
    Tk_Item3d *itemPtr;		/* Sphere to be rotated. */
    int axis;
    double angle;
 {
    SphrItem *sphrPtr = (SphrItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (axis == AXIS_X) {
	GE_RotateXShape(sphrPtr->shapeData, angle);
    } else if (axis == AXIS_Y) {
	GE_RotateYShape(sphrPtr->shapeData, angle);
    } else {
	GE_RotateZShape(sphrPtr->shapeData, angle);
    }

    ExBufSphr(canvas3d, (Tk_Item3d *)sphrPtr);
  }

/*
 *--------------------------------------------------------------
 *
 * ScaleSphr --
 *
 *	This procedure is invoked to rescale a sphere item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
ScaleSphr(canvas3d, itemPtr, xScale, yScale, zScale)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing sphr. */
    Tk_Item3d *itemPtr;		/* Sphere to be rotated. */
    double xScale;		    
    double yScale;			    
    double zScale;			    		
 {
    SphrItem *sphrPtr = (SphrItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_ScaleShape(sphrPtr->shapeData, xScale, yScale, zScale);
    
    ExBufSphr(canvas3d, (Tk_Item3d *)sphrPtr);
  }

/*
 *--------------------------------------------------------------
 *
 * TransformSphr --
 *
 *	This procedure is called to transform a sphere by a 
 *	specified matrix.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TransformSphr(canvas3d, itemPtr, matrixId)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    int matrixId;
{
    SphrItem *sphrPtr = (SphrItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    //GE_TransformShape(sphrPtr->shapeData, matrixPtr);

    ExBufSphr(canvas3d, (Tk_Item3d *)sphrPtr);
}
  
/*
 *--------------------------------------------------------------
 *
 * TranslateSphr --
 *
 *	This procedure is called to move a sphere by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TranslateSphr(canvas3d, itemPtr, xTrans, yTrans, zTrans)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    double xTrans;
    double yTrans;		
    double zTrans;
{
    SphrItem *sphrPtr = (SphrItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_TranslateShape(sphrPtr->shapeData, xTrans, yTrans, zTrans);

    ExBufSphr(canvas3d, (Tk_Item3d *)sphrPtr);
}
 
/*
 *--------------------------------------------------------------
 *
 * FillSphr --
 *
 *	This procedure is called to change the fill type of sphere.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
FillSphr(canvas3d, itemPtr, fillType)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;			/* Item that is being filled. */
    int fillType;
{
    SphrItem *sphrPtr = (SphrItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_FillShape(sphrPtr->shapeData, fillType);

    ExBufSphr(canvas3d, (Tk_Item3d *)sphrPtr);
}

/*
 *--------------------------------------------------------------
 *
 * WrapSphr --
 *
 *	This procedure is called to change the wrap style of sphere.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
WrapSphr(canvas3d, itemPtr, wrapStyle)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;			/* Item that is being wrapped. */
    int wrapStyle;
{
    SphrItem *sphrPtr = (SphrItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_WrapShape(sphrPtr->shapeData, wrapStyle);

    ExBufSphr(canvas3d, (Tk_Item3d *)sphrPtr);
}


/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dSurfParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-surface) for the Sphere item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dSurfParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    SphrItem *sphrPtr = (SphrItem *) widgRec;
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) sphrPtr->canvas3d;

    if (offset != Tk_Offset(SphrItem, surface)) {
	panic("Canvas3dSurfParseProc received bogus offset");
    }

    /* 
     * Get the Surface Item Id from command option input value. 
     */
    if (Tcl_GetInt(interp, value, &id) == TCL_ERROR ) {
	Tcl_AppendResult(interp, 
	    " : bad value for -surface option",(char *)NULL);
		return TCL_ERROR;
    }

    /*
     * Search for the surface in the list of items.
     */
    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
		    if (itemPtr->id == id) {
			if (itemPtr->itemCode != ITEM_SURFACE) {
			    Tcl_AppendResult(interp, 
				"bad value for -surface option",(char *)NULL);
			    return TCL_ERROR;
			}
			sphrPtr->surface = (SurfItem *)itemPtr;
			break;
		    }
    }
    
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dSurfPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-surface" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-surface" option value for the sphere item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
Tk_Canvas3dSurfPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    SphrItem *sphrPtr = (SphrItem *) widgRec;
    char *buffer;
    
    if(sphrPtr->surface == NULL) {
	buffer = (char *)ckalloc(5 * sizeof(char));
	sprintf(buffer, "None");
    } else {
	buffer = (char *) ckalloc(150 * sizeof(char));
	    sprintf(buffer, "%.2f %.2f %.2f {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} %i %s",
	    sphrPtr->surface->ambientIntensity, sphrPtr->surface->shininess,
	    sphrPtr->surface->transparency,
	    sphrPtr->surface->diffuseColor.r, sphrPtr->surface->diffuseColor.g,
	    sphrPtr->surface->diffuseColor.b, sphrPtr->surface->diffuseColor.a,
	    sphrPtr->surface->emissiveColor.r,sphrPtr->surface->emissiveColor.g,
	    sphrPtr->surface->emissiveColor.b, sphrPtr->surface->emissiveColor.a,
	    sphrPtr->surface->specularColor.r, sphrPtr->surface->specularColor.g,
	    sphrPtr->surface->specularColor.b, sphrPtr->surface->specularColor.a,
	    sphrPtr->surface->ramp, sphrPtr->surface->textureString);
    }
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
} 
/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dCenterParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	"-center" options for sphere items.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dCenterParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    SphrItem *sphrPtr = (SphrItem *) widgRec;
    double x, y, z;
    int argc;
    char **argv = NULL;

    if (offset != Tk_Offset(SphrItem, center)) {
	panic("Canvas3dCenterParseProc received bogus offset");
    }
    
    /*
     * Break the value up into the individual coordinate values.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad center value \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }
    /*
     * Convert the string values into double.
     */
    if ((Tcl_GetDouble(interp, argv[0], &x) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &y) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &z) != TCL_OK)) {
	goto syntaxError;
    }

    /*
     * Set the values in the SphrItem structure.
     */
    if (offset == Tk_Offset(SphrItem, center)) {
	sphrPtr->center.x = x;
	sphrPtr->center.y = y;
	sphrPtr->center.z = z;
    }

    ckfree((char *) argv);
    return TCL_OK;   
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dCenterPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-center" configuration
 *	option for sphere items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-center" option value for the sphere item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dCenterPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    SphrItem *sphrPtr = (SphrItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.2f %.2f %.2f", sphrPtr->center.x,
	    sphrPtr->center.y, sphrPtr->center.z);
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}


