
/* 
 * tkCanv3dPolygon.c --
 *
 *	This file implements polygon items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"

/*
 * Prototypes for procedures defined in this file.
 */

static int		ConfigurePolygon _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreatePolygon _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeletePolygon _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		DisplayPolygon _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		ExBufPolygon _ANSI_ARGS_((Tk_Canvas3d canvas3d, 
			    Tk_Item3d *itemPtr));
static void		RotatePolygon(Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int axis, double angle);
static void		ScalePolygon _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xScale,
			    double yScale, double zScale));
static void		TransformPolygon _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, int matrixId));
static void		TranslatePolygon _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xTrans, 
			    double yTrans, double zTrans));
static void		FillPolygon _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int fillType));
static void		WrapPolygon _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int wrapStyle));

static int		Tk_Canvas3dCenterParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dCenterPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dSurfParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dSurfPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreatePolygon.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption centerOption = {Tk_Canvas3dCenterParseProc,
    Tk_Canvas3dCenterPrintProc, (ClientData) NULL
};
static Tk_CustomOption surfOption = {Tk_Canvas3dSurfParseProc,
    Tk_Canvas3dSurfPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-center", (char *) NULL, (char *) NULL,
	"0.0 0.0 0.0", Tk_Offset(PolygonItem, center), TK_CONFIG_NULL_OK, &centerOption},
    {TK_CONFIG_DOUBLE, "-radius", (char *) NULL, (char *) NULL,
	"10.0", Tk_Offset(PolygonItem, radius), 0},
    {TK_CONFIG_INT, "-edges", (char *) NULL, (char *) NULL,
	"10", Tk_Offset(PolygonItem, edges), 0},
    {TK_CONFIG_CUSTOM, "-surface", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(PolygonItem, surface), TK_CONFIG_NULL_OK, &surfOption},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the polygon item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkPolygonType = {
    "polygon",				/* name */
    sizeof(PolygonItem),		/* itemSize */
    CreatePolygon,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigurePolygon,			/* configureProc */
    DeletePolygon,			/* deleteProc */
    DisplayPolygon,			/* displayProc */
    ExBufPolygon,			/* exbufProc */
    RotatePolygon,			/* rotateProc */
    ScalePolygon,			/* scaleProc */
    TransformPolygon,			/* transformProc */
    TranslatePolygon,			/* translateProc */
    FillPolygon,			/* fillProc */
    WrapPolygon,			/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreatePolygon --
 *
 *	This procedure is invoked to create a new polygon item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new polygon item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreatePolygon(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing polygon. */
{
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    polygonPtr->canvas3d = canvas3d;
    polygonPtr->itemCode = ITEM_POLYGON;
    itemPtr->itemCode = ITEM_POLYGON;
    polygonPtr->shapeData = (ShapeData *)ckalloc(sizeof(ShapeData));
    polygonPtr->center.x = 0.0;
    polygonPtr->center.y = 0.0;
    polygonPtr->center.z = 0.0;
    polygonPtr->radius = 10.0;		
    polygonPtr->edges = 10;		
    polygonPtr->surface = (SurfItem *)ckalloc(sizeof(SurfItem));
    polygonPtr->surface = NULL;

    /*
     * Initialize the ShapeData associated with the PolygonItem. ShapeData
     * is a graphics engine structure that holds all the internal details
     * about shapes.
     */
    GE_InitShape(polygonPtr->shapeData);
    
    if (ConfigurePolygon(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeletePolygon(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
 

/*
 *--------------------------------------------------------------
 *
 * ConfigurePolygon --
 *
 *	This procedure is invoked to configure various aspects
 *	of a polygon item such as its radius.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Execute Buffers for the shape are created too
 *
 *--------------------------------------------------------------
 */

static int
ConfigurePolygon(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Polygon item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) polygonPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /* 
     * Create the polygon according to configure options.
     */
    if (GE_CreatePolygon(polygonPtr) != GE_OK) {
	Tcl_AppendResult(interp, "cannot create polygon", NULL);
	return TCL_ERROR;
    }

    /*
     * Create execute buffers for the shape. 
     */
    ExBufPolygon(canvas3d, (Tk_Item3d *)polygonPtr);
   
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeletePolygon --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a polygon item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeletePolygon(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;
   
    /* 
     * Release the information held in the ShapeData.
     */
    GE_DestroyShape(polygonPtr->shapeData);
}
  
/*
 *--------------------------------------------------------------
 *
 * DisplayPolygon --
 *
 *	This procedure is invoked to draw a polygon item in a given
 *	canvas3d viewport.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in viewport using the transformation
 *	information in canvas3d.
 *
 *--------------------------------------------------------------
 */

static void
DisplayPolygon(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d that contains item. */
    Tk_Item3d *itemPtr;			/* Item to be displayed. */
{
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    
    /* 
     * Render the shape execute buffers.
     */
    GE_RenderShape(*facePtr, *polygonPtr->shapeData);
}

/*
 *--------------------------------------------------------------
 *
 * ExBufPolygon --
 *
 *	This procedure is invoked to create execute buffers for
 *	the shape. Whenever the shape configuration, or any other
 *	of its property is changed, execute buffers are recreated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
static void
ExBufPolygon(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing polygon. */
    Tk_Item3d *itemPtr;			/* Polygon to be scaled. */
 {
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (polygonPtr->surface != NULL) {
	GE_CreateShapeExBuf(facePtr, polygonPtr->shapeData, 
				polygonPtr->surface->surfData);
    } else {
	GE_CreateShapeExBuf(facePtr, polygonPtr->shapeData, 
				NULL);
    }
  }
  
/*
 *--------------------------------------------------------------
 *
 * RotatePolygon --
 *
 *	This procedure is invoked to rotate a polygon item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
RotatePolygon(canvas3d, itemPtr, axis, angle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing polygon. */
    Tk_Item3d *itemPtr;		/* Polygon to be rotated. */
    int axis;
    double angle;
 {
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (axis == AXIS_X) {
	GE_RotateXShape(polygonPtr->shapeData, angle);
    } else if (axis == AXIS_Y) {
	GE_RotateYShape(polygonPtr->shapeData, angle);
    } else {
	GE_RotateZShape(polygonPtr->shapeData, angle);
    }

    ExBufPolygon(canvas3d, (Tk_Item3d *)polygonPtr);
  }

/*
 *--------------------------------------------------------------
 *
 * ScalePolygon --
 *
 *	This procedure is invoked to rescale a polygon item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
ScalePolygon(canvas3d, itemPtr, xScale, yScale, zScale)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing polygon. */
    Tk_Item3d *itemPtr;		/* Polygon to be scaled. */
    double xScale;		    
    double yScale;			    
    double zScale;			    		
 {
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_ScaleShape(polygonPtr->shapeData, xScale, yScale, zScale);
    
    ExBufPolygon(canvas3d, (Tk_Item3d *)polygonPtr);
  }

/*
 *--------------------------------------------------------------
 *
 * TransformPolygon --
 *
 *	This procedure is called to transform a polygon by a
 *	specified matrix.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TransformPolygon(canvas3d, itemPtr, matrixId)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being transformed. */
    int matrixId;
{
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    //GE_TransformShape(polygonPtr->shapeData, matrixPtr);

    ExBufPolygon(canvas3d, (Tk_Item3d *)polygonPtr);
}
  
/*
 *--------------------------------------------------------------
 *
 * TranslatePolygon --
 *
 *	This procedure is called to move a polygon by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TranslatePolygon(canvas3d, itemPtr, xTrans, yTrans, zTrans)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    double xTrans;
    double yTrans;		
    double zTrans;
{
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_TranslateShape(polygonPtr->shapeData, xTrans, yTrans, zTrans);

    ExBufPolygon(canvas3d, (Tk_Item3d *)polygonPtr);
}

/*
 *--------------------------------------------------------------
 *
 * FillPolygon --
 *
 *	This procedure is called to change the fill type of the polygon.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
FillPolygon(canvas3d, itemPtr, fillType)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;			/* Item that is being filled. */
    int fillType;
{
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_FillShape(polygonPtr->shapeData, fillType);
    
    ExBufPolygon(canvas3d, (Tk_Item3d *)polygonPtr);
} 
/*
 *--------------------------------------------------------------
 *
 * FillPolygon --
 *
 *	This procedure is called to change the wrap style of the polygon.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
WrapPolygon(canvas3d, itemPtr, wrapStyle)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;			/* Item that is being wrapped. */
    int wrapStyle;
{
    PolygonItem *polygonPtr = (PolygonItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_WrapShape(polygonPtr->shapeData, wrapStyle);
    
    ExBufPolygon(canvas3d, (Tk_Item3d *)polygonPtr);
} 

/*
 *--------------------------------------------------------------
 *	    NOT TOUCHED AT ALL - STILL TO WRITE
 * Tk_Canvas3dColorPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-tags" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing all the tags for
 *	the item referred to by "widgRec".  In addition, *freeProcPtr
 *	is filled in with the address of a procedure to call to free
 *	the result string when it's no longer needed (or NULL to
 *	indicate that the string doesn't need to be freed).
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
Tk_Canvas3dColorPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    register Tk_Item3d *itemPtr = (Tk_Item3d *) widgRec;

    if (itemPtr->numTags == 0) {
	*freeProcPtr = (Tcl_FreeProc *) NULL;
	return "";
    }
    if (itemPtr->numTags == 1) {
	*freeProcPtr = (Tcl_FreeProc *) NULL;
	return (char *) itemPtr->tagPtr[0];
    }
    *freeProcPtr = TCL_DYNAMIC;
    return Tcl_Merge(itemPtr->numTags, (char **) itemPtr->tagPtr);
} 

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dSurfParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-surface) for the polygon item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dSurfParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    PolygonItem *polygonPtr = (PolygonItem *) widgRec;
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) polygonPtr->canvas3d;

    if (offset != Tk_Offset(PolygonItem, surface)) {
	panic("Canvas3dSurfParseProc received bogus offset");
    }
    
    /* 
     * Get the Surface Item Id from command option input value. 
     */
    if (Tcl_GetInt(interp, value, &id) == TCL_ERROR ) {
	Tcl_AppendResult(interp, 
	    " : bad value for -surface option",(char *)NULL);
		return TCL_ERROR;
    }  
    
    /*
     * Search for the surface in the list of items.
     */
    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
		    if (itemPtr->id == id) {
			if (itemPtr->itemCode != ITEM_SURFACE) {
			    Tcl_AppendResult(interp, 
				"bad value for -surface option",(char *)NULL);
			    return TCL_ERROR;
			}
			polygonPtr->surface = (SurfItem *)itemPtr;
			break;
		    }
    }
    
    return TCL_OK; 
}


/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dSurfPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-surface" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-surface" option value for the polygon item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dSurfPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    PolygonItem *polygonPtr = (PolygonItem *) widgRec;
    char *buffer;

    if(polygonPtr->surface == NULL) {
	buffer = (char *)ckalloc(5 * sizeof(char));
	sprintf(buffer, "None");
    } else {
	buffer = (char *) ckalloc(150);
	sprintf(buffer, "%.2f %.2f %.2f {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} %i %s",
	    polygonPtr->surface->ambientIntensity, polygonPtr->surface->shininess,
	    polygonPtr->surface->transparency,
	    polygonPtr->surface->diffuseColor.r, polygonPtr->surface->diffuseColor.g,
	    polygonPtr->surface->diffuseColor.b, polygonPtr->surface->diffuseColor.a,
	    polygonPtr->surface->emissiveColor.r,polygonPtr->surface->emissiveColor.g,
	    polygonPtr->surface->emissiveColor.b, polygonPtr->surface->emissiveColor.a,
	    polygonPtr->surface->specularColor.r, polygonPtr->surface->specularColor.g,
	    polygonPtr->surface->specularColor.b, polygonPtr->surface->specularColor.a,
	    polygonPtr->surface->ramp, polygonPtr->surface->textureString);
    }

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
} 

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dCenterParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	"-center" options for polygon items.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dCenterParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    PolygonItem *polygonPtr = (PolygonItem *) widgRec;
    double x, y, z;
    int argc;
    char **argv = NULL;

    if (offset != Tk_Offset(PolygonItem, center)) {
	panic("Canvas3dCenterParseProc received bogus offset");
    }
    
    /*
     * Break the value up into the individual coordinate values.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad center value \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }
    /* 
     * Convert all the string values into double.
     */
    if ((Tcl_GetDouble(interp, argv[0], &x) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &y) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &z) != TCL_OK)) {
	goto syntaxError;
    }

    /*
     * Set the values into the PolygonItem structure.
     */
    if (offset == Tk_Offset(PolygonItem, center)) {
	polygonPtr->center.x = x;
	polygonPtr->center.y = y;
	polygonPtr->center.z = z;
    }

    ckfree((char *) argv);
    return TCL_OK;   
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dCenterPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-center" configuration
 *	option for polygon items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-center" option value for the polygon item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dCenterPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    PolygonItem *polygonPtr = (PolygonItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.2f %.2f %.2f", polygonPtr->center.x,
	    polygonPtr->center.y, polygonPtr->center.z);
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}



