/* 
 * tkCanv3dMesh.c --
 *
 *	This file implements mesh items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"

/*
 * Prototypes for procedures defined in this file. 
 */

static int		ConfigureMesh _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateMesh _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteMesh _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		DisplayMesh _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		ExBufMesh _ANSI_ARGS_((Tk_Canvas3d canvas3d, 
			    Tk_Item3d *itemPtr));
static void		RotateMesh(Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int axis, double angle);
static void		ScaleMesh _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xScale,
			    double yScale, double zScale));
static void		TransformMesh _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, int meshId));
static void		TranslateMesh _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xTrans, 
			    double yTrans, double zTrans));
static void		FillMesh _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int fillType));
static void		WrapMesh _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int wrapStyle));

static int		Tk_Canvas3dSurfParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dSurfPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dVertexParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dVertexPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dFaceParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dFacePrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dStatsParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dStatsPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dEdgeParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dEdgePrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));

/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateMesh.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption surfOption = {Tk_Canvas3dSurfParseProc,
    Tk_Canvas3dSurfPrintProc, (ClientData) NULL
};
static Tk_CustomOption vertexOption = {Tk_Canvas3dVertexParseProc,
    Tk_Canvas3dVertexPrintProc, (ClientData) NULL
};
static Tk_CustomOption faceOption = {Tk_Canvas3dFaceParseProc,
    Tk_Canvas3dFacePrintProc, (ClientData) NULL
};
static Tk_CustomOption statsOption = {Tk_Canvas3dStatsParseProc,
    Tk_Canvas3dStatsPrintProc, (ClientData) NULL
};
static Tk_CustomOption edgeOption = {Tk_Canvas3dEdgeParseProc,
    Tk_Canvas3dEdgePrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-edge", (char *) NULL, (char *) NULL,
	(char *)NULL, 0, TK_CONFIG_NULL_OK, &edgeOption},
    {TK_CONFIG_INT, "-expF", (char *) NULL, (char *) NULL,
	"50", Tk_Offset(MeshItem, expNumFace), 0},
    {TK_CONFIG_INT, "-expV", (char *) NULL, (char *) NULL,
	"50", Tk_Offset(MeshItem, expNumVert), 0},
    {TK_CONFIG_CUSTOM, "-faces", (char *) NULL, (char *) NULL,
	NULL, 0, TK_CONFIG_NULL_OK, &faceOption},
    {TK_CONFIG_CUSTOM, "-surface", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(MeshItem, surface), TK_CONFIG_NULL_OK, &surfOption},
    {TK_CONFIG_CUSTOM, "-stats", (char *) NULL, (char *) NULL,
	NULL, 0, TK_CONFIG_NULL_OK, &statsOption},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_CUSTOM, "-vertices", (char *) NULL, (char *) NULL,
	NULL, 0, TK_CONFIG_NULL_OK, &vertexOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the mesh item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkMeshType = {
    "mesh",				/* name */
    sizeof(MeshItem),			/* itemSize */
    CreateMesh,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureMesh,			/* configureProc */
    DeleteMesh,				/* deleteProc */
    DisplayMesh,			/* displayProc */
    ExBufMesh,				/* exbufProc */
    RotateMesh,				/* rotateProc */
    ScaleMesh,				/* scaleProc */
    TransformMesh,			/* transformProc */
    TranslateMesh,			/* translateProc */
    FillMesh,				/* fillProc */
    WrapMesh,				/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateMesh --
 *
 *	This procedure is invoked to create a new mesh item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new mesh item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateMesh(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing mesh. */
{
    MeshItem *meshPtr = (MeshItem *) itemPtr;
 
    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */
    
    meshPtr->canvas3d = canvas3d;
    meshPtr->itemCode = ITEM_MESH;
    itemPtr->itemCode = ITEM_MESH;
    meshPtr->shapeData = (ShapeData *)ckalloc(sizeof(ShapeData));
    meshPtr->center.x = 0.0;
    meshPtr->center.y = 0.0;
    meshPtr->center.z = 0.0;
    meshPtr->surface = (SurfItem *)ckalloc(sizeof(SurfItem));
    meshPtr->surface = NULL;
    meshPtr->expNumVert = 50;
    meshPtr->availNumVert = 50;
    meshPtr->expNumFace = 50;
    meshPtr->availNumFace = 50;

    /*
     * Initialize the ShapeData associated with the MeshItem. ShapeData
     * is a graphics engine structure that holds all the internal details
     * about shapes.
     */
    GE_InitShape(meshPtr->shapeData);
    
    if (ConfigureMesh(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteMesh(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}

 
/*
 *--------------------------------------------------------------
 *
 * ConfigureMesh --
 *
 *	This procedure is invoked to configure various aspects
 *	of a mesh item such as its surface.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Execute Buffers for the shape are created too
 *
 *--------------------------------------------------------------
 */

static int
ConfigureMesh(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Mesh item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    MeshItem *meshPtr = (MeshItem *) itemPtr;
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) meshPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Create execute buffers for the shape. 
     */
    ExBufMesh(canvas3d, (Tk_Item3d *)meshPtr);

    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteMesh --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a mesh item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteMesh(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    MeshItem *meshPtr = (MeshItem *) itemPtr;
   
    /* 
     * Release the information held in the ShapeData.
     */
    GE_DestroyShape(meshPtr->shapeData);
}
  
/*
 *--------------------------------------------------------------
 *
 * DisplayMesh --
 *
 *	This procedure is invoked to draw a mesh item in a given
 *	canvas3d viewport.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in viewport using the transformation
 *	information in canvas3d.
 *
 *--------------------------------------------------------------
 */

static void
DisplayMesh(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d that contains item. */
    Tk_Item3d *itemPtr;			/* Item to be displayed. */
{
    MeshItem *meshPtr = (MeshItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    
    /* 
     * Render the shape execute buffers.
     */
    GE_RenderShape(*facePtr, *meshPtr->shapeData);
}

/*
 *--------------------------------------------------------------
 *
 * ExBufMesh --
 *
 *	This procedure is invoked to create execute buffers for
 *	the shape. Whenever the shape configuration, or any other
 *	of its property is changed, execute buffers are recreated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */ 

static void
ExBufMesh(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing mesh. */
    Tk_Item3d *itemPtr;			/* Mesh to be scaled. */
 {
    MeshItem *meshPtr = (MeshItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (meshPtr->surface != NULL) {
	GE_CreateShapeExBuf(facePtr, meshPtr->shapeData, 
				meshPtr->surface->surfData);
    } else {
	GE_CreateShapeExBuf(facePtr, meshPtr->shapeData, 
				NULL);
    }
  }
 
/*
 *--------------------------------------------------------------
 *
 * RotateMesh --
 *
 *	This procedure is invoked to rotate a mesh item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
RotateMesh(canvas3d, itemPtr, axis, angle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing mesh. */
    Tk_Item3d *itemPtr;		/* Mesh to be rotated. */
    int axis;
    double angle;
 {
    MeshItem *meshPtr = (MeshItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (axis == AXIS_X) {
	GE_RotateXShape(meshPtr->shapeData, angle);
    } else if (axis == AXIS_Y) {
	GE_RotateYShape(meshPtr->shapeData, angle);
    } else {
	GE_RotateZShape(meshPtr->shapeData, angle);
    }

    ExBufMesh(canvas3d, (Tk_Item3d *)meshPtr);
  }

/*
 *--------------------------------------------------------------
 *
 * ScaleMesh --
 *
 *	This procedure is invoked to rescale a mesh item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
ScaleMesh(canvas3d, itemPtr, xScale, yScale, zScale)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing mesh. */
    Tk_Item3d *itemPtr;		/* Mesh to be scaled. */
    double xScale;		    
    double yScale;			    
    double zScale;			    		
 {
    MeshItem *meshPtr = (MeshItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_ScaleShape(meshPtr->shapeData, xScale, yScale, zScale);
    
    ExBufMesh(canvas3d, (Tk_Item3d *)meshPtr);
  }

/*
 *--------------------------------------------------------------
 *
 * TransformMesh --
 *
 *	This procedure is called to transform a mesh by a specified
 *	matrix.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TransformMesh(canvas3d, itemPtr, meshId)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    int meshId;
{
    MeshItem *meshPtr = (MeshItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    //GE_TransformShape(meshPtr->shapeData, meshPtr);

    ExBufMesh(canvas3d, (Tk_Item3d *)meshPtr);
}
  
/*
 *--------------------------------------------------------------
 *
 * TranslateMesh --
 *
 *	This procedure is called to move a mesh by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TranslateMesh(canvas3d, itemPtr, xTrans, yTrans, zTrans)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    double xTrans;
    double yTrans;		
    double zTrans;
{
    MeshItem *meshPtr = (MeshItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_TranslateShape(meshPtr->shapeData, xTrans, yTrans, zTrans);

    ExBufMesh(canvas3d, (Tk_Item3d *)meshPtr);
}
 
/*
 *--------------------------------------------------------------
 *
 * FillMesh --
 *
 *	This procedure is called to change the fill type of the mesh.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
FillMesh(canvas3d, itemPtr, fillType)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being filled. */
    int fillType;
{
    MeshItem *meshPtr = (MeshItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_FillShape(meshPtr->shapeData, fillType);
    
    ExBufMesh(canvas3d, (Tk_Item3d *)meshPtr);
}

/*
 *--------------------------------------------------------------
 *
 * WrapMesh --
 *
 *	This procedure is called to change the wrap style of the mesh.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
WrapMesh(canvas3d, itemPtr, wrapStyle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being wrapped. */
    int wrapStyle;
{
    MeshItem *meshPtr = (MeshItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_WrapShape(meshPtr->shapeData, wrapStyle);
    
    ExBufMesh(canvas3d, (Tk_Item3d *)meshPtr);
}


/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dSurfParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-surface) for the Mesh item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dSurfParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    MeshItem *meshPtr = (MeshItem *) widgRec;
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) meshPtr->canvas3d;

    if (offset != Tk_Offset(MeshItem, surface)) {
	panic("Canvas3dSurfParseProc received bogus offset");
    }
    /* 
     * Get the Surface Item Id from command option input value. 
     */
    if (Tcl_GetInt(interp, value, &id) == TCL_ERROR ) {
	Tcl_AppendResult(interp, 
	    " : bad value for -surface option",(char *)NULL);
		return TCL_ERROR;
    }
    
    /*
     * Search for the surface in the list of items.
     */
    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
		    if (itemPtr->id == id) {
			if (itemPtr->itemCode != ITEM_SURFACE) {
			    Tcl_AppendResult(interp, 
				"bad value for -surface option",(char *)NULL);
			    return TCL_ERROR;
			}
			meshPtr->surface = (SurfItem *)itemPtr;
			break;
		    }
    }
    
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dSurfPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-surface" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-surface" option value for the mesh item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dSurfPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    MeshItem *meshPtr = (MeshItem *) widgRec;
    char *buffer;

    if(meshPtr->surface == NULL) {
	buffer = (char *)ckalloc(5 * sizeof(char));
	sprintf(buffer, "None");
    } else {
	buffer = (char *) ckalloc(150);
	sprintf(buffer, "%.2f %.2f %.2f {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} %i %s",
	    meshPtr->surface->ambientIntensity, meshPtr->surface->shininess,
	    meshPtr->surface->transparency,
	    meshPtr->surface->diffuseColor.r, meshPtr->surface->diffuseColor.g,
	    meshPtr->surface->diffuseColor.b, meshPtr->surface->diffuseColor.a,
	    meshPtr->surface->emissiveColor.r,meshPtr->surface->emissiveColor.g,
	    meshPtr->surface->emissiveColor.b, meshPtr->surface->emissiveColor.a,
	    meshPtr->surface->specularColor.r, meshPtr->surface->specularColor.g,
	    meshPtr->surface->specularColor.b, meshPtr->surface->specularColor.a,
	    meshPtr->surface->ramp, meshPtr->surface->textureString);
    }

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;  
} 

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dValueParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	-vertices option for mesh item. The new vertices are added
 *	to the earlier list of vertices.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dVertexParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    MeshItem *meshPtr = (MeshItem *) widgRec;
    double x, y, z;
    int argc, argcV;
    char **argv = NULL;
    char **vertex = NULL;
    int i;
    int vertexNum;
    char *vNumStr = (char *) ckalloc (4 * sizeof(char));
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) meshPtr->canvas3d;
    geInterface *facePtr = canvas3dPtr->facePtr;

    /* 
     * Break the entire list up into the individual vertices. 
     */
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad vertices list \"", value,
		"\": must be list of four-valued sub-lists", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
   
    for ( i = 0; i < argc; ++i) {
	/* 
	 * Break the vertex list into x, y and z values. 
	 */
	if (Tcl_SplitList(interp, argv[i], &argcV, &vertex) != TCL_OK) {
	    syntaxError2:
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "bad vertex value \"", argv[i],
			"\": must be a sublist of format { x y z }. ", 
			"All earlier vertices in the list, if any,",
			" have been added.", (char *)NULL);
	    if (argv != NULL) {
		ckfree((char *) argv);
	    }
	    return TCL_ERROR;
	} else {
	    if (argcV != 3) {
		goto syntaxError2;
	    }
	    /*
	     * Convert the string values into doubles.
	     */
	    if ((Tcl_GetDouble(interp, vertex[0], &x) != TCL_OK)
		|| (Tcl_GetDouble(interp, vertex[1], &y) != TCL_OK)
		|| (Tcl_GetDouble(interp, vertex[2], &z) != TCL_OK)) {
			goto syntaxError2;
	    }

	    Tcl_ResetResult(interp);
	    /* 
	     * Add the vertex to the earlier vertex list. 
	     */
	    if (GE_AddVertex(meshPtr, x, y, z, &vertexNum)) {
		sprintf(vNumStr, "%d", vertexNum);
		Tcl_AppendResult(interp, vNumStr, " ", (char *)NULL);
	    } else {
		Tcl_AppendResult(interp, "Cannot add vertex \"", argv[i],
			"\": possible insufficient memory.", (char *)NULL);
		break;
	    }
	}
    }
    ckfree((char *) vNumStr);
    ckfree((char *) argv);
    ckfree((char *) vertex);
    return TCL_OK; 
}


/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dVertexPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-vertices" configuration
 *	option for mesh items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-vertices" option value for the mesh item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dVertexPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * informeshion about how to reclaim
					 * storage for return string. */
{
    MeshItem *meshPtr = (MeshItem *)widgRec;
    char *buffer = (char *)ckalloc(50);
    int index = meshPtr->shapeData->num_vertices - 1;
    
    if (meshPtr->shapeData->num_vertices < 1) {
	sprintf(buffer, "None");
    } else {
	sprintf(buffer, "%d : {%.2f %.2f %.2f}", 
		index, meshPtr->shapeData->lpV[index].x, 
		meshPtr->shapeData->lpV[index].y, 
		meshPtr->shapeData->lpV[index].z);
    }
    *freeProcPtr = TCL_DYNAMIC; 
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dFaceParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	-faces option for mesh item. The new faces are added to 
 *	the earlier list of faces.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dFaceParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    MeshItem *meshPtr = (MeshItem *) widgRec;
    int v1, v2, v3;
    int argc, argcF;
    char **argv = NULL;
    char **face = NULL;
    int i;
    int faceNum;
    char *fNumStr = (char *) ckalloc (4 * sizeof(char));
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) meshPtr->canvas3d;
    geInterface *facePtr = canvas3dPtr->facePtr;

    /* 
     * Break the entire list up into the individual vertices. 
     */
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad faces list \"", value,
		"\": must be list of four-valued sub-lists", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
   
    for ( i = 0; i < argc; ++i) {
	/* 
	 * Break the face list into v1, v2 and v3 values. 
	 */
	if (Tcl_SplitList(interp, argv[i], &argcF, &face) != TCL_OK) {
	    syntaxError2:
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "bad face value \"", argv[i],
			"\": must be a sublist of format { v1 v2 v3 }. ", 
			"All earlier faces in the list, if any,",
			" have been added.", (char *)NULL);
	    if (argv != NULL) {
		ckfree((char *) argv);
	    }
	    return TCL_ERROR;
	} else {
	    if (argcF != 3) {
		goto syntaxError2;
	    }
	    /* 
	     * Convert the string value into integers.
	     */
	    if ((Tcl_GetInt(interp, face[0], &v1) != TCL_OK)
		|| (Tcl_GetInt(interp, face[1], &v2) != TCL_OK)
		|| (Tcl_GetInt(interp, face[2], &v3) != TCL_OK)) {
			goto syntaxError2;
	    }

	    Tcl_ResetResult(interp);
	    /* 
	     * Add the vertex to the earlier vertex list.
	     */
	    if (GE_AddFace(meshPtr, v1, v2, v3, &faceNum)) {
		sprintf(fNumStr, "%d", faceNum);
		Tcl_AppendResult(interp, fNumStr, " ", (char *)NULL);
	    } else {
		Tcl_AppendResult(interp, "Cannot add face \"", argv[i],
		    "\": possible inexistent vertex or insufficient memory.",
		    (char *)NULL);
		break;
	    }
	}
    }

    ckfree((char *) fNumStr);
    ckfree((char *) argv);
    ckfree((char *) face);
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dFacePrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-faces" configuration
 *	option for mesh items. 
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-faces" option value for the mesh item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dFacePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * informeshion about how to reclaim
					 * storage for return string. */
{
    MeshItem *meshPtr = (MeshItem *)widgRec;
    char *buffer = (char *)ckalloc(50);
    int index = meshPtr->shapeData->num_faces - 1;
    
    if (meshPtr->shapeData->num_faces < 1) {
	sprintf(buffer, "None");
    } else {
	sprintf(buffer, "%d : {%d %d %d}", 
		index, meshPtr->shapeData->lpTri[index].v1, 
		meshPtr->shapeData->lpTri[index].v2, 
		meshPtr->shapeData->lpTri[index].v3);
    }

    *freeProcPtr = TCL_DYNAMIC; 
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dStatsParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	-stats option for mesh item. This option prints out the 
 *	statistics of vertices and faces that have been created uptil
 *	now. -stats takes in "vertices", "faces" or "both" as its 
 *	value to print out data accordingly.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dStatsParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    MeshItem *meshPtr = (MeshItem *) widgRec;
    char *buffer = (char *)ckalloc(120);
    int i;

    Tcl_ResetResult(interp);
    /*
     * If the option value is "vertices" or "both".
     */
    if ((strcmp(value, "vertices") == 0) || (strcmp(value, "both") == 0)) {
	Tcl_AppendResult(interp, "Vertex v : { x y z } \n", (char *)NULL);
	for ( i = 0; i < meshPtr->shapeData->num_vertices; i++) {
	    sprintf(buffer, "%d : {%f %f %f} \n", 
		i, meshPtr->shapeData->lpV[i].x, 
		meshPtr->shapeData->lpV[i].y, meshPtr->shapeData->lpV[i].z);
	    Tcl_AppendResult(interp, buffer, (char *)NULL);
	}	
    }

    /*
     * If the option value is "faces" of "both".
     */
    if ((strcmp(value, "faces") == 0) || (strcmp(value, "both") == 0)) {
	Tcl_AppendResult(interp, "Face f : { v1 v2 v3 } \n", (char *)NULL);
	for ( i = 0; i < meshPtr->shapeData->num_faces; i++) {
	    sprintf(buffer, "%d : {%d %d %d} \n", 
		i, meshPtr->shapeData->lpTri[i].v1, 
		meshPtr->shapeData->lpTri[i].v2, meshPtr->shapeData->lpTri[i].v3);
	    Tcl_AppendResult(interp, buffer, (char *)NULL);
	}	
    }
    if ((strcmp(value, "vertices") != 0) 
	&& (strcmp(value, "faces") != 0) 
	&& (strcmp(value, "both") != 0)) {
	Tcl_AppendResult(interp, "Usage : -stats value, ",
	    "where value is vertices, faces or both.", (char *)NULL);
	return TCL_ERROR;
    }
    ckfree((char *) buffer);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dStatsPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-stats" configuration
 *	option for mesh items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-stats" option value for the mesh item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dStatsPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * informeshion about how to reclaim
					 * storage for return string. */
{
    MeshItem *meshPtr = (MeshItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(70);
    sprintf(buffer, "Usage : -stats value, where value is vertices, faces or both");
    
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dEdgeParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	-edge option for mesh item. This option takes an edge enable
 *	type as spcified in the constants, to decide which edge(s) of
 *	the face should be made visible in wireframe mode.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dEdgeParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    MeshItem *meshPtr = (MeshItem *) widgRec;
    int f, m;
    int argc, argcF;
    char **argv = NULL;
    char **face = NULL;
    int i, j;
    char *buffer = (char *) ckalloc (4 * sizeof(char));
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) meshPtr->canvas3d;
    geInterface *facePtr = canvas3dPtr->facePtr;

    /* 
     * Break the entire list up into the individual pairs. 
     */
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad edge enable list \"", value,
		"\": must be list of two-valued sub-lists", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
   
    for ( i = 0; i < argc; ++i) {
	/* 
	 * Break the face list into v1, v2 and v3 values. 
	 */
	if (Tcl_SplitList(interp, argv[i], &argcF, &face) != TCL_OK) {
	    syntaxError2:
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "bad edge enable value \"", argv[i],
			"\": must be a sublist of format { face edge_mode }.", 
			"All earlier faces in the list, if any,",
			" have been enabled with the new edge_mode.", 
			(char *)NULL);
	    if (argv != NULL) {
		ckfree((char *) argv);
	    }
	    return TCL_ERROR;
	} else {
	    if (argcF != 2) {
		goto syntaxError2;
	    }

	    /* 
	     * If change the edge enable mode for all the edges. 
	     */
	    if (strcmp(face[0], "all") == 0) {
		/* 
		 * Parse the edge_enable mode string, 
		 * and get the equivalent value. 
		 */
		if ((GE_GetConstFaceEdge(interp, face[1], &m) != TCL_OK)) {
			goto syntaxError2;
		}
		/* 
		 * Change mode for all the faces in the mesh. 
		 */
		for ( j = 0; j < meshPtr->shapeData->num_faces; j++) {
		    if (!GE_FaceEdgeEnable(meshPtr->shapeData, j, m)) {
			goto syntaxError2;
		    }
		}
		continue;
	    } 
	    
	    /* 
	     * If the sub-list has two values, and first value is not 'all'. 
	     */
	    if ((Tcl_GetInt(interp, face[0], &f) != TCL_OK)
		|| (GE_GetConstFaceEdge(interp, face[1], &m) != TCL_OK)) {
			goto syntaxError2;
	    } else {
		/* 
		 * Check for invalid face. 
		 */
		if ((f < 0) || (f >= meshPtr->shapeData->num_faces)) {
		    goto syntaxError2;
		}
		/* 
		 * Parse the edge_enable mode string, 
		 * and get the equivalent value 
		 */
		if ((GE_GetConstFaceEdge(interp, face[1], &m) != TCL_OK)) {
			goto syntaxError2;
		}
		/* 
		 * Change mode for the face in the mesh. 
		 */
		if (!GE_FaceEdgeEnable(meshPtr->shapeData, f, m)) {
		    goto syntaxError2;
		}
	    }
	}
    }

    ckfree((char *) buffer);
    ckfree((char *) argv);
    ckfree((char *) face);
    return TCL_OK; 
}



/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dEdgePrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-edge" configuration
 *	option for mesh items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-edge" option value for the mesh item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dEdgePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * informeshion about how to reclaim
					 * storage for return string. */
{
    register Tk_Item3d *itemPtr = (Tk_Item3d *) widgRec;

    if (itemPtr->numTags == 0) {
	*freeProcPtr = (Tcl_FreeProc *) NULL;
	return "";
    }
    if (itemPtr->numTags == 1) {
	*freeProcPtr = (Tcl_FreeProc *) NULL;
	return (char *) itemPtr->tagPtr[0];
    }
    *freeProcPtr = TCL_DYNAMIC;
    return Tcl_Merge(itemPtr->numTags, (char **) itemPtr->tagPtr);
}
