/* 
 * tkCanv3dMatrix.c --
 *
 *	This file implements matrix items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"


/*
 * Prototypes for procedures defined in this file:
 */

static int		ConfigureMatrix _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateMatrix _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteMatrix _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));

static int		Tk_Canvas3dValueParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dValuePrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dMultiplyParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dMultiplyPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));

/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateMatrix.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};

static Tk_CustomOption valueOption = {Tk_Canvas3dValueParseProc,
    Tk_Canvas3dValuePrintProc, (ClientData) NULL
};

static Tk_CustomOption multiplyOption = {Tk_Canvas3dMultiplyParseProc,
    Tk_Canvas3dMultiplyPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-value", (char *) NULL, (char *) NULL,
	"{ 1.0 0.0 0.0 0.0 } { 0.0 1.0 0.0 0.0 } { 0.0 0.0 1.0 0.0 } { 0.0 0.0 0.0 1.0 }",
	Tk_Offset(MatrixItem, matrix), TK_CONFIG_NULL_OK, &valueOption},
    {TK_CONFIG_CUSTOM, "-multiply", (char *) NULL, (char *) NULL,
	NULL, 0, TK_CONFIG_NULL_OK, &multiplyOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the matrix item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkMatrixType = {
    "matrix",				/* name */
    sizeof(MatrixItem),			/* itemSize */
    CreateMatrix,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureMatrix,			/* configureProc */
    DeleteMatrix,			/* deleteProc */
    (Tk_Item3dDisplayProc *)NULL,	/* displayProc */
    (Tk_Item3dExBufProc *)NULL,		/* exbufProc */
    (Tk_Item3dRotateProc *)NULL,	/* rotateProc */
    (Tk_Item3dScaleProc *)NULL,		/* scaleProc */
    (Tk_Item3dTransformProc *)NULL,	/* transformProc */
    (Tk_Item3dTranslateProc *)NULL,	/* translateProc */
    (Tk_Item3dFillProc *)NULL,		/* fillProc */
    (Tk_Item3dWrapProc *)NULL,		/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateMatrix --
 *
 *	This procedure is invoked to create a new matrix item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new matrix item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateMatrix(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing matrix. */
{
    MatrixItem *matrixPtr = (MatrixItem *) itemPtr;
   
    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    matrixPtr->canvas3d = canvas3d;
    matrixPtr->itemCode = ITEM_MATRIX;
    itemPtr->itemCode = ITEM_MATRIX;
    matrixPtr->matrixData = (MatrixData *) ckalloc(sizeof(MatrixData));
    matrixPtr->matrix.a1 = 1.0;	    matrixPtr->matrix.a2 = 0.0;
    matrixPtr->matrix.a3 = 0.0;	    matrixPtr->matrix.a4 = 0.0;		
    matrixPtr->matrix.b1 = 0.0;	    matrixPtr->matrix.b2 = 1.0;
    matrixPtr->matrix.b3 = 0.0;	    matrixPtr->matrix.b4 = 0.0;
    matrixPtr->matrix.c1 = 0.0;	    matrixPtr->matrix.c2 = 0.0;
    matrixPtr->matrix.c3 = 1.0;	    matrixPtr->matrix.c4 = 0.0;
    matrixPtr->matrix.d1 = 0.0;	    matrixPtr->matrix.d2 = 0.0;
    matrixPtr->matrix.d3 = 0.0;	    matrixPtr->matrix.d4 = 1.0;
    matrixPtr->createFlag = 1;

    if (ConfigureMatrix(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteMatrix(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
  
/*
 *--------------------------------------------------------------
 *
 * ConfigureMatrix --
 *
 *	This procedure is invoked to configure various aspects
 *	of a matrix item such as its value.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureMatrix(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Matrix item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    MatrixItem *matrixPtr = (MatrixItem *) itemPtr;
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) matrixPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteMatrix --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a matrix item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteMatrix(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    MatrixItem *matrixPtr = (MatrixItem *) itemPtr;
    
    /* 
     * Release the information held in the MatrixData.
     */
    GE_DestroyMatrix(matrixPtr);
}
 

  
/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dValueParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	-value option for matrix items. 
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dValueParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    MatrixItem *matrixPtr = (MatrixItem *) widgRec;
    int argc;
    char **argv = NULL;
    char **row = NULL;
    int i;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) matrixPtr->canvas3d;
    geInterface *facePtr = canvas3dPtr->facePtr;

    /*
     * Break the value up into the individual sublists.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError1:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad matrix value \"", value,
		"\": must be list with four lists", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }

    if (argc != 4) {
	goto syntaxError1;
    }

    for ( i = 0; i < 4; ++i) {
	/*
	 * Break each sublist into individual values for each row.
	 */
	if (Tcl_SplitList(interp, argv[i], &argc, &row) != TCL_OK) {
	    syntaxError2:
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "bad matrix value \"", argv[i],
			"\": must be a sublist with four numbers", (char *) NULL);
	    if (argv != NULL) {
	    ckfree((char *) argv);
	    }
	    return TCL_ERROR;
	} else {
	    if (argc != 4) {
		goto syntaxError2;
	    }
	    /* 
	     * Store first row values into MatrixItem structure.
	     */
	    if ( i == 0) {
		if ((Tcl_GetDouble(interp, row[0], &matrixPtr->matrix.a1) != TCL_OK)
		    || (Tcl_GetDouble(interp,row[1], &matrixPtr->matrix.a2)
			!= TCL_OK)
		    || (Tcl_GetDouble(interp, row[2], &matrixPtr->matrix.a3)
			!= TCL_OK)
		    || (Tcl_GetDouble(interp, row[3], &matrixPtr->matrix.a4)
			!= TCL_OK)) {
		goto syntaxError2;
		}
	    }
	    /* 
	     * Store second row values into MatrixItem structure.
	     */
	    if ( i == 1) {
		if ((Tcl_GetDouble(interp, row[0], &matrixPtr->matrix.b1) != TCL_OK)
		    || (Tcl_GetDouble(interp, row[1], &matrixPtr->matrix.b2)
			!= TCL_OK)
		    || (Tcl_GetDouble(interp, row[2], &matrixPtr->matrix.b3)
			!= TCL_OK)
		    || (Tcl_GetDouble(interp, row[3], &matrixPtr->matrix.b4)
			!= TCL_OK)) {
		goto syntaxError2;
		}
	    }
	    /* 
	     * Store third row values into MatrixItem structure.
	     */
	    if ( i == 2) {
		if ((Tcl_GetDouble(interp, row[0], &matrixPtr->matrix.c1) != TCL_OK)
		    || (Tcl_GetDouble(interp, row[1], &matrixPtr->matrix.c2)
			!= TCL_OK)
		    || (Tcl_GetDouble(interp, row[2], &matrixPtr->matrix.c3)
			!= TCL_OK)
		    || (Tcl_GetDouble(interp, row[3], &matrixPtr->matrix.c4)
			!= TCL_OK)) {
		goto syntaxError2;
		}
	    }
	    /* 
	     * Store fourth row values into MatrixItem structure.
	     */
	    if ( i == 3) {
		if ((Tcl_GetDouble(interp, row[0], &matrixPtr->matrix.d1) != TCL_OK)
		    || (Tcl_GetDouble(interp, row[1], &matrixPtr->matrix.d2)
			!= TCL_OK)
		    || (Tcl_GetDouble(interp, row[2], &matrixPtr->matrix.d3)
			!= TCL_OK)
		    || (Tcl_GetDouble(interp, row[3], &matrixPtr->matrix.d4)
			!= TCL_OK)) {
		goto syntaxError2;
		}
	    }
	}
    }

    if(matrixPtr->createFlag == 1) {
	/* 
	 * Create the matrix in Graphics Engine with the passed values.
	 */
	if(!GE_CreateMatrix(*facePtr, matrixPtr)) {
	    Tcl_AppendResult(interp, "Graphics Engine CreateMatrix Error", (char *) NULL);	
	    return TCL_ERROR;
	}
	matrixPtr->createFlag = 0;
    }

    ckfree((char *) argv);
    ckfree((char *) row);
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dValuePrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-value" configuration
 *	option for matrix items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-value" option value for the matrix item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dValuePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * informatrixion about how to reclaim
					 * storage for return string. */
{
    MatrixItem *matrixPtr = (MatrixItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(100 * sizeof(char));

    sprintf(buffer, "{%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f}", 
	matrixPtr->matrix.a1, matrixPtr->matrix.a2,
	matrixPtr->matrix.a3, matrixPtr->matrix.a4,
	matrixPtr->matrix.b1, matrixPtr->matrix.b2,
	matrixPtr->matrix.b3, matrixPtr->matrix.b4,
	matrixPtr->matrix.c1, matrixPtr->matrix.c2,
	matrixPtr->matrix.c3, matrixPtr->matrix.c4,
	matrixPtr->matrix.d1, matrixPtr->matrix.d2,
	matrixPtr->matrix.d3, matrixPtr->matrix.d4);
    
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;  
}


/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dMultiplyParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	"-multiply" options for matrix items.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static int
Tk_Canvas3dMultiplyParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    MatrixItem *matrixPtr = (MatrixItem *) widgRec;
    int argc;
    char **argv = NULL;
    int i, id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) matrixPtr->canvas3d;
    geInterface *facePtr = canvas3dPtr->facePtr;
    MatrixItem *tempPtr = (MatrixItem *)ckalloc(sizeof(MatrixItem));

    /* 
     * If the final matrix is a new matrix, then the final result
     * after multiply is anyway going to be zeroes...so return 
     * without a fuss!
     */
    if(matrixPtr->createFlag == 1) {
	return TCL_OK;
    }
    /*
     * Break the value up into the individual matrix item ids.
     */
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad matrix multiply value \"", value,
		"\": must be a list of matrix items", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }

    /*
     * Search the list to find the matrix items passed as option values.
     * The search is done for every matrix in the list that was passed.
     */
    for ( i = 0; i < argc; i++) {
	Tcl_GetInt(interp, argv[i], &id);

	for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
	    if (itemPtr->id == id) {
		if (itemPtr->itemCode != ITEM_MATRIX) {
		    Tcl_AppendResult(interp, 
			"bad matrix value",(char *)NULL);
			return TCL_ERROR;
		}
		/*
		 * If the matrix is found, multiply it to the matrix item
		 * for which the -multiply options is specified.
		 */
		tempPtr = (MatrixItem *)itemPtr;
		GE_MatrixMultiply(matrixPtr, tempPtr);
		break;
	    }
	}
    }

    ckfree((char *) argv);
    ckfree((char *)tempPtr);
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dMultiplyPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-multiply" configuration
 *	option for matrix items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-multiply" option value for the matrix item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dMultiplyPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * informatrixion about how to reclaim
					 * storage for return string. */
{
    MatrixItem *matrixPtr = (MatrixItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(40 * sizeof(char));

    sprintf(buffer, "Usage : -multiply { list of matrices }");
    
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}


