/* 
 * tkCanv3dLight.c --
 *
 *	This file implements light items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"


/*
 * Prototypes for procedures defined in this file:
 */

static int		ConfigureLight _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateLight _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteLight _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		RotateLight(Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int axis, double angle);
static void		ScaleLight _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xScale,
			    double yScale, double zScale));
static void		TransformLight _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, int matrixId));
static void		TranslateLight _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xTrans, 
			    double yTrans, double zTrans));

static int		Tk_Canvas3dColorParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dColorPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));

/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateLight.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption colorOption = {Tk_Canvas3dColorParseProc,
    Tk_Canvas3dColorPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_INT, "-type", (char *) NULL, (char *) NULL,
	"3", Tk_Offset(LightItem, lightType), 0},
    {TK_CONFIG_CUSTOM, "-color", (char *) NULL, (char *) NULL,
	"1.0 1.0 1.0 1.0", Tk_Offset(LightItem, color), TK_CONFIG_NULL_OK, &colorOption},
    {TK_CONFIG_DOUBLE, "-range", (char *) NULL, (char *) NULL,
	"10.0", Tk_Offset(LightItem, range), 0},
    {TK_CONFIG_DOUBLE, "-fallOff", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(LightItem, fallOff), 0},
    {TK_CONFIG_DOUBLE, "-attenuation0", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(LightItem, attenuation0), 0},
    {TK_CONFIG_DOUBLE, "-attenuation1", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(LightItem, attenuation1), 0},
    {TK_CONFIG_DOUBLE, "-attenuation2", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(LightItem, attenuation2), 0},
    {TK_CONFIG_DOUBLE, "-thetaAngle", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(LightItem, thetaAngle), 0},
    {TK_CONFIG_DOUBLE, "-phiAngle", (char *) NULL, (char *) NULL,
	"1.5", Tk_Offset(LightItem, phiAngle), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}

};

/*
 * The structures below defines the light item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkLightType = {
    "light",				/* name */
    sizeof(LightItem),			/* itemSize */
    CreateLight,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureLight,			/* configureProc */
    DeleteLight,			/* deleteProc */
    (Tk_Item3dDisplayProc *)NULL,	/* displayProc */
    (Tk_Item3dExBufProc *)NULL,		/* exbufProc */
    RotateLight,			/* rotateProc */
    (Tk_Item3dScaleProc *)NULL,		/* scaleProc */
    (Tk_Item3dTransformProc *)NULL,	/* transformProc */
    TranslateLight,			/* translateProc */
    (Tk_Item3dFillProc *)NULL,		/* fillProc */
    (Tk_Item3dWrapProc *)NULL,		/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateLight --
 *
 *	This procedure is invoked to create a new light item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new light item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateLight(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing light. */
{
    LightItem *lightPtr = (LightItem *) itemPtr;
   
    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    lightPtr->canvas3d = canvas3d;
    lightPtr->itemCode = ITEM_LIGHT;
    itemPtr->itemCode = ITEM_LIGHT;
    lightPtr->lightData = (LightData *) ckalloc(sizeof(LightData));
    lightPtr->lightType = LIGHT_DIRECTIONAL;		
    lightPtr->color.r = 1.0;	
    lightPtr->color.g = 1.0;	
    lightPtr->color.b = 1.0;	
    lightPtr->color.a = 1.0;	
    lightPtr->position.x = 0.0;		
    lightPtr->position.y = 0.0;
    lightPtr->position.z = 0.0;
    lightPtr->direction.x = 0.0;
    lightPtr->direction.y = 0.0;
    lightPtr->direction.z = 1.0;
    lightPtr->range = 10.0;
    lightPtr->fallOff = 0.0;		
    lightPtr->attenuation0 = 0.0;	
    lightPtr->attenuation1 = 0.0;	
    lightPtr->attenuation2 = 1.0;	
    lightPtr->thetaAngle = 1.0;		
    lightPtr->phiAngle = 1.5;		

    if (ConfigureLight(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteLight(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
 

 
/*
 *--------------------------------------------------------------
 *
 * ConfigureLight --
 *
 *	This procedure is invoked to configure various aspects
 *	of a light item such as its color.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureLight(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Light item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    LightItem *lightPtr = (LightItem *) itemPtr;
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) lightPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /* 
     * Create the light according to configure options.
     */
    GE_CreateLight(*facePtr, lightPtr);
    
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteLight --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a light item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteLight(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    LightItem *lightPtr = (LightItem *) itemPtr;

    /* 
     * Release the information held in the LightData.
     */
    GE_DestroyLight(lightPtr);
}

/*
 *--------------------------------------------------------------
 *
 * RotateLight --
 *
 *	This procedure is invoked to rotate a light item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
RotateLight(canvas3d, itemPtr, axis, angle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing light. */
    Tk_Item3d *itemPtr;		/* Light to be rotated. */
    int axis;
    double angle;
 {
    LightItem *lightPtr = (LightItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    /*if (axis == AXIS_X) {
	GE_RotateXLight(lightPtr->lightData, angle);
    } else if (axis == AXIS_Y) {
	GE_RotateYLight(lightPtr->lightData, angle);
    } else {
	GE_RotateZLight(lightPtr->lightData, angle);
    }
    */
  }
  
/*
 *--------------------------------------------------------------
 *
 * TranslateLight --
 *
 *	This procedure is called to move a light by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TranslateLight(canvas3d, itemPtr, xTrans, yTrans, zTrans)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    double xTrans;
    double yTrans;		
    double zTrans;
{
    LightItem *lightPtr = (LightItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    /*GE_TranslateLight(lightPtr->lightData, xTrans, yTrans, zTrans);
	Just add the three values to light position.
    */
}
 
 
/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dColorParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	canv3dColor options (-color) for light item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dColorParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    LightItem *lightPtr = (LightItem *) widgRec;
    double r, g, b, a;
    int argc;
    char **argv = NULL;

    if ((offset != Tk_Offset(LightItem, color))) {
	panic("Canvas3dColorParseProc received bogus offset");
    }
    
    /*
     * Break the value up into the individual color values.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad color value \"", value,
		"\": must be list with four numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 4) {
	goto syntaxError;
    }
    /* 
     * Convert the string input values into doubles.
     */
    if ((Tcl_GetDouble(interp, argv[0], &r) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &g) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &b) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[3], &a) != TCL_OK)) {
	goto syntaxError;
    }

    /* 
     * Set the values into LightItem structure.
     */
    if (offset == Tk_Offset(LightItem, color)) {
	lightPtr->color.r = r;
	lightPtr->color.g = g;
	lightPtr->color.b = b;
	lightPtr->color.a = a;
    }

    ckfree((char *) argv);
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dColorPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-color" configuration
 *	option for light items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-color" option value for the light item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
Tk_Canvas3dColorPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * inforlightion about how to reclaim
					 * storage for return string. */
{
    LightItem *lightPtr = (LightItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.1f %.1f %.1f %.1f", lightPtr->color.r,
	    lightPtr->color.g, lightPtr->color.b, lightPtr->color.a);
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}
