/* 
 * tkCanv3dCylin.c --
 *
 *	This file implements cylinder items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"

/*
 * Prototypes for procedures defined in this file. 
 */

static int		ConfigureCylin _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateCylin _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteCylin _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		DisplayCylin _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		ExBufCylin _ANSI_ARGS_((Tk_Canvas3d canvas3d, 
			    Tk_Item3d *itemPtr));
static int		CylinCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		RotateCylin(Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int axis, double angle);
static void		ScaleCylin _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xScale,
			    double yScale, double zScale));
static void		TransformCylin _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, int matrixId));
static void		TranslateCylin _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xTrans, 
			    double yTrans, double zTrans));
static void		FillCylin _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int fillType));
static void		WrapCylin _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int wrapStyle));

static int		Tk_Canvas3dCenterParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dCenterPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dSurfParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dSurfPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateCylin.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption centerOption = {Tk_Canvas3dCenterParseProc,
    Tk_Canvas3dCenterPrintProc, (ClientData) NULL
};
static Tk_CustomOption surfOption = {Tk_Canvas3dSurfParseProc,
    Tk_Canvas3dSurfPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-center", (char *) NULL, (char *) NULL,
	"0.0 0.0 0.0", Tk_Offset(CylinItem, center), TK_CONFIG_NULL_OK, &centerOption},
    {TK_CONFIG_DOUBLE, "-radius", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(CylinItem, radius), 0},
    {TK_CONFIG_DOUBLE, "-height", (char *) NULL, (char *) NULL,
	"10.0", Tk_Offset(CylinItem, height), 0},
    {TK_CONFIG_INT, "-sections", (char *) NULL, (char *) NULL,
	"10", Tk_Offset(CylinItem, numSections), 0},
    {TK_CONFIG_CUSTOM, "-surface", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(CylinItem, surface), TK_CONFIG_NULL_OK, &surfOption},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the cylin item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkCylinType = {
    "cylinder",				/* name */
    sizeof(CylinItem),			/* itemSize */
    CreateCylin,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureCylin,			/* configureProc */
    DeleteCylin,			/* deleteProc */
    DisplayCylin,			/* displayProc */
    ExBufCylin,				/* exbufProc */
    RotateCylin,			/* rotateProc */
    ScaleCylin,				/* scaleProc */
    TransformCylin,			/* transformProc */
    TranslateCylin,			/* translateProc */
    FillCylin,				/* fillProc */
    WrapCylin,				/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateCylin --
 *
 *	This procedure is invoked to create a new cylin item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new cylin item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateCylin(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing cylin. */
{
    CylinItem *cylinPtr = (CylinItem *) itemPtr;

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    cylinPtr->canvas3d = canvas3d;
    cylinPtr->itemCode = ITEM_CYLINDER;
    itemPtr->itemCode = ITEM_CYLINDER;
    cylinPtr->shapeData = (ShapeData *)ckalloc(sizeof(ShapeData));
    cylinPtr->center.x = 0.0;
    cylinPtr->center.y = 0.0;
    cylinPtr->center.z = 0.0;
    cylinPtr->radius = 1.0;
    cylinPtr->height = 10.0;
    cylinPtr->numSections = 15;		
    cylinPtr->surface = (SurfItem *)ckalloc(sizeof(SurfItem));
    cylinPtr->surface = NULL;
   
    /*
     * Initialize the ShapeData associated with the CylinItem. ShapeData
     * is a graphics engine structure that holds all the internal details
     * about shapes.
     */
    GE_InitShape(cylinPtr->shapeData);
   
    if (ConfigureCylin(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteCylin(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
 
/*
 *--------------------------------------------------------------
 *
 * ConfigureCylin --
 *
 *	This procedure is invoked to configure various aspects
 *	of a cylin item such as its height.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Execute Buffers for the shape are created too.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureCylin(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Cylin item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    CylinItem *cylinPtr = (CylinItem *) itemPtr;
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) cylinPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /* 
     * Create the cylinder according to configure options.
     */
    if (GE_CreateCylinder(cylinPtr) != GE_OK) {
	Tcl_AppendResult(interp, "cannot create cylinder", NULL);
	return TCL_ERROR;
    }

    /*
     * Create execute buffers for the shape. 
     */
    ExBufCylin(canvas3d, (Tk_Item3d *)cylinPtr);
   
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteCylin --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a cylinder item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteCylin(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    CylinItem *cylinPtr = (CylinItem *) itemPtr;
   
    /* 
     * Release the information held in the ShapeData.
     */
    GE_DestroyShape(cylinPtr->shapeData);
}
 
 
/*
 *--------------------------------------------------------------
 *
 * DisplayCylin --
 *
 *	This procedure is invoked to draw a cylin item in a given
 *	canvas3d viewport.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in viewport using the transformation
 *	information in canvas3d.
 *
 *--------------------------------------------------------------
 */

static void
DisplayCylin(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d that contains item. */
    Tk_Item3d *itemPtr;			/* Item to be displayed. */
{
    CylinItem *cylinPtr = (CylinItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    
    /* 
     * Render the shape execute buffers.
     */
    GE_RenderShape(*facePtr, *cylinPtr->shapeData);
}

/*
 *--------------------------------------------------------------
 *
 * ExBufCylin --
 *
 *	This procedure is invoked to create execute buffers for
 *	the shape. Whenever the shape configuration, or any other
 *	of its property is changed, execute buffers are recreated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
static void
ExBufCylin(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing cylin. */
    Tk_Item3d *itemPtr;			/* Cylin to be scaled. */
 {
    CylinItem *cylinPtr = (CylinItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (cylinPtr->surface != NULL) {
	GE_CreateShapeExBuf(facePtr, cylinPtr->shapeData, 
				cylinPtr->surface->surfData);
    } else {
	GE_CreateShapeExBuf(facePtr, cylinPtr->shapeData, NULL);
    }
  }
 
/*
 *--------------------------------------------------------------
 *
 * RotateCylin --
 *
 *	This procedure is invoked to rotate a cylinder item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
RotateCylin(canvas3d, itemPtr, axis, angle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing cylin. */
    Tk_Item3d *itemPtr;		/* Cylin to be scaled. */
    int axis;
    double angle;
 {
    CylinItem *cylinPtr = (CylinItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (axis == AXIS_X) {
	GE_RotateXShape(cylinPtr->shapeData, angle);
    } else if (axis == AXIS_Y) {
	GE_RotateYShape(cylinPtr->shapeData, angle);
    } else {
	GE_RotateZShape(cylinPtr->shapeData, angle);
    }

    ExBufCylin(canvas3d, (Tk_Item3d *)cylinPtr);
  }

/*
 *--------------------------------------------------------------
 *
 * ScaleCylin --
 *
 *	This procedure is invoked to rescale a cylin item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
ScaleCylin(canvas3d, itemPtr, xScale, yScale, zScale)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing cylin. */
    Tk_Item3d *itemPtr;		/* Cylin to be scaled. */
    double xScale;		    
    double yScale;			    
    double zScale;			    		
 {
    CylinItem *cylinPtr = (CylinItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_ScaleShape(cylinPtr->shapeData, xScale, yScale, zScale);
    
    ExBufCylin(canvas3d, (Tk_Item3d *)cylinPtr);
  }

/*
 *--------------------------------------------------------------
 *
 * TransformCylin --
 *
 *	This procedure is called to transform a cylinder by a 
 *	specified matrix.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TransformCylin(canvas3d, itemPtr, matrixId)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being transformed. */
    int matrixId;
{
    CylinItem *cylinPtr = (CylinItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    //GE_TransformShape(cylinPtr->shapeData, matrixPtr);

    ExBufCylin(canvas3d, (Tk_Item3d *)cylinPtr);
}
  
/*
 *--------------------------------------------------------------
 *
 * TranslateCylin --
 *
 *	This procedure is called to move a cylinder by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TranslateCylin(canvas3d, itemPtr, xTrans, yTrans, zTrans)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    double xTrans;
    double yTrans;		
    double zTrans;
{
    CylinItem *cylinPtr = (CylinItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_TranslateShape(cylinPtr->shapeData, xTrans, yTrans, zTrans);

    ExBufCylin(canvas3d, (Tk_Item3d *)cylinPtr);
}
 
/*
 *--------------------------------------------------------------
 *
 * FillCylin --
 *
 *	This procedure is called to change the fill type of the
 *	cylinder.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
FillCylin(canvas3d, itemPtr, fillType)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being filled. */
    int fillType;
{
    CylinItem *cylinPtr = (CylinItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_FillShape(cylinPtr->shapeData, fillType);
    
    ExBufCylin(canvas3d, (Tk_Item3d *)cylinPtr);
}

/*
 *--------------------------------------------------------------
 *
 * WrapCylin --
 *
 *	This procedure is called to change the wrap style of the 
 *	cylinder.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
WrapCylin(canvas3d, itemPtr, wrapStyle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being wrapped. */
    int wrapStyle;
{
    CylinItem *cylinPtr = (CylinItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_WrapShape(cylinPtr->shapeData, wrapStyle);
    
    ExBufCylin(canvas3d, (Tk_Item3d *)cylinPtr);
}


/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dSurfParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem options (-surface) for the Cylinder item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dSurfParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    CylinItem *cylinPtr = (CylinItem *) widgRec;
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) cylinPtr->canvas3d;

    if (offset != Tk_Offset(CylinItem, surface)) {
	panic("Canvas3dSurfParseProc received bogus offset");
    }

    /* 
     * Get the Surface Item Id from command option input value. 
     */
    if (Tcl_GetInt(interp, value, &id) == TCL_ERROR ) {
	Tcl_AppendResult(interp, 
	    " : bad value for -surface option",(char *)NULL);
		return TCL_ERROR;
    }

    /*
     * Search for the surface in the list of items.
     */
    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
		    if (itemPtr->id == id) {
			if (itemPtr->itemCode != ITEM_SURFACE) {
			    Tcl_AppendResult(interp, 
				"bad value for -surface option",(char *)NULL);
			    return TCL_ERROR;
			}
			cylinPtr->surface = (SurfItem *)itemPtr;
			break;
		    }
    }
    
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dSurfPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-surface" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-surface" option value for the cylinder item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dSurfPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    CylinItem *cylinPtr = (CylinItem *) widgRec;
    char *buffer;

    if(cylinPtr->surface == NULL) {
	buffer = (char *)ckalloc(5 * sizeof(char));
	sprintf(buffer, "None");
    } else {
	buffer = (char *) ckalloc(150);
	sprintf(buffer, "%.2f %.2f %.2f {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} %i %s",
	    cylinPtr->surface->ambientIntensity, cylinPtr->surface->shininess,
	    cylinPtr->surface->transparency,
	    cylinPtr->surface->diffuseColor.r, cylinPtr->surface->diffuseColor.g,
	    cylinPtr->surface->diffuseColor.b, cylinPtr->surface->diffuseColor.a,
	    cylinPtr->surface->emissiveColor.r,cylinPtr->surface->emissiveColor.g,
	    cylinPtr->surface->emissiveColor.b, cylinPtr->surface->emissiveColor.a,
	    cylinPtr->surface->specularColor.r, cylinPtr->surface->specularColor.g,
	    cylinPtr->surface->specularColor.b, cylinPtr->surface->specularColor.a,
	    cylinPtr->surface->ramp, cylinPtr->surface->textureString);
    }

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
} 

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dCenterParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	"-center" options for cylinder items.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static int
Tk_Canvas3dCenterParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    CylinItem *cylinPtr = (CylinItem *) widgRec;
    double x, y, z;
    int argc;
    char **argv = NULL;

    if (offset != Tk_Offset(CylinItem, center)) {
	panic("Canvas3dCenterParseProc received bogus offset");
    }
    
    /*
     * Break the value up into the individual coordinate values.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad center value \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }

    /*
     * Convert the string options values into doubles.
     */
    if ((Tcl_GetDouble(interp, argv[0], &x) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &y) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &z) != TCL_OK)) {
	goto syntaxError;
    }

    /*
     * Set the values into the CylinItem structure.
     */
    if (offset == Tk_Offset(CylinItem, center)) {
	cylinPtr->center.x = x;
	cylinPtr->center.y = y;
	cylinPtr->center.z = z;
    }

    ckfree((char *) argv);
    return TCL_OK;   
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dCenterPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-center" configuration
 *	option for cylinder items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-center" option value for the cylinder item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
Tk_Canvas3dCenterPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    CylinItem *cylinPtr = (CylinItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.2f %.2f %.2f", cylinPtr->center.x,
	    cylinPtr->center.y, cylinPtr->center.z);
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}


