/* 
 * tkCanv3dCone.c --
 *
 *	This file implements cone items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"

/*
 * Prototypes for procedures defined in this file. 
 */

static int		ConfigureCone _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateCone _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteCone _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		DisplayCone _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		ExBufCone _ANSI_ARGS_((Tk_Canvas3d canvas3d, 
			    Tk_Item3d *itemPtr));
static void		RotateCone(Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int axis, double angle);
static void		ScaleCone _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xScale,
			    double yScale, double zScale));
static void		TransformCone _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, int matrixId));
static void		TranslateCone _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xTrans, 
			    double yTrans, double zTrans));
static void		FillCone _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int fillType));
static void		WrapCone _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int wrapStyle));

static int		Tk_Canvas3dCenterParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dCenterPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dSurfParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dSurfPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateCone.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption centerOption = {Tk_Canvas3dCenterParseProc,
    Tk_Canvas3dCenterPrintProc, (ClientData) NULL
};
static Tk_CustomOption surfOption = {Tk_Canvas3dSurfParseProc,
    Tk_Canvas3dSurfPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-center", (char *) NULL, (char *) NULL,
	"0.0 0.0 0.0", Tk_Offset(ConeItem, center), TK_CONFIG_NULL_OK, &centerOption},
    {TK_CONFIG_DOUBLE, "-radius", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(ConeItem, radius), 0},
    {TK_CONFIG_DOUBLE, "-height", (char *) NULL, (char *) NULL,
	"10.0", Tk_Offset(ConeItem, height), 0},
    {TK_CONFIG_INT, "-sections", (char *) NULL, (char *) NULL,
	"10", Tk_Offset(ConeItem, numSections), 0},
    {TK_CONFIG_CUSTOM, "-surface", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(ConeItem, surface), TK_CONFIG_NULL_OK, &surfOption},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the cone item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkConeType = {
    "cone",				/* name */
    sizeof(ConeItem),			/* itemSize */
    CreateCone,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureCone,			/* configureProc */
    DeleteCone,				/* deleteProc */
    DisplayCone,			/* displayProc */
    ExBufCone,				/* exbufProc */
    RotateCone,				/* rotateProc */
    ScaleCone,				/* scaleProc */
    TransformCone,			/* transformProc */
    TranslateCone,			/* translateProc */
    FillCone,				/* fillProc */
    WrapCone,				/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateCone --
 *
 *	This procedure is invoked to create a new cone item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new cone item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateCone(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing cone. */
{
    ConeItem *conePtr = (ConeItem *) itemPtr;
   
    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    conePtr->canvas3d = canvas3d;
    conePtr->itemCode = ITEM_CONE;
    itemPtr->itemCode = ITEM_CONE;
    conePtr->shapeData = (ShapeData *)ckalloc(sizeof(ShapeData));
    conePtr->center.x = 0.0;
    conePtr->center.y = 0.0;
    conePtr->center.z = 0.0;
    conePtr->radius = 1.0;		
    conePtr->height = 10.0;
    conePtr->numSections = 10;		
    conePtr->surface = (SurfItem *)ckalloc(sizeof(SurfItem));
    conePtr->surface = NULL;

    /*
     * Initialize the ShapeData associated with the ConeItem. ShapeData
     * is a graphics engine structure that holds all the internal details
     * about shapes.
     */
    GE_InitShape(conePtr->shapeData);
    
    if (ConfigureCone(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteCone(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
 

/*
 *--------------------------------------------------------------
 *
 * ConfigureCone --
 *
 *	This procedure is invoked to configure various aspects
 *	of a cone item such as its height.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Execute Buffers for the shape are created too.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureCone(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Cone item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    ConeItem *conePtr = (ConeItem *) itemPtr;
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) conePtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /* 
     * Create the cone according to configure options.
     */
    if (GE_CreateCone(conePtr) != GE_OK) {
	Tcl_AppendResult(interp, "cannot create cone", NULL);
	return TCL_ERROR;
    }

    /*
     * Create execute buffers for the shape. 
     */
    ExBufCone(canvas3d, (Tk_Item3d *)conePtr);
   
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteCone --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a cone item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteCone(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    ConeItem *conePtr = (ConeItem *) itemPtr;

    /* 
     * Release the information held in the ShapeData.
     */
    GE_DestroyShape(conePtr->shapeData);
}
 
/*
 *--------------------------------------------------------------
 *
 * DisplayCone --
 *
 *	This procedure is invoked to draw a cone item in a given
 *	canvas3d viewport.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in viewport using the transformation
 *	information in canvas3d.
 *
 *--------------------------------------------------------------
 */

static void
DisplayCone(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d that contains item. */
    Tk_Item3d *itemPtr;			/* Item to be displayed. */
{
    ConeItem *conePtr = (ConeItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    
    /* 
     * Render the shape execute buffers.
     */
    GE_RenderShape(*facePtr, *conePtr->shapeData);
}

/*
 *--------------------------------------------------------------
 *
 * ExBufCone --
 *
 *	This procedure is invoked to create execute buffers for
 *	the shape. Whenever the shape configuration, or any other
 *	of its property is changed, execute buffers are recreated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
static void
ExBufCone(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing cone. */
    Tk_Item3d *itemPtr;			/* Cone to be scaled. */
 {
    ConeItem *conePtr = (ConeItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (conePtr->surface != NULL) {
	GE_CreateShapeExBuf(facePtr, conePtr->shapeData, 
	    conePtr->surface->surfData);
    } else {
	GE_CreateShapeExBuf(facePtr, conePtr->shapeData, NULL);
    }
  }
  

/*
 *--------------------------------------------------------------
 *
 * RotateCone --
 *
 *	This procedure is invoked to rotate a cone item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
RotateCone(canvas3d, itemPtr, axis, angle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing cone. */
    Tk_Item3d *itemPtr;		/* Cone to be rotated. */
    int axis;
    double angle;
 {
    ConeItem *conePtr = (ConeItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (axis == AXIS_X) {
	GE_RotateXShape(conePtr->shapeData, angle);
    } else if (axis == AXIS_Y) {
	GE_RotateYShape(conePtr->shapeData, angle);
    } else {
	GE_RotateZShape(conePtr->shapeData, angle);
    }

    ExBufCone(canvas3d, (Tk_Item3d *)conePtr);
  }

/*
 *--------------------------------------------------------------
 *
 * ScaleCone --
 *
 *	This procedure is invoked to rescale a cone item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
ScaleCone(canvas3d, itemPtr, xScale, yScale, zScale)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing cone. */
    Tk_Item3d *itemPtr;		/* Cone to be scaled. */
    double xScale;		    
    double yScale;			    
    double zScale;			    		
{
    ConeItem *conePtr = (ConeItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_ScaleShape(conePtr->shapeData, xScale, yScale, zScale);
    
    ExBufCone(canvas3d, (Tk_Item3d *)conePtr);
}

/*
 *--------------------------------------------------------------
 *
 * TransformCone --
 *
 *	This procedure is called to transform a cone by the
 *	specified matrix.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TransformCone(canvas3d, itemPtr, matrixId)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being transformed. */
    int matrixId;
{
    ConeItem *conePtr = (ConeItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    //GE_TransformShape(conePtr->shapeData, matrixPtr);

    ExBufCone(canvas3d, (Tk_Item3d *)conePtr);
}
  
/*
 *--------------------------------------------------------------
 *
 * TranslateCone --
 *
 *	This procedure is called to move a cone by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TranslateCone(canvas3d, itemPtr, xTrans, yTrans, zTrans)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    double xTrans;
    double yTrans;		
    double zTrans;
{
    ConeItem *conePtr = (ConeItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_TranslateShape(conePtr->shapeData, xTrans, yTrans, zTrans);

    ExBufCone(canvas3d, (Tk_Item3d *)conePtr);
}

/*
 *--------------------------------------------------------------
 *
 * FillCone --
 *
 *	This procedure is called to change the fill type of the cone.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
FillCone(canvas3d, itemPtr, fillType)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being filled. */
    int fillType;
{
    ConeItem *conePtr = (ConeItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_FillShape(conePtr->shapeData, fillType);    
    
    ExBufCone(canvas3d, (Tk_Item3d *)conePtr);
} 
/*
 *--------------------------------------------------------------
 *
 * FillCone --
 *
 *	This procedure is called to change the wrap style of the cone.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
WrapCone(canvas3d, itemPtr, wrapStyle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being wrapped. */
    int wrapStyle;
{
    ConeItem *conePtr = (ConeItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_WrapShape(conePtr->shapeData, wrapStyle);    
    
    ExBufCone(canvas3d, (Tk_Item3d *)conePtr);
} 

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dSurfParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-surface) for the cone item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dSurfParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    ConeItem *conePtr = (ConeItem *) widgRec;
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) conePtr->canvas3d;

    if (offset != Tk_Offset(ConeItem, surface)) {
	panic("Canvas3dSurfParseProc received bogus offset");
    }

    /*  
     * Get the Surface Item Id from command option input value. 
     */
    if (Tcl_GetInt(interp, value, &id) == TCL_ERROR ) {
	Tcl_AppendResult(interp, 
	    " : bad value for -surface option",(char *)NULL);
		return TCL_ERROR;
    }

    /*
     * Search for the surface in the list of items.
     */
    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
		    if (itemPtr->id == id) {
			if (itemPtr->itemCode != ITEM_SURFACE) {
			    Tcl_AppendResult(interp, 
				"bad value for -surface option",(char *)NULL);
			    return TCL_ERROR;
			}
			conePtr->surface = (SurfItem *)itemPtr;
			break;
		    }
    }
    
    return TCL_OK; 
}


/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dSurfPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-surface" configuration
 *	option for cone items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-surface" option value for the cone item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
Tk_Canvas3dSurfPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    ConeItem *conePtr = (ConeItem *) widgRec;
    char *buffer;

    if(conePtr->surface == NULL) {
	buffer = (char *)ckalloc(5 * sizeof(char));
	sprintf(buffer, "None");
    } else {
	buffer = (char *) ckalloc(150);
	sprintf(buffer, "%.2f %.2f %.2f {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} %i %s",
	    conePtr->surface->ambientIntensity, conePtr->surface->shininess,
	    conePtr->surface->transparency,
	    conePtr->surface->diffuseColor.r, conePtr->surface->diffuseColor.g,
	    conePtr->surface->diffuseColor.b, conePtr->surface->diffuseColor.a,
	    conePtr->surface->emissiveColor.r,conePtr->surface->emissiveColor.g,
	    conePtr->surface->emissiveColor.b, conePtr->surface->emissiveColor.a,
	    conePtr->surface->specularColor.r, conePtr->surface->specularColor.g,
	    conePtr->surface->specularColor.b, conePtr->surface->specularColor.a,
	    conePtr->surface->ramp, conePtr->surface->textureString);
    }

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
} 

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dCenterParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	"-center" options for cone items.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dCenterParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    ConeItem *conePtr = (ConeItem *) widgRec;
    double x, y, z;
    int argc;
    char **argv = NULL;

    if (offset != Tk_Offset(ConeItem, center)) {
	panic("Canvas3dCenterParseProc received bogus offset");
    }
    
    /*
     * Break the value up into the individual coordinate values.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad center value \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }
    /* 
     * Convert the string input values to double.
     */
    if ((Tcl_GetDouble(interp, argv[0], &x) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &y) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &z) != TCL_OK)) {
	goto syntaxError;
    }

    /*
     * Set the center into the ConeItem structure.
     */
    if (offset == Tk_Offset(ConeItem, center)) {
	conePtr->center.x = x;
	conePtr->center.y = y;
	conePtr->center.z = z;
    }

    ckfree((char *) argv);
    return TCL_OK;   
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dCenterPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-center" configuration
 *	option for cone items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-center" option value for the cone item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dCenterPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    ConeItem *conePtr = (ConeItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.2f %.2f %.2f", conePtr->center.x,
	    conePtr->center.y, conePtr->center.z);
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}


