/* 
 * tkCanv3dBox.c --
 *
 *	This file implements box items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"

/*
 * Prototypes for procedures defined in this file. 
 */

static int		ConfigureBox _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateBox _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteBox _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		DisplayBox _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		ExBufBox _ANSI_ARGS_((Tk_Canvas3d canvas3d, 
			    Tk_Item3d *itemPtr));
static void		RotateBox _ANSI_ARGS_((Tk_Canvas3d canvas3d, 
			    Tk_Item3d *itemPtr, int axis, double angle));
static void		ScaleBox _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xScale,
			    double yScale, double zScale));
static void		TransformBox _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, int matrixId));
static void		TranslateBox _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xTrans, 
			    double yTrans, double zTrans));
static void		FillBox _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int fillType));
static void		WrapBox _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int wrapStyle));

static int		Tk_Canvas3dCenterParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dCenterPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dSurfParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dSurfPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateBox.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption centerOption = {Tk_Canvas3dCenterParseProc,
    Tk_Canvas3dCenterPrintProc, (ClientData) NULL
};
static Tk_CustomOption surfOption = {Tk_Canvas3dSurfParseProc,
    Tk_Canvas3dSurfPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-center", (char *) NULL, (char *) NULL,
	"0.0 0.0 0.0", Tk_Offset(BoxItem, center), TK_CONFIG_NULL_OK, &centerOption},
    {TK_CONFIG_DOUBLE, "-depth", (char *) NULL, (char *) NULL,
	"2.0", Tk_Offset(BoxItem, depth), 0},
    {TK_CONFIG_DOUBLE, "-height", (char *) NULL, (char *) NULL,
	"2.0", Tk_Offset(BoxItem, height), 0},
    {TK_CONFIG_CUSTOM, "-surface", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(BoxItem, surface), TK_CONFIG_NULL_OK, &surfOption},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_DOUBLE, "-width", (char *) NULL, (char *) NULL,
	"2.0", Tk_Offset(BoxItem, width), 0},	
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the box item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkBoxType = {
    "box",				/* name */
    sizeof(BoxItem),			/* itemSize */
    CreateBox,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureBox,			/* configureProc */
    DeleteBox,				/* deleteProc */
    DisplayBox,				/* displayProc */
    ExBufBox,				/* exbufProc */
    RotateBox,				/* rotateProc */
    ScaleBox,				/* scaleProc */
    TransformBox,			/* transformProc */
    TranslateBox,			/* translateProc */
    FillBox,				/* fillProc */
    WrapBox,				/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateBox --
 *
 *	This procedure is invoked to create a new box item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new box item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateBox(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing box. */
{
    BoxItem *boxPtr = (BoxItem *) itemPtr;
   
    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */
   
    boxPtr->canvas3d = canvas3d;
    boxPtr->itemCode = ITEM_BOX;
    itemPtr->itemCode = ITEM_BOX;
    boxPtr->shapeData = (ShapeData *)ckalloc(sizeof(ShapeData));
    boxPtr->width = 2.0;
    boxPtr->height = 2.0;
    boxPtr->depth = 2.0;
    boxPtr->center.x = 0.0;
    boxPtr->center.y = 0.0;
    boxPtr->center.z = 0.0;
    boxPtr->surface = (SurfItem *)ckalloc(sizeof(SurfItem));
    boxPtr->surface = NULL;
  
    /*
     * Initialize the ShapeData associated with the BoxItem. ShapeData
     * is a graphics engine structure that holds all the internal details
     * about shapes.
     */
    GE_InitShape(boxPtr->shapeData);

    if (ConfigureBox(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteBox(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
 
 
/*
 *--------------------------------------------------------------
 *
 * ConfigureBox --
 *
 *	This procedure is invoked to configure various aspects
 *	of a box item such as its surface.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Execute Buffers for the shape are created too.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureBox(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* Box item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    BoxItem *boxPtr = (BoxItem *) itemPtr;
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    
    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) boxPtr, flags) != TCL_OK) {
	    return TCL_ERROR;
    }

    /* 
     * Create the box according to configure options.
     */
    if (GE_CreateBox(boxPtr) != GE_OK) {
	Tcl_AppendResult(interp, "cannot create box", NULL);
	return TCL_ERROR;
    }

    /*
     * Create execute buffers for the shape. 
     */
    ExBufBox(canvas3d, (Tk_Item3d *)boxPtr);

    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteBox --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a box item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteBox(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    BoxItem *boxPtr = (BoxItem *) itemPtr;

    /* 
     * Release the information held in the ShapeData.
     */
    GE_DestroyShape(boxPtr->shapeData);
}
 

 
/*
 *--------------------------------------------------------------
 *
 * DisplayBox --
 *
 *	This procedure is invoked to draw a box item in a given
 *	canvas3d viewport.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in the viewport using the transformation
 *	information in canvas3d.
 *
 *--------------------------------------------------------------
 */

static void
DisplayBox(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d that contains item. */
    Tk_Item3d *itemPtr;			/* Item to be displayed. */
{
    BoxItem *boxPtr = (BoxItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    Tcl_Interp *interp = Tk_Canvas3dInterp(canvas3d);
    int errorId;
    char *errorStr;

    /* 
     * Render the shape execute buffers.
     */
    errorId = GE_RenderShape(*facePtr, *boxPtr->shapeData);
    if (errorId != GE_OK) {
	errorStr = D3DErrorToString(errorId);
	//GE_ErrorString(errorId, errorStr);
	Tcl_AppendResult(interp, errorStr, (char *)NULL);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ExBufBox --
 *
 *	This procedure is invoked to create execute buffers for
 *	the shape. Whenever the shape configuration, or any other
 *	of its property is changed, execute buffers are recreated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */

static void
ExBufBox(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d containing box. */
    Tk_Item3d *itemPtr;			/* Box to be scaled. */
 {
    BoxItem *boxPtr = (BoxItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (boxPtr->surface != NULL) {
	GE_CreateShapeExBuf(facePtr, boxPtr->shapeData,
		boxPtr->surface->surfData);
    } else {
	GE_CreateShapeExBuf(facePtr, boxPtr->shapeData, NULL);
    }
  }
 
 
/*
 *--------------------------------------------------------------
 *
 * RotateBox --
 *
 *	This procedure is invoked to rotate a box item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
RotateBox(canvas3d, itemPtr, axis, angle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing box. */
    Tk_Item3d *itemPtr;		/* Box to be scaled. */
    int axis;
    double angle;
 {
    BoxItem *boxPtr = (BoxItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (axis == AXIS_X) {
	GE_RotateXShape(boxPtr->shapeData, angle);
    } else if (axis == AXIS_Y) {
	GE_RotateYShape(boxPtr->shapeData, angle);
    } else {
	GE_RotateZShape(boxPtr->shapeData, angle);
    }

    ExBufBox(canvas3d, (Tk_Item3d *)boxPtr);

  }

/*
 *--------------------------------------------------------------
 *
 * ScaleBox --
 *
 *	This procedure is invoked to rescale a box item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
ScaleBox(canvas3d, itemPtr, xScale, yScale, zScale)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing box. */
    Tk_Item3d *itemPtr;		/* Box to be scaled. */
    double xScale;		    
    double yScale;			    
    double zScale;			    		
 {
    BoxItem *boxPtr = (BoxItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_ScaleShape(boxPtr->shapeData, xScale, yScale, zScale);
    
    ExBufBox(canvas3d, (Tk_Item3d *)boxPtr);

  }
 
/*
 *--------------------------------------------------------------
 *
 * TransformBox --
 *
 *	This procedure is called to transform a box by a 
 *	specified matrix.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TransformBox(canvas3d, itemPtr, matrixId)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being transformed. */
    int matrixId;
{
    BoxItem *boxPtr = (BoxItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    //GE_TransformShape(sphrPtr->shapeData, matrixPtr);

    ExBufBox(canvas3d, (Tk_Item3d *)boxPtr);

}

/*
 *--------------------------------------------------------------
 *
 * TranslateBox --
 *
 *	This procedure is called to move a box by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
TranslateBox(canvas3d, itemPtr, xTrans, yTrans, zTrans)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being moved. */
    double xTrans;
    double yTrans;		
    double zTrans;
{
    BoxItem *boxPtr = (BoxItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_TranslateShape(boxPtr->shapeData, xTrans, yTrans, zTrans);

    ExBufBox(canvas3d, (Tk_Item3d *)boxPtr);

}
 
/*
 *--------------------------------------------------------------
 *
 * FillBox --
 *
 *	This procedure is called to change the fill type of the box.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
FillBox(canvas3d, itemPtr, fillType)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being filled. */
    int fillType;
{
    BoxItem *boxPtr = (BoxItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_FillShape(boxPtr->shapeData, fillType);
   
    ExBufBox(canvas3d, (Tk_Item3d *)boxPtr);

}

/*
 *--------------------------------------------------------------
 *
 * WrapBox --
 *
 *	This procedure is called to change the wrap style of the box.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Execute Buffers of the shape are recreated.
 *
 *--------------------------------------------------------------
 */

static void
WrapBox(canvas3d, itemPtr, wrapStyle)
    Tk_Canvas3d canvas3d;	/* Canvas3d containing item. */
    Tk_Item3d *itemPtr;		/* Item that is being wrapped. */
    int wrapStyle;
{
    BoxItem *boxPtr = (BoxItem *) itemPtr; 
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    GE_WrapShape(boxPtr->shapeData, wrapStyle);
   
    ExBufBox(canvas3d, (Tk_Item3d *)boxPtr);

}


/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dSurfParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem options (-surface) for the box item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dSurfParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    BoxItem *boxPtr = (BoxItem *) widgRec;
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) boxPtr->canvas3d;

    if  (offset != Tk_Offset(BoxItem, surface)) {
	panic("Canvas3dSurfParseProc received bogus offset");
    }

    /* 
     * Get the Surface Item Id from command option input value. 
     */
    if (Tcl_GetInt(interp, value, &id) == TCL_ERROR ) {
	Tcl_AppendResult(interp, 
	    " : bad value for -surface option",(char *)NULL);
		return TCL_ERROR;
    }

    /*
     * Search for the surface in the list of items.
     */
    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
		    if (itemPtr->id == id) {
			if (itemPtr->itemCode != ITEM_SURFACE) {
			    Tcl_AppendResult(interp, 
				"bad value for -surface option",(char *)NULL);
			    return TCL_ERROR;
			}
			boxPtr->surface = (SurfItem *)itemPtr;
			break;
		    }
    }
    return TCL_OK; 
}


/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dSurfPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-surface" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-surface" option value for the box item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
Tk_Canvas3dSurfPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    BoxItem *boxPtr = (BoxItem *) widgRec;
    char *buffer;

    if(boxPtr->surface == NULL) {
	buffer = (char *)ckalloc(5 * sizeof(char));
	sprintf(buffer, "None");
    } else {
	buffer = (char *) ckalloc(150 * sizeof(char));
	sprintf(buffer, "%.2f %.2f %.2f {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} %i %s",
	    boxPtr->surface->ambientIntensity, boxPtr->surface->shininess,
	    boxPtr->surface->transparency,
	    boxPtr->surface->diffuseColor.r, boxPtr->surface->diffuseColor.g,
	    boxPtr->surface->diffuseColor.b, boxPtr->surface->diffuseColor.a,
	    boxPtr->surface->emissiveColor.r,boxPtr->surface->emissiveColor.g,
	    boxPtr->surface->emissiveColor.b, boxPtr->surface->emissiveColor.a,
	    boxPtr->surface->specularColor.r, boxPtr->surface->specularColor.g,
	    boxPtr->surface->specularColor.b, boxPtr->surface->specularColor.a,
	    boxPtr->surface->ramp, boxPtr->surface->textureString);
    }

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
} 

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dCenterParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	"-center" options for box items.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dCenterParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    BoxItem *boxPtr = (BoxItem *) widgRec;
    double x, y, z;
    int argc;
    char **argv = NULL;

    if (offset != Tk_Offset(BoxItem, center)) {
	panic("Canvas3dCenterParseProc received bogus offset");
    }
    
    /*
     * Break the value up into the individual coordinate values.
     */

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad center value \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }
    /*
     * Convert the string values parsed into double.
     */
    if ((Tcl_GetDouble(interp, argv[0], &x) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &y) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &z) != TCL_OK)) {
	goto syntaxError;
    }

    /*
     * Set the values into the BoxItem structure.
     */
    if (offset == Tk_Offset(BoxItem, center)) {
	boxPtr->center.x = x;
	boxPtr->center.y = y;
	boxPtr->center.z = z;
    }

    ckfree((char *) argv);
    return TCL_OK;   
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dCenterPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-center" configuration
 *	option for box items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-center" option value for the box item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dCenterPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    BoxItem *boxPtr = (BoxItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.2f %.2f %.2f", boxPtr->center.x,
	    boxPtr->center.y, boxPtr->center.z);
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
    
}


