/* 
 * geUtils.c --
 *
 *	This file implements most of the graphics engine library
 *	functions other than those specific to initialization, shapes 
 *      and matrices. 
 *
 */

#include "tkCanvas3dInt.h"

static BOOL RestoreSurf(geInterface face);

/*
 *--------------------------------------------------------------
 *
 * GE_ErrorString --
 *
 *	Get an error string for an error constant value to
 *	print it out to the user.
 *
 * Results:
 *	Return the error string in the passed parameter.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void	    
GE_ErrorString(id, error)
    int id;		/* error constant value */
    char *error;	/* string to return error */
{
    if (id == GE_OK) {
	error = NULL;
	return;
    }
    /* 
     * Return a Direct3D error.
     */
    error =  D3DErrorToString(id);

}

/*
 *--------------------------------------------------------------
 *
 * GE_Init --
 *
 *	This procedure initializes the graphics engine depending
 *	on the system in which this software is being used. The 
 *	geInterface structure in TkCanvas3d holds the data for 
 *	the graphics engine. Every graphics engine will have its 
 *	own geInterface.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
void		    
GE_Init(TkCanvas3d *canvas3dPtr)
{
    HWND hWnd;

    /*
     * Get the MS Window Handle of the canvas3d.
     */
    hWnd = TkWinGetHWND(Tk_WindowId(canvas3dPtr->tkwin));
    
    /*
     * Store the handle and instance in geInterface structure.
     */
    canvas3dPtr->facePtr->msWinHandle = hWnd;
    canvas3dPtr->facePtr->msWinInstance = TkWinGetAppInstance();

    /*
     * If Microsoft Direct3d is your graphics engine then initialize it.
     */
    Direct3d_Init(canvas3dPtr->facePtr);

}

/*
 *--------------------------------------------------------------
 *
 * GE_SetViewport --
 *
 *	A viewport for Canvas3D is first created by default 
 *	during initialization. This function changes the current
 *	viewport settings.
 *
 * Results:
 *	Return a GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_SetViewport(facePtr, xOrigin, yOrigin, width, height)
    geInterface *facePtr; 
    int xOrigin; 
    int yOrigin; 
    int width; 
    int height; 
{
    D3DAppInfo* d3dapp = facePtr->d3dapp;
    D3DVIEWPORT viewData;
    HRESULT retVal;

    /*
     * Prepare the viewport data structure with the new values.
     */
    memset(&viewData, 0, sizeof(D3DVIEWPORT));
    viewData.dwSize = sizeof(D3DVIEWPORT);
    viewData.dwX = xOrigin;
    viewData.dwY = yOrigin;
    viewData.dwWidth = width;
    viewData.dwHeight = height;
    viewData.dvScaleX = viewData.dwWidth / (float)2.0;
    viewData.dvScaleY = viewData.dwHeight / (float)2.0;
    viewData.dvMaxX = (float)D3DDivide(D3DVAL(viewData.dwWidth),
                                       D3DVAL(2 * viewData.dvScaleX));
    viewData.dvMaxY = (float)D3DDivide(D3DVAL(viewData.dwHeight),
                                       D3DVAL(2 * viewData.dvScaleY));

    /*
     * Setup the viewport for a reasonable viewing area
     */
    if ((retVal  = d3dapp->lpD3DViewport->lpVtbl->SetViewport(d3dapp->lpD3DViewport, &viewData)) != D3D_OK) {
        return retVal;
    }

    return GE_OK;
}


/*
 *--------------------------------------------------------------
 *
 * GE_CreateSurf --
 *
 *	SurfData is a structure that holds the surface info required
 *	by Direct3d. This function creates a new SurfData(texture+material) 
 *	and stores the structure in SurfItem.  
 *
 * Results:
 *	Return GE_OK if successful, -1 if the texture is already
 *	read in earlier and GE_ERROR if any other error occurs.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_CreateSurf(face, itemPtr)
    geInterface face; 
    SurfItem *itemPtr; 
{
    D3DAppInfo* d3dapp = face.d3dapp;
    D3DMATERIAL mat;
    LPDIRECT3DMATERIAL matObject;
    LPD3DTEXTUREHANDLE TextureHandle = d3dapp->TextureHandle;
    SurfData *dataPtr;

    int arrayPos;

    dataPtr = (SurfData *) ckalloc(sizeof(SurfData));
    
    /*
     * Everytime you enter this function, a new surface data is to be created. 
     * If a surface item already has surface data, then it is destroyed and
     * recreated with the latest option values!
     */ 
    if (itemPtr->surfData->matObject != NULL) {
        GE_DestroySurf(itemPtr);
    }
    
    /*
     * See \tkDirect3d\d3dapp.h for D3DAppAddTexture info.
     */
    if (itemPtr->textureString != NULL) {
	D3DAppAddTexture(itemPtr->textureString);
	
	/* 
	 * If the texture has already been present before. 
	 */
	if ((arrayPos = D3DAppGetTextureHandle(itemPtr->textureString)) == -1) {
	    return -1;
	}
	/* 
	 * Store texture info in surface data.
	 */
	dataPtr->tex = TextureHandle[arrayPos];
    }
    /*
     * Prepare a material object according to the specifications.
     */
    memset(&mat, 0, sizeof(D3DMATERIAL));
    mat.dwSize = sizeof(D3DMATERIAL);
    mat.diffuse.r = (D3DVALUE)itemPtr->diffuseColor.r;
    mat.diffuse.g = (D3DVALUE)itemPtr->diffuseColor.g;
    mat.diffuse.b = (D3DVALUE)itemPtr->diffuseColor.b;
    mat.diffuse.a = (D3DVALUE)itemPtr->diffuseColor.a;
    mat.ambient.r = (D3DVALUE)itemPtr->diffuseColor.g *
		    (D3DVALUE)itemPtr->ambientIntensity;
    mat.ambient.g = (D3DVALUE)itemPtr->diffuseColor.g * 
		    (D3DVALUE)itemPtr->ambientIntensity;
    mat.ambient.b = (D3DVALUE)itemPtr->diffuseColor.b * 
		    (D3DVALUE)itemPtr->ambientIntensity;
    mat.ambient.a = (D3DVALUE)itemPtr->diffuseColor.a * 
		    (D3DVALUE)itemPtr->ambientIntensity;
    mat.emissive.r = (D3DVALUE)itemPtr->emissiveColor.r;
    mat.emissive.g = (D3DVALUE)itemPtr->emissiveColor.g;
    mat.emissive.b = (D3DVALUE)itemPtr->emissiveColor.b;
    mat.emissive.a = (D3DVALUE)itemPtr->emissiveColor.a;
    mat.specular.r = (D3DVALUE)itemPtr->specularColor.r;
    mat.specular.g = (D3DVALUE)itemPtr->specularColor.g;
    mat.specular.b = (D3DVALUE)itemPtr->specularColor.b;
    mat.specular.a = (D3DVALUE)itemPtr->specularColor.a;
    mat.power = (D3DVALUE)itemPtr->shininess;
    mat.dwRampSize = itemPtr->ramp;
    if (itemPtr->textureString != NULL) {
	mat.hTexture = dataPtr->tex;
    }

    /*
     * Create the material and get a handle to the material.
     */
    if (d3dapp->lpD3D->lpVtbl->CreateMaterial(d3dapp->lpD3D, &matObject, NULL) 
	    != D3D_OK) {
        return GE_ERROR;
    }
    matObject->lpVtbl->SetMaterial(matObject, &mat);
    matObject->lpVtbl->GetHandle(matObject, d3dapp->lpD3DDevice, &dataPtr->mat);
    dataPtr->matObject = matObject;
    
    /*
     * The required surface is created now. Store the surface data in 
     * surface item.
     */
    itemPtr->surfData = dataPtr;
    
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_CreateViewBackground --
 *
 *	Set a background surface for the viewport.	  
 *
 * Results:
 *	Return GE_OK if successful, else GE_ERROR.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_CreateViewBackground(face, itemPtr)
    geInterface face;
    SurfItem *itemPtr;
{
    D3DAppInfo* d3dapp = face.d3dapp;
    LPDIRECT3DMATERIAL bmat = itemPtr->surfData->matObject;
    D3DMATERIALHANDLE mat;
    HRESULT retVal;
    
    /* 
     * Create a surface.
     */
    GE_CreateSurf(face, itemPtr);

    /*
     * Set the surface as the background of the viewport.
     */
    retVal = bmat->lpVtbl->GetHandle(bmat,d3dapp->lpD3DDevice, &mat);
    retVal = d3dapp->lpD3DViewport->lpVtbl->
			SetBackground(d3dapp->lpD3DViewport, mat);
    
    itemPtr->surfData->mat = mat;

    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_DestroySurf --
 *
 *	Destroy a surface item.	
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void 
GE_DestroySurf(itemPtr)
    SurfItem *itemPtr;
{
    if (itemPtr->surfData->matObject != NULL) {
        RELEASE(itemPtr->surfData->matObject);
    }
}


/*
 *--------------------------------------------------------------
 *
 * GE_CreateLight --
 *
 *	Create a new light item according the the specifications
 *	passed.
 *
 * Results:
 *	Return GE_OK if successful, else returns a Direct3D error
 *	constant value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_CreateLight(face, itemPtr)
    geInterface face;
    LightItem *itemPtr;
{
    D3DAppInfo* d3dapp = face.d3dapp;
    D3DLIGHT light; 
    LPDIRECT3DLIGHT lightObj;
    HRESULT retVal;

    /* 
     * Prepare the light item structure with new values.
     */
    memset(&light, 0, sizeof(D3DLIGHT));
    light.dwSize = sizeof(D3DLIGHT);
    light.dltType = itemPtr->lightType;
    light.dcvColor.r = (float)itemPtr->color.r;
    light.dcvColor.g = (float)itemPtr->color.g;
    light.dcvColor.b = (float)itemPtr->color.b;
    light.dcvColor.a = (float)itemPtr->color.a;
    light.dvPosition.x = (float)itemPtr->position.x;
    light.dvPosition.y = (float)itemPtr->position.y;
    light.dvPosition.z = (float)itemPtr->position.z;
    light.dvDirection.x = D3DVALP((float)itemPtr->direction.x, 12); 
    light.dvDirection.y = D3DVALP((float)itemPtr->direction.x, 12);
    light.dvDirection.z = D3DVALP((float)itemPtr->direction.x, 12);
    light.dvRange = (float)itemPtr->range;
    light.dvFalloff = (float)itemPtr->fallOff;
    light.dvAttenuation0 = (float)itemPtr->attenuation0;
    light.dvAttenuation1 = (float)itemPtr->attenuation1;
    light.dvAttenuation2 = (float)itemPtr->attenuation2;
    light.dvTheta = (float)itemPtr->thetaAngle;
    light.dvPhi = (float)itemPtr->phiAngle;
    
    /* 
     * Create the light item and get handle to it.
     */
    if ((retVal = d3dapp->lpD3D->lpVtbl->CreateLight(d3dapp->lpD3D, &lightObj, NULL)) != D3D_OK)
        return retVal;
    if ((retVal = lightObj->lpVtbl->SetLight(lightObj, &light)) != D3D_OK)
        return retVal;
    if ((retVal = d3dapp->lpD3DViewport->lpVtbl->AddLight(d3dapp->lpD3DViewport, lightObj)) != D3D_OK)
        return retVal;

    itemPtr->lightData->lightPtr = lightObj;

    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_DestroyLight --
 *
 *	Destroy a light item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void 
GE_DestroyLight(itemPtr)
    LightItem *itemPtr;
{
    RELEASE(itemPtr->lightData->lightPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderViewWorldProj --
 *
 *	Render the projection, world and view matrices.	
 *
 * Results:
 *	Return GE_OK if successful, else a Direct3D error 
 *	constant value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_RenderProjWorldView(face, projPtr, worldPtr, viewPtr)
    geInterface face; 
    MatrixItem *projPtr;
    MatrixItem *worldPtr;
    MatrixItem *viewPtr;
{
    D3DAppInfo* d3dapp = face.d3dapp;
    LPDIRECT3DEXECUTEBUFFER lpD3DExCmdBuf;
    LPVOID lpBufStart, lpInsStart, lpPointer;
    size_t size;
    
    D3DEXECUTEDATA d3dExData;
    D3DEXECUTEBUFFERDESC debDesc;

    D3DMATRIXHANDLE hView  =  viewPtr->matrixData->matrixPtr;
    D3DMATRIXHANDLE hWorld =  worldPtr->matrixData->matrixPtr;
    D3DMATRIXHANDLE hProj  =  projPtr->matrixData->matrixPtr;
    
    HRESULT retVal;

    /* 
     * Prepare the execute buffer. Get some space for it. 
     */
    size = 0;
    size += sizeof(D3DINSTRUCTION) * 3;
    size += sizeof(D3DSTATE) * 4;
    memset(&debDesc, 0, sizeof(D3DEXECUTEBUFFERDESC));
    debDesc.dwSize = sizeof(D3DEXECUTEBUFFERDESC);
    debDesc.dwFlags = D3DDEB_BUFSIZE;
    debDesc.dwBufferSize = size;
    if ((retVal = d3dapp->lpD3DDevice->lpVtbl->
	CreateExecuteBuffer(d3dapp->lpD3DDevice, &debDesc, &lpD3DExCmdBuf,
                                           NULL)) != D3D_OK)
        return retVal;
    if ((retVal = lpD3DExCmdBuf->lpVtbl->Lock(lpD3DExCmdBuf, &debDesc)) != D3D_OK)
        return retVal;
    lpBufStart = debDesc.lpData;
    memset(lpBufStart, 0, size);
    lpPointer = lpBufStart;

    lpInsStart = lpPointer;
    /*
     * Set the instructions in the execute buffers.
     */
    OP_STATE_TRANSFORM(3, lpPointer);
        STATE_DATA(D3DTRANSFORMSTATE_VIEW, hView, lpPointer);
	STATE_DATA(D3DTRANSFORMSTATE_WORLD, hWorld, lpPointer);
	STATE_DATA(D3DTRANSFORMSTATE_PROJECTION, hProj, lpPointer);
    OP_STATE_LIGHT(1, lpPointer);
        STATE_DATA(D3DLIGHTSTATE_AMBIENT, RGBA_MAKE(64, 64, 64, 64), lpPointer);
    OP_EXIT(lpPointer);
    
    lpD3DExCmdBuf->lpVtbl->Unlock(lpD3DExCmdBuf);
    memset(&d3dExData, 0, sizeof(D3DEXECUTEDATA));
    d3dExData.dwSize = sizeof(D3DEXECUTEDATA);
    d3dExData.dwInstructionOffset = (ULONG) 0;
    d3dExData.dwInstructionLength = (ULONG)((char *)lpPointer - (char *)lpInsStart);
    lpD3DExCmdBuf->lpVtbl->SetExecuteData(lpD3DExCmdBuf, &d3dExData);
    
    /*
     * Render the execute buffers just created.
     */
    if ((retVal = d3dapp->lpD3DDevice->lpVtbl->BeginScene(d3dapp->lpD3DDevice)) != D3D_OK)
	return retVal;
    if ((retVal = d3dapp->lpD3DDevice->lpVtbl->Execute(d3dapp->lpD3DDevice, lpD3DExCmdBuf, 
			d3dapp->lpD3DViewport, D3DEXECUTE_CLIPPED)) != D3D_OK)
	return retVal;
    if ((retVal = d3dapp->lpD3DDevice->lpVtbl->EndScene(d3dapp->lpD3DDevice)) != D3D_OK)
	return retVal;
    lpD3DExCmdBuf->lpVtbl->Release(lpD3DExCmdBuf); 
    
    return GE_OK;

}

/*
 *--------------------------------------------------------------
 *
 * GE_CreateShapeExBuf --
 *
 *	Create Execute buffers for the shape with the depending
 *	on its specifications. The shape is not displayed, just
 *	the execute buffers are created.
 *
 * Results:
 *	Return GE_OK if successful, else a Direct3D error constant
 *	value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_CreateShapeExBuf(facePtr, shapePtr, surfPtr)
    geInterface *facePtr;
    ShapeData *shapePtr;
    SurfData *surfPtr;
{
    D3DAppInfo* d3dapp = facePtr->d3dapp;
    
    LPD3DTRIANGLE lpTri = shapePtr->lpTri;
    LPVOID lpBufStart, lpInsStart, lpPointer;
    size_t size; 
  
    D3DEXECUTEDATA d3dExData;
    LPDIRECT3DEXECUTEBUFFER lpD3DExBuf;
    D3DEXECUTEBUFFERDESC debDesc;
    
    HRESULT retVal;

    int stateNum = 0;
    /*
     * Calculate the different states to be set.
     */
    if (surfPtr != NULL) {
	stateNum += 4;	    /* Light associated with material, 
			     * Texture and wrap-styles */
    }
    stateNum += 1;	    /* Fill Type */

    /* 
     * Get some space to store the execute buffers.
     */
    size = sizeof(D3DVERTEX) * shapePtr->num_vertices;
    size += sizeof(D3DPROCESSVERTICES) * 1;
    size += sizeof(D3DSTATUS) * 1;
    size += sizeof(D3DINSTRUCTION) * 8;
    size += sizeof(D3DSTATE) * stateNum;
    size += sizeof(D3DTRIANGLE) * shapePtr->num_faces;
    memset(&debDesc, 0, sizeof(D3DEXECUTEBUFFERDESC));
    debDesc.dwSize = sizeof(D3DEXECUTEBUFFERDESC);
    debDesc.dwFlags = D3DDEB_BUFSIZE;
    debDesc.dwBufferSize = size;
    if ((retVal = d3dapp->lpD3DDevice->lpVtbl->
	CreateExecuteBuffer(d3dapp->lpD3DDevice, &debDesc, &lpD3DExBuf, 
                                           NULL)) != D3D_OK)
	    return retVal;
    /*
     * lock it so it can be filled
     */
    if ((retVal = lpD3DExBuf->lpVtbl->Lock(lpD3DExBuf, &debDesc)) != D3D_OK)
        return retVal;
    lpBufStart = debDesc.lpData;
    memset(lpBufStart, 0, size);
    lpPointer = lpBufStart;

    /*
     * Copy Vertices to execute buffers.
     */

    VERTEX_DATA(shapePtr->lpV, shapePtr->num_vertices, lpPointer);

    /*
     * Save the location of the first instruction and add instructions to 
     * execute buffer.
     */
    lpInsStart = lpPointer;
    /*
     * To associate light with the material.
     */
    if (surfPtr != NULL) {
	OP_STATE_LIGHT(1, lpPointer);
        STATE_DATA(D3DLIGHTSTATE_MATERIAL, surfPtr->mat, lpPointer);
    }
    
    /* 
     * (2048, 2048, 0, 0) define a bounding box for all the relevant vertices.
     */

    OP_SET_STATUS(D3DSETSTATUS_ALL, D3DSTATUS_DEFAULT, 2048, 2048, 0, 0, lpPointer);
    
    OP_PROCESS_VERTICES(1, lpPointer);
        PROCESSVERTICES_DATA(D3DPROCESSVERTICES_TRANSFORMLIGHT, 0, shapePtr->num_vertices, lpPointer);
    
    /*
     * 14 different Render states are available. 2 are used here.
     */
    if (surfPtr != NULL) {
	OP_STATE_RENDER(3, lpPointer);
        STATE_DATA(D3DRENDERSTATE_TEXTUREHANDLE, surfPtr->tex, lpPointer);
        STATE_DATA(D3DRENDERSTATE_WRAPU, shapePtr->wrapU, lpPointer);
	STATE_DATA(D3DRENDERSTATE_WRAPV, shapePtr->wrapV, lpPointer);
    }

    OP_STATE_RENDER(1, lpPointer);
    STATE_DATA(D3DRENDERSTATE_FILLMODE, shapePtr->fillType, lpPointer);
    
    /*
     * Make sure that the triangle data (not OP) will be QWORD aligned
     */
    if (QWORD_ALIGNED(lpPointer)) {
        OP_NOP(lpPointer);
    }
    OP_TRIANGLE_LIST(shapePtr->num_faces, lpPointer);
        TRIANGLE_LIST_DATA(shapePtr->lpTri, shapePtr->num_faces, lpPointer);
    OP_EXIT(lpPointer);
    /*
     * Setup the execute data describing the buffer
     */
    lpD3DExBuf->lpVtbl->Unlock(lpD3DExBuf);
    memset(&d3dExData, 0, sizeof(D3DEXECUTEDATA));
    d3dExData.dwSize = sizeof(D3DEXECUTEDATA);
    d3dExData.dwVertexCount = shapePtr->num_vertices;
    d3dExData.dwInstructionOffset = (ULONG)((char*)lpInsStart - (char*)lpBufStart);
    d3dExData.dwInstructionLength = (ULONG)((char*)lpPointer - (char*)lpInsStart);
    lpD3DExBuf->lpVtbl->SetExecuteData(lpD3DExBuf, &d3dExData);
   
    shapePtr->lpD3DExBuf = lpD3DExBuf;
    shapePtr->d3dExData  = d3dExData; 
    
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderShape --
 *
 *	Renders a shape into the coordinate system. The shapes are
 *	displayed only if they lie within the viewing volume of 
 *	the camera, else discarded. 
 *
 * Results:
 *	Return GE_OK if successful, else Direct3D error constant
 *	value is returned.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_RenderShape(face, data)
    geInterface face;
    ShapeData data;
{
    D3DAppInfo* d3dapp = face.d3dapp;
    D3DTRANSFORMDATA transData;
    DWORD offscreen;
    HRESULT retVal;
    D3DRECT extents[D3DAPP_MAXCLEARRECTS];	/* Extents are used to blt 
						 * only the dirty portion of 
						 * the front buffers */
    int count;
    LPD3DVERTEX out = (LPD3DVERTEX)malloc(data.num_vertices * sizeof(D3DVERTEX)); 
    D3DHVERTEX homo;
    /*
     * Check if the shape lies in the viewing volume. If not then, return.
     */
    memset(&transData, 0, sizeof(D3DTRANSFORMDATA));
    transData.dwSize = sizeof(D3DTRANSFORMDATA);
    transData.lpIn = data.lpV;
    transData.dwInSize = 5;	/* 3 normal + 2 texture values */
    transData.lpOut = out;
    transData.dwOutSize = 5;	/* 3 normal + 2 texture values */
    transData.lpHOut = &homo;
    transData.dwClip = D3DCLIP_BACK | D3DCLIP_BOTTOM | D3DCLIP_FRONT | 
	    D3DCLIP_LEFT | D3DCLIP_RIGHT | D3DCLIP_TOP;
    transData.dwClipIntersection = 
	    D3DSTATUS_CLIPINTERSECTIONBACK  | D3DSTATUS_CLIPINTERSECTIONBOTTOM |
	    D3DSTATUS_CLIPINTERSECTIONFRONT | D3DSTATUS_CLIPINTERSECTIONLEFT |
	    D3DSTATUS_CLIPINTERSECTIONRIGHT | D3DSTATUS_CLIPINTERSECTIONTOP;
    transData.dwClipUnion = 
	    D3DSTATUS_CLIPUNIONBACK   | D3DSTATUS_CLIPUNIONBOTTOM |
	    D3DSTATUS_CLIPUNIONFRONT  | D3DSTATUS_CLIPUNIONLEFT |
	    D3DSTATUS_CLIPUNIONBOTTOM | D3DSTATUS_CLIPUNIONTOP ;
    retVal  = d3dapp->lpD3DViewport->lpVtbl->
	TransformVertices(d3dapp->lpD3DViewport, 1, &transData, D3DTRANSFORM_CLIPPED, &offscreen );
    if (retVal != D3D_OK) {
        return retVal;
    }
    /* 
     * Return without rendering if all the resulting vertices are offscreen. 
     */
    if ( offscreen != 0 ) {
	return 0;
    }
    

    /*
     * If all the DD and D3D objects have been initialized we can render
     */
    if (face.d3dapp->bRenderingIsOK) {
        /*
         * Restore any lost surfaces
         */
        if (!RestoreSurf(face)) {
            /*
             * Restoring surfaces sometimes fails because the surfaces cannot
             * yet be restored.  If this is the case, the error will show up
             * somewhere else and we should return success here to prevent
             * unnecessary error's being reported.
             */
            return TRUE;
        }
        /*
         * Call the sample's RenderScene to render this frame
         */

	retVal = d3dapp->lpD3DDevice->lpVtbl->BeginScene(d3dapp->lpD3DDevice);
	if (retVal != D3D_OK) {
	    return retVal;
	}
	retVal = d3dapp->lpD3DDevice->lpVtbl->
	    Execute(d3dapp->lpD3DDevice, data.lpD3DExBuf,d3dapp->lpD3DViewport, D3DEXECUTE_CLIPPED);
	if (retVal != D3D_OK) { 
	    return retVal;
	}
	retVal = d3dapp->lpD3DDevice->lpVtbl->EndScene(d3dapp->lpD3DDevice);
	if (retVal != D3D_OK) {
	    return retVal;
	}
	retVal = data.lpD3DExBuf->lpVtbl->GetExecuteData(data.lpD3DExBuf, &data.d3dExData);
	if (retVal != D3D_OK) {
	    return retVal;
	}
   
	extents[0] = data.d3dExData.dsStatus.drExtent;

	count = 1;

	/*
	 * Give D3DApp the extents so it can keep track of dirty sections of
	 * the back and front buffers
	 */
	if (!D3DAppRenderExtents(count, extents, face.myglobs.bResized ?
	                        D3DAPP_CLEARALL : (BOOL)NULL)) {
        return FALSE;
	}

	/*
	 * Blt or flip the back buffer to the front buffer.  Don't report an
	 * error if this fails.
	 */
	D3DAppShowBackBuffer(face.myglobs.bResized ? D3DAPP_SHOWALL : (BOOL)NULL);

	/*
	 * Reset the resize flag
	 */
	face.myglobs.bResized = FALSE;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_ViewClears --
 *	Change the option for clearing view after a render. The 
 *	current option is stored in the geInterface structure.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_ViewClears(facePtr, clearFlag)
    geInterface *facePtr; 
    int clearFlag;
{
    if (facePtr->myglobs.bClearsOn == clearFlag) {
	return GE_OK;
    }	
    /*
     * Toggle the clearing the the back buffer and Z-buffer
     * and set the resized flag to clear the entire window
     */
    facePtr->myglobs.bClearsOn = clearFlag;
    if (facePtr->myglobs.bClearsOn) {
	facePtr->myglobs.bResized = FALSE; /* TRUE in d3dmain.cpp...check how it works!*/
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_UpdateClears --
 *
 *	This function is called when the viewport is updated. 
 *	
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_UpdateClears(canvas3d)
    Tk_Canvas3d canvas3d;
{
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);

    if (facePtr->myglobs.bClearsOn) {
	D3DAppClearBackBuffer(D3DAPP_CLEARALL);
    }

    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderFill --
 *
 *	Set the Render option for filltype of a shape. 
 *	Refer tkCanv3dConsts for the different types of filtypes. 
 *	The current choice is stored in the geInterface structure.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RenderFill(facePtr, fillType)
    geInterface *facePtr;
    int fillType;
{
    if (facePtr->myglobs.rstate.FillMode == fillType) {
	return GE_OK;
    }
    facePtr->myglobs.rstate.FillMode = fillType;
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
	return GE_ERROR;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderShade --
 *
 *	Set the Render option for shading style of a shape. 
 *	Refer tkCanv3dConsts for the different types of shading
 *	styles. The current choice is stored in the geInterface 
 *	structure.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RenderShade(facePtr, shadeType)
    geInterface *facePtr;
    int shadeType;
{
    if (facePtr->myglobs.rstate.ShadeMode == shadeType) {
	return GE_OK;
    }
    facePtr->myglobs.rstate.ShadeMode = shadeType;
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
	return GE_ERROR;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderFilter --
 *
 *	Set the Render option for texture filtering type. 
 *	Refer tkCanv3dConsts for the different types of filtering
 *	textures. The current choice is stored in the geInterface 
 *	structure.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int		    
GE_RenderFilter(facePtr, filterType)
    geInterface *facePtr;
    int filterType;
{
    if (facePtr->myglobs.rstate.TextureFilter == filterType) {
	return GE_OK;
    }
    facePtr->myglobs.rstate.TextureFilter = filterType;
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
	return GE_ERROR;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderZbuffer --
 *
 *	Set flag for the option of z-buffering while rendering.
 *	The current choice is stored in the geInterface structure.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RenderZbuffer(facePtr, zFlag)
    geInterface *facePtr; 
    int zFlag;
{
    if (facePtr->myglobs.rstate.bZBufferOn == zFlag) {
	return GE_OK;
    }
    facePtr->myglobs.rstate.bZBufferOn = zFlag;
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
	return GE_ERROR;                                          
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderDither --
 *	
 *	Set flag for the option of dithering while rendering.
 *	The current choice is stored in the geInterface structure.	
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RenderDither(facePtr, ditherFlag)
    geInterface *facePtr;
    int ditherFlag;
{
    if (facePtr->myglobs.rstate.bDithering == ditherFlag) {
	return GE_OK;
    }
    facePtr->myglobs.rstate.bDithering = ditherFlag;
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
	return GE_ERROR;
    }
    return GE_OK;
}


/*
 *--------------------------------------------------------------
 *
 * GE_RenderFog --
 *
 *	Set flag for the fog while rendering.
 *	The current choice is stored in the geInterface structure.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RenderFog(facePtr, fogFlag)
    geInterface *facePtr; 
    int fogFlag;
{
    if (facePtr->myglobs.rstate.bFogEnabled == fogFlag) {
	return GE_OK;
    }
    facePtr->myglobs.rstate.bFogEnabled = fogFlag;
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
	return GE_ERROR;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderAntiAlias --
 *
 *	Set flag for the anti aliasing while rendering.
 *	The current choice is stored in the geInterface structure.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RenderAntiAlias(facePtr,antiAliasFlag)
geInterface *facePtr; 
    int antiAliasFlag;
{
    if (facePtr->myglobs.rstate.bAntialiasing == antiAliasFlag) {
	return GE_OK;
    }
    facePtr->myglobs.rstate.bAntialiasing = antiAliasFlag;
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
	return GE_ERROR;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RenderPerspCor --
 *
 *	Set flag for the perspective correction while rendering.
 *	The current choice is stored in the geInterface structure.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RenderPerspCor(facePtr, perspCorFlag)
    geInterface *facePtr; 
    int perspCorFlag;
{
    if (facePtr->myglobs.rstate.bPerspCorrect == perspCorFlag) {
	return GE_OK;
    }
    facePtr->myglobs.rstate.bPerspCorrect = perspCorFlag;
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
	return GE_ERROR;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RestoreSurf --
 * 
 *	Restores any lost surfaces.  Returns TRUE if all surfaces are not 
 *	lost and FALSE if one or more surfaces is lost and can not be 
 *	restored at the moment.	
 *
 * Results:
 *	Return GE_OK if successful, else GE_ERROR.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static BOOL
RestoreSurf(face)
    geInterface face;
{
    /*
     * Have D3DApp check all the surfaces it's in charge of,
     * including front, back and Z buffers. Restore if lost.
     */
    if (!D3DAppCheckForLostSurfaces()) {
            return GE_ERROR;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_DestroyAllItems --
 *	
 *	Destroy all the items in Canvas3D. This function is
 *	called when the Canvas3d configure options such as 
 *	height and width are changed.
 *  
 * Results:
 *	None.	
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
GE_DestroyAllItems(canvas3dPtr)
    TkCanvas3d *canvas3dPtr;
{
    Tk_Item3d *itemPtr;

    for (itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL; itemPtr = itemPtr->nextPtr) {
	(*itemPtr->typePtr->deleteProc)((Tk_Canvas3d) canvas3dPtr, itemPtr);
		    }
}

/*
 *--------------------------------------------------------------
 *
 * GE_ReCreateAllItems --
 *	Recreate all the items in Canvas3D. This function is
 *	called when the Canvas3d configure options such as 
 *	height and width are changed. 
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
GE_ReCreateAllItems(interp, canvas3dPtr)
    Tcl_Interp *interp;
    TkCanvas3d *canvas3dPtr;
{
    Tk_Item3d *itemPtr;

    /* 
     * Value of argc = 0 does not make any changes inside the 
     * Tk_ConfigureWidget call. This indirectly, does all the 
     * activities in the configProc other than Tk_ConfigureWidget.
     */
    for (itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL; itemPtr = itemPtr->nextPtr) {
		(*itemPtr->typePtr->configProc)(interp,
		(Tk_Canvas3d) canvas3dPtr, itemPtr, 0, NULL, 0);
		    }
}

