/*
 * geShapes.c --
 *
 *	All graphics engine library support for any shape is in this 
 *	file. Supporting utilities for shapes are also here.
 */

#include "tkCanvas3dInt.h"

static int	GenerateBox(float width, float height, float depth, 
			    canv3dCoord center,
			    LPD3DVERTEX* plpv, LPD3DTRIANGLE* plptri, 
			    int* pnum_v, int* pnum_tri);
static int	GenerateCone(float radius, float height, int num_sections,
			    canv3dCoord center,  
			    LPD3DVERTEX* plpv, LPD3DTRIANGLE* plptri, 
			    int* pnum_v, int* pnum_tri);
static int	GenerateCylinder(float radius, float height, int num_sections,
			    canv3dCoord center,
			    LPD3DVERTEX* plpv, LPD3DTRIANGLE* plptri, 
			    int* pnum_v, int* pnum_tri);
static int	GenerateSphere(float sphere_r, int num_rings, int num_sections,
			    canv3dCoord center,
			    LPD3DVERTEX* plpv, LPD3DTRIANGLE* plptri, 
			    int* pnum_v, int* pnum_tri);
static int	GeneratePolygon(float radius, int edges,
			    canv3dCoord center,
			    LPD3DVERTEX* plpv, LPD3DTRIANGLE* plptri, 
			    int* pnum_v, int* pnum_tri);

/*
 * Vertex Specifications :
 *	{x, y, z, nx, ny, nz, u, v}
 *	where x, y, z are homogenous coordinates of the vertex.
 *	nx, ny, nz are normal coordinates of the vertex.
 *	u, v are texture coordinates of the vertex.
 *
 * Triangles Specifications :
 *	v1, v2, v3 - vertices in clockwise direction.
 *	flags - decide which edges of triangle are to be visible.
 */

/*
 *--------------------------------------------------------------
 *
 * GE_InitShape --
 *	
 *	Initialize a shape. Called from the CreateProc of shape
 *	items.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void 
GE_InitShape(dataPtr) 
    ShapeData *dataPtr;
{
    dataPtr->num_vertices = 0;
    dataPtr->num_faces = 0;
    dataPtr->lpV = NULL;
    dataPtr->lpTri = NULL;

    dataPtr->fillType = FILL_SOLID;
    dataPtr->wrapU = FALSE;
    dataPtr->wrapV = FALSE;
}

/*
 *--------------------------------------------------------------
 *
 * GE_CreateBox --
 *
 *	Create a box shape according to the specifications 
 *	in the BoxItem. The box is not displayed, only the 
 *	execute buffers are created.
 *
 * Results:
 *	Return GE_OK if box is created successfully, 
 *	else GE_ERROR.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */


int 
GE_CreateBox(itemPtr)
    BoxItem *itemPtr; 
{   
    if (!(GenerateBox((float)itemPtr->width, (float)itemPtr->height, 
		(float)itemPtr->depth, itemPtr->center,
		&itemPtr->shapeData->lpV, &itemPtr->shapeData->lpTri,
                &itemPtr->shapeData->num_vertices, &itemPtr->shapeData->num_faces)))	
                                return GE_ERROR;
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_CreateCone --
 *
 *	Create a cone shape according to the specifications 
 *	in the ConeItem. The cone is not displayed, only the 
 *	execute buffers are created.
 *
 * Results:
 *	Return GE_OK if box is created successfully, 
 *	else GE_ERROR.	
 *
 * Side effects:
 *
 *
 *--------------------------------------------------------------
 */

int 
GE_CreateCone(itemPtr)
    ConeItem *itemPtr;
{
    if (!(GenerateCone((float)itemPtr->radius, (float)itemPtr->height, 
			itemPtr->numSections, itemPtr->center, 
			&itemPtr->shapeData->lpV, &itemPtr->shapeData->lpTri,
                        &itemPtr->shapeData->num_vertices, &itemPtr->shapeData->num_faces)))
                                return GE_ERROR;
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_CreateCylinder --
 *
 *	Create a cylinder shape according to the specifications 
 *	in the CylinderItem. The cylinder is not displayed, only the 
 *	execute buffers are created.	
 *
 * Results:
 *	Return GE_OK if box is created successfully, 
 *	else GE_ERROR.	
 *
 * Side effects:
 *
 *
 *--------------------------------------------------------------
 */

int 
GE_CreateCylinder(itemPtr)
    CylinItem *itemPtr;
{
    if (!(GenerateCylinder((float)itemPtr->radius, (float)itemPtr->height, 
			itemPtr->numSections, itemPtr->center, 
			&itemPtr->shapeData->lpV, &itemPtr->shapeData->lpTri,
                        &itemPtr->shapeData->num_vertices, &itemPtr->shapeData->num_faces)))
                                return GE_ERROR;
    
     return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_CreateSphere --
 *
 *	Create a sphere shape according to the specifications 
 *	in the SphrItem. The sphere is not displayed, only the 
 *	execute buffers are created.	
 *
 * Results:
 *	Return GE_OK if box is created successfully, 
 *	else GE_ERROR.	
 *
 * Side effects:
 *
 *
 *--------------------------------------------------------------
 */

int 
GE_CreateSphere(itemPtr)
    SphrItem *itemPtr; 
{
    /*
     * Generate the sphere which will be used as the twisted object.  Scale
     * one axis long.  Rotate the sphere around the x axis so the end caps
     * are around the z axis (for effect).  Stretch one end of the sphere 
     * out to allow an easier view of the twisting.
     */
    if (!(GenerateSphere((float)itemPtr->radius, itemPtr->numRings, 
			itemPtr->numSections, itemPtr->center,
			&itemPtr->shapeData->lpV, &itemPtr->shapeData->lpTri,
                        &itemPtr->shapeData->num_vertices, &itemPtr->shapeData->num_faces)))
                                return GE_ERROR;
    
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_CreatePolygon --
 *
 *	Create a polygon shape according to the specifications 
 *	in the PolygonItem. The polygon is not displayed, only the 
 *	execute buffers are created.	
 *
 * Results:
 *	Return GE_OK if box is created successfully, 
 *	else GE_ERROR.	
 *
 * Side effects:
 *
 *
 *--------------------------------------------------------------
 */

int 
GE_CreatePolygon(itemPtr)
    PolygonItem *itemPtr;
{
    if (!(GeneratePolygon((float)itemPtr->radius, itemPtr->edges, 
			itemPtr->center,
			&itemPtr->shapeData->lpV, &itemPtr->shapeData->lpTri,
                        &itemPtr->shapeData->num_vertices, &itemPtr->shapeData->num_faces)))
                                return GE_ERROR;
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_DestroyShape --
 *
 *	Destroys any shape item.
 *
 * Results:
 *	Returns GE_OK.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int 
GE_DestroyShape(dataPtr)
    ShapeData *dataPtr; 
{
    if (dataPtr->lpTri) {
        free(dataPtr->lpTri);
	dataPtr->lpTri = NULL;
    }
    if (dataPtr->lpV) {
        free(dataPtr->lpV);
	dataPtr->lpV = NULL;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GenerateBox --
 *
 *	Generate a list of vertices and triangles depending
 *	upon the specified dimensions and its center.
 *
 * Results:
 *	Returns back the pointers to list of vertices, list 
 *	of faces, and also returns the number of vertices
 *	and faces through passed parameters. If unsuccessful, 
 *	then a GE_ERROR is returned, else GE_OK.
 *
 * Side effects:
 *
 *
 *--------------------------------------------------------------
 */

static int
GenerateBox(float width,
    float height,
    float depth,
    canv3dCoord c,
    LPD3DVERTEX* plpv,
    LPD3DTRIANGLE* plptri,
    int *pnum_v,
    int *pnum_tri)
{
    int num_v, num_tri; 
    LPD3DVERTEX lpv;		/* Internal pointer for vertices */
    LPD3DTRIANGLE lptri;	/* Internal pointer for triangles */
    int i;

    /* 
     * Adjust the box equally around the origin.
     */
    float w = width/(float)2.0;
    float h = height/(float)2.0;
    float d = depth/(float)2.0; 

    D3DVERTEX CubeVertices[] = {
	/* top */
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(1.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(0.0) },
	/* front */
	{D3DVAL((1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(0.0),D3DVAL(-1.0),D3DVAL(1.0),D3DVAL(0.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(0.0),D3DVAL(-1.0),D3DVAL(1.0),D3DVAL(1.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(0.0),D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(1.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(0.0),D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(0.0) },
	/* right */
	{D3DVAL((1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(1.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(0.0) },
	/* back */
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(1.0),D3DVAL(0.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(1.0),D3DVAL(1.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0),D3DVAL(1.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0),D3DVAL(0.0) },
	/* left */
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(1.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(0.0) },
	/* bottom */
	{D3DVAL((1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(1.0) },
	{D3DVAL((1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(1.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(0.0),D3DVAL(0.0) },
	{D3DVAL((-1.0 * w) + c.x),D3DVAL((-1.0 * h) + c.y),D3DVAL((-1.0 * d) + c.z),
	    D3DVAL(0.0),D3DVAL(-1.0),D3DVAL(0.0),D3DVAL(1.0),D3DVAL(0.0) }
      };
    
    int CubeTri[] = {
	0,  1,  2, 2, 3, 0,	    /* top */
	4,  5,  6,  6, 7, 4,	    /* front */
	8,  9, 10, 10, 11, 8,	    /* right */
	12, 13, 14, 14, 15, 12,	    /* back */
	16, 17, 18, 18, 19, 16,	    /* left */
	20, 21, 22, 22, 23, 20	    /* bottom */
	};

    if ((width <= 0) || (height < 0) || (depth < 0))
        return GE_ERROR;
    
    /* 
     * Allocate space for the box.
     */
    num_v = 24;
    num_tri = 12;
    *plpv = (LPD3DVERTEX) malloc(sizeof(D3DVERTEX) * num_v);
    *plptri = (LPD3DTRIANGLE) malloc(sizeof(D3DTRIANGLE) * num_tri);
    *pnum_v = num_v;
    *pnum_tri = num_tri;
    lpv = *plpv;
    lptri = *plptri;
  
    /* 
     * Calcuate the vertex positions.
     */
    for (i = 0; i < num_v; i++) {
            lpv[i].x = CubeVertices[i].x;
            lpv[i].y = CubeVertices[i].y;
            lpv[i].z = CubeVertices[i].z;    
            lpv[i].nx = CubeVertices[i].nx;
            lpv[i].ny = CubeVertices[i].ny;
            lpv[i].nz = CubeVertices[i].nz;    
	    lpv[i].tu = CubeVertices[i].tu;
            lpv[i].tv = CubeVertices[i].tv;
    }
    
    /* 
     * Calculate the faces connections. 
     */
    for (i = 0; i < num_tri; i++) {
            lptri[i].v1 = CubeTri[3*i];
            lptri[i].v2 = CubeTri[(3*i)+1];
            lptri[i].v3 = CubeTri[(3*i)+2];    
            /* Enable correct edges.*/
            lptri[i].wFlags = D3DTRIFLAG_EDGEENABLE1 |
		              D3DTRIFLAG_EDGEENABLE2;
            /* Build fans. */
            if (i == 0) {
                lptri[i].wFlags |= D3DTRIFLAG_START;
            } else {
                lptri[i].wFlags |= D3DTRIFLAG_EVEN;
            }
            
    }
   return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GenerateCone --
 *
 *	Generate a cone according to required specifications
 *	and complexity.
 *
 * Results:
 *	Returns back the pointers to list of vertices, list 
 *	of faces, and also returns the number of vertices
 *	and faces through passed parameters. If unsuccessful, 
 *	then a GE_ERROR is returned, else GE_OK.	
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int 
GenerateCone(float radius,
    float height,
    int num_sections,
    canv3dCoord c,
    LPD3DVERTEX* plpv,
    LPD3DTRIANGLE* plptri,
    int* pnum_v,
    int* pnum_tri)
{
    float theta, phi;    /* Angles used to sweep around sphere */
    float x, y, z; /* Temporary variables */
    int i, j, n;      /* counters */
    int num_v, num_tri;  /* Internal vertex and triangle count */
    LPD3DVERTEX lpv;     /* Internal pointer for vertices */
    LPD3DTRIANGLE lptri; /* Internal pointer for trianlges */

    /*
     * Check the parameters to make sure they are valid.
     */
    if ((radius <= 0) || (height <= 0.0) || (num_sections < 3))
        return FALSE;
    /*
     * Generate space for the required triangles and vertices.
     */
    num_tri = 2 * num_sections;   /* for two disks and surface */
    num_v = num_sections + 2; /* top point + 
				     * (num_sections + center point)*/
    *plpv = (LPD3DVERTEX) malloc(sizeof(D3DVERTEX) * num_v);
    *plptri = (LPD3DTRIANGLE) malloc(sizeof(D3DTRIANGLE) * num_tri);
    lpv = *plpv;
    lptri = *plptri;
    *pnum_v = num_v;
    *pnum_tri = num_tri;

    /* Make the base disk for the cone. */
    n = 0;	     /* First point is the center point */
    y = -height/(float)2.0; /* At negative half_height */

    /* center point */
    lpv[n].x = D3DVAL(c.x); lpv[n].y = D3DVAL(y + c.y); lpv[n].z = D3DVAL(c.z);
    lpv[n].tu = D3DVAL(0.0); lpv[n].tv = D3DVAL(0.0);
    n++;
    theta = (float)0.0;
    phi = (float)0.0;
    for ( ;n <= num_sections; n++) {
	x = radius * (float)sin(theta);
	z = radius * (float)cos(theta);
	lpv[n].x = D3DVAL(x + c.x);
	lpv[n].z = D3DVAL(z + c.y);
	lpv[n].y = D3DVAL(y + c.z);
	lpv[n].tu = D3DVAL(x/radius);
	lpv[n].tv = D3DVAL(y/radius);
	theta += (float)(2.0 * PI / (double) num_sections);
    } 
    y += height;
    /* top point */
    lpv[n].x = D3DVAL(c.x);  lpv[n].y = D3DVAL(y + c.y);  lpv[n].z = D3DVAL(c.z);
    lpv[n].tu = D3DVAL(0.0); lpv[n].tv = D3DVAL(1.0);

    /* 
     * Make the triangles for the disk, and show only the outer border. 
     */
    for (i = 0; i < num_sections; i++) {
	if(i < (num_sections - 1)) {
	    lptri[i].v1 = 0;
	    lptri[i].v2 = i + 2;
	    lptri[i].v3 = i + 1;
	} else {
	    lptri[i].v1 = 0;
	    lptri[i].v2 = 1;
	    lptri[i].v3 = i+1;
	}             
	/* 
	 * Enable correct edges.
	 */
	lptri[i].wFlags = D3DTRIFLAG_EDGEENABLE2; 
    }
    
    /* 
     * Connect the top point to the disk to form the cone's surface 
     */
    i = num_sections;
    for (j = 0; j < (num_sections - 1); j++) {
	lptri[i].v1 = num_sections + 1;	/* top point */
	lptri[i].v2 = j + 1;
	lptri[i].v3 = j + 2;
	/* Enable two edges. */
	lptri[i].wFlags = D3DTRIFLAG_EDGEENABLETRIANGLE;

	i++;
    }
    lptri[i].v1 = num_sections + 1;
    lptri[i].v2 = num_sections;
    lptri[i].v3 = 1;
    /* Enable two edges. */
    lptri[i].wFlags = D3DTRIFLAG_EDGEENABLETRIANGLE;

    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * GenerateCylinder --
 *
 *	Generate a cylinder according to the required specifications
 *	and dimensions.
 *
 * Results:
 *	Returns back the pointers to list of vertices, list 
 *	of faces, and also returns the number of vertices
 *	and faces through passed parameters. If unsuccessful, 
 *	then a GE_ERROR is returned, else GE_OK.	
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static int 
GenerateCylinder(float radius,
    float height,
    int num_sections,
    canv3dCoord c,
    LPD3DVERTEX* plpv,
    LPD3DTRIANGLE* plptri,
    int* pnum_v,
    int* pnum_tri)
{

    float theta, phi;    /* Angles used to sweep around sphere */
    float x, y, z;	 /* Temporary variables */
    int i, j, n;         /* counters */
    int num_v, num_tri;  /* Internal vertex and triangle count */
    LPD3DVERTEX lpv;     /* Internal pointer for vertices */
    LPD3DTRIANGLE lptri; /* Internal pointer for trianlges */
    int disk, disk_vertex;

    /*
     * Check the parameters to make sure they are valid.
     */
    if ((radius <= 0) || (height <= 0.0) || (num_sections < 3))
        return FALSE;
    /*
     * Generate space for the required triangles and vertices.
     */
    num_tri = 4 * num_sections;   /* for two disks and surface */
    num_v = 2 * (num_sections + 1); /*(num_sections + center point)
				   * for two disks */
    *plpv = (LPD3DVERTEX) malloc(sizeof(D3DVERTEX) * num_v);
    *plptri = (LPD3DTRIANGLE) malloc(sizeof(D3DTRIANGLE) * num_tri);
    lpv = *plpv;
    lptri = *plptri;
    *pnum_v = num_v;
    *pnum_tri = num_tri;

    /* 
     * Make two disks, for the cylinder top and bottom 
     */
    disk = 0;
    n = 0;
    y = height/(float)2.0; /* First disk at positive half_height */
    disk_vertex = num_sections;
    while(disk < 2) {
	/* center point */
	lpv[n].x = D3DVAL(c.x);
	lpv[n].y = D3DVAL(y + c.y);
	lpv[n].z = D3DVAL(c.z);
	lpv[n].tv = D3DVAL(0.0);
	lpv[n].tu = D3DVAL(1.0);
	n++;
	theta = (float)0.0;
	phi = (float)0.0;
	for ( ;n <= disk_vertex; n++) {
            x = radius * (float)sin(theta);
            z = radius * (float)cos(theta);
            lpv[n].x = D3DVAL(x + c.x);
            lpv[n].z = D3DVAL(z + c.y);
            lpv[n].y = D3DVAL(y + c.z);
            lpv[n].nx = D3DVAL(x / radius);
	    lpv[n].tv = D3DVAL(x/radius);
	    lpv[n].tu = D3DVAL(y/radius);
	    theta += (float)(2.0 * PI / (double) num_sections);
        }
	disk++;		/* Next disk. */
	y -= height;	/* Second disk at negative half_height */
	disk_vertex += (num_sections + 1);
    }
    
    /* 
     * Make the top disk. 
     */
    for (i = 0; i < num_sections; i++) {
	if(i < (num_sections - 1)) {
	    lptri[i].v1 = 0;
	    lptri[i].v2 = i + 1;
	    lptri[i].v3 = i + 2;
	} else {
	    lptri[i].v1 = 0;
	    lptri[i].v2 = i+1;
	    lptri[i].v3 = 1;
	} 
               
	/* 
	 * Enable correct edges. 
	 */
	lptri[i].wFlags = D3DTRIFLAG_EDGEENABLE2; 
    }

    /* 
     * Make the bottom disk. 
     */
    for (i = 0; i < num_sections; i++) {
	if(i < (num_sections - 1)) {
	    lptri[i + num_sections].v1 = num_sections + 1;
	    lptri[i + num_sections].v2 = i + num_sections + 3;
	    lptri[i + num_sections].v3 = i + num_sections + 2;
	} else {
	    lptri[i + num_sections].v1 = num_sections + 1;
	    lptri[i + num_sections].v2 = num_sections + 2;
	    lptri[i + num_sections].v3 = i + num_sections + 2;
	} 
	/* 
	 * Enable correct edges. 
	 */
	lptri[i + num_sections].wFlags = D3DTRIFLAG_EDGEENABLE2;              
    }
    
    /* 
     * Connect the two disks to form the cylinder's surface 
     */
    i = num_sections * 2;
    for (j = 1; j < num_sections; j++) {
	lptri[i].v1 = j;
	lptri[i].v2 = j + num_sections + 1;
	lptri[i].v3 = j + num_sections + 2;
	
	lptri[i+1].v1 = j + num_sections + 2;
	lptri[i+1].v2 = j + 1;
	lptri[i+1].v3 = j;

	lptri[i].wFlags = D3DTRIFLAG_EDGEENABLE1
			  | D3DTRIFLAG_EDGEENABLE2; 
	lptri[i+1].wFlags = D3DTRIFLAG_EDGEENABLE1
			  | D3DTRIFLAG_EDGEENABLE2; 
	
	i += 2;
    }

    lptri[i].v1 = num_sections;
    lptri[i].v2 = num_sections + num_sections + 1;
    lptri[i].v3 = num_sections + 2;
    
    lptri[i+1].v1 = num_sections + 2;
    lptri[i+1].v2 = 1;
    lptri[i+1].v3 = num_sections;

    lptri[i].wFlags = D3DTRIFLAG_EDGEENABLE1
		      | D3DTRIFLAG_EDGEENABLE2; 
    lptri[i+1].wFlags = D3DTRIFLAG_EDGEENABLE1
			| D3DTRIFLAG_EDGEENABLE2; 

    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * GenerateSphere --
 *  
 *	Generate a sphere shape according to required dimensions
 *	and complexity.
 *
 * Results:
 *	Returns back the pointers to list of vertices, list 
 *	of faces, and also returns the number of vertices
 *	and faces through passed parameters. If unsuccessful, 
 *	then a GE_ERROR is returned, else GE_OK.	
 *
 * Side effects:
 *
 *
 *--------------------------------------------------------------
 */

/*
 * Generates a sphere around the y-axis centered at the origin including
 * normals and texture coordiantes.  Returns TRUE on success and FALSE on
 * failure.
 *     sphere_r     Radius of the sphere.
 *     num_rings    Number of full rings not including the top and bottom
 *                  caps.
 *     num_sections Number of sections each ring is divided into.  Each
 *                  section contains two triangles on full rings and one 
 *                  on top and bottom caps.
 *     plpv         On exit points to the vertices of the sphere.  The
 *                  function allocates this space.  Not allocated if
 *                  function fails.
 *     plptri       On exit points to the triangles of the sphere which 
 *                  reference vertices in the vertex list.  The function
 *                  allocates this space. Not allocated if function fails.
 *     pnum_v       On exit contains the number of vertices.
 *     pnum_tri     On exit contains the number of triangles.
 */

static int 
GenerateSphere(float sphere_r,	
    int num_rings,
    int num_sections,
    canv3dCoord c,
    LPD3DVERTEX* plpv,
    LPD3DTRIANGLE* plptri,    
    int* pnum_v,	
    int* pnum_tri)	
{

    float theta, phi;    /* Angles used to sweep around sphere */
    float dtheta, dphi;  /* Angle between each section and ring */
    float x, y, z, v, rsintheta; /* Temporary variables */
    int i, j, n, m;      /* counters */
    int num_v, num_tri;  /* Internal vertex and triangle count */
    LPD3DVERTEX lpv;     /* Internal pointer for vertices */
    LPD3DTRIANGLE lptri; /* Internal pointer for trianlges */

    /*
     * Check the parameters to make sure they are valid.
     */
    if ((sphere_r <= 0) || (num_rings < 1) || (num_sections < 3))
        return FALSE;
    /*
     * Generate space for the required triangles and vertices.
     */
    num_tri = (num_rings + 1) * num_sections * 2;
    num_v = (num_rings + 1) * num_sections + 2;
    *plpv = (LPD3DVERTEX) malloc(sizeof(D3DVERTEX) * num_v);
    *plptri = (LPD3DTRIANGLE) malloc(sizeof(D3DTRIANGLE) * num_tri);
    lpv = *plpv;
    lptri = *plptri;
    *pnum_v = num_v;
    *pnum_tri = num_tri;

    /*
     * Generate vertices at the top and bottom points.
     */
    lpv[0].x = D3DVAL(c.x);
    lpv[0].y = D3DVAL(sphere_r + c.y);
    lpv[0].z = D3DVAL(c.z);
    lpv[0].nx = D3DVAL(0.0);
    lpv[0].ny = D3DVAL(1.0);
    lpv[0].nz = D3DVAL(0.0);
    lpv[0].tu = D3DVAL(0.0);
    lpv[0].tv = D3DVAL(0.0);
    lpv[num_v - 1].x = D3DVAL(c.x);
    lpv[num_v - 1].y = D3DVAL(-sphere_r + c.y);
    lpv[num_v - 1].z = D3DVAL(c.z);
    lpv[num_v - 1].nx = D3DVAL(0.0);
    lpv[num_v - 1].ny = D3DVAL(-1.0);
    lpv[num_v - 1].nz = D3DVAL(0.0);
    lpv[num_v - 1].tu = D3DVAL(0.0);
    lpv[num_v - 1].tv = D3DVAL(1.0);


    /*
     * Generate vertex points for rings
     */
    dtheta = (float)(PI / (double)(num_rings + 2));
    dphi = (float)(2.0 * PI / (double) num_sections);
    n = 1; /* vertex being generated, begins at 1 to skip top point */
    theta = dtheta;
    for (i = 0; i <= num_rings; i++) {
        y = sphere_r * (float)cos(theta); /* y is the same for each ring */
        v = theta / (float)PI;     /* v is the same for each ring */
        rsintheta = sphere_r * (float)sin(theta);
        phi = (float)0.0;
        for (j = 0; j < num_sections; j++) {
            x = rsintheta * (float)sin(phi);
            z = rsintheta * (float)cos(phi);
            lpv[n].x = D3DVAL(x + c.x);
            lpv[n].z = D3DVAL(z + c.y);
            lpv[n].y = D3DVAL(y + c.z);
            lpv[n].nx = D3DVAL(x / sphere_r);
            lpv[n].ny = D3DVAL(y / sphere_r);
            lpv[n].nz = D3DVAL(z / sphere_r);
            lpv[n].tv = D3DVAL(v);
            lpv[n].tu = D3DVAL((float)(1.0 - phi / (2.0 * PI)));
            phi += dphi;
            ++n;
        }
        theta += dtheta;
    }

    /*
     * Generate triangles for top and bottom caps.
     */
    if (num_sections < 30) {
    /*
     * we can put the whole cap in a tri fan.
     */
        for (i = 0; i < num_sections; i++) {
            lptri[i].v1 = 0;
            lptri[i].v2 = i + 1;
            lptri[i].v3 = 1 + ((i + 1) % num_sections);
            
            lptri[num_tri - num_sections + i].v1 = num_v - 1;
            lptri[num_tri - num_sections + i].v2 = num_v - 2 - i;
            lptri[num_tri - num_sections + i].v3 = num_v - 2 - 
                    ((1 + i) % num_sections);
                    
                   
            /*
             * Enable correct edges.
             */
            lptri[i].wFlags = D3DTRIFLAG_EDGEENABLE1 |
                              D3DTRIFLAG_EDGEENABLE2;
                              
            lptri[num_tri - num_sections + i].wFlags= D3DTRIFLAG_EDGEENABLE1 |
                                                      D3DTRIFLAG_EDGEENABLE2;
            /*
             * build fans.
             */
            if (i == 0) {
                lptri[i].wFlags |= D3DTRIFLAG_START;
                lptri[num_tri - num_sections + i].wFlags |= D3DTRIFLAG_START;
            } else {
                lptri[i].wFlags |= D3DTRIFLAG_EVEN;
                lptri[num_tri - num_sections + i].wFlags |= D3DTRIFLAG_EVEN;
            }
            
        }
    } else {
        for (i = 0; i < num_sections; i++) {
            lptri[i].v1 = 0;
            lptri[i].v2 = i + 1;
            lptri[i].v3 = 1 + ((i + 1) % num_sections);
            lptri[i].wFlags = D3DTRIFLAG_EDGEENABLE1;
                              D3DTRIFLAG_EDGEENABLE2;
            lptri[num_tri - num_sections + i].v1 = num_v - 1;
            lptri[num_tri - num_sections + i].v2 = num_v - 2 - i;
            lptri[num_tri - num_sections + i].v3 = num_v - 2 - 
                    ((1 + i) % num_sections);
            lptri[num_tri - num_sections + i].wFlags= D3DTRIFLAG_EDGEENABLE1 |
                                                      D3DTRIFLAG_EDGEENABLE2;
        }
    }

    /*
     * Generate triangles for the rings
     */
    m = 1; /* first vertex in current ring,begins at 1 to skip top point*/
    n = num_sections; /* triangle being generated, skip the top cap */
        for (i = 0; i < num_rings; i++) {
        for (j = 0; j < num_sections; j++) {
            lptri[n].v1 = m + j;
            lptri[n].v2 = m + num_sections + j;
            lptri[n].v3 = m + num_sections + ((j + 1) % num_sections);
            lptri[n].wFlags = D3DTRIFLAG_EDGEENABLETRIANGLE;
            
            /*
             * Start a two triangle flat fan for each face.
             */
                    
            lptri[n].wFlags = D3DTRIFLAG_STARTFLAT(1);
            
            /*
             * only need two edges for wireframe.
             */ 
            lptri[n].wFlags |= D3DTRIFLAG_EDGEENABLE1 |
                               D3DTRIFLAG_EDGEENABLE2;
        
            
            lptri[n + 1].v1 = lptri[n].v1;
            lptri[n + 1].v2 = lptri[n].v3;
            lptri[n + 1].v3 = m + ((j + 1) % num_sections);
            
            lptri[n + 1].wFlags = D3DTRIFLAG_EVEN;
            /*
             * only need two edges for wireframe.
             */ 
            lptri[n + 1].wFlags |= D3DTRIFLAG_EDGEENABLE2 |
                                   D3DTRIFLAG_EDGEENABLE3;
            n += 2;
        }
        m += num_sections;
    }
    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * GeneratePolygon --
 * 
 *	Generate a polygon shape according the required dimensions
 *	and complexity.
 *
 * Results:
 *	Returns back the pointers to list of vertices, list 
 *	of faces, and also returns the number of vertices
 *	and faces through passed parameters. If unsuccessful, 
 *	then a GE_ERROR is returned, else GE_OK.	
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int 
GeneratePolygon(float radius,
    int edges,
    canv3dCoord c,
    LPD3DVERTEX* plpv,
    LPD3DTRIANGLE* plptri,
    int* pnum_v,
    int* pnum_tri)
{
    float theta;    /* Angles used to sweep around. */
    float x, y, z; /* Temporary variables */
    int i, n;      /* counters */
    int num_v, num_tri;  /* Internal vertex and triangle count */
    LPD3DVERTEX lpv;     /* Internal pointer for vertices */
    LPD3DTRIANGLE lptri; /* Internal pointer for trianlges */

    /*
     * Check the parameters to make sure they are valid.
     */
    if ((radius <= 0.0) || (edges < 1))
        return FALSE;
    /*
     * Generate space for the required triangles and vertices.
     */
    num_tri = edges;   /* for one plane */
    num_v = edges + 1; /* num_sections + center point*/
    *plpv = (LPD3DVERTEX) malloc(sizeof(D3DVERTEX) * num_v);
    *plptri = (LPD3DTRIANGLE) malloc(sizeof(D3DTRIANGLE) * num_tri);
    lpv = *plpv;
    lptri = *plptri;
    *pnum_v = num_v;
    *pnum_tri = num_tri;

    n = 0;	     /* First point-lpv[0] is the center point */

    /* center point at origin */
    lpv[n].x = D3DVAL(c.x); lpv[n].y = D3DVAL(c.y); lpv[n].z = D3DVAL(c.z);
    lpv[n].tu = D3DVAL(0.0); lpv[n].tv = D3DVAL(0.0);
    n++;
    theta = (float)0.0;
    z = (float)0.0;
    for ( ;n <= edges; n++) {
	x = radius * (float)sin(theta);
	y = radius * (float)cos(theta);
	lpv[n].x = D3DVAL(x + c.x);
	lpv[n].y = D3DVAL(y + c.y);
	lpv[n].z = D3DVAL(z + c.z);
	lpv[n].tu = D3DVAL(x/radius);
	lpv[n].tv = D3DVAL(y/radius);
	theta += (float)(2.0 * PI/ (double) edges);
    } 
    
    /* Make the triangles for the disk, and show only the outer border */
    for (i = 0; i < edges; i++) {
	if(i < (edges - 1)) {
	    lptri[i].v1 = 0;
	    lptri[i].v2 = i + 1;
	    lptri[i].v3 = i + 2;
	} else {
	    lptri[i].v1 = 0;
	    lptri[i].v2 = i + 1;
	    lptri[i].v3 = 1;
	}             
	/* Enable correct edges.*/
	lptri[i].wFlags = D3DTRIFLAG_EDGEENABLETRIANGLE; 
    }

    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * GE_FillShape --
 *
 *	Change the fill type (point, wireframe or solid) of the 
 *      shape	
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int	
GE_FillShape(dataPtr, fillType)
    ShapeData *dataPtr;
    int fillType;
{
    dataPtr->fillType = fillType;
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_WrapShape --
 *
 *	Change the wrap style of the shape.	
 *
 * Results:
 *	Returns GE_OK if successful, else GE_ERROR.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int	
GE_WrapShape(dataPtr, wrapStyle)
    ShapeData *dataPtr;
    int wrapStyle;
{
    if (wrapStyle == WRAP_FLAT) { 
	dataPtr->wrapU = FALSE;
	dataPtr->wrapV = FALSE;
    } else if (wrapStyle == WRAP_CYL_U) {
	dataPtr->wrapU = TRUE;
	dataPtr->wrapV = FALSE;
    } else if (wrapStyle == WRAP_CYL_V) {
	dataPtr->wrapU = FALSE;
	dataPtr->wrapV = TRUE;
    } else if (wrapStyle == WRAP_TORUS) {
	dataPtr->wrapU = TRUE;
	dataPtr->wrapV = TRUE;
    } else {
	return GE_ERROR;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_TranslateShape --
 *
 *	Translate the shape vertices into the coordinate system.	
 *
 * Results:
 *	Returns GE_OK if successful. 
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_TranslateShape(dataPtr, tx, ty, tz)
    ShapeData *dataPtr; 
    double tx; 
    double ty; 
    double tz;
{
    int i;
    
    for(i = 0; i < dataPtr->num_vertices; i++) {
	dataPtr->lpV[i].x += (float)tx;
	dataPtr->lpV[i].y += (float)ty;
	dataPtr->lpV[i].z += (float)tz;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_ScaleShape --
 *
 *	Scale the vertices of the shape in the coordinate system.
 *
 * Results:
 *	Return GE_OK if successful, else GE_ERROR.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_ScaleShape(dataPtr, sx, sy, sz)
    ShapeData *dataPtr;
    double sx;
    double sy;
    double sz;
{
    int i;
    
    /* 
     * Scale factor cannot be zero. 
     */
    if (( sx == 0 ) || ( sy == 0 ) || ( sz == 0 ) ) {
	return GE_ERROR;
    }

    for(i = 0; i < dataPtr->num_vertices; i++) {
	dataPtr->lpV[i].x *= (float)sx;
	dataPtr->lpV[i].y *= (float)sy;
	dataPtr->lpV[i].z *= (float)sz;
    }

    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RotateXShape --
 *
 *	Rotate the shape vertices around the X axis in the 
 *	coordinate system. Angle of rotation is specified 
 *	in radians, and the positive/negative values
 *	correspond to clockwise/anticlockwise rotations
 *	respectively.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RotateXShape(dataPtr, angle)
    ShapeData *dataPtr;
    double angle;
{
    float st, ct;
    int i;
    float y, z;

    st = (float)sin(angle); ct = (float)cos(angle);

    for (i = 0; i < dataPtr->num_vertices; i++) {
        y = dataPtr->lpV[i].y; 
	z = dataPtr->lpV[i].z;
        dataPtr->lpV[i].y = ct * y + st * z;
        dataPtr->lpV[i].z = -st * y + ct * z;

        y = dataPtr->lpV[i].ny; 
	z = dataPtr->lpV[i].nz;
        dataPtr->lpV[i].ny = ct * y + st * z;
        dataPtr->lpV[i].nz = -st * y + ct * z;
    }
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RotateYShape --
 *
 *	Rotate the shape vertices around the Y axis in the 
 *	coordinate system. Angle of rotation is specified 
 *	in radians, and the positive/negative values
 *	correspond to clockwise/anticlockwise rotations
 *	respectively.	
 *
 * Results:
 *	Returns GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int		    
GE_RotateYShape(dataPtr, angle)
    ShapeData *dataPtr; 
    double angle;
{
    float st, ct;
    int i;
    float z, x;

    st = (float)sin(angle); ct = (float)cos(angle);
    
    for (i = 0; i < dataPtr->num_vertices; i++) {
        z = dataPtr->lpV[i].z; 
	x = dataPtr->lpV[i].x;
        dataPtr->lpV[i].z = ct * z + st * x;
        dataPtr->lpV[i].x = -st * z + ct * x;
        
	z = dataPtr->lpV[i].nz; 
	x = dataPtr->lpV[i].nx;
        dataPtr->lpV[i].nz = ct * z + st * x;
        dataPtr->lpV[i].nx = -st * z + ct * x;
    }

    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_RotateZShape --
 *
 *	Rotate the shape vertices around the Z axis in the 
 *	coordinate system. Angle of rotation is specified 
 *	in radians, and the positive/negative values
 *	correspond to clockwise/anticlockwise rotations
 *      respectively.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */

int		    
GE_RotateZShape(dataPtr, angle)
    ShapeData *dataPtr;	
    double angle;
{
    float st, ct;
    int i;
    float x, y;

    st = (float)sin(angle); ct = (float)cos(angle);
    
    for (i = 0; i < dataPtr->num_vertices; i++){
        x = dataPtr->lpV[i].x; 
	y = dataPtr->lpV[i].y;
        dataPtr->lpV[i].x = ct * x + st * y;
        dataPtr->lpV[i].y = -st * x + ct * y;
        
	x = dataPtr->lpV[i].nx; 
	y = dataPtr->lpV[i].ny;
        dataPtr->lpV[i].nx = ct * x + st * y;
        dataPtr->lpV[i].ny = -st * x + ct * y;
    } 
    
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_TransformShape --
 *
 *	Apply a transformation matrix to the vertices of the 
	shape in the coordinate system.
 *
 * Results:
 *	Return GE_OK if successful.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int 	    
GE_TransformShape(dataPtr, item)
    ShapeData *dataPtr; 
    MatrixItem item;
{
    int i;

    for (i = 0; i < dataPtr->num_vertices; i++){
	dataPtr->lpV[i].x = (float)((dataPtr->lpV[i].x * item.matrix.a1)
			    + (dataPtr->lpV[i].y * item.matrix.b1)
			    + (dataPtr->lpV[i].z * item.matrix.c1)
			    + item.matrix.d1);
	dataPtr->lpV[i].y = (float)((dataPtr->lpV[i].x * item.matrix.a2)
			    + (dataPtr->lpV[i].y * item.matrix.b2)
			    + (dataPtr->lpV[i].z * item.matrix.c2)
			    + item.matrix.d2);
	dataPtr->lpV[i].z = (float)((dataPtr->lpV[i].x * item.matrix.a3)
			    + (dataPtr->lpV[i].y * item.matrix.b3)
			    + (dataPtr->lpV[i].z * item.matrix.c3)
			    + item.matrix.d3);
    }

    return GE_OK;
}


/*
 *--------------------------------------------------------------
 *
 * GE_AddVertex --
 *
 *	Add a vertex to the list of vertices for a mesh. Memory
 *	adjustments are taken care of inside the function.
 *
 * Results:
 *	Return GE_OK if successful, else GE_ERROR.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int 
GE_AddVertex(itemPtr, x, y, z, vertexNum)
    MeshItem *itemPtr;
    double x;
    double y;
    double z;
    int *vertexNum;
{    
    ShapeData *dataPtr = itemPtr->shapeData;

    /* 
     * Return a number for the vertex added. 
     */
    *vertexNum = dataPtr->num_vertices;
    
    /* 
     * Assign memory to store the vertex. 
     */
    if(dataPtr->num_vertices == 0) {
	itemPtr->availNumVert = itemPtr->expNumVert;
	if((dataPtr->lpV = 
	    (LPD3DVERTEX)malloc(sizeof(D3DVERTEX) * itemPtr->availNumVert)) == NULL) {
	    return GE_ERROR;
	}
    } else if (dataPtr->num_vertices  == itemPtr->availNumVert){
	/* 
	 * Reallocate only for expected number of vertices. 
	 */
	itemPtr->availNumVert += itemPtr->expNumVert;
	if(realloc(dataPtr->lpV, 
	    sizeof(D3DVERTEX) * itemPtr->availNumVert) == NULL) {
	    return GE_ERROR;
	}
    }
    /* 
     * Store the values in the vertex. 
     */
    dataPtr->lpV[dataPtr->num_vertices].x = (float)x;
    dataPtr->lpV[dataPtr->num_vertices].y = (float)y;
    dataPtr->lpV[dataPtr->num_vertices].z = (float)z;

    /* 
     * Increment the total vertex count. 
     */
    dataPtr->num_vertices++;
 
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_AddFace --
 *
 *	Add a new face to the earlier list of faces for a mesh.
 *	Memory adjustments are taken care in the system.
 *  
 * Results:
 *	Returns GE_OK if successful, else GE_ERROR.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int 
GE_AddFace(itemPtr, v1, v2, v3, faceNum)
    MeshItem *itemPtr;
    int v1;
    int v2;
    int v3;
    int *faceNum;
{    
    ShapeData *dataPtr = itemPtr->shapeData;

    /* 
     * Validate the vertex numbers. 
     */
    if((v1 >= dataPtr->num_vertices) 
	|| (v2 >= dataPtr->num_vertices)
	|| (v3 >= dataPtr->num_vertices)) {
	    return FALSE;
    }
    /* 
     * Return a number for the face added. 
     */
    *faceNum = dataPtr->num_faces;
    
    /* 
     * Assign memory to store the faces. 
     */
    if(dataPtr->num_faces == 0) {
	itemPtr->availNumFace = itemPtr->expNumFace;
	if((dataPtr->lpTri = 
	    (LPD3DTRIANGLE)malloc(sizeof(D3DTRIANGLE) * itemPtr->availNumFace)) == NULL) {
	    return GE_ERROR;
	}
    } else if (dataPtr->num_faces == itemPtr->availNumFace){
	/* 
	 * Reallocate only for expected number of faces. 
	 */
	itemPtr->availNumFace += itemPtr->expNumFace;
	if(realloc(dataPtr->lpTri, 
	    sizeof(D3DTRIANGLE) * itemPtr->availNumFace) == NULL) {
	    return GE_ERROR;
	}
    }
    
    /* 
     * Store the values in the face. 
     */
    dataPtr->lpTri[dataPtr->num_faces].v1 = v1;
    dataPtr->lpTri[dataPtr->num_faces].v2 = v2;
    dataPtr->lpTri[dataPtr->num_faces].v3 = v3;

    /* 
     * By default, set all the edges of the face to be visible in wireframe.
     */
    dataPtr->lpTri[dataPtr->num_faces].wFlags = D3DTRIFLAG_EDGEENABLETRIANGLE;
    
    /* 
     * Increment the total number of faces count. 
     */
    dataPtr->num_faces++;
    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_FaceEdgeEnable --
 *
 *	Set the edge enable type for a specified face of the mesh.	
 *
 * Results:
 *	Return a GE_OK if successful, else GE_ERROR.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_FaceEdgeEnable(dataPtr, faceNum, enableType)
    ShapeData *dataPtr;
    int faceNum;
    int enableType;
{
    if(enableType == FACE_EDGE1) {
	dataPtr->lpTri[faceNum].wFlags = D3DTRIFLAG_EDGEENABLE1;
    } else if(enableType == FACE_EDGE2) {
	dataPtr->lpTri[faceNum].wFlags = D3DTRIFLAG_EDGEENABLE2;
    } else if(enableType == FACE_EDGE3) {
	dataPtr->lpTri[faceNum].wFlags = D3DTRIFLAG_EDGEENABLE3;
    } else if(enableType == FACE_EDGE12) {
	dataPtr->lpTri[faceNum].wFlags = D3DTRIFLAG_EDGEENABLE1
					 | D3DTRIFLAG_EDGEENABLE2;
    } else if(enableType == FACE_EDGE23) {
	dataPtr->lpTri[faceNum].wFlags = D3DTRIFLAG_EDGEENABLE2
					 | D3DTRIFLAG_EDGEENABLE3;
    } else if(enableType == FACE_EDGE31) {
	dataPtr->lpTri[faceNum].wFlags = D3DTRIFLAG_EDGEENABLE3
					 | D3DTRIFLAG_EDGEENABLE1 ;
    } else if(enableType == FACE_EDGE123) {
	dataPtr->lpTri[faceNum].wFlags = D3DTRIFLAG_EDGEENABLETRIANGLE;
    } else {
	return GE_ERROR;
    }
    return GE_OK;	    
}
