/* 
 * geMatrix.c --
 *
 *	This file implements all the graphics engine utilities 
 *	for matrices.
 */

#include "tkCanvas3dInt.h"

static void GE_MatrixDataInItem(MatrixItem  *mat1);

/*
 *--------------------------------------------------------------
 *
 * GE_CreateMatrix --
 *
 *	Create a new matrix item.
 *	
 * Results:
 *	Return GE_OK if successful, else return a Direct3D error
 *	constant values.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_CreateMatrix(face, matrixPtr)
    geInterface face; 
    MatrixItem *matrixPtr; 
{
    D3DAppInfo* d3dapp = face.d3dapp;
    D3DMATRIXHANDLE handle;
    HRESULT retVal;

    /* 
     * Prepare a matrix item with the passed values.
     */
    matrixPtr->matrixData->matData._11=(float)matrixPtr->matrix.a1;  
    matrixPtr->matrixData->matData._12=(float)matrixPtr->matrix.a2;  
    matrixPtr->matrixData->matData._13=(float)matrixPtr->matrix.a3;  
    matrixPtr->matrixData->matData._14=(float)matrixPtr->matrix.b4;
    
    matrixPtr->matrixData->matData._21=(float)matrixPtr->matrix.b1;  
    matrixPtr->matrixData->matData._22=(float)matrixPtr->matrix.b2;  
    matrixPtr->matrixData->matData._23=(float)matrixPtr->matrix.b3;  
    matrixPtr->matrixData->matData._24=(float)matrixPtr->matrix.b4;
    
    matrixPtr->matrixData->matData._31=(float)matrixPtr->matrix.c1;  
    matrixPtr->matrixData->matData._32=(float)matrixPtr->matrix.c2;  
    matrixPtr->matrixData->matData._33=(float)matrixPtr->matrix.c3;  
    matrixPtr->matrixData->matData._34=(float)matrixPtr->matrix.c4;
    
    matrixPtr->matrixData->matData._41=(float)matrixPtr->matrix.d1;  
    matrixPtr->matrixData->matData._42=(float)matrixPtr->matrix.d2;  
    matrixPtr->matrixData->matData._43=(float)matrixPtr->matrix.d3;  
    matrixPtr->matrixData->matData._44=(float)matrixPtr->matrix.d4;

    /* 
     * Create a new matrix and get a handle to it.
     */
    retVal = d3dapp->lpD3DDevice->lpVtbl->CreateMatrix(d3dapp->lpD3DDevice, &handle);
    if (retVal != D3D_OK) {
        return retVal;
    }
    retVal = d3dapp->lpD3DDevice->lpVtbl->SetMatrix(d3dapp->lpD3DDevice, handle, &matrixPtr->matrixData->matData);

    if (retVal != D3D_OK) {
	return retVal;
    }

    /*
     * Store the handle in matrixData structure.
     */
    matrixPtr->matrixData->matrixPtr = handle;

    return GE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GE_SetMatrix --
 * 
 *	Change the values of an already created matrix item.	
 *
 * Results:
 *	Return GE_OK if successful, else return the Direct3D 
 *	error constant value.
 *
 * Side effects:
 *
 *
 *--------------------------------------------------------------
 */

int
GE_SetMatrix(face, matrixPtr)
    geInterface face; 
    MatrixItem *matrixPtr; 
{
    D3DAppInfo* d3dapp = face.d3dapp;
    HRESULT retVal;

    /* 
     * Prepate the matrix object with new values.
     */
    matrixPtr->matrixData->matData._11=(float)matrixPtr->matrix.a1;  
    matrixPtr->matrixData->matData._12=(float)matrixPtr->matrix.a2;  
    matrixPtr->matrixData->matData._13=(float)matrixPtr->matrix.a3;  
    matrixPtr->matrixData->matData._14=(float)matrixPtr->matrix.b4;
    
    matrixPtr->matrixData->matData._21=(float)matrixPtr->matrix.b1;  
    matrixPtr->matrixData->matData._22=(float)matrixPtr->matrix.b2;  
    matrixPtr->matrixData->matData._23=(float)matrixPtr->matrix.b3;  
    matrixPtr->matrixData->matData._24=(float)matrixPtr->matrix.b4;
    
    matrixPtr->matrixData->matData._31=(float)matrixPtr->matrix.c1;  
    matrixPtr->matrixData->matData._32=(float)matrixPtr->matrix.c2;  
    matrixPtr->matrixData->matData._33=(float)matrixPtr->matrix.c3;  
    matrixPtr->matrixData->matData._34=(float)matrixPtr->matrix.c4;
    
    matrixPtr->matrixData->matData._41=(float)matrixPtr->matrix.d1;  
    matrixPtr->matrixData->matData._42=(float)matrixPtr->matrix.d2;  
    matrixPtr->matrixData->matData._43=(float)matrixPtr->matrix.d3;  
    matrixPtr->matrixData->matData._44=(float)matrixPtr->matrix.d4;

    /* 
     * Set the new value for the matrix handle.
     */
    retVal = d3dapp->lpD3DDevice->lpVtbl->SetMatrix(d3dapp->lpD3DDevice, 
		matrixPtr->matrixData->matrixPtr, &matrixPtr->matrixData->matData);
    if (retVal != D3D_OK) {
	return retVal;
    }

    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * GE_MatrixDataInItem --
 *
 *	Puts matrixData values in matrix item values for an item.
 *	This function is useful when user wants to check the
 *	current matrix values.
 *
 * Results:
 *	None.	
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
GE_MatrixDataInItem(mat1)
    MatrixItem  *mat1;
{
    /* Now put the temp values in mat1 */
    mat1->matrix.a1 = mat1->matrixData->matData._11;
    mat1->matrix.a2 = mat1->matrixData->matData._12;
    mat1->matrix.a3 = mat1->matrixData->matData._13;
    mat1->matrix.a4 = mat1->matrixData->matData._14;
    mat1->matrix.b1 = mat1->matrixData->matData._21;
    mat1->matrix.b2 = mat1->matrixData->matData._22;
    mat1->matrix.b3 = mat1->matrixData->matData._23;
    mat1->matrix.b4 = mat1->matrixData->matData._24;
    mat1->matrix.c1 = mat1->matrixData->matData._31;
    mat1->matrix.c2 = mat1->matrixData->matData._32;
    mat1->matrix.c3 = mat1->matrixData->matData._33;
    mat1->matrix.c4 = mat1->matrixData->matData._34;
    mat1->matrix.d1 = mat1->matrixData->matData._41;
    mat1->matrix.d2 = mat1->matrixData->matData._42;
    mat1->matrix.d3 = mat1->matrixData->matData._43;
    mat1->matrix.d4 = mat1->matrixData->matData._44;
}

/*
 *--------------------------------------------------------------
 *
 * GE_DestroyMatrix --
 *
 *	Destroy a matrix item.
 *	
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void		    
GE_DestroyMatrix(matrixPtr)
    MatrixItem *matrixPtr;
{
    //free(matrixPtr->matrixData);
}

/*
 *--------------------------------------------------------------
 *
 * GE_MatrixMultiply --
 *
 *	Multiply two homogenous matrices.Two matrix items are 
 *	multiplied and the result stored in the first one.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
GE_MatrixMultiply(mat1, mat2)
    MatrixItem  *mat1;
    MatrixItem  *mat2;
{
    MultiplyD3DMATRIX(&mat1->matrixData->matData, 
	    &mat1->matrixData->matData, &mat2->matrixData->matData);

    GE_MatrixDataInItem(mat1);
}

/*
 *--------------------------------------------------------------
 *
 * GE_InverseMatrix --
 *
 *	Creates an inverse of the matrix and stores it in the
 *	same matrix item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void    
GE_InverseMatrix(face, matrixPtr)
    geInterface face;
    MatrixItem *matrixPtr;
{
    /*
     * Inverse the matrix.
     */
    D3DMATRIXInvert( &matrixPtr->matrixData->matData, 
	    &matrixPtr->matrixData->matData);

    /*
     * Update the new matrix object values in the matrix item.
     */
    GE_MatrixDataInItem(matrixPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GE_SetMatrixZY --
 *
 *	Set the rotation part of a matrix such that the 
 *	vector lpD is the new z-axis and lpU is the new y-axis.	
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
void
GE_SetMatrixZY(face, vectorD, vectorU, matrixPtr)
    geInterface face;
    canv3dDir vectorD;
    canv3dDir vectorU;
    MatrixItem *matrixPtr;
{
    D3DVECTOR d, u;
    
    d.x = (float)vectorD.x; d.y = (float)vectorD.y; d.z = (float)vectorD.z;
    u.x = (float)vectorU.x; u.y = (float)vectorU.y; u.z = (float)vectorU.z;

    /* 
     * Set the rotation as required.
     */
    D3DMATRIXSetRotation(&matrixPtr->matrixData->matData, &d, &u);

    /*
     * Update the values in matrix object to reflect in the matrix item.
     */
    GE_MatrixDataInItem(matrixPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GE_IdentityMatrix -- 
 *
 *	Set a matrix item to the identity matrix.	
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
void	    
GE_IdentityMatrix(face, matrixPtr)
    geInterface face;
    MatrixItem *matrixPtr;
{
    D3DMATRIX tempMat;
    
    /* Prepare a temporary identity matrix. */
    tempMat._11 = (float)1.0;	    tempMat._12 = (float)0.0;
    tempMat._13 = (float)0.0;	    tempMat._14 = (float)0.0;		
    tempMat._21 = (float)0.0;	    tempMat._22 = (float)1.0;
    tempMat._23 = (float)0.0;	    tempMat._24 = (float)0.0;
    tempMat._31 = (float)0.0;	    tempMat._32 = (float)0.0;
    tempMat._33 = (float)1.0;	    tempMat._34 = (float)0.0;
    tempMat._41 = (float)0.0;	    tempMat._42 = (float)0.0;
    tempMat._43 = (float)0.0;	    tempMat._44 = (float)1.0;

    matrixPtr->matrixData->matData = tempMat;
    
    /*
     * Update the new matrix object values in the matrix item.
     */
    GE_MatrixDataInItem(matrixPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GE_Translate --
 *
 *	Apply translation to a matrix by the given positions.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void	    
GE_TranslateMatrix(face, tx, ty, tz, matrixPtr)
    geInterface face;
    double tx;
    double ty;
    double tz;
    MatrixItem *matrixPtr;
{
    D3DMATRIX tempMat;
    
    /* Prepare a temporary translate matrix. */
    tempMat._11 = (float)1.0;	    tempMat._12 = (float)0.0;
    tempMat._13 = (float)0.0;	    tempMat._14 = (float)0.0;		
    tempMat._21 = (float)0.0;	    tempMat._22 = (float)1.0;
    tempMat._23 = (float)0.0;	    tempMat._24 = (float)0.0;
    tempMat._31 = (float)0.0;	    tempMat._32 = (float)0.0;
    tempMat._33 = (float)1.0;	    tempMat._34 = (float)0.0;
    tempMat._41 = (float)tx;	    tempMat._42 = (float)ty;
    tempMat._43 = (float)tz;	    tempMat._44 = (float)1.0;

    MultiplyD3DMATRIX(&matrixPtr->matrixData->matData, 
	    &matrixPtr->matrixData->matData, &tempMat);

    /*
     * Update the new matrix object values in the matrix item.
     */
    GE_MatrixDataInItem(matrixPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GE_Scale --
 *
 *	Apply the specified scale factors to a matrix.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
GE_ScaleMatrix(face, sx, sy, sz, matrixPtr)
    geInterface face;
    double sx;
    double sy;
    double sz;
    MatrixItem *matrixPtr;
{
    D3DMATRIX tempMat;
    
    /* Prepare a temporary scale matrix. */
    tempMat._11 = (float)sx;	    tempMat._12 = (float)0.0;
    tempMat._13 = (float)0.0;	    tempMat._14 = (float)0.0;		
    tempMat._21 = (float)0.0;	    tempMat._22 = (float)sy;
    tempMat._23 = (float)0.0;	    tempMat._24 = (float)0.0;
    tempMat._31 = (float)0.0;	    tempMat._32 = (float)0.0;
    tempMat._33 = (float)sz;	    tempMat._34 = (float)0.0;
    tempMat._41 = (float)0.0;	    tempMat._42 = (float)0.0;
    tempMat._43 = (float)0.0;	    tempMat._44 = (float)1.0;

    MultiplyD3DMATRIX(&matrixPtr->matrixData->matData, 
	    &matrixPtr->matrixData->matData, &tempMat);

    /*
     * Update the new matrix object values in the matrix item.
     */
    GE_MatrixDataInItem(matrixPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GE_RotateX --
 *
 *	Apply a rotation along the x axis on a matrix by the 
 *	specified angle.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
GE_RotateXMatrix(face, angle, matrixPtr)
    geInterface face;
    double angle;
    MatrixItem *matrixPtr;
{
    D3DMATRIX tempMat;
    
    /* Prepare a temporary rotate matrix. */
    tempMat._11 = (float)1.0;		tempMat._12 = (float)0.0;
    tempMat._13 = (float)0.0;		tempMat._14 = (float)0.0;		
    tempMat._21 = (float)0.0;		tempMat._22 = (float)cos(angle);
    tempMat._23 = (float)sin(angle);	tempMat._24 = (float)0.0;
    tempMat._31 = (float)0.0;		tempMat._32 = (float)-sin(angle);
    tempMat._33 = (float)cos(angle);	tempMat._34 = (float)0.0;
    tempMat._41 = (float)0.0;		tempMat._42 = (float)0.0;
    tempMat._43 = (float)0.0;		tempMat._44 = (float)1.0;

    MultiplyD3DMATRIX(&matrixPtr->matrixData->matData, 
	    &matrixPtr->matrixData->matData, &tempMat);

    /*
     * Update the new matrix object values in the matrix item.
     */
    GE_MatrixDataInItem(matrixPtr); 
}

/*
 *--------------------------------------------------------------
 *
 * GE_RotateY --
 *
 *	Apply a rotation along the y axis on a matrix by the 
 *	specified angle.	
 *
 * Results:
 *	None.	
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
GE_RotateYMatrix(face, angle, matrixPtr)
    geInterface face;
    double angle;
    MatrixItem *matrixPtr;
{
    D3DMATRIX tempMat;
    
    /* Prepare a temporary rotate matrix. */
    tempMat._11 = (float)cos(angle);	    tempMat._12 = (float)0.0;
    tempMat._13 = (float)-sin(angle);	    tempMat._14 = (float)0.0;		
    tempMat._21 = (float)0.0;		    tempMat._22 = (float)1.0;
    tempMat._23 = (float)0.0;		    tempMat._24 = (float)0.0;
    tempMat._31 = (float)sin(angle);	    tempMat._32 = (float)0.0;
    tempMat._33 = (float)cos(angle);	    tempMat._34 = (float)0.0;
    tempMat._41 = (float)0.0;		    tempMat._42 = (float)0.0;
    tempMat._43 = (float)0.0;		    tempMat._44 = (float)1.0;

    MultiplyD3DMATRIX(&matrixPtr->matrixData->matData, 
	    &matrixPtr->matrixData->matData, &tempMat);

    /*
     * Update the new matrix object values in the matrix item.
     */
    GE_MatrixDataInItem(matrixPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GE_RotateZ --
 *
 *	Apply a rotation along the z axis on a matrix by the 
 *	specified angle.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
GE_RotateZMatrix(face, angle, matrixPtr)
    geInterface face;
    double angle;
    MatrixItem *matrixPtr;
{
    D3DMATRIX tempMat;
    
    /* Prepare a temporary rotate matrix. */
    tempMat._11 = (float)cos(angle);	    tempMat._12 = (float)sin(angle);
    tempMat._13 = (float)0.0;		    tempMat._14 = (float)0.0;		
    tempMat._21 = (float)-sin(angle);	    tempMat._22 = (float)cos(angle);
    tempMat._23 = (float)0.0;		    tempMat._24 = (float)0.0;
    tempMat._31 = (float)0.0;		    tempMat._32 = (float)0.0;
    tempMat._33 = (float)1.0;		    tempMat._34 = (float)0.0;
    tempMat._41 = (float)0.0;		    tempMat._42 = (float)0.0;
    tempMat._43 = (float)0.0;		    tempMat._44 = (float)1.0;

    MultiplyD3DMATRIX(&matrixPtr->matrixData->matData, 
	    &matrixPtr->matrixData->matData, &tempMat);

    /*
     * Update the new matrix object values in the matrix item.
     */
    GE_MatrixDataInItem(matrixPtr);
}

/*
 *--------------------------------------------------------------
 *
 * GE_MakeProjMatrix --
 *
 *	Make a projection matrix for the viewing volume of the
 *	camera depending on the parameters passed. 
 *
 * Results:
 *	Return the Direct3D error constant value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_MakeProjMatrix(face, halfHeight, backPlanePos, frontPlanePos, matrixPtr)
    geInterface face;
    double halfHeight;
    double backPlanePos;
    double frontPlanePos;
    MatrixItem *matrixPtr;
{
    int retVal;
    double value = (halfHeight * backPlanePos)/(backPlanePos - frontPlanePos);

    /* The matrix is not an item */
    matrixPtr->matrixData = (MatrixData *) ckalloc(sizeof(MatrixData));
    matrixPtr->matrix.a1 = 1.0;	 matrixPtr->matrix.a2 = 0.0;
    matrixPtr->matrix.a3 = 0.0;	 matrixPtr->matrix.a4 = 0.0;		
    matrixPtr->matrix.b1 = 0.0;  matrixPtr->matrix.b2 = 1.0;
    matrixPtr->matrix.b3 = 0.0;	 matrixPtr->matrix.b4 = 0.0;
    matrixPtr->matrix.c1 = 0.0;	 matrixPtr->matrix.c2 = 0.0;
    matrixPtr->matrix.c3 = value/frontPlanePos;	 
    matrixPtr->matrix.c4 = halfHeight/frontPlanePos;
    matrixPtr->matrix.d1 = 0.0;	 matrixPtr->matrix.d2 = 0.0;
    matrixPtr->matrix.d3 = -value;	 
    matrixPtr->matrix.d4 = 0.0;
    
    /* 
     * Create the Matrix Item. 
     */
    retVal = GE_CreateMatrix(face, matrixPtr);
    
    return retVal;
}

/*
 *--------------------------------------------------------------
 *
 * GE_MakeWorldMatrix --
 *
 *	Make a window scaling matrix for projecting the 3d 
 *	coordinates of the shapes into a 2D window.
 *
 * Results:
 *	Return the Direct3D error constant value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
GE_MakeWorldMatrix(face, scaleX, scaleY, originX, originY, matrixPtr)
    geInterface face;
    double scaleX;
    double scaleY;
    double originX;
    double originY;
    MatrixItem *matrixPtr;
{
    int retVal;

    /* The matrix is not an item!!! */
    matrixPtr->matrixData = (MatrixData *) ckalloc(sizeof(MatrixData));
    matrixPtr->matrix.a1 = scaleX;  matrixPtr->matrix.a2 = 0.0;
    matrixPtr->matrix.a3 = 0.0;	    matrixPtr->matrix.a4 = 0.0;		
    matrixPtr->matrix.b1 = 0.0;	    matrixPtr->matrix.b2 = -scaleY;
    matrixPtr->matrix.b3 = 0.0;	    matrixPtr->matrix.b4 = 0.0;
    matrixPtr->matrix.c1 = 0.0;	    matrixPtr->matrix.c2 = 0.0;
    matrixPtr->matrix.c3 = 1.0;	    matrixPtr->matrix.c4 = 0.0;
    matrixPtr->matrix.d1 = originX; matrixPtr->matrix.d2 = originY;
    matrixPtr->matrix.d3 = 0.0;	    matrixPtr->matrix.d4 = 1.0;
    
    /* 
     * Create the Matrix Item 
     */
    retVal = GE_CreateMatrix(face, matrixPtr);

    return retVal;
}

/*
 *--------------------------------------------------------------
 *
 * GE_MakeViewMatrix --
 *
 *	Make a view matrix as a combination of the projection
 *	and the window scaling matrices.
 *
 * Results:
 *	Return the Direct3D error constant value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
void
GE_MakeViewMatrix(projPtr, worldPtr, viewPtr)
    MatrixItem *projPtr;
    MatrixItem *worldPtr;
    MatrixItem *viewPtr;
{
    
    /* 
     * View Matrix is initially an identity matrix. 
     */
    viewPtr->matrixData = (MatrixData *) ckalloc(sizeof(MatrixData));
    viewPtr->matrix.a1 = 1.0;	    viewPtr->matrix.a2 = 0.0;
    viewPtr->matrix.a3 = 0.0;	    viewPtr->matrix.a4 = 0.0;		
    viewPtr->matrix.b1 = 0.0;	    viewPtr->matrix.b2 = 1.0;
    viewPtr->matrix.b3 = 0.0;	    viewPtr->matrix.b4 = 0.0;
    viewPtr->matrix.c1 = 0.0;	    viewPtr->matrix.c2 = 0.0;
    viewPtr->matrix.c3 = 1.0;	    viewPtr->matrix.c4 = 0.0;
    viewPtr->matrix.d1 = 0.0;	    viewPtr->matrix.d2 = 0.0;
    viewPtr->matrix.d3 = 0.0;	    viewPtr->matrix.d4 = 1.0;

    /* 
     * Multiply projection matrix to view matrix.
     */
    GE_MatrixMultiply(viewPtr, projPtr);

    /* 
     * Multiply world matrix to view matrix. 
     */
    GE_MatrixMultiply(viewPtr, worldPtr);
}


/*
 *--------------------------------------------------------------
 *
 * GE_SetCamera --
 *
 *	Set the camera into the coordinate system according to its
 *	current position, and the up and normal vectors.
 *
 * Results:
 *	Return the Direct3D error constant value if any error occurs,
 *	else return GE_OK.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int
GE_SetCamera(facePtr, viewPtr, viewpoint, normal, up)
    geInterface *facePtr;
    ViewItem *viewPtr;
    canv3dCoord viewpoint;
    canv3dDir normal;
    canv3dDir up;

{
    HRESULT retVal;
    MatrixItem *camera;
    camera = (MatrixItem *)ckalloc (sizeof(MatrixItem));
    camera->matrixData = (MatrixData *)ckalloc(sizeof(MatrixData));
    camera->canvas3d = viewPtr->canvas3d;
    camera->itemCode = ITEM_MATRIX;
 
    /* 
     * First initialize the camera to identity matrix. 
     */
    GE_IdentityMatrix(*facePtr, camera);

    /* 
     * Translate to the viewpoint. 
     */
    GE_TranslateMatrix(*facePtr, -viewpoint.x, 
		    -viewpoint.y, -viewpoint.z, camera);
    
    /* 
     * Set the view directions. 
     */
    GE_SetMatrixZY(*facePtr, normal, up, camera);

    /* 
     * If setting the camera for the first time.
     */
    if(viewPtr->worldMat == NULL) {
	GE_CreateMatrix(*facePtr, camera);
	viewPtr->worldMat = camera;
    } else {
	/* 
	 * Every other than the first time.
	 */
	viewPtr->worldMat->matrixData->matData = camera->matrixData->matData;
	retVal = facePtr->d3dapp->lpD3DDevice->lpVtbl->SetMatrix(facePtr->d3dapp->lpD3DDevice, viewPtr->worldMat->matrixData->matrixPtr, &camera->matrixData->matData);
	if (retVal != D3D_OK)
	    return retVal;
	GE_MatrixDataInItem(viewPtr->worldMat);
    }
    
    ckfree((char *)camera);
    return GE_OK;
}


















