/* 
 * geInit.cpp
 *
 *     This file has graphics engine utilities to start Direct3D.
 *     Because of the specific way of accessing COM object methods,
 *     this file is a C++ file.
 */

#include "geUtils.h"

/*
 * GLOBAL VARIABLES
 */

D3DAppInfo* d3dapp;         /* Pointer to read only collection 
			     * of DD and D3D objects maintained 
			     * by D3DApp */
d3dmainglobals myglobs;     /* collection of global variables */

BOOL(*MouseHandler)(UINT, WPARAM, LPARAM);    /* sample's function which traps
                                               * mouse input */
BOOL(*KeyboardHandler)(UINT, WPARAM, LPARAM); /* sample's functions which traps
                                               * keyboard input */


/*
 *  INTERNAL FUNCTION PROTOTYPES
 */

static BOOL BeforeDeviceDestroyed(LPVOID lpContext);
static BOOL AfterDeviceCreated(int w, int h, LPDIRECT3DVIEWPORT* lpViewport,
                               LPVOID lpContext);
void CleanUpAndPostQuit(void);
static void InitGlobals(geInterface *facePtr);
static BOOL AppPause(BOOL f);
extern void ReportD3DAppError(void);
extern BOOL CreateD3DApp(LPSTR lpCmdLine, geInterface *facePtr);

/*
 *--------------------------------------------------------------
 *
 * Direct3D_Init --
 *
 *	This procedure initializes Direct3D for the Canvas3D.
 *
 * Results:
 *	Returns all the required and important pointers to 
 *	hardware devices in the geInterface structure. This 
 *	structure is then used throughout the rest of the 
 *      processing. Almost all the graphics engine functions 
 *      take geInterface parameter.
 *
 * Side effects:
 *	Direct3D reserves the requested memory blocks in for the
 *      primary, flipping and z-buffers surfaces.
 *
 *--------------------------------------------------------------
 */
int 
Direct3d_Init(geInterface *facePtr)
{
    /* 
     * Set appropriate value for system requirements.
     * Can take the value -emulation, -systemmemory.
     */

    LPSTR lpCmdLine = "-systemmemory";	

    /*
     * Call D3DApp to initialize all DD and D3D objects necessary to render.
     * D3DApp will call the device creation callback which will initialize the
     * viewport and the sample's execute buffers.
     */
    if (!CreateD3DApp(lpCmdLine, facePtr))
        return FALSE;

     return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * InitGlobals --
 *
 *	Initialize various globals in the geInterface structure.
 *      Called before creating the Direct3D interface.  
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */  
void
InitGlobals(geInterface *facePtr)
{
    facePtr->d3dapp = NULL;
    memset(&facePtr->myglobs.rstate, 0, sizeof(facePtr->myglobs.rstate));
    memset(&facePtr->myglobs, 0, sizeof(facePtr->myglobs));
    facePtr->myglobs.bClearsOn = TRUE;
    facePtr->myglobs.bShowFrameRate = TRUE;
    facePtr->myglobs.bShowInfo = TRUE;
    MouseHandler = NULL;
    KeyboardHandler = NULL;

    facePtr->myglobs.hInstApp = facePtr->msWinInstance;
}

/*
 *--------------------------------------------------------------
 *
 * CreateD3DApp --
 * 
 *     Create all DirectDraw and Direct3D objects necessary 
 *     to begin rendering. Add the list of D3D drivers to the 
 *     file menu.	
 *
 * Results:
 *	All the object pointers get stored into the geInterface
 *	structure. Returns TRUE if successful, else FALSE.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static BOOL
CreateD3DApp(LPSTR lpCmdLine, geInterface *facePtr)
{
    LPSTR option;
    BOOL bOnlySystemMemory, bOnlyEmulation;
    DWORD flags;
    Defaults defaults;

    /*
     * Initialize a few global values.
     */
    InitGlobals(facePtr);


    /*
     * Parse the command line in seach of one of the following options:
     *     systemmemory  All surfaces should be created in system memory.
     *                   Hardware DD and D3D devices are disabled, but
     *                   debugging during the Win16 lock becomes possible.
     *     emulation     Do not use hardware DD or D3D devices.
     */
    bOnlySystemMemory = FALSE;
    bOnlyEmulation = FALSE;
    option = strtok(lpCmdLine, " -");
    while(option != NULL )   {
        if (!lstrcmp(option, "systemmemory")) {
            bOnlySystemMemory = TRUE;
        } else if (!lstrcmp(option, "emulation")) {
            bOnlyEmulation = TRUE;
        } else {
            Msg("Invalid command line options given.\nLegal options: -systemmemory, -emulation\n");
            return FALSE;
        }
        option = strtok(NULL, " -");
    }
    /*
     * Set the flags to pass to the D3DApp creation based on command line
     */
    flags = ((bOnlySystemMemory) ? D3DAPP_ONLYSYSTEMMEMORY : 0) | 
            ((bOnlyEmulation) ? (D3DAPP_ONLYD3DEMULATION |
                                 D3DAPP_ONLYDDEMULATION) : 0);
    /*
     * Create all the DirectDraw and D3D objects neccesary to render.  The
    * AfterDeviceCreated callback function is called by D3DApp to create the
     * viewport and the example's execute buffers.
     */
    if (!D3DAppCreateFromHWND(flags, facePtr->msWinHandle, AfterDeviceCreated,
                              NULL, BeforeDeviceDestroyed, NULL, &d3dapp)) {
        ReportD3DAppError();
        return FALSE;
    }
    facePtr->d3dapp = d3dapp;
    
    /*
     * Allow the sample to override the default render state and other
     * settings
     */
    if (!D3DAppGetRenderState(&defaults.rs)) {
        ReportD3DAppError();
        return FALSE;
    }
    lstrcpy(defaults.Name, "D3D Example");
    defaults.bTexturesDisabled = FALSE;
    defaults.bResizingDisabled = facePtr->myglobs.bResizingDisabled;
    defaults.bClearsOn = facePtr->myglobs.bClearsOn;
    myglobs.bClearsOn = defaults.bClearsOn;
    myglobs.bResizingDisabled = defaults.bResizingDisabled;
    /*
     * Apply any changes to the render state
     */
    memcpy(&facePtr->myglobs.rstate, &defaults.rs, sizeof(D3DAppRenderState));
    if (!D3DAppSetRenderState(&facePtr->myglobs.rstate)) {
        ReportD3DAppError();
        return FALSE;
    }

    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * AfterDeviceCreated --
 * 
 *     D3DApp will call this function immediately after the D3D 
 *     device has been created (or re-created).  D3DApp expects 
 *     the D3D viewport to be created and returned.  The sample's 
 *     execute buffers are also created (or re-created) here.    	
 *
 * Results:
 *	Returns TRUE if successful, else FALSE.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static BOOL
AfterDeviceCreated(int w, int h, LPDIRECT3DVIEWPORT* lplpViewport, 
		   LPVOID lpContext)
{
    LPDIRECT3DVIEWPORT lpD3DViewport;
    HRESULT rval;

    /*
     * Create the D3D viewport object
     */
    rval = d3dapp->lpD3D->CreateViewport(&lpD3DViewport, NULL);
    if (rval != D3D_OK) {
        Msg("Create D3D viewport failed.\n%s", D3DAppErrorToString(rval));
        CleanUpAndPostQuit();
        return FALSE;
    }
    /*
     * Add the viewport to the D3D device
     */
    rval = d3dapp->lpD3DDevice->AddViewport(lpD3DViewport);
    if (rval != D3D_OK) {
        Msg("Add D3D viewport failed.\n%s", D3DAppErrorToString(rval));
        CleanUpAndPostQuit();
        return FALSE;
    }
    /*
     * Setup the viewport for a reasonable viewing area
     */
    D3DVIEWPORT viewData;
    memset(&viewData, 0, sizeof(D3DVIEWPORT));
    viewData.dwSize = sizeof(D3DVIEWPORT);
    viewData.dwX = viewData.dwY = 0;
    viewData.dwWidth = w;
    viewData.dwHeight = h;
    viewData.dvScaleX = viewData.dwWidth / (float)2.0;
    viewData.dvScaleY = viewData.dwHeight / (float)2.0;
    viewData.dvMaxX = (float)D3DDivide(D3DVAL(viewData.dwWidth),
                                       D3DVAL(2 * viewData.dvScaleX));
    viewData.dvMaxY = (float)D3DDivide(D3DVAL(viewData.dwHeight),
                                       D3DVAL(2 * viewData.dvScaleY));
    rval = lpD3DViewport->SetViewport(&viewData);
    if (rval != D3D_OK) {
        Msg("SetViewport failed.\n%s", D3DAppErrorToString(rval));
        CleanUpAndPostQuit();
        return FALSE;
    }

    /*
     * Return the viewport to D3DApp so it can use it
     */
    *lplpViewport = lpD3DViewport;
    
    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * BeforeDeviceCreated --
 *     
 *     D3DApp will call this function before the current D3D 
 *     device is destroyed to give the app the opportunity to 
 *     destroy objects it has created with the DD or D3D objects.     	
 *
 * Results:
 *     Returns TRUE if successful, else FALSE.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static BOOL
BeforeDeviceDestroyed(LPVOID lpContext)
{
   
    /*
     * Since we created the viewport it is our responsibility to release
     * it.  Use D3DApp's pointer to it since we didn't save one.
     */
    d3dapp->lpD3DViewport->Release();

    return TRUE;
}

/*
 * AppPause
 * Pause and unpause the application
 */
static BOOL
AppPause(BOOL f)
{
    /*
     * Flip to the GDI surface and halt rendering
     */
    if (!D3DAppPause(f))
        return FALSE;
    
    return TRUE;
}



/*
 *--------------------------------------------------------------
 *
 * WindowProc --
 *     
 *     Main window message handler.  	
 *
 * Results:
 *     Does action in the window depending on the message.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

long
FAR PASCAL WindowProc(HWND hWnd, UINT message, WPARAM wParam,
                           LPARAM lParam )
{
    BOOL bStop;
    LRESULT lresult;
   
    /*
     * Give D3DApp an opportunity to process any messages it MUST see in order
     * to perform it's function.
     */
    if (!D3DAppWindowProc(&bStop, &lresult, hWnd, message, wParam, lParam)) {
        ReportD3DAppError();
        CleanUpAndPostQuit();
        return 0;
    }

    /*
     * Prevent resizing through this message
     */
    /*if (message == WM_GETMINMAXINFO && myglobs.bResizingDisabled && !d3dapp->bFullscreen && !d3dapp->bMinimized) {
        RECT rc;
        GetWindowRect(hWnd, &rc);
        ((LPMINMAXINFO)lParam)->ptMaxTrackSize.x = START_WIN_SIZE;
        ((LPMINMAXINFO)lParam)->ptMaxTrackSize.y = START_WIN_SIZE;
        ((LPMINMAXINFO)lParam)->ptMinTrackSize.x = START_WIN_SIZE;
        ((LPMINMAXINFO)lParam)->ptMinTrackSize.y = START_WIN_SIZE;
        return 0;
    }*/

    /* 
     * If bStop is set by D3DApp, the app should not process the message but
     * return lresult.
     */
    if (bStop)
        return lresult;

    switch( message ) {
        case WM_MOUSEMOVE:
        case WM_LBUTTONDOWN:
        case WM_LBUTTONUP:
        case WM_RBUTTONDOWN:
        case WM_RBUTTONUP:
        case WM_MBUTTONDOWN:
        case WM_MBUTTONUP:
            /*
             * Call the sample's MouseHandler if available
             */
            if (!MouseHandler)
                break;
            if ((MouseHandler)(message, wParam, lParam))
                return 1;
            break;
        case WM_KEYDOWN:
            /*
             * Call the sample's keyboard handler if available
             */
            if (!KeyboardHandler)
                break;
            if ((KeyboardHandler)(message, wParam, lParam))
                return 1;
            break;
        case WM_DESTROY:
            myglobs.hWndMain = NULL;
            CleanUpAndPostQuit();
            break;
        //case WM_GETMINMAXINFO:
            /*
             * Some samples don't like being resized, such as those which use
             * screen coordinates (TLVERTEXs).
             */
            /*if (myglobs.bResizingDisabled) {
                ((LPMINMAXINFO)lParam)->ptMaxTrackSize.x = START_WIN_SIZE;
                ((LPMINMAXINFO)lParam)->ptMaxTrackSize.y = START_WIN_SIZE;
                ((LPMINMAXINFO)lParam)->ptMinTrackSize.x = START_WIN_SIZE;
                ((LPMINMAXINFO)lParam)->ptMinTrackSize.y = START_WIN_SIZE;
                return 0;
            }
            break;*/
        case WM_COMMAND:
            /*
             * Whenever we receive a command in single step mode, draw a frame
             */
            if (myglobs.bSingleStepMode)
                myglobs.bDrawAFrame = TRUE;
            return 0L;
    }
    return DefWindowProc(hWnd, message, wParam, lParam);
}


/*
 *--------------------------------------------------------------
 *
 *    Setting up callbacks for Mouse and Keyboard interaction.        
 *
 *--------------------------------------------------------------
 */ 
/*
 * SetMouseCallback
 * Called in an example to set a callback function for all WM messages
 * dealing with the mouse.  The handler should return whether or not
 * it handled the message.
 */
BOOL
SetMouseCallback(BOOL(*Handler)(UINT, WPARAM, LPARAM))
{
    MouseHandler = Handler;
    return TRUE;
}
 
/*
 * SetKeyboardCallback
 * Called in an example to set a callback function for all WM messages
 * dealing with the keyboard. The handler should return whether or not it
 * handled the message.
 */
BOOL
SetKeyboardCallback(BOOL(*Handler)(UINT, WPARAM, LPARAM)) {
    
    KeyboardHandler = Handler;
    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 *    Initialization, error reporting and release functions.        
 *
 *--------------------------------------------------------------
 */

/*
 * Called once at program initialization to initialize global variables.
 */
static void
InitGlobals(void)
{
    d3dapp = NULL;
    memset(&myglobs.rstate, 0, sizeof(myglobs.rstate));
    memset(&myglobs, 0, sizeof(myglobs));
    myglobs.bClearsOn = TRUE;
    myglobs.bShowFrameRate = TRUE;
    myglobs.bShowInfo = TRUE;
    MouseHandler = NULL;
    KeyboardHandler = NULL;
}

/*
 * Release all D3D objects, post a quit message and set the bQuit flag
 */
void
CleanUpAndPostQuit(void)
{
    if (myglobs.bQuit)
        return;
    if (!D3DAppDestroy())
        ReportD3DAppError();
    myglobs.bQuit = TRUE;
    PostQuitMessage( 0 );
}

/*
 * Reports an error during a d3d app call.
 */
void
ReportD3DAppError(void)
{
    Msg("%s", D3DAppLastErrorString());
}

/* 
 * Message output for error notification.
 */
void __cdecl
Msg( LPSTR fmt, ... )
{
    char buff[256];

    wvsprintf(buff, fmt, (char *)(&fmt+1));
    lstrcat(buff, "\r\n");
    AppPause(TRUE);
    if (d3dapp && d3dapp->bFullscreen)
        SetWindowPos(myglobs.hWndMain, HWND_NOTOPMOST, 0, 0, 0, 0,
                     SWP_NOSIZE | SWP_NOMOVE);
    MessageBox( NULL, buff, "D3D Example Message", MB_OK );
    if (d3dapp && d3dapp->bFullscreen)
        SetWindowPos(myglobs.hWndMain, HWND_TOPMOST, 0, 0, 0, 0,
                     SWP_NOSIZE | SWP_NOMOVE);
    AppPause(FALSE);
}

