(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Dan Grossman                        *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

(* talout.ml
 * Useful functions for programs to write TAL modules out and call an
 * external verifier/assembler/linker.
 *
 *)

(* RLS 4/23/99 - 5/7/99: Added support for GNU linking under Linux. *)

let do_write_dasm_info = ref false;;
(* Set to true to produce an "annots" file. *)

type bintool = MS | TALC | GNU;;
(* NG: Microsoft is more stable so is default *)

type binformat = COFF | ELF;;

let object_file_suffix,library_suffix,default_executable,
   asm_bintool,link_bintool,objformat =
  match Sys.os_type with
    "Unix" -> ".o",".a","a.out",ref TALC,ref GNU,ref ELF
  | "Win32" -> ".obj",".lib","a.exe",ref MS, ref MS,ref COFF
  | _ -> Printf.eprintf "Unknown operating system\n"; exit 1
;;

let runtime =
  ref (try Sys.getenv "TALCLIB" with Not_found -> Filename.current_dir_name)
;;
let set_runtime s = runtime := s;;

(* Modified by Dan to avoid inserting duplicates *)
let includes = ref [];;
let add_include p = 
  if not (List.mem p (!includes))
  then includes := p :: !includes
  else ()

let ri_options () =
  let aux s p = " -I "^p^s in
  " --runtime "^(!runtime)^(List.fold_left aux "" !includes)
;;

let write_options = ref Talpp.std_options (* TEMPORARY; CYCLONE/MASM *)

let write_int modname filename talint =
  Format.set_margin 9999;
  Format.set_max_boxes 9999;
  let oc = open_out filename in
  Format.set_formatter_out_channel oc;
  Talpp.print_tal_int Format.std_formatter
    (!write_options) (* TEMPORARY; CYCLONE/MASM *)
    modname talint;
  Format.print_flush ();
  Format.set_formatter_out_channel stdout;
  close_out oc;
  Format.set_margin 79;
  ()
;;

let write_imp modname filename talimp =
  Format.set_margin 9999;  (* JGM: too large a value causes stack overflow *)
  let oc = open_out filename in
  Format.set_formatter_out_channel oc;
  Talpp.print_tal_imp Format.std_formatter
    (!write_options) (* TEMPORARY; CYCLONE/MASM *)
    modname talimp;
  Format.print_flush ();
  Format.set_formatter_out_channel stdout;
  close_out oc;
  Format.set_margin 79;
  ()
;;

 (* NG: for development purposes this is on by default, probably should be
  *     off for production use.
  *)
let verbose_sys_commands = ref true;;
let sys_command s =
  if !verbose_sys_commands then begin Printf.printf "%s\n" s; flush stdout end;
  (Sys.command s)=0
;;


let verify filename =
  let cmd = "talc.exe"^(ri_options ())^" --elaborate-only "^filename in
  sys_command cmd
;;

exception Exit;;

let do_asm doverify talfn objfile =
  try
    let cmd =
      match !asm_bintool with
      	MS ->
	  if doverify then
	    if not (verify talfn) then raise Exit;
	  (match !objformat with
	    COFF -> 
	      let aux s p = " /I"^p^s in
	      let includes = aux (List.fold_left aux "" !includes) !runtime in
	      let objfile =
	      	match objfile with None -> "" | Some fn -> " /Fo"^fn in
	      "ml /nologo /coff"^includes^" /Fl /Sc /c"^objfile^" /Ta"^talfn
	  | ELF ->
	      Printf.eprintf "%s: masm cannot output ELF\n" talfn; raise Exit)
      | TALC ->
	  let objfile = match objfile with None -> "" | Some fn -> " -o "^fn in
	  let fmt =
	    match !objformat with COFF -> " --coff" | ELF -> " --elf" in
	  let extra_options = 
	     (if !do_write_dasm_info then " --dasm-info" else "") in
	  "talc.exe"^(ri_options ())^extra_options^objfile^
	        " --TALC"^fmt^" -c "^talfn
      |	GNU ->
	   Printf.eprintf "%s: GNU assembler not supported\n" talfn;
	   raise Exit    
    in 
    sys_command cmd
  with
    Exit -> false
;;

let asm talfn objfile = do_asm false talfn objfile;;
let verify_and_asm talfn objfile = do_asm true talfn objfile;;


(* find_file_in_paths: Searches for fn in the given paths, in order, 
   and returns the path to fn. 
   Added by RLS to put real file paths into gcc command *)

let rec find_file_in_paths fn paths = 
   if (String.contains fn '/') then fn    
   else match paths with
      [] -> fn
    | (path :: tail) -> 
	 if (Sys.file_exists (Filename.concat path fn)) then
	    (Filename.concat path fn)
	 else 
	    (find_file_in_paths fn tail)
;;
    
let link objfiles exename =
  let cmd =
    match !link_bintool with
      MS ->
    	"link /nologo /subsystem:console /libpath:" ^ !runtime ^
	(List.fold_left (fun r d -> " /libpath:" ^d ^ " " ^ r) "" !includes) ^
    	" /out:" ^ exename ^ " tal_start" ^ object_file_suffix ^ " " ^
    	(List.fold_right (fun fn s -> fn^" "^s) objfiles "") ^
    	"tal_util" ^ object_file_suffix ^ " gc" ^ library_suffix
     | TALC -> failwith "Talout.link - TALC linker unimplemented"
     | GNU ->
	  let all_objfiles = (("tal_start"^object_file_suffix) :: 
	     ("tal_util"^object_file_suffix) ::
				objfiles) in
	  let paths = ("." :: !runtime :: !includes) in
	  "gcc -o " ^ exename ^ " " ^
	  (List.fold_right 
		(fun fn s -> (find_file_in_paths fn paths)^" "^s)
		all_objfiles "") ^
	  !runtime^"/gc"^library_suffix
  in
  sys_command cmd
;;
   
      

(* EOF: talout.ml *)
