
(* cyclone.ml
   Contains macros used by Cyclone. *)

open Numtypes;;
open Tal;;
open Identifier;;


(**********************************************
*                                             *
* These macros save and restore the registers *
*                                             *
* x4REGS treats regs EAX, EBX, ECX, and EDX   *
*                                             *
* x3REGS only treats regs EBX, ECX, and EDX   *
* (for when return value is stored in EAX)    *
*                                             *
***********************************************)

let push4regs = 
   [  Push (Reg Eax, []);
      Push (Reg Ebx, []);
      Push (Reg Ecx, []);
      Push (Reg Edx, [])  ]

and pop4regs = 
   [  Pop (Reg Edx);
      Pop (Reg Ecx);
      Pop (Reg Ebx);
      Pop (Reg Eax)  ]

and push3regs reg =
   match reg with 
      Eax ->
	 [  Push (Reg Ebx, []);
	    Push (Reg Ecx, []);
	    Push (Reg Edx, [])  ]
    | Ebx ->
	 [  Push (Reg Eax, []);
	    Push (Reg Ecx, []);
	    Push (Reg Edx, [])  ]
    | Ecx ->
	 [  Push (Reg Eax, []);
	    Push (Reg Ebx, []);
	    Push (Reg Edx, [])  ]
    | Edx ->
	 [  Push (Reg Eax, []);
	    Push (Reg Ebx, []);
	    Push (Reg Ecx, [])  ]
    | _ -> invalid_arg "push3regs"
;;

let pop3regs reg =
   match reg with 
      Eax ->
	 [  Pop (Reg Edx);
	    Pop (Reg Ecx);
	    Pop (Reg Ebx)  ]
    | Ebx ->
	 [  Pop (Reg Edx);
	    Pop (Reg Ecx);
	    Pop (Reg Eax)  ]
    | Ecx ->
	 [  Pop (Reg Edx);
	    Pop (Reg Ebx);
	    Pop (Reg Eax)  ]
    | Edx ->
	 [  Pop (Reg Ecx);
	    Pop (Reg Ebx);
	    Pop (Reg Eax)  ]
    | _ -> invalid_arg "pop3regs"
;;


(***** Label Arithmetic *****)

(* assert_template_labels 
   Should raise an error if any label in the list is _not_ in the template
   section. For now, does nothing. *)

let assert_template_labels env label_list =
   () (**)(* Unimplemented *)
;;


let cg_start_label = (id_of_string "_CG_start");;
let cg_dump_label = (id_of_string "_CG_dump");;
let cg_abort_label = (id_of_string "_CG_abort");;
let cg_end_label = (id_of_string "_CG_end");;
let cg_mark_label = (id_of_string "_CG_mark");;
let external_addr = (id_of_string "_external_addr");;

let cyclone_labels = [ (cg_start_label, cempty);
			(cg_dump_label, cempty);
			(cg_abort_label, cempty);
			(cg_end_label, cempty);
			(cg_mark_label, cempty);
			(external_addr, cempty) ]
;;


(***** Implicit labels *****)

(* cg_get_implicit_label
   If there is an implicit label (i.e. a hole label) in the given
   instruction, returns Some l where l is the label in the instruction.
   Otherwise, returns None. *)

let cg_get_implicit_label instr = 
   match instr with
    | CgHole (reg, labeloftmpl, labelofhole) ->
	 Some labelofhole 
    | CgHoleBtagi (reg, tag, tmpllab, (holelab,crc), cc) ->
	 Some holelab 
    | CgHoleBtagvar (reg, sum, tag, tmpllab, (holelab,crc), cc) ->
	 Some holelab 
    | CgHoleJmp (tmpllab, (holelab, crc)) ->
	 Some holelab 
    | CgHoleJcc (cc, tmpllab, (holelab,crc)) ->
	 Some holelab 
    | _ -> None
;;



(***** Macros *****)


(* CGSTART type			      
   Begin a new code generation region. *)

let cgstart_macro arg = 
   push4regs @ 
   [  Mov (Reg Eax, (Addr cg_start_label, []));
      Call (Reg Eax, [])  ]
   @ pop4regs
;;

(* CGDUMP tv, reg, lab				   
 						   
  Dump the template at label lab into the current  
  code generation region.  A pointer to the first  
  instruction of the copied template is put into   
  register reg.	 The type variable tv is used	   
  by the verifier; it is a BINDING OCCURRENCE.	  *)

let cgdump_macro tv reg lab = 
   (push3regs reg) @
   [  Push (Addr lab, []);
      Mov (Reg Eax, (Addr cg_dump_label, []));
      Call (Reg Eax, []);
      ArithBin (Add, (Reg Esp), (Immed i32_4))  ]
   @ (if reg != Eax then [Mov (Reg reg, (Reg Eax, []))] else [])
   @ (pop3regs reg)
;;
      


(* CGFILL (env) reg1, labeloftmpl, labelofhole, reg2			  
 								  
  reg1 is a register pointing into the code being generated,	  
  labeloftmpl is the label at the start of the template,	  
  labelofhole is the label just before the hole,		  
  reg2 is a register containing the value to stuff into the hole *)
      
let cgfill_macro reg1 tmpl_pos hole_pos reg2 =
   let disp = (hole_pos -$ tmpl_pos -$ i32_3) in
   [  Mov (Prjr ((reg1, []), disp), (Reg reg2, [])) ] 
   @ (push3regs reg2) @
   [  Push (Reg reg2, []);
      Mov (Reg Eax, (Addr cg_mark_label, []));
      Call (Reg Eax, []);
      Pop (Reg reg2)  ]
   @ (pop3regs reg2)
;;




(* CGFILLBTAG (env) holereg, htmplab, holelab, targreg, ttmplab, targlab
  (used for BTAGI and BTAGVAR)					            
 								            
  holereg is a register pointing to the generated hole code                 
  htmplab is the label of the hole template                                 
  holelab is the label of the hole                                          
  targreg is a register pointing to the generated target code               
  ttmplab is the label of the target template                               
  targlab is the label of the target                                        
 								            
   | BTAGI ... target |                                                     
   --------------------                                                     
                                                                            
   --------------------                                                     
   | target ...       |                                                     
                                                                            
  The location of the hole is calculated as follows:                        
    holereg + (holelab + 3 + 2 - htmplab - 4)                               
                                                                            
  The values 3 and 2 are added in due to the CMP and JMP instructions.      
  The 4 is subtracted because of the template length.  The final            
  expressions is:                                                           
    holereg + (holelab - htmplab + 1)                                       
                                                                            
  The target address is calculated as follows:                              
   (targreg-holereg)+(targlab-targtmpllab-4)-(holelab-holetmpllab-4+3+2+4)  
                                                                            
  4 is subtracted from each template due to template length.  Adding        
  3, 2, and 4 is due to the CMP instruction, JMP opcode, and JMP            
  address.  The final expression is:                                        
    (targreg-holereg)+(targlab-targtmpllab)-(holelab-holetmpllab)-9        *)


let cgfillbtag_macro holereg htmp_pos hole_pos targreg ttmp_pos targ_pos =
   [  Push (Reg targreg, []); 
      ArithBin (Sub, Reg targreg, Reg holereg); 
      ArithBin (Add, Reg targreg, 
		  Immed ((targ_pos -$ ttmp_pos)
			      -$ (hole_pos -$ htmp_pos) -$ (int_to_int32 9)));
      Mov (Prjr ((holereg,[]), (hole_pos -$ htmp_pos +$ i32_1)),
	     (Reg targreg,[]));
      Pop (Reg targreg)
   ]
;;


(* CGFILLJMP holereg, htmplab, holelab, targreg, ttmplab, targlab           
 								           
  This is the same as CGITBTAGI, except a jmp instr filled in.             
                                                                           
                                                                           
  The location of the hole is calculated as follows:                       
   holereg+(holelab+1-htmplab-4)=holereg+(holelab-htmplab-3)               
  1 is added for the JMP opcode, 4 is subtracted due to template length:   
                                                                           
  The target address is calculated as follows:                             
   (targreg-holereg)+(targlab-targtmpllab-4)-(holelab-holetmpllab-4+1+4)=  
   (targreg-holereg)+(targlab-targtmpllab)-(holelab-holetmpllab)-5         
                                                                           
  4 is subtracted from each template due to template length.  Adding 1     
  and and 4 are due to the JMP opcode and address.                       *)  
                                                                         

let cgfilljmp_macro holereg htmp_pos hole_pos targreg ttmp_pos targ_pos =
   [  Push (Reg targreg, []);
      ArithBin (Sub, Reg targreg, Reg holereg); 
      ArithBin (Add, Reg targreg, 
		  Immed ((targ_pos -$ ttmp_pos) 
			      -$ (hole_pos -$ htmp_pos) -$ (int_to_int32 5)));
      Mov (Prjr ((holereg,[]),(hole_pos -$ htmp_pos -$ i32_3)), 
	     (Reg targreg,[]));
      Pop (Reg targreg)
   ]
;;
      

(*
* CGFILLJCC holereg, htmplab, holelab, targreg, ttmplab, targlab          * 
*								          * 
* This is the same as CGITBTAGI, except a jcc instr filled in.            * 
*                                                                         * 
* The location of the hole is calculated as follows:                      * 
*  holereg+(holelab+2-htmplab-4)=holereg+(holelab-htmplab-2)              * 
* 2 is added for the JCC opcode, 4 is subtracted due to template length:  * 
*                                                                         * 
* The target address is calculated as follows:                            *
*  (targreg-holereg)+(targlab-targtmpllab-4)-(holelab-holetmpllab-4+2+4)= *
*  (targreg-holereg)+(targlab-targtmpllab)-(holelab-holetmpllab)-6        * 
*                                                                         * 
* 4 is subtracted from each template due to template length.  Adding 2    * 
* and 4 are due to the JMP opcode and address.                            *) 

let cgfilljcc_macro holereg htmp_pos hole_pos targreg ttmp_pos targ_pos =
   [  Push (Reg targreg, []);
      ArithBin (Sub, Reg targreg, Reg holereg); 
      ArithBin (Add, Reg targreg, 
		  Immed ((targ_pos -$ ttmp_pos)
			      -$ (hole_pos -$ htmp_pos) -$ (int_to_int32 6)));
      Mov (Prjr ((holereg,[]),(hole_pos -$ htmp_pos -$ i32_2)),
	     (Reg targreg,[]));
      Pop (Reg targreg)
   ]
;;


(*			   *
* Trevor's original guess. *
*			   *)

let cgfillrel_macro reg1 tmpl_pos hole_pos reg2 tmpl2_pos targ_pos =
   [  Push (Reg reg2, []);
      Lea (reg2, Prjr ((reg2,[]),(targ_pos -$ tmpl2_pos -$ i32_4)));
      Mov (Prjr ((reg1,[]),(hole_pos -$ tmpl_pos +$ i32_1)), (Reg reg2,[]));
      Pop (Reg reg2)
   ]
;;



(* CGFORGET tv			*
*				*
* Needed for the verifier only. *)

let cgforget_macro tv = [];;


(* CGABORT val				      *
*					      * 
* Abort the most recent val code generations. *)


let cgabort_macro i = 
   push4regs @
   [  Push (Immed i, []);
      Mov (Reg Eax, (Addr cg_abort_label, []));
      Call (Reg Eax, []);
      ArithBin (Add, Reg Esp, Immed i32_4)  ]
   @ pop4regs
;;
     

(* CGEND reg					     *
*						     * 
* Complete the most recent code generation and leave * 
* a pointer to the new function in register reg.     *) 

let cgend_macro reg =
   (push3regs reg) @
   [  Mov (Reg Eax, (Addr cg_end_label, []));
      Call (Reg Eax, [])  ]
   @ (if reg <> Eax then [Mov (Reg reg, (Reg Eax,[]))] else [])
   @ (pop3regs reg)
;;
   
      
(* CGHOLE reg, labeloftmpl, labelofhole				* 
*								* 
* Declares a new hole labelofhole in the template whose first	* 
* instruction has label labeloftmpl.  The instruction will move	*  
* the contents of the hole into register reg (when the hole has * 
* been filled.							* 
*								* 
* labelhole is a BINDING OCCURRENCE.				*)

let cghole_macro reg labeloftmpl labelofhole =
   (**) (* Bind labelofhole to this instruction. *)
   [ Mov (Reg reg, (Immed i32_0,[])) ]
;;


(* CGHOLEBTAGI                                                   *
*								* 
* Declares a new hole holelab in the template, which consists   * 
* of a BTAGI instruction with the label holelab.  An external   * 
* address is used to create a four byte hole.  The label        * 
* tmpllab is used by the verifier, but not used in the macro.	*)

let cgholebtagi_macro reg tag tmpllab holelab cc = 
   (**)(* Bind holelab to this instr. *)
   [ Btagi (reg, tag, (external_addr,[]), cc) ]
;;



(* CGHOLEBTAGVAR                                                 *
* Same as CGHOLEBTAGI, except for a BTAGVAR instruction         *)

let cgholebtagvar_macro reg sum tag tmpllab holelab cc =
   (**)(* Bind holelab to this instr *)
   [ Btagvar (reg, sum, tag, (external_addr,[]), cc) ]
;;



(* CGHOLEJMP                                                     *
* Same as CGHOLEBTAGI, except for a JMP instruction             *)

let cgholejmp_macro tmpllab holelab =
   (**)(* Bind holelab to this instr. *)
   [ Jmp (Addr external_addr, []) ]
;;


(* CGHOLEJCC                                                     *
* Same as CGHOLEBTAGI, except for a JCC instruction             *)

let cgholejcc_macro cc tmpllab holelab =
   (**)(* Bind holelab to this instr *)
   [ Jcc (cc, (external_addr,[])) ]
;;










