/* Standard Library for TAL Code
 *
 * Copyright Greg Morrisett, Neal Glew, Dan Grossman
 *   January 1999, all rights reserved
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

void *GC_malloc(int size);

typedef struct {int size; char chars[1];} *string;


/***************************** I/O *******************************/

FILE *tal_stdin = stdin;
FILE *tal_stdout = stdout;
FILE *tal_stderr = stderr;

static void check_fd(FILE *fd) {
  if(!fd) {
    fprintf(stderr,"Attempt to print to null file descriptor.\n");
    exit(255);
  }
}

void fprint_int    (FILE *fd, int x)    { 
  check_fd(fd);
  if(fprintf(fd, "%d", x) < 0)
    fprintf(stderr,"File write failed (non-fatal)\n");
}
void fprint_char   (FILE *fd, char c)   { 
  check_fd(fd);
  if (putc(c, fd) == EOF)
    fprintf(stderr,"File write failed (non-fatal)\n");
}
void fprint_string (FILE *fd, string s) { 
  check_fd(fd);
  if (fwrite(s->chars, 1, s->size, fd) < s->size) 
    fprintf(stderr,"File write failed (non-fatal)\n");
}
void fprint_newline(FILE *fd)           { 
  check_fd(fd);
  if (putc('\n', fd) == EOF)
    fprintf(stderr,"File write failed (non-fatal)\n");
}

void pop_debug_never_null(int i,string s) {
  fprintf(stderr,"Null pointer exception: ");
  fprint_string(tal_stderr,s);
  fprintf(stderr,", char ");
  fprint_int(tal_stderr,i);
  exit(255);
}
/***************************** Timing *******************************/

/* This is ad hoc and doesn't work under Unix, but for now...
 */

#ifdef __linux__
#  define _ftime ftime
#  define _timeb timeb
#  include <sys/timeb.h>
#else

typedef long time_t;
struct _timeb {
        time_t time;
        unsigned short millitm;
        short timezone;
        short dstflag;
        };

#endif

int int_time()
{
  struct _timeb tstr;
  long result;

  /* places secs in "time field", msecs in "millitm" field */
  _ftime( &tstr );
  
  /* get secs */
  result = tstr.time;

  /* free up top ten 10 bits (to avoid overflow of following multiplication) */
  result = result & 0x003fffff;

  /* convert from secs to msecs (now uses all 32 bits) */
  result = result * 1000;
  
  /* add in msecs */
  result = result + tstr.millitm;

  return result;
}

/***************************** Temporary *******************************/

/* This stuff should really go into a standard library or utility file
 * somewhere else.  But its here for now.
 */

void print_time(int start_time, int end_time, int iterations)
{
  printf("Time: %d msec\n", (end_time - start_time) / iterations);
}

void get_input(string out)
{
  char c;

  fprint_string(tal_stdout, out);
  printf("(Hit <RETURN> to continue, 'q' + <RETURN> to quit)");
  fflush(stdout);
  c = getchar();
  if (c == 'q')
    exit(0);
}

/* EOF: stdlib.c */
