(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel, Frederick Smith   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

(* Reason for inexhaustive match in tcExp is because it doesn't have a
   UnionMember case.  This is because things are parsed as StructMember
   and converted in this function.
 *)

(* Stopped in the middle:
 *  check_* need to make sure their definitions are consistent with
 *  global_env
 *)

(* Breaking news:
 *
 * Open and prefix provide crude mechanisms for encapsulation.  The names in
 * TypeName are refs so that they can be updated with their
 * fully qualified path, once that is known.
 *
 * Type checking proceeds as follows:
 *   1. Eliminate Prefixes by embedding them in the names.
 *   2. Blindly build an initial global environment from the top decls.
 *   3. Check that all types mentioned in global_env are well-formed, and
 *      expand their names to be fully qualified.
 *   4. Now type check the function bodies, and the initializers for global
 *      variables.
 *
 * The following represent unique namespaces:
 *   1. Unions/Structures/Abstracts
 *   2. Globals/Exception
 * No duplicates are allowed within one namespace unless they agree.
 * We always keep the declaration with the most information.
 * 
 * Still need to verify that values do not have types that are not
 * exported (i.e., types that are not static.)
 *)

let debug = ref false

open Numtypes
open Popsyntax

exception Unimplemented
exception UnboundVar of (var*location)

type global_env = { structs : (type_name,structdecl) Dict.dict;
		    unions  : (type_name,uniondecl)  Dict.dict;
		    abstracts : (type_name,var list * bool) Dict.dict;
		    globals : (var,typ) Dict.dict;
		    exceptions : (var,typ) Dict.dict;
		    open_typs : (type_name,type_name) Dict.dict;
		    open_vals : (var,var) Dict.dict
		  }

(* Environment of a function *)
type f_env =
    { f_name : var;
      f_tyvars : var list;
      f_args : (var * typ) list;
      f_ret_type : typ;
      f_locals : (var * typ) list;
      f_inloop : bool;
      f_labels : var list;
    }

let deSome opt = (match opt with None -> raise Not_found | Some v -> v);;

let set_inloop fenv  = { fenv with f_inloop = true                 };;
let add_var x t fenv = { fenv with f_locals = (x,t)::fenv.f_locals };;
let add_label x fenv = { fenv with f_labels =     x::fenv.f_labels };;

let fndecl2f_env fd=
  { f_name = fd.fn_name;
    f_tyvars = fd.fn_tyvars;
    f_args = fd.fn_args;
    f_ret_type = fd.fn_ret_type;
    f_locals = fd.fn_args;
    f_inloop = false;
    f_labels = []
  } 

(* DEBUG *)
let dummy_location = Gcdfec.seg_of_abs 42 42
let make_exp  re   = { exp_typ = None; raw_exp = re; exp_loc = dummy_location}
let make_stmt rs   = { raw_stmt = rs; stmt_loc = dummy_location}
let make_call name args =
  make_exp(FunCall(make_exp(Var name), ref (Some []), args))
let make_call_stmt name args =
  make_stmt(Exp(make_call name args))
let debug_push name = 
  make_call_stmt "callStackPush" [make_exp(Const (String name))]
let debug_pop () = 
  make_call_stmt "callStackPop"  []
let debug_dump() =
  make_call_stmt "callStackDump" []

(* END DEBUG *)

exception Unbound

(* Takes a name and a function defined : var -> bool.
   Returns the fully qualified name.
*)
let complete_typ_name global_env defined n =
  begin try
    if defined n then n else Dict.lookup global_env.open_typs n
  with Dict.Absent -> raise Unbound
  end

let complete_val_name global_env defined n =
  begin try
    if defined n then n else Dict.lookup global_env.open_vals n
  with Dict.Absent -> raise Unbound
  end
  
let open_prefix global_env p =
  begin
    let pLen = String.length p in
    let has_prefix n = 
      if String.length n <= pLen then None
      else 
	let rec aux i = 
	  (* foo?bar foo is the prefix.
	     When opened we get bar not ?bar so we need to add 1. *)
	  if i=pLen then Some (String.sub n (i+1) (String.length n - pLen - 1))
	  else if n.[i] <> p.[i] then None else aux (i+1)
	in
	aux 0
    in
    let process_dict opened d =
      let process_elt id b opened =
	match has_prefix id with
	  None -> opened
	| Some id' -> (Dict.insert opened id' id)
      in
      Dict.fold_dict process_elt d opened
    in
    let process_opened opened od =
      let process_elt id id' opened =
	(* id' is the fully-qualified name of this thing.  It just happens to
	   have been opened twice. *)
	match has_prefix id with
	  None -> opened
	| Some id'' -> (Dict.insert opened id'' id')
      in
      Dict.fold_dict process_elt od opened
    in
    let open_typs = process_opened global_env.open_typs global_env.open_typs in
    let open_vals = process_opened global_env.open_vals global_env.open_vals in
    let open_typs = process_dict open_typs global_env.structs in
    let open_typs = process_dict open_typs global_env.unions in
    let open_typs = process_dict open_typs global_env.abstracts in
    let open_vals = process_dict open_vals global_env.globals in
    let open_vals = process_dict open_vals global_env.exceptions in
    { global_env with open_typs = open_typs; open_vals = open_vals }
  end

let inloop f_env  = f_env.f_inloop;;
let retType f_env = f_env.f_ret_type;;
let terr loc s =
    let e = Poperr.Etypecheck (Poperr.TypeError s) in
    let e = Gcdfec.mk_err_elab loc (Poperr.error_message e) in
    Gcdfec.post_error e
let tyvars fenv = fenv.f_tyvars
let mk_env fenv = fenv
let lookup fenv id = List.assoc id fenv.f_locals

(* Cyclone *)

(****************************************************************)
(* We define Cyclone environments, and redefine functions that  *)
(* depend on the representation of environments.                *)
(*                                                              *)
(* A version of poptype without Cyclone support can be          *)
(* recovered simply by commenting out this code, and subsequent *)
(* code marked CYCLONE.                                         *)
(*                                                              *)
(* VARIABLE CONVENTIONS: fenv is used to name f_env's, cenv is  *)
(* used to name c_env's, and env is used where it could be      *)
(* either (depending on whether the Cyclone code is commented   *)
(* out).  Global environments are always named by global_env.   *)
(****************************************************************)
type c_env =
    Outermost of f_env
  | Frame of f_env * c_env
  | Hidden of f_env * c_env

let rec fenv_of_cenv cenv =
  match cenv with
    Outermost fenv -> fenv
  | Frame(fenv,_) -> fenv
  | Hidden(_,cenv) -> fenv_of_cenv cenv

let rec put_fenv cenv fenv =
  match cenv with
    Outermost _ -> Outermost fenv
  | Frame(_,cenv) -> Frame(fenv,cenv)
  | Hidden(fenv',cenv) -> Hidden(fenv',put_fenv cenv fenv)

let set_inloop cenv =
  put_fenv cenv (set_inloop(fenv_of_cenv cenv))

let add_var x t cenv =
  put_fenv cenv (add_var x t (fenv_of_cenv cenv))

let add_label x cenv =
  put_fenv cenv (add_label x (fenv_of_cenv cenv))

let label_bound x cenv =
  (List.mem x (fenv_of_cenv cenv).f_labels)

let lookup cenv id =
  (********************************************************************)
  (* Special case for Cyclone: if we are looking up a function, we    *)
  (* need to look in the frame.  All Popcorn functions are outermost, *)
  (* and their types are given in the global environment as well as   *)
  (* the frame.  In Cyclone, an inner function won't be in the        *)
  (* global environment.                                              *)
  (********************************************************************)
  let fenv = fenv_of_cenv cenv in
  if fenv.f_name = id then
    FnType(fenv.f_tyvars,fenv.f_ret_type,List.map snd fenv.f_args)
  else lookup fenv id

let inloop cenv = inloop (fenv_of_cenv cenv)

let tyvars cenv = tyvars (fenv_of_cenv cenv)

let retType cenv = retType (fenv_of_cenv cenv)

let mk_env fenv = Outermost fenv

let set_outloop fenv = { fenv with f_inloop = false }

let set_outloop cenv =
  put_fenv cenv (set_outloop(fenv_of_cenv cenv))

(* End Cyclone *)

let rec list_match test t1 t2 =
  match t1,t2 with
    [],[] -> true
  | [],_ | _,[] -> false
  | hd1::_,hd2::_ when not (test hd1 hd2) -> false
  | hd1::tl1,hd2::tl2 -> list_match test tl1 tl2
;;

let scope_match s1 s2 =
  match s1,s2 with
    Static,Static | Public,Public | Extern,Extern | Abstract,Abstract
  | Extern,Public | Public,Extern -> true
  | _ -> false
;;

(* returns true if the type constructor n is a ? type *)
let possibly_null global_env n =
  (try (Dict.lookup global_env.structs n).st_possibly_null with
    Dict.Absent ->
      (try snd (Dict.lookup global_env.abstracts n) with
	Dict.Absent -> false))


(* compute the size in words of a type. *)
let rec size_of_typ global_env t =
  begin match t with
    VoidType -> 0
  | Evar(c,tor) ->
      begin match c with
	Any ->
	  begin match !tor with
	    Some t -> size_of_typ global_env t
	  | None -> -1
	  end
      |	Byte4 | Option -> 4
      end
  | _ -> 4
  end 

let rec check_arraysize_expression loc e =
  (* Check that an expression is a "constant-expression" suitable for *)
  (* an array size in a type or declaration. *)
  (* We should also check that it has type int!! Sometimes but not
     always this is done elsewhere... *)
  let err s = terr loc s in
  let raise_err s = err s; raise Gcdfec.Exit in
  match e.raw_exp with
    Const(Int _) -> ()
  | Primop(p,[e1;e2]) ->
      begin
        check_arraysize_expression loc e1;
        check_arraysize_expression loc e2;
        match p with
          Plus | Times | Minus | Div | Mod | Bitand | Bitor | Bitxor
        | Bitlshift | Bitlrshift | Bitarshift -> ()
        | _ -> raise_err "non-constant-expression in type"
      end
  | _ -> raise_err "non-constant-expression in type"

(* checks that t is a valid type in the current global environment.
 * The set of VarType's must be drawn from tyvars.
 *)
let rec check_valid_type loc global_env (tyvars:var list) t =
  let err s = terr loc s in
  let raise_err s = err s; raise Gcdfec.Exit in
  match t with
    ArrayType(t,None) ->
      check_valid_type loc global_env tyvars t
  | ArrayType(t,Some e) ->
      check_valid_type loc global_env tyvars t;
      check_arraysize_expression loc e
  | FnType(vs,t,ts) -> check_valid_types loc global_env (vs@tyvars) (t::ts)
  | NamedType(n,ts) ->
      check_valid_types loc global_env tyvars ts;
      let defined name =
	(Dict.member global_env.structs name ||
	 Dict.member global_env.unions name ||
	 Dict.member global_env.abstracts name)
      in
      n :=
	 begin try
	   complete_typ_name global_env defined !n
	 with Unbound -> raise_err ("bad type "^ !n)
	 end;
      let vs =
      	(try (Dict.lookup global_env.structs !n).st_tyvars with
	  Dict.Absent ->
	    (try (Dict.lookup global_env.unions !n).un_tyvars with
	      Dict.Absent ->
		(try fst (Dict.lookup global_env.abstracts !n) with
		  Dict.Absent -> raise_err ("bad type "^ !n))))
      in if (List.length vs) = (List.length ts) then ()
      else raise_err ("wrong # of type arguments for "^ !n)
  (* Evars should only occur as pointers to other actual
   * types -- in fact, they shouldn't occur at all but what the hell.
   *)
  | Evar(_,r) ->
      (match !r with
	Some t -> check_valid_type loc global_env tyvars t
      |	_ -> raise_err "free existentially-bound variable!")
  | VarType v ->
      if List.mem v tyvars then () else
      raise_err ("free type variable #"^v)
  | TupleType ts -> 
      check_valid_types loc global_env tyvars ts
  | _ -> ()
and check_valid_types loc global_env (tyvars:var list) ts =
  List.iter (check_valid_type loc global_env tyvars) ts

let check_fundecl_valid_type global_env loc fd =
  begin
    check_valid_type loc global_env fd.fn_tyvars fd.fn_ret_type;
    List.iter (fun (_,t) -> check_valid_type loc global_env fd.fn_tyvars t)
      fd.fn_args;
    FnType(fd.fn_tyvars,fd.fn_ret_type,List.map snd fd.fn_args)
  end

(*************************************************************************)
(* Check a statement to make sure it does a return rather than fall off  *)
(* the end without doing a return. This is a conservative approximation: *)
(* doesRet returns false for some statements that always do a return.    *)
(*                                                                       *)
(*   EXAMPLE:                              WE RETURN:      CONSERVATIVE? *)
(*   { return(5); 0; }                     true            no            *)
(*   while (1==2) return(5);               false           no            *)
(*   while (1<2) return(5);                false           yes           *)
(*   codegen(int f()                                                     *)
(*           { cut return(5); });          false           yes           *)
(*                                                                       *)
(*************************************************************************)
let rec doesRet {raw_stmt=s} =
  match s with
    Skip -> false
  | Exp e ->
      begin match e.raw_exp with
	Raise _ -> true
      |	_ -> false
      end
  | Seq(s1,s2) -> doesRet s1 or doesRet s2
  | Return _ -> true
  | IfThenElse(_,s1,s2) -> doesRet s1 & doesRet s2
  | While(e,s) -> false (* s may never execute *)
  | Break _ -> false
  | Continue _ -> true (* FMS : Changed to true. *)
  | For(e1,e2,e3,s) -> false (* s may never execute *)
  | IntSwitch(_,ss,s) ->
      (List.for_all (fun (_,s) -> doesRet s) ss) & (doesRet s)
  | CharSwitch(_,ss,s) ->
      (List.for_all (fun (_,s) -> doesRet s) ss) & (doesRet s)
  | UnionSwitch(_,ss,sopt) ->
      (List.for_all (fun a -> doesRet a.arm_body) ss) &
      (match sopt with None -> true | Some s -> doesRet s)
  | ExnSwitch(_,ss,sopt) ->
      (List.for_all (fun a -> doesRet a.arm_body) ss) &
      (match sopt with None -> true | Some s -> doesRet s)      
  | Decl(_,_,_,s) -> doesRet s
  | Label(_,s) -> doesRet s
  | Do(s,e) -> doesRet s
  | Try(s1,_,s2) -> doesRet s1 & doesRet s2
(* Cyclone *)
  | Cut s -> doesSpliceRet s
  | Splice s -> false (* this will be a type error *)
(**********************************************************************)
(* Check a statement to be sure that it does a splice followed by a   *)
(* return.  This is needed for the codegen, cut, splice, and fill     *)
(* special forms.  Like doesRet, this is a conservative analysis.     *)
(**********************************************************************)
and doesSpliceRet {raw_stmt=s} =
  match s with
    Skip -> false
  | Exp _ -> false
  | Seq(s1,s2) -> doesSpliceRet s1 or doesSpliceRet s2
  | Return _ -> false (* return in the wrong context *)
  | IfThenElse(_,s1,s2) -> doesSpliceRet s1 & doesSpliceRet s2
  | While(e,s) -> false (* s may never execute *)
  | Break _ -> false
  | Continue _ -> false
  | For(e1,e2,e3,s) -> false (* s may never execute *)
  | IntSwitch(_,ss,s) ->
      (List.for_all (fun (_,s) -> doesSpliceRet s) ss) & (doesSpliceRet s)
  | CharSwitch(_,ss,s) ->
      (List.for_all (fun (_,s) -> doesSpliceRet s) ss) & (doesSpliceRet s)
  | UnionSwitch(_,ss,sopt) ->
      (List.for_all (fun a -> doesSpliceRet a.arm_body) ss) &
      (match sopt with None -> true | Some s -> doesSpliceRet s)
  | ExnSwitch(_,ss,sopt) ->
      (List.for_all (fun a -> doesSpliceRet a.arm_body) ss) &
      (match sopt with None -> true | Some s -> doesSpliceRet s)
  | Decl(_,_,_,s) -> doesSpliceRet s
  | Label(_,s) -> doesSpliceRet s
  | Cut s -> false (* this will be a type error *)
  | Splice s -> doesRet s
  | Do(s,e) -> doesSpliceRet s
  | Try(s1,_,s2) -> doesSpliceRet s1 & doesSpliceRet s2
(* End Cyclone *)
let rec subst inst t =
  match t with
    VarType v ->
      (try List.assoc v inst with Not_found -> t)
  | ArrayType(t,iopt) -> ArrayType(subst inst t,iopt)
  | FnType(vs,t,ts) ->
      let inst = (List.map (fun v -> (v,VarType v)) vs) @ inst in
      FnType(vs,subst inst t,substs inst ts)
  | TupleType ts -> TupleType(substs inst ts)
  | NamedType(n,ts) -> NamedType(n,substs inst ts)
  | _ -> t
and substs inst ts = List.map (subst inst) ts

(* Do path compression on Evar's so that they point directly to the 
 * defined type (if any).
 *)
let rec compress t =
  match t with
    Evar (_,r) ->
      (match !r with
	None -> t
      | Some t' ->
	  (compress t'; r := Some t'; t'))
  | _ -> t

(* Returns true if t1 and t2 unify, false otherwise.  Is destructive so
 * not suitable for backtracking.
 *)
exception Unify
let unify global_env t1 t2 =
  (* check to see if an Evar occurs in a type, raising Unify if so. *)
  let rec occurs r t =
    match t with
      Evar (_,r') -> if r = r' then raise Unify else ()
    | ArrayType(t,_) -> occurs r t
    | FnType(_,t,ts) -> occurslist r (t::ts)
    | TupleType ts -> occurslist r ts
    | NamedType(_,ts) -> occurslist r ts
    | _ -> ()
  and occurslist r ts =
    match ts with
      [] -> ()
    | t::ts -> occurs r t; occurslist r ts in
  (* inner loop of unify -- raises Unify if the two types do not unify. *)
  let rec un t1 t2 =
    (* compress the types to get rid of any indirection *)
    let t1 = compress t1 in
    let t2 = compress t2 in
    (* check for structural/physical equality.  This is the common case and
     * in particular, succeeds when we have the same refs for Evars.
     *)
    if t1 = t2 then () else
    match t1,t2 with
      Evar(c1,r1),_ ->
	begin match !r1 with
	  (* this case shouldn't really happen due to compression *)
	  Some t1' -> un t1' t2
	  (* as long as r1 doesn't occur in t2, set r1 to t2 *)
	| None -> 
	    begin match c1 with
	      Any -> occurs r1 t2; r1 := (Some t2)
	    | Byte4 -> occurs r1 t2; 
		if size_of_typ global_env t2 <> 4 
		then raise Unify;
		r1 := (Some t2)
	    | Option -> 
		begin match t2 with
		  Evar(Option,_) -> r1 := (Some t2)
		| NamedType(tn,ts) ->
		    if possibly_null global_env !tn then r1 := (Some t2)
		    else raise Unify
		| _ -> raise Unify
		end
	    end
	end
    | _,Evar _ -> un t2 t1
    | ArrayType(t1,_),ArrayType(t2,_) -> un t1 t2
    | FnType([],t1,ts1),FnType([],t2,ts2) -> uns (t1::ts1) (t2::ts2)
    | FnType(vs1,t1,ts1),FnType(vs2,t2,ts2) ->
	let inst = 
	  try List.combine vs1 (List.map (fun v -> VarType v) vs2) 
	with Invalid_argument _-> raise Unify in
	uns (t1::ts1) (substs inst (t2::ts2))
    | TupleType(ts1),TupleType(ts2) -> uns ts1 ts2
    | NamedType(tn1,ts1),NamedType(tn2,ts2) ->
	if !tn1 = !tn2 then uns ts1 ts2 else raise Unify
    | VarType(x),VarType(y) -> if x = y then () else raise Unify
    | _,_ -> raise Unify
  and uns ts1 ts2 =
    match ts1, ts2 with
      [],[] -> ()
    | t1::ts1,t2::ts2 -> un t1 t2; uns ts1 ts2
    | _,_ -> raise Unify
  in try un t1 t2; true with Unify -> false
;;

(* Insert a cast. *)
let coerce e t =
  let e_new = { exp_typ = e.exp_typ; 
		raw_exp = e.raw_exp; 
		exp_loc = e.exp_loc } 
  in
  e.raw_exp <- Cast(t,e_new);
  e.exp_typ <- Some t;
  ()

let coercable t =
  match t with
    IntType _ | CharType -> true
  | _                    -> false
;;

(* unify two the types of two expressions if possible.  
   If not attempt to coerce one to the other. 
   We assume the expressions already have types on them. 
*)
let unify_coerce2 global_env e1 e2 =
  let t1 = compress (deSome e1.exp_typ) in
  let t2 = compress (deSome e2.exp_typ) in
  let def_coerce e = coerce e (IntType(true,B4)) in
  if (unify global_env t1 t2) then true
  else
    begin
      match t1,t2 with
      |	IntType(false,B4),IntType(true ,B4) -> coerce e2 (IntType(false,B4));
	  true
      |	IntType(true ,B4),IntType(false,B4) -> coerce e1 (IntType(false,B4));
	  true
      |	IntType(b1,s1)   ,IntType(b2,s2)    ->
	  (* Wow C made this really easy.
	     Everything becomes an int *)
	  if s1 <> B4 then def_coerce e1;
	  if s2 <> B4 then def_coerce e2;	  
	  true
      |	CharType         ,IntType(b2,s2)    ->
	  if s2 <> B4 then def_coerce e2;
	  def_coerce e1;
	  true
      |	IntType(b1,s1)   ,CharType          ->
	  if s1 <> B4 then def_coerce e1;
	  def_coerce e2;
	  true
      |	CharType        ,CharType           ->
	  def_coerce e1;
	  def_coerce e2;
	  true
      |	_ -> false
    end
;;

(* Given a target type and a typed expression coerce e to the target type if
   possible. *)
let unify_coerce1 global_env e t =
  let t = compress t in
  let te = compress (deSome e.exp_typ) in
  if unify global_env te t then true
  else
    begin match te,t with
      IntType(_,_),IntType(_,_) | IntType(_,_),CharType 
    | CharType,IntType(_,_) | CharType,CharType -> coerce e t; true
    | _,_                       -> false
    end

(* check that an expression, used for initialization of globals, is
 * "constant" (i.e., can be computed at compile time.)
 *)
let rec check_constexp e =
  begin
    let loc = e.exp_loc in
    match e.raw_exp with
      Const _ -> ()
    | ConstArray (el,t) -> List.iter check_constexp el
    | NewStruct (_,tn,el) -> List.iter check_constexp el
    | NewUnion (_,tn,f,eo) ->
	(match eo with
	  None -> ()
	| Some e -> check_constexp e)
    | NewTuple (el) -> List.iter check_constexp el;
    | _ ->
 	terr loc "Global variable initializer must be a constant expression."
  end

(* returns an expression for a default initializer for a type t (if any) *)
let rec default_initializer global_env t loc =
  let def_init t = default_initializer global_env t loc in
  begin
    let err() = terr loc ("declaration of type "^(typ2string t)
			  ^" requires initializer") in
    let abort () = err (); raise Gcdfec.Exit in
    let make_exp re = { exp_typ = None; raw_exp = re; exp_loc = loc } in
    match t with
      IntType(true,_) -> make_exp(Const (Int i32_0))
    | IntType(false,_) -> make_exp(Cast(t,make_exp(Const (Int i32_0))))
    | BooleanType     -> make_exp(Const (Bool false))
    | StringType      -> make_exp(Const (String ""))
    | CharType        -> make_exp(Const (Char '\000'))
    | ArrayType(t,None) ->
        make_exp(ConstArray([],Some t))
    | ArrayType(t,Some e) ->
        (* Too bad if e is 0, we'll construct the default element anyway *)
        make_exp(FunCall(make_exp(Var "new_array"),
                         ref None,
                         [e;default_initializer global_env t loc]))
    | TupleType ts    -> make_exp(NewTuple (List.map def_init ts))
    | NamedType (n,ts) ->
(* let n = !n in
   should suffice but relies on tricky invariants.  So we burn a few
   cycles to cover up our sloppiness. *)
	let n = complete_typ_name global_env (* Dan *)
	    (fun v -> (Dict.member global_env.structs v ||
	               Dict.member global_env.unions  v ||
                       Dict.member global_env.abstracts v))
	    (!n) in
	if (possibly_null global_env n) then make_exp(Const(Null))
	else 
	  begin try 
	    let sd = Dict.lookup global_env.structs n in
	    let init = List.combine sd.st_tyvars ts in
	    let proc_field (f,c,t) =
	      def_init (subst init t)
	    in
	    let es = List.map proc_field sd.st_fields in
	    make_exp (NewStruct (n,ref (Some ts),es))
	  with Dict.Absent ->
	    begin try 
	      let ud = Dict.lookup global_env.unions n in
	      let (field,e_opt) = 
		match ud.un_fields with 
		  [] -> terr loc "Union with no cases."; raise Gcdfec.Exit 
		| (f,t)::tl -> 
		    if t=VoidType then (f,None)
		    else 	      
		      let init = List.combine ud.un_tyvars ts in
		      (f,Some (def_init (subst init t)))
	      in
	      make_exp (NewUnion (n,ref (Some ts),field,e_opt))
	    with Dict.Absent -> terr loc "Abstract types require initializers.";
	      abort ()
	    end
	  end
    | FnType _  -> abort () (* Should initialize with dummy function. *)
    | ExnType   -> abort () (* Should initialize with default exception. *)
    | _ -> abort ()
  end

(* returns (true,log_2(y)) if y is an even power of 2, otherwise returns
 * (false,?).  Assumes y >= 0.
 *)
let is_power_of_2 y = 
  let rec power2 log x y = 
    if y < x then (false,log)
    else if y = x then (true,log)
    else power2 (log+1) (x lsl 1) y
  in power2 0 1 y

let rec optimize_binop p e1 e2 = 
  let default_r = Primop(p,[e1;e2]) in
  match p,e1.raw_exp,e2.raw_exp with
    Plus,Const(Int i1),Const(Int i2) -> Const(Int(i1 +$ i2))
  | Plus,Const(Int i1),r when (i1 = i32_0) -> r
  | Plus,r,Const(Int i1) when (i1 = i32_0) -> r
  | Times,Const(Int i1),Const(Int i2) -> Const(Int(i1 *$ i2))
  | Times,Const(Int i1),r when (i1 = i32_1) -> r
  | Times,r,Const(Int i1) when (i1 = i32_1) -> r
  | Times,r,Const(Int i2) when (i2 >$ i32_1) ->
      let (isp2,log2) = is_power_of_2 (int32_to_int i2)
      in if isp2 then
	Primop(Bitlshift,[e1;{ exp_typ = e2.exp_typ;
			       raw_exp = Const(Int (int_to_int32 log2));
			       exp_loc = e2.exp_loc}])
      else default_r
  | Times,Const(Int i2),r -> optimize_binop p e2 e1
  | Minus,Const(Int i1),Const(Int i2) -> Const(Int(i1 -$ i2))
  | Minus,r,Const(Int i2) when (i2 = i32_0) -> r
  | Div,Const(Int i1),Const(Int i2) ->
      if (i2 = i32_0) then
	(terr e2.exp_loc ("compile-time division by zero"); default_r)
      else (Const(Int(i1 /$ i2)))
  | Mod,Const(Int i1),Const(Int i2) -> Const(Int(mod32 i1 i2))
  | Bitand,Const(Int i1),Const(Int i2) -> Const(Int(land32 i1 i2))
  | Bitor,Const(Int i1),Const(Int i2) -> Const(Int(lor32 i1 i2))
  | Bitxor,Const(Int i1),Const(Int i2) -> Const(Int(lxor32 i1 i2))
  | Bitlshift,Const(Int i1),Const(Int i2) -> Const(Int(lsl32 i1 i2))
  | Bitlrshift,Const(Int i1),Const(Int i2) -> Const(Int(lsr32 i1 i2))
  | Bitarshift,Const(Int i1),Const(Int i2) -> Const(Int(asr32 i1 i2))
  | _,_,_ -> default_r

let optimize_unop p e = 
  let default_r = Primop(p,[e]) in
  match p,e.raw_exp with
    Not,Const(Bool b) -> Const(Bool(not b))
  | Bitnot,Const(Int i) -> Const(Int (lnot32 i))
  | Ord,Const(Char c) -> Const(Int (int_to_int32 (Char.code c))) 
  | Chr,Const(Int i) -> Const(Char (Char.chr ((int32_to_int i) land 255)))
  | Size,Const(String s) -> Const(Int (int_to_int32 (String.length s)))
  | _,_ -> default_r

(**********************************************************************)
(* The following recursive function typecheck Stmts, Expressions, ... *)
(* given a global environment and an initial function environment.    *)
(**********************************************************************)
let rec tcStmt global_env env ({raw_stmt = s;stmt_loc = loc} as stmt) =
  begin
    let tcExp' e = tcExp global_env env e in
    let tcStmt' s = tcStmt global_env env s in
    let unify = unify global_env in
    let unify_coerce1 = unify_coerce1 global_env in
    match s with
    Skip -> ()
  | Exp e -> (tcExp' e; ())
  | Seq(s1,s2) -> (tcStmt' s1; tcStmt' s2)
  | Return(eopt) ->
(* DEBUG *)
      if !debug
      then 
	stmt.raw_stmt <-
	   (match eopt with 
	     None   -> Seq(debug_pop(), {raw_stmt=s; stmt_loc=loc});
	   | Some e -> 
	       let t  = retType env in
	       let t' = tcExp' e    in
	       if not (unify_coerce1 e t)
	       then terr loc ("returns value of type "^(typ2string t')^
			      " but requires " ^(typ2string t));
	       match t with
		 VoidType -> Seq(make_stmt(Exp e), debug_pop())
	       | _ ->
		   let vname = "?R" in 
		   Decl(vname, t, ref (Some e),
		    	make_stmt 
			  (Seq (debug_pop(),
			       {raw_stmt = Return (Some (make_exp (Var vname)));
				stmt_loc = loc}))));
(* END DEBUG *)
      (match eopt,(retType env) with
	None,VoidType -> ()
      | None,t -> terr loc ("must return a value of type "^(typ2string t))
      | Some e,t ->
	  let t' = tcExp' e in
	  if not (unify_coerce1 e t) then
	    terr loc ("returns value of type "^(typ2string t')^" but requires "
		      ^(typ2string t)))
  | IfThenElse(e,s1,s2) ->
      let t = tcExp' e in
      if not (unify BooleanType t)
      then terr e.exp_loc ("argument of if has type "^(typ2string t)
			   ^" instead of bool");
      tcStmt' s1;
      tcStmt' s2
  | While(e,s) ->
      let t = tcExp' e in
      if not (unify BooleanType t)
      then terr e.exp_loc ("while argument has type "^(typ2string t)
			   ^" instead of bool");
      tcStmt global_env (set_inloop env) s
  | Break None ->
      if not (inloop env) then terr loc "break not inside loop"
  | Break (Some x) ->
      if not (label_bound x env) 
      then terr loc ("Label "^x^" undefined.")
  | Continue None ->
      if not (inloop env) then terr loc "continue not inside loop"
  | Continue (Some x) ->
      if not (label_bound x env)
      then terr loc ("Label "^x^" undefined.")
  | For(e1,e2,e3,s) ->
      begin
	tcExp' e1;
	tcExp' e3;
	let t = tcExp' e2 in
	if not (unify BooleanType t)
	then terr e2.exp_loc ("2nd for argument has type "^(typ2string t)
			      ^" instead of bool");
	tcStmt global_env (set_inloop env) s
      end
  | IntSwitch(e,ss,s) ->
      begin
	let t = tcExp' e in
	if not (unify_coerce1 e (IntType(true,B4))) (* XXX - should allow unsigned values too. *)
	then terr loc ("switch has argument type "^(typ2string t)
		       ^" but int cases");
	let rec check_unique ss =
	  match ss with
	    [] -> ()
	  | [_] -> ()
	  | (i1,_)::(((i2,_)::_) as rest) ->
	      if i1 =$ i2 then
		(terr loc ("switch has two cases for "^(string_of_int32 i1));
		 raise Gcdfec.Exit)
	      else check_unique rest in
	let ss = Sort.list (fun (i1,s1) (i2,s2) -> (i1 <=$ i2)) ss in
	check_unique ss;
	List.iter (fun (_,s) -> tcStmt' s) ss;
	tcStmt' s
      end
  | CharSwitch(e,ss,s) ->
      begin
	let t = tcExp' e in
	if not (unify CharType t)
	then terr loc ("switch has argument type "^(typ2string t)
		       ^" but char cases");
	let rec check_unique ss =
	  match ss with
	    [] -> ()
	  | [_] -> ()
	  | (i1,_)::(((i2,_)::_) as rest) ->
	      if i1 = i2 then
		terr loc ("switch has two cases for "^(Char.escaped i1))
	      else check_unique rest in
	let ss = Sort.list (fun (i1,s1) (i2,s2) -> (i1 <= i2)) ss in
	begin
	  check_unique ss;
	  List.iter (fun (_,s) -> tcStmt' s) ss;
	  tcStmt' s
	end
      end
  | UnionSwitch(e,ss,def) ->
      (* UnionSwitch's can actually be one of two things:  a switch on a
       * union value or a switch on an exn value.  We can't tell
       *syntactically. *)
      begin
	let err t = terr loc ("switch has argument type "^(typ2string t)
			      ^" but union cases") in
	match compress (tcExp' e) with
	  NamedType(u,ts) ->
	    (try
	      let u = !u in
	      (let ud = Dict.lookup global_env.unions u in
	      (* compute instantiation for union type *)
	      let inst = List.combine ud.un_tyvars ts in
	      (* sort statements by data constructor name *)
	      let ss =
		Sort.list (fun a1 a2 -> a1.arm_field <= a2.arm_field) ss in
	      (* check for duplicate cases -- assumes sorted cases *)
	      let rec check_unique ss =
		match ss with
		  [] -> ()
		| [_] -> ()
		| a1::a2::_ when a1.arm_field = a2.arm_field ->
		    terr loc ("switch has two cases for field "^a1.arm_field)
		| hd::tl -> check_unique tl
	      in
	      (* find missing field for error message *)
	      let rec check_complete fields =
		match fields with
		  [] -> ()
		| (f,_)::rest ->
		    if List.exists 
			(fun arm -> f = arm.arm_field) ss then
		      check_complete rest
		    else
		      terr loc ("switch missing case for field "^f) in
	      (* check that fields are complete -- assumes default is empty *)
	      let rec check_exhaustive fields =
		List.for_all 
		  (fun f->List.exists (fun a ->(fst f)=a.arm_field) ss)
		  fields
	      in
	      (* type-check a clause of the switch *)
	      let tc_arm ({arm_field=f;arm_var=x;arm_body=s} as arm) =
		let loc' = s.stmt_loc in
		let t =
		  (try List.assoc f ud.un_fields with
		    Not_found -> terr loc' ("bad switch field "^f);
		      raise Gcdfec.Exit) in
		(* add x to the environment with the type asssociated with f.
		 * if f has VoidType then x shouldn't really be there. *)
		let env =
		  (match x,t with
		    None,VoidType -> env
		  | Some _,VoidType ->
		      terr loc' ("field "^f^" has type void");
		      raise Gcdfec.Exit
		  | None,_ ->
		      terr loc' ("field "^f^" has non-void type -- an identifier is required");
		      env
		    | Some x,t -> 
			arm.arm_typ <- (subst inst t);
			add_var x arm.arm_typ env)
		  in tcStmt global_env env s in
		begin
		  check_unique ss;
		  List.iter tc_arm ss;
		  let exhaustive = check_exhaustive ud.un_fields in
		  (match def with
		    None ->
		      begin
			if ud.un_possibly_null then
			  terr loc ("switch on ? type "^u^" requires default");
			if not exhaustive then
			  terr loc ("non-exhaustive switch requires default");
			check_complete ud.un_fields
		      end
		  | Some s ->
		      begin
			tcStmt global_env env s;
			if not ud.un_possibly_null & exhaustive then
			  terr loc ("switch has redundant default")
		      end)
		end) with Dict.Absent -> err (NamedType(u,[])))
	(* the switch is really an exception switch *)
	| ExnType -> 
	    stmt.raw_stmt <- ExnSwitch(e,ss,def);
	    tcStmt' stmt
	| t -> err t
      end	
  | ExnSwitch(e,ss,def) ->
      begin 
	(* DAN HACK: if you leave off a default and you're switching
	   on a variable, we'll re-raise.
	   More importantly, perhaps, we'll dump the stack on re-raise.
	   If you have a default, then no stack dumping will occur.
	   *)
	let def = 
	  match e.raw_exp, def with 
	    Var v, None ->
	      let reraise_stmt = make_stmt(Exp(make_exp(Raise e))) in
	      let inserted_default =
	      	Some (if !debug
	      	then make_stmt
		    (Seq 
		     (make_call_stmt "callStackRestore" [make_exp(Var "?old")],
		      reraise_stmt))
	      	else reraise_stmt) in
	      (stmt.raw_stmt <- ExnSwitch(e, ss, inserted_default);
	       inserted_default)
	  | _ -> def 
	in
	(* END HACK *)

	begin match compress (tcExp' e) with 
	  ExnType -> ()
	| t -> terr loc 
	      ("exception switch has non-exception type "^(typ2string t)) 
	end;
      (* FMS: must first complete the exception names of the fields *)
	let complete_exn n = 
	  try complete_val_name global_env (Dict.member global_env.exceptions) n 
	  with Unbound -> 
	    (terr loc ("Exception "^n^" is unknown."); raise Gcdfec.Exit)
	in
	let ss = List.map 
	    (fun a -> { a with arm_field = complete_exn a.arm_field }) ss in	
      (* sort the cases by exn name *)
	let ss = Sort.list (fun a1 a2 -> a1.arm_field <= a2.arm_field) ss in
      (* Now rewrite the statement with the completed, sorted names *)
	stmt.raw_stmt <- ExnSwitch(e,ss,def);
      (* check that the cases are unique *)
	let rec check_unique ss =
	  match ss with
	    [] -> ()
	  | [_] -> ()
	  | a1::a2::_ when a1.arm_field = a2.arm_field ->
	      (terr loc ("exn switch has two cases for exn "^a1.arm_field);
	       raise Gcdfec.Exit)
	  | hd::tl -> check_unique tl 
	in
	(* check a case, adding x to the environment with the appropriate
	 * type.  If the exception f carries void, then x shouldn't
	 * really be there. *)
	let tc_arm ({arm_field=f;arm_var=x;arm_body=s} as arm) =
	  let loc' = s.stmt_loc in
	  let t =
	    (try Dict.lookup global_env.exceptions f with
	      Dict.Absent ->
		terr loc' ("unknown exception "^f);
		raise Gcdfec.Exit) in
	  let env =
	    match x,t with
	      None,VoidType -> env
	    | Some _,VoidType ->
		terr loc' ("exception "^f^" carries no type");
		env
	    | None,_ ->
		terr loc' ("exception "^f^" has non-void type -- an identifier is required");
		env
	    | Some x,t -> arm.arm_typ <- t; add_var x t env
	  in 
	  tcStmt global_env env s 
	in
	begin
	  check_unique ss;
	  List.iter tc_arm ss;
	  (match def with
	    None -> (* With Dan's hack, only possible when e not a variable *)
	      terr loc ("exception switch requires default");
	      raise Gcdfec.Exit
	  |	Some s -> tcStmt global_env env s)
	end
      end
    | Decl(x,t,eopt,s) ->
	begin
          (* check_valid_type completes type names in t *)
	  check_valid_type loc global_env (tyvars env) t;
	  let e =
	    match !eopt with
	      None   -> default_initializer global_env t loc
	    | Some e -> e
	  in 
	  eopt := (Some e);
	  let t' = tcExp' e in
	  if not (unify_coerce1 e t)
	  then terr loc (x^" declared with type "^(typ2string t)
			 ^" but initializer has type "^(typ2string t'));
	  tcStmt global_env (add_var x t env) s
	end
    | Label(x,s) -> tcStmt global_env (add_label x env) s
(* Cyclone *)
    | Cut s ->
        begin
          match env with
            Frame(fenv,cenv) ->
              (* set_outloop to prevent break/continue across cut *)
              tcStmt global_env (set_outloop(Hidden(fenv,cenv))) s
          | Hidden _ -> terr loc "can't cut while codegen is in cut"
          | Outermost _ ->
	      terr loc "cut can only be used within codegen"
        end
    | Splice s ->
        begin
          match env with
            Hidden(fenv,cenv) ->
              (* set_outloop to prevent break/continue across splice *)
              tcStmt global_env (set_outloop(Frame(fenv,cenv))) s
          | Frame _ -> terr loc "can't splice while already in codegen"
          | Outermost _ ->
	      terr loc "splice can only be used within cut"
        end
(* End Cyclone *)
    | Do(s,e) ->
	let t = tcExp' e in
	if not (unify BooleanType t)
	then terr e.exp_loc ("do-while guard has type "^(typ2string t)
			     ^" instead of bool");
	tcStmt global_env (set_inloop env) s
    | Try(s1,x,s2) ->
	begin
(* DEBUG *)
	  let env = 
	  if !debug
	  then 
	    (let vname = "?Q" in
	    stmt.raw_stmt <- 
	       Decl(vname, IntType(true,B4),
		    ref (Some(make_call "callStackSave" [])), 
		    {raw_stmt = stmt.raw_stmt; stmt_loc = stmt.stmt_loc});
	    s2.raw_stmt <-
	       (let oldname = "?old" in (* used in inserted default *)
	       Decl(oldname, IntType(true,B4),
		    ref(Some (make_call "callStackSave" [])),
		    make_stmt
		      (Seq(make_call_stmt "callStackRestore" 
			     [make_exp(Var vname)],
			   {raw_stmt = s2.raw_stmt; stmt_loc = s2.stmt_loc}))));
	    add_var vname (IntType(true,B4)) env)
	  else
	    env in
(* END DEBUG *)
	  tcStmt' s1;
	  tcStmt global_env (add_var x ExnType env) s2
	end
  end(* end of tcStmt *)
and tcExp global_env env exp =
  begin
    let e = exp.raw_exp in
    let loc = exp.exp_loc in
    let tcExp' e = tcExp global_env env e in
    let unify = unify global_env in
    let unify_un t1 t2 = (unify t1 t2; ()) in
    let unify_coerce1 = unify_coerce1 global_env in
    let unify_coerce2 = unify_coerce2 global_env in
    let err s = terr loc s in
    let abort s = terr loc s; raise Gcdfec.Exit in
    let t2s = typ2string in
    let t =
      begin
	match e with
	  Const(Int _) -> IntType(true,B4)
	| Const(Bool _) -> BooleanType
	| Const(String _) -> StringType
	| Const(Char _) -> CharType
	| Const(Null) -> Evar(Option,ref None)

	| ConstArray([],Some t) ->
	    (check_valid_type loc global_env (tyvars env) t; ArrayType(t,None))
	| ConstArray([],None) ->
	    abort "empty array has no type"
	| ConstArray((e1::rest) as es,_) ->
	    begin
	      let t = tcExp' e1 in
	      let rec aux es =
		match es with
		  [] -> ()
		| (e::rest) ->
		    let _ = tcExp' e in
		    if unify_coerce1 e t then aux rest
		    else 
		      (terr e.exp_loc "type mismatch in array constant";
		       raise Gcdfec.Exit)
	      in
	      aux rest;
	      ArrayType(t,None)
	    end
	| Var x ->
	    begin try lookup env x
	    with Not_found -> try (Dict.lookup global_env.globals x)
	    with Dict.Absent ->
	      begin
		let x' = try
		  let ge = global_env in
		  complete_val_name ge (Dict.member ge.globals) x
		with Unbound -> raise (UnboundVar (x,loc))
		in
		exp.raw_exp <- Var x';
		Dict.lookup global_env.globals x'
	      end
	    end
	| Primop(p,[e1;e2]) ->
	    let t1 = tcExp' e1 in
	    let t2 = tcExp' e2 in
	    let (p,t) = tcBinPrimop global_env env loc p e1 e2 in
	    let r = optimize_binop p e1 e2 in
	    exp.raw_exp <- r;
	    t
	| Primop(p,[e]) ->
	    let t1 = tcExp' e in
	    let t = tcUnPrimop global_env env loc p e in
	    let r = optimize_unop p e in
	    exp.raw_exp <- r;
	    t
	| Primop(_,_) -> abort "primop wrong # of args"
	| Conditional(e1,e2,e3) ->
	    let t1 = tcExp' e1 in
	    if not (unify BooleanType t1)
	    then err ("conditional argument has type "^(t2s t1)
			   ^"instead of bool");
	    let t2 = tcExp' e2 in
	    let t3 = tcExp' e3 in
	    if unify_coerce2 e2 e3 then (deSome e2.exp_typ)
	    else
	      (abort ("clauses of conditional do not match type: "
			 ^(t2s t2)^" != "^(t2s t3)))
	| AssignOp(e1,po,e2) ->
	    let t1 = tcExp' e1 in
	    let t2 = tcExp' e2 in
	    check_valid_lhs global_env e1;
	    begin match po with
	      None -> 
		if not (unify_coerce1 e2 t1)
		then err ("type mismatch: "^(t2s t1)^" != "^(t2s t2));
		t1
	    | Some p -> 
		let (p',t_result) = tcBinPrimop global_env env loc p e1 e2 in
		if not (unify t_result t1 or coercable t_result)
		then err "Cannot use this operator in front of an assignment.";
		if p'<>p then exp.raw_exp <- (AssignOp (e1,Some p',e2));
		t1
	    end
	| FunCall(e,ts,es) ->
	    let t = tcExp' e in
	    begin match t with
	      FnType(vs,rt,argts) ->
		(* inst is the instantiation of the bound type variables vs.
		 * we substitute Evar's for them to be unified.  *)
		let inst = List.map (fun v -> (v,Evar(Byte4,ref None))) vs in
		let rt = (subst inst) rt in
		let argts = List.map (subst inst) argts in
		(try
		  (match !ts with
		    (* implicit instantiation *)
		    None -> ts := Some(List.map snd inst)
		    (* explicit instantiation *)
		  | Some ts -> List.iter2 unify_un ts (List.map snd inst));
		  (List.iter2
		     (fun e t ->
		       let t' = tcExp' e in
		       if not (unify_coerce1 e t)
		       then terr e.exp_loc ("argument type mismatch.  "^
					    "actual has type "^(t2s t')
					    ^" but formal has type "
					    ^(t2s t)))
		     es argts);
		 rt
		with Invalid_argument _ -> abort "argument number mismatch")
	    | t -> abort ("attempt to apply non function type " ^(t2s t))
	    end
	| TypInst (e,ts) ->
	    let t = tcExp' e in
	    begin match t with
	      FnType(vs,rt,argts) ->
		begin
		  let (vs,inst) = 
		    let rec aux vs ts inst =
		      match vs,ts with
			vs,[] -> (vs,inst) (* Allows partial instantiation *)
		      |	[],_ -> 
			  abort "instantiation with too many arguments"
		      |	hd1::tl1,hd2::tl2 -> aux tl1 tl2 ((hd1,hd2)::inst)
		    in
		    aux vs ts []
		  in
		  let rt = (subst inst) rt in
		  let argts = List.map (subst inst) argts in
		  let new_t = FnType(vs,rt,argts) in 
		  check_valid_type loc global_env (tyvars env) new_t; (*Dan*)
		  new_t
		end
	    | t -> abort ("attempt to instantiate non function type "^(t2s t))
	    end
	| NewStruct(n,ts,es) ->
	    (* The parser does not distinguish NewExn from NewStruct so
	       we have to make this distinction here.  If a struct and an
	       exception have the same name we will consider that an error.
	     *)
	    begin let name =
	      try complete_typ_name global_env 
		  (Dict.member global_env.structs) n
	      with Unbound ->
		begin try complete_val_name global_env 
		    (Dict.member global_env.exceptions) n
		with Unbound ->
		  abort (n ^ " refers to neither a struct nor an exception.")
		end
	    in
	    match (Dict.member global_env.structs name,
		   Dict.member global_env.exceptions name)
	    with
	      (true,true) ->
		abort ("Ambiguous new (struct? or exception?) " ^ n)
	    | (false,false) -> 
		(* Dan bug fix -- occurs if n is an opened union or abstract *)
		(* Was thought to be Impossible *)
		abort (n ^ " refers to neither a struct nor an exception.")
	    | (false,true) ->
		(* It is a NewExn *)
		begin
		  let eopt =
		    match (!ts,es) with
		      (None,[])  -> None
		    | (None,[e]) -> (Some e)
		    | (Some _,_) ->
		    	abort (n^" is an exception. Type arguments are not allowed.")
		    | _ ->
		    	abort (n^" is an exception. Too many arguments")
		  in
		  exp.raw_exp <- NewExn (name,eopt);
		  tcExp' exp
		end
	    | (true,false) ->
		(* It is a NewStruct. *)
	    	begin try
		  exp.raw_exp <- NewStruct (name,ts,es);
		  let sd = Dict.lookup global_env.structs name in
		  (* inst is the instantiat'n for the bound type variables with
		   * Evar's used for unification. *)
		  let inst = 
		    List.map (fun v->(v,Evar(Byte4,ref None))) sd.st_tyvars
		  in
		  begin match !ts with
		    (* implicit instantiation *)
		    None -> ts := Some(List.map snd inst)
		    (* explicit instantiation *)
		  | Some ts -> 
		      (try List.iter2 unify_un ts (List.map snd inst)
		      with Invalid_argument _ ->
			abort "Explicit instantiation has wrong # of arguments")
		  end;
		  let checkexp e (f,_,t) =
		    let t = subst inst t in
		    let t' = tcExp' e in
		    if not (unify_coerce1 e t)
		    then terr e.exp_loc
		    	("struct "^n^", field "^f^" has type "^(t2s t)^
			 " but argument has type "^(t2s t'))
		  in
		  (try List.iter2 checkexp es sd.st_fields
		  with Invalid_argument _ -> 
		    abort ("struct "^n^" argument # mismatch"));
		  NamedType(ref name,List.map snd inst)
	    	with
		  Dict.Absent -> failwith "poptype.ml:NewStruct:Impossible!"	
	    	end
	    end
	| StructMember(e,f) ->
	    begin
	      match compress (tcExp' e) with
	      	NamedType(n,ts) ->
		  begin try
		    let n = !n in
		    let sd = Dict.lookup global_env.structs n in
		    let rec aux fs =
		      match fs with
		      	((f',_,t)::rest) ->
		    	  if f = f' then
			    let inst = List.map2 (fun v t -> (v,t))
			  	sd.st_tyvars ts in
			    subst inst t
			  else aux rest
		      | [] ->
			  abort ("struct "^n^" has no "^f^" field")
		    in aux sd.st_fields
		  with
		    Dict.Absent -> 
		      exp.raw_exp <- UnionMember(e,f);
		      tcExp' exp
		  end
	      | t -> abort ((t2s t)^" not a struct or union")
	    end
	| UnionMember(e,f) ->
	    begin match (compress (tcExp' e)) with
	      NamedType(n,ts) ->
		begin try 
		  let n = !n in
		  let ud = Dict.lookup global_env.unions n in
		  let rec aux fs =
		    match fs with
		      (f',t)::rest -> 
			if f = f' then
			  let inst = List.map2 (fun v t -> (v,t)) 
			      ud.un_tyvars ts 
			  in
			  (match subst inst t with
			    VoidType ->abort ("union projection has void type")
			  |        t -> t)
			else aux rest
		    | [] -> abort ("union "^n^" has no "^f^" field")
		  in 
		  aux ud.un_fields
		with Dict.Absent -> 
		  abort (!n ^ " not a struct or union")
		end
	    | t -> abort ((t2s t)^" not a struct or union type")
	    end
	| NewUnion(n,ts,f,eopt) ->
	    let name =
	      try complete_typ_name global_env (Dict.member global_env.unions) n
	      with Unbound -> abort (n^" is not a union")
	    in
	    exp.raw_exp <- NewUnion (name,ts,f,eopt);
	    begin try
	      (* similar to NewStruct *)
	      let ud = Dict.lookup global_env.unions name in
	      let inst = 
		List.map (fun v->(v,Evar(Byte4,ref None))) ud.un_tyvars in
	      let evars = List.map snd inst in

	      let rty = NamedType(ref name,evars) in
	      (match !ts with
		None -> ts := Some(evars)
	      |	Some ts -> 
		  (try List.iter2 unify_un evars ts
		  with Invalid_argument _ -> 
		    abort "Explicit instantiation has wrong # of arguments"));
	      let t = subst inst (List.assoc f ud.un_fields) in
	      begin match t,eopt with
		VoidType,None -> ()
	      |	_,None -> abort ("union "^n^", field "^f^
				 " requires argument of type "^(t2s t))
	      |	_,Some e ->
		  let t' = tcExp' e in
		  if unify_coerce1 e t then ()
		  else err ("union "^n^", field "^f^
				 " requires argument of type "
				 ^(t2s t)^" not "^(t2s t'))
	      end;
	      rty
	    with
	      Dict.Absent -> abort(n^" is not a union type")
	    | Not_found -> abort("union "^n^" has no "^f^" field")
	    end
	| NewTuple(es) -> TupleType(List.map tcExp' es)
	| TupleMember(e,i) ->
	    (match compress (tcExp' e) with
	      TupleType ts ->
		(try List.nth ts (i - 1) with
		  Invalid_argument _ -> 
		    if i=0 then abort ("First element of tuple is 1 not 0")
		    else abort ("Invalid offset of tuple")
		| Failure _ ->
		    let num_fields = string_of_int (List.length ts) in
		    abort("tuple has "^num_fields^" fields"))
	    | Evar _ ->
		abort ("cannot determine # of fields in tuple type")
	    | t -> abort ("tuple projection applied to "^(t2s t)^" value"))
	| Subscript(e1,e2) ->
	    begin
	      let t1 = tcExp' e1 in
	      let t2 = tcExp' e2 in
	      if not (unify_coerce1 e2 (IntType(false,B4))) (* XXX *)
	      then terr e2.exp_loc
		  ("subscript type "^(t2s t2)^" not an int");
	      match compress t1 with
	      	StringType -> CharType
	      | _ ->
		  let t = Evar(Any,ref None) in
		  if unify (ArrayType(t,None)) t1 then t
		  else  (terr e1.exp_loc
			   ("subscript on non-array/string type "^(t2s t1));
			 raise Gcdfec.Exit)
	    end
(*	| NewArray(e1,e2) ->
	    let t2 = tcExp' e2 in
	    let t1 = tcExp' e1 in
	    if not (unify_coerce1 e1 (IntType(false,B4))) 
	    then err ((t2s t1)^" not an int");
	    ArrayType(t2,None) *)
	| NewExn(id,eopt) ->
	    (* FMS : id is completed when converting from NewStruct to
	       NewExn!!! However sometimes we parse NewExn's directly so
	       we still need to complete the name. *)
	    begin try
	      let name = 
		try complete_val_name global_env 
		    (Dict.member global_env.exceptions) id
		with Unbound -> abort ("Unknown exception "^id)
	      in
	      exp.raw_exp <- NewExn(name,eopt);	      
	      let t1 = Dict.lookup global_env.exceptions name in
	      begin match t1,eopt with
		VoidType,None -> ()
	      |	t,Some e ->
		  let t2 = tcExp' e in
		  if not (unify_coerce1 e t)
		  then err ("exception "^id^" requires "^(t2s t)^"<>"^(t2s t2));
		  ()
	      |	_,None -> err ("exception "^id^" requires "^(t2s t1)^" value")
	      end;
	      ExnType
	    with Dict.Absent -> 
	      failwith("Impossible: Bound exception without type.")
	    end
	| Raise e ->
	    let t = tcExp' e in
	    if not (unify t ExnType)
	    then err ("expected type exn found type " ^ (t2s t));
	    Evar(Any,ref None)
	| SeqExp es ->
	    let rec aux es = 
	      match es with
		[] -> abort ("Impossible. Sequence exp with no expressions.")
	      |	[hd] -> tcExp' hd
	      |	hd :: tl -> tcExp' hd; aux tl
	    in
	    aux es
	| Nop -> VoidType
	| Cast(t,e) -> 
	    let t' = tcExp' e in
	    (* Changed by Dan *)
            (* begin match (t,t') with
	      IntType(_,_),IntType(_,_) -> ()
	    | _,_ -> abort ("Cast is only allowed on integral types.")
	    end; *)
	    (if (not (coercable t && coercable t'))
	    then abort ("Cast is only allowed on integral types."));
	    t
(* Cyclone *)
        | Codegen fd ->
            let env' = Frame(fndecl2f_env fd, env) in
	    let _ = check_fundecl_valid_type global_env loc fd in
            tcFunDecl global_env env' loc fd;
            FnType(fd.fn_tyvars,fd.fn_ret_type,List.map snd fd.fn_args)
        | Fill e ->
            begin
              match env with
                Frame(fenv,cenv) -> tcExp global_env (Hidden(fenv,cenv)) e
              | Hidden _ ->
                  abort "fill cannot be used while codegen is in cut"
              | Outermost _ ->
		  abort "fill can only be used within codegen"
            end
(* End Cyclone *)
      end
    in (exp.exp_typ <- Some t; t)
  end (* end of tcExp *)
and tcBinPrimop global_env env loc p e1 e2 =
  begin
    let t1 = deSome e1.exp_typ in
    let t2 = deSome e2.exp_typ in 
    let err = terr loc in
    let unify_coerce2 = unify_coerce2 global_env in
    let abort s = terr loc s; raise Gcdfec.Exit in
    match p with
      (Plus | Times | TimesU | Minus | Div | DivU | Mod | ModU | Bitand | 
      Bitor | Bitxor | Bitlshift | Bitlrshift | Bitarshift) ->
	 if not (unify_coerce2 e1 e2)
	 then err "operator applied to non-numeric type";
	 let t = compress (deSome e1.exp_typ) in
	 if not (coercable t)
	 then err "operator applied to non-numeric type";
	 begin match t with 
	   IntType(false,_) ->
	     begin match p with
	       Times        -> (TimesU,t) 
	     | Div          -> (DivU  ,t)
	     | Mod          -> (ModU  ,t)
	     | _            -> (p     ,t)
	     end
	 | _                -> (p     ,t)
	 end
    | (Gt | GtU | Lt | LtU | Gte | GteU | Lte | LteU) ->
	if not (unify_coerce2 e1 e2)
	then err "comparison applied to non-numeric type";
        let t = compress (deSome e1.exp_typ) in
	if not (coercable t)
	then err "operator applied to non-numeric type";
	let p =
	  match t with
	    IntType(false,_) ->
	      (match p with
		Gt -> GtU | Lt -> LtU | Gte -> GteU | Lte -> LteU | p -> p)
	  | _ -> p
	in
	(p,BooleanType)
    | (Eq | Neq) ->
	if not (unify_coerce2 e1 e2)
	then err ("comparison of two types: "^(typ2string t1)^" != "
		       ^(typ2string t2));
	(p,BooleanType)
    | _ -> abort ("wrong # of args to primop");
  end
and tcUnPrimop global_env env loc p e =
  begin
    let t = deSome e.exp_typ in
    let err = terr loc in
    let abort s = terr loc s; raise Gcdfec.Exit in
    let unify_t = unify global_env t in
    let unify_coerce1 = unify_coerce1 global_env in
    match p with
      Not ->
	if not (unify_t BooleanType)
	then err ("! requires bool <> "^(typ2string t));
	BooleanType
    | Bitnot ->
	if not (unify_coerce1 e (IntType(true,B4)))
	then err ("bitwise negation requires int <> "^(typ2string t));
	IntType(true,B4)
    | Size ->
	begin match compress t with
	  StringType -> ()
	| ArrayType _ -> ()
	| _ -> err ("size requires string/array <> "^(typ2string t))
	end;
	IntType(true,B4)
    | Ord ->
	if not (unify_t CharType)
	then err ("ord requires char <> "^(typ2string t));
	IntType(true,B4)
    | Chr ->
	if not (unify_coerce1 e (IntType(true,B4)))
	then err ("chr requires int <> "^(typ2string t));
	CharType
    | _ -> abort "wrong # of args for primop"
  end
and check_valid_lhs global_env e =
  begin
    match e.raw_exp with
      Var _ -> ()
    | StructMember(e1,f) ->
	(match e1.exp_typ with
	  Some t ->
	    (match compress t with
	      NamedType(n,_) ->
		let n = !n in
	    	let rec aux fs =
		  match fs with
		    ((f',rw,_)::rest) ->
		      if f = f' then
		    	(match rw with
			  ReadWrite -> ()
		    	| ReadOnly -> terr e.exp_loc ("field "^f^" is const"))
		      else aux rest
		  | [] -> failwith "field lookup"
	    	in aux (Dict.lookup global_env.structs n).st_fields
	    | _ -> failwith "type lookup 0")
	| _ -> failwith "type lookup")
    | Subscript(e1,e2) ->
	(match e1.exp_typ with
	  Some t ->
	    (match compress t with
	      StringType -> ()
	    | ArrayType _ -> ()
	    | _ -> terr e.exp_loc "not a valid left-hand-side")
	| None -> failwith "type lookup 1")
    | TupleMember(e,i) -> ()
    | _ -> terr e.exp_loc "not a valid left-hand-side"
  end
and tcFunDecl global_env env loc fd =
  begin
    let returns = doesRet fd.fn_body in
    if fd.fn_ret_type <> VoidType & not returns
    then terr loc ("function body must return a value of type "^
		   (typ2string fd.fn_ret_type));
(* DEBUG *)
    if !debug
    then 
      (fd.fn_body.raw_stmt <- 
	 Seq(debug_push fd.fn_name, 
	     {raw_stmt = fd.fn_body.raw_stmt; stmt_loc = fd.fn_body.stmt_loc});
       if (not returns)
       then fd.fn_body.raw_stmt <- 
	  Seq({raw_stmt = fd.fn_body.raw_stmt; stmt_loc = fd.fn_body.stmt_loc},
	      debug_pop()));
(* END DEBUG *)
    try
      tcStmt global_env env fd.fn_body
	with UnboundVar (x,loc) ->
	  (terr loc ("unbound variable "^x); raise Gcdfec.Exit)
  end
(* end of tcFunDecl *)

let tcGlobalDecl loc global_env (s,v,t,eor) =
  begin
    let env =
      mk_env { f_name = "*BOGUS*";
	       f_tyvars = [];
	       f_args = [];
	       f_ret_type = VoidType;
	       f_locals = [];
	       f_inloop = false;
	       f_labels = [];
	     }
    in
    let e =
      match !eor with
	None -> default_initializer global_env t loc
      |	Some e -> check_constexp e; e
    in
    let t' = tcExp global_env env e in
    (* FMS:
       We need to unify this so the uninstantiated type variables in
       t' get instantiated for null.
       It is unfortunate but we must do this even when we create
       a default initializer.
       *)
    if not (unify global_env t t')
    then (terr loc ("The type of the initializer ("
		    ^ (typ2string t')
		    ^ ") does not match the declared type ("
		    ^ typ2string t ^").");
	  raise Gcdfec.Exit);
    eor := Some(e);
  end

let rec eliminate_open tds =
  match tds with
    [] -> []
  | (OpenDecl(v,tds'),_)::tl -> 
      List.rev_append (eliminate_open tds') (eliminate_open tl)
  | hd::tl -> hd :: (eliminate_open tl)
;;

(* decls is used to check that values are only declared a single time. *)
let rec add_global_env ((structs,unions,abstracts,globals,exns,decls) as e) 
    (d,loc) =
  let abort s = terr loc s; raise Gcdfec.Exit in
  begin match d with
    StructDecl sd ->
      let n = sd.st_name in
      let keep_sd () = Dict.insert structs n sd in      
      if Dict.member abstracts n 
      then (keep_sd (), unions, Dict.delete abstracts n,globals,exns,decls)
      else if Dict.member structs n then 
	begin
	  let sd_g = Dict.lookup structs n in
	  match sd_g.st_scope,sd.st_scope with
	    Extern,Public ->
	      (keep_sd(),unions,abstracts,globals,exns,decls)
	  | _ -> e 
	end
      else if Dict.member unions n then  
	abort ("struct and union share name "^(var2string n))
      else (keep_sd (),unions,abstracts,globals,exns,decls)
  | UnionDecl ud ->
      let n = ud.un_name in
      let keep_ud () = Dict.insert unions n ud in
      if Dict.member abstracts n 
      then (structs,keep_ud (),Dict.delete abstracts n,globals,exns,decls)
      else if Dict.member unions n then 
	begin
	  let ud_g = Dict.lookup unions n in
	  match ud_g.un_scope, ud.un_scope with
	    Extern,Public -> (* We keep the public declaration*)
	      (structs,keep_ud(),abstracts,globals,exns,decls)
	  | _ -> e (* Other error conditions are checked later. *)
	end
      else if Dict.member structs n then
	abort ("struct and union share name " ^(var2string n))
      else (structs,keep_ud (),abstracts,globals,exns,decls)
  | ExternType (n,vs,b) ->
      if Dict.member structs   n or 
	 Dict.member unions    n or 
	 Dict.member abstracts n
      then e 
      else (structs,unions,Dict.insert abstracts n (vs,b),globals,exns,decls)
  | ExceptionDecl(v,s,t) ->
      if Dict.member decls v 
      then abort ("Multiple declarations of "^(var2string v));
      if Dict.member exns v 
      then (structs,unions,abstracts,globals,exns,Dict.insert decls v ())
      else if Dict.member globals v then
	abort ("exception and global share name "^(var2string v))
      else (structs, unions, abstracts, globals, Dict.insert exns v t,
	    Dict.insert decls v ())
  | FunDecl fd ->
      let v = fd.fn_name in
      let arg_typs = List.map snd fd.fn_args in
      let t = FnType(fd.fn_tyvars,fd.fn_ret_type,arg_typs) in
      if Dict.member decls v
      then abort ("Multiple declarations of "^(var2string v));
      if Dict.member globals v 
      then (structs,unions,abstracts,globals,exns,Dict.insert decls v ())
      else if Dict.member exns v then
	abort ("function and exception share name "^(var2string v))
      else (structs,unions,abstracts,Dict.insert globals v t,exns,
	    Dict.insert decls v ())
  | ExternVal (v,t) -> 
      if Dict.member globals v then e
      else if Dict.member exns v then
	abort ("extern and exception share name "^(var2string v))
      else (structs,unions,abstracts,Dict.insert globals v t,exns,decls)
  | GlobalDecl (s,v,t,eor) ->
      if Dict.member decls v
      then abort ("Multiple declarations of "^(var2string v));
      if Dict.member globals v then e
      else if Dict.member exns v then
	abort ("global and exception share name " ^(var2string v))
      else (structs,unions,abstracts,Dict.insert globals v t,exns,
	    Dict.insert decls v ())
  | PrefixDecl _ -> failwith "poptype.ml:add_global_env: prefixes should have been eliminated"
  | OpenDecl(prefix,ds') -> add_globals_env e ds'
  end
and add_globals_env e ds = List.fold_left add_global_env e ds

let initial_global_env ds =
  let d () = Dict.empty compare in
  let empty_env =
    (d(), d(), d(), d(), d(), d())
  in
  let (s,u,a,g,e,_) = add_globals_env empty_env ds in
  { structs = s; unions = u; abstracts = a; globals = g; exceptions = e;
    open_typs = d(); open_vals = d()
  }

(* Must check_valid_type first to ensure that names are fully expanded. *)
let rec check_fundecl global_env loc fd =
  let fd_type = check_fundecl_valid_type global_env loc fd in
  let fd_genv = Dict.lookup global_env.globals fd.fn_name in
  if not (unify global_env fd_type fd_genv)
  then 
    begin 
      terr loc ("Inconsistent declarations for " ^ (var2string fd.fn_name));
      raise Gcdfec.Exit;
    end;
  ()
;;

let check_structdecl global_env loc sd =
  let abort s = 
    let n = var2string sd.st_name in
    terr loc ("Multiple declarations of "^n^" disagree on " ^ s); 
    raise Gcdfec.Exit 
  in
  begin
    List.iter (fun (_,_,t) -> check_valid_type loc global_env sd.st_tyvars t)
      sd.st_fields;
    let sd_genv = Dict.lookup global_env.structs sd.st_name in
    if not (scope_match sd.st_scope sd_genv.st_scope)
    then abort "scope.";
    if not (list_match (=) sd.st_tyvars sd_genv.st_tyvars)
    then abort "type variables.";
    if sd.st_possibly_null <> sd_genv.st_possibly_null
    then abort "?.";
    (try List.iter2 (fun (f1,c1,t1) (f2,c2,t2) ->
      if f1<>f2 
      then abort ("fields ("^(var2string f1)^" != "^(var2string f2)^").")
      else if c1<>c2 or not (unify global_env t1 t2)
      then abort ("field "^(var2string f1)^".");
      ()) sd.st_fields sd_genv.st_fields	
    with Invalid_argument _ -> abort " # of fields")
  end

let check_uniondecl global_env loc ud =
  begin
    let abort s = 
      let n = var2string ud.un_name in
      terr loc ("Multiple declarations of "^n^" disagree on "^s);
      raise Gcdfec.Exit
    in
    List.iter (fun (_,t) -> check_valid_type loc global_env ud.un_tyvars t)
      ud.un_fields;
    let ud_genv = Dict.lookup global_env.unions ud.un_name in
    if not (scope_match ud.un_scope ud_genv.un_scope)
    then abort "scope";
    if not (list_match (=) ud.un_tyvars ud_genv.un_tyvars)
    then abort "type variables";
    if ud.un_possibly_null <> ud_genv.un_possibly_null
    then abort "?";
    (try List.iter2 
      (fun (f1,t1) (f2,t2) ->
	if f1!=f2 
	then abort ("fields ("^(var2string f1)^" != "^(var2string f2)^")");
	if not (unify global_env t1 t2)
	then abort ("type of field " ^ (var2string f1));
	())
      ud.un_fields ud_genv.un_fields
    with Invalid_argument _ -> abort "# of fields")
  end
  
let check_globaldecl loc global_env (s,v,t,eor) =
  begin 
    check_valid_type loc global_env [] t;
    (* Should only be one declaration of v although there may be many externs.*)
    (* So I shouldn't have to check this, but what the hell. *)
    let t_genv = Dict.lookup global_env.globals v in
    if not (unify global_env t t_genv)
    then begin
      terr loc ("Global "^(var2string v)^" has types "
		^(typ2string t)^" != "^(typ2string t_genv));
      raise Gcdfec.Exit
    end
  end

let check_top_decls ds =
  let global_env = initial_global_env ds in
  let rec check_typ_decls global_env ds =
    let check_typ_decl (d,loc) =
      let abort s = terr loc s; raise Gcdfec.Exit in
      begin match d with
    	FunDecl fd -> check_fundecl global_env loc fd
      | StructDecl sd -> check_structdecl global_env loc sd
      | UnionDecl ud -> check_uniondecl global_env loc ud
      | ExceptionDecl(v,s,t) ->
	  begin
	    if s = Abstract
	    then terr loc "Exceptions cannot be declared abstract.";
	    check_valid_type loc global_env [] t;
	    let t_g = Dict.lookup global_env.exceptions v in
	    if not (unify global_env t t_g)
	    then abort ("Exception "^(var2string v)^" has types "
			^(typ2string t)^" != "^(typ2string t_g));
	  end
      | ExternType (tn,vs,null) ->
	  begin
	    (* Check whether this is defined anywhere and defn matches. *)
	    if Dict.member global_env.abstracts tn
	    then begin
	      let (vs',null') = Dict.lookup global_env.abstracts tn in
	      if null <> null'
	      then abort ("External type "^(var2string tn)^" disagree on ?");
	      if not (list_match (=) vs vs') 
	      then abort ("Abstract type "^(var2string tn)
			  ^"disagree on type variables");
	      ()
	    end
	    else if Dict.member global_env.structs tn
	    then begin
	      let sd = Dict.lookup global_env.structs tn in
	      if not (list_match (=) vs sd.st_tyvars)
	      then abort ("Extern and struct declaration of "^(var2string tn)
			  ^" disagree on type variables");
	      if null <> sd.st_possibly_null 
	      then abort ("Extern and declaration of "^
			  (var2string tn)^"disagree on ?");
	      ()
	    end
	    else if Dict.member global_env.unions tn
	    then begin
	      let ud = Dict.lookup global_env.unions tn in
	      if not (list_match (=) vs ud.un_tyvars)
	      then abort ("Extern and union declaration of "^(var2string tn)
			  ^" disagree on type variables.");
	      if null <> ud.un_possibly_null 
	      then abort ("Extern and union declaration of "^(var2string tn)
			  ^" disagree on ?");
	      ()
	    end
	    else failwith "poptype.ml:check_top_decls: Unbound."
	  end
      | ExternVal (v,t) -> 
	  begin 
	    check_valid_type loc global_env [] t;
	    let t_g = Dict.lookup global_env.globals v in
	    if not (unify global_env t t_g)
	    then abort ("Extern has types "^(typ2string t)^" != "
			^(typ2string t_g))	     
	  end
      | GlobalDecl d ->  check_globaldecl loc global_env d
      | PrefixDecl _ -> failwith "poptype.ml:check_decl:prefixes should have been eliminated.\n"
      |	OpenDecl (prefix,ds) ->
	  check_typ_decls (open_prefix global_env prefix) ds
      end in
    List.iter check_typ_decl ds
  in
  let rec check_val_decls global_env ds =
    let check_val_decl (d,loc) =
      begin match d with
    	FunDecl fd -> 
	  let env = mk_env (fndecl2f_env fd) in
	  tcFunDecl global_env env loc fd
      | GlobalDecl d -> tcGlobalDecl loc global_env d
      | PrefixDecl _ -> failwith "poptype.ml:check_decl:prefixes should have been eliminated.\n"
      |	OpenDecl (prefix,ds) ->
	  check_val_decls (open_prefix global_env prefix) ds
      |	_ -> ()
      end in
    List.iter check_val_decl ds
  in
  check_typ_decls global_env ds;
  check_val_decls global_env ds;
  let ds = eliminate_open ds in
  (* PrefixDecl and OpenDecl have both been eliminated. *)
  let less_decl (d1,l1) (d2,l2) =
    let val_d d = 
      let val_sc sc =
	match sc with
	  Static -> 0 | Public -> 1 | Extern -> 2 | Abstract -> 3
      in
      let (vals,typs) = (1,0) in
      let (funs,exns,globals,extern_vals) = (0,1,2,3) in
      let (structs,unions,extern_types) = (0,1,2) in
      match d with
	FunDecl fd             -> (vals,fd.fn_name ,        funs,0)
      |	StructDecl sd          -> (typs,sd.st_name ,     structs,
				   val_sc sd.st_scope)
      |	UnionDecl ud           -> (typs, ud.un_name,      unions,
				   val_sc ud.un_scope)
      |	ExceptionDecl (v,sc,_) -> (vals,          v,        exns,
				   val_sc sc)
      |	ExternType (tn,_,_)    -> (typs,         tn,extern_types,
				   val_sc Extern)
      |	ExternVal (v,t)        -> (vals,          v, extern_vals,
				   val_sc Extern) 
      |	GlobalDecl(sc,v,_,_)   -> (vals,          v,     globals,
				   val_sc sc) 
      |	_ -> failwith "poptype.ml:less_decl:prefix/open should be eliminated.\n"
    in
    let vd1,vd2 = val_d d1, val_d d2 in
    vd1 <= vd2
  in
  let ds = Sort.list less_decl ds in
  let rec remove_dups ds =
    let id_decl d =
      match d with
	FunDecl fd            -> fd.fn_name
      |	StructDecl sd         -> sd.st_name
      | UnionDecl ud          -> ud.un_name
      |	ExceptionDecl (v,_,_) -> v
      |	ExternType(tn,_,_)    -> tn
      |	ExternVal(v,_)        -> v
      |	GlobalDecl(_,v,_,_)   -> v
      |	_ -> failwith "poptype.ml:remove_dups:prefix/open eliminated\n"
    in
    match ds with
      ((d1,_) as dl1)::(d2,_)::tl when (id_decl d1) = (id_decl d2) ->
	remove_dups (dl1::tl)
    | dl1 :: (dl2 :: tl' as tl) -> dl1 :: (remove_dups tl)
    | l -> l
  in
  (remove_dups ds,global_env)

(* This approach has the drawback that error messages will show funny names
   but it is the easiest to implement and seems appropriate at this time. *)
let eliminate_prefixes ds =
  let rec aux current_prefix_opt ds =
    let aux' = aux current_prefix_opt in
    match (current_prefix_opt,ds) with
      (None,(PrefixDecl (prefix,ds'),loc) :: tl) ->
	(aux (Some prefix) ds') @ (aux' tl)
    | (None,(OpenDecl (prefix,ds'),loc) :: tl) ->
	(OpenDecl (prefix,aux' ds'),loc) :: (aux' tl)
    | (None,hd::tl) -> hd :: (aux' tl)
    | (Some p,(hd,loc)::tl) ->
	begin
	  let loop td = td :: (aux' tl) in
	  match hd with
	    FunDecl fd -> loop
	      	(FunDecl { fd with fn_name = add_prefix p fd.fn_name}, loc)
	  | StructDecl sd -> loop
		(StructDecl { sd with st_name = add_prefix p sd.st_name }, loc)
	  | UnionDecl ud -> loop
		(UnionDecl { ud with un_name = add_prefix p ud.un_name }, loc)
	  | ExceptionDecl(v,s,t) -> loop
	      	(ExceptionDecl(add_prefix p v, s,t), loc)
	  | ExternType (tn,vs,b) -> loop
	      	(ExternType (add_prefix p tn,vs,b), loc)
	  | ExternVal (v,t) -> loop
	      	(ExternVal (add_prefix p v,t), loc)
	  | GlobalDecl (sc,v,t,eor) -> loop
	      	(GlobalDecl (sc, add_prefix p v,t,eor),loc)
	  | PrefixDecl (p',ds') ->
	      (aux (Some (add_prefix p p')) ds') @ (aux' tl)
	  | OpenDecl (p',ds') -> loop
		(OpenDecl (p',aux' ds'),loc)
  	end
    | (_,[]) -> []
  in
  aux None ds

let type_check ds = 
(* DEBUG *)
  let ds = 
    if !debug
    then
      let vs = 
	  ["callStackDump",    FnType([],VoidType,[]);
	   "callStackPush",    FnType([],VoidType,[StringType]);
      	   "callStackPop",     FnType([],VoidType,[]);
           "callStackSave",    FnType([],IntType(true,B4),[]);
      	   "callStackRestore", FnType([],VoidType,[IntType(true,B4)])] in
      let vs = List.map (fun (x,y) -> (ExternVal(x, y), dummy_location)) vs in
      List.append vs ds
    else
      ds in
(* END DEBUG *)
  let answer = check_top_decls (eliminate_prefixes ds) in
(* DEBUG *)
(* Have to get rid of these to avoid duplicate imports! *)
  let answer =
    if !debug
    then
      let ds,gs = answer in
      let ds = Utilities.filter 
	  (fun td ->
	    match td with
	      ExternVal("callStackDump",_),_    -> false
	    | ExternVal("callStackPush",_),_    -> false
	    | ExternVal("callStackPop",_),_     -> false
	    | ExternVal("callStackSave",_),_    -> false
	    | ExternVal("callStackRestore",_),_ -> false
	    | _ -> true)
	  ds in
      ds,gs
    else answer
  in
(* END DEBUG *)
  answer
