(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel, Frederick Smith   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

open Numtypes;;

type location = Gcdfec.seg
type var = string
type type_name = string
type field_name = string
type scope = Static | Public | Extern | Abstract
type capability = ReadOnly | ReadWrite
type size = B1 | B2 | B4

type var_class = Any | Option | Byte4

type const = 
    Int of int32
  | Bool of bool
  | String of string
  | Char of char
  | Null

type primop = 
    Plus | Times | TimesU | Minus | Div | DivU | Mod | ModU |
    Eq | Neq | Gt | GtU | Lt | LtU | Gte | GteU | Lte | LteU | Not |
    Bitnot | Bitand | Bitor | Bitxor | Bitlshift | Bitlrshift | Bitarshift |
    Size | Ord | Chr

type typ =
    VoidType
  | Evar of  var_class * (typ option ref)  (* used for unification only *)
  | VarType of var
  | IntType of bool * size (* true corresponds to signed. *)
  | BooleanType
  | StringType
  | CharType
  | ArrayType of typ * exp option (* optional size *)
  | FnType of var list * typ * (typ list)
  | TupleType of (typ list)
  | NamedType of type_name ref * (typ list)
  | ExnType

and raw_exp =
    Const of const
  | ConstArray of exp list * typ option
  | Var of var
  | Primop of primop * (exp list)
  | Conditional of exp * exp * exp
  | AssignOp of exp * (primop option) * exp  (* e.g., x+=1, x[i] *= 3 *)
  | FunCall of exp * ((typ list) option ref) * (exp list)
  | TypInst of exp * typ list
  | NewStruct of type_name * ((typ list) option ref) * (exp list)
  | StructMember of exp * field_name
  | NewUnion of type_name * ((typ list) option ref) * field_name * (exp option)
  | UnionMember of exp * field_name
  | NewTuple of (exp list)
  | TupleMember of exp * int
  | Subscript of exp * exp
(*  | NewArray of exp * exp *)
(* Cyclone *)
  | Codegen of fndecl
  | Fill of exp
(* End Cyclone *)
  | NewExn of var * (exp option)
  | Raise of exp
  | SeqExp of exp list
  | Nop
  | Cast of typ * exp
and exp = 
    { mutable exp_typ: typ option;
      mutable raw_exp: raw_exp;
      exp_loc: location
    } 

and raw_stmt =
    Skip
  | Exp of exp
  | Seq of stmt * stmt
  | Return of exp option
  | IfThenElse of exp * stmt * stmt
  | While of exp * stmt
  | Break of var option
  | Continue of var option
  | For of exp * exp * exp * stmt
  | IntSwitch of exp * (int32 * stmt) list * stmt
  | CharSwitch of exp * (char * stmt) list * stmt
  | UnionSwitch of exp * switch_arm list * (stmt option)
  | ExnSwitch of exp * switch_arm list * (stmt option)
  | Decl of var * typ * (exp option) ref * stmt
  | Label of var * stmt
(* Cyclone *)
  | Cut of stmt
  | Splice of stmt
(* End Cyclone *)
  | Do of stmt * exp
  | Try of stmt * var * stmt
and switch_arm = { arm_field: field_name;
		   arm_var: var option;
		   mutable arm_typ: typ;
		   arm_body: stmt
		 } 
and stmt = { mutable raw_stmt: raw_stmt; 
	     stmt_loc: location 
	   }
and fndecl = { fn_static: bool;
	       fn_name: var;
	       fn_tyvars: var list;
	       fn_ret_type: typ;
	       fn_args: (var * typ) list;
	       fn_body: stmt
	      } 

type struct_field = (field_name * capability * typ)
type union_field = (field_name * typ)


type structdecl = { st_scope: scope;
		    st_name: type_name;
		    st_tyvars: var list;
		    st_possibly_null: bool;
		    st_fields: struct_field list
		  }	
type uniondecl = { un_scope: scope;
		   un_name: type_name;
		   un_tyvars: var list;
		   un_possibly_null: bool;
		   un_fields: union_field list
		 }	

type raw_top_decl =
    FunDecl of fndecl
  | StructDecl of structdecl
  | UnionDecl of uniondecl
  | ExceptionDecl of (var * scope * typ)
  | ExternType of type_name * (var list) * bool (* bool indicates option *)
  | ExternVal of var * typ
  | GlobalDecl of (scope * var * typ * (exp option) ref)
  | PrefixDecl of var * top_decl list
  | OpenDecl of var * top_decl list
and top_decl = raw_top_decl * location

let var2string v =
  let len = String.length v in
  let i = ref 0 in
  let questions = 
    let count = ref 0 in
    while !i < len do
      if v.[!i] = '?' then incr count;
      incr i;
    done;
    !count
  in
  let w = String.create (len+questions) in
  let j = ref 0 in
  i:=0;
  while !i < len do
    if v.[!i]='?' 
    then begin w.[!j] <- ':'; w.[!j+1] <- ':'; incr j; end
    else w.[!j] <- v.[!i];
    incr j;
    incr i;
  done;
  w

(* should fix function and array types to be inline with the parsed syntax *)
let rec typ2string t = 
  match t with
    VoidType -> "void"
  | Evar(c,r) ->
      (match c,!r with
	Option,None -> "?%"
      |	_,None -> "%"
      |	_,Some t -> typ2string t)
  | VarType v -> "`" ^ v
  | IntType(sign,size) -> 
      (if sign then "" else "unsigned ")^
      (match size with B1 -> "byte" | B2 -> "short" | B4 -> "int")
  | BooleanType -> "bool"
  | StringType -> "string"
  | CharType -> "char"
  | ArrayType(t,None) -> (typ2string t)^"[]"
  | ArrayType(t,Some e) -> (typ2string t)^"[]" (* INCOMPLETE!! *)
  | FnType ([],t,ts) -> 
      (typ2string t)^" ("^types2string ts^")"
  | FnType (vs,t,ts) -> 
      (typ2string t)^" <"^(tyvars2string vs)^">("^types2string ts^")"
  | TupleType ts -> "*("^(types2string ts)^")"
  | NamedType (n,[]) -> var2string !n
  | NamedType (n,ts) -> (var2string !n) ^ "<" ^ (types2string ts)^ ">"
  | ExnType -> "exn"
and types2string ts = 
  match ts with
    [] -> ""
  | [t] -> typ2string t
  | t::ts -> ((typ2string t)^","^(types2string ts))
and tyvars2string vs =
  match vs with
    [] -> ""
  | [v] -> "'" ^ v
  | v::vs -> "'" ^ v ^ "," ^ (tyvars2string vs)

let eqtype t1 t2 = (t1 = t2)

let add_prefix prefix v = (prefix ^ "?" ^ v)

let size_leq s1 s2 = 
  begin match s1,s2 with
    B1,B1 | B1,B2 | B1,B4 | B2,B2 | B2,B4 | B4,B4 -> true
  | _,_ -> false
  end
