(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,Frederick Smith    *)
(*     Dan Grossman                                                   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

(* Prior to Release 1.7, this was part of Popcompile; now it is opened by
   that module.*)

open Numtypes
module T    = Tal
module P    = Popsyntax 
module PT   = Poptype 
module Id   = Identifier
module Peep = Poppeep
module X    = Poperr

type typ   = Popsyntax.typ
type con   = Tal.con
type int32 = Numtypes.int32
type id    = Id.identifier

(**************************** Exceptions ****************)
exception Unimplemented of string
exception Impossible    of string
exception Void_Type (* Void_Types have no TAL constructor *)

let unimpl s = raise (Unimplemented ("popcompile: "^s))
let impos  s = raise (Impossible    ("popcompile: "^s))
let deSome x = match x with Some(v) -> v | None -> impos "deSome"

let to_i32      = int_to_int32 
let dc  = T.defcon

(*********************** Convert to TAL Identifiers ************)
let tid_val  p = Id.id_of_string ("_" ^ p)
let tid_mem  p = Id.id_of_string (p   ^ "?mem")
let tid_type p = Id.id_of_string (p   ^ "?")
let tid_tyv  p = Id.id_of_string ("v" ^ p)
let tid_exn  p = Id.id_of_string (p   ^ "?exn")

(********************** Type Variable Utilities ***************)
let tyvars_to_cons tyvars = List.map (fun v -> T.cvar (tid_tyv v)) tyvars
let tyvars_to_lam tyvars con =  
  List.fold_right (fun v -> T.clam (tid_tyv v) T.k4byte) tyvars con
let rec tyvars_to_kind ts k =
  List.fold_right (fun t k -> T.Karrow(T.k4byte,k)) ts k

(********************** Cons and Abbrevs **********************)
let stack1_v              = Id.id_of_string "s1" 
let stack2_v              = Id.id_of_string "s2" 
let array_size_var        = Id.id_of_string "?sz"
let array_abbrev_var      = Id.id_of_string "?arr"
let string_abbrev_var     = Id.id_of_string "?str"
let handle_abbrev_var     = Id.id_of_string "?H"
let handler_abbrev_var    = Id.id_of_string "?Ha"
let exn_stack_abbrev_var  = Id.id_of_string "?E"
let exn_var               = Id.id_of_string "?exn"
let exnname_var           = Id.id_of_string "?exnname"
let exnname_arg_var       = Id.id_new       "c"
let stack_abbrev_var      = Id.id_of_string "?S"
let stack_void_abbrev_var = Id.id_of_string "?Sv"
let bogus_option_var      = Id.id_new       "bogus_option"
(* Cyclone *) let cg_v    = Id.id_of_string "cg" (* End Cyclone *)

let stack1_c = T.cvar stack1_v
let stack2_c = T.cvar stack2_v
(* Cyclone *) let cg_c     = T.cvar cg_v (* End Cyclone *)

let exn_con           = T.cvar exn_var
let exnname_arg_con   = T.cvar exnname_arg_var
let exnname_con'      = T.cvar exnname_var
let exnname_con    c  = T.capp exnname_con' c
let int_con           = T.cbyte4
let bool_con          = T.chptr [i32_0;i32_1] None None
let char_con          = int_con
let string_con        = T.cvar string_abbrev_var
let opt_con        c  = T.chptr [i32_0] (Some c) None
let array_con      c  = T.capp (T.cvar array_abbrev_var) c
let exn_stack_con  s  =
  (* Cyclone *) T.capp ((* End Cyclone *)
    T.capp (T.cvar exn_stack_abbrev_var) s
  (* Cyclone *) ) cg_c (* End Cyclone *)
let handle_con     s  =
  (* Cyclone *) T.capp ((* End Cyclone *) T.capp (T.cvar handle_abbrev_var) s
  (* Cyclone *) ) cg_c (* End Cyclone *)
let array_real_con c  = T.carray_s array_size_var (T.cfield c T.ReadWrite)
let handler_con s1 s2 = T.capp (T.capp (T.cvar handler_abbrev_var) s1) s2
let bogus_option_con  = T.chptr [i32_0] None None

let array_abbrev =
  let c = Id.id_of_string "c" in
  T.clam c T.k4byte (array_real_con (T.cvar c))

let string_abbrev = array_real_con (T.pcbytes T.Byte1)

let exnname_abbrev = 
  T.clam exnname_arg_var T.Kmem
    (T.chptr [] (Some (T.cprod [T.pcjunk i32_4]))
       (Some (exnname_arg_con,T.ReadWrite)))

let exn_body =
  T.cprod_b [ T.cfield (exnname_con exnname_arg_con) T.Read; exnname_arg_con ]

let exn_abbrev = T.cexist exnname_arg_var T.Kmem exn_body

(* Type of the handler as seen in function types. *)
let handle_abbrev = 
  (* Cyclone *)
  let tla_con = T.cprod_b [T.cfield cg_c T.ReadWrite] in
  (* End Cyclone *)
  let sv  = Id.id_of_string "s" in
  let svc = T.cvar sv           in
  T.clam sv T.Kstack 
  (* Cyclone *)
    (T.clam cg_v T.Ktstack
  (* End Cyclone *)
       (T.ccode_l_tla [T.Eax,exn_con; T.Esp,T.csptr svc]
         (* Cyclone *) tla_con (* End Cyclone *)))

(* Abbreviation for shape of stack pointer in EBP *)
let exn_stack_abbrev = 
  let sv  = Id.id_of_string "s" in
  let svc = T.cvar sv           in
  T.clam sv T.Kstack
    (* Cyclone *)
    (T.clam cg_v T.Ktstack
    (* End Cyclone *)
       (T.ccons (handle_con svc) svc))

(* Abbreviate the stack shape common to all labels.
   We need two variables because void functions return nothing in EAX. *)

let (stack_abbrev, stack_abbrev_void) =
  let rv    = Id.id_of_string "?ret" in let rvc  = T.cvar rv    in
  let s1v   = Id.id_of_string "?s1"  in let s1vc = T.cvar s1v   in
  let s2v   = Id.id_of_string "?s2"  in let s2vc = T.cvar s2v   in
  let (abbrev,abbrev_void) = 
    let (ret_state,ret_state_void) =
      let ret_ctxt = 
	let ebp_stk = exn_stack_con s2vc in
	[T.Esp, T.csptr (T.cappend s1vc ebp_stk);
	 T.Ebp, T.csptr ebp_stk]
      in
      (T.ccode_l_tla ((T.Eax, rvc):: ret_ctxt)
         (* Cyclone *) (T.cprod_b [T.cfield cg_c T.ReadWrite]) (* End Cyclone *)
         ,
       T.ccode_l_tla ret_ctxt
        (* Cyclone *) (T.cprod_b [T.cfield cg_c T.ReadWrite]) (* End Cyclone *))
    in
    let aux r =
      T.clam s1v T.Kstack
        (T.clam s2v T.Kstack
           (* Cyclone *) (T.clam cg_v T.Ktstack (* End Cyclone *)
	      (T.ccons r s1vc))) in
    (T.clam rv T.k4byte (aux ret_state), aux ret_state_void) in
  (abbrev, abbrev_void)

(* stack has s1 @ h(s2) @ s2 *)
let stack_con ret_con_opt s1 s2 (* Cyclone *) cg (* End Cyclone *) =
  match ret_con_opt with
    Some ret_con -> 
      (* Cyclone *) (T.capp (* End Cyclone *)
	  (T.capp (T.capp (T.capp (T.cvar stack_abbrev_var) ret_con) s1) s2)
      (* Cyclone *) cg)     (* End Cyclone *)
  | None -> 
      (* Cyclone *) (T.capp (* End Cyclone *)
         (T.capp (T.capp (T.cvar stack_void_abbrev_var) s1) s2)
      (* Cyclone *) cg)     (* End Cyclone *)

let handler_abbrev =
  let s1v = Id.id_of_string "?s1" in let s1vc = T.cvar s1v in
  let s2v = Id.id_of_string "?s2" in let s2vc = T.cvar s2v in
  let esp_stk = T.ccons (T.csptr (exn_stack_con s2vc))
                        (T.cappend s1vc (exn_stack_con s2vc)) in
  T.clam s1v T.Kstack (T.clam s2v T.Kstack esp_stk)

(************************ Con utilities *********************)
let app_cons     c cs = List.fold_left T.capp c cs
let raw_name_con n cs = app_cons     (T.clab   n) cs
let name_con     n cs = raw_name_con (tid_type n) cs
let mem_name_con n cs = raw_name_con (tid_mem  n) cs

let close_code tyvars code = (* Given type, attach forall tyvars,s1,s2,cg *)
  List.fold_right (fun v c -> T.cforall (tid_tyv v) T.k4byte c) tyvars
    (T.cforall stack1_v T.Kstack
       (T.cforall stack2_v T.Kstack
          (* Cyclone *) (T.cforall cg_v T.Ktstack (* End Cyclone *)
             code)))

(************************** Type Translation ******************)

let rec typ2con t = (* Translate popcorn type to Tal type (i.e. con) *)
  match t with 
    P.VoidType -> raise Void_Type
  | P.Evar(c,r)   ->
	(* an unconstrained evar can be instantiated with any 4-byte type,
         * we choose int to be simple *)
      begin match !r with
	Some t -> typ2con t
      | None   -> 
	  begin match c with
	    P.Any | P.Byte4 -> (r := Some(P.IntType(true,P.B4)); int_con)
	  | P.Option -> T.clab bogus_option_var
	  end
      end
  | P.VarType v    -> T.cvar (tid_tyv v)
  | P.IntType _    -> int_con
  | P.BooleanType  -> bool_con
  | P.StringType   -> string_con
  | P.CharType     -> char_con
  | P.ExnType      -> exn_con
  | P.ArrayType(t',_) -> array_con (typ2con t')
  | P.FnType    (vs,t',tl) -> let (res,_,_) = fun_con vs t' tl in res
  | P.NamedType (n,ts)     -> name_con !n (types2cons ts)
  | P.TupleType ts -> 
      T.cprod_b (List.map (fun t -> T.cfield (typ2con t) T.ReadWrite) ts)

and fun_con vs ret_typ params = (* vs are type arguments. *)
    (* Calling convention pushes args from right to left, return value in EAX *)
    let rec map params a =
      match params with
	[]     -> a 
      | hd::tl ->(try map tl ((typ2con hd) :: a) with Void_Type -> (map tl a))in
    let p_cons      = List.rev (map params [])                          in
    let stack_p     = List.fold_right T.ccons p_cons stack1_c           in
    let ret_con_opt = try Some (typ2con ret_typ) with Void_Type -> None in
    let stack = stack_con ret_con_opt stack_p stack2_c
          (* Cyclone *) cg_c (* End Cyclone *)  in
    let exn_stack   = exn_stack_con stack2_c in
    let fun_state   =
      T.ccode_l_tla [ (T.Esp,T.csptr (T.cappend stack exn_stack));
		      (T.Ebp,T.csptr exn_stack)]
         (* Cyclone *) (T.cprod_b [T.cfield cg_c T.ReadWrite]) (* End Cyclone *)
    in
    let lab_con = close_code vs fun_state in
    (lab_con,stack,stack2_c)
and types2cons ts = List.map typ2con ts

let mallocarg2con m =
  let rec aux m =
    match m with
      T.Mprod  mas -> T.cprod (List.map aux mas)
    | T.Mfield c   -> T.cfield c T.Uninit
    | T.Mbytearray (scale,size) ->
 	T.carray (T.pcint size) (T.cfield (T.pcbytes scale) T.ReadWrite) in
  aux m

let con2field c = T.cfield c T.Read

let bogus_option_con_block = (bogus_option_var,T.k4byte,bogus_option_con)

 (* To turn off annotation hack, use this instead of following and
    modify popcompile.ml popcomptypes.mli as directed there *)
(* let fun_coercion stack1 stack2 cg cons =
   cg::(T.Tapp stack2)::(T.Tapp stack1)::
   (List.rev (List.map (fun c -> T.Tapp c) cons)) *)
let fun_coercion i1 i2 bottom cg cons =
  cg
  ::(T.Tapp (T.StackTail(T.Ebp,1)))
  ::(T.Tapp (T.StackSlice(T.Esp, i1, i2, bottom)))
  ::(List.rev (List.map (fun c -> T.Tapp (T.Con c)) cons))

(* Functions are already pointers so we don't need indirection to get the 
   * actual value.  In all other cases we do. *)
let needs_indirect t = match t with P.FnType _ -> false | _ -> true

let rec get_name t = (* get the name of a named type *)
  let aux r = 
    match r with
      None   -> impos "get_name: uninstantiated evar"
    | Some t -> get_name t
  in
  match t with
    P.NamedType  (n,ts) -> !n
  | P.Evar       (_,r)  -> aux !r 
  | _                   -> impos "get_name: unnamed type"
      
(**************************** Struct Information *************)
type struct_info =
    { struct_null:     bool;
      struct_con:      T.con;
      struct_mem_con:  T.con;
      struct_mem_kind: T.kind;
      struct_kind:     T.kind;
      sfield_infos:    (P.field_name * int * T.con * T.variance) list
    } 

let info_structdecl st = (* : P.structdecl -> struct_info *)
  let scope,name,tyvars,null,fields =
    st.P.st_scope,st.P.st_name,st.P.st_tyvars,st.P.st_possibly_null,
    st.P.st_fields in
    
  let tycons = tyvars_to_cons tyvars          in
  let tylam  = tyvars_to_lam  tyvars          in    
  let kind   = tyvars_to_kind tyvars T.k4byte in

  let mem_kind = 
    let base_kind = T.Kmemi (to_i32 (4 * (List.length fields))) in
    tyvars_to_kind tyvars base_kind
  in
  let offset = ref 0 in
  let post_incr  i = (let j = !i in i:=j+1;j) in
  let info_field typ2con sf =
    let (n,cap,t) = sf        in
    let con       = typ2con t in
    let t_cap = 
      match cap with
	P.ReadOnly  -> T.Read
      | P.ReadWrite -> T.ReadWrite in
    n,post_incr offset,con,t_cap in
  let field_infos = List.map (info_field typ2con) fields in

  let aux (_,_,con,cap) = T.cfield con cap                   in
  let struct_mem_con'   = T.cprod (List.map aux field_infos) in

  let struct_con = tylam (T.chptr 
			    (if null then [i32_0] else []) 
			    (Some (mem_name_con name tycons))
			    None) in
  let struct_mem_con = tylam struct_mem_con' in
  { sfield_infos    = field_infos;
    struct_null     = null;
    struct_con      = struct_con;
    struct_mem_con  = struct_mem_con;
    struct_kind     = kind;
    struct_mem_kind = mem_kind
  }
let struct_null  s_info = s_info.struct_null 
let struct_t     s_info = s_info.struct_con,     s_info.struct_kind
let struct_mem_t s_info = s_info.struct_mem_con, s_info.struct_mem_kind

let struct_field_offset s_info f =
  let rec aux fields =
    match fields with
      [] -> impos "struct_field_offset: No such field"
    |	((n,o,_,_)::tl) -> if n=f then o else (aux tl) in
  aux s_info.sfield_infos

let roll_struct s_info name cs = (* Coerce a value to a struct name. *)
  let coerce_to_mem = T.Roll      (app_cons s_info.struct_con cs ) in
  let coerce_to_sum = T.RollTosum (name_con name cs)               in
  let coerce        = T.Roll      (name_con name cs)               in
  if struct_null s_info
  then [coerce_to_sum;coerce_to_mem]
  else [coerce;coerce_to_mem]

(******************** Union Information *****************)
type union_info = 
    { union_tyvars: id list;  
      union_null:   bool;
      union_con:    T.con;
      union_kind:   T.kind;
      void_infos:   (P.field_name * int32)           list;
      value_infos:  (P.field_name * (int32 * T.con)) list
    } 

let info_uniondecl ud = (* : P.uniondecl -> union_info *)  
  if ud.P.un_possibly_null
  then unimpl "info_uniondecl : option unions";
  let scope,name,tyvars,union_fields = 
    ud.P.un_scope, ud.P.un_name, ud.P.un_tyvars, ud.P.un_fields in

  let tylam = tyvars_to_lam  tyvars          in
  let kind  = tyvars_to_kind tyvars T.k4byte in

  let rec sort_fields fields void_fields value_fields =
    match fields with
      [] -> (List.rev void_fields, List.rev value_fields)
    | (f,P.VoidType)::tl -> sort_fields tl (f::void_fields) value_fields
    | (f,t)::tl          -> sort_fields tl void_fields ((f,t)::value_fields) in
  let void_fields, value_fields = sort_fields union_fields [] [] in
  let value_field_con (f,t)     = (f,typ2con t)                  in
  let value_field_cons = List.map value_field_con value_fields   in

  let i = ref 1 in
  let post_incr i = (let j = !i in  i:=j+1;to_i32 j)                in
  let void_infos  = List.map (fun f -> (f,post_incr i)) void_fields in
  let value_info (f,con) = let j = post_incr i in (f,(j,con))       in
  let value_infos =
    try
      List.map value_info value_field_cons
    with Void_Type -> impos "info_uniondecl: void_type" in
  let tags = List.map snd void_infos                    in
  let sum  = 
    match value_infos with
      [] -> None
    | _  ->
	let aux (n,(tag,con)) =
	  T.cprod [ T.cfield (T.csing (T.pcint tag)) T.Read; con2field con ] in
	let vars = List.map aux value_infos in
	Some (T.csum vars) in
  { union_tyvars = List.map tid_tyv tyvars; 
    void_infos   = void_infos;
    value_infos  = value_infos;
    union_con    = tylam (T.chptr tags sum None);
    union_kind   = kind;
    union_null   = ud.P.un_possibly_null
  } 	
let union_t              u_info        = u_info.union_con, u_info.union_kind
let union_void_tag_assoc u_info fn     = List.assoc fn u_info.void_infos
let union_val_tag_assoc  u_info fn     = List.assoc fn u_info.value_infos
let union_num_voids      u_info        = List.length u_info.void_infos
let union_instantiate    u_info con ts = (* would be nice not to need Talcon *)
  Talcon.substs 
    (Dict.inserts (Dict.empty Id.id_compare)
       (List.combine u_info.union_tyvars ts))
    con

(************************** Other **************)
let rec compress t =
  let aux r = 
    match !r with 
      None    -> t
    | Some t' -> let t' = compress t' in r := Some t'; t'
  in
  match t with
    P.Evar   (_,r) -> aux r 
  | _              -> t

let bool b = (T.Immed (if b then i32_1 else i32_0),[T.Tosum bool_con])

let rec is_value e =
  match e.P.raw_exp with
    P.Const        _      -> true
  | P.Var          _      -> true
  | P.StructMember (e',_) -> is_value e'
  | P.TupleMember  (e',i) -> is_value e'
  | P.NewUnion(nt,_,f,eo) -> (match eo with None -> true | _    -> false)
  | P.Primop     (p,[e']) ->
      (is_value e' & 
       match p with
	 P.Bitnot -> true
       | P.Size   -> true
       | P.Ord    -> true
       | P.Chr    -> true
       | _        -> false)
  | _ -> false

let exp2typ e = deSome e.P.exp_typ
let exp2con e = typ2con (exp2typ e)

