(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,Frederick Smith    *)
(*     Dan Grossman                                                   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

(* We no longer import tal.tali, so you have to extern what you want.
   If we have main, this means we must import tal_exit *)

(* NOTES:
 * Option unions are NOT supported
 * Single value unions are not special-cased
 *)

(* Call map -- here's a top down view of the code generation functions,
 *             where indentation means uses as a helper
 *
 * code_gen  
 *    cg_all
 *       cg_typedecls 
 *       cg_global_decls
 *       cg_exn_decls
 *       cg_fundecl
 *          cg_stmt
 *             cg_exp (also used by cg_bool_exp)
 *                cg_lhs_exp (call cg_exp)
 *                cg_bop (also used by cg_lhs_exp)
 *             cg_bool_exp
 *             cg_fill_holes (also called by cg_exp)
 *    postprocess
 *)  

(* File contents: 
 *  (Note type compilation and env manipulation are now in other modules.)
 *  o Modules and such
 *  o type definitions
 *  o STATE 
 *  o Label names, and main information
 *  o Emit utilities
 *  o Push and pop utilities
 *  o Operator Translation
 *  o Code Generation::Main (bottom up of call map described above)
 *)

let print_comments = ref true
let peephole = true
let debug_null = ref false

(*********************************Modules and such ****************************)
open Popcomptypes
open Popcompenv
open Numtypes
module T    = Tal
module P    = Popsyntax 
module PT   = Poptype 
module Id   = Identifier
module Peep = Poppeep
module X    = Poperr

type id = Id.identifier

let dc          = T.defcon
let to_i32      = int_to_int32 
let nonsense_id = Id.id_new "nonsense"

let deSome x = match x with Some v -> v | None -> impos "deSome"

(***************************************Types**********************************)
type stack_or_reg = TopStack | Stack of int | Register of T.reg

(* Cyclone *)
type code_mark =
    M_Label       of id * T.con option
  | M_ISeq        of T.instruction array
  | M_TypeAbbrev  of id * T.con
      (* Delimiters for templates *)
  | M_TemplateBeg of id * id * T.con * fenv
  | M_TemplateEnd of T.con
      (* Expression to fill a hole *)
  | M_Fill        of id * P.exp
      (* Jumps from inside templates might need to be patched.
      ** First id is the target, second id is id of the template
      ** containing the jump, if the jump is an inter-template jump. *)
  | M_Jmp   of id * (id * id) option ref * T.con * T.coercion list
  | M_Jcc   of T.condition * id * (id * id) option ref * T.con * T.coercion list
  | M_Btagi of T.reg * Numtypes.int32 * id T.coerce * T.condition
        * (id * id) option ref * T.con
  | M_Btagvar of T.reg * Numtypes.int32 * Numtypes.int32 * id T.coerce
        * T.condition * (id * id) option ref * T.con
(* End Cyclone *)

(**************************************** STATE *******************************)
type gen_state = (* Encapsulates the state of the code generator. *)
    { mutable export_cons : T.int_con     list;
      mutable export_vals : (id * T.con)  list;
      mutable import_cons : T.int_con     list;
      mutable import_vals : (id * T.con)  list;
      mutable code_marks  : code_mark     list;
      mutable data_blocks : T.data_block  list;
      mutable instrs      : T.instruction list; (* Instr in  current block *)
      mutable label       : id * T.con option;  (* Label for current block *)
      mutable module_string : id (* label for name of module *)
    } 

(* Cyclone *)
(* Eventually put this in gen_state *)
let forward_itjs  = Hashtbl.create 11 (*map labels to inter-template-jump info*)
let tmpl_of_label = Hashtbl.create 11 (*map labels to templates they occur in *)
(* End Cyclone *)

let gen_state = 
  { export_cons = []; export_vals = [];
    import_cons = []; import_vals = [];
    code_marks  = []; data_blocks = [];
    instrs      = []; label = (nonsense_id, None);
    module_string = nonsense_id;
  } 

let gen_export_con c = gen_state.export_cons <- (c :: gen_state.export_cons)
let gen_export_val v = gen_state.export_vals <- (v :: gen_state.export_vals)
let gen_import_con c = gen_state.import_cons <- (c :: gen_state.import_cons)
let gen_import_val v = gen_state.import_vals <- (v :: gen_state.import_vals)
let gen_data_block d = gen_state.data_blocks <- (d :: gen_state.data_blocks)

let in_template() = 
  let rec aux marks depth = 
    match marks with
      [] -> depth <> 0
    | M_TemplateBeg(_,_,_,_)::tl -> aux tl (depth+1)
    | (M_TemplateEnd _)::tl -> aux tl (depth - 1)
    | _::tl -> aux tl depth
  in aux gen_state.code_marks 0

(* Pushes the block on the current list of code blocks and
 * starts a new block.  Does some simple peephole optimization.
 *)
let flush_code () =
  match gen_state.instrs with
    [] -> ()
  | ilist ->
      let opt = if peephole then Peep.optimize (in_template())
                else (fun x -> x) in
      let iv = Array.of_list(opt(List.rev ilist)) in
      gen_state.code_marks <- ((M_ISeq iv) :: gen_state.code_marks);
      gen_state.instrs     <- []

let gen_code_mark m =
  flush_code();
  gen_state.code_marks <- (m :: gen_state.code_marks)
let gen_set_label  (l,c)     = gen_code_mark(M_Label(l,c))
let gen_code_block (l,c,iv)  = gen_set_label (l,c); gen_code_mark(M_ISeq iv)
let emit           i         = gen_state.instrs <- (i::gen_state.instrs)
let emit_label     l con_opt = flush_code (); gen_set_label(l,con_opt)
let emit_comment   s         = if !print_comments then emit(T.Comment("\t"^s))

let add_string s = 
  let lab = Id.id_new "string"                                 in
  let len = to_i32 (String.length s)                           in
  let l   = if (String.length s) = 0 then [] else [T.Dbytes s] in 
  gen_data_block (lab,None,
		  (T.D4bytes (len,[]):: T.Dup ::l,
		   [ T.Pack    (T.pcint len,string_con);
		     T.Toarray (i32_4,0,T.cfield (T.pcbytes T.Byte1)
				  T.ReadWrite)]));
  lab

let reset_generator module_name = 
  let gs = gen_state in
  gs.export_cons<-[]; gs.export_vals<-[]; gs.import_cons<-[]; 
  gs.import_vals<-[]; gs.code_marks <-[]; gs.data_blocks<-[];
  gs.instrs     <-[]; gs.label<-(nonsense_id, None);
  if (!debug_null) then
      gs.module_string <- add_string module_name;
(* Cyclone *)
  Hashtbl.clear forward_itjs;
  Hashtbl.clear tmpl_of_label;
(* End Cyclone *)
  ()
(********************** Label names and main information **********************)
let pop_exit_label    = Id.id_of_string "_tal_exit"
let new_array_label   = Id.id_of_string "_new_array" 
let never_null_label  = Id.id_of_string "_pop_never_null"
let debug_never_null_label = Id.id_of_string "_pop_debug_never_null"
let exn_handler_label = Id.id_of_string "_pop_exn_handler"
let main_label        = Id.id_of_string "__main$"

let main_id = "main"

let pop_exit_con = 
  T.cforall stack1_v T.Kstack 
    (T.ccode_l [T.Esp, T.csptr (T.ccons T.cbyte4 (T.cvar stack1_v))])

let (tal_main_code_block,tal_main_con) = 
  let tal_main_label = Id.id_of_string "_tal_main" in
  let b4rw = T.cfield T.cbyte4 T.ReadWrite in
  let tal_main_con =
    T.ccode_l_tla [(T.Esp,T.csptr T.cempty)] (T.cprod_b [
(* Cyclone *)
					      T.cfield T.ctempty T.ReadWrite
(* End Cyclone *)
					      ;b4rw;b4rw;b4rw])
  in
  let tal_main_code = 
    [|T.Push (T.Addr exn_handler_label,[]);
      T.Mov (T.Reg T.Ebp, (T.Reg T.Esp,[]));
      T.Call (T.Addr main_label,[
(* Cyclone *)
              T.Tapp (T.Con (dc T.Ctempty));
(* End Cyclone *)
              T.Tapp (T.Con T.cempty);
              T.Tapp (T.Con T.cempty)]);
      T.Pop (T.Reg T.Ebx); 
      T.Push (T.Immed i32_0,[]);
      T.Jmp (T.Addr pop_exit_label,[T.Tapp (T.Con T.cempty)])
    |] 
  in
  ((tal_main_label,Some tal_main_con,tal_main_code),tal_main_con)

(*********************************Emit Utilities*****************************)

let branch_coercion env = 
(* Cyclone *)
  (T.Tapp (T.Con cg_c))::
(* End Cyclone *)
  (T.Tapp (T.Con stack2_c))::(T.Tapp (T.Con stack1_c))::
  (List.rev (List.map 
	       (fun v -> T.Tapp(T.Con (T.cvar(tid_tyv v)))) (env_tyvars env)))

let exncase_coercion env =
  (branch_coercion env) @ [T.Tapp (T.Con exnname_arg_con)]

let fallthru env = 
  let tcons = List.map (fun v -> T.cvar(tid_tyv v)) (env_tyvars env) in
  emit(T.Fallthru (tcons @ [stack1_c;stack2_c
(* Cyclone *)
                               ;cg_c
(* End Cyclone *)
                           ]))

let exncase_fallthru env =
  let tcons = List.map (fun v -> T.cvar(tid_tyv v)) (env_tyvars env) in
  emit(T.Fallthru (exnname_arg_con::
		   (tcons @ [stack1_c;stack2_c
(* Cyclone *)
                               ;cg_c
(* End Cyclone *)
                           ])))

let mk_label_con env =
  let exn_part = exn_stack_con (env_stack2 env) in
  let stack_state = 
    let esp_set = T.rs_set_reg (env_regs env) T.Esp 
	(T.csptr(T.cappend (env_stack1 env) exn_part)) in
    T.rs_set_reg esp_set T.Ebp (T.csptr exn_part) in
  close_code (env_tyvars env) (T.ccode stack_state)

let mk_exncase_label_con env =
  T.cforall exnname_arg_var T.Kmem (mk_label_con env)

(* In Cyclone jumps between templates have to be patched.
   Therefore we added emit_jmp, emit_jcc, emit_btagi,
   and emit_btagvar. *)
let emit_jmp env label coercion =
(* Cyclone *)
  if not(outermost env)
  then
    (* Make sure we have a coercion (not just an empty coercion)
       in case the jump is an inter-template jump. *)
    let coercion   = branch_coercion env in
    let target_con = mk_label_con    env in
    gen_code_mark(M_Jmp(label,ref None,target_con,coercion))
  else
(* End Cyclone *)
    emit (T.Jmp(T.Addr label,coercion))

let emit_jcc env cc label coercion =
(* Cyclone *)
  if not(outermost env)
  then
    let coercion   = branch_coercion env in
    let target_con = mk_label_con    env in
    gen_code_mark(M_Jcc(cc,label,ref None,target_con,coercion))
  else
(* End Cyclone *)
    emit (T.Jcc(cc,(label,coercion)))

let emit_btagi (reg,i,(label,coercion),condition) env =
(* Cyclone *)
  if not(outermost env)
  then
    let coercion   = branch_coercion env in
    let target_con = mk_label_con    env in
    gen_code_mark(M_Btagi(reg,i,(label,coercion),condition,ref None,target_con))
  else
(* End Cyclone *)
    emit (T.Btagi(reg,i,(label,coercion),condition))

let emit_btagvar (reg,i1,i2,(label,coercion),condition) env =
(* Cyclone *)
  if not(outermost env)
  then
    let coercion   = branch_coercion env in
    let target_con = mk_label_con    env in
    gen_code_mark
      (M_Btagvar(reg,i1,i2,(label,coercion),condition,ref None,target_con))
  else
(* End Cyclone *)
    emit (T.Btagvar(reg,i1,i2,(label,coercion),condition))

let emit_lab env l =  emit_label l (Some (mk_label_con env))
let emit_lab_nocon env l = 
(* Cyclone *) (* Dan: Can this be changed -- union makes unnecessary cons *)
  if not(outermost env)
  then (fallthru env; emit_lab env l)
  else
(* End Cyclone *)
    emit_label l None
let emit_exncase env l = emit_label l (Some (mk_exncase_label_con env))

let emit_handler env l =
(* Cyclone *)
  let tla_con = deSome (T.rs_get_tla (env_regs env)) in
(* End Cyclone *)
  let code = 
    T.ccode_l_tla [ T.Eax, exn_con;
		T.Esp, T.csptr(handler_con (env_stack1 env) (env_stack2 env))]
(* Cyclone *)
      tla_con
(* End Cyclone *)
  in
  let l_con = close_code (env_tyvars env) code in
  emit_label l (Some l_con)


(* Check that r != null, raise null exception if r == null. *)
let check_not_null pos_loc r env = 
  if !debug_null then
    begin
      let lab = Id.id_new "not_null" in
      let pos = Numtypes.int_to_int32 (Gcdfec.seg_start pos_loc) in
      (* generate type instantiation from environment *)
(* let inst_c = T.cappend (env_stack1 env) (exn_stack_con (env_stack2 env))
      in 
*)
      emit_btagi (r,i32_0,(lab,[]),T.NotEq) env;   (* if not null, goto lab *)
      emit(T.Push(T.Addr gen_state.module_string,[]));  (* push module name *)
      emit(T.Push(T.Immed pos,[]));              (* push error position *)
      (* we need to push another argument for the "return address" 
       * because pop_debug_never_null is a C function expecting it. *)
      emit(T.Push(T.Immed pos,[]));              
      emit(T.Jmp(T.Addr debug_never_null_label,
		 [T.Tapp (T.StackTail (T.Esp,3))]));
      emit_lab_nocon env lab                     (* lab: *)
    end
  else 
    emit_btagi (r,i32_0,(never_null_label,[]),T.Eq) env
(********************** Push and pop utilities **************************)

let push_con env reg reg_con = 
  emit (T.Push (T.Reg reg, []));
  env_push_con env reg_con

let push env reg t = push_con env reg (typ2con t) 

let pop reg = emit(T.Pop (T.Reg reg))

let pop_free i = (* pop freeing i stack slots. *)
  (if i>0 then emit(T.ArithBin(T.Add,T.Reg T.Esp,T.Immed (to_i32 (4*i)))))

(* peek - load the ith word from the top of the stack without popping. *)
let peek reg i = emit(T.Mov (T.Reg reg,(T.Prjr ((T.Esp,[]),to_i32 (4*i)),[])))

(* Cyclone *)
let abbrev_con con =
  match con.T.rcon with
    T.Cvar _  -> con (* Don't bother to abbreviate a variable *)
  | T.Cprim _ -> con (* or a primitive type *)
  | _ ->
      let abbrev_var = Id.id_new "t"         in 
      let abbrev_con = dc(T.Cvar abbrev_var) in
      gen_code_mark(M_TypeAbbrev(abbrev_var,con));
      abbrev_con

let cg_push env ti ls hs post_con_opt =
  let cg_type =
    match (get_cyclone_con env).T.rcon with
      T.Ctcons({T.rcon=T.Ctrgn(pre,post,t)},tl) ->
        dc(T.Ctcons(dc(T.Ctrgn(pre,post_con_opt,(ti,ls,hs)::t)),
                    tl))
    | _ -> impos "cg_push: bad type for ECG"
  in put_vis_cg_type env cg_type

let cg_pop_hole env h =
  let cg_type =
    match (get_cyclone_con env).T.rcon with
      T.Ctcons({T.rcon=T.Ctrgn(pre,post,(ti,ls,hs)::t)},tl) ->
        let hs' =
          List.concat
            (List.map (fun (l,c) -> if l=h then [] else [(l,c)]) hs) in
        dc(T.Ctcons(dc(T.Ctrgn(pre,post,(ti,ls,hs')::t)), tl))
    | _ -> impos "cg_pop_hole: bad type for ECG"
  in put_vis_cg_type env cg_type

let cg_pop_free env n =
  let dumped_tmpls =
    match (get_cyclone_con env).T.rcon with
      T.Ctcons({T.rcon=T.Ctrgn(pre,post,t)},tl) -> t
    | _ -> impos "cg_pop_free: bad type for ECG" in
  let rec aux n l =
    if n<=0 then () else 
      match l with
        (ti,_,_)::tl -> emit(T.CgForget ti); aux (n-1) tl
      | _ -> impos "Popcompile.cg_pop_free: bad type for ECG 2" in
  aux n dumped_tmpls
(* End Cyclone *)


(************************* Operator Translation *****************************)
let arith_op p =
  match p with
    P.Plus   -> T.Add | P.Times  -> T.Imul2
  | P.Minus  -> T.Sub | P.Bitand -> T.And 
  | P.Bitor  -> T.Or  | P.Bitxor -> T.Xor
  | _ -> impos "arith_op: Expected arithmetic operator."

let arithsr_op p =
  match p with
    P.Bitlshift -> T.Shl | P.Bitlrshift -> T.Shr | P.Bitarshift -> T.Sar
  | _ -> impos "arithsr_op: Expected arith_sr operator."

let cond_op p = 
  match p with
    P.Gt   -> T.Greater  | P.Lt  -> T.Less  | P.Gte  -> T.GreaterEq
  | P.GtU  -> T.Above    | P.LtU -> T.Below | P.GteU -> T.AboveEq 
  | P.Lte  -> T.LessEq   | P.Eq  -> T.Eq    | P.Neq  -> T.NotEq
  | P.LteU -> T.BelowEq
  | _ -> impos "cond_op : Expected conditional op."
    
(******************************Code Generation::Main***************************)

(* Code generate a cast from an expression of type tsrc to type tdest
   Assumes the value starts off in Eax.
 *)
let cg_cast tsrc tdest =
  let movpart sign_extend reg_part1 reg_part2 =
    emit (T.Movpart (not sign_extend, T.Reg T.Eax, reg_part1, T.Reg T.Eax,
		     reg_part2))
  in
  let rec compress t = 
    match t with
      P.Evar(_,r) -> 
	(match !r with
	  Some t -> compress t
	| _ -> t)
    | _ -> t in
  let part s = match s with P.B1 -> T.RPl | P.B2 -> T.RPx | P.B4 -> T.RPe in 
  let gen_cast (b,s) (be,se) =
    if s=se then () 
    else if P.size_leq se s 
    then (* s wider se *)
      begin if not be
      then (*   zero-fill *) movpart false (part s) (part se)
      else (* sign-extend *) movpart true  (part s) (part se)
      end
    else  
      let mask s =  
	match s with P.B1 -> 0xFF | P.B2 -> 0xFFFF | P.B4 -> 0xFFFFFFFF(* XXX *)
      in
      if se=P.B4 then () 
      else emit(T.ArithBin(T.And,T.Reg T.Eax, T.Immed (to_i32 (mask se))))
  in
  begin match compress tdest,compress tsrc with
    P.IntType(b,s)  ,P.IntType(be,se) -> gen_cast (b,s) (be,se)
  | P.CharType      ,P.IntType(be,se) -> gen_cast (false,P.B1) (be,se)
  | P.IntType(b,s)  ,P.CharType       -> gen_cast (b,s) (false,P.B1)
  | t1,t2 when t1=t2 -> () (* XXX - may be wrong *)
  | t1,t2 -> 
      let ts1 = P.typ2string t1 in
      let ts2 = P.typ2string t2 in
      impos ("Cast of non-integral types: "^ts1^" and "^ts2)
  end

(* Code generate a binary operator.
 * One value is assumed to be in Eax.  The other is either on the stack or
 * in a register. 
 * If the other value is on the top of the stack it is popped off the stack.
 * Generate the code for the operation leaving the result in Eax. 
 * Returns the constructor for the result if different from EAX. 
 *)
let cg_bop env p sor = (* : env -> P.Primop -> stack_or_reg -> con option *)
  let get_val =
    match sor with
      TopStack   -> (fun r  -> pop r;r)
    | Stack    i -> (fun r  -> peek r i;r)
    | Register r -> (fun r' -> r) 
  in
  let div_or_mod is_signed is_mod =
    (* TJIM: division by zero or other overflow here will cause a
       machine exception.  Currently this means TAL will dump core. *)
    let reg2 = get_val T.Ecx in
    if is_signed
    then begin
      (* Edx = sign extension of Eax *)
      emit(T.Conv T.Cdq);
      emit(T.ArithMD (T.Idiv,T.Reg reg2))
    end
    else begin
      (* Edx = 0 since we are performing unsigned division *)
      emit(T.Mov (T.Reg T.Edx,(T.Immed i32_0,[])));
      emit(T.ArithMD (T.Div,T.Reg reg2))
    end;
    if is_mod then emit(T.Mov (T.Reg T.Eax, (T.Reg T.Edx,[])));
    None
  in
  match p with
    (P.Plus | P.Times | P.Minus | P.Bitand | P.Bitor | P.Bitxor) ->
      let op   = arith_op p    in
      let reg2 = get_val T.Ebx in
      emit(T.ArithBin (op,T.Reg T.Eax,T.Reg reg2));
      None
  | (P.Bitlshift | P.Bitlrshift | P.Bitarshift) ->
      let op   = arithsr_op p  in
      let reg2 = get_val T.Ecx in
      (if not(reg2 = T.Ecx) then emit(T.Mov (T.Reg T.Ecx,(T.Reg reg2,[]))));
      emit (T.ArithSR (op,T.Reg T.Eax, None));
      None
  | P.TimesU ->
      let reg2 = get_val T.Ecx in
      emit(T.ArithMD (T.Mul,T.Reg reg2));
      (* XXX - We should check for overflow. *)
      None
  | P.Div  -> div_or_mod true  false
  | P.Mod  -> div_or_mod true  true
  | P.DivU -> div_or_mod false false
  | P.ModU -> div_or_mod false true
  | (P.Gt | P.GtU | P.Lt | P.LtU | P.Gte | P.GteU | P.Lte | P.LteU 
  | P.Eq | P.Neq) ->
      let op      = cond_op p                             in
      let reg2    = get_val T.Ecx                         in
      let aux_reg = if reg2 = T.Ebx then T.Ecx else T.Ebx in
      emit (T.Mov (T.Reg aux_reg, (T.Reg T.Eax,[])));
      emit (T.Mov (T.Reg T.Eax,   bool false));
      emit (T.Cmp (T.Reg aux_reg, T.Reg reg2));
      emit (T.Setcc (op,T.Reg T.Eax));
      Some bool_con
  | _ -> impos "cg_exp: Binary Primop"  


(* EAX is "caller"-save. Results are returned in EAX *)
let rec cg_exp_push env e = 
  cg_exp env e; 
  push env T.Eax (exp2typ e)
and cg_exp env e = 
  match e.P.raw_exp with
    P.Const c ->
      let cgop = 
	match c with
	  P.Int    i -> (T.Immed i,[])
	| P.Bool   b -> (bool b)
	| P.String s -> (T.Addr (add_string s),[])
	| P.Char   c -> (T.Immed (to_i32 (Char.code c)),[])
	| P.Null     -> (T.Immed i32_0,[T.RollTosum (exp2con e)]) in 
      emit(T.Mov(T.Reg T.Eax,cgop))
  | P.ConstArray (es,ot) -> 
      let length    = List.length es                in
      let scale     = 4                             in
      let real_size = to_i32 ((length * scale) + 4) in
      let typ = 
	match (es,ot) with
	  (_,Some t) -> t 
	| (hd::tl,None) -> exp2typ hd
	| _ -> (impos "cg_exp : ConstArray without type.")            in
      let con       = typ2con typ                                     in
      let cons      = Utilities.replicate con length                  in
      let size_con  = T.csing (T.pcint (to_i32 length))               in
      let mallocarg = T.Mprod [T.Mfield size_con; T.malloc_prod cons] in
	  (* con_uninit i = type with the first i fields uninitialized.
	     We initialize from right-to-left. *)
      let con_uninit i = 
	let init_field    = T.cfield con      T.ReadWrite in
	let uninit_field  = T.cfield con      T.Uninit    in
	let size_con      = T.cfield size_con T.ReadWrite in
	let rec aux pos a =
	  if pos<=0 then a 
	  else if pos>i then aux (pos-1) (init_field :: a)
	  else aux (pos-1) (uninit_field :: a)
	in
	T.cprod_b [size_con; T.cprod (aux length [])]
      in
      let rec cg_fields env' elt es =
	match es with
	  [] -> env'
	| (hd::tl) ->
	    let offset = to_i32 ((elt-1)*scale+4) (*4 for tag*) in
	    cg_exp env' hd;
	    pop T.Ebx;	    
	    emit(T.Mov (T.Prjr ((T.Ebx,[]),offset),(T.Reg T.Eax,[])));
	    let con = con_uninit (elt-1) in
	    let env'' = push_con env T.Ebx con in
	    cg_fields env'' (elt-1) tl
      in
      emit (T.Malloc (real_size,mallocarg,None));
          
      let length' = to_i32 length in (* Initialize the tag. *)
      emit(T.Mov (T.Prjr ((T.Eax,[]),i32_0),(T.Immed length',[])));
      let env'  = push_con env T.Eax (con_uninit length) in
      let env'' = cg_fields env' length (List.rev es)    in
      pop T.Eax;
      emit(T.Coerce(T.Eax,[ T.Pack (T.pcint length',array_con con);
			    T.Toarray (i32_4,0,T.cfield con T.ReadWrite)]))
  | P.Var x ->
      (try 
	let offset = env_local_var_offset env x in
	emit (T.Mov(T.Reg T.Eax,(T.Prjr((T.Esp,[]),to_i32 (4*offset)),[])))
      with Not_found -> 
	try 
	  let t   = lookup_global env x in
          let n   = tid_val x           in
          let src = 
	    if needs_indirect t  then T.Prjl ((n,[]),i32_0) else  T.Addr n in
          emit(T.Mov(T.Reg T.Eax,(src,[])))
	with Dict.Absent -> impos "cg_exp: Var without type.")
  | P.Primop(p,[e]) ->
      cg_exp env e;
      (match p with
	P.Not ->
	  emit(T.Mov (T.Reg T.Ebx, bool false));
	  emit(T.Cmp (T.Reg T.Eax, T.Reg T.Ebx));
	  emit(T.Setcc(T.Eq,T.Reg T.Eax))
      | P.Bitnot -> emit(T.ArithUn (T.Not,T.Reg T.Eax))
      | P.Size -> 
	  emit (T.Unpack (array_size_var,T.Eax,(T.Reg T.Eax,[])));
	  emit (T.Mov (T.Reg T.Eax,(T.Prjr ((T.Eax,[]),i32_0),[])));
	  emit (T.Coerce (T.Eax,[T.Subsume T.cbyte4]))
      | P.Ord -> () (* do nothing *)
      | P.Chr -> () (* do nothing *)
      | _ -> impos "cg_exp: Unary op expected.")
  | P.Primop(p,[e1;e2]) ->
      let env' = cg_exp_push env e2 in
      cg_exp env' e1;
      cg_bop env' p TopStack; ()
  | P.Primop _ -> impos "cg_exp: ?? Primop"
  | P.Conditional (e1,e2,e3) ->
      let true_lab  = Id.id_new "condtrue"  in
      let false_lab = Id.id_new "condfalse" in
      let end_lab   = Id.id_new "condend"   in
      cg_bool_exp env e1 (true_lab,[]) (false_lab,[]) true;
      emit_lab_nocon env true_lab;
      cg_exp env e2;
      emit_jmp env end_lab [];
      emit_lab_nocon env false_lab;
      cg_exp env e3;
      emit_lab_nocon env end_lab
  | P.AssignOp (e1,po,e2) ->
      cg_exp env e2;
      cg_lhs_exp env (exp2typ e2) e1 po
  | P.FunCall (e,ts,es) ->
      let ts       = deSome !ts in
      let numargs  = List.length es in
      let coercion = 
	(* to turn off annotation hack, use next line instead of following
	   and modify popcomptypes.ml/mli as directed there *)
	(* fun_coercion (env_stack1 env) (env_stack2 env) *)
	fun_coercion numargs (numargs + (env_s1len env))
	  (if env_in_try env then T.cempty else stack1_c)
(* Cyclone *)
          (try T.Tapp (T.Con (get_cyclone_con env))
          with
            x ->
              (match e.P.raw_exp with
                P.Var z ->
                  (Printf.printf "Cyclone error in funcall %s\n" z;
                   raise x)
              | _ -> raise x))
(* End Cyclone *)
          (types2cons ts) in
      let env' = List.fold_left cg_exp_push env (List.rev es) in
      cg_exp env' e;
      emit(T.Call (T.Reg T.Eax,coercion));
      pop_free (List.length es)
  | P.TypInst(e,ts) ->
      cg_exp env e;
      if ts=[] then ()
      else 
	begin
	  let cs = types2cons ts in
	  let coercions = List.rev (List.map (fun c -> T.Tapp (T.Con c)) cs) in
	  emit(T.Coerce (T.Eax,coercions))
	end
  | P.NewStruct(n,ts,es) -> 
      let cs         = List.map typ2con (deSome(!ts)) in
      let es_cs      = List.map exp2con es            in
      let s_info     = lookup_struct env n            in
      let num_fields = List.length es                 in
      let mallocarg  = T.malloc_prod es_cs            in
      let rev_es     = List.rev es                    in
	  (* for expressions that we can't calculate easily (e.g., constants)
	   * and that might involve control-flow, we calculate the result and
	   * push it on the stack. *)
      let cg_field envs e = 
	if   is_value e 
	then envs 
	else (cg_exp_push (List.hd envs) e)::envs in
      let envs = List.fold_left cg_field [env] rev_es in
	 
      emit (T.Malloc (to_i32 (4*num_fields),mallocarg,None));
      emit (T.Mov (T.Reg T.Ebx,(T.Reg T.Eax,[])));
      List.fold_left
	(fun (offset,envs) e ->
	 (* If we've generated the value and pushed it on the stack,
	  * then pop it off, otherwise generate code for the value. *)
	  let envs = 
	    if is_value e 
	    then ((cg_exp (List.hd envs) e); envs)
	    else (pop T.Eax; List.tl envs) in
	  emit (T.Mov (T.Prjr ((T.Ebx,[]),i32_4*$offset), (T.Reg T.Eax,[])));
	  (offset +$ i32_1, envs))
	(i32_0, envs) es;
      emit(T.Mov(T.Reg T.Eax,(T.Reg T.Ebx,roll_struct s_info n cs)))

    | P.StructMember (e',n) ->
	cg_exp env e';
	let t      = exp2typ e'                   in
	let s_info = typ2struct_info env t        in
	let offset = struct_field_offset s_info n in
	emit(T.Coerce(T.Eax,[T.Unroll;T.Unroll]));
	if struct_null s_info then check_not_null e.P.exp_loc T.Eax env;
	emit(T.Mov (T.Reg T.Eax,
		    (T.Prjr ((T.Eax,[]),to_i32 (4*offset)),[])))
    | P.NewUnion (nt,ts,f,eo) ->
	let u_info = typ2union_info env (exp2typ e) in
	let cs     = types2cons     (deSome !ts)    in
	(try match eo with
	  None    -> (* void_case *)
	    let tag = union_void_tag_assoc u_info f in
	    emit(T.Mov (T.Reg T.Eax, 
			(T.Immed tag,[T.RollTosum (name_con nt cs)])))
	| Some e' -> (* value_case *)
	    let tag,con   = union_val_tag_assoc u_info f   in
	    let tag_con   = T.csing (T.pcint tag)          in
	    let e'_con    = exp2con e'                     in
	    let mallocarg = T.malloc_prod [tag_con;e'_con] in 
	    if is_value e' then 
	      (emit (T.Malloc(i32_8,mallocarg,None));
	       emit (T.Mov(T.Reg T.Ebx,(T.Reg T.Eax,[])));
	       emit (T.Mov(T.Prjr ((T.Ebx,[]),i32_0),(T.Immed tag,[])));
	       cg_exp env e';
	       emit (T.Mov(T.Prjr ((T.Ebx,[]),i32_4),(T.Reg T.Eax,[])));
	       emit (T.Mov(T.Reg T.Eax,(T.Reg T.Ebx,
				     	[T.RollTosum (name_con nt cs)]))))
	    else
	      (cg_exp env e';
	       emit (T.Push (T.Reg T.Eax,[]));
	       emit (T.Malloc (i32_8,mallocarg,None));
	       emit (T.Pop (T.Reg T.Ebx));
	       emit (T.Mov(T.Prjr ((T.Eax,[]),i32_4),(T.Reg T.Ebx,[])));
	       emit (T.Mov(T.Prjr ((T.Eax,[]),i32_0),(T.Immed tag,[])));
	       emit (T.Coerce (T.Eax,[T.RollTosum (name_con nt cs)])))
	with Not_found -> impos "cg_exp: NewUnion: No such tag.")
(*JGM*)  
    | P.UnionMember (e',n) ->
	(* generate code for e' *)
	cg_exp env e';
	let t      = exp2typ e'                   in
	let u_info = typ2union_info env t         in
	let tag    = fst(union_val_tag_assoc u_info n) in
	emit(T.Coerce(T.Eax,[T.Unroll]));
	(* branch to never_null_label -- really should "raise" an exception
	 * suitable for this sum but what the hell. *)
	if !debug_null then
	  let lab = Id.id_new "right_case" in
	  let pos = Numtypes.int_to_int32 (Gcdfec.seg_start e'.P.exp_loc) in
(*	  let inst_c = T.cappend (env_stack1 env)
	      (exn_stack_con (env_stack2 env)) in 
*)
	  (if ((union_num_voids u_info) > 0)
	  then emit_btagi(T.Eax, T.min_pointer_integer,(never_null_label,[]),
			  T.Below) env);
	  emit_btagvar(T.Eax,i32_0,tag,(lab,[]),T.Eq) env;
	  emit(T.Push(T.Addr gen_state.module_string,[]));
	  emit(T.Push(T.Immed pos,[]));
	  (* we need to push another integer as the "return address"
	   * because pop_debug_never_null is implemeneted as a C function.
	   *)
	  emit(T.Push(T.Immed pos,[]));
	  emit(T.Jmp(T.Addr debug_never_null_label,
		     [T.Tapp (T.StackTail (T.Esp,3))]));
	  emit_lab_nocon env lab;
	else 
	  (
	  (if ((union_num_voids u_info) > 0)
	  then emit_btagi(T.Eax, T.min_pointer_integer,(never_null_label,[]),
			  T.Below) env);
	  emit_btagvar(T.Eax,i32_0,tag,(never_null_label,[]),T.NotEq) env;
	  );
	(* load value out of sum *)
	emit(T.Mov(T.Reg T.Eax,(T.Prjr((T.Eax,[T.Fromsum]),i32_4),[])))
(*JGM*)
    | P.NewTuple es ->
	let num_fields = List.length es      in
	let margs      = List.map exp2con es in
	let mallocarg  = T.malloc_prod margs in
	let c = T.cprod_b (List.map (fun c -> T.cfield c T.ReadWrite) margs) in
	let envs = List.fold_left
	    (fun envs e -> 
	      if is_value e 
	      then envs 
	      else (cg_exp_push (List.hd envs) e)::envs)
	    [env] (List.rev es) in
	emit (T.Malloc (to_i32 (4*num_fields),mallocarg,None));
	emit (T.Mov (T.Reg T.Ebx,(T.Reg T.Eax,[])));
      	List.fold_left
	  (fun (offset,envs) e ->
	   (* If we've generated the value and pushed it on the stack,
	    * then pop it off, otherwise generate code for the value. *)
	    let envs = 
	      if is_value e 
	      then ((cg_exp (List.hd envs) e); envs)
	      else (pop T.Eax; List.tl envs) in
	    emit (T.Mov (T.Prjr ((T.Ebx,[]),i32_4*$offset), (T.Reg T.Eax,[])));
	    (offset +$ i32_1, envs))
	  (i32_0,envs) es;
	emit (T.Mov (T.Reg T.Eax,(T.Reg T.Ebx,[T.Subsume c])))
    | P.TupleMember(e,i) ->
	cg_exp env e;
	emit(T.Mov (T.Reg T.Eax,
		    (T.Prjr((T.Eax,[]),to_i32 (4*(i-1))),[])));
    | P.Subscript (e1,e2) ->
	let env' = cg_exp_push env e2 in
	cg_exp env' e1;
	pop T.Ebx;
	emit (T.Unpack (array_size_var,T.Eax,(T.Reg T.Eax,[])));
	(match exp2typ e1 with 
	  P.StringType -> 
	    emit (T.Mov (T.Reg T.Ecx,(T.Immed i32_0,[])));
	    emit (T.Asub (T.Ecx, T.Prjr ((T.Eax,[]),i32_4), i32_1, T.Ebx,
			  T.Prjr ((T.Eax,[]),i32_0)));
	    emit (T.Mov (T.Reg T.Eax,(T.Reg T.Ecx,[])))
	| _ -> emit (T.Asub (T.Eax, T.Prjr ((T.Eax,[]),i32_4), i32_4, T.Ebx,
			     T.Prjr ((T.Eax,[]),i32_0))))
(*    | P.NewArray(e1,e2) -> (* e1 is the size, e2 is the initializer. *)
	let env'  = cg_exp_push env  e2 in
	let env'' = cg_exp_push env' e1 in
	  (* FMS: new_array only treats EBP as callee-save.  It knows nothing 
	     about exceptions so we need the following special type 
	     applications for the call. *)
	emit(T.Call (T.Addr new_array_label,
		     [ T.Tapp (T.Con (T.cappend (env_stack1 env)
				 (exn_stack_con (env_stack2 env))));
                       T.Tapp (T.Con(T.csptr (exn_stack_con (env_stack2 env))));
(* Cyclone *)
                       (try
                         let c =
                           T.cprod_b
                             [T.cfield (get_cyclone_con env) T.ReadWrite] in
                         T.Tapp (T.Con c)
                       with
                         x ->
                           Printf.printf "Cyclone error in newarray\n";
                           raise x);
(* End Cyclone *)
		       T.Tapp (T.Con (exp2con e2));
                     ]));
	pop_free 2 *)
    | P.NewExn (x,eopt) -> 
	let con = 
	  try typ2con (lookup_exn env x) with Void_Type -> T.cbyte4 
	in
	let con' = T.cfield con T.Read in
	let mallocarg = T.Mprod [ T.Mfield (exnname_con con'); T.Mfield con] 
	in
	  (* Construct the exn *)
	(match eopt with 
	  None    -> emit (T.Mov (T.Reg T.Eax,(T.Immed i32_0,[])))
	| Some e' -> cg_exp env e');
	emit(T.Push (T.Reg T.Eax,[]));
	emit(T.Malloc(i32_8,mallocarg,None));
	emit(T.Pop (T.Reg T.Ebx));
	emit(T.Mov (T.Prjr ((T.Eax,[]),i32_4),(T.Reg T.Ebx,[])));
	emit(T.Mov (T.Reg T.Ebx,(T.Addr (tid_exn x),[])));
	emit(T.Mov (T.Prjr ((T.Eax,[]),i32_0),(T.Reg T.Ebx,[]))); 
	emit(T.Coerce (T.Eax,[T.Pack (con',exn_con)]))
    | P.Raise e ->
	cg_exp env e;
	emit (T.Mov (T.Reg T.Esp,(T.Reg T.Ebp,[])));
	emit (T.Pop (T.Reg T.Ebx));
	emit (T.Jmp (T.Reg T.Ebx,[]))
    | P.SeqExp es -> List.iter (cg_exp env) es
    | P.Nop -> () 
    | P.Cast (t,e) -> 
	let te = deSome e.P.exp_typ in
	cg_exp env e;
	cg_cast te t
(* Cyclone *)
    | P.Codegen fd ->
        let env = flush_vis_fenv env in
        let fn_con = abbrev_con(
          typ2con (P.FnType(fd.P.fn_tyvars,
                            fd.P.fn_ret_type,
			    List.map snd fd.P.fn_args))) in
        let cg_type = get_cyclone_con env in
        let cg_type = dc(T.Ctcons(dc(T.Ctrgn(fn_con,Some fn_con,[])),cg_type))in
        let env     = put_vis_cg_type env cg_type in

	let env2, env3 = env_codegen_body env fd in

        let pre_con = abbrev_con(mk_label_con env2) in
        emit(T.CgStart fn_con);
        gen_code_mark(M_TemplateBeg(Id.id_new "tmpl_start",
                                    Id.id_new "tmpl_end",
                                    pre_con,
                                    get_vis_fenv env));
        cg_fundecl env2 fd;
        cg_fill_holes env3 None true;
        emit(T.CgEnd T.Eax)
    | P.Fill e -> 
	if not (in_frame env)
	then impos "fill can only be used within a codegen or splice";

	let temp_id =
          let rec aux marks depth =
            match marks with
              [] -> impos "can't find template id"
            | M_TemplateBeg(temp_id,_,_,_)::tl ->
                if depth=0 then temp_id
                else aux tl (depth-1)
            | (M_TemplateEnd _)::tl -> aux tl (depth+1)
            | _::tl -> aux tl depth
          in aux gen_state.code_marks 0 in
        let hole_id = Id.id_new "hole" in
        emit(T.CgHole(T.Eax, temp_id, hole_id));
        gen_code_mark(M_Fill(hole_id,e))
(* End Cyclone *)
(* On entry EAX contains the value computed by the RHS.
   e is the LHS expression.
   Must compute address of e.
   if Some op then push stuff for address onto stack if dynamic.
                   load the value
                   compute op
   Store EAX into address.
   Must leave EAX in original state for cascaded assignments.
   If there is an op, and the LHS is a cast then we compute
*)
and cg_lhs_exp env e2type e op_opt = (* : env -> exp -> P.op option -> unit *)
  let (raw_exp,arg_cast,result_cast) =
    match e.P.raw_exp with
      P.Cast (t,e') ->
	let te' = deSome e'.P.exp_typ in
	(e'.P.raw_exp,(fun () -> cg_cast te' t),(fun () -> cg_cast t te'))
    | rx -> let none () = () in (rx,none,none)
  in
  match raw_exp with
    P.Var x -> 
      let gop = 
	try 
	  let offset = env_local_var_offset env x in
	  T.Prjr ((T.Esp,[]),to_i32 (4*offset))
	with Not_found -> 
	  try 
            let t = lookup_global env x in
            let n = tid_val x in
            if needs_indirect t 
            then T.Prjl ((n,[]),i32_0)
            else T.Addr n
	  with Dict.Absent -> impos "cg_exp: Var without type." 
      in
      begin match op_opt with
	    None -> ()
	  | Some op ->
	      let env' = push env T.Eax e2type in
	      cg_exp env' e;
	      arg_cast ();
	      cg_bop env' op TopStack;
	      result_cast ();
	      ()
      end;   
      emit(T.Mov (gop,(T.Reg T.Eax,[])));
  | P.StructMember (e,f) ->
      let env' = push env T.Eax e2type in
      cg_exp env' e;
      let struct_info = typ2struct_info env (exp2typ e) in
      let gop r =
	let offset = struct_field_offset struct_info f in
	T.Prjr ((r,[]),to_i32 (4*offset)) in
      let unroll = [T.Unroll;T.Unroll] in
      begin match op_opt with
	None -> 
	  emit(T.Mov (T.Reg T.Ebx,(T.Reg T.Eax,[])));
	  pop T.Eax;
	  emit(T.Coerce(T.Ebx,unroll));
	  if struct_null struct_info then check_not_null e.P.exp_loc T.Ebx env;
	  emit(T.Mov (gop T.Ebx,(T.Reg T.Eax,[])))
      | Some op ->
	  emit (T.Coerce (T.Eax,unroll));
	  let env'' = push env' T.Eax (exp2typ e) in (* struct address in Eax *)
	  if struct_null struct_info then check_not_null e.P.exp_loc T.Eax env;
	  emit (T.Mov (T.Reg T.Eax, (T.Prjr((T.Esp,[]),i32_0),[]))); (* Dan bug fix *)
	  emit (T.Mov (T.Reg T.Eax, (gop T.Eax,[])));
	  arg_cast ();
	  cg_bop env'' op (Stack 1);
	  result_cast ();
	  pop T.Ebx; (* EBX = address of struct. EAX = value *)
	  emit (T.Mov (gop T.Ebx,(T.Reg T.Eax,[])));
	  pop_free 1
      end
    | P.Subscript (e1,e2) ->
	let env' = push env T.Eax e2type in (* RHS *)
	cg_exp env' e2;
	let env'' = push env' T.Eax (exp2typ e2) in (* subscript *)
	cg_exp env'' e1;
	let scale = 
	  match exp2typ e1 with
	    P.StringType -> i32_1
	  | _ -> i32_4
	in
	  (* Eax = address of array; Stack = offset,value *)
	begin match op_opt with
	  None ->
	    emit (T.Unpack (array_size_var, T.Ecx, (T.Reg T.Eax,[])));
	    pop T.Ebx;
	    pop T.Eax;
	    emit (T.Aupd (T.Prjr ((T.Ecx,[]),i32_4),scale, T.Ebx, T.Eax,
			  T.Prjr ((T.Ecx,[]),i32_0)))
	| Some op ->
	    emit (T.Unpack (array_size_var, T.Ecx, (T.Reg T.Eax,[])));
	    let env'' = push env'' T.Eax (exp2typ e1) in
	    (* stack = array addr :: subscript :: RHS :: .... *)
	    peek T.Ebx 1; (* subscript *)
	    if scale <$ i32_4 then 
	      emit (T.Mov (T.Reg T.Eax,(T.Immed i32_0,[])));
	    emit (T.Asub (T.Eax, T.Prjr ((T.Ecx,[]),i32_4), scale, T.Ebx,
			  T.Prjr ((T.Ecx,[]),i32_0)));
	    arg_cast ();
	    cg_bop env'' op (Stack 2);
	    result_cast ();
	    pop T.Ecx; (* array address *)
	    pop T.Ebx; (* subscript     *)
	    emit (T.Unpack (array_size_var, T.Ecx, (T.Reg T.Ecx,[])));
            emit (T.Aupd (T.Prjr ((T.Ecx,[]),i32_4), scale, T.Ebx, T.Eax,
			  T.Prjr ((T.Ecx,[]),i32_0)));
	    pop_free 1 (* free RHS value. *)
	end
    | P.TupleMember (e,i) ->
	let env' = push env T.Eax e2type in
	cg_exp env' e;
	let gop r = (T.Prjr ((r,[]),to_i32 (4*(i-1)))) in
	begin match op_opt with
	  None -> 	
	    emit(T.Mov(T.Reg T.Ebx,(T.Reg T.Eax,[])));
	    pop T.Eax;
	    emit (T.Mov (gop T.Ebx,(T.Reg T.Eax,[])))
	| Some op ->
	    let env'' = push env' T.Eax (exp2typ e) in
	    emit (T.Mov (T.Reg T.Eax, (gop T.Eax,[])));
	    arg_cast ();
	    cg_bop env'' op (Stack 1);
	    result_cast ();
	    pop T.Ebx; (* EBX = address of tuple. EAX = value *)
	    emit (T.Mov (gop T.Ebx,(T.Reg T.Eax,[])));
	    pop_free 1
	end
    | _ -> impos "cg_lhs_exp: Not allowed by type-checker."

(* generates code for a boolean expression that's part of a test --
 * avoids generating the boolean value and instead branches to the
 * appropriate label based on the test.  Special support for && and ||
 * is baked in to generate (surprisingly) decent code.
 *
 * true_l is the label to branch to if the expression evaluates to true,
 * false_l is the label to branch to if the expression evaluates to false.
 * true_is_fallthru = true when the true_l is the next block, whereas
 * true_is_fallthru = false when the false_l is the next block. 
 *)
and cg_bool_exp env e true_l false_l true_is_fallthru = 
  match e.P.raw_exp with
    P.Const (P.Bool b) ->
      if b & (not true_is_fallthru) then emit_jmp env (fst true_l) (snd true_l)
      else if (not b) & true_is_fallthru then 
	emit_jmp env (fst false_l) (snd false_l)
      else ()
  | P.Primop(P.Not,[e]) -> 
      cg_bool_exp env e false_l true_l (not true_is_fallthru)
  | P.Primop(((P.Gt | P.Lt | P.Gte | P.Lte | P.Eq | P.Neq) as p),[e1;e2]) ->
      let env' = cg_exp_push env e2 in
      cg_exp env' e1;
      emit(T.Pop (T.Reg T.Ecx));
      emit(T.Cmp (T.Reg T.Eax,T.Reg T.Ecx));
      if true_is_fallthru then
        let (lab,coerce) = false_l in
	emit_jcc env (T.negate_condition(cond_op p)) lab coerce
      else 
        let (lab,coerce) = true_l in
	emit_jcc env (cond_op p) lab coerce
  (* special case for e1 && e2 *)
  | P.Conditional(e1,e2,{P.raw_exp = P.Const(P.Bool false)}) ->
      let true_l2 = Id.id_new "condtrue" in
      cg_bool_exp env e1 (true_l2,[]) false_l true;
      emit_lab_nocon env true_l2;
      cg_bool_exp env e2 true_l false_l true_is_fallthru;
  (* special case for e1 || e2 *)
  | P.Conditional(e1,{P.raw_exp = P.Const(P.Bool true)},e2) ->
      let false_l2 = Id.id_new "condfalse" in
      cg_bool_exp env e1 true_l (false_l2,[]) false;
      emit_lab_nocon env false_l2;
      cg_bool_exp env e2 true_l false_l true_is_fallthru;
  | P.Conditional(e1,e2,e3) ->
      let true_l2  = Id.id_new "condtrue"  in
      let false_l2 = Id.id_new "condfalse" in
      let end_l    = Id.id_new "condend"   in
      cg_bool_exp env e1 (true_l2,[]) (false_l2,[]) true;
      emit_lab_nocon env true_l2;
      cg_bool_exp env e2 true_l false_l true;
      emit_jmp env end_l [];
      emit_lab_nocon env false_l2;
      cg_bool_exp env e3 true_l false_l true;
      emit_lab_nocon env end_l;
  | _ ->
      cg_exp env e;
      if true_is_fallthru then emit_btagi (T.Eax,i32_0,false_l,T.Eq) env
                          else emit_btagi (T.Eax,i32_1,true_l, T.Eq) env

 
(* Cyclone *)
(****************************************************************)
(* cg_fill_holes is called at the end of a template to generate *)
(* code to dump the template and fill its holes.                *)
(****************************************************************)
and cg_fill_holes cenv0 post_con_opt restore_stack =
    (* Scan back in the TAL code that has been emitted so far,
    ** until we reach the beginning of the template.  Gather
    ** the labels and holes of the template, as well as 
    ** any jumps in the template.  Also gather the environment
    ** in effect just before the template code was emitted. *)
  let rec gather marks labels holes jumps depth =
    match marks with
      [] -> impos "gather"
    | M_Fill(hole_id,e)::tl ->
        if depth=0 then gather tl labels ((hole_id,e)::holes) jumps depth
        else gather tl labels holes jumps depth
    | M_Label(i,Some c)::tl -> 
        if depth=0
        then gather tl ((i,abbrev_con c)::labels) holes jumps depth
        else gather tl labels holes jumps depth
    | M_Label(i,None)::tl -> gather tl labels holes jumps depth
    | (M_Jmp _ as m)::tl ->
        if depth=0 then gather tl labels holes (m::jumps) depth
        else gather tl labels holes jumps depth
    | (M_Jcc _ as m)::tl ->
        if depth=0 then gather tl labels holes (m::jumps) depth
        else gather tl labels holes jumps depth
    | (M_Btagi _ as m)::tl ->
        if depth=0 then gather tl labels holes (m::jumps) depth
        else gather tl labels holes jumps depth
    | (M_Btagvar _ as m)::tl ->
        if depth=0 then gather tl labels holes (m::jumps) depth
        else gather tl labels holes jumps depth
    | M_TemplateBeg(tmpl_id,_,pre_con,fenv)::tl ->
        if depth=0 then (tmpl_id,pre_con,labels,holes,jumps,fenv)
        else gather tl labels holes jumps (depth-1)
    | (M_TemplateEnd _)::tl -> gather tl labels holes jumps (depth+1)
    | (M_ISeq _)::tl -> gather tl labels holes jumps depth
    | (M_TypeAbbrev _)::tl -> gather tl labels holes jumps depth
  in
  let (this_tmpl,pre_con,labels,holes,jumps,fenv) =
    gather (gen_state.code_marks) [] [] [] 0 in
  List.iter (fun (l,_) -> Hashtbl.add tmpl_of_label l this_tmpl) labels;
    (* Type variable for the template copy *)
  let tmpl_tv = Id.id_new "tv" in
    (* Find out which jump targets are not defined in the template;
     ** these are inter-template jumps that will have to be patched. *)
  let jumps = 
    List.concat 
      (List.map
	 (fun m ->
	   match m with
	     M_Jmp(t,x,c,coercion) ->
	       if List.exists (fun (i,_) -> t=i) labels then []
               else
                 let hole = Id.id_new "jump_hole" in
                 x := Some(this_tmpl,hole);
                 [(t,c,this_tmpl,hole,
                   fun (a,b,c,d,e,f) -> T.CgFillJmp(a,b,c,d,e,f))]
           | M_Jcc(cc,t,x,c,coercion) ->
               if List.exists (fun (i,_) -> t=i) labels then []
               else
                 let hole = Id.id_new "jcc_hole" in
                 x := Some(this_tmpl,hole);
                 [(t,c,this_tmpl,hole,
                   fun (a,b,c,d,e,f) -> T.CgFillJcc(a,b,c,d,e,f))]
           | M_Btagi(_,_,(t,_),_,x,c) ->
               if List.exists (fun (i,_) -> t=i) labels then []
               else 
                 let hole = Id.id_new "btagi_hole" in
                 x := Some(this_tmpl,hole);
                 [(t,c,this_tmpl,hole,
                   fun (a,b,c,d,e,f) -> T.CgFillBtag(a,b,c,d,e,f))]
           | M_Btagvar(_,_,_,(t,_),_,x,c) ->
               if List.exists (fun (i,_) -> t=i) labels then []
               else
                 let hole = Id.id_new "btagvar_hole" in
                 x := Some(this_tmpl,hole);
                 [(t,c,this_tmpl,hole,
                   fun (a,b,c,d,e,f) -> T.CgFillBtag(a,b,c,d,e,f))]
           | _ -> impos "cg_fill_holes: mark is not a jump")
         jumps) in
    (* Get a list of all the holes and their cons *)
    let holes2 =
      (List.map (fun (v,e) -> (v,abbrev_con(exp2con e))) holes)
        @ (List.map (fun (_,c,_,h,_) -> (h,c)) jumps) in
    (match post_con_opt with
      None -> ()
    | Some _ -> (* Emit gratuitous fallthru needed by the verifier for now *)
        fallthru cenv0);
    (* Mark the end of the template *)
    let template_con =
      abbrev_con(dc(T.Ctmpl(pre_con,post_con_opt,labels,holes2))) in
    gen_code_mark(M_TemplateEnd template_con);
    (* Restore the stack and cg type from before the last template *)
    let cenv = put_vis_fenv cenv0 fenv in 
    (* Dump the template *)
    emit(T.CgDump(tmpl_tv, T.Ebx, this_tmpl));
    let cenv = cg_push cenv tmpl_tv labels holes2 post_con_opt in
    (* Push the pointer to the template copy *)
    emit(T.Push(T.Reg T.Ebx,[]));
    let cenv =
      env_add_local_var cenv (Id.id_to_string this_tmpl) (dc(T.Ctptr tmpl_tv))in
    (* Fill the move holes *)
    let cenv = 
      List.fold_left
        (fun cenv (hole,e) ->
          cg_exp cenv e; (* Value to fill hole in goes in EAX *)
          peek T.Ebx 0;  (* Get the pointer to the temp. from top of stack *)
          emit (T.CgFill(T.Ebx, this_tmpl, hole, T.Eax));
          cg_pop_hole cenv hole)
        cenv holes in

    (*** INVARIANT: T.EBX HOLDS THE POINTER TO THE TEMPLATE COPY ***)

    (* Patch forward jumps to this template *)
    let cenv =           
      List.fold_left                (* for every label in this template *)
        (fun cenv (label,_) ->
          List.fold_left            (* for every forward itj to the label *)
            (fun cenv (back_tmpl,back_hole,fill_inst) ->
              let i = env_local_var_offset cenv (Id.id_to_string back_tmpl) in
                (* Can this fail? *)
                (* List.assoc (Id.id_to_string back_tmpl) cenv.local_env in *)
              peek T.Eax i;
              emit(fill_inst(T.Eax,back_tmpl,back_hole,T.Ebx,this_tmpl,label));
              cg_pop_hole cenv back_hole)
            cenv
            (Hashtbl.find_all forward_itjs label))
        cenv labels in

    (*** STILL INVARIANT: T.EBX HOLDS THE POINTER TO THE TEMPLATE COPY ***)

    (* Patch backward jumps from this template, mark fwd jumps as such *)
    let cenv =  
      List.fold_left
        (fun cenv (t,_,_,hole,fill_inst) ->
          try 
            let back_tmpl = Hashtbl.find tmpl_of_label t in
            let i = env_local_var_offset cenv (Id.id_to_string back_tmpl) in
              (* Can this fail? *)
              (* List.assoc (Id.id_to_string back_tmpl) cenv.local_env in *)
            peek T.Eax i;
            emit(fill_inst(T.Ebx,this_tmpl,hole,T.Eax,back_tmpl,t));
            cg_pop_hole cenv hole
          with Not_found ->
            (* Not a backward jump, must be a forward jump *)
            Hashtbl.add forward_itjs t (this_tmpl,hole,fill_inst);
            cenv)
        cenv jumps in
    
    (* Finally, restore the stack if asked *)
    if restore_stack then begin
        let n = env_stack_height cenv - env_stack_height cenv0 in
        pop_free n;
        cg_pop_free cenv n
    end;
    (* The returned cenv is not correct if restore_stack=true, because
    ** it doesn't take the pop_free into account.  But it is only used
    ** if restore_stack=false. *)
    cenv
(* End Cyclone *)

and cg_stmt env {P.raw_stmt=s;P.stmt_loc=loc} = (* true iff stmt must return *)
  let cg_control_flow env n =
    let local_depth = env_local_depth  env   in
    let ebp_opt     = env_next_handler env n in
    (match ebp_opt with 
      None        -> ()
    | Some offset -> peek T.Ebp (local_depth - offset));
    pop_free (local_depth - n) in
  let cg_goto env (label,n) coercion = 
    cg_control_flow env n; emit_jmp env label coercion 
  in
  match s with
    P.Skip  -> false
  | P.Exp e -> 
      cg_exp env e; 
      (match e.P.raw_exp with
	P.Raise _ -> true
      | _         -> false)
  | P.Seq (s1,s2) -> cg_stmt env s1 or cg_stmt env s2
  | P.Return eopt ->
      (match eopt with
	None   -> ()
      | Some e -> cg_exp env e; ());
      cg_control_flow env (env_args_on_stk env); (* pop local vars *)
      emit(T.Retn None); 
      true
    | P.IfThenElse(e,s1,s2) ->
	let true_lab  = Id.id_new "iftrue"  in
      	let false_lab = Id.id_new "iffalse" in
      	let end_lab   = Id.id_new "ifend"   in
	cg_bool_exp env e (true_lab,[]) (false_lab,[]) true;
	emit_lab_nocon env true_lab;
	let s1_returns = cg_stmt env s1 in
	if not s1_returns then emit_jmp env end_lab [];
	emit_lab_nocon env false_lab;
	let s2_returns = cg_stmt env s2 in
	if not (s1_returns & s2_returns)
	then (emit_lab_nocon env end_lab; false)
	else true
    | P.While(e,s) ->
	let loopbody = Id.id_new "whilebody" in
        let looptest = Id.id_new "whiletest" in
	let loopend  = Id.id_new "whileend"  in
	let env' = env_set_loop_labels env loopend looptest in
	emit_jmp env  looptest [];
	emit_lab env' loopbody;
	cg_stmt  env' s;
	emit_lab_nocon env' looptest;
	cg_bool_exp env' e 
	  (loopbody,branch_coercion env') (loopend,[]) false;
	emit_lab_nocon env' loopend;
	false
    | P.Break xo  -> 
	begin match xo with
	  None   -> cg_goto env (env_break_label env) []
	| Some l -> 
	    let (s,e,o) = lookup_label env l in
	    cg_goto env (e,o) (branch_coercion env)
	end;
	false
    | P.Continue xo -> 
	begin match xo with
	  None   -> cg_goto env (env_cont_label env) []
	| Some l -> 
	    let (s,e,o) = lookup_label env l in
	    cg_goto env (s,o) (branch_coercion env)
	end;
	true
    | P.For(e1,e2,e3,s) ->
	cg_exp env e1;
	let looptest  = Id.id_new "fortest"  in
	let loopbody  = Id.id_new "forbody"  in
	let loopcount = Id.id_new "forcount" in
	let loopend   = Id.id_new "forend"   in
	let env' = env_set_loop_labels env loopend loopcount in
	emit_jmp env  looptest [];
	emit_lab env' loopbody;
	cg_stmt  env' s;
	emit_lab_nocon env' loopcount;
	cg_exp   env' e3;
	emit_lab_nocon env' looptest;
	cg_bool_exp env' e2 
	  (loopbody,branch_coercion env') (loopend,[]) false;
	emit_lab_nocon env' loopend;
	false
    | P.CharSwitch(e,ss,s) -> (* compile char switch as integer switch *)
	let e' = { P.exp_typ = Some (P.IntType(true,P.B4)); 
		   P.raw_exp = P.Primop(P.Ord,[e]);
		   P.exp_loc = e.P.exp_loc
		 } in
	let ss' = List.map (fun (c,s) -> (to_i32 (Char.code c),s)) ss in
      	cg_stmt env {P.raw_stmt=P.IntSwitch(e', ss', s); P.stmt_loc=loc}
    | P.IntSwitch(e,ss,s) -> (* just does iterated ifs *)
      	cg_exp env e;
	let env2      = env_add_reg env T.Eax int_con in
      	let end_label = Id.id_new "endswitch"     in
      	let rec aux r l arms returns = 
	  match arms with
	    (i,s)::rest -> 
	      let l' = Id.id_new ("l" ^ (string_of_int32 i)) in
	      emit(T.Cmp(T.Reg r,T.Immed i));
	      emit_jcc env2 T.NotEq l' [];
	      let s_returns = cg_stmt env s in
	      (if not s_returns
              then emit_jmp env end_label (branch_coercion env));
	      emit_lab_nocon env2 l';
	      aux r l' rest (s_returns & returns) 
	  | [] -> 
	      let s_returns = cg_stmt env s in
	      if not s_returns then fallthru env;
	      (if not (s_returns & returns)  
              then (emit_lab env end_label; false)
	      else true) in
	aux T.Eax nonsense_id ss true
    | P.UnionSwitch(e,ss,d) ->
	let r             = T.Eax                          in
      	let u_info        = typ2union_info env (exp2typ e) in
	let end_label     = Id.id_new "endswitch"          in
      	let default_label = Id.id_new "default"            in

	(* Several auxiliary functions *)
	let rec split cs value_cs void_cs = 
	  match cs with
	    [] -> (List.rev value_cs, List.rev void_cs)
	  | hd::tl when hd.P.arm_var = None  ->
	      let f = hd.P.arm_field in
	      let s = hd.P.arm_body in
	      let lab = Id.id_new (f ^ "_void")          in
	      let tag = union_void_tag_assoc u_info f    in
	      split tl value_cs ((tag,lab,s) :: void_cs) 
	  | hd :: tl -> 
	      let f = hd.P.arm_field in
	      let x = deSome hd.P.arm_var in
	      let t = hd.P.arm_typ in
	      let s = hd.P.arm_body in
	      let lab     = Id.id_new (f ^ "_value")     in
	      let tag,con = union_val_tag_assoc u_info f in
(*
 	      let ts   = 
		match deSome e.P.exp_typ with 
		  P.NamedType(n,tl) -> List.map typ2con tl
		| _ -> impos "cg_exp: Unionswitch: type not named" in
	      let con = union_instantiate u_info con ts in
*)
	      let con = typ2con t in
	      split tl ((tag,con,lab,x,s)::value_cs) void_cs
	in
	
	let cg_void_branch (tag,lab,_) = 
	  emit_btagi   (r,tag,(lab,[]),T.Eq)              env in
	let cg_default_void_case tag = 
	  emit_btagi   (r,tag,(default_label,[]),T.Below) env in
	let cg_value_branch (tag,_,lab,_,_) = 
	  emit_btagvar (r,i32_0,tag,(lab,[]),T.Eq)        env in

      	let cg_void_case (tag,lab,s) = 
	  emit_lab_nocon env lab; 
	  (if not (cg_stmt env s)
	  then (emit_jmp env end_label (branch_coercion env); false)
	  else true) in
	
      	let cg_value_case (tag,con,lab,x,s) = 
	  let rc = (* Dan: we can avoid making this after rtcg infers labels *)
	    T.cprod [ T.cfield (T.csing (T.pcint tag)) T.Read;
		      T.cfield con T.Read ] in
	  let rc   = T.cptr (T.csum [rc])   in
	  let env' = env_add_reg env r rc       in
	  emit_lab_nocon env' lab;
	  emit(T.Mov(T.Reg T.Eax,(T.Prjr((r,[T.Fromsum]),i32_4),[])));
	  emit(T.Push(T.Reg T.Eax,[]));        
	  let s_returns = cg_stmt (env_add_local_var env x con) s in
	  if not s_returns 
	  then (pop_free 1; emit_jmp env end_label (branch_coercion env)); 
	  s_returns in

	(* Reorganize cases, etc. *)
	let (value_cases,void_cases) = split ss [] []                   in
	let value_cases = Sort.list 
	    (fun (tag1,_,_,_,_)(tag2,_,_,_,_)->(tag1>tag2)) value_cases in
	let void_cases = Sort.list 
	    (fun (tag1,_,_) (tag2,_,_)->(tag1>tag2)) 	    void_cases  in

	let voids_exhaust   = List.length void_cases = union_num_voids u_info in
	let has_value_cases = match value_cases with [] -> false | _ -> true  in
	let has_default     = match d with None -> false | _ -> true          in

	(* Actually generate code *)
	cg_exp env e;                 (* gen expression          *)
	emit(T.Coerce(r,[T.Unroll])); (* unroll expression       *)
      	
	if has_default                (* gen branch instructions *)
	then 
	  (List.iter cg_void_branch void_cases;
	   (if (not voids_exhaust) & has_value_cases
	   then 
	     let tag,_,_,_,_ = List.hd (List.rev value_cases) in
	     cg_default_void_case tag);
	   List.iter cg_value_branch value_cases;
	   emit_jmp env default_label [])
	else
	  if has_value_cases
	  then
	    (List.iter cg_void_branch  void_cases;
	     List.iter cg_value_branch (List.tl value_cases))
	  else
	    List.iter cg_void_branch (List.tl void_cases);
	  
	let values_return   =         (* gen value   blocks *)
	  List.for_all (fun x -> x) (List.map cg_value_case value_cases) in
	let voids_return    =         (* gen void    blocks *)
	  List.for_all (fun x -> x) (List.map cg_void_case void_cases)   in
	let default_returns =         (* gen default block  *)
	  match d with
	    None   -> true
	  | Some s -> emit_lab_nocon env default_label; 
	              if not (cg_stmt env s) then (fallthru env; false) 
		                             else true                   in
	                              (* finish and return *)
	let returns = voids_return & (values_return & default_returns)   in
	if not returns then emit_lab env end_label;
	returns

    | P.ExnSwitch(e,ss,d) -> (* Exception switch *)
	let end_label = Id.id_new "switchend"                                in
	let env'      = env_add_reg env  T.Eax exn_body                      in
	let env'      = env_add_reg env' T.Ebx (exnname_con exnname_arg_con) in

	let rec cg_arm { P.arm_field = exn; P.arm_var = vopt; P.arm_body = s} =
	  let typ     = lookup_exn env exn   in
	  let nextlab = Id.id_new "exn_case" in
	  
	  emit(T.Cmp(T.Reg T.Ebx, T.Addr(tid_exn exn)));
	  emit(T.Jcc(T.NotEq, (nextlab, exncase_coercion env')));
	  let arm_returns = 
	    match vopt with
	      None   -> cg_stmt env s
	    | Some v -> 
		(emit (T.Push (T.Prjr ((T.Eax,[]),i32_4), []));
		 let ret = cg_stmt (env_add_local_var env v (typ2con typ)) s in
		 pop_free 1;
		 ret) in
	  if not arm_returns then emit_jmp env end_label (branch_coercion env);
	  emit_exncase env' nextlab;
	  arm_returns in
	
	cg_exp env e;
	emit(T.Unpack(exnname_arg_var,T.Eax,(T.Reg T.Eax,[])));
	emit(T.Mov(T.Reg T.Ebx,(T.Prjr ((T.Eax,[]),i32_0),[])));
	let env' = env_add_reg env' T.Eax exn_body in
	let env' = env_add_reg env' T.Ebx (exnname_con exnname_arg_con) in
	exncase_fallthru env';
	emit_exncase env' (Id.id_new "exn_case");
                              (* DJG: Bug fix -- Avoid short-circuiting *)
	let arms_return     = List.for_all (fun x -> x) (List.map cg_arm ss) in
	let default_returns = cg_stmt env (deSome d)        in
	let returns         = arms_return & default_returns in
	if not default_returns then fallthru env;
	if not returns         then emit_lab env end_label;
	returns
    | P.Decl(x,t,roe,s) ->
	cg_exp env (deSome(!roe));
	emit(T.Push (T.Reg T.Eax,[]));
	if not (cg_stmt (env_add_local_var env x (typ2con t)) s)
	then (pop_free 1; false)
	else true
    | P.Label(x,s) ->
	let label_start = Id.id_new (x^"_start") in
	let label_end   = Id.id_new (x^"_end")   in
	let env'        = env_add_label env x label_start label_end in
	fallthru env;
	emit_lab env' label_start;
	let returns = cg_stmt env' s in
	if not returns then 
	  (fallthru env'; emit_lab env' label_end);
	returns
    | P.Do(s,e) ->
        let loopstart = Id.id_new "loopstart" in
        let looptest  = Id.id_new "looptest"  in
        let loopend   = Id.id_new "loopend"   in
        fallthru env;
        let env' = env_set_loop_labels env loopend loopstart in
        emit_lab env' loopstart;
	cg_stmt env' s;
        emit_lab_nocon env' looptest;
        cg_bool_exp env' e 
	  (loopstart,branch_coercion env') (loopend,[]) false;
        emit_lab_nocon env' loopend;
        false
    | P.Try(s1,x,s2) ->
	let trycatch = Id.id_new "trycatch" in
	let tryend   = Id.id_new "tryend"   in
	let env' = push_con env T.Ebp (T.csptr(exn_stack_con (env_stack2 env))) 
	in
	let handler_coercion = branch_coercion env' in 
	emit(T.Push (T.Addr trycatch, handler_coercion));
	emit(T.Mov (T.Reg T.Ebp,(T.Reg T.Esp,[])));

	emit_comment("Begin try body.");
	let env'' = env_try_body env' in
	let s1_returns = cg_stmt env'' s1 in
	emit_comment("End try body.");
	if not s1_returns 
	then begin
	  pop_free 1; (* pop handler *)
	  pop T.Ebp;  (* restore EBP *)
	  emit_jmp env tryend (branch_coercion env);
	end;
	emit_handler env trycatch;
	pop T.Ebp; (* restore EBP *)
	emit(T.Push (T.Reg T.Eax,[]));
	let env' = env_add_local_var env x exn_con in
	let s2_returns = cg_stmt env' s2 in
	if not s2_returns
	then begin 
	  pop_free 1; 
	  fallthru env;     
	end;
	let s_returns = s1_returns & s2_returns in
	if not s_returns then emit_lab env tryend;
	s_returns
(* Cyclone *)
    | P.Cut stmt ->
        let env = flush_vis_fenv env in
	if not (in_frame env)
	then impos "cut can only be used within a codgen or splice";

              (* Hack with type abbrevs to avoid MASM string too long *)
	let post_con        = mk_label_con env    in
        let post_con_abbrev = abbrev_con post_con in
              (* Find the next visible frame. *)
              (* This is too convoluted, but requires less modification
                 of the original environment type.  Someday we should just
                 go for it. *)
        let env2 = env_cut_body env in
              (* This is tricky... 
              ** here returns is true if the CUT statement
              ** lays out templates that all return *)
        let returns = PT.doesSpliceRet stmt in
        let post_con_opt =
          if returns then None else (Some post_con_abbrev) in
              (* Type of ECG will change as we dump and fill *)
        let env3 =
          cg_fill_holes
            env2
            post_con_opt
            false in
              (* WATCH OUT: can the type of ECG change if there are
		 ** splices in stmt?? *)
        cg_stmt env3 stmt;
              (* Note, the post-condition of the last template becomes
                 the pre-condition of the next template *)
        gen_code_mark(M_TemplateBeg(Id.id_new "tmpl_start",
                                    Id.id_new "tmpl_end",
                                    post_con_abbrev,
                                    get_vis_fenv env3));
              (* The TAL parser wants a label at the beginning of each
              ** template; the label serves no other purpose. *)
              (* We can't use the abbreviated post_con here because of
              ** talverify; maybe talverify should be changed. *)
        gen_code_mark(M_Label(Id.id_new "cut",Some post_con));
        returns
    | P.Splice stmt ->
        let env     = flush_vis_fenv  env           in
        let env2    = env_splice_body env           in
        let pre_con = abbrev_con(mk_label_con env2) in
        gen_code_mark(M_TemplateBeg(Id.id_new "tmpl_start",
                                    Id.id_new "tmpl_end",
                                    pre_con,
                                    get_vis_fenv env));
              (* The TAL parser wants a label at the beginning of each
              ** template; the label serves no other purpose. *)
        emit_lab env2 (Id.id_new "splice");
        let returns      = cg_stmt env2 stmt                      in
        let post_con_opt = if returns then None else Some pre_con in
        cg_fill_holes env post_con_opt true;
        false
(* End Cyclone *)

(* Cyclone *)
(* cg_fundecl has been changed so that whoever calls it is required
   to set up the environment properly.  This is needed by Cyclone. *)
(* End Cyclone *)
and cg_fundecl env fd =
  let name    = fd.P.fn_name              in
  let tyargs  = fd.P.fn_tyvars            in
  let ret_typ = fd.P.fn_ret_type          in
  let body    = fd.P.fn_body              in
  let params  = List.map snd fd.P.fn_args in
  let (con,stack1_type,stack2_type) = fun_con tyargs ret_typ params   in
  let lab     = if name = main_id then main_label else (tid_val name) in
  (if fd.P.fn_static then
    let (con,_,_) = fun_con tyargs ret_typ params in
    gen_export_val (lab,con));
  gen_set_label (lab,Some con);
  if (not (cg_stmt env body)) then emit(T.Retn None);
  flush_code ();
  (name = main_id)

(* cg_typedecls : env -> P.top_decl list -> T,con_block list -> 
 *                 (env * T.con_block list)
 *  Generates the tal constructors for popcorn type declarations.
 *)
let rec cg_typedecls env tds con_blocks = 
  let rec aux tds con_blocks structs unions =
    match tds with
      [] -> (con_blocks,structs,unions)
    | ((P.StructDecl sd,_) :: tl) ->
	let label     = tid_type sd.P.st_name                               in
	let mem_label = tid_mem  sd.P.st_name                               in
	let s_info    = info_structdecl sd                                  in
	let c,   k    = struct_t     s_info                                 in
	let memc,memk = struct_mem_t s_info                                 in
	let real_mem  = mem_label, memk, T.ConcCon memc                     in
	let mem       = mem_label, memk, memc                               in
	let int_con   = label,     k,    T.ConcCon c                        in
	let con       = label,     k,    c                                  in
	let abs_con   = label,     k,    T.AbsCon                           in
	let abs_mem   = mem_label, memk, T.AbsCon                           in
	let aux' cons = aux tl cons ((sd.P.st_name,s_info)::structs) unions in
	begin match sd.P.st_scope with
	  P.Public -> 
	    gen_export_con int_con; 
	    gen_export_con real_mem;
	    aux' (con::mem::con_blocks)
	| P.Abstract -> 
	    if struct_null s_info    (* Export null for all option types *)
	    then (gen_export_con int_con; gen_export_con abs_mem)
	    else gen_export_con abs_con;
	    aux' (con::mem::con_blocks)
	| P.Extern -> 
	    gen_import_con int_con; 
	    gen_import_con real_mem;
	    aux' con_blocks
	| P.Static -> aux' (con::mem::con_blocks) 
	end
    | ((P.UnionDecl ud,_) :: tl) ->
	let label     = tid_type ud.P.un_name                                in
	let u_info    = info_uniondecl ud                                    in
	let c,k       = union_t u_info                                       in
	let real_con  = label, k, T.ConcCon c                                in
	let con       = label, k, c                                          in
	let abs_con   = label, k, T.AbsCon                                   in
	let aux' cons =  aux tl cons structs ((ud.P.un_name,u_info)::unions) in
	begin match ud.P.un_scope with
	  P.Public   -> gen_export_con real_con; aux' (con::con_blocks)
	| P.Abstract -> gen_export_con abs_con;  aux' (con::con_blocks)
	| P.Extern   -> gen_import_con real_con; aux' con_blocks
	| P.Static   ->                          aux' (con::con_blocks)
	end
    | ((P.ExternType (tn,tyvs,opt),_)::tl) ->
	let kind_con  = tyvars_to_kind tyvs T.k4byte             in
	let kind_mem  = tyvars_to_kind tyvs T.Kmem               in
	let tylam     = tyvars_to_lam  tyvs                      in
	let label     = tid_type tn                              in
	let mem_label = tid_mem  tn                              in
	let abs_con   = (label,     kind_con, T.AbsCon)          in
	let abs_mem   = (mem_label, kind_mem, T.AbsCon)          in
	let tycons    = tyvars_to_cons tyvs                      in
	let conc_con  = tylam (opt_con (mem_name_con tn tycons)) in
	let con       = (label,kind_con,T.ConcCon conc_con)      in
	(if opt 
	then (gen_import_con con; gen_import_con abs_mem)
	else gen_import_con abs_con);
    	aux tl con_blocks structs unions
    | ((P.ExternVal (v,t),_) :: tl) ->
	let ptr_to con = T.cprod_b [T.cfield con T.ReadWrite]           in
	let con'       = typ2con t                                      in
	let con        = if needs_indirect t then ptr_to con' else con' in
	gen_import_val (tid_val v,con);
	aux tl con_blocks structs unions
    | _ -> impos "cg_typedecl : This is not a type declaration."
  in
  let (con_blocks,structs,unions) = aux tds [] [] []           in
  let env = env_add_structs env structs in
  let env = env_add_unions  env unions  in
  (env,con_blocks)

(* cg_global_decls:env->(scope*var*typ*exp option ref) list->data_block list *)
let cg_global_decls env globals =
  let cg_global dbs (scope,n,typ,eor) =
    let return con data blocks =
      let label = Id.id_unique (tid_val n) in
      (((label, Some con, data)::blocks), con, T.Dlabel(label,[]))  in
    let rec compile_data e =
      let con = exp2con e in
      match e.P.raw_exp with
	P.Const(P.Int  i) -> ([], con, T.D4bytes (i,
                                                  [T.Subsume T.cbyte4]))
      | P.Const(P.Char c) -> ([], con, T.D4bytes (to_i32 (Char.code c),
			                          [T.Subsume T.cbyte4]))
      | P.Const(P.Bool b) -> ([], con, T.D4bytes ((if b then i32_1 else i32_0),
			                          [T.Tosum con]))
      | P.Const(P.Null)   -> ([], con, T.D4bytes (i32_0, 
						  [T.RollTosum con]))
      | P.Const(P.String s) -> ([], con, T.Dlabel (add_string s,[])) 
      | P.ConstArray(es,ot) ->
	  let typ = 
	    match (es,ot) with
	      (_,      Some t) -> t 
	    | (hd::tl, None)   -> exp2typ hd
	    | _  -> impos "cg_global_decls : ConstArray without type." in
	  let len = to_i32 (List.length es)         in
	  let (blocks, _, datas) = compile_datas es in
	  let c = typ2con typ                       in
	  let this_data = 
	    (((T.D4bytes (len,[]))::(T.Dup)::datas),
 	     [T.Pack   (T.pcint len,con);
	      T.Toarray(i32_4,0,T.cfield c T.ReadWrite)]) in 
	  return con this_data blocks
      | P.NewStruct(tn, tsor, es) ->
	  let ts     = deSome !tsor                 in
	  let cs     = types2cons ts                in
	  let s_info = lookup_struct env tn         in
	  let (blocks, _, datas) = compile_datas es in
	  let coerce = roll_struct s_info tn cs     in
	  return con (datas, coerce) blocks 
      | P.NewUnion(tn,tsor,f,eopt) ->
	  let ts     = deSome !tsor        in
	  let cs     = types2cons ts       in
	  let u_info = lookup_union env tn in
	  let coerce = T.RollTosum con     in
	  (match eopt with
	    None   -> 
              let tag = union_void_tag_assoc u_info f in
	      ([],con,T.D4bytes (tag, [coerce]))
	  | Some e ->
	      let blocks, _, data = compile_data e     in
	      let tag,_ = union_val_tag_assoc u_info f in
	      return con ([(T.D4bytes (tag,[])); data], [coerce]) blocks)
      | P.NewTuple es ->
	  let (blocks, cons, datas) = compile_datas es in
	  let con = 
	    T.cprod_b (List.map (fun c -> (T.cfield c T.ReadWrite)) cons) 
	  in
	  return con (datas,[]) blocks 
      | _ -> impos "cg_global_decl: Expected a global variable declaration"
    and compile_datas es =
      let aux elt (blocks, cons, datas) =  
	let (subBlocks, con, data) = compile_data elt in
	((subBlocks@blocks), (con::cons), (data::datas))
      in
      List.fold_right aux es ([], [], []) 
    in
    let (blocks, con, data) = compile_data (deSome !eor) in
    
    let con' = T.cprod_b [T.cfield con T.ReadWrite] in
    let nt   = tid_val n                            in
    if scope = P.Public then gen_export_val (nt,con');
    (nt, Some con', ([data],[])) :: (blocks@dbs)
  in  
  List.fold_left cg_global [] globals

let cg_exn_decls exns = (* : (var * scope * typ) list -> data_block list *)
  let rec aux (v,sc,typ) rest =
    let con       = try typ2con typ with Void_Type -> T.cbyte4   in
    let con'      = T.cfield con T.Read                          in
    let con''     = exnname_con con'                             in
    let label     = tid_exn v                                    in
    let data_item = (label, Some con'', ([T.Djunk],[]))          in
    match sc with
      P.Static   -> data_item :: rest
    | P.Public   -> gen_export_val (label,con''); data_item :: rest
    | P.Extern   -> gen_import_val (label,con''); rest
    | P.Abstract -> impos "cg_exn_decls: Abstract exception declaration."
  in
  List.fold_right aux exns []

(* cg_all : env -> P.top_decl list -> (T.code_block list * bool) *)
(* Sort top decls, Generate type decls, Generate functions *)
let cg_all mod_name env topdecls = 
  let (type_decls,fun_decls,global_decls,exn_decls) = 
    let rec sort_decls ds ts fs gs xs = 
      match ds with
	[]                        -> (ts,fs,gs,xs)
      | (P.FunDecl fd,_)::tl      -> sort_decls tl ts (fd::fs) gs xs
      | (P.GlobalDecl g,_)::tl    -> sort_decls tl ts fs (g :: gs) xs
      | (P.ExceptionDecl x,_)::tl -> sort_decls tl ts fs gs (x::xs)
      | (P.OpenDecl(prefix,decls),_)::tl ->
	  let (ts,fs,gs,xs) = sort_decls decls ts fs gs xs in
	  sort_decls tl ts fs gs xs
      | hd::tl                    -> sort_decls tl (hd::ts) fs gs xs in
    sort_decls topdecls [] [] [] []  in
  reset_generator mod_name;
  let (env',con_blocks) = cg_typedecls env type_decls [] in 
  let data_blocks = 
    (cg_exn_decls exn_decls) @ (cg_global_decls env' global_decls) in
  List.iter gen_data_block data_blocks;

  let has_main_list =
    List.map (fun fd -> cg_fundecl (env_fun_start env' fd) fd) fun_decls in

  let has_main = List.exists (fun x -> x) has_main_list in
  if has_main then 
    (gen_code_block tal_main_code_block;
     gen_import_val (pop_exit_label, pop_exit_con));
  (con_blocks,has_main)

(* Note: marks are processed in *REVERSE* order *)
let postprocess marks =
  let rec pp marks cbs istack tmps cuts tcons abbrevs =
    match marks with
      [] ->
        begin
          match istack,cuts with
            [],[] -> (tmps,cbs,abbrevs)
          | _,[]  -> impos "instructions without label"
          | [],_  -> impos "cuts not empty"
          | _     -> impos "instructions without label, cuts not empty"
        end
    | m::ms ->
        let issue x = pp ms cbs ([|x|]::istack) tmps cuts tcons abbrevs in
        match m with
          M_Label(id,con) ->
            let cb = (id,con,Array.concat istack) in
            pp ms (cb::cbs) [] tmps cuts tcons abbrevs
        | M_ISeq i ->
            pp ms cbs (i::istack) tmps cuts tcons abbrevs
        | M_TypeAbbrev(v,c) ->
            pp ms cbs istack tmps cuts tcons ((v,c)::abbrevs)
(* Cyclone *)
        | M_Jmp(label,{contents=None},target_con,coercion) ->
            issue (T.Jmp(T.Addr label,coercion))
        | M_Jmp(_,{contents=Some(temp,hole)},target_con,coercion) ->
            issue (T.CgHoleJmp(temp,(hole,coercion)))
        | M_Jcc(cc,label,{contents=None},target_con,coercion) ->
            issue (T.Jcc(cc,(label,coercion)))
        | M_Jcc(cc,_,{contents=Some(temp,hole)},target_con,coercion) ->
            issue (T.CgHoleJcc(cc,temp,(hole,coercion)))
        | M_Btagi(reg,i,lc,cc,{contents=None},target_con) ->
            issue (T.Btagi(reg,i,lc,cc))
        | M_Btagi(reg,i,(_,c),cc,{contents=Some(temp,hole)},target_con) ->
            issue (T.CgHoleBtagi(reg,i,temp,(hole,c),cc))
        | M_Btagvar(reg,i1,i2,lc,cc,{contents=None},target_con) ->
              issue (T.Btagvar(reg,i1,i2,lc,cc))
        | M_Btagvar(reg,i1,i2,(_,c),cc,
                    {contents=Some(temp,hole)},target_con) ->
		      issue (T.CgHoleBtagvar(reg,i1,i2,temp,(hole,c),cc))
        | M_TemplateEnd c ->
            pp ms [] [] tmps ((cbs,istack)::cuts) (c::tcons) abbrevs
        | M_TemplateBeg(id1,id2,pre_con,_) ->
            begin
              match istack,cuts,tcons with
                [],(cbs0,istack0)::tl,c::tcons ->
                  pp ms cbs0 istack0 ((id1,id2,c,cbs)::tmps) tl tcons abbrevs
              | _,_,_ -> impos "pp M_TemplateBeg"
            end
        | M_Fill _ -> pp ms cbs istack tmps cuts tcons abbrevs
(* End Cyclone *)
  in pp marks [] [] [] [] [] []


(************************************Entry Point*******************************)
let code_gen mod_name import_file export_file (td,global_env) =

  let env = env_empty global_env in

  let (con_blocks,has_main)           = cg_all mod_name env td in
  let (templates,code_blocks,abbrevs) = postprocess gen_state.code_marks in
  let abbrevs =
    Array.of_list
      ([array_abbrev_var,       array_abbrev;
	 string_abbrev_var,     string_abbrev; 
	 exnname_var,           exnname_abbrev;
	 exn_var,               exn_abbrev;
	 handle_abbrev_var,     handle_abbrev; 
	 exn_stack_abbrev_var,  exn_stack_abbrev;
	 stack_abbrev_var,      stack_abbrev;
	 stack_void_abbrev_var, stack_abbrev_void;
	 handler_abbrev_var,    handler_abbrev] @
       abbrevs) in
  let implementation =     
    { T.imports     = [|"pop_runtime.tali"; import_file |];
      T.exports     = if has_main then [|"tal_prog.tali"; export_file|]
                                  else [|export_file|];
      T.imp_abbrevs = [||];
      T.con_blocks  = Array.of_list (bogus_option_con_block :: con_blocks);
      T.code_blocks = Array.of_list code_blocks;
      T.data_blocks = Array.of_list gen_state.data_blocks;
(* Cyclone *)
      T.templates   = Array.of_list templates
(* End Cyclone *)
    } in
  let export_interface = 
    { T.int_abbrevs = abbrevs;
      T.int_cons    = Array.of_list gen_state.export_cons;
      T.int_vals    = Array.of_list gen_state.export_vals
    }	in
  let import_interface = 
    { T.int_abbrevs = abbrevs;
      T.int_cons    = Array.of_list gen_state.import_cons;
      T.int_vals    = Array.of_list gen_state.import_vals
    }	in
  (implementation,import_interface,export_interface)
