(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,Frederick Smith    *)
(*     Dan Grossman                                                   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

(* Prior to Release 1.7, this was part of Popcompile; now it is opened by 
   that module.
   Here we make the environment an abstract data type.
 *)

type con = Tal.con
type typ = Popsyntax.typ
type id  = Identifier.identifier

open Popcomptypes
module P  = Popsyntax
module T  = Tal
module PT = Poptype

let deSome x = match x with Some v -> v | None -> impos "deSome"

(***************** Type Definitions **********)

(* The global_env tracks definitions of types and types of global variables.
 * The local_env tracks the types and stack positions (in bytes relative to
 * the current stack pointer) of local variables.  The stack types track
 * the current type of the stack.
 * 
 * The break label, when Some(lab,n), is the label that a break should
 * jump to.  However, before jumping, the break should pop s - n local 
 * variables from the stack, where s is the current size of the stack.
 * Similarly, the continue label, when Some(lab,n) is the label that a 
 * continue should jump to.  
 *
 * sptr_offsets tracks the location of the stack pointers pushed in
 * a try.  A continue or break must restore the appropriate one into EBP.
 * Similarly a return must restore the outermost one. (last one)
 *)
type fenv = (* Environment local to a function *)
    { f_tyvars:         P.var list;
      f_local_env:      (P.var * (T.con * int)) list;
      f_args_on_stack:  int;
      f_stack1_type:    T.con;
      f_stack2_type:    T.con; (* Stack after exception handler. *)
      f_s1len:          int;
      f_break_label:    (id*int) option;
      f_continue_label: (id*int) option;
      f_regs:           T.register_state;
      f_labels:         (P.var,id * id * int) Dict.dict
    } 
type cenv =
    Hidden    of fenv
  | Frame     of fenv
  | Outermost of fenv

type type_info =
    { unions:  (P.type_name, union_info)  Dict.dict;
      structs: (P.type_name, struct_info) Dict.dict
    } 
type env = 
    { global_env:     PT.global_env;
      f_env:          fenv;
      sptr_offsets:   int list;
      type_info:      type_info;
(* Cyclone *) cenvs:  cenv list (* End Cyclone *)
    } 

(*************** Constructors **************)
let env_empty global_env =
  { global_env = global_env;
    f_env = { f_tyvars         = [];
	      f_local_env      = [];
	      f_args_on_stack  = 0;
	      f_stack1_type    = T.cempty;
	      f_stack2_type    = T.cempty;
	      f_s1len          = 0;
	      f_break_label    = None;
	      f_continue_label = None;
	      f_regs           = T.rs_empty;
	      f_labels         = Dict.empty compare
	    };
    sptr_offsets = [];
    type_info    = { structs=Dict.empty compare; unions=Dict.empty compare};
(* Cyclone *) cenvs = [] (* End Cyclone *)
  } 

let mk_fenv fd =
  let tyargs    = fd.P.fn_tyvars    in
  let ret_typ   = fd.P.fn_ret_type  in
  let args      = fd.P.fn_args      in
  let params    = List.map snd args in
  let (_,stack1_type,stack2_type) = fun_con tyargs ret_typ params in
  let local_env = 
    snd (List.fold_left 
	   (fun (i, vars) (x,t) -> (i+1, (x, (typ2con t,i))::vars))
	   (1,[]) args)
  in
  { f_tyvars         = tyargs;
    f_local_env      = local_env;
    f_args_on_stack  = List.length local_env;
    f_stack1_type    = stack1_type;
    f_stack2_type    = stack2_type;
    f_s1len          = List.length params + 1; (* 1 for retn addr *)
    f_break_label    = None;
    f_continue_label = None;
    f_labels         = Dict.empty compare;
    f_regs           =
      (* Cyclone *) T.rs_set_tla (* End Cyclone *)      
                  T.rs_empty
      (* Cyclone *) (T.cprod_b [T.cfield cg_c T.ReadWrite]) (* End Cyclone *)
  }
let env_add_structs env s_info_list =
  { env with type_info =
      { env.type_info with 
          structs = Dict.inserts env.type_info.structs s_info_list
      } 
  } 
let env_add_unions env u_info_list =
  { env with type_info =
      { env.type_info with 
          unions = Dict.inserts env.type_info.unions u_info_list
      } 
  } 
let env_fun_start env fd =
  { global_env   = env.global_env;
    f_env        = mk_fenv fd;
    sptr_offsets = [];
    type_info    = env.type_info;
(* Cyclone *) cenvs = [] (* End Cyclone *)
  } 

(***************** Selectors **************)
let env_tyvars      env = env.f_env.f_tyvars
let env_regs        env = env.f_env.f_regs
let env_stack1      env = env.f_env.f_stack1_type
let env_stack2      env = env.f_env.f_stack2_type
let env_s1len       env = env.f_env.f_s1len
let env_break_label env = deSome env.f_env.f_break_label
let env_cont_label  env = deSome env.f_env.f_continue_label
let env_args_on_stk env = env.f_env.f_args_on_stack

let lookup_label    env x  = Dict.lookup env.f_env.f_labels           x      
let lookup_struct   env n  = Dict.lookup env.type_info.structs        n
let lookup_union    env n  = Dict.lookup env.type_info.unions         n
let lookup_global   env x  = Dict.lookup env.global_env.PT.globals    x
let lookup_exn      env x  = Dict.lookup env.global_env.PT.exceptions x
let typ2struct_info env nt = lookup_struct env (get_name nt)
let typ2union_info  env nt = lookup_union  env (get_name nt)

let env_local_depth env = List.length env.f_env.f_local_env

let env_local_var_offset env v = (* for globals, raises Not_found *)
  snd (List.assoc v env.f_env.f_local_env)

let env_next_handler env below_here = 
(* fixed by Dan for when in a handler but break does _not_ leave it *)
  match env.sptr_offsets with
    []      -> None
  | hd::tl  -> 
	(let rec aux prev rest =
	  match rest with
	    hd::tl when hd>below_here -> aux (Some hd) tl
	  | _ -> prev in
	aux None env.sptr_offsets)

let env_in_try env = match env.sptr_offsets with [] -> false | _ -> true

(****************** Modifiers ***************)
let env_add_local_var env v con = (*v has already been pushed, now update env*)
  let new_local_env = (v,(con,0)) ::
    (List.map (fun (v,(c,i)) -> (v,(c,i+1))) env.f_env.f_local_env) in
  let new_stack1_type = T.ccons con (env_stack1 env) in
  { env with f_env =
     { env.f_env with f_local_env   = new_local_env; 
                      f_stack1_type = new_stack1_type;
                      f_s1len       = env.f_env.f_s1len+1;} 
  } 
let env_push_con env reg_con = env_add_local_var env "*BOGUS*" reg_con 
let env_add_reg  env reg c   =
  { env with f_env =
    { env.f_env with f_regs = T.rs_set_reg (env_regs env) reg c }
  } 
let env_set_loop_labels env loopend looptest =
  let stack_depth = List.length env.f_env.f_local_env in
  { env with f_env =
    { env.f_env with f_break_label    = Some(loopend,  stack_depth);
                     f_continue_label = Some(looptest, stack_depth); } 
  } 

let env_add_label env popL startL endL =
  let stack_depth = List.length env.f_env.f_local_env in
  { env with f_env = 
      { env.f_env with f_labels = 
           Dict.insert env.f_env.f_labels popL (startL,endL,stack_depth) }
  }

let env_try_body env = 
(* call on try entry,  stack already has shape handler::old_ebp::.... 
   but env does not yet record the location of old_ebp. *)
  let new_stack1 = T.cempty in
  let new_stack2 = T.cappend (env_stack1 env) (exn_stack_con (env_stack2 env))in
  let old_ebp_offset = List.length env.f_env.f_local_env in
  let env = env_add_local_var env "*BOGUS*" int_con in(*place hold for handler*)
  { env with 
    f_env = { env.f_env with 
               f_stack1_type  = new_stack1; 
               f_stack2_type  = new_stack2;
               f_s1len        = 0;
             };
    sptr_offsets = old_ebp_offset :: env.sptr_offsets
  }

(* Cyclone *)
let get_cyclone_con env    = fst(deSome(T.get_cyclone_field (env_regs env)))
let set_cyclone_con rs con = T.set_cyclone_field rs con T.ReadWrite

let outermost env =
  let rec aux cs =
    match cs with
      [] -> true
    | (Outermost _)::_  -> true
    | (Hidden    _)::cs -> aux cs
    | (Frame     _)::_  -> false in
  aux env.cenvs
    
let in_frame  env = match env.cenvs with (Frame _)::_ -> true | _ -> false
    
let get_vis_fenv env      = env.f_env
let put_vis_fenv env fenv =
  let new_cenvs =
    let rec aux cs =
      match cs with
	[]                    -> []
      | (Hidden    fenv2)::tl -> (Hidden    fenv2) ::(aux tl)
      | (Frame     _)    ::tl -> (Frame     fenv)  ::tl
      | (Outermost _)    ::tl -> (Outermost fenv)  ::tl in
    aux env.cenvs in
  { env with f_env = fenv; cenvs = new_cenvs } 
let flush_vis_fenv env =
  let new_cenvs =
    let rec aux cs =
      match cs with
        []                 -> []
      | (Frame     _)::tl  ->let fenv = get_vis_fenv env in (Frame     fenv)::tl
      | (Outermost _)::tl  ->let fenv = get_vis_fenv env in (Outermost fenv)::tl
      | (Hidden fenv2)::tl ->(Hidden fenv2)::(aux tl) 
    in
    aux env.cenvs in
  { env with cenvs = new_cenvs} 
let put_vis_cg_type env cg_type =
  { env with f_env = 
     { env.f_env with f_regs = set_cyclone_con (env_regs env) cg_type }
  } 

let env_stack_height env =
  let rec aux con =
    match con.T.rcon with
      T.Ccons(_,con) -> 1 + aux con
    | _              -> 0 in
  aux (env_stack1 env)

let env_codegen_body env fd = (* env should already be flushed *)
  let fenv2 = mk_fenv fd in
  let cenvs =
    match env.cenvs with
      []     -> [ Frame fenv2; Outermost(get_vis_fenv env) ]
    | cenvs' ->  (Frame fenv2)::cenvs' in
  let env2 = { env with f_env = fenv2; cenvs = cenvs} in
  let env3 = { env with cenvs = (Hidden fenv2)::env.cenvs } in
  env2, env3

let env_cut_body env = (* env should already be flushed *)
  let fenv,cenvs  = 
    match env.cenvs with (Frame x)::y -> x,y
    | _ -> impos "cut: bad cenvs type in env_cut_body" in
  let fenv2 =
    let rec aux cs =
      match cs with
        []                    -> impos "cut: can't find previous env"
      | (Outermost fenv2)::_  -> fenv2
      | (Frame     fenv2)::_  -> fenv2
      | (Hidden        _)::tl -> aux tl in
    aux cenvs in
  { env with f_env = fenv2; cenvs = (Hidden fenv)::cenvs }
    
let env_splice_body env = (* env should already be flushed *)
  match env.cenvs with
    (Hidden fenv2)::cenvs ->
      let fenv0 = get_vis_fenv env in
      { env with f_env = fenv2; cenvs = (Frame fenv2)::cenvs } 
  | _ -> impos "splice can only be used within a cut"

(* End Cyclone *)
