/**********************************************************************
 * (c) Greg Morrisett, Neal Glew, Chris Hawblitzel, Dan Grossman      *
 *     June 1998, all rights reserved.                                *
 **********************************************************************/

%{

(* Does unary minus inneficiently -- should be trivial update to abstract
   syntax
 *)

(* Grammar taken largely from Java's syntax (see Java spec) *)

open Numtypes;;
open Popsyntax;;

type type_modifier =
    ModArray
  | ModParams of string list * (string option * typ) list

let rec make_type t ms =
  match ms with
    [] -> t
  | ModArray::tl -> make_type (ArrayType t) tl
  | (ModParams (tvs,tps))::tl ->
      make_type (FnType(tvs,t, List.map (fun (_,tp) -> tp) tps)) tl

module X = Poperr;;

let locus () = Gcdfec.seg_symbol ();;

let err e = (Gcdfec.post_error (X.mk_parse_error e))

let parse_error s = err (X.Syntax s)

type switch_clause = 
    AnyClause
  | IntClauses of (int32 * stmt) list
  | CharClauses of (char * stmt) list
  | UnionClauses of (field_name * (var option) * stmt) list;;

let add_int_clause i s (scs,d) = 
  match scs with
    IntClauses cs -> (IntClauses ((i,s)::cs),d)
  | AnyClause -> (IntClauses [(i,s)],d)
  | _ -> err X.SwitchClausesDontMatch; (scs,d)
;;

let add_char_clause c s (scs,d) = 
  match scs with
    CharClauses cs -> (CharClauses ((c,s)::cs),d)
  | AnyClause -> (CharClauses [(c,s)],d)
  | _ -> err X.SwitchClausesDontMatch; (scs,d)
;;

let add_union_clause f v s (scs,d) = 
  match scs with
    UnionClauses cs -> (UnionClauses ((f,v,s)::cs),d)
  | AnyClause -> (UnionClauses [(f,v,s)],d)
  | _ -> err X.SwitchClausesDontMatch; (scs,d)
;;

let make_exp re = { exp_typ = None; raw_exp = re; exp_loc = locus() };;
let make_binop lft oper rgt = make_exp (Primop(oper, [lft; rgt]));;
 
let make_extern_val vtel =
  begin
    let rec aux vtel a = 
      match vtel with
	[] -> List.rev a
      |	(v,t,e) :: tl ->
	  begin match !e with 
	    Some _ -> err X.ExternNoInit
	  | None -> ()
	  end;
	  aux tl ((v,t) :: a)
    in
    aux vtel []
  end

%}

%token <string> ID
%token <string> QID
%token <Numtypes.int32> CONSTINT
%token <bool> CONSTBOOLEAN
%token <string> CONSTSTRING
%token <char> CONSTCHAR
%token EOF

%token LPAREN RPAREN LBRACE RBRACE LBRACKET RBRACKET LESSTHAN
%token GREATERTHAN LESSTHANEQ GREATERTHANEQ PLUSPLUS MINUSMINUS
%token PLUSEQUAL MINUSEQUAL TIMESEQUAL DIVEQUAL AMPERAMPER BACKQUOTE
%token PIPEPIPE EQUALS EE NE PLUS MINUS TIMES DIV BANG QUESTION
%token COLON SEMICOLON DOT COMMA AMPER PIPE LESSLESS GREATERGREATER PERCENT

%token AMPEREQUAL PIPEEQUAL LESSLESSEQ GREATERGREATEREQ MODEQUAL TILDE 
%token GREATERGREATERGREATER GREATERGREATERGREATEREQ CARET CARETEQUAL

%token ARROW EXCEPTION EXN HANDLE RAISE TRY

%token PREFIX COLONCOLON OPEN
%token BOOL BREAK CASE CHAR CHR CONST CONTINUE DEFAULT ELSE EXTERN FOR IF 
%token INT NEW NEWARRAY NULL ORD RETURN SIZE STATIC 
%token STRING STRUCT SWITCH UNION VOID WHILE
%token DO PRIVATE PUBLIC ABSTRACT
/* Cyclone */
%token CODEGEN CUT SPLICE FILL
/* End Cyclone */

%right PLUSPLUS MINUSMINUS TILDE BANG
%left  TIMES DIV PERCENT
%left  PLUS MINUS
%left  LESSLESS GREATERGREATER GREATERGREATERGREATER
%left  LESSTHAN LESSTHANEQ GRWATERTHAN GREATERTHANEQ
%left  EE NE
%left  AMPER
%left  CARET
%left  PIPE
%left  AMPERAMPER
%left  PIPEPIPE
%right EQUALS PLUSEQUAL MINUSEQUAL TIMESEQUAL DIVEQUAL MODEQUAL 
%right AMPEREQUAL PIPEEQUAL CARETEQUAL
%right LESSLESSEQ GREATERGREATEREQ GREATERGREATERGREATEREQ

%type <Popsyntax.top_decl list> top

%start top

%%

top:
    top_decls EOF           { List.rev($1) }
  ;

top_decls:
  | top_decl                { $1 }
  | PREFIX prefixed_type_name SEMICOLON top_decls 
      { [(PrefixDecl($2,$4),locus ())] }
  | OPEN prefixed_type_name SEMICOLON top_decls 
      { [(OpenDecl($2,$4),locus ())] }
  | top_decl top_decls   { $1 @ $2 }
  ;

top_decl:
    func_decl               { [$1] }
  | struct_decl             { [$1] }
  | union_decl              { [$1] }
  | exn_decl                { [$1] }
  | prefix_decl             { [$1] }
  | open_decl               { [$1] }
  | extern                  { $1 }
  | scope var_decl          
      { List.map (fun (v,t,e) -> (GlobalDecl($1,v,t,e),locus())) $2 }
  ;

prefix_decl:
      PREFIX prefixed_type_name LBRACE top_decls RBRACE 
      { (PrefixDecl ($2,$4),locus ()) }
  ;

open_decl:
      OPEN prefixed_type_name LBRACE top_decls RBRACE
      { (OpenDecl ($2,$4),locus ()) }
  ;

scope:
    PRIVATE { Static }
  | PUBLIC  { Public }
  ;

func_decl:
    static_opt prim_type ID type_params LPAREN RPAREN type_mods block
                        { (FunDecl { fn_static = $1;
				     fn_name = $3;
				     fn_tyvars = $4;
				     fn_ret_type = make_type $2 $7;
				     fn_args = [];
				     fn_body = $8
				   },
                           locus()) }
  | static_opt prim_type ID type_params LPAREN named_param_list RPAREN type_mods block
                        { (FunDecl { fn_static = $1;
				     fn_name = $3;
				     fn_tyvars = $4;
				     fn_ret_type = make_type $2 $8;
				     fn_args = $6;
				     fn_body = $9
				   },
                           locus()) }
  ;

struct_decl:
    scope_opt question_opt STRUCT type_params ID LBRACE struct_field_decls RBRACE
                       { (StructDecl { st_scope = $1;
				       st_name = $5;
				       st_tyvars = $4;
				       st_possibly_null = $2;
				       st_fields = $7}, locus()) }
  ;

union_decl:
    scope_opt question_opt UNION type_params ID LBRACE union_field_decls RBRACE
                       {   (UnionDecl { un_scope = $1;
				      	un_name = $5;
					un_tyvars = $4;
				      	un_possibly_null = $2;
				      	un_fields = $7}, locus ()) }
  ;

exn_decl:
    scope_opt EXCEPTION ID LPAREN param_decl RPAREN SEMICOLON
      { (ExceptionDecl($3,$1,snd $5),locus()) }
  | scope_opt EXCEPTION ID SEMICOLON
      { (ExceptionDecl($3,$1,VoidType),locus()) }
  ;
	  

type_params:
      { [] }
  | LESSTHAN tyvar_list { $2 }
  ;

tyvar_list:
    BACKQUOTE ID GREATERTHAN  { [$2] }
  | BACKQUOTE ID COMMA tyvar_list { $2 :: $4 }

type_mods:
    /* empty */                          { [] }
  | type_mods LBRACKET RBRACKET          { ModArray::($1) }
  | type_mods type_params LPAREN RPAREN  { (ModParams($2,[]))::($1) }
  | type_mods type_params LPAREN param_list RPAREN   { (ModParams($2,$4))::($1) }
  ;

param_list:
    param_decl                    { [($1)] }
  | param_decl COMMA param_list   { ($1)::($3) }
  ;

param_decl:
    prim_type ID type_mods  { (Some($2), make_type ($1) ($3)) }
  | prim_type type_mods     { (None, make_type ($1) ($2)) }
  ;

named_param_list:
    named_param_decl                          { [($1)] }
  | named_param_decl COMMA named_param_list   { ($1)::($3) }
  ;

named_param_decl:
    prim_type ID type_mods  { ($2, make_type ($1) ($3)) }
  ;

prim_type:
    VOID       { VoidType }
  | INT        { IntType }
  | BOOL       { BooleanType }
  | CHAR       { CharType }
  | STRING     { StringType }
  | prefixed_type_name { NamedType (ref $1,[]) }
  | TIMES LPAREN RPAREN { TupleType [] }
  | TIMES LPAREN param_list RPAREN { TupleType(List.map snd $3) }
  | LESSTHAN param_list GREATERTHAN prefixed_type_name 
      { NamedType (ref $4,List.map snd $2) } 
  | BACKQUOTE ID    { VarType ($2) } 
  | EXN             { ExnType }
  ;

prefixed_type_name:
  | ID    { $1 }
  | ID COLONCOLON prefixed_type_name { add_prefix $1 $3 }
  ;

extern:
    EXTERN var_decl
      { (List.map (fun (v,t) -> (ExternVal(v,t),locus())) 
	   (make_extern_val $2)) }
  | EXTERN prefixed_type_name question_opt type_params SEMICOLON             
      { [ExternType ($2,$4,$3),locus()] }
  ;
static_opt:
                        { true }
  | STATIC              { false }
  ;

scope_opt:
    STATIC            { Static   }
  | ABSTRACT          { Abstract }
  | EXTERN            { Extern   }
  | PUBLIC	      { Public   }
  |                   { Public   }
  ;

question_opt:
                       { false }
  | QUESTION           { true }
  ;

constopt:
          { ReadWrite }
  | CONST { ReadOnly }
  ;

iddecl:
    prim_type ibl  { let t = $1 in 
                   let idbs : (string * type_modifier list) list = $2 in
                   List.map(fun (x,b) -> (x,make_type t b)) idbs }
  ;

ibl:
    ID type_mods             { [($1,$2)] }
  | ID type_mods COMMA ibl   { ($1,$2)::$4 }
  ;

struct_field_decls:
                                          { [] }
  | struct_field_decl struct_field_decls  { $1 @ $2 }
  ;

struct_field_decl:
    constopt iddecl SEMICOLON   { let rw = $1 in 
				  List.map (fun (x,t) -> (x,rw,t)) $2 }
  ;

union_field_decls:
                                        { [] }
  | union_field_decl union_field_decls  { $1 @ $2 }
  ;

union_field_decl:
    iddecl SEMICOLON   { $1 }
  ;

block:
    LBRACE stmts RBRACE  { $2 }
  ;

stmts:
    stmt  { $1 }
  | var_decl stmts { List.fold_right 
		       (fun (x,t,eopt) s -> (Decl(x,t,eopt,s),locus())) $1 $2 }
  | stmt stmts { (Seq($1,$2),locus()) }
  ;

var_decl:
    prim_type init_list SEMICOLON
    { let t = $1 in let il = $2 in
      List.map (fun (x,b,e) -> (x,make_type t b,e)) il }
  ; 

init_list:
    init_var                  { [$1] }
  | init_var COMMA init_list  { $1 :: $3 }
  ;

init_var:
    ID type_mods             { ($1,$2,ref None) }
  | ID type_mods EQUALS exp  { ($1,$2,ref (Some $4)) }
  ;

stmt:
    stmt_no_trailing              { $1 }
  | IF LPAREN exp RPAREN stmt
      { (IfThenElse ($3, $5, (Skip,locus())),locus()) }
  | IF LPAREN exp RPAREN stmt_no_short_if ELSE stmt   
      { (IfThenElse ($3, $5, $7),locus()) }
  | WHILE LPAREN exp RPAREN stmt  { (While ($3, $5),locus()) }
  | FOR LPAREN exp SEMICOLON exp SEMICOLON exp RPAREN stmt 
      { (For($3,$5,$7,$9),locus()) }
/* Cyclone */
  | CUT stmt { (Cut($2),locus()) }
  | SPLICE stmt { (Splice($2),locus()) }
/* End Cyclone */
  | TRY stmt HANDLE ID stmt { (Try($2,$4,$5),locus()) }
  ;

stmt_no_short_if:
    stmt_no_trailing    { $1 }
  | IF LPAREN exp RPAREN stmt_no_short_if ELSE stmt_no_short_if   
      { (IfThenElse ($3, $5, $7),locus()) }
  | WHILE LPAREN exp RPAREN stmt_no_short_if  { (While ($3, $5), locus()) }
  | FOR LPAREN exp SEMICOLON exp SEMICOLON exp RPAREN stmt_no_short_if 
      { (For($3,$5,$7,$9),locus()) }
/* Cyclone */
  | CUT stmt_no_short_if { (Cut($2),locus()) }
  | SPLICE stmt_no_short_if { (Splice($2),locus()) }
/* End Cyclone */
  | TRY stmt HANDLE ID stmt_no_short_if { (Try($2,$4,$5),locus()) }
  ;

switch_clauses:
                          { (AnyClause,None) }
  | DEFAULT COLON stmts   { (AnyClause,Some $3) }
  | CASE ID LPAREN ID RPAREN COLON stmts switch_clauses 
      { add_union_clause $2 (Some $4) $7 $8 }
  | CASE ID COLON stmts switch_clauses 
      { add_union_clause $2 None $4 $5 }
  | CASE CONSTINT COLON stmts switch_clauses
      { add_int_clause $2 $4 $5 }
  | CASE CONSTCHAR COLON stmts switch_clauses
      { add_char_clause $2 $4 $5 }
  ;

stmt_no_trailing:
    block                 { $1 }
  | RETURN SEMICOLON      { (Return None,locus()) }
  | RETURN exp SEMICOLON  { (Return (Some($2)),locus()) }
  | BREAK SEMICOLON       { (Break,locus()) }
  | CONTINUE SEMICOLON    { (Continue,locus()) }
  | exp SEMICOLON         { (Exp $1,locus()) }
  | SEMICOLON             { (Skip,locus()) }
  | SWITCH exp LBRACE switch_clauses RBRACE  
      { let (cs,d) = $4 in
        let must_init d =
	  match d with
	    Some s -> s
	  | None -> err (X.Syntax "switch requires default"); (Skip,locus()) in
        match cs with
	  IntClauses cs -> (IntSwitch($2,cs,must_init d),locus())
	| CharClauses cs -> (CharSwitch($2,cs,must_init d),locus())
        | UnionClauses cs -> (UnionSwitch($2,cs,d),locus())
	| AnyClause ->
	    err (X.Syntax "No non-default clauses in switch");
	    raise Parsing.Parse_error
      }	
  | DO stmt WHILE LPAREN exp RPAREN SEMICOLON    { (Do($2,$5),locus()) }
  ;

exp1:
    LPAREN exp RPAREN        { $2 }
  | ID                       { make_exp (Var $1) }
  | CONSTINT                 { make_exp (Const(Int $1)) }
  | CONSTBOOLEAN             { make_exp (Const(Bool $1)) }
  | CONSTSTRING              { make_exp (Const(String $1)) }
  | CONSTCHAR                { make_exp (Const(Char $1)) }
  | LBRACE COLON prim_type type_mods RBRACE
      { make_exp (ConstArray ([],Some(make_type $3 $4))) }
  | LBRACE arg_list RBRACE   { make_exp (ConstArray ($2,None)) }
  | NULL                     { make_exp (Const(Null)) }
  | SIZE LPAREN exp RPAREN   { make_exp (Primop(Size,[$3])) }
  | ORD LPAREN exp RPAREN    { make_exp (Primop(Ord,[$3])) }
  | CHR LPAREN exp RPAREN    { make_exp (Primop(Chr,[$3])) }
  | RAISE ID LPAREN RPAREN   { make_exp (Raise(make_exp (NewExn ($2,None)))) }
  | RAISE ID LPAREN exp RPAREN { make_exp (Raise(make_exp (NewExn($2,Some $4))))}
  | RAISE LPAREN exp RPAREN { make_exp (Raise $3) }
/* Cyclone */
  | CODEGEN LPAREN prim_type ID LPAREN named_param_list RPAREN type_mods block RPAREN
      { make_exp (Codegen { fn_static = false;
			    fn_name = $4;
			    fn_tyvars = [];
			    fn_ret_type = make_type $3 $8;
			    fn_args = $6;
			    fn_body = $9 }) }
  | CODEGEN LPAREN prim_type ID LPAREN RPAREN type_mods block RPAREN
      { make_exp (Codegen { fn_static = false;
			    fn_name = $4;
			    fn_tyvars = [];
			    fn_ret_type = make_type $3 $7;
			    fn_args = [];
			    fn_body = $8 }) }
  | FILL LPAREN exp RPAREN   { make_exp (Fill $3) }
/* End Cyclone */
  | exp1 LPAREN RPAREN             { make_exp (FunCall ($1, ref None, [])) }
  | exp1 LPAREN arg_list RPAREN    { make_exp (FunCall ($1, ref None, $3)) }
  | exp1 LBRACKET exp RBRACKET          { make_exp (Subscript($1,$3)) }
  | exp1 DOT ID                         { make_exp (StructMember ($1, $3)) }
  | exp1 DOT CONSTINT                   { make_exp (TupleMember($1,int32_to_int $3)) }
  ;

new_kw:
      NEW { () }
  | CARET { () }
  ;

exp2:
    exp1                                { $1 }
  | NEWARRAY LPAREN exp COMMA exp RPAREN { make_exp (NewArray ($3, $5)) }
  | new_kw prefixed_type_name LPAREN RPAREN               
      { make_exp (NewStruct ($2,ref None,[])) }
  | new_kw prefixed_type_name LPAREN arg_list RPAREN      
      { make_exp (NewStruct ($2,ref None,$4)) }
  | new_kw prefixed_type_name DOT ID LPAREN exp RPAREN    
      { make_exp (NewUnion($2,ref None,$4,Some $6)) }
  | new_kw prefixed_type_name DOT ID                      
      { make_exp (NewUnion($2,ref None,$4,None)) }
  | new_kw LPAREN RPAREN                  { make_exp (NewTuple([])) }
  | new_kw LPAREN arg_list RPAREN         { make_exp (NewTuple($3)) }
  ;

exp3:
    exp2              { $1 }

  | exp3 PLUSPLUS      { 
    make_exp(Primop(Minus,
		    [make_exp(AssignOp($1,Plus, make_exp(Const(Int i32_1))));
		     make_exp(Const(Int i32_1))]))}
  | exp3 MINUSMINUS      { 
    make_exp(Primop(Plus,
		    [make_exp(AssignOp($1,Minus,make_exp(Const(Int i32_1))));
		     make_exp(Const(Int i32_1))]))}
  | MINUSMINUS exp3 { make_exp(AssignOp($2,Minus,make_exp(Const(Int i32_1)))) }
  | PLUSPLUS exp3   { make_exp(AssignOp($2,Plus, make_exp(Const(Int i32_1)))) }
  | PLUS exp3       { $2 }
      /* make this negate but requires change to abstract syntax! */
  | MINUS exp3      
      { match $2.raw_exp with
      	Const(Int i) -> make_exp (Const(Int (i32_0 -$ i)))
      |	_ -> make_exp(Primop(Minus, [make_exp (Const(Int i32_0)); $2])) 
      }
  | TILDE exp3      { make_exp(Primop(Bitnot,[$2])) }
  | BANG exp3         { make_exp (Primop(Not,[$2])) }
  ;

exp4:
    exp3             { $1 }
  | exp4 exp4op exp3 { make_binop $1 $2 $3 }
  ;
exp5:
    exp4             { $1 }
  | exp5 exp5op exp4 { make_binop $1 $2 $3 }
  ;
exp6:
    exp5             { $1 }
  | exp6 exp6op exp5 { make_binop $1 $2 $3 }
  ;
exp7:
    exp6              { $1 }
  | exp7 exp7op exp6  { make_binop $1 $2 $3 }
  ;
exp8:
    exp7              { $1 }
  | exp8 exp8op exp7  { make_binop $1 $2 $3 }
  ;
exp9:
    exp8              { $1 }
  | exp9 exp9op exp8  { make_binop $1 $2 $3 }
  ;
exp10:
    exp9              { $1 }
  | exp10 exp10op exp9  { make_binop $1 $2 $3 }
  ; 
exp11:
    exp10              { $1 }
  | exp11 exp11op exp10  { make_binop $1 $2 $3 }
  ;

exp4op:
    TIMES                        {Times}
  | DIV                          {Div}
  | PERCENT                      {Mod}
  ;
exp5op:
    PLUS                         {Plus}
  | MINUS                        {Minus}
  ;
exp6op:
    GREATERGREATER               {Bitarshift}
  | LESSLESS                     {Bitlshift}
  | GREATERGREATERGREATER        {Bitlrshift}
  ;
exp7op:
    LESSTHAN                     {Lt}
  | GREATERTHAN                  {Gt}
  | LESSTHANEQ                   {Lte}
  | GREATERTHANEQ                {Gte}
  ;
exp8op:
    EE                           {Eq}
  | NE                           {Neq}
  ;
exp9op:
    AMPER                        {Bitand}
  ;
exp10op:
    CARET                        {Bitxor}
  ;
exp11op:
    PIPE                         {Bitor}
  ;

exp12: 
    exp11             { $1 }
  | exp12 AMPERAMPER exp11    
       { make_exp (Conditional($1,$3,make_exp(Const(Bool false)))) }
  ;
exp13:
    exp12            { $1 }
  | exp13 PIPEPIPE exp12
       { make_exp (Conditional($1,make_exp(Const(Bool true)),$3)) }
  ;
exp14:
    exp13                           { $1 }
  | exp13 QUESTION exp COLON exp14  { make_exp (Conditional ($1, $3, $5)) }
  ;

exp15:
    exp14                { $1 }
  | exp14 exp15op exp15  { make_exp(AssignOp($1,$2,$3)) }
  | exp14 EQUALS exp15   { make_exp (Assign($1, $3))}
  ;
exp15op:
  | PLUSEQUAL               {Plus}
  | MINUSEQUAL              {Minus}
  | TIMESEQUAL              {Times}
  | DIVEQUAL                {Div}
  | MODEQUAL                {Mod}
  | AMPEREQUAL              {Bitand}
  | PIPEEQUAL               {Bitor}
  | CARETEQUAL              {Bitxor}
  | LESSLESSEQ              {Bitlshift}
  | GREATERGREATEREQ        {Bitarshift}
  | GREATERGREATERGREATEREQ {Bitlrshift}
  ;

exp:
   exp15 { $1 }
  ;

arg_list:
    exp                    { [$1] }
  | exp COMMA arg_list     { $1 :: $3 }
  ;

