/**********************************************************************
 * (c) Greg Morrisett, Neal Glew,                                     *
 *     June 1998, all rights reserved.                                *
 **********************************************************************/

/* talparser.mly
 *
 * Basic TAL parser
 *
 */

%{
open Utilities;;
open Identifier;;
open Tal;;

let dword_error () =
  failwith "operand requires dword ptr"
;;

let ck_dword_l op =
  match op with
    Prjl(_,_) -> dword_error ()
  | _ -> ()
;;

let ck_dword_lr op =
  match op with
    Prjr(_,_) | Prjl(_,_) -> dword_error ()
  | _ -> ()
;;

let ck_dword_b op1 op2 =
  match op1 with
    Prjr(_,_) | Prjl(_,_) ->
      (match op2 with
	Immed _ | Tag _ -> dword_error ()
      |	_ -> ())
  | _ -> ()
;;

let mk_label_coerce (o,cs) =
  match o with
    Addr l -> (l,cs)
  | _ -> failwith "bad label coerce"
;;

let mk_reg_coerce (o,cs) =
  match o with
    Reg r -> (r,cs)
  | _ -> failwith "bad register coerce"
;;

let mk_reg o =
  match o with
    Reg r -> r 
  | _ -> failwith "operand must be a register"
;;

let mk_scale n =
  match n with
    1 -> Byte1
  | 2 -> Byte2
  | 4 -> Byte4
  | 8 -> Byte8
  | _ -> failwith "bad scale"
;;

let mk_cc s =
  match String.uppercase s with
    "A" | "NBE" -> Above                  | "LE" | "NG" -> LessEq       
  | "AE" | "NB" | "NC" -> AboveEq	  | "NE" | "NZ" -> NotEq        
  | "B" | "NAE" | "C" -> Below		  | "NO" -> NotOverflow         
  | "BE" | "NA" -> BelowEq		  | "NS" -> NotSign             
  | "E" | "Z" -> Eq			  | "O" -> Overflow             
  | "G" | "NLE" -> Greater		  | "PE" | "P" -> ParityEven    
  | "GE" | "NL" -> GreaterEq		  | "PO" | "NP" -> ParityOdd    
  | "L" | "NGE" -> Less			  | "S" -> Sign                 
  | _ -> failwith "bad condition code"
;;

let process_prj (o,cs) n =
  match o with
    Reg r -> Prjr ((r,cs),n)
  | Addr l -> Prjl ((l,cs),n)
  | _ -> failwith "bad projection for genop"
;;

type bi = BInum of int | BIstr of string;;

let mk_data_coerce (go,clist) =
  match go with None -> ((),clist) | Some _ -> failwith "bad data coerce"
;;

let process_byte_list bis =
  let f bi = match bi with BInum _ -> 1 | BIstr s -> String.length s in
  let len = List.fold_left (+) 0 (List.map f bis) in
  let s = String.create len in
  let rec g off bis =
    match bis with
      [] -> Dbytes s
    | (BInum n)::bis -> s.[off] <- Char.chr (n land 255); g (off+1) bis
    | (BIstr s1)::bis ->
 	let l = String.length s1 in
	String.blit s1 0 s off l; g (off+l) bis
  in
  g 0 bis
;;

let process_dd_coerce (go,clist) =
  match go with
    None -> if clist<>[] then failwith "bad dd coerce"; Djunk
  | Some (Immed i) -> if clist<>[] then failwith "bad dd coerce"; D4bytes i
  | Some (Tag i) -> Dtag (i,clist)
  | Some (Addr l) -> Dlabel (l,clist)
  | _ -> failwith "bad dd coerce"
;;

let rec make_tapps (o,cs) tcs =
  match tcs with
    [] -> (o,cs)
  | c::tcs -> make_tapps (o,(Tapp c)::cs) tcs
;;

type int_item =
    INTinclude of int_ref
  | INTcon of int_con
  | INTval of identifier*con
;;

let process_int_items is =
  let incs = ref []
  and cons = ref []
  and vals = ref [] in
  let rec loop is =
    match is with
      [] -> ()
    | (INTinclude intref)::is -> incs := intref :: !incs; loop is
    | (INTcon lkcd)::is -> cons := lkcd :: !cons; loop is
    | (INTval (l,c))::is -> vals := (l,c) :: !vals; loop is in
  loop is;
  {int_includes=Array.of_list (List.rev !incs);
    int_cons=Array.of_list (List.rev !cons);
    int_vals=Array.of_list (List.rev !vals)}
;;

type imp_item =
    IMPimport of string
  | IMPexport of string
  | IMPabbrev of (identifier*con)
  | IMPconblock of (identifier*kind*con)
  | IMPcode
  | IMPdata
  | IMPlabel of identifier
  | IMPlabeltype of con
  | IMPinst of instruction
  | IMPcoerce of unit coerce
  | IMPdi of data_item
;;

(* Allowable patterns:
 *   i*a*((.c (l lc i* )* )|(.d (l lc? di* )* ))*
 *   exports allowed anywhere, conblocks allowed anywhere after typedefs
 *)

(* Implement as a state machine:
 *   0: looking for imports
 *   1: seen an abbrev
 *   2: seen a .code
 *   3: seen a .data
 *   5: seen a con block but no code or data
 *)

let process_imp_items items =
  let imports = ref []
  and exports = ref []
  and abbrevs = ref []
  and con_blocks = ref []
  and code_blocks = ref []
  and data_blocks = ref [] in
  let code_block l is =
    let (c,is) = 
      match is with
      	(IMPlabeltype c)::is -> (c,is)
      |	_ -> failwith "no type for code label"
    and insts = ref [] in
    let rec loop is =
      match is with
      	(IMPinst i)::is -> insts := i :: !insts; loop is
      |	_ -> is in
    let is = loop is in
    ((l,c,Array.of_list (List.rev !insts)),is) in
  let data_block l is =
    let (co,is) = 
      match is with
      	(IMPlabeltype c)::is -> (Some c,is)
      |	_ -> (None,is) in
    let (clist,is ) =
      match is with
	(IMPcoerce ((),clist))::is -> (clist,is)
      |	_ -> ([],is) in
    let dis = ref [] in
    let rec loop is =
      match is with
	(IMPdi di)::is -> dis := di :: !dis; loop is
      |	_ -> is in
    let is = loop is in
    ((l,co,(List.rev !dis,clist)),is) in
  let rec loop s is =
    match s,is with
      _,[] -> ()
    | 0,(IMPimport intref)::is -> imports := intref :: !imports; loop s is
    | _,(IMPexport intref)::is -> exports := intref :: !exports; loop s is
    | (0 | 1),(IMPabbrev lc)::is -> abbrevs := lc :: !abbrevs; loop 1 is
    | (0 | 1 | 2 | 3 | 5),(IMPconblock lkc)::is ->
	con_blocks := lkc :: !con_blocks; loop (if s<2 then 5 else s) is
    | (0 | 1 | 2 | 3 | 5),(IMPcode)::is -> loop 2 is
    | (0 | 1 | 2 | 3 | 5),(IMPdata)::is -> loop 3 is
    | 2,(IMPlabel l)::is ->
 	let (cb,is) = code_block l is in
	code_blocks := cb :: !code_blocks; loop s is
    | 3,(IMPlabel l)::is ->
 	let (db,is) = data_block l is in
	data_blocks := db :: !data_blocks; loop s is
    | _,_ -> failwith "bad items"
  in
  loop 0 items;
  { imports=Array.of_list (List.rev !imports);
    exports=Array.of_list (List.rev !exports);
    imp_abbrevs=Array.of_list (List.rev !abbrevs);
    con_blocks=Array.of_list (List.rev !con_blocks);
    code_blocks=Array.of_list (List.rev !code_blocks);
    data_blocks=Array.of_list (List.rev !data_blocks)
  }
;;

let empty_regs = Dict.empty (compare);;

let mk_id s =
  let l = String.length s in
  let i = ref (l - 1) in
  while !i>=0 & s.[!i]<>'$' do decr i done;
  if !i<0 or !i=l-1 then
    id_of_string s
  else
    id_make (String.sub s 0 !i) (int_of_string (String.sub s (!i+1) (l- !i-1)))
;;
%}

%token <string> Tident
%token <int> Tnumber
%token <string> Tstring
%token <Tal.variance> Tvar

%token Teol Teof Tlab Trab Tlb Trb Tlsb Trsb Tlcb Trcb Tbar Tcomma Tdot Tstar
%token Tplus Tcolon Tequal Tarrow Ttalinc Tat Tcons Tbackquote Tquestion Tleq
%token Tcaret

%token <string> Tinclude Ttal_export Ttal_import
%token <Tal.scale> TB

%token T_begin_TAL T_end_TAL Tadc Tadd Tal TAll Tand
%token Tarray Tasub Taupd
%token Tbexn Tbl Tbswap Tbtagi Tbtagvar
%token Tcall Tcbw Tcdq Tcl Tclc Tcmc
%token Tcmova Tcmovae Tcmovb Tcmovbe Tcmovc Tcmove Tcmovg Tcmovge Tcmovl
%token Tcmovle Tcmovna Tcmovnae Tcmovnb Tcmovnbe Tcmovnc Tcmovne Tcmovng
%token Tcmovnge Tcmovnl Tcmovnle Tcmovno Tcmovnp Tcmovns Tcmovnz Tcmovo
%token Tcmovp Tcmovpe Tcmovpo Tcmovs Tcmovz
%token Tcmp Tcode Tcoerce Tcwd Tcwde Tdata Tdb Tdd Tdec Tdiv Tdl Tdw Tdword
%token Teax Tebp Tebx Tecx Tedi Tedx
%token Tend Tesi Tesp TExist Texn Texnname Tfallthru Tfn
%token Tidiv Timul Tinc Tint Tinto
%token Tja Tjae Tjb Tjbe Tjc Tje Tjecxz Tjg Tjge Tjl
%token Tjle Tjmp Tjna Tjnae Tjnb Tjnbe Tjnc Tjne Tjng
%token Tjnge Tjnl Tjnle Tjno Tjnp Tjns Tjnz Tjo
%token Tjp Tjpe Tjpo Tjs Tjunk Tjz Tlabeltype
%token Tlahf Tlea
%token Tloopd Tlooped Tloopned Tmalloc Tmov Tmovsx Tmovzx
%token Tmul Tneg Tnop Tnot
%token Tor Tpack Tpop Tpopad Tpopfd Tptr Tpush Tpushad Tpushfd
%token TR Trcl Trcr Trec Tretn TRL
%token Trol Troll Trollsum Tror TS Tsahf Tsal Tsar Tsbb Tse
%token Tseta Tsetae Tsetb Tsetbe Tsetc Tsete Tsetg Tsetge Tsetl
%token Tsetle Tsetna Tsetnae Tsetnb Tsetnbe Tsetnc Tsetne Tsetng
%token Tsetnge Tsetnl Tsetnle Tsetno Tsetnp Tsetns Tsetnz Tseto
%token Tsetp Tsetpe Tsetpo Tsets Tsetz
%token Tshl Tshld Tshr Tshrd TSint Tslot Tsptr Tstc Tsub Tsubsume Tsum 
%token TT TT1 TT2 TT4 TT8 Ttal_struct Ttal_ends Ttapp Ttest TTm TTs Ttype
%token Tunpack Tunroll Tval Txchg Txor

%right Tarrow

%start tal_int
%start tal_imp
%type <Tal.tal_int> tal_int
%type <Tal.tal_imp> tal_imp
%type <Tal.register_state> register_state
%type <Tal.genop> unary_op
%type <Tal.genop Tal.coerce> unary_op_coerce
%type <Tal.genop> anop
%type <Tal.genop Tal.coerce> anop_coerce
%%

tal_int:
  int_items Teof {process_int_items $1}
;

int_items:
   {[]}
| int_item int_items {$1::$2}
| Teol int_items {$2}
;

int_item:
  Tinclude Teol {INTinclude $1}
| Ttype Tlab label Tcolon kind Trab Teol {INTcon ($3,$5,AbsCon)}
| Ttype Tlab label Tcolon kind Tequal con Trab Teol {INTcon ($3,$5,ConcCon $7)}
| Ttype Tlab label Tcolon kind Tleq con Trab Teol {INTcon ($3,$5,BoundCon $7)}
| Tval label Tcomma econ Teol {INTval ($2,$4)}
;

tal_imp:
  prolog imp_items epilog {process_imp_items $2}
;

prolog:
  eols Tinclude Teol eols T_begin_TAL Teol
       {if (String.lowercase $2)<>"tal.inc" then failwith "invalid prolog"}
;

epilog:
  T_end_TAL Teol eols Tend eols Teof {()}
;

eols: {()} | Teol eols {()};

imp_items:
   {[]}
| imp_item imp_items {$1::$2}
| Teol imp_items {$2}
;

imp_item:
  Ttal_import Teol {IMPimport $1}
| Ttal_export Teol {IMPexport $1}
| Ttype Tlab tvar Tequal con Trab Teol {IMPabbrev ($3,$5)}
| Ttype Tlab label Tcolon kind Tequal con Trab Teol {IMPconblock ($3,$5,$7)}
| Tcode Teol {IMPcode}
| Tdata Teol {IMPdata}
| label Tcolon {IMPlabel $1}
| Tlabeltype econ Teol {IMPlabeltype $2}
| instruction Teol {IMPinst $1}
| Tcoerce Tlab coerce1 Trab Teol {IMPcoerce (mk_data_coerce $3)}
| data_item Teol {IMPdi $1}
;

/* TODO: movsx/movzx */

instruction:
/* Generic x86 instructions */
  Tadc binop {ArithBin (Adc,fst $2,snd $2)}
| Tadd binop {ArithBin (Add,fst $2,snd $2)}
| Tand binop {ArithBin (And,fst $2,snd $2)}
| Tbswap reg {Bswap $2}
| Tcall unary_op_coerce {Call $2}
| Tcbw {Conv Cbw}
| Tcdq {Conv Cdq}
| Tclc {Clc}
| Tcmc {Cmc}
| cmovcc reg Tcomma anop_coerce {Cmovcc ($1,$2,$4)}
| Tcmp binop {Cmp (fst $2,snd $2)}
| Tcwd {Conv Cwd}
| Tcwde {Conv Cwde}
| Tdec unary_op {ArithUn (Dec,$2)}
| Tdiv unary_op {ArithMD (Div,$2)}
| Tidiv unary_op {ArithMD (Idiv,$2)}
| imul {$1}
| Tinc unary_op {ArithUn (Inc,$2)}
| Tint Tnumber {Int $2}
| Tinto {Into}
| jcc coerce {Jcc ($1,mk_label_coerce $2)}
| Tjecxz coerce {Jecxz (mk_label_coerce $2)}
| Tjmp anop_coerce {Jmp $2}
| Tlahf {Lahf}
| Tlea reg Tcomma anop {Lea ($2,$4)}
| Tloopd coerce {Loopd (mk_label_coerce $2,None)}
| Tlooped coerce {Loopd (mk_label_coerce $2,Some true)}
| Tloopned coerce {Loopd (mk_label_coerce $2,Some false)}
| Tmov binop2 {Mov (fst $2,snd $2)}
| Tmul unary_op {ArithMD (Mul,$2)}
| Tneg unary_op {ArithUn (Neg,$2)}
| Tnop {Nop}
| Tnot unary_op {ArithUn (Not,$2)}
| Tor binop {ArithBin (Or,fst $2,snd $2)}
| Tpop unary_op {Pop $2}
| Tpopad {Popad}
| Tpopfd {Popfd}
| Tpush unary_op_coerce {Push $2}
| Tpushad {Pushad}
| Tpushfd {Pushfd}
| Trcl unary_op Tcomma shift_amount {ArithSR (Rcl,$2,$4)}
| Trcr unary_op Tcomma shift_amount {ArithSR (Rcr,$2,$4)}
| Tretn {Retn None}
| Tretn Tnumber {Retn (Some $2)}
| Trol unary_op Tcomma shift_amount {ArithSR (Rol,$2,$4)}
| Tror unary_op Tcomma shift_amount {ArithSR (Ror,$2,$4)}
| Tsahf {Sahf}
| Tsal unary_op Tcomma shift_amount {ArithSR (Sal,$2,$4)}
| Tsar unary_op Tcomma shift_amount {ArithSR (Sar,$2,$4)}
| Tsbb binop {ArithBin (Sbb,fst $2,snd $2)}
| setcc reglow {Setcc ($1,Reg $2)}
| Tshl unary_op Tcomma shift_amount {ArithSR (Shl,$2,$4)}
| Tshld anop Tcomma reg Tcomma shift_amount {Shld ($2,$4,$6)}
| Tshr unary_op Tcomma shift_amount {ArithSR (Shr,$2,$4)}
| Tshrd anop Tcomma reg Tcomma shift_amount {Shrd ($2,$4,$6)}
| Tstc {Stc}
| Tsub binop {ArithBin (Sub,fst $2,snd $2)}
| Ttest binop {Test (fst $2,snd $2)}
| Txchg anop Tcomma reg {Xchg ($2,$4)}
| Txor binop {ArithBin (Xor,fst $2,snd $2)}
/* TAL specific instructions */
| Tasub reg Tcomma array_arg Tcomma Tnumber Tcomma reg Tcomma genop
    {Asub ($2,$4,$6,$8,$10)}
| Taupd array_arg Tcomma Tnumber Tcomma reg Tcomma reg Tcomma genop
    {Aupd ($2,$4,$6,$8,$10)}
| Tbexn reg Tcomma genop Tcomma coerce {Bexn ($2,$4,mk_label_coerce $6)}
| Tbtagi Tident Tcomma reg Tcomma Tnumber Tcomma coerce
    {Btagi ($4,$6,mk_label_coerce $8,mk_cc $2)}
| Tbtagvar Tident Tcomma Tlsb reg Tplus Tnumber Trsb Tcomma Tnumber Tcomma
    coerce
    {Btagvar ($5,$7,$10,mk_label_coerce $12,mk_cc $2)}
| Tcoerce coerce {Coerce (mk_reg_coerce $2)}
| Tfallthru {Fallthru []}
| Tfallthru Tlab Trab {Fallthru []}
| Tfallthru econlist {Fallthru $2}
| Tmalloc Tnumber Tcomma mallocarg {Malloc ($2,$4)}
| Tunpack tvar Tcomma reg Tcomma anop_coerce {Unpack ($2,$4,$6)}
;

array_arg:
  coerce Tplus Tnumber {process_prj $1 $3}
| coerce {process_prj $1 0}
;

cmovcc:
  Tcmova {Above}
| Tcmovae {AboveEq}
| Tcmovb {Below}
| Tcmovbe {BelowEq}
| Tcmovc {Below}
| Tcmove {Eq}
| Tcmovg {Greater}
| Tcmovge {GreaterEq}
| Tcmovl {Less}
| Tcmovle {LessEq}
| Tcmovna {BelowEq}
| Tcmovnae {Below}
| Tcmovnb {AboveEq}
| Tcmovnbe {Above}
| Tcmovnc {AboveEq}
| Tcmovne {NotEq}
| Tcmovng {LessEq}
| Tcmovnge {Less}
| Tcmovnl {GreaterEq}
| Tcmovnle {Greater}	
| Tcmovno {NotOverflow}
| Tcmovnp {ParityOdd}
| Tcmovns {NotSign}                
| Tcmovnz {NotEq}
| Tcmovo {Overflow}                    
| Tcmovp {ParityEven}      
| Tcmovpe {ParityEven}
| Tcmovpo {ParityOdd}
| Tcmovs {Sign}
| Tcmovz {Eq}	
;

jcc:
  Tja {Above}
| Tjae {AboveEq}
| Tjb {Below}
| Tjbe {BelowEq}
| Tjc {Below}
| Tje {Eq}
| Tjg {Greater}
| Tjge {GreaterEq}
| Tjl {Less}
| Tjle {LessEq}
| Tjna {BelowEq}
| Tjnae {Below}
| Tjnb {AboveEq}
| Tjnbe {Above}
| Tjnc {AboveEq}
| Tjne {NotEq}
| Tjng {LessEq}
| Tjnge {Less}
| Tjnl {GreaterEq}
| Tjnle {Greater}	
| Tjno {NotOverflow}
| Tjnp {ParityOdd}
| Tjns {NotSign}                
| Tjnz {NotEq}
| Tjo {Overflow}                    
| Tjp {ParityEven}      
| Tjpe {ParityEven}
| Tjpo {ParityOdd}
| Tjs {Sign}
| Tjz {Eq}	
;

/* Because Imul has three cases we need to do the dword checks here to
 * avoid parsing conflicts
 */

imul:
  Timul ptr_opt genop {if $2 then ck_dword_lr $3; ArithMD (Imul1,$3)}
| Timul ptr_opt genop Tcomma ptr_opt genop
    {if $2 then ck_dword_b $3 $6;
     if $5 then ck_dword_b $6 $3;
     ArithBin (Imul2,$3,$6)}
| Timul ptr_opt genop Tcomma ptr_opt genop Tcomma Tnumber
    {if not $2 then failwith "imul register has dword ptr";
     if $5 then ck_dword_l $6;
     Imul3 (mk_reg $3,$6,$8)}
;

mallocarg:
  Tlab mallocarg_aux Trab {$2}
| Tlab Texnname Tlb con Trb Trab {Mexnname $4}
;

mallocarg_aux:
  Tlsb mallocarg_auxs0 Trsb {Mprod $2}
| Tcolon con {Mfield $2}
| Tarray Tlb Tnumber Tcomma TB Trb {Mbytearray ($5,$3)}
;

mallocarg_auxs0:
    {[]}
| mallocarg_auxs {$1}
;

mallocarg_auxs:
  mallocarg_aux  {[$1]}
| mallocarg_aux Tcomma mallocarg_auxs {$1::$3}
;

shift_amount:
  Tnumber {Some $1}
| Tcl {None}
;

setcc:
  Tseta {Above}
| Tsetae {AboveEq}
| Tsetb {Below}
| Tsetbe {BelowEq}
| Tsetc {Below}
| Tsete {Eq}
| Tsetg {Greater}
| Tsetge {GreaterEq}
| Tsetl {Less}
| Tsetle {LessEq}
| Tsetna {BelowEq}
| Tsetnae {Below}
| Tsetnb {AboveEq}
| Tsetnbe {Above}
| Tsetnc {AboveEq}
| Tsetne {NotEq}
| Tsetng {LessEq}
| Tsetnge {Less}
| Tsetnl {GreaterEq}
| Tsetnle {Greater}	
| Tsetno {NotOverflow}
| Tsetnp {ParityOdd}
| Tsetns {NotSign}                
| Tsetnz {NotEq}
| Tseto {Overflow}                    
| Tsetp {ParityEven}      
| Tsetpe {ParityEven}
| Tsetpo {ParityOdd}
| Tsets {Sign}
| Tsetz {Eq}	
;

anop: ptr_opt genop {if $1 then ck_dword_l $2; $2};
anop_coerce: ptr_opt coerce {if $1 then ck_dword_l (fst $2); $2};
unary_op: ptr_opt genop {if $1 then ck_dword_lr $2; $2};
unary_op_coerce: ptr_opt coerce {if $1 then ck_dword_lr (fst $2); $2};
binop:
  ptr_opt genop Tcomma ptr_opt genop
    {if $1 then ck_dword_b $2 $5;
     if $4 then ck_dword_b $5 $2;
     ($2,$5)}
;
binop2:
  ptr_opt genop Tcomma ptr_opt coerce
    {if $1 then ck_dword_b $2 (fst $5);
     if $4 then ck_dword_b (fst $5) $2;
     ($2,$5)}
;

ptr_opt:
   {true}
| Tdword Tptr {false}
;

genop:
  Tnumber {Immed $1}
| TS Tlb Tnumber Trb {Tag $3}
| reg {Reg $1}
| label {Addr $1}
| Tlsb coerce Tplus Tnumber Trsb {process_prj $2 $4}
| Tlsb coerce Trsb {process_prj $2 0}
;

coerce:
  coerce1
    {let (go,clist) = $1 in
    match go with None -> failwith "bad coercion"
    | Some g -> (g,clist)}
;

coerce1:
  genop {(Some $1,[])}
| Tquestion {(None,[])}
| Tpack Tlb econ Tcomma coerce1 Tcomma econ Trb
    {let (o,cs) = $5 in (o,Pack ($3,$7)::cs)}
| Ttapp Tlb coerce1 Tcomma econlist Trb {make_tapps $3 $5}
| Troll Tlb econ Tcomma coerce1 Trb
    {let (o,cs) = $5 in (o,Roll $3::cs)}
| Tunroll Tlb coerce1 Trb 
    {let (o,cs) = $3 in (o,Unroll ::cs)}
| Tsum Tlb econ Tcomma coerce1 Trb
    {let (o,cs) = $5 in (o,Tosum $3::cs)}
| Trollsum Tlb econ Tcomma coerce1 Trb
    {let (o,cs) = $5 in (o,RollTosum $3::cs)}
| Trec Tlb coerce1 Trb
    {let (o,cs) = $3 in (o,Fromsum ::cs)}
| Texn Tlb coerce1 Trb
    {let (o,cs) = $3 in (o,Toexn ::cs)}
| Tarray Tlb Tnumber Tcomma Tnumber Tcomma coerce1 Trb
    {let (o,cs) = $7 in (o,Toarray ($3,$5)::cs)}
| Tslot Tlb Tnumber Tcomma Tnumber Tcomma coerce1 Trb
    {let (o,cs) = $7 in (o,Slot ($3,$5)::cs)}
| Tsubsume Tlb econ Tcomma coerce1 Trb
    {let (o,cs) = $5 in (o,Subsume $3::cs)}
;

reg:
  Teax {Eax}
| Tebx {Ebx}
| Tecx {Ecx}
| Tedx {Edx}
| Tesi {Esi}
| Tedi {Edi}
| Tebp {Ebp}
| Tesp {Esp}
| TR Tlb Tident Trb {Virt (mk_id $3)}
;

reglow:
  Tal {Eax}
| Tbl {Ebx}
| Tcl {Ecx}
| Tdl {Edx}
| TRL Tlb Tident Trb {Virt (mk_id $3)}
;

/* todo: dd label_coerce */

data_item:
  Tdb byte_list {process_byte_list $2}
| Tdw Tnumber {D2bytes $2}
| Tdd coerce1 {process_dd_coerce $2}
| Texnname econ {Dexnname $2}
| Ttal_struct {Dup}
| Ttal_ends {Ddown}
;

byte_list:
  byte_item byte_list_rest {$1::$2}
;

byte_list_rest:
   {[]}
| Tcomma byte_item byte_list_rest {$2::$3}
;

byte_item:
  Tnumber {BInum $1}
| Tstring {BIstr $1}
;

econ:
  Tlab con Trab {$2}
;

econlist:
  Tlab conlist Trab {$2}
;

/* NG - not used right now
ekind:
  Tlab kind Trab {$2}
;
*/

conlist0:
   {[]}
| conlist {$1}
;

conlist:
  con {[$1]}
| con Tcomma conlist {$1::$3}
;

con: con1 {$1};

con1:
  con2 {$1}
| Tfn vck vcks Tdot con1
    {List.fold_right (fun (v,k) c -> defcon(Clam (v,k,c))) ($2::$3) $5}
;

con2:
  con3 {$1}
| con2 con3 {defcon (Capp ($1,$2))}
| Tsptr con3 {defcon (Csptr $2)}
;

con3:
  con4 {$1}
| con4 Tdot Tnumber {defcon(Cproj ($3,$1))}
;

con4:
  con5 {$1}
| TAll Tlsb vck vcks Trsb Tdot con4
    {List.fold_right (fun (v,k) c -> defcon(Cforall (v,k,c))) ($3::$4) $7}
| TExist Tlsb vck vcks Trsb Tdot con4
    {List.fold_right (fun (v,k) c -> defcon(Cexist (v,k,c))) ($3::$4) $7}
| Tcaret Tdot Tlsb tags Trsb {defcon (Chptr ($4,None))}
| Tcaret opt_tags con4 {defcon (Chptr ($2,Some $3))}
;

con5:
  con6 {$1}
| con6 Tat con5 {defcon(Cappend ($1,$3))}
;

con6:
  con7 {$1}
| con7 Tcons con6 {defcon(Ccons ($1,$3))}
;

con7:
  con100 {$1}
| con7 Tvar {defcon (Cfield ($1,$2))}
;

con100:
  tvar {defcon(Cvar $1)}
| Tlsb conlist0 Trsb {defcon(Ctuple $2)}
| Tbackquote label {defcon(Clab $2)}
| pcon {defcon(Cprim $1)}
| Trec Tlb reclist Trb {defcon(Crec $3)}
| register_state {defcon(Ccode $1)}
| Tstar Tlsb conlist0 Trsb {defcon(Cprod $3)}
| Tplus Tlsb conlist0 Trsb {defcon(Csum $3)}
| Tarray Tlb con Tcomma con Trb {defcon (Carray ($3,$5))}
| TS Tlb con Trb {defcon (Csing $3)}
| Tse {defcon(Cempty)}
| Tlb con Trb {$2}
;

vcks:
   {[]}
| vck vcks {$1::$2}
;

vck:
  tvar Tcolon kind {($1,$3)}
;

opt_tags:
   {[]}
| Tdot Tlb tags Trb {$3}
;

tags:
  Tnumber {[$1]}
| Tnumber Tcomma tags {$1::$3}
;

pcon:
  TB {PCbytes $1}
| Tjunk Tnumber {PCjunk $2}
| Texn {PCexn}
| Texnname {PCexnname}
| Tnumber {PCint $1}
;

reclist:
  recitem {[$1]}
| recitem Tcomma reclist {$1::$3}
;

recitem:
  tvar Tcolon kind Tdot con {($1,$3,$5)}
;

register_state:
  Tlcb rccs Trcb {$2}
;

rccs:
   {rs_empty}
| rccs1 {$1 (*| reg Tcolon con rccs1 {rs_set_reg $4 $1 $3}*)}
;

rccs1:
  reg Tcolon con {rs_set_reg rs_empty $1 $3}
| rccs1 Tcomma reg Tcolon con {rs_set_reg $1 $3 $5}
;

kind:
  Tlb kind Trb {$2}
| TT {Ktype}
| TT1 {Kbyte Byte1}
| TT2 {Kbyte Byte2}
| TT4 {Kbyte Byte4}
| TT8 {Kbyte Byte8}
| TTm {Kmem}
| TTm Tnumber {Kmemi $2}
| TTs {Kstack}
| TSint {Kint}
| kind Tarrow kind {Karrow ($1,$3)}
| Tstar Tlsb kind_list Trsb {Kprod $3}
;

kind_list:
   {[]}
| kind kind_list_rest {$1::$2}
;

kind_list_rest:
   {[]}
| Tcomma kind kind_list_rest {$2::$3}
;

label:
  Tident {mk_id $1}
;

tvar:
  Tident {mk_id $1}
;

/* EOF: talparser.mly */
