(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,                   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

{
open Popparse;;

module X = Poperr;;
module L = Poplocus;;

let get_lexeme = Lexing.lexeme;;
let get_lexeme_char = Lexing.lexeme_char;;
let get_lexeme_start = Lexing.lexeme_start;;
let get_lexeme_end = Lexing.lexeme_end;;
let get_pos lexbuf = (get_lexeme_start lexbuf, get_lexeme_end lexbuf) ;;
let err e buf = raise (X.CompilerError (X.Elexer e,
					L.create(get_lexeme_start buf, 
						 get_lexeme_end buf))) ;;


let runaway_start = ref (0, 0) ;;
let runaway_err e =  raise (X.CompilerError (X.Elexer e, L.create !runaway_start));;

let line = ref 1;;
let process_newline () = line := (!line) + 1;;
let reset_lexer () = line := 1;;

let char_for_backslash c = 
  match c with
    'n' -> '\010'
  | 'r' -> '\013'
  | 'b' -> '\008'
  | 't' -> '\009'
  | c -> c

let char_for_decimal_code lexbuf i = 
  let c = 100 * (Char.code(Lexing.lexeme_char lexbuf i) - 48) +
           10 * (Char.code(Lexing.lexeme_char lexbuf (i+1)) - 48) +
                (Char.code(Lexing.lexeme_char lexbuf (i+2)) - 48) in
  Char.chr(c land 0xff)

let rw = Hashtbl.create 101;;
List.iter (fun (s,t) -> Hashtbl.add rw s t)
    [ "boolean",BOOLEAN; 
      "break",BREAK; 
      "case",CASE; 
      "char",CHAR;
      "chr",CHR;
      "compile",COMPILE;
      "const",CONST;
      "continue",CONTINUE; 
      "do",DO; 
      "default",DEFAULT; 
      "else",ELSE; 
      "extern",EXTERN;
      "false",CONSTBOOLEAN false;
      "for",FOR; 
      "hole",HOLE; 
      "if",IF; 
      "int",INT; 
      "new",NEW; 
      "newarray",NEWARRAY; 
      "null",NULL; 
      "ord",ORD;
      "private",PRIVATE;
      "public",PUBLIC;
      "abstract",ABSTRACT;
      "resume",RESUME; 
      "return",RETURN; 
      "size",SIZE; 
      "static",STATIC; 
      "string",STRING;
      "struct",STRUCT; 
      "suspend",SUSPEND; 
      "switch",SWITCH;
      "union",UNION;
      "true",CONSTBOOLEAN true; 
      "void",VOID; 
      "while",WHILE
    ]
;;

let process_id s =
  try Hashtbl.find rw s with Not_found -> ID s
;;

let comment_depth = ref 0 ;;

let string_buffer = ref (String.create 100) ;;
let string_size = ref 100;;
let string_pos = ref 0;;
let store_string_char char =
  begin
    if !string_pos >= !string_size then
      let str = String.create(2 * (!string_size))
      in
         begin
	   String.blit !string_buffer 0 str 0 !string_size;
	   string_buffer := str;
	   string_size := 2 * !string_size
	 end
    else ();
    String.set !string_buffer !string_pos char;
    string_pos := 1 + !string_pos
  end
;;

let get_stored_string () = 
  let str = String.sub !string_buffer 0 !string_pos
  in (string_pos := 0; str)
;;

}

rule token = parse
  ['A'-'Z''a'-'z']['A'-'Z''a'-'z''0'-'9''_']*
                                  { process_id (Lexing.lexeme lexbuf) }
| ['0'-'9']+  
                                  { CONSTINT
				      (int_of_string (Lexing.lexeme lexbuf)) }
| "("             		  { LPAREN }
| ")"             		  { RPAREN }
| "{"             		  { LBRACE }
| "}"             		  { RBRACE }
| "["             		  { LBRACKET }
| "]"             		  { RBRACKET }
| "+"             		  { PLUS }
| "-"             		  { MINUS }
| "*"             		  { TIMES }
| "/"             		  { DIV }
| "->"                            { ARROW }
| "=="            		  { EE }
| "!="            		  { NE }
| "="             		  { EQUALS }
| "!"             		  { BANG }
| "?"             		  { QUESTION }
| ":"             		  { COLON }
| ";"             		  { SEMICOLON }
| "."             		  { DOT }
| ","             		  { COMMA }
| "<="            		  { LESSTHANEQ }
| ">="            		  { GREATERTHANEQ }
| "<"             		  { LESSTHAN }
| ">"             		  { GREATERTHAN }
| "++"            		  { PLUSPLUS }
| "--"            		  { MINUSMINUS }
| "+="            		  { PLUSEQUAL }
| "-="            		  { MINUSEQUAL }
| "*="            		  { TIMESEQUAL }
| "/="            		  { DIVEQUAL }
| "&&"            		  { AMPERAMPER }
| "||"            		  { PIPEPIPE }  
| "&"                             { AMPER }
| "|"                             { PIPE }
| "<<"                            { LESSLESS }
| ">>"                            { GREATERGREATER }
| "%"                             { PERCENT }
| ['\n' ]                         { process_newline(); token lexbuf }
| [' ' '\010' '\013' '\009' '\012']+  { token lexbuf }
| "//"[^'\n']*'\n'                { process_newline(); token lexbuf }
| "/*"                            { comment_depth := 1; 
				    runaway_start := get_pos lexbuf; 
				    comment lexbuf; 
				    token lexbuf }
| "\""                            { string_pos := 0; 
				    runaway_start := get_pos lexbuf;
				    string lexbuf; 
				    CONSTSTRING(get_stored_string()) }
| "'" [^ '\\' '\''] "'"           
    { CONSTCHAR(Lexing.lexeme_char lexbuf 1) }                             
| "'" '\\' ['\\' '\'' 'n' 't' 'b' 'r'] "'"
    { CONSTCHAR(char_for_backslash (Lexing.lexeme_char lexbuf 2)) }
| "'" '\\' ['0'-'9'] ['0'-'9'] ['0'-'9'] "'"
    { CONSTCHAR(char_for_decimal_code lexbuf 2) }
| eof                           { EOF }
| _ { err (X.NonWhitespace(get_lexeme lexbuf)) lexbuf }

and string = parse
    "\""            { () }
  | "\\\n"          { process_newline(); string lexbuf }
  | "\\\t"          { string lexbuf }
  | "\\ "           { string lexbuf }
  | "\\\\"          { store_string_char '\\'; string lexbuf }
  | "\\n"           { store_string_char '\n'; string lexbuf }
  | "\\t"           { store_string_char '\t'; string lexbuf }
  | "\\\""          { store_string_char '\034'; string lexbuf }
  | [' '-'~']       { store_string_char (Lexing.lexeme_char lexbuf 0);
		      string lexbuf }
  | eof             { runaway_err X.RunawayString }
  | _               { err (X.IllegalStringCharacter (get_lexeme lexbuf)) lexbuf }
and comment = parse
   "/*"             { incr comment_depth; comment lexbuf }
 | "*/"             { decr comment_depth; 
		      if !comment_depth > 0 then (comment lexbuf)
		    }
 | '\n'             { process_newline(); comment lexbuf }
 | eof      { runaway_err X.RunawayComment}
 | _        { comment lexbuf }
