(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew,                                     *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

open Utilities;;
open Talutil;;
open Talcomp;;
open Talicache;;

let verbose = ref false;;
let stats = ref false;;
let interfaces = ref false;;
let linkc = ref false;;
let progc = ref false;;

let intts = ref [];;

let compileFile filename = 
  let basename = Filename.chop_extension filename in
  let ic = open_in filename in
  let lb = Lexing.from_channel ic in
  try
    Tallex.reset_lexer ();
    let talm = Talparser.tal_imp Tallex.main lb in
    close_in ic;
    if !verbose then dumptalm basename talm;
    begin
      try 
	let intt = Talverify.verify_imp get_tali Talctxt.empty_ctxt talm in
	print_string "TAL verified"; 
	print_newline ();
	if !stats then
	  begin
	    Talcon.print_stats();
	    Gc.print_stat stdout
	  end;
	if !interfaces then
	  begin
	    let (it,et) = intt in
	    print_string "TAL Interface"; print_newline ();
	    print_string "Imports"; print_newline ();
	    Talpp.print_tal_int_type Format.std_formatter Talpp.std_options it;
	    Format.print_newline ();
	    print_string "Exports"; print_newline ();
	    Talpp.print_tal_int_type Format.std_formatter Talpp.std_options et;
	    Format.print_newline ()
	  end;
	intts := intt :: !intts
      with 
	Talctxt.Talverify (ctxt,ve) ->
	  print_newline();
	  print_string "TAL verification failed";
	  print_newline ();
	  Talpp.print_Talverify
	    Format.std_formatter tal_print_options (ctxt,ve);
	  Format.print_newline ()
      |	Talctxt.Talfail -> ()
    end 
  with x -> try dumpreaderror filename lb x with Talctxt.Talfail -> ()
;;

let imports = ref ["stdlib.tali";"tal.tali"];;

let main () =
  Arg.parse
    ["-v",Arg.Set verbose,"verbose output";
      "-s",Arg.Set stats,"print statsitics";
      "-I",Arg.String add_include,"add directory to search list";
      "-i",Arg.Set interfaces,"print module interfaces";
      "-lc",Arg.Set linkc,"do linking checks";
      "-pc",Arg.Set progc,"do program completeness checks";
      "-S",Arg.String (fun s -> imports := s :: !imports),
        "add library interface";
      "-runtime",Arg.String (fun s -> runtime:=s),"set runtime directory"]
    compileFile
    "talc: usage: options file ...";
  if !linkc then
    begin try
      let intt = Tallinkchk.verify_link (List.rev !intts) in
      Printf.printf "LINK Verified\n";
      if !interfaces then
	begin
	  let (it,et) = intt in
	  print_string "TAL Linked Interface"; print_newline ();
	  print_string "Imports"; print_newline ();
	  Talpp.print_tal_int_type Format.std_formatter Talpp.std_options it;
	  Format.print_newline ();
	  print_string "Exports"; print_newline ();
	  Talpp.print_tal_int_type Format.std_formatter Talpp.std_options et;
	  Format.print_newline ()
	end;
      if !progc then
	begin try
	  Tallinkchk.set_program_interface
	    get_tali (Array.of_list (List.rev !imports)) [|"tal_prog.tali"|];
	  Tallinkchk.verify_program intt;
	  Printf.printf "PROG Verified\n"
	with
	  Talctxt.Talverify (ctxt,ve) ->
	    print_newline();
	    print_string "PROG verification failed";
	    print_newline ();
	    Talpp.print_Talverify
	      Format.std_formatter tal_print_options (ctxt,ve);
	    Format.print_newline ()
    	| Talctxt.Talfail -> ()
	end
    with 
      Talctxt.Talverify (ctxt,ve) ->
	print_newline();
	print_string "LINK verification failed";
	print_newline ();
	Talpp.print_Talverify
	  Format.std_formatter tal_print_options (ctxt,ve);
	Format.print_newline ()
    | Talctxt.Talfail -> ()
    end;
  ()
;;

Printexc.catch main ();;

(* EOF: talc.ml *)
