(**********************************************************************)

(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,                   *)

(*     June 1998, all rights reserved.                                *)

(**********************************************************************)

exception Terminate



let link = ref true;;

let set_no_link() = link := false;;

let assemble = ref true;;

let set_no_assemble() = (assemble := false; set_no_link());;

let executable = ref "a.out";;

let set_executable s = executable := s;;

let talfile = ref "";;

let set_talfile s = talfile := s;;

let objectfile = ref "";;

let set_objectfile s = objectfile := s;;

let set_no_optimize () = () (* Popcompile.optimize_tal := false;; *)





let filenames s =

  begin

    let in_chars = Stream.of_string s in

    let rec from_list (a:char list):string = 

      match a with [] -> "" | (c::tl) -> (Char.escaped c) ^ (from_list tl) 

    in  

    let rec next_filename c (a:char list) : string*bool = 

      try

	let next_c = Stream.next c in

	if (next_c = ' ' or next_c = '\t' or next_c = '\n') 

	then (from_list (List.rev a),false)

	else (next_filename c (next_c::a))

      with Stream.Failure -> (from_list (List.rev a),true)

    in

    let rec aux c a =  

      begin

	let (f,eof) = next_filename c [] in

	let a' = if f = "" then a else f::a in

	if eof then a' else (aux c a')

      end in

    List.rev (aux in_chars [])

  end

let command s =

  Printf.printf "%s\n" s; flush stdout;

  Sys.command s;

  ()

;;



let objfiles : string list ref = ref [];;



let compileFile filename =

  let fileBase = Filename.chop_extension filename in

  let modname = Filename.basename fileBase in

  let talfile = if (!talfile) = "" then fileBase ^ ".tal" else !talfile in

  let impfile = fileBase ^ "_i.tali" in

  let expfile = fileBase ^ "_e.tali" in

  let objectfile = 

      if (!objectfile) = "" then fileBase ^ ".obj" else !objectfile in

  Printf.printf "Compiling file: %s\n" fileBase; flush stdout;

  let fileChannel = open_in filename in

  let lexbuf = Lexing.from_channel fileChannel in

  let _ = Poperrhandle.set_filename filename true in

  try

    let (implementation,imports,exports) =

      try

      	let decls = Popparse.top Poplex.token lexbuf in

      	let (decls,env) = Poptype.type_check decls in

      	Popcompile.code_gen modname impfile expfile (decls,env)     

      with 

      	Poperr.CompilerError(e,loc) -> (Poperrhandle.signal_error e loc;

				      	close_in fileChannel;

				      	raise Terminate)

      | exn -> (close_in fileChannel; raise exn)

    in

    Talcomp.write_int (modname^"_i") impfile imports;

    Talcomp.write_int (modname^"_e") expfile exports;

    Talcomp.write_imp fileBase talfile implementation;

    command ("talc.exe " ^ talfile);

    if (!assemble) then

      Talcomp.asm true talfile (Some objectfile);

    if (!link) then

      objfiles := objectfile :: !objfiles;

    ()

  with 

    Sys_error str ->

      (print_string "System Error: "; 

       print_string str; 

       print_newline();

       link := false)

  | Parsing.Parse_error -> (link:=false; ())

;;



let doIt s = 

  begin

    let fs = filenames s in 

    let rec aux fs = 

      match fs with 

      	[] -> ()

      |  (f::tl) -> 

	  (try (compileFile f) with Terminate -> link := false);

	  aux tl

    in

    aux fs

  end;;



let do_link () =

    begin

      (if (!link) then

    	(Talcomp.link true (List.rev ("pop_runtime.obj" :: !objfiles)) !executable))

    end;;



let main () =

  Arg.parse 

    [ (*"-r",Arg.String Talcomp.set_runtime,"runtime directory"; *)

      "-c",Arg.Unit set_no_link,"do not link";

      "-o",Arg.String set_executable,"executable file name";

      "-A",Arg.Unit set_no_assemble,"do not assemble";

      "-t",Arg.String set_talfile,"tal file name";

      "-nopt",Arg.Unit set_no_optimize,"don't peephole optimize"

    ] 

    doIt "popcorn: usage: <option> filename ... ";

  do_link ()

;;



Printexc.catch main () ;;

