(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, David Walker,                       *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

(* talverify.mli
 * TAL Verifier
 *
 * Checks operands, instructions, interfaces, and implementations for well
 * formedness.
 *)

open Utilities;;
open Identifier;;
open Tal;;
open Talctxt;;
open Talcon;;

(* Some types used by the verifier *)
let whcon rc = { rcon=rc; isnorm=false; freevars=None };;

let prcon rc = { rcon=rc; isnorm=true;
                 freevars=Some(Set.empty id_compare) };;

let bool_con = prcon(Csum {sum_tags = [0;1]; sum_vt= None});;
let byte4 = prcon(Cprim(PCbytes Byte4));;
let exn_con = prcon(Cprim PCexn);;
let junk_con = prcon(Cprim PCjunk);;
let tag0_con = prcon(Cprim(PCtag 0));;
let tag1_con = prcon(Cprim(PCtag 1));;
let sptr_con = prcon(Cprim(PCstackptr));;
let empty_con = prcon(Cempty);;
let array_con = prcon(Cprim(PCarray));;
let exnname_con = prcon(Cprim(PCexnname));;

(************************************************************************)
(* Coercions                                                            *)
(************************************************************************)

let raw a : 'a coerce = (a,[]) (* raw: 'a -> 'a coerce *)
let coerce (a,clist : 'a coerce) c : 'a coerce = (a, c:: clist)
   (* coerce: 'a coerce -> coercion -> 'a coerce  *)
let get_raw (raw, clist : 'a coerce) = raw

(* Must check constructors before normalising or odd exceptions could result *)

let check_to_sum error ctxt c con =
  begin match c.rcon with
      Csum {sum_tags = st; sum_vt = vt} ->
        (* check to see that o is either a tag or a record *)
      	(match (whnorm ctxt con).rcon with
	  Cprim (PCtag i) -> 
	   (* check that the tag is in the sum *)
	    if not (List.mem i st) then error "tosum: tag not in sum"
       	| Cprod(fs) ->
	   (* check to see if this is the only record in the sum
	    * or otherwise, it begins with a tag and is in the
	    * list of variants for this sum.
	    *)
	    (match vt with
	      None -> error "tosum: no variant or tuple"
	    | Some (Tuple fs') ->
		eqcon ctxt (defcon(Cprod fs)) (defcon(Cprod fs'))
	    | Some (Variants vs) ->
		(match fs with
                  (ctag,_,Init)::fs ->
                    (match (whnorm ctxt ctag).rcon with
                      (Cprim(PCtag i)) ->
		        (try let fs' = List.assoc i vs in
 		   	   eqcon ctxt (defcon(Cprod fs)) 
                             (defcon(Cprod fs'))
		        with Not_found ->
			  error "tosum: variants do not match tuple")
                    | _ -> error "tosum: no tag on variants")
		| _ -> error "Tosum: not a variant"))
       	| Csum {sum_tags = ost; sum_vt = ovt} -> 
	    if List.for_all (fun i -> List.mem i st) ost then 
	      match ovt with 
		None -> ()
	      | Some (Tuple ofs) ->
		  (match vt with
		    None -> error "widden sum: no variant or tuple"
		  | Some(Tuple fs') -> 
                      eqcon ctxt (defcon(Cprod ofs)) (defcon(Cprod fs'))
		  | Some(Variants vs) ->
		      (match ofs with
			(ctag,Read,Init)::fs ->
                          (match (whnorm ctxt ctag).rcon with
                            Cprim(PCtag i) ->
			      (try let fs' = List.assoc i vs in
			           eqcon ctxt (defcon(Cprod ofs)) 
                                   (defcon(Cprod fs'))
			      with Not_found ->
			        error "widden sum: tuple not in variants")
                          | _ -> error "widden sum: no tag on variant")
		      | _ -> error "widden sum: not a variant"))
	      | Some (Variants ovs) ->
		  (match vt with
		    None -> error "widden sum: no variant or tuple"
		  | Some(Tuple fs') ->
		      error "widden sum: can't coerce variants to tuple"
		  | Some (Variants vs) ->
		      let check (i,ofs) =
			try
			  let fs' = List.assoc i vs in
			  eqcon ctxt (defcon(Cprod ofs)) (defcon(Cprod fs'))
			with Not_found ->
			  error "widden sum: variant not in variants" in
		      List.iter check ovs)
            else error "tosum: tag not in sum"
	| _ -> error "tosum: not a tag, record, or variant");
    | _ -> error "tosum: not a sum"
  end;
  c
;;

(* In context ctxt return the type constructor resulting from coercion of con
   by coercion.
   con must be normalized.
 *)
let coercion_con ctxt coercion con =
  let error s = generate_error ctxt (Coercion (con,s,coercion)) in
  match coercion with
    Pack (c1,c2) ->
      let ctxt = set_verify_ctxt ctxt "pack coercion" in
      let (k1,c1) = check ctxt c1
      and (k2,c2) = check_whnorm ctxt c2 in
      (match c2.rcon with
	Cexist (v,k,c2') ->
	  kindleq ctxt k1 k;
	  (* let c = subst c1 v c2' in *)
	  eqcon ctxt con (defcon(Capp(whcon(Clam(v,k,c2')),c1)));
	  c2
      | _ -> error "pack: not an existential"; c2)
  | Tapp c ->
      let ctxt = set_verify_ctxt ctxt "tapp coercion" in
      let (k,c) = check_whnorm ctxt c in
      (match (whnorm ctxt con).rcon with
	Cforall (v1,k1,c1) ->
	  kindleq ctxt k k1;
	  defcon(Capp(whcon(Clam(v1,k1,c1)),c))
      |	_ -> error "tapp: not a forall"; raise Talfail)
  | Roll c ->
      let ctxt = set_verify_ctxt ctxt "roll coercion" in
      let (k,c) = check_whnorm ctxt c in
      eqcon ctxt con (unroll_rec ctxt c); c
  | Unroll ->
      let ctxt = set_verify_ctxt ctxt "unroll coercion" in
      let c = whnorm ctxt con in
      unroll_rec ctxt c
  | Tosum c ->
      let ctxt = set_verify_ctxt ctxt "tosum coercion" in
      let (k,c) = check_whnorm ctxt c in 
      check_to_sum error ctxt c con
  | Fromsum ->
      let ctxt = set_verify_ctxt ctxt "fromsum coercion" in
      (match (whnorm ctxt con).rcon with
	  (* if it's a singleton tag coerce to a tag *)
	Csum {sum_tags = [i]; sum_vt = None} -> whcon(Cprim (PCtag i))
	  (* if it's a singleton record coerce to a record *)
      | Csum {sum_tags = []; sum_vt = Some (Tuple fs)} -> whcon(Cprod fs)
	  (* if it's a singleton variant coerce to a tagged record *)
      | Csum {sum_tags = []; sum_vt = Some (Variants [(i,fs)])} ->
	  whcon(Cprod ((prcon(Cprim (PCtag i)),Read,Init)::fs))
      | _ -> error "fromsum: not a singleton sum"; raise Talfail)
  | RollTosum c ->
      let ctxt = set_verify_ctxt ctxt "rollsum coercion" in
      let (k,c) = check_whnorm ctxt c in
      let cunroll = whnorm ctxt (unroll_rec ctxt c) in 
      check_to_sum error ctxt cunroll con; c
  | Toexn ->
      let ctxt = set_verify_ctxt ctxt "toexn coercion" in
      (match (whnorm ctxt con).rcon with
        Cprod[(ca,Read,Init);(c2,Read,Init)] ->
          (match (whnorm ctxt ca).rcon with
             (Capp(cexn,c1)) -> eqcon ctxt cexn exn_con; eqcon ctxt c1 c2
           | _ -> error "toexn: first component of product not an exnname")
      | _ -> error "toexn: not a product");
      exn_con
  | Toarray ->
      let ctxt = set_verify_ctxt ctxt "toarray coercion" in
      (match (whnorm ctxt con).rcon with
	Cprod ((c1,_,Init)::((c,rw,Init)::_ as fs)) ->
          (match (whnorm ctxt c1).rcon with
             (Cprim (PCtag n)) ->
	  let rec check_fs n fs =
	    if n<>0 then
	      match fs with
		[] -> error "toarray: too few fields"
	      | (_,_,Uninit)::_ -> error "toarray: field not initialised"
	      |	(c1,rw1,Init)::fs ->
		  if rw<>rw1 then error "toarray: ro/rw inconsistency";
		  eqcon ctxt c c1;
		  check_fs (n-1) fs in
	  check_fs n fs;
	  (match (whnorm ctxt c).rcon with
	    Cprim (PCbytes (Byte1 | Byte2 | Byte8 as s)) ->
	      prcon(Cprim (if rw=ReadWrite then PCbytearray s 
                           else PCbytevector s))
	  | _ ->
	      kindleq ctxt (con_kind ctxt c) K4byte;
	      whcon (Capp (prcon (Cprim
                (if rw=ReadWrite then PCarray else PCvector)),c)))
           | _ -> error "toarray: first field not a tag"; raise Talfail)
      |	_ -> error "toarray: not an array form"; raise Talfail)
  | Slot i ->
        (* ctxt.gamma(Esp) = Stackptr(c1::...::cn::tail)
	 * i = sizeof(c1) + ... + sizeof(cn-1)  
         * ------------------------------------------------
         * ctxt |- CoerceS snew:ctxt[Esp:Stackptr(c1::...::cn-1::Junk::tail)] 
	 *
         *)
      let ctxt = set_verify_ctxt ctxt "slot coercion" in
      let rec forget_slot ctxt i cstack =
      	match (i,(whnorm ctxt cstack).rcon) with
	  (0,Ccons(c,tail)) -> 
	    let csz = sizeof ctxt c in
	    if csz = 4 then
	      whcon(Ccons (junk_con, tail))
	    else if csz = 8 then
	      whcon(Ccons (junk_con, whcon(Ccons (junk_con, tail))))
	    else
	      begin error "slot: slot size not 4,8"; empty_con end
	| (0,_) -> begin error "slot: bad offset"; empty_con end
	| (i,Ccons(c,c')) -> 
	    let i' = i - (sizeof ctxt c) in
	    if i' >= 0 then whcon(Ccons (c, forget_slot ctxt i' c'))
	    else begin error "slot: bad offset"; c' end
	| (i,_) -> error "slot: bad offset"; raise Talfail
      in
      (match (whnorm ctxt con).rcon with
	Capp (csptr, sold) ->
          eqcon ctxt csptr sptr_con;
	  whcon(Capp (sptr_con, forget_slot ctxt i sold))
      | _ -> error "slot: not a stack pointer"; raise Talfail)
;;
     
let coerce_con f ctxt (raw,clist) =
  let c = f ctxt raw in
  List.fold_right (coercion_con ctxt) clist c
;;

(************************************************************************)
(* Operands                                                             *)
(************************************************************************)

(* get the current stack type *)
let current_stack_con ctxt =
  match (whnorm ctxt (get_reg_con ctxt Esp)).rcon with
    Capp(sptr_con,c) -> c
  | _ -> generate_error ctxt No_stack_type; empty_con
;;

(* is con a valid stack type if so return the mutator function *)
let valid_stack_con ctxt con =
  verify_stack_tail ctxt con (current_stack_con ctxt)
;;

(* ctxt |- coerce(reg) : c *)
let coerce_reg_con = coerce_con get_reg_con;;

(* ctxt |- coerce(label) : c *)
let coerce_label_con = coerce_con get_label_con;;

(* ctxt |- genop : c *)
let genop_con ctxt genop = 
  let error s = generate_error ctxt (Genop (s,genop)) in
  match genop with
    Immed i -> byte4
  | Tag i ->
      (* try to get some sharing out of common tags *)
      if i = 0 then tag0_con else if i = 1 then tag1_con else
      if i>=2 & i<=255 then prcon(Cprim (PCtag i))
      else begin error "genop: tag out of range"; prcon(Cprim (PCtag i)) end
  | Reg r -> get_reg_con ctxt r
  | Addr l -> get_label_con ctxt l
    (* rc and lc must be either product types or else stack pointer types *)
  | Prjr (rc,i) -> 
      let rc_con = coerce_reg_con ctxt rc in
      (match (whnorm ctxt rc_con).rcon with
	Cprod fs -> 
          (match get_field_offset ctxt i fs with
	    (c,_,Init) -> c
	  | (c,_,Uninit) ->
	      error "genop: prjr: reading uninitialized field"; c)
      | Capp (c1,c) ->
          begin
            match (whnorm ctxt c1).rcon with
             (Cprim PCstackptr) -> 
                begin
                  match rc with
                    (Esp,[]) -> get_stack_offset ctxt i c
                  | _ -> let f = valid_stack_con ctxt c in
	                 get_stack_offset ctxt i c
                end
            | _ -> error "genop: prjr: app not a stackptr"; c
          end
      | _ -> error "genop: prjr: projection not a tuple/stack"; raise Talfail)
  | Prjl (lc,i) -> 
      let lc_con = coerce_label_con ctxt lc in
      (match (whnorm ctxt lc_con).rcon with
        Cprod fs -> 
	  (match get_field_offset ctxt i fs with
	    (c,_,Init) -> c
	  | (c,_,Uninit) ->
	      error "genop: prjl: reading uninitialized field"; c)
      | _ -> error "genop: prjl: projection not a tuple/stack"; raise Talfail)
;;
let coerce_genop_con = coerce_con genop_con;;
    
(* verify that only one of the operands is a path to memory *)
let is_mem gop =
  match gop with
    Immed _ -> false
  | Tag _ -> false
  | Reg _ -> false
  | Addr _ -> false
  | Prjr (_,_) -> true
  | Prjl (_,_) -> true
;;

let valid_binops ctxt gop1 gop2 = 
  if is_mem gop1 & is_mem gop2 then
    generate_error ctxt (Both_mem (gop1,gop2))
;;

let valid_cbinops ctxt gop1 cgop2 = valid_binops ctxt gop1 (get_raw cgop2);;

(* Can a con be written to a genop? *)
(* Checks also that gop has a vaild type *)
let writeable_genop ctxt gop con =
  let error s = generate_error ctxt (Genop (s,gop)) in
  match gop with
    Reg _ -> ()
  | Prjr (rc,i) ->
      let rc_con = coerce_reg_con ctxt rc in
      (match (whnorm ctxt rc_con).rcon with
	Cprod fs -> writeable_field ctxt (get_field_offset ctxt i fs) con
      | Capp (csptr,c) -> (eqcon ctxt csptr sptr_con;
                           writeable_stack_offset ctxt i c con)
      | _ -> error "genop: prjr: projection not a tuple/stack")
  | Prjl (lc,i) ->
      let lc_con = coerce_label_con ctxt lc in
      (match (whnorm ctxt lc_con).rcon with
	Cprod fs -> writeable_field ctxt (get_field_offset ctxt i fs) con
      | _ -> error "genop: prjl: projection not a tuple")
  | _ -> generate_error ctxt Readonly
;;

(* Is operand register or memory? *)
let reg_or_mem_genop ctxt g =
  match g with
    Immed _ -> generate_error ctxt (Genop ("operand is an immediate",g))
  | Tag _ -> generate_error ctxt (Genop ("operand is tag",g))
  | Reg _ -> ()
  | Addr _ -> generate_error ctxt (Genop ("operand is a label",g))
  | Prjr (_,_) -> ()
  | Prjl (_,_) -> ()
;;

(************************************************************************)
(* Instructions                                                         *)
(************************************************************************)

let inst_form ctxt s = generate_error ctxt (Inst_form s);;

exception Terminal_Jmp;;
exception Fall_Thru of ctxt * (con list);;

(*** Generic x86 Instructions ***)

(* Normal case: two byte4s, first is writeable, not both mem.
 * Special case: add to stack pointer type is "free"
 * Special case: sub from ESP is a "stack allocate"
 *
 *  ctxt |- genop1 : byte4  ctxt |- genop2 : byte4  
 *     ctxt |- (genop1,genop2) valid binops
 *     ctxt |- genop1 writeable
 *  ----------------------------------------------
 *  ctxt |- ArithBin ab genop1,genop2 : ctxt
 *
 *  ctxt |- r : Stackptr(c1::....::ci::c)
 *  ctxt |- n : byte4  
 *     ctxt |- (genop1,genop2) valid binops
 *     ctxt |- genop1 writeable
 *  ----------------------------------------------(n=sizeof(c1::...::ci))
 *  ctxt |- Add r,n : ctxt[r:Stacpkptr(c)]
 *
 *  ctxt |- ESP : Sptr(c)
 *  ---------------------------------------------------------
 *  ctxt |- Sub ESP,4n : ctxt[ESP:Sptr(junk1::...::junkn::c)]
 *)
   
let verify_ArithBin ctxt ab genop1 genop2 =
  if ab=Sub & genop1=(Reg Esp) then begin
    match genop2 with
      Immed n ->
   	let rec incr_stack i cstack =
	  if i = 0 then cstack
	  else if i > 0 then
	    incr_stack (i-4) (whcon(Ccons(junk_con, cstack)))
	  else
	    begin inst_form ctxt "ESP - negative or non-4-multiple"; cstack end
	in
    	let cstack = incr_stack n (current_stack_con ctxt) in
 	add_reg ctxt Esp (whcon(Capp(sptr_con,cstack)))
    | _ -> inst_form ctxt "ESP - non-immed"; raise Talfail
  end else begin
    valid_binops ctxt genop1 genop2;
    writeable_genop ctxt genop1 byte4;
    match (whnorm ctxt (genop_con ctxt genop1)).rcon with
      (* A normal binop *)
      Cprim (PCbytes Byte4) ->
	eqcon ctxt byte4 (genop_con ctxt genop2); ctxt
      (* adjusting a pointer onto the stack -- note that this provides
       * "stack free" but it's more general in that we can calculate
       * pointers into the middle of the stack from other pointers into
       * the middle of the stack, as long as the sizes of the intervening
       * types are known (i.e., there's no intervening append). *)
    | Capp (spc,c) ->
        eqcon ctxt spc sptr_con;
	(match ab,genop1,genop2 with
	  (Add,Reg r,Immed i) -> 
	    let c' = get_stack_tail ctxt i c
	    in add_reg ctxt r (whcon(Capp(sptr_con,c')))
	| (_,_,_) -> inst_form ctxt "ArithBin: stack pointer"; raise Talfail)
    | _ -> inst_form ctxt "ArithBin"; raise Talfail
  end
;;

(* ctxt |- genop : byte4   ctxt |- genop writeable
 * -----------------------------------------------
 * ctxt |- ArithUn arithun genop : ctxt
 *)

let verify_ArithUn ctxt arithun genop =
  writeable_genop ctxt genop byte4;
  eqcon ctxt byte4 (genop_con ctxt genop);
  ctxt
;;

(* Div,IDiv: divides Eax by the operand and puts the quotient in Eax and 
 *  remainder in Edx. (Div is unsigned, IDiv is signed)
 * Mul,Imul1: multiples Eax by the operand and puts the 64-bit result in
 *  Eax and Edx.  (Mul is unsigned, Imul1 is signed)
 * In any of the cases, Eax must be a Byte4, the operand must be a Byte4,
 * and the resulting context maps Edx to Byte4.
 *
 * ctxt |- genop : byte4  ctxt |- Eax : byte4
 * -----------------------------------------------
 * ctxt |- ArithMd arithmd genop : ctxt[Edx:byte4]
 *)
let verify_ArithMD ctxt arithmd genop =
  eqcon ctxt byte4 (genop_con ctxt genop);
  eqcon ctxt byte4 (get_reg_con ctxt Eax);
  add_reg ctxt Edx byte4
;;

(* Arithmatic shift operations. If iopt is None, then Ecx is used
 * as the shift amount.  Otherwise, the integer immediate is used
 * as the shift amount.
 *
 * ctxt |- genop : byte4  ctxt |- genop writeable
 * ctxt |- Ecx : byte4
 * ----------------------------------------------
 * ctxt |- ArithSR arithsr genop None : ctxt
 *
 * ctxt |- genop : byte4  ctxt |- genop writeable
 * ----------------------------------------------
 * ctxt |- ArithSR arithsr genop (Some i) : ctxt
 *)

let verify_ArithSR ctxt arithsr genop iopt =
  writeable_genop ctxt genop byte4;
  eqcon ctxt byte4 (genop_con ctxt genop);
  (match iopt with
    Some _ -> ()
  | None -> eqcon ctxt byte4 (get_reg_con ctxt Ecx));
  ctxt
;;

(* Byte-swap 
 * ctxt |- reg : byte4
 * ------------------------
 * ctxt |- Bswap reg : ctxt
 *)

let verify_Bswap ctxt reg = eqcon ctxt byte4 (get_reg_con ctxt reg); ctxt;;

(* Call:
 * This is a little delicate.  The called label must have a code type which is
 * a supertype of the current code type with the return address pushed;
 * however, the return address type is garned from the code type of the called
 * label; however, the called label might ignore the return label and there may
 * be no return type.  In the later case we should treat call as a terminal
 * jump.
 *
 * For now: if the called label defines ESP then it must have a code type on
 * top, use it.  Otherwise its a terminal jump.
 * 
 * ctxt |- cgop : Ccode g1
 * ctxt |- g1(ESP) = Sptr (Ccode g2::c')
 * ctxt |- ctxt.gamma [ESP: Sptr (Ccode g2::(ctxt.gamma(ESP)))] <= g1
 * ------------------------------------------------------------------ESP in g1
 * ctxt |- Call gop : ctxt[gamma: g2]
 *
 * ctxt |- cgop : Ccode g1
 * ctxt |- ctxt.gamma <= g1
 * --------------------------------ESP not in g1
 * ctxt |- Call gop : terminal jump
 *)

let verify_Call ctxt cgop =
  let c = (whnorm ctxt (coerce_genop_con ctxt cgop)) in
  match c.rcon with
    Ccode g2 ->
      let g1 = get_register_state ctxt in
      (try
	let called_st = Dict.lookup g2 Esp in
        let c = whnorm ctxt (current_stack_con ctxt) in
	let cstack = whnorm ctxt called_st in
	match cstack.rcon with
	  Capp(c1,c2) ->
            (match (whnorm ctxt c1).rcon,(whnorm ctxt c2).rcon with
               (Cprim PCstackptr,Ccons(ca,c')) -> 
                (match (whnorm ctxt ca).rcon with
                  (Ccode gret) -> 
   	             eqcon ctxt c c';
	             reg_state_leq ctxt (Dict.insert g1 Esp cstack) g2;
	             set_register_state ctxt gret
                 | _ -> 
                     inst_form ctxt "Call: no return address in pre-condition";
                     raise Talfail)
              | _ -> inst_form ctxt "Call: no ret address in pre-condition";
                     raise Talfail)     
	 | _ ->
	    inst_form ctxt "Call: bad return address or stack";
	    raise Talfail
      with Dict.Absent ->
	reg_state_leq ctxt g1 g2; raise Terminal_Jmp)
  | _ -> inst_form ctxt "Call: operand not code"; raise Talfail
;;

(* Conditional move: it must be the case that if cgop : c, then
 * r : c.  This ensures that after the conditional move, r : c.
 *
 * ctxt |- r : c    ctxt |- cgop : c
 * ----------------------------------
 * ctxt |- Cmovcc r,cgop : ctxt
 *)

let verify_Cmovcc ctxt cond r cgop = 
  eqcon ctxt (get_reg_con ctxt r) (coerce_genop_con ctxt cgop);
  ctxt
;;

(* We generalize to allow comparisons of any two values of equal type.
 * 
 * ctxt |- (genop1,genop2) valid binops
 * ctxt |- genop1 reg_or_mem_genop
 * ctxt |- genop1 : c   ctxt |- genop2 : c
 * -----------------------------------------------
 * ctxt |- Cmp genop1,genop2 : byte4
 *)

let verify_Cmp ctxt genop1 genop2 =
  valid_binops ctxt genop1 genop2; reg_or_mem_genop ctxt genop1;
  eqcon ctxt (genop_con ctxt genop1) (genop_con ctxt genop2);
  ctxt
;;

(* ctxt |- Eax : byte4
 * -----------------------
 * ctxt |- Cbw/Cwde : ctxt
 *
 * ctxt |- Eax : byte4
 * ---------------------------------
 * ctxt |- Cdq/Cwd : ctxt[Edx:byte4]
 *
 * ctxt |- gop : bytei
 * -------------------------------------------(j,i) in {(32,16),(32,8),(16,8)}
 * ctxt |- Mov(s/z)x r,j,gop,i : ctxt[r:byte4]
 *)

let verify_Conv ctxt c =
  let aux r s1 gop s2 =
    (match s1,s2 with Byte2,Byte1 | Byte4,Byte1 | Byte4,Byte2 -> ()
    | _ -> inst_form ctxt "Movsx/Movzx: invalid size combination");
    (match (whnorm ctxt (genop_con ctxt gop)).rcon with
      Cprim (PCbytes s3) ->
        if s2<>s3 then 
          inst_form ctxt "Movsx/Movzx: operand size does not match scale"
    | _ -> inst_form ctxt "Movsx/Movzx: operand not bytes");
    add_reg ctxt r byte4
  in
  match c with
    (Cbw | Cwde) -> 
      eqcon ctxt byte4 (get_reg_con ctxt Eax); ctxt
  | (Cdq | Cwd) -> 
      eqcon ctxt byte4 (get_reg_con ctxt Eax); add_reg ctxt Edx byte4
  | Movsx (r,s1,gop,s2) -> aux r s1 gop s2
  | Movzx (r,s1,gop,s2) -> aux r s1 gop s2
;;

(* ctxt |- genop : byte4
 * ---------------------------------------
 * ctxt |- Imul3 r,genop,i : ctxt[r:byte4]
 *)

let verify_Imul3 ctxt r genop i =
  eqcon ctxt byte4 (genop_con ctxt genop); add_reg ctxt r byte4
;;

(* ctxt |- lc : Ccode(g)    ctxt |- ctxt.gamma <= g
 * ------------------------------------------------
 * ctxt |- Jcc cc,lc : ctxt
 *)

let verify_Jcc ctxt lc =
  (match (whnorm ctxt (coerce_label_con ctxt lc)).rcon with
    Ccode rs -> reg_state_leq ctxt (get_register_state ctxt) rs
  | _ -> inst_form ctxt "Jcc: label for branch is not code");
  ctxt
;;

(* ctxt |- lc : Ccode(g)   ctxt |- ctxt.gamma <= g
 * ctxt |- Ecx : byte4
 * -----------------------------------------------
 * ctxt |- Jecxz lc : ctxt
 *)

let verify_Jecxz ctxt lc =
  eqcon ctxt byte4 (get_reg_con ctxt Ecx); verify_Jcc ctxt lc
;;

(* ctxt |- gc : Ccode(g)    ctxt |- ctxt.gamma <= g
 * ------------------------------------------------
 * ctxt |- branch lc : ctxt
 *)

let verify_Jmp ctxt gc =
  (match (whnorm ctxt (coerce_genop_con ctxt gc)).rcon with
    Ccode rs -> reg_state_leq ctxt (get_register_state ctxt) rs
  | _ -> inst_form ctxt "Jmp: label for branch is not code");
  raise Terminal_Jmp
;;

(* this is really only useful for moving an offset from the stack
 * into a register or moving a label into a register which is more
 * easily accomplished via a mov.
 *
 *  ctxt |- l : c
 *  ---------------------------
 *  ctxt |- Lea r,[l+0] : ctxt[r:c]
 *
 *  ctxt |- rc : Cprod(fs)
 *  ----------------------
 *  ctxt |- Lea r,[rc+0] : ctxt[r:Cprod(fs)]
 *
 *  ctxt |- rc : Stackptr(c1::...::cn::c)
 *  Stackptr(c1::...::cn::c) tailof ctxt.gamma(Esp)  <-- not really needed
 *  -----------------------------------------------
 *  ctxt |- Lea r,rc[i] : ctxt[r:Stackptr(c)]
 *)

let verify_Lea ctxt r g =
  match g with
    Immed _ -> inst_form ctxt "Lea: immediate"; raise Talfail
  | Reg _ -> inst_form ctxt "Lea: register"; raise Talfail
  | Tag _ -> inst_form ctxt "Lea: tag"; raise Talfail
  | Addr _ -> inst_form ctxt "Lea: label"; raise Talfail
  | Prjr (rc,i) ->
      let rc_con = coerce_reg_con ctxt rc in
      (match (whnorm ctxt rc_con).rcon with
	Capp (spc,c) ->
          eqcon ctxt spc sptr_con;
	  let f = valid_stack_con ctxt c in
	  add_reg ctxt r
	    (whcon(Capp (sptr_con,get_stack_tail ctxt i c)))
      |	Cprod _ ->
	  if i<>0 then inst_form ctxt "Lea: middle tuple pointer";
	  add_reg ctxt r rc_con
      | _ -> inst_form ctxt "Lea: non stack-pointer type"; raise Talfail)
  | Prjl (lc,0) -> add_reg ctxt r (coerce_label_con ctxt lc)
  | Prjl (_,_) -> inst_form ctxt "Lea: middle tuple pointer"; raise Talfail
;;
	      
(* sigh (Dave interjects: Buck up Greg. You will get through this). 
 * There are many possible situations:
 *  
 * The simplest scenerio:  we move a value into a register:
 * N.B. moving junk into a register does not hurt
 *  ctxt |- c : K4byte
 *  ctxt |- cgop : c   
 *  ------------------------------
 *  ctxt |- mov r,cgop : ctxt[r:c]
 *
 * The next simplest scenerio:  the target is already initialized but
 * has read/write capability, in which case the context does not change:
 *   ctxt |- gop : c   <--- implies gop is initialized
 *   ctxt |- gop writeable
 *   ctxt |- cgop : c
 *   ctxt |- gop,cgop valid_binops
 *   -----------------------------
 *   ctxt |- mov gop,cgop : ctxt
 *
 * The next simplest scenerio:  the target is the offset of a label.
 * The context remains unchanged, as label types are invariant.  (We
 * may want to relax this constraint but it would require putting label
 * type assumptions on code as well as type variable kinds and register
 * types.)
 *   ctxt |- l : <(c1,cap1,init1),...,(cn,capn,initn)>
 *   ctxt |- sizeof(c1) + ... + sizeof(cj-1) = i
 *   ctxt |- l[i] writeable
 *   ctxt |- cgop : cj
 *   ctxt |- gop,cgop valid_binops
 *   -----------------------------------------------
 *   ctxt |- mov [l+i],cgop : ctxt
 *
 * Finally, the most complicated scenerios: the target is the offset of
 * an (uncoerced) register in which case we update the type of the register
 * to reflect the fact that this field is initialized.  For stack pointers,
 * we must also update the type of the stack pointer.
 *   ctxt |- r : <(c1,cap1,init1),...,(cn,capn,initn)>
 *   ctxt |- sizeof(c1) + ... + sizeof(cj-1) = i
 *   ctxt |- r[i] writeable
 *   ctxt |- cgop : cj
 *   ctxt |- gop,cgop valid_binops
 *   --------------------------------------------------------------------------
 *   ctxt |- mov [r+i],cgop :
 *              ctxt[r:<(c1,cap1,init1),...,(cj,capj,Init),...(cn,capn,initn)>]
 *
 *   ctxt |- r : Stackptr([c1,...,cj] @ ctail)
 *   ctxt |- Esp:Stackptr(chead@[c1,...,cj]@ctail)
 *   ctxt |- sizeof(c1) + ... + sizeof(cj-1) = i  
 *   ctxt |- sizeof(carg) = sizeof(cj) 
 *   ctxt |- r[i] writeable(carg)
 *   ctxt |- cgop : carg
 *   ctxt |- gop,cgop valid_binops
 *   ----------------------------------------------------------------------
 *   ctxt |- mov [r+i],cgop : ctxt[r:Stackptr([c1,...,carg]@ctail), 
 *                                 Esp:Stackptr(chead@[c1,...,carg]@ctail)]
 *
 * (Neal) These rules boil down to three cases:
 *   1) move to register - first rule above
 *   2) move to projection of uncoerced register - last two rules
 *   3) move to other projections - essentially the middle rule
 *)

let verify_Mov ctxt gop cgop =
  valid_cbinops ctxt gop cgop;
  let c_source = coerce_genop_con ctxt cgop in
  match gop with
  (* This is the easy case:  just update the context with r
   * now having the type c_source
   *)
    Reg r ->
      kindleq ctxt (con_kind ctxt c_source) K4byte;
      add_reg ctxt r c_source
  (* This is the hard case.  We must update the type of r: for tuples this
   * sets the init flag to Init and checking that the new type matches the old
   * type, for stacks this requires replacing the previous type with the
   * new type.
   *)
  | Prjr ((r,[]),i) ->
      (match (whnorm ctxt (get_reg_con ctxt r)).rcon,r with
	Cprod fs,_ ->
	  let f = get_field_offset ctxt i fs in 
	  writeable_field ctxt f c_source;
	  add_reg ctxt r (whcon(Cprod (init_field_offset ctxt i fs)))
      | Capp (csp,c_stptr),_ ->
          eqcon ctxt csp sptr_con;
	  let f = valid_stack_con ctxt c_stptr in
	  (* the following checks writeability, gets new type *)
	  let c_stptr' = init_stack_offset ctxt i c_stptr c_source in
	  let newesp = f c_stptr' in
	  let ctxt = add_reg ctxt r (whcon(Capp (sptr_con,c_stptr'))) in
	  let ctxt = add_reg ctxt Esp (whcon(Capp (sptr_con,newesp))) in
	  ctxt
      | _ ->
	  inst_form ctxt "mov: dest register not a product or stack pointer";
	  raise Talfail)
  (* (Neal) I've simplified the following two cases down to the following.
   * The target must have a valid type, be either a field or stack slot, and
   * must accept values of type c_source; however the target may be
   * uninitialised and we lose the fact that the target is initialised by the
   * instruction.  Thus, writeable_genop will check all the conditions and the
   * context remains the same.
   *)
  | Prjr (rc,i) ->
      writeable_genop ctxt gop c_source; ctxt
  | Prjl (lc,i) -> 
      writeable_genop ctxt gop c_source; ctxt
  | _ ->
      inst_form ctxt "mov: destination not register, tuple, or stack";
      raise Talfail
;;

(* Pop
 *
 * Expand to: Mov genop,[esp+0]; Add Esp,4
 *)

let verify_Pop ctxt genop =
  let ctxt = verify_Mov ctxt genop (Prjr ((Esp,[]),0),[]) in
  let ctxt = verify_ArithBin ctxt Add (Reg Esp) (Immed 4) in
  ctxt
;;

(* Popad pops all of the registers.  However, the value for the
 * Esp register is discarded.
 *
 * ctxt |- Esp : sptr(c0::c1::c2::c3::c4::c5::c6::c7::c)
 * ctxt |- ci : K4byte 0<=i<=7
 * ---------------------------------------------------------
 * ctxt |- Popad : ctxt[Eax:c7, Ebx:c4, Ecx:c6, Edx:c5,
 *                      Esp:sptr(c), Ebp:c2, Esi:c1, Edi:c0]
 *)

let verify_Popad ctxt =
  let strip_check ctxt r stkcon =
    match (whnorm ctxt stkcon).rcon with
      Ccons (c1,c2) ->
	kindleq ctxt (con_kind ctxt c1) K4byte; (add_reg ctxt r c1,c2)
    | _ -> inst_form ctxt "Popad: stack not a cons"; raise Talfail in
  let stkcon = current_stack_con ctxt in
  let (ctxt,stkcon) = strip_check ctxt Edi stkcon in
  let (ctxt,stkcon) = strip_check ctxt Esi stkcon in
  let (ctxt,stkcon) = strip_check ctxt Ebp stkcon in
  let (ctxt,stkcon) = strip_check ctxt Ebx stkcon in
  let (ctxt,stkcon) = strip_check ctxt Ebx stkcon in
  let (ctxt,stkcon) = strip_check ctxt Edx stkcon in
  let (ctxt,stkcon) = strip_check ctxt Ecx stkcon in
  let (ctxt,stkcon) = strip_check ctxt Eax stkcon in
  add_reg ctxt Esp (whcon(Capp (sptr_con,stkcon)))
;;

(* Popfd pops a 4-byte value into the flags register.  Hence, the
 * stack pointer must point to a byte4 before hand, and afterwards,
 * points to the tail. 
 *   ctxt |- Esp : sptr(byte4::c)
 *   -----------------------------------------
 *   ctxt |- Popfd : ctxt[Esp:Stackptr(c)]
 *)

let verify_Popfd ctxt =
  match (whnorm ctxt (current_stack_con ctxt)).rcon with
    Ccons (c,c') ->
      eqcon ctxt byte4 c;
      add_reg ctxt Esp (whcon(Capp (sptr_con,c')))
  | _ -> inst_form ctxt "Popfd: Esp has no top slot"; raise Talfail
;;

(* ctxt |- Esp : sptr(c')
 * ctxt |- gc : c
 * ctxt |- c : K4byte
 * ----------------------------------------
 * ctxt |- pushd gc : ctxt[Esp:sptr(c::c')]
 *)

(* FLOAT SUPPORT NEEDED *)
let verify_Push ctxt gc =
  let c = coerce_genop_con ctxt gc in
  kindleq ctxt (con_kind ctxt c) K4byte;
  let cstack = current_stack_con ctxt in
  let cstack' = whcon(Ccons(c,cstack)) in
  add_reg ctxt Esp (whcon(Capp (sptr_con,cstack')))
;;

(* ctxt |- Eax,Ebx,Ecx,Edx,Esi,Edi,Ebp : c7,c4,c6,c5,c1,c0,c2
 * ctxt |- Esp : sptr(c)
 * -----------------------------------------------------------------------
 * ctxt |- pushad : ctxt[Esp:sptr(c0::c1::c2::sptr(c)::c4::c5::c6::c7::c)]
 *
 * N.B. registers have to have K4byte-s in them.
 *)

let verify_Pushad ctxt = 
  let c3 = (whnorm ctxt (get_reg_con ctxt Esp)) in
  match c3.rcon with
    Capp (csp,cs) ->
      eqcon ctxt csp sptr_con;
      let c0 = get_reg_con ctxt Edi
      and c1 = get_reg_con ctxt Esi
      and c2 = get_reg_con ctxt Ebp
      and c4 = get_reg_con ctxt Ebx
      and c5 = get_reg_con ctxt Edx
      and c6 = get_reg_con ctxt Ecx
      and c7 = get_reg_con ctxt Eax in
      let cs = List.fold_right (fun (c:con) (cs:con) -> whcon(Ccons(c,cs))) 
              [c0;c1;c2;c3;c4;c5;c6;c7] cs in
      add_reg ctxt Esp (whcon(Capp (sptr_con,cs)))
  | _ -> inst_form ctxt "Pushad: Esp not a stack pointer"; raise Talfail
;;

(* ctxt |- Esp : sptr(c)
 * -----------------------------------------
 * ctxt |- Pushfd : ctxt[Esp:sptr(byte4::c)]
 *)

let verify_Pushfd ctxt =
  match (whnorm ctxt (get_reg_con ctxt Esp)).rcon with
    Capp(csp,cs) -> 
      eqcon ctxt csp sptr_con;
      add_reg ctxt Esp
        (whcon(Capp (sptr_con,whcon(Ccons (byte4,cs)))))
  | _ -> inst_form ctxt "Pushfd: Esp not a stack pointer"; raise Talfail
;;

(* Retn: pops the return address off the stack and then pops an additional
 * (optional) i bytes off the stack, then jumps to the return address.
 *
 *  ctxt |- Esp : Ccode(G)::c1::c2::...::cn::ctail'
 *  ctxt |- sizeof(c1) + ... + sizeof(cn) = i
 *  ctxt |- ctxt.gamma[Esp:ctail'] <= G
 *  -----------------------------------------------
 *  ctxt |- Retn (Some i) : ctxt'
 *)

let verify_Retn ctxt iopt =
  (match (whnorm ctxt (get_reg_con ctxt Esp)).rcon with
    Capp (csp, c) ->
      (match (whnorm ctxt csp).rcon,(whnorm ctxt c).rcon with
         (Cprim PCstackptr,Ccons(cg,ctail)) ->
           (match (whnorm ctxt cg).rcon with
              Ccode g2 -> 
                let ctail' = 
                   match iopt with 
                     None -> ctail 
                   | Some i -> get_stack_tail ctxt i ctail in
                let newctxt = add_reg ctxt Esp 
                   (whcon(Capp (sptr_con,ctail'))) in
                reg_state_leq ctxt (get_register_state newctxt) g2
            | _ -> inst_form ctxt "Retn: no return address on stack")
       | _ -> inst_form ctxt "Retn: bad stack pointer")
  | _ -> inst_form ctxt "Retn: bad stack pointer (no app)");
  raise Terminal_Jmp
;;

(*   ctxt |- gop writeable   ctxt |- gop : [S(0)+S(1)]
 *   -------------------------------------------------
 *   ctxt |- Setcc gop : ctxt
 *)

let verify_Setcc ctxt cc gop =
  writeable_genop ctxt gop bool_con;
  eqcon ctxt bool_con (genop_con ctxt gop);
  ctxt
;;

(* shifts: gop must be writeable and a byte4, r must be a byte4
 *         Ecx must be byte4 if it is the shift amount
 *
 * ctxt |- gop : byte4  ctxt |- gop writeable
 * ctxt |- Ecx : byte4    ctxt |- r : byte4
 * ----------------------------------------------
 * ctxt |- Shld/Shrd gop,r,None : ctxt
 *
 * ctxt |- gop : byte4  ctxt |- gop writeable
 * ctxt |- r : byte4
 * ----------------------------------------------
 * ctxt |- Shld/Shrd gop,r,(Some i) : ctxt
 *)

let verify_Shld_Shrd ctxt gop r iopt =
  writeable_genop ctxt gop byte4;
  eqcon ctxt byte4 (genop_con ctxt gop);
  eqcon ctxt byte4 (get_reg_con ctxt r);
  (match iopt with
    Some _ -> ()
  | None -> eqcon ctxt byte4 (get_reg_con ctxt Ecx));
  ctxt
;;

(* ctxt |- (gop1,gop2) valid_binops
 * ctxt |- gop1 : byte4  ctxt |- gop2 : byte4
 * ------------------------------------------
 * ctxt |- test gop1,gop2 : ctxt
 *)

let verify_Test ctxt gop1 gop2 =
  valid_binops ctxt gop1 gop2;
  eqcon ctxt byte4 (genop_con ctxt gop1);
  eqcon ctxt byte4 (genop_con ctxt gop2);
  ctxt
;;

(* exchange gop1 and gop2.
 *   ctxt |- (gop1,gop2) valid binops  
 *   ctxt |- gop1 writeable  ctxt |- gop2 writeable
 *   ctxt |- gop1 : c   ctxt |- gop2 : c
 *   ----------------------------------------------
 *   ctxt |- Xchg gop1,gop2 : ctxt
 *)

let verify_Xchg ctxt gop1 r =
  let c1 = genop_con ctxt gop1
  and c2 = get_reg_con ctxt r in
  writeable_genop ctxt gop1 c2;
  eqcon ctxt c1 c2;
  ctxt
;;

(*** TAL specific instructions ***)

(* array/vector length: always returns a byte4
 *  ctxt |- gop : array(c)/vector(c)
 *  ------------------------------------
 *  ctxt |- Alen r1,gop : ctxt[r1:byte4]
 *
 *  ctxt |- gop : bytearray/vector 
 *  ------------------------------------
 *  ctxt |- Alen r1,gop : ctxt[r1:byte4]
 *)

let verify_Alen ctxt s r1 gop =
  (match (whnorm ctxt (genop_con ctxt gop)).rcon with
    Cprim (PCbytearray s1) ->
      if s1 <> s then inst_form ctxt "Alen: bytearray scales do not match"
  | Cprim (PCbytevector s1) ->
      if s1 <> s then inst_form ctxt "Alen: bytevector scales do not match"
  | Capp (cav,c1) ->
      (match (whnorm ctxt cav).rcon with
	Cprim PCarray ->
	  if s <> Byte4 then inst_form ctxt "Alen: array scale not byte 4"
      | Cprim PCvector ->
	  if s <> Byte4 then inst_form ctxt "Alen: vector scale not byte 4"
      |	_ -> inst_form ctxt "Alen: constructor not array/vector")
  | _ -> inst_form ctxt "Alen: bad argument");
  add_reg ctxt r1 byte4
;;

(* array/vector subscript.  float arrays?
 *  ctxt |- gop : array(c)  ctxt |- r2 : byte4
 *  ------------------------------------------
 *  ctxt |- Asub r1,gop,r2 : ctxt[r1:c]
 *
 *  ctxt |- gop : bytearray  ctxt |- r2 : byte4
 *  -------------------------------------------
 *  ctxt |- Asub r1,gop,r2 : ctxt[r1:byte4]
 *)

let verify_Asub ctxt s r1 gop r2 =
  (eqcon ctxt byte4 (get_reg_con ctxt r2);
   match (whnorm ctxt (genop_con ctxt gop)).rcon with
     Capp (cav,c1) ->
       (match (whnorm ctxt cav).rcon with
	 Cprim PCarray ->
	   if s <> Byte4 then inst_form ctxt "Asub: array scale not byte 4";
	   add_reg ctxt r1 c1
       | Cprim PCvector ->
	   if s <> Byte4 then inst_form ctxt "Asub: vector scale not byte 4";
	   add_reg ctxt r1 c1
       | _ -> inst_form ctxt "Asum: constructor not array/vector"; raise Talfail)
   | Cprim (PCbytearray s1) ->
       if s1 <> s then inst_form ctxt "Asub: bytearray scales do not match";
       if s1 = Byte8 then inst_form ctxt "Asub: loading 8 bytes into reg";
       add_reg ctxt r1 byte4
   | Cprim (PCbytevector s1) ->
       if s1 <> s then inst_form ctxt "Asub: bytevector scales do not match";
       if s1 = Byte8 then inst_form ctxt "Asub: loading 8 bytes into reg";
       add_reg ctxt r1 byte4
   | _ -> inst_form ctxt "Asub: operand not an array/vector"; raise Talfail)
;;

(* array subscript.  float arrays? 
 *  ctxt |- gop : array(c)  ctxt |- r1 : byte4  ctxt |- r2 : c
 *  ----------------------------------------------------------
 *  ctxt |- Aupd gop,r1,r2 : ctxt
 *
 *  ctxt |- gop : bytearray  ctxt |- r1 : byte4  ctxt |- r2 : byte4
 *  ---------------------------------------------------------------
 *  ctxt |- Aupd gop,r1,r2 : ctxt
 *)

let verify_Aupd ctxt s gop r1 r2 =
  (eqcon ctxt byte4 (get_reg_con ctxt r1);
   match (whnorm ctxt (genop_con ctxt gop)).rcon with
     Capp(ca,c) ->
       eqcon ctxt ca array_con;
       if s <> Byte4 then inst_form ctxt "Aupd: array scale not byte 4";
       eqcon ctxt c (get_reg_con ctxt r2);
       ctxt
   | Cprim(PCbytearray s1) -> 
       if s1 <> s then inst_form ctxt "Aupd: bytearray scales do not match";
       if s1 = Byte8 then inst_form ctxt "Aupd: storing 8 bytes from reg";
       eqcon ctxt byte4 (get_reg_con ctxt r2);
       ctxt
   | _ -> inst_form ctxt "Aupd: operand not an array"; ctxt)
;;

(* ctxt |- r : exn           ctxt |- gop : exnname(c) 
 * ctxt |- lc : code(G)      ctxt |- ctxt.gamma <= G
 * --------------------------------------------------------------------------
 * ctxt |- Bexn r,gop,lc : ctxt[r:Prod<(exnname(c),Read,Init),(c,Read,Init)>]
 *
 * Operationally, Bexn extracts the exception name from an exception
 * packet in r, compares it to the exception name in gop, and if they're
 * _not_ equal, branches to the (possibly coerced) label lc.  If they
 * are equal, then we fall through and refine the type of r so that
 * it's a pair of an exnname(c) * c value
 *)

let verify_Bexn ctxt r gop lc =
  eqcon ctxt exn_con (get_reg_con ctxt r);
  let cexnname = (whnorm ctxt (genop_con ctxt gop)) in
  match cexnname.rcon,
        (whnorm ctxt (coerce_label_con ctxt lc)).rcon with
    Capp (cexnn,c),Ccode gamma' ->
      eqcon ctxt cexnn exnname_con;
      reg_state_leq ctxt (get_register_state ctxt) gamma';
      add_reg ctxt r (whcon(Cprod [(cexnname,Read,Init);(c,Read,Init)]))
  | _ ->
      inst_form ctxt "Bexn: either r not an exn or gop not an exnname";
      raise Talfail
;;

(* The immediate i is in the range [0..255].  r is either an integer in
 * the range [0..255] or else a pointer to a tuple and has type
 * Sum{sum_tags[i1,...,in],vt=svt} in the current context.  If r is
 * a pointer, then we assume i < r The "instruction"
 * explodes into the following code:
 *    
 *     cmp r,i
 *     jcc lc
 *
 * Hence, if
 *
 * cc is:      lc assigns r:                    fall through assigns r:
 * -------- ------------------------------    ---------------------------
 * Above    Sum{sum_tags[j1,...,jm],vt=svt}   Sum{sum_tags[k1,...,kp],vt=None}
 *    where each jl,kl in [i1,...,in] ^ jl > i and kl <= i
 * AboveEq  Sum{sum_tags[j1,...,jm],vt=svt}   Sum{sum_tags[k1,...,kp],vt=None}
 *    where each jl,kl in [i1,...,in] ^ jl >= i and kl < i
 * Below    Sum{sum_tags[j1,...,jm],vt=None}   Sum{sum_tags[k1,...,kp],vt=svt}
 *    where each jl,kl in [i1,...,in] ^ jl < i and kl >= i
 * BelowEq  Sum{sum_tags[j1,...,jm],vt=None}   Sum{sum_tags[k1,...,kp],vt=svt}
 *    where each jl,kl in [i1,...,in] ^ jl <= i and kl > i
 *
 * Notice that if r is actually a pointer, then for instance, doing
 *   cmp r,i
 *   jab lc
 * will always branch to the (coerced) label lc.
 *)

let interpret_condition ctxt cc =
  match cc with
    Above -> (>)
  | AboveEq -> (>=)
  | Below -> (<)
  | BelowEq -> (<=)
  | Eq -> (=)
  | NotEq -> (!=)
  | _ ->
      inst_form ctxt "tag comparison: condition code is invalid"; raise Talfail
;;

let split_condition_tag ctxt cc i st svt =
  let f = interpret_condition ctxt cc in
  let condition x = f x i in
  let rec split l matches not_matches =
    match l with
      [] -> (List.rev matches,List.rev not_matches)
    | h::t ->
 	if condition h then split t (h::matches) not_matches
    	else split t matches (h::not_matches) in
  let (tag_matches,tag_not_matches) = split st [] [] in
  let (svt_matches,svt_not_matches) = 
    if condition 256 then (svt,None) else (None,svt) in
  (whcon(Csum{sum_tags=tag_matches;sum_vt=svt_matches}),
   whcon(Csum{sum_tags=tag_not_matches;sum_vt=svt_not_matches}))
;;

let verify_Btagi ctxt r i lc cc =
  if i < 0 or i > 255 then inst_form ctxt "Btagi: not(0 <= i <= 255)";
  match (whnorm ctxt (get_reg_con ctxt r)).rcon with
    Csum {sum_tags=st;sum_vt=svt} ->
      let (sum_branch,sum_fallthru) = split_condition_tag ctxt cc i st svt in
      (match (whnorm ctxt (coerce_label_con ctxt lc)).rcon with
	Ccode gamma' ->
	  reg_state_leq ctxt
	    (get_register_state (add_reg ctxt r sum_branch)) gamma'
      | _ -> inst_form ctxt "Btagi: label not code");
      add_reg ctxt r sum_fallthru
  | _ -> inst_form ctxt "Btagi: register doesn't contain sum"; raise Talfail
;;

(* The immediate i is in the range [0..255].  r is a pointer to a tuple
 * where the first component is an integer in the range [0..255] and has type
 * Sum{sum_tags,vt=Some(Variant[(i1,fs1),...,(in,fsn)])} in the current 
 * context.  The instruction explodes into:
 *    
 *     cmp r[0],i
 *     jcc lc
 *
 * cc is:      lc assigns r:                     fall through assigns r:
 * -------- -----------------------------     --------------------------------
 * Above    Sum{[(j1,fsj1),...,(jm,fsjm)]}    Sum{vt=[(k1,fsk1),...(kp,fskp)]}
 *    where each jl,kl in [i1,...,in] ^ jl > i and kl <= i
 * AboveEq  Sum{[(j1,fsj1),...,(jm,fsjm)]}    Sum{vt=[(k1,fsk1),...(kp,fskp)]}
 *    where each jl,kl in [i1,...,in] ^ jl >= i and kl < i
 * Below    Sum{[(j1,fsj1),...,(jm,fsjm)]}    Sum{vt=[(k1,fsk1),...(kp,fskp)]}
 *    where each jl,kl in [i1,...,in] ^ jl < i and kl >= i
 * BelowEq  Sum{[(j1,fsj1),...,(jm,fsjm)]}    Sum{vt=[(k1,fsk1),...(kp,fskp)]}
 *    where each jl,kl in [i1,...,in] ^ jl <= i and kl > i
 *
 * where we've dropped off the tag, Some, Variant etc. to fit this on the
 * screen 
 *)

let split_condition_var ctxt cc i st =
  let f = interpret_condition ctxt cc in
  let condition x = f x i in
  let rec split l matches not_matches =
    match l with
      [] -> (List.rev matches,List.rev not_matches)
    | h::t ->
 	if condition (fst h) then split t (h::matches) not_matches
    	else split t matches (h::not_matches) in
  let (svt_matches,svt_not_matches) = split st [] [] in
  (whcon(Csum{sum_tags=[];sum_vt=Some(Variants svt_matches)}),
   whcon(Csum{sum_tags=[];sum_vt=Some(Variants svt_not_matches)}))
;;

let verify_Btagvar ctxt r i lc cc =
  if i < 0 or i > 255 then inst_form ctxt "Btagvar: not(0 <= i <= 255)";
  match (whnorm ctxt (get_reg_con ctxt r)).rcon with
    Csum{sum_tags=[];sum_vt=Some(Variants vs)} ->
      let (sum_branch,sum_fallthru) = split_condition_var ctxt cc i vs in
      (match (whnorm ctxt (coerce_label_con ctxt lc)).rcon with
	Ccode gamma' ->
	  reg_state_leq ctxt
	    (get_register_state (add_reg ctxt r sum_branch)) gamma'
      | _ -> inst_form ctxt "Btagi: label not code");
      add_reg ctxt r sum_fallthru
  | Csum{sum_tags=_::_;sum_vt=_} ->
      inst_form ctxt "Btagvar: on tags"; raise Talfail
  | Csum{sum_tags=_;sum_vt=Some(Tuple _)} ->
      inst_form ctxt "Btagvar: on tuple"; raise Talfail
  | _ ->
      inst_form ctxt "Btagvar: register doesn't contain sum"; raise Talfail
;;

(* Coerce: coerce register
 * N.B. the second condition may be redundant
 *
 * ctxt |- coerce(r) : c    ctxt |- c : K4byte
 * -------------------------------------------
 * ctxt |- Coerce coerce(r) : ctxt[r:c]
 *)

let verify_Coerce ctxt (r,_ as arg) =
  let c = coerce_reg_con ctxt arg in
  kindleq ctxt (con_kind ctxt c) K4byte;
  add_reg ctxt r c
;;

(* We can malloc the following things:
 *  tuples      :  fields in the tuple must be uninitialized and 
 *                 we must be able to determine the field sizes
 *  exnname's   :
 *  bytearray's :
 *)

let scale_to_int s =
   match s with
     Byte1 -> 1
   | Byte2 -> 2
   | Byte4 -> 4
   | Byte8 -> 8

let verify_Malloc ctxt r i ma =
  if i < 0 then inst_form ctxt "malloc of negative value";
  match ma with
    Mprod fs ->
      let check_field f (sum,fields) =
	let (k,c) = check ctxt (fst f) in
	kindleq ctxt k Ktype;
	let sum' = sum + (sizeof ctxt c) in
        (sum',(c,snd f,Uninit)::fields) in
      let (sum,fields) = List.fold_right check_field fs (0,[]) in
      if i <> sum then inst_form ctxt "malloc product: byte count is off";
      add_reg ctxt r (whcon(Cprod fields))
  | Mbytearray scale -> 
      if i mod (scale_to_int scale) <> 0 then
        inst_form ctxt "malloc bytearray: byte count not multiple of scale";
      add_reg ctxt r (prcon(Cprim(PCbytearray scale)))
  | Mexnname c ->
      let (k,c) = check ctxt c in
      kindleq ctxt k K4byte;
      if i <> 4 then inst_form ctxt "malloc exnname: byte count not 4";
      add_reg ctxt r (whcon(Capp (exnname_con,c)))
;;

(* verify gc is an exists, and introduce the new type variable a into
 * scope, as well as moving the value under the existential into the
 * register r.  Also, it's necessary to alpha-convert a so that it
 * is unique and does not conflict with something already in the
 * context...
 *)

let verify_Unpack ctxt a r gc =
  match (whnorm ctxt (coerce_genop_con ctxt gc)).rcon with
    Cexist(a',k,c') ->
      let a_con = whcon(Cvar a) in
      let c = subst a_con a' c' in
      let ctxt = add_var ctxt a k in
      let ctxt = add_reg ctxt r c in
      ctxt
  | _ -> inst_form ctxt "Unpack: genop not an Exists"; raise Talfail
;;

(*** Verify an instruction ***)

let verify_instr ctxt i =
  match i with
  (* Generic x86 instructions *)
    ArithBin(ab,g1,g2) -> verify_ArithBin ctxt ab g1 g2
  | ArithUn(au,g) -> verify_ArithUn ctxt au g
  | ArithMD(amd,g) -> verify_ArithMD ctxt amd g
  | ArithSR(sr,g,iopt) -> verify_ArithSR ctxt sr g iopt
  | Bswap(r) -> verify_Bswap ctxt r
  | Call(gc) -> verify_Call ctxt gc
  | Clc -> ctxt
  | Cmc -> ctxt
  | Cmovcc(c,r,gopt) -> verify_Cmovcc ctxt c r gopt
  | Cmp(g1,g2) -> verify_Cmp ctxt g1 g2
  | Conv(c) -> verify_Conv ctxt c
  | Imul3(r,g,i) -> verify_Imul3 ctxt r g i
  | Int i -> inst_form ctxt "interrupt unimplemented"; raise Talfail
  | Into -> inst_form ctxt "interrupt on overflow unimplemented"; raise Talfail
  | Jcc (c,lc) -> verify_Jcc ctxt lc
  | Jecxz lc -> verify_Jecxz ctxt lc
  | Jmp gc -> verify_Jmp ctxt gc
  | Lahf -> add_reg ctxt Eax byte4
  | Lea (r,g) -> verify_Lea ctxt r g
  | Loopd (lc,bo) -> verify_Jecxz ctxt lc (* For typing loop is jecxz *)
  | Mov (gop,gcop) -> verify_Mov ctxt gop gcop
  | Nop -> ctxt
  | Pop genop -> verify_Pop ctxt genop
  | Popad -> verify_Popad ctxt
  | Popfd -> verify_Popfd ctxt
  | Push gc -> verify_Push ctxt gc
  | Pushad -> verify_Pushad ctxt
  | Pushfd -> verify_Pushfd ctxt
  | Retn iopt -> verify_Retn ctxt iopt
  | Sahf -> eqcon ctxt byte4 (get_reg_con ctxt Eax); ctxt
  | Setcc (c,g) -> verify_Setcc ctxt c g
  | Shld (gop,r,iopt) -> verify_Shld_Shrd ctxt gop r iopt
  | Shrd (gop,r,iopt) -> verify_Shld_Shrd ctxt gop r iopt
  | Stc -> ctxt
  | Test (gop1,gop2) -> verify_Test ctxt gop1 gop2
  | Xchg (gop,r) -> verify_Xchg ctxt gop r
  (* TAL specific instructions *)
  | Alen (s,r1,gop) -> verify_Alen ctxt s r1 gop
  | Asub (s,r1,gop,r2) -> verify_Asub ctxt s r1 gop r2
  | Aupd (s,gop,r1,r2) -> verify_Aupd ctxt s gop r1 r2
  | Bexn (r,gop,lc) -> verify_Bexn ctxt r gop lc
  | Btagi (rc,i,lc,cc) -> verify_Btagi ctxt rc i lc cc
  | Btagvar (rc,i,lc,cc) -> verify_Btagvar ctxt rc i lc cc
  | Coerce rc -> verify_Coerce ctxt rc
  | Comment _ -> ctxt
  | Fallthru cs -> raise (Fall_Thru (ctxt,cs))
  | Malloc (r,i,ma) -> verify_Malloc ctxt r i ma
  | Unpack (a,r,gc) -> verify_Unpack ctxt a r gc
;;

(**********************************************************************)    
(* Code Blocks                                                        *)
(**********************************************************************)

(* Add the code labels to psi:
 * Check and normalise the constructor and check it has kind K4byte
 *)
let add_code_labels ctxt cbv =
  let aux ctxt (l,c,_) =
    let ctxt =
      set_verify_ctxt (set_loc ctxt (Locc (l,-1))) "adding code labels" in
    let (k,c') = check ctxt c in
    kindleq ctxt k K4byte; add_val ctxt l c' in
  vector_fold aux ctxt cbv
;;

(* Verifying a code block:
 *   1) build context.
 *     i) seperate the labels type in a (variable,kind) list and register state
 *     ii) add each variable to context left to right
 *     iii) check and normalise register state in this context
 *     iv) make checked and normalised register state the register state
 *   2) check instructions.
 *     i) verify left to right chaining context through
 *     ii) if get to end have a Fallsthru error
 *     iii) otherwise one of two exceptions is raised:
 *       Terminal_Jmp: return None to indicate no fall thru
 *       Fall_Thru d: return Some d to indicate fall thru
 *)

let verify_code_block ctxt (l,c,insts) =
  (* Build ctxt *)
  let ctxt = set_loc ctxt (Locc (l,-1)) in
  let (vks,rs) = separate_fun_type ctxt c in
  let f ctxt (v,k) = add_var ctxt v k in
  let ctxt = List.fold_left f ctxt vks in
  let gamma = verify_gamma ctxt rs in
  let ctxt = set_register_state ctxt gamma in
  (* Verify instructions *)
  let ctxt = set_verify_ctxt ctxt "verifying instructions" in
  let rec loop ctxt n i =
    let ctxt = set_loc ctxt (Locc (l,i)) in
    if i=n then
      ()
    else
      let ctxt = verify_instr ctxt insts.(i) in loop ctxt n (i+1) in
  try
    loop ctxt (Array.length insts) 0;
    generate_error ctxt Fallsthru;
    None
  with
    Terminal_Jmp -> None
  | Fall_Thru (ctxt,cs) -> Some (ctxt,cs)
  | Talfail -> None
;;

let verify_code_blocks ctxt cbv =
  let rec loop ft n i =
    if i=n then
      ft
    else
      let (l,_,_ as cb) = cbv.(i) in
      (match ft with
	None -> ()
      |	Some (ctxt,cs) ->
	  let lc =
	    List.fold_left (fun gc c -> coerce gc (Tapp c))
	      (raw (Addr l)) cs in
	  try verify_instr ctxt (Jmp lc); generate_error ctxt Fallsthru with
	    Terminal_Jmp -> ()
	  | Fall_Thru (_,_) -> generate_error ctxt Fallsthru
	  | Talfail -> ());
      let ft = verify_code_block ctxt cb in
      loop ft n (i+1) in
  match loop None (Array.length cbv) 0 with
    None -> ()
  | Some (_,_) -> generate_error ctxt Fallsthru
;;

(**********************************************************************)    
(* Data Blocks                                                        *)
(**********************************************************************)

(* Add the data labels to psi:
 * Check and normalise the constructor and check it has kind K4byte
 *)
let add_data_labels ctxt dbv =
  let aux ctxt (l,co,(dis,clist)) =
    let ctxt =
      set_verify_ctxt (set_loc ctxt (Locc (l,-1))) "adding data labels" in
    let c = 
      match co with
	None -> failwith "Talverify.add_data_labels - unimplemented"
      |	Some c -> c in
    let (k,c') = check ctxt c in
    kindleq ctxt k K4byte; add_val ctxt l c' in
  vector_fold aux ctxt dbv
;;

let verify_data_item ctxt di =
  match di with
    Dlabel cl -> [(coerce_label_con ctxt cl,ReadWrite)]
  | Dtag ci ->
      let tag_con ctxt i = prcon (Cprim (PCtag i)) in
      [(coerce_con tag_con ctxt ci,ReadWrite)]
  | Dbytes s ->
      replicate (prcon (Cprim (PCbytes Byte1)),ReadWrite) (String.length s)
  | D2bytes _ -> [(prcon (Cprim (PCbytes Byte2)),ReadWrite)]
  | D4bytes _ -> [(byte4,ReadWrite)]
  | Djunk -> [(junk_con,ReadWrite)]
  | Dexnname _ ->
      generate_error ctxt (Data_form "Dexnname not allowed"); raise Talfail
;;

let verify_data_block ctxt (l,co,(dis,clist)) = 
  let ctxt = set_loc ctxt (Locd (l,-1)) in
  let c1 =
    match dis,clist with
      [Dexnname c],[] -> 
	let (k,c') = check ctxt c in
    	kindleq ctxt k K4byte;
 	whcon (Capp (exnname_con,c'))
    | _,_ ->
	let ctxt = set_verify_ctxt ctxt "verifying data items" in
	let rec aux n dis =
	  match dis with
	    [] -> []
	  | di::dis ->
	      let fs = aux (n+1) dis in
	      let cs = verify_data_item (set_loc ctxt (Locd (l,n))) di in
	      List.fold_right (fun (c,cap) fs -> (c,cap,Init)::fs) cs fs in
	let fs = aux 0 dis in
	List.fold_right (coercion_con ctxt) clist (whcon(Cprod fs)) in
  match co with
    None -> c1
  | Some c2 -> eqcon ctxt c1 c2; c2
;;

let verify_data_blocks ctxt dbv =
  for i=0 to Array.length dbv - 1 do
    try 
      verify_data_block ctxt dbv.(i); ()
    with Talfail -> ()
  done
;;

(**********************************************************************)    
(* Interfaces/Implementations                                         *)
(**********************************************************************)

type ref2int = int_ref -> tal_int

let intrefs2inttype ref2int ctxt refs includes =
  let current = ref []
  and added = ref []
  and cons = ref []
  and vals = ref [] in
  let rec loop intref =
    if List.mem intref !current then
      generate_error ctxt (Cyclic_imports intref);
    if not (List.mem intref !added) then begin
      added := intref :: !added;
      let oldcur = !current in
      current := intref::oldcur;
      let tal_int = ref2int intref in
      if includes then Array.iter loop tal_int.int_includes;
      Array.iter (fun ic -> cons := ic :: !cons) tal_int.int_cons;
      Array.iter (fun lc -> vals := lc :: !vals) tal_int.int_vals;
      current := oldcur
    end in
  Array.iter loop refs;
  {it_cons=List.rev !cons; it_vals=List.rev !vals}
;;

let add_imported_cons ctxt lkcds =
  let aux ctxt (l,k,_) =
    add_con (set_loc ctxt (Loccon l)) l k in
  List.fold_left aux ctxt lkcds
;;

let verify_int_con_def ctxt (l,k,cd) =
  let ctxt = set_loc ctxt (Loccon l) in
  let cd =
    match cd with
    AbsCon -> cd
  | ConcCon c ->
      let (k',c') = check_whnorm ctxt c in
      kindleq ctxt k' k;
      ConcCon c' in
  ((l,k,cd),add_con_def ctxt l cd)
;;

let add_imported_vals ctxt lcs =
  let aux ctxt (l,c) =
    let ctxt = set_loc ctxt (Locval l) in
    let (k,c) = check_whnorm ctxt c in
    kindleq ctxt k K4byte;
    ((l,c),add_val ctxt l c) in
  foldmap_left aux ctxt lcs
;;

let add_con_blocks ctxt lkcs =
  let aux ctxt (l,k,_) =
    add_con (set_loc ctxt (Loccon l)) l k in
  vector_fold aux ctxt lkcs
;;

let verify_con_block ctxt (l,k,c) =
  let ctxt = set_loc ctxt (Loccb l) in
  let (k',c') = check_whnorm ctxt c in
  kindleq ctxt k' k;
  add_con_def ctxt l (ConcCon c')
;;

let verify_exports ref2int ctxt tal_imp =
  let ext = intrefs2inttype ref2int ctxt tal_imp.exports false in
  (* Check exported constructors *)
  let f (l,k,cd as lkcd) = 
    let ctxt = set_loc ctxt (Loccon l) in
    let g (l',_,_) = l=l' in
    if not (vector_exists g tal_imp.con_blocks) then
      generate_error ctxt Doesnt_export;
    match cd with
      AbsCon ->
 	kindleq ctxt (get_label_kind ctxt l) k; lkcd
    | ConcCon c ->
 	let (k1,c1) = check_whnorm ctxt c in
	kindleq ctxt k1 k;
	(match get_label_def ctxt l with
	  AbsCon ->
	    (* This shouldn't happen as conblocks are added concrete *)
	    failwith "Talverify.verify_exports - internal error"
	| ConcCon c2 ->
	    eqcon ctxt c2 c1);
	(l,k,(ConcCon c1)) in
  let econs = List.map f ext.it_cons in
  (* Check exported labels *)
  let f (l,c) =
    let ctxt = set_loc ctxt (Locval l) in
    let (k1,c1) = check_whnorm ctxt c in
    kindleq ctxt k1 K4byte;
    let g (l',_,_) = l=l' in
    if not (vector_exists g tal_imp.code_blocks)
 	& not (vector_exists g tal_imp.data_blocks) then
      generate_error ctxt Doesnt_export;
    eqcon ctxt (get_label_con ctxt l) c1;
    (l,c1) in
  let evals = List.map f ext.it_vals in
  {it_cons=econs; it_vals=evals}
;;

let verify_imp ref2int ctxt tal_imp =
  (* Get imported interface *)
  let imt = intrefs2inttype ref2int ctxt tal_imp.imports true in
  (* Build phi *)
  let ctxt = add_imported_cons ctxt imt.it_cons in
  let ctxt = add_con_blocks ctxt tal_imp.con_blocks in
  (* Process label constructor definitions *)
  let (icons,ctxt) = foldmap_left verify_int_con_def ctxt imt.it_cons in
  let ctxt = vector_fold verify_con_block ctxt tal_imp.con_blocks in
  (* Build psi *)
  let (ivals,ctxt) = add_imported_vals ctxt imt.it_vals in
  let ctxt = add_code_labels ctxt tal_imp.code_blocks in
  let ctxt = add_data_labels ctxt tal_imp.data_blocks in
  (* Check code blocks *)
  verify_code_blocks ctxt tal_imp.code_blocks;
  (* Check data blocks *)
  verify_data_blocks ctxt tal_imp.data_blocks;
  (* Check exports *)
  let ctxt = set_verify_ctxt ctxt "verifying exports" in
  let ext = verify_exports ref2int ctxt tal_imp in
  ({it_cons=icons; it_vals=ivals},ext)
;;

(**********************************************************************)    
(* EOF: talverify.ml                                                  *)
(**********************************************************************)
