/**********************************************************************
 * (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,                   *
 *     June 1998, all rights reserved.                                *
 **********************************************************************/

%{

(* Grammar taken largely from Java's syntax (see Java spec) *)

open Popsyntax;;

module X = Poperr;;
module L = Poplocus;;

let locus () = (L.create (symbol_start(),symbol_end()));;

let err e = ((Poperrhandle.signal_error (X.Eparse e) (locus ()));
      	            (raise Parsing.Parse_error));;

let parse_error s =
  begin
    Poperrhandle.signal_error (X.Eparse (X.Syntax s)) (locus ());
    flush stdout;
    ()
  end;;

type switch_clause = 
    AnyClause
  | IntClauses of (int * stmt) list
  | CharClauses of (char * stmt) list
  | UnionClauses of (field_name * (var option) * stmt) list;;

let add_int_clause i s (scs,d) = 
  match scs with
    IntClauses cs -> (IntClauses ((i,s)::cs),d)
  | AnyClause -> (IntClauses [(i,s)],d)
  | _ -> err X.SwitchClausesDontMatch
;;

let add_char_clause c s (scs,d) = 
  match scs with
    CharClauses cs -> (CharClauses ((c,s)::cs),d)
  | AnyClause -> (CharClauses [(c,s)],d)
  | _ -> err X.SwitchClausesDontMatch;;

let add_union_clause f v s (scs,d) = 
  match scs with
    UnionClauses cs -> (UnionClauses ((f,v,s)::cs),d)
  | AnyClause -> (UnionClauses [(f,v,s)],d)
  | _ -> err X.SwitchClausesDontMatch;;

let make_exp re = { exp_typ = None; raw_exp = re; exp_loc = locus() };;
  
let rec make_type base_type num_brackets = 
  if num_brackets = 0 then base_type
  else ArrayType(make_type base_type (num_brackets - 1));;
%}

%token <string> ID
%token <int> CONSTINT
%token <bool> CONSTBOOLEAN
%token <string> CONSTSTRING
%token <char> CONSTCHAR
%token EOF

%token LPAREN RPAREN LBRACE RBRACE LBRACKET RBRACKET LESSTHAN
%token GREATERTHAN LESSTHANEQ GREATERTHANEQ PLUSPLUS MINUSMINUS
%token PLUSEQUAL MINUSEQUAL TIMESEQUAL DIVEQUAL AMPERAMPER
%token PIPEPIPE EQUALS EE NE PLUS MINUS TIMES DIV BANG QUESTION
%token COLON SEMICOLON DOT COMMA AMPER PIPE LESSLESS GREATERGREATER PERCENT
%token ARROW 

%token BOOLEAN BREAK CASE CHAR CHR CONST CONTINUE DEFAULT ELSE EXTERN FOR IF 
%token INT NEW NEWARRAY NULL ORD RETURN SIZE STATIC 
%token STRING STRUCT SWITCH UNION VOID WHILE

%right EQUALS
%left PLUSEQUAL MINUSEQUAL TIMESEQUAL DIVEQUAL
%left AMPERAMPER, PIPEPIPE
%left EE, NE, LESSTHAN, GREATERTHAN, LESSTHANEQ, GREATERTHANEQ
%left PLUS MINUS
%left TIMES DIV
%left BANG

%type <Popsyntax.top_decl list> top

%start top

%%

top:
    top_decls EOF           { List.rev($1) }
  ;

top_decls:
    top_decl                { [$1] }
  | top_decl top_decls   { $1 :: $2 }
  ;

top_decl:
    func_decl               { $1 }
  | struct_decl             { $1 }
  | union_decl              { $1 }
  | extern                  { $1 }
  ;

func_decl:
    static_opt typ ID LPAREN param_list RPAREN block
                        { (FunDecl { fn_static = $1;
				     fn_name = $3;
				     fn_ret_type = $2;
				     fn_args = $5;
				     fn_body = $7;
				     fn_returns = false 
				   }, locus()) }
  | static_opt typ ID LPAREN RPAREN block
                        { (FunDecl { fn_static = $1;
				     fn_name = $3;
				     fn_ret_type = $2;
				     fn_args = [];
				     fn_body = $6;
				     fn_returns = false 
				   }, locus()) }
  ;

struct_decl:
    scope_opt question_opt STRUCT ID LBRACE struct_field_decls RBRACE
                       { (StructDecl { st_scope = $1;
				        st_name = $4;
				        st_possibly_null = $2;
				        st_fields = $6}, locus()) }
  ;

union_decl:
    scope_opt question_opt UNION ID LBRACE union_field_decls RBRACE
                       {   (UnionDecl { un_scope = $1;
				      	un_name = $4;
				      	un_possibly_null = $2;
				      	un_fields = $6}, locus ()) }
  ;

extern:
    EXTERN typ ID LPAREN type_list RPAREN SEMICOLON
      { (ExternFun ($2,$3,$5),locus()) }
  | EXTERN typ ID LPAREN RPAREN SEMICOLON
      { (ExternFun ($2,$3,[]),locus()) }
  | EXTERN ID question_opt SEMICOLON             
      { (ExternType ($2,$3),locus()) }
  ;

static_opt:
                        { true }
  | STATIC              { false }
  ;

scope_opt:
    STATIC            { Static }
  | EXTERN            { Extern }
  |                   { Public }
  ;

question_opt:
                       { false }
  | QUESTION           { true }
  ;

param_list:
    decltyp ID brackets  { [($2,make_type $1 $3)] }
  | decltyp ID brackets COMMA param_list { ($2,make_type $1 $3)::$5 }
  ;

constopt:
          { ReadWrite }
  | CONST { ReadOnly }
  ;

iddecl:
    decltyp ibl  { let t = $1 in 
                   let idbs : (string * int) list = $2 in
                   List.map(fun (x,b) -> (x,make_type t b)) idbs }
  ;

ibl:
    ID brackets                       { [($1,$2)] }
  | ID brackets COMMA ibl             { ($1,$2)::$4 }
  ;

struct_field_decls:
                                          { [] }
  | struct_field_decl struct_field_decls  { $1 @ $2 }
  ;

struct_field_decl:
    constopt iddecl SEMICOLON   { let rw = $1 in 
				  List.map (fun (x,t) -> (x,rw,t)) $2 }
  ;

union_field_decls:
                                        { [] }
  | union_field_decl union_field_decls  { $1 @ $2 }
  ;

union_field_decl:
    iddecl SEMICOLON   { $1 }
  ;

block:
    LBRACE stmts RBRACE  { $2 }
  ;

stmts:
    stmt  { $1 }
  | var_decl stmts { List.fold_right 
		       (fun (x,t,eopt) s -> (Decl(x,t,eopt,s),locus())) $1 $2 }
  | stmt stmts { (Seq($1,$2),locus()) }
  ;
var_decl:
    decltyp init_list SEMICOLON
    { let t = $1 in let il = $2 in
      List.map (fun (x,b,e) -> (x,make_type t b,e)) il }
  ; 

init_list:
    init_var                  { [$1] }
  | init_var COMMA init_list  { $1 :: $3 }
  ;

init_var:
    ID brackets             { ($1,$2,ref None) }
  | ID brackets EQUALS exp  { ($1,$2,ref (Some $4)) }
  ;

brackets:
    { 0 }
  | LBRACKET RBRACKET brackets { 1 + $3 }
  ;
    
stmt:
    stmt_no_trailing              { $1 }
  | IF LPAREN exp RPAREN stmt     {(IfThenElse ($3, $5, (Skip,L.none)),locus())}
  | IF LPAREN exp RPAREN stmt_no_short_if ELSE stmt   
      { (IfThenElse ($3, $5, $7),locus()) }
  | WHILE LPAREN exp RPAREN stmt  { (While ($3, $5),locus()) }
  | FOR LPAREN exp SEMICOLON exp SEMICOLON exp RPAREN stmt 
      { (For($3,$5,$7,$9),locus()) }
  ;

stmt_no_short_if:
    stmt_no_trailing    { $1 }
  | IF LPAREN exp RPAREN stmt_no_short_if ELSE stmt_no_short_if   
      { (IfThenElse ($3, $5, $7),locus()) }
  | WHILE LPAREN exp RPAREN stmt_no_short_if  { (While ($3, $5), locus()) }
  | FOR LPAREN exp SEMICOLON exp SEMICOLON exp RPAREN stmt_no_short_if 
      { (For($3,$5,$7,$9),locus()) }
  ;

switch_clauses:
                         { (AnyClause,None) }
  | DEFAULT COLON stmt   { (AnyClause,Some $3) }
  | CASE ID LPAREN ID RPAREN COLON stmts switch_clauses 
      { add_union_clause $2 (Some $4) $7 $8 }
  | CASE ID COLON stmts switch_clauses 
      { add_union_clause $2 None $4 $5 }
  | CASE CONSTINT COLON stmts switch_clauses
      { add_int_clause $2 $4 $5 }
  | CASE CONSTCHAR COLON stmts switch_clauses
      { add_char_clause $2 $4 $5 }
  ;

stmt_no_trailing:
    block                 { $1 }
  | RETURN SEMICOLON      { (Return None,locus()) }
  | RETURN exp SEMICOLON  { (Return (Some($2)),locus()) }
  | BREAK SEMICOLON       { (Break,locus()) }
  | CONTINUE SEMICOLON    { (Continue,locus()) }
  | exp SEMICOLON         { (Exp $1,locus()) }
  | SEMICOLON             { (Skip,locus()) }
  | SWITCH exp LBRACE switch_clauses RBRACE  
      { let (cs,d) = $4 in 
          match (cs,d) with
	    (IntClauses cs,Some s) -> (IntSwitch($2,cs,s),locus())
	  | (IntClauses _,None) -> (err (X.Syntax "integer switch requires default"))
	  | (CharClauses cs,Some s) -> (CharSwitch($2,cs,s),locus())
	  | (CharClauses _,None) -> (err (X.Syntax "char switch requires default"))
          | (UnionClauses cs,_) -> (UnionSwitch($2,cs,d),locus())
	  | (AnyClause,_) -> (err (X.Syntax "No non-default clauses in switch"))
      }	
  ;

decltyp:
    prim_type                    { $1 }
  | ID                           { NamedType ($1) }
  ;

typ:
    decltyp                      { $1 }
  | array_type                   { $1 }
  ;

prim_type:
    VOID       { VoidType }
  | INT        { IntType }
  | BOOLEAN    { BooleanType }
  | CHAR       { CharType }
  | STRING     { StringType }
  ;

/* NG - N.B. this category gets around some type / array access conflicts */

array_type:
    prim_type LBRACKET RBRACKET  { ArrayType $1 }
  | ID LBRACKET RBRACKET         { ArrayType (NamedType $1) }
  | array_type LBRACKET RBRACKET { ArrayType $1 }
  ;

type_list:
    typ                         { [$1] }
  | typ COMMA type_list         { $1 :: $3 }
  ;

exp6:
    LPAREN exp RPAREN        { $2 }
  | ID                       { make_exp (Var $1) }
  | CONSTINT                 { make_exp (Const(Int $1)) }
  | CONSTBOOLEAN             { make_exp (Const(Bool $1)) }
  | CONSTSTRING              { make_exp (Const(String $1)) }
  | CONSTCHAR                { make_exp (Const(Char $1)) }
  | LBRACE COLON typ RBRACE  { make_exp (ConstArray ([],Some $3)) }
  | LBRACE arg_list RBRACE   { make_exp (ConstArray ($2,None)) }
  | NULL ID                  { make_exp (Const(Null $2)) }
  | SIZE LPAREN exp RPAREN   { make_exp (Primop(Size,[$3])) }
  | ORD LPAREN exp RPAREN    { make_exp (Primop(Ord,[$3])) }
  | CHR LPAREN exp RPAREN    { make_exp (Primop(Chr,[$3])) }
  ;

exp5:
    exp6                                 { $1 }
  | exp5 LBRACKET exp RBRACKET           { make_exp (Subscript($1,$3)) }
  | exp5 DOT ID                          { make_exp (StructMember ($1, $3)) }
  | NEWARRAY LPAREN exp COMMA exp RPAREN { make_exp (NewArray ($3, $5)) }
  | NEW ID LPAREN RPAREN               { make_exp (NewStruct ($2, [])) }
  | NEW ID LPAREN arg_list RPAREN      { make_exp (NewStruct ($2, $4)) }
  | NEW ID DOT ID LPAREN exp RPAREN    { make_exp (NewUnion($2,$4,Some $6)) }
  | NEW ID DOT ID                      { make_exp (NewUnion($2,$4,None)) }
  | ID LPAREN RPAREN             { make_exp (FunCall (make_exp (Var $1), [])) }
  | ID LPAREN arg_list RPAREN    { make_exp (FunCall (make_exp (Var $1), $3)) }
  ;

exp4:
    exp5              { $1 }
  | exp4 TIMES exp4   { make_exp (Primop(Times,[$1;$3])) }
  | exp4 DIV exp4     { make_exp (Primop(Div,[$1;$3])) }
  | BANG exp4         { make_exp (Primop(Not,[$2])) }
  ;

exp3:
    exp4                    { $1 }
  | exp3 PLUS exp3          { make_exp (Primop(Plus,[$1;$3])) }
  | exp3 MINUS exp3         { make_exp (Primop(Minus,[$1;$3])) }
  | exp3 EE exp3            { make_exp (Primop(Eq,[$1;$3])) }
  | exp3 NE exp3            { make_exp (Primop(Neq,[$1;$3])) }
  | exp3 LESSTHAN exp3      { make_exp (Primop(Lt,[$1;$3])) }
  | exp3 GREATERTHAN exp3   { make_exp (Primop(Gt,[$1;$3])) }
  | exp3 LESSTHANEQ exp3    { make_exp (Primop(Lte,[$1;$3])) }
  | exp3 GREATERTHANEQ exp3 { make_exp (Primop(Gte,[$1;$3])) }
  | exp3 AMPERAMPER exp3    
       { make_exp (Conditional($1,$3,make_exp(Const(Bool false)))) }
  | exp3 PIPEPIPE exp3 
       { make_exp (Conditional($1,make_exp(Const(Bool true)),$3)) }
  ;

exp2:
    exp3                          { $1 }
  | exp3 QUESTION exp COLON exp2  { make_exp (Conditional ($1, $3, $5)) }
  ;

exp1:
    exp2                { $1 }
    /* XXX this should be exp1 EQUALS exp1: */
  | exp1 PLUSPLUS      { make_exp(AssignOp($1,Plus,make_exp(Const(Int 1)))) }
  | exp1 MINUSMINUS    { make_exp(AssignOp($1,Minus,make_exp(Const(Int 1)))) }
  | exp1 PLUSEQUAL exp3  { make_exp(AssignOp($1,Plus,$3)) }
  | exp1 MINUSEQUAL exp3 { make_exp(AssignOp($1,Minus,$3)) }
  | exp1 TIMESEQUAL exp3 { make_exp(AssignOp($1,Times,$3)) }
  | exp1 DIVEQUAL exp3   { make_exp(AssignOp($1,Div,$3)) }
  ;


exp:
    exp EQUALS exp1     { make_exp(Assign($1,$3)) }
  | exp1        { $1 }
  ;

arg_list:
    exp                    { [$1] }
  | exp COMMA arg_list     { $1 :: $3 }
  ;

