(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Stephanie Weirich,                  *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

let link = ref true;;
let set_no_link() = link := false;;
let assemble = ref true;;
let set_no_assemble() = (assemble := false; set_no_link());;
let executable = ref "";;
let set_executable s = executable := s;;
let set_no_print_comments() = Scomp.print_comments := false;;

let command s =
  Printf.printf "%s\n" s; flush stdout;
  Sys.command s;
  ()
;;

let compileFile filename =
  let (filename,fileBase) = 
    (try (filename,Filename.chop_extension filename) with
      Invalid_argument _ -> (filename^".s",filename)) in
  let talfile = fileBase ^ ".tal" in
  let objectfile = fileBase ^ ".obj" in
  let executable = 
      if (!executable) = "" then fileBase ^ ".exe" else !executable in
  Printf.printf "Compiling file: %s\n" fileBase; flush stdout;
  let fileChannel = open_in filename in
  let syntaxTree = 
    begin
      try 
      	Sparse.prog
      	  Slex.token (Lexing.from_channel fileChannel) 
      with 
	 Parsing.Parse_error ->
	    begin
	       print_string ("Syntax error, line "^
			     (string_of_int (!Slex.line)));
	       print_string ("\n"); print_newline(); close_in fileChannel; 
	       raise Parsing.Parse_error
	    end
       | x  ->
	    begin
	       print_string ("Unknown exception during parsing, line "^
			     (string_of_int (!Slex.line)));
	       (* print_string s; *)
	       print_string ("\n"); print_newline(); close_in fileChannel; 
	       raise x
	   end

    end in
  let ilTree = Sil.xprog syntaxTree in
  let tal_imp = Scomp.code_gen ilTree in
  try
    Talcomp.write_imp fileBase talfile tal_imp;
    command ("talc.exe "^talfile);
    if (!assemble) then
      Talcomp.asm true talfile (Some objectfile);
    if (!link) then
      Talcomp.link true [objectfile] executable;
    ()
  with 
    Sys_error str ->
      print_string "System Error: "; print_string str; print_newline()
;;

let main () =
  begin
    Arg.parse 
      [ "-r",Arg.String Talcomp.set_runtime,"runtime directory";
	"-c",Arg.Unit set_no_link,"do not link";
	"-o",Arg.String set_executable,"executable file name";
	"-A",Arg.Unit set_no_assemble,"do not assemble";
	"-C",Arg.Unit set_no_print_comments,
	  "do not print comments in TAL code"
      ] 
      compileFile
      "scheme: usage: (options) filename ..."
  end
;;

Printexc.catch main () ;;
