(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,                   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

open Popast;;

type typ_env = (string * typ) list

(*
 *  typed_exp:
 *    gamma before expression executes
 *    gamma after expression executes (to keep track of var initialization)
 *    type of expression
 *    expression
 *)
type typed_exp = typ_env * typ * tc_exp
and tc_exp =
    TcIntop of int_op * typed_exp * typed_exp
  | TcCompare of compare * typed_exp * typed_exp
  | TcNull of string
  | TcNot of typed_exp
  | TcIf of typed_exp * typed_exp * typed_exp
  | TcWhile of typed_exp * typed_exp
  | TcCompound of typed_exp * typed_exp
  | TcLet of tc_let_decl * typed_exp
  | TcVarExp of string
  | TcAssign of string * typed_exp
  | TcConstInt of int
  | TcConstBoolean of bool
  | TcConstString of string
  | TcConstVoid
  | TcConstArray of typed_exp list
  | TcPrintInt of typed_exp
  | TcPrintString of typed_exp
  | TcPrintNewline 
  | TcIntOfString of typed_exp
  | TcStringOfInt of typed_exp
  | TcStdArgs
  | TcFunCall of string * (typed_exp list)
  | TcReturn of typed_exp
  | TcNewStruct of string * (typed_exp list)
  | TcStructMember of typed_exp * string
  | TcAssignStructMember of typed_exp * string * typed_exp
  | TcSubscript of typed_exp * typed_exp
  | TcNewArray of typed_exp * typed_exp
  | TcArraySize of typed_exp
  | TcArrayAssign of typed_exp * typed_exp * typed_exp
and tc_let_decl =
    TcVarDecl of string * typ * typed_exp

type tc_func_decl = typ * (decl list)
type tc_struct_decl = bool * Popast.struct_type * decl list

type tc_env = {
    env_fun : (string * tc_func_decl) list;
    env_tids : string list;
    env_sdefs : (string * tc_struct_decl) list
  }
;;

type tc_extern_decl =
    TcExternFun of typ * string * typ list
  | TcExternStruct of string

type tc_top_decl =
    TcFunDecl of bool * string * tc_func_decl * typed_exp

exception TypeError of string
exception NotImplemented of string
exception DeclaredTwice of string
exception MustReturnValue of string
exception RequiresMainFunction of string
exception NoSuchFunction of string
exception NoSuchStruct of string
exception NoSuchVariable of string
exception NoSuchField of string

(* needsReturn is an option:
 *   Some(tau) indicates that the expression must return a value of type tau.
 *   None indicates that the expression need not return a value
 *)
let tcFunExp env =
  let rec tcExp(e, gamma, needsReturn) =
    let needsRetFlag =
      (match needsReturn with Some tau -> true | None -> false) in
    match e with
      Intop (iop, e1, e2) ->
	let te1 = tcExp (e1, gamma, needsReturn)
	and te2 = tcExp (e2, gamma, needsReturn) in
      	(match (te1, te2) with
          ((upGamma1, IntType, upExp1), (upGamma2, IntType, upExp2)) ->
            (gamma, IntType, TcIntop(iop, te1, te2))
        | (_, _) -> raise (TypeError "operator requires 2 integer arguments"))
    | Compare (icmp, e1, e2) ->
	let te1 = tcExp (e1, gamma, needsReturn)
	and te2 = tcExp (e2, gamma, needsReturn) in
      	(match (icmp, te1, te2) with
          ((IEq | INe), (_,tau1,_), (_,tau2,_)) ->
	    if (tau1 = tau2) then
	      (gamma, BooleanType, TcCompare(icmp, te1, te2))
	    else 
	      raise (TypeError "comparison requires 2 arguments of same type")
        | (_, (_,IntType,_), (_,IntType,_)) ->
            (gamma, BooleanType, TcCompare(icmp, te1, te2))
        | (_, _, _) -> raise (TypeError "comparison requires 2 int arguments"))
    | Null s ->
      	(try
	  (match List.assoc s env.env_sdefs with
	    (_,OptionNull,_) ->
	      (gamma, StructType s, TcNull s)
          | _ -> raise (TypeError ("null: "^s^" not a ? struct type")))
	with Not_found -> raise (TypeError ("null: "^s^" invalid type")))
    | Not e1 ->
      	let te1 = tcExp(e1, gamma, needsReturn) in
      	(match te1 with
          (upGamma1, BooleanType, upExp1) ->
            (gamma, BooleanType, TcNot(te1))
        | _ -> raise (TypeError "Not requires 1 boolean argument"))
    | If (e1, e2, e3) ->
      	let te1 = tcExp(e1, gamma, None)
      	and te2 = tcExp(e2, gamma, needsReturn)
      	and te3 = tcExp(e3, gamma, needsReturn) in
      	(match te1 with
          (upGamma1, BooleanType, upExp1) ->
            (match (te2, te3) with
              ((upGamma2, upTau2, upExp2), (upGamma3, upTau3, upExp3)) ->
                if upTau2 = upTau3 then
                  (gamma, upTau2, TcIf(te1, te2, te3))
                else raise (TypeError "Types in if expression must match"))
        | _ -> raise (TypeError "If expression requires a boolean argument"))
    | While (e1, e2) ->
      	if needsRetFlag then raise (MustReturnValue "");
        let te1 = tcExp(e1, gamma, None)
        and te2 = tcExp(e2, gamma, None) in
        (match te1 with
          (upGamma1, BooleanType, upExp1) ->
            (gamma, VoidType, TcWhile(te1, te2))
        | _ ->
            raise (TypeError "While expression requires a boolean argument"))
    | Compound (e1, e2) ->
      (* XXX: could treat e1 = Return (e) as a special case
       * when dealing with needsReturn *)
      	let te1 = tcExp(e1, gamma, None)
      	and te2 = tcExp(e2, gamma, needsReturn) in
      	(match te2 with
          (upGamma2, upTau2, upExp2) ->
            (gamma, upTau2, TcCompound(te1, te2)))
    | VarExp (id) ->
      	if needsRetFlag then raise (MustReturnValue "");
        let tau =
	  try List.assoc id gamma
 	  with Not_found -> raise (NoSuchVariable id) in
        (gamma, tau, TcVarExp (id))
    | Let (VarDecl (id, tau, e1), e2) ->
      	let te1 = tcExp(e1, gamma, None) in
      	(match te1 with
          (upGamma1, upTau1, upExp1) ->
            if tau=upTau1 then
              let te2 = tcExp(e2, (id, tau)::gamma, needsReturn) in
              match te2 with
                (upGamma2, upTau2, upExp2) ->
                  (gamma, upTau2, TcLet(TcVarDecl (id, tau, te1), te2))
            else
	      raise (TypeError ("Wrong type in declaration of " ^ id)))
    | Assign (id, e1) -> 
      let varTau =
 	try List.assoc id gamma
	with Not_found -> raise (NoSuchVariable id)
      and te1 = tcExp(e1, gamma, needsReturn) in
      (match te1 with
        (upGamma1, upTau1, upExp1) ->
          if upTau1 = varTau then (gamma, upTau1, TcAssign(id, te1))
          else raise (TypeError ("Wrong type in assignment to " ^ id)))
    | ConstInt (i) ->
      	if needsRetFlag then raise (MustReturnValue "")
      	else (gamma, IntType, TcConstInt(i))
    | ConstBoolean (b) ->
      	if (needsRetFlag) then raise (MustReturnValue "")
      	else (gamma, BooleanType, TcConstBoolean(b))
    | ConstString (s) ->
      	if (needsRetFlag) then raise (MustReturnValue "")
      	else (gamma, StringType, TcConstString(s))
    | ConstVoid ->
      	if (needsRetFlag) then raise (MustReturnValue "")
      	else (gamma, VoidType, TcConstVoid)
    | ConstArray (es,t) ->
      	(match t with
	  Some t -> (gamma, ArrayType t, TcConstArray [])
      	| None -> 
	    let tes = List.map (fun e -> tcExp (e, gamma, needsReturn)) es in
	    let t =
 	      match tes with (_,t,_)::_ -> t
 	      | _ ->
		  raise
		    (TypeError "zero element array literal must have type") in
	    let f (_,t1,_) =
	      if t<>t1 then
	      	raise
		  (TypeError "array literal elements must have same type") in
	    List.iter f tes;
	    (gamma, ArrayType t, TcConstArray tes))
    | PrintInt (e1) ->
      	let te1 = tcExp(e1, gamma, needsReturn) in
      	(match te1 with
          (upGamma1, IntType, upExp1) ->
            (gamma, VoidType, TcPrintInt(te1))
        | _ -> raise (TypeError "PrintInt requires 1 integer argument"))
    | PrintString (e1) ->
      	let te1 = tcExp(e1, gamma, needsReturn) in
      	(match te1 with
	  (upGamma1, StringType, upExp1) ->
	    (gamma, VoidType, TcPrintString(te1))
      	| _ -> raise (TypeError "PrintString requires 1 string argument"))
    | PrintNewline -> (gamma, VoidType, TcPrintNewline)
    | IntOfString (e1) ->
      	let te1 = tcExp(e1, gamma, needsReturn) in
      	(match te1 with
	  (upGamma1, StringType, upExp1) ->
	    (gamma, IntType, TcIntOfString(te1))
      	| _ -> raise (TypeError "IntOfString requires 1 string argument"))
    | StringOfInt (e1) ->
      	let te1 = tcExp(e1, gamma, needsReturn) in
      	(match te1 with
	  (upGamma1, IntType, upExp1) ->
	    (gamma, StringType, TcStringOfInt(te1))
      	| _ -> raise (TypeError "StringOfInt requires 1 int argument"))
    | StdArgs -> (gamma, ArrayType StringType, TcStdArgs)
    | FunCall (funId, expList) ->
      	let (retType, argDecls) =
	  try List.assoc funId env.env_fun
	  with Not_found -> raise (NoSuchFunction funId)
      	and teList =
	  List.map
            (fun e -> tcExp (e, gamma, needsReturn))
            expList in
        begin
          (try
            List.iter2
              (fun (Decl (id, tau)) (upGamma, upTau, upExp) ->
                if (tau = upTau) then ()
                else raise (TypeError ("Wrong argument type for " ^ id
				       ^ " to function " ^ funId)))
              argDecls
              teList
          with Invalid_argument _ ->
	    raise (TypeError ("Wrong number of arguments to function "^funId))
          );
          (gamma, retType, TcFunCall (funId, teList))
        end
    | Return e1 ->
      	let te1 = tcExp(e1, gamma, None) in
        let (upGamma1, upTau1, upExp1) = te1 in
        begin
          (match needsReturn with
            None -> ()
          | Some (tau) ->
              if (tau = upTau1) then ()
              else raise (TypeError "Function returns wrong type")
          );
          (gamma, VoidType, TcReturn (te1))
        end
    | NewStruct (structId, expList) ->
      	let (_,struct_ty,fieldDecls) = 
	  try List.assoc structId env.env_sdefs
	  with Not_found -> 
	    raise (TypeError ("new struct "^structId^": invalid struct type"))
      	and teList =
	  List.map (fun e -> tcExp (e, gamma, needsReturn)) expList in
        begin
          (try
            List.iter2
              (fun (Decl (id, tau)) (upGamma, upTau, upExp) ->
                if (tau = upTau) then ()
                else raise (TypeError ("Wrong argument type for " ^ id
				       ^ " to new struct " ^ structId)))
              fieldDecls
              teList
          with Invalid_argument (_) -> raise (TypeError
              ("Wrong number of arguments to new struct " ^ structId))
          );
          (gamma, StructType structId, TcNewStruct (structId, teList))
        end
    | StructMember (e1, memberId) ->
      	let te1 = tcExp(e1, gamma, needsReturn) in
      	let check id env = 
	  let (_,_,fieldDecls) =
	    try List.assoc id env
	    with Not_found -> raise (NoSuchStruct id) in
          let fields =
	    List.map (fun (Decl (id, tau)) -> (id, tau)) fieldDecls in
          let memberTau = 
	    try List.assoc memberId fields
	    with Not_found -> raise (NoSuchField memberId) in
          (gamma, memberTau, TcStructMember(te1, memberId)) in
      	(match te1 with
          (_, StructType id, _) -> check id env.env_sdefs
        | _ ->
	    raise
	      (TypeError ("Attempt to project "^memberId^" of a non-struct")))
    | AssignStructMember (e1, memberId, e2) ->
      	let te1 = tcExp(e1, gamma, None)
      	and te2 = tcExp(e2, gamma, needsReturn) in
      	let check id env =
          let (_,_,fieldDecls) =
	    try List.assoc id env
	    with Not_found -> raise (NoSuchStruct id) in
          let fields =
	    List.map (fun (Decl (id, tau)) -> (id, tau)) fieldDecls in
          let memberTau =
	    try List.assoc  memberId fields
	    with Not_found -> raise (NoSuchField memberId)
          and (_, upTau2, _) = te2 in
          if memberTau = upTau2 then
            (gamma, memberTau, TcAssignStructMember(te1, memberId, te2))
          else raise (TypeError "Type mismatch in assignment to struct") in
      	(match te1 with
          (_, StructType id, _) -> check id env.env_sdefs
        | _ ->
	    raise
	      (TypeError ("Attempt to assign "^memberId^" of a non-struct")))
    | Subscript(e1,e2) ->
      	let te1 = tcExp(e1, gamma, needsReturn)
      	and te2 = tcExp(e2, gamma, needsReturn) in
      	begin
	  match (te1, te2) with
	    ((upGamma1, ArrayType tau, upExp1), (upGamma2, IntType, upExp2)) ->
	      (gamma, tau, TcSubscript(te1,te2))
	  | ((_, ArrayType _,_), _) ->
	      raise (TypeError "subscript not an integer")
	  | _ -> raise (TypeError "subscript on non-array")
      	end
    | NewArray (e1,e2) ->
      	let te1 = tcExp(e1, gamma, needsReturn)
      	and te2 = tcExp(e2, gamma, needsReturn) in
      	begin
	  match (te1, te2) with
	    ((upGamma1, IntType, upExp1), (upGamma2, tau, upExp2)) ->
	      (gamma, ArrayType tau, TcNewArray(te1,te2))
	  | _ -> raise (TypeError "first argument of new array not int")
      	end
    | ArraySize e ->
      	let te = tcExp(e,gamma,needsReturn) in
      	begin
	  match te with
	    (upGamma, ArrayType tau, upExp) ->
	      (gamma, IntType, TcArraySize(te))
	  | _ -> raise (TypeError "size argument not array")
      	end
    | ArrayAssign(e1, e2, e3) ->
      	let te1 = tcExp(e1, gamma, needsReturn)
      	and te2 = tcExp(e2, gamma, needsReturn) 
      	and te3 = tcExp(e3, gamma, needsReturn) in
      	begin
	  match (te1, te2, te3) with
	    ((_,ArrayType tau1,_),(_,IntType,_),(_,tau2,_)) ->
	      if tau1 = tau2 then 
	      	(gamma, tau1, TcArrayAssign(te1,te2,te3))
	      else raise (TypeError "array assignment requires equal types")
	  | ((_,ArrayType tau1,_),_,_) ->
	      raise (TypeError "array assignment requires integer index")
	  | _ -> raise (TypeError "array assignment requires array type")
      	end
  in tcExp
;;

let tcExternDecls topDecls =
  let rec aux topDecls defs =
    match topDecls with
      [] -> defs
    | (ExternFun (rt, id, ats))::tl -> aux tl ((TcExternFun (rt,id,ats))::defs)
    | (ExternStruct id)::tl -> aux tl ((TcExternStruct id)::defs)
    | _::tl -> aux tl defs in
  List.rev (aux topDecls [])

let tcFunDecl env (static, id, tau, declList, e) =
  let initialEnv =
    List.map (fun (Decl (argId, argTau)) -> (argId, argTau)) declList in
  let typedExp = tcFunExp env (e, initialEnv, Some (tau)) in
  TcFunDecl (static, id, (tau, declList), typedExp)
;;

let rec tcTopDecls (topDecls, env) = 
  match topDecls with
    [] -> []
  | (FunDecl fDecl)::tl ->
      (tcFunDecl env fDecl)::(tcTopDecls (tl, env))
  | _::tl -> tcTopDecls (tl, env)
;;

(* XXX: need to check that no name is declared twice *)
(* Actually, we don't need to, but this would make more sense to the user *)
(* XXX: need to check publics use only public types *)
let buildEnvs topDecls =
  let funs = ref [] in
  let tids = ref [] in
  let sdefs = ref [] in
  let rec aux topDecls =
    match topDecls with
      [] -> ()
    | (FunDecl (_, id, tau, declList, e))::tl ->
	funs := (id, (tau, declList)) :: !funs;
	aux tl
    | (StructDecl (static, id, st, declList))::tl ->
	tids := id :: !tids;
	sdefs := (id, (static, st, declList)) :: !sdefs;
	aux tl
    | (ExternFun (rt, id, ats))::tl ->
	funs := (id, (rt, List.map (fun t -> Decl ("",t)) ats)) :: !funs;
	aux tl
    | (ExternStruct id)::tl ->
	tids := id :: !tids;
	aux tl in
  aux topDecls;
  {env_fun=List.rev !funs; env_tids=List.rev !tids; env_sdefs=List.rev !sdefs}
;;

let typecheck a =
  match a with
    Ast topDecls ->
      let env = buildEnvs topDecls in
      (env, tcExternDecls topDecls, tcTopDecls (topDecls, env))
;;

(* EOF: poptype.ml *)
