(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,                   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

type typ =
    VoidType
  | IntType
  | BooleanType
  | StringType
  | ArrayType of typ
  | StructType of string

type typ_list =
    NilTypes
  | TypesLeft of typ_list * typ

type exp =
    Intop of Popast.int_op * exp * exp
  | Compare of Popast.compare * exp * exp
  | Null of string
  | Not of exp
  | Conditional of exp * exp * exp
  | VarExp of string
  | Assign of string * exp    (* this should really be exp * exp *)
  | ConstInt of int
  | ConstBoolean of bool
  | ConstString of string
  | ConstArray of exp_list * typ option
  | PrintInt of exp
  | PrintString of exp
  | PrintNewline 
  | IntOfString of exp
  | StringOfInt of exp
  | StdArgs
  | FunCall of string * exp_list
  | NewStruct of string * exp_list
  | StructMember of exp * string
  | AssignStructMember of exp * string * exp
  | Subscript of exp * exp
  | NewArray of exp * exp
  | ArraySize of exp
  | ArrayAssign of exp * exp * exp
and exp_list =
    NilExps
  | ExpsLeft of exp_list * exp
  | ExpsRight of exp * exp_list

type var_decls =
    VarDecl of string * exp
  | VarDecls of var_decls * var_decls

type stmt =
    Block of stmt
  | IfThen of exp * stmt
  | IfThenElse of exp * stmt * stmt
  | While of exp * stmt
  | Compound of stmt * stmt
  | LocalDecl of typ * var_decls
  | ExpStmt of exp
  | Return of exp
  | ReturnVoid
  | EmptyStmt

type decl = Decl of string * typ

type decls =
    NilDecls
  | DeclsLeft of decls * decl
  | DeclsRight of decl * decls

type top_decl =
    FunDecl of (bool * string * typ * decls * stmt)
  | StructDecl of (bool * string * Popast.struct_type * decls)
  | ExternFun of typ * string * typ_list
  | ExternStruct of string
  | TopDecls of top_decl * top_decl

type syntax = Syntax of top_decl

exception NoMatch

let rec typToAst tau =
  match tau with
    VoidType -> Popast.VoidType
  | IntType -> Popast.IntType
  | BooleanType -> Popast.BooleanType
  | StringType -> Popast.StringType
  | StructType id -> Popast.StructType id
  | ArrayType atype -> Popast.ArrayType (typToAst atype)

let rec rightAssocTypes oldTree =
  rightAssocTypesIn (oldTree, [])
and rightAssocTypesIn (oldTree, newTree) =
  match oldTree with
    NilTypes -> newTree
  | TypesLeft (ts,t) -> rightAssocTypesIn (ts, t::newTree)

let rec rightAssocExpList (oldTree) =
  rightAssocExpListIn (oldTree, [])
and rightAssocExpListIn (oldTree, newTree) =
  match oldTree with
    NilExps -> newTree
  | ExpsLeft (es, e) -> rightAssocExpListIn (es, e::newTree)
  | ExpsRight (e, es) -> e::(rightAssocExpListIn (es, newTree))

let rec expToAst syntaxTree =
  match syntaxTree with
    Intop (iop,e1, e2) -> Popast.Intop (iop, expToAst e1, expToAst e2)
  | Compare (icmp, e1, e2) ->
      Popast.Compare (icmp, expToAst e1, expToAst e2)
  | Null s -> Popast.Null s
  | Not (e1) -> Popast.Not (expToAst e1)
  | Conditional (e1, e2, e3) ->
      Popast.If (expToAst e1, expToAst e2, expToAst e3)
  | VarExp (id) -> Popast.VarExp (id)
  | Assign (id, e1) -> Popast.Assign (id, expToAst e1)
  | ConstInt (i) -> Popast.ConstInt (i)
  | ConstBoolean (b) -> Popast.ConstBoolean (b)
  | ConstString (s) -> Popast.ConstString (s)
  | ConstArray (eList,t) ->
      let t = match t with Some t -> Some (typToAst t) | None -> None in
      Popast.ConstArray (List.map expToAst (rightAssocExpList eList),t)
  | PrintInt (e1) -> Popast.PrintInt (expToAst e1)
  | PrintString (e1) -> Popast.PrintString (expToAst e1)
  | PrintNewline -> Popast.PrintNewline
  | IntOfString (e1) -> Popast.IntOfString (expToAst e1)
  | StringOfInt (e1) -> Popast.StringOfInt (expToAst e1)
  | StdArgs -> Popast.StdArgs
  | FunCall (id, eList) -> Popast.FunCall (id,
      List.map expToAst (rightAssocExpList eList))
  | NewStruct (id, eList) -> Popast.NewStruct (id,
      List.map expToAst (rightAssocExpList eList))
  | StructMember (e1, id) -> Popast.StructMember (expToAst e1, id)
  | AssignStructMember (e1, id, e2) ->
      Popast.AssignStructMember (expToAst e1, id, expToAst e2)
  | Subscript (e1, e2) -> Popast.Subscript(expToAst e1, expToAst e2)
  | NewArray (e1, e2) -> Popast.NewArray(expToAst e1, expToAst e2)
  | ArraySize e -> Popast.ArraySize(expToAst e)
  | ArrayAssign(e1, e2, e3) -> 
      Popast.ArrayAssign(expToAst e1, expToAst e2, expToAst e3)

let rec stmtToAst syntaxTree =
  match syntaxTree with
    Block (s1) ->
      stmtToAst (s1)
  | IfThen (e1, s2) ->
      Popast.If (expToAst e1,
        Popast.Compound (stmtToAst s2, Popast.ConstVoid),
        Popast.ConstVoid)
  | IfThenElse (e1, s2, s3) ->
      Popast.If (expToAst e1,
        Popast.Compound (stmtToAst s2, Popast.ConstVoid),
        Popast.Compound (stmtToAst s3, Popast.ConstVoid))
  | While (e1, s2) ->
      Popast.While (expToAst e1, stmtToAst s2)
  | Compound (s1, s2) ->
    (
      match s1 with
        LocalDecl (tau, varDecls) -> declareVars (tau, varDecls, s2)
      | _ -> Popast.Compound (stmtToAst s1, stmtToAst s2)
    )
  | LocalDecl (tau, varDecls) -> declareVars (tau, varDecls, EmptyStmt)
  | ExpStmt (e1) -> expToAst (e1)
  | Return (e1) -> Popast.Return (expToAst e1)
  | ReturnVoid -> Popast.Return (Popast.ConstVoid)
  | EmptyStmt -> Popast.ConstVoid
and declareVars (tau, varDecls, s2) =
  match varDecls with
    VarDecl (id, e1) ->
      Popast.Let (Popast.VarDecl (id, typToAst tau, expToAst e1), stmtToAst s2)
  | VarDecls (VarDecl (id, e1), d2) ->
      Popast.Let (Popast.VarDecl (id, typToAst tau, expToAst e1),
        declareVars (tau, d2, s2))
  | VarDecls (_, _) -> raise NoMatch

let rec declsToAst ds =
  match ds with
    NilDecls -> []
  | DeclsLeft (dsLeft, d) -> raise NoMatch
  | DeclsRight ((Decl (id, tau)), dsRight) ->
      (Popast.Decl (id, typToAst tau))::(declsToAst dsRight)

let rec topToAst syntaxTree =
  let topDeclToAst topDecl =
    match topDecl with
      FunDecl (static, id, tau, ds, s) ->
        Popast.FunDecl (static, id, typToAst tau, declsToAst ds, stmtToAst s)
    | StructDecl (static, id, st, ds) ->
        Popast.StructDecl (static, id, st, declsToAst ds)
    | ExternFun (rt, id, ts) ->
 	Popast.ExternFun
	  (typToAst rt, id, List.map typToAst (rightAssocTypes ts))
    | ExternStruct id -> Popast.ExternStruct id
    | TopDecls (t1, t2) -> raise NoMatch
  in
    match syntaxTree with
      FunDecl _ | StructDecl _ | ExternFun _ | ExternStruct _ ->
 	[topDeclToAst syntaxTree]
    | TopDecls (t1, t2) -> (topDeclToAst t1)::(topToAst t2)

let rec rightAssocVarDecls (oldTree) =
  match oldTree with
    VarDecl (id, e1) -> VarDecl (id, e1)
  | VarDecls (d1, d2) -> rightAssocVarDeclsIn (d1, rightAssocVarDecls (d2))
and rightAssocVarDeclsIn (oldTree, newTree) =
  match oldTree with
    VarDecl (id, e1) -> VarDecls (VarDecl (id, e1), newTree)
  | VarDecls (d1, d2) ->
      let d2right = rightAssocVarDeclsIn (d2, newTree) in
        rightAssocVarDeclsIn (d1, d2right)

let rec rightAssocStmt (oldTree) =
  match oldTree with
    Compound (s1, s2) -> rightAssocStmtIn (s1, rightAssocStmt (s2))
  | Block (s1) -> Block (rightAssocStmt (s1))
  | IfThen (e1, s2) -> IfThen (e1, rightAssocStmt (s2))
  | IfThenElse (e1, s2, s3) ->
      IfThenElse (e1, rightAssocStmt (s2), rightAssocStmt (s3))
  | While (e1, s2) -> While (e1, rightAssocStmt (s2))
  | LocalDecl (tau, varDecls) -> LocalDecl (tau, rightAssocVarDecls varDecls)
  | ExpStmt (e1) -> ExpStmt (e1)
  | Return (e1) -> Return (e1)
  | ReturnVoid -> ReturnVoid
  | EmptyStmt -> EmptyStmt
and rightAssocStmtIn (oldTree, newTree) =
  match oldTree with
    Compound (s1, s2) ->
      let s2right = rightAssocStmtIn (s2, newTree) in
        rightAssocStmtIn (s1, s2right)
  | _ -> Compound (rightAssocStmt (oldTree), newTree)

let rec rightAssocDecls (oldTree) =
  rightAssocDeclsIn (oldTree, NilDecls)
and rightAssocDeclsIn (oldTree, newTree) =
  match oldTree with
    NilDecls -> newTree
  | DeclsLeft (ds, d) -> rightAssocDeclsIn (ds, DeclsRight (d, newTree))
  | DeclsRight (d, ds) -> DeclsRight (d, rightAssocDeclsIn (ds, newTree))

let rec rightAssocTop (oldTree) =
  match oldTree with
    TopDecls (t1, t2) -> rightAssocTopIn (t1, rightAssocTop t2)
  | FunDecl (static, id, tau, decls, s) ->
      FunDecl (static, id, tau, rightAssocDecls decls, rightAssocStmt s)
  | StructDecl (static, id, st, decls) ->
      StructDecl (static, id, st, rightAssocDecls decls)
  | ExternFun (_,_,_) -> oldTree
  | ExternStruct _ -> oldTree
and rightAssocTopIn (oldTree, newTree) =
  match oldTree with
    TopDecls (t1, t2) ->
      let t2right = rightAssocTopIn (t2, newTree) in
        rightAssocTopIn (t1, t2right)
    | _ -> TopDecls (rightAssocTop oldTree, newTree)

let syntaxToAst syntaxTree =
  match syntaxTree with
    Syntax (s) -> Popast.Ast (topToAst (rightAssocTop s))
  
