/**********************************************************************
 * (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,                   *
 *     June 1998, all rights reserved.                                *
 **********************************************************************/

%{

(* Grammar taken largely from Java's syntax (see Java spec) *)

open Popsyntax;;
%}

%token <string> ID
%token <int> CONSTINT
%token <bool> CONSTBOOLEAN
%token <string> CONSTSTRING
%token EOF

%token LPAREN RPAREN LBRACE RBRACE LBRACKET RBRACKET LESSTHAN
%token GREATERTHAN LESSTHANEQ GREATERTHANEQ PLUSPLUS MINUSMINUS
%token PLUSEQUAL MINUSEQUAL TIMESEQUAL DIVEQUAL AMPERAMPER
%token PIPEPIPE EQUALS EE NE PLUS MINUS TIMES DIV BANG QUESTION
%token COLON SEMICOLON DOT COMMA

%token BOOLEAN ELSE EXTERN FOR IF INT INTOFSTRING NEW NEWARRAY NULL
%token PRINTINT PRINTNEWLINE PRINTSTRING RETURN SIZE STATIC STDARGS
%token STRING STRINGOFINT STRUCT VOID WHILE

%right EQUALS
%left PLUSEQUAL MINUSEQUAL TIMESEQUAL DIVEQUAL
%left AMPERAMPER, PIPEPIPE
%left EE, NE, LESSTHAN, GREATERTHAN, LESSTHANEQ, GREATERTHANEQ
%left PLUS MINUS
%left TIMES DIV
%left BANG

%type <Popsyntax.syntax> yy_top

%start yy_top

%%

yy_top:
    yy_top_decls EOF           { Syntax ($1) }
  ;

yy_top_decls:
    yy_top_decl                { $1 }
  | yy_top_decls yy_top_decl   { TopDecls ($1, $2) }
  ;

yy_top_decl:
    yy_func_decl               { $1 }
  | yy_struct_decl             { $1 }
  | yy_extern                  { $1 }
  ;

yy_func_decl:
    yy_static_opt yy_type ID LPAREN RPAREN yy_block
                        { FunDecl ($1, $3, $2, NilDecls, $6) }
  | yy_static_opt yy_type ID LPAREN yy_param_list RPAREN yy_block
                        { FunDecl ($1, $3, $2, $5, $7) }
  ;

yy_static_opt:
                        { true }
  | STATIC              { false }
  ;

yy_param_list:
    yy_param                     { DeclsLeft (NilDecls, $1) }
  | yy_param_list COMMA yy_param    { DeclsLeft ($1, $3) }
  ;

yy_param:
    yy_type ID        { Decl ($2, $1) }
  ;

yy_struct_decl:
    yy_static_opt yy_question_opt STRUCT ID LBRACE RBRACE
                       { StructDecl ($1, $4, $2, NilDecls) }
  | yy_static_opt yy_question_opt STRUCT ID LBRACE yy_field_decls RBRACE
                       { StructDecl ($1, $4, $2, $6) }
  ;

yy_question_opt:
                       { Popast.NeverNull }
  | QUESTION           { Popast.OptionNull }
  ;

yy_field_decls:
    yy_field_decl        { DeclsLeft (NilDecls, $1) }
  | yy_field_decls yy_field_decl    { DeclsLeft ($1, $2) }
  ;

yy_field_decl:
    /* XXX: deal with commas */
    yy_type ID SEMICOLON            { Decl ($2, $1) }
  ;

yy_extern:
    EXTERN yy_type ID LPAREN RPAREN SEMICOLON  { ExternFun ($2,$3,NilTypes) }
  | EXTERN yy_type ID LPAREN yy_type_list RPAREN SEMICOLON
      { ExternFun ($2,$3,$5) }
  | EXTERN ID SEMICOLON             { ExternStruct $2 }
  ;

yy_block:
    LBRACE RBRACE                 { EmptyStmt }
  | LBRACE yy_block_stmts RBRACE   { Block ($2) }
  ;

yy_block_stmts:
    yy_block_stmt                   { $1 }
  | yy_block_stmts yy_block_stmt    { Compound ($1, $2) }
  ;

yy_block_stmt:
    yy_local_decl_stmt                { $1 }
  | yy_stmt                    { $1 }
  ;

yy_local_decl_stmt:
    yy_type yy_var_decls SEMICOLON    { LocalDecl ($1, $2) }
  ;

yy_stmt:
    yy_stmt_no_trailing    { $1 }
  | IF LPAREN yy_exp RPAREN yy_stmt   { IfThen ($3, $5) }
  | IF LPAREN yy_exp RPAREN yy_stmt_no_short_if ELSE yy_stmt   { IfThenElse ($3, $5, $7) }
  | WHILE LPAREN yy_exp RPAREN yy_stmt  { While ($3, $5) }
  | FOR LPAREN yy_exp SEMICOLON yy_exp SEMICOLON yy_exp RPAREN yy_stmt { Compound(ExpStmt $3,While($5,Compound($9,ExpStmt $7))) }
  ;

yy_stmt_no_short_if:
    yy_stmt_no_trailing    { $1 }
  | IF LPAREN yy_exp RPAREN yy_stmt_no_short_if ELSE yy_stmt_no_short_if   { IfThenElse ($3, $5, $7) }
  | WHILE LPAREN yy_exp RPAREN yy_stmt_no_short_if  { While ($3, $5) }
  | FOR LPAREN yy_exp SEMICOLON yy_exp SEMICOLON yy_exp RPAREN yy_stmt_no_short_if { Compound(ExpStmt $3,While($5,Compound($9,ExpStmt $7))) }
  ;

yy_stmt_no_trailing:
    yy_block             { $1 }
  | RETURN SEMICOLON   { ReturnVoid }
  | RETURN yy_exp SEMICOLON  { Return ($2) }
  | yy_stmt_exp SEMICOLON  { $1 }
  | SEMICOLON            { EmptyStmt }
  ;

yy_stmt_exp:
    /* XXX note: the following is more lax than the Java syntax: */
    yy_exp        { ExpStmt ($1) }
  ;

yy_type:
    yy_prim_type  { $1}
  | ID            { StructType ($1) }
  | yy_array_type { $1 }
  ;

yy_prim_type:
    VOID       { VoidType }
  | INT        { IntType }
  | BOOLEAN    { BooleanType }
  | STRING     { StringType }
  ;

/* NG - N.B. this category gets around some type / array access conflicts */

yy_array_type:
    yy_prim_type LBRACKET RBRACKET  { ArrayType $1 }
  | ID LBRACKET RBRACKET            { ArrayType (StructType $1) }
  | yy_array_type LBRACKET RBRACKET { ArrayType $1 }
  ;

yy_type_list:
    yy_type                         { TypesLeft (NilTypes,$1) }
  | yy_type_list COMMA yy_type      { TypesLeft ($1,$3) }
  ;

yy_var_decls:
    yy_var_decl    { $1 }
  | yy_var_decls COMMA yy_var_decl    { VarDecls ($1, $3) }
  ;

yy_var_decl:
/* XXX:   yy_var_decl_id    */
    yy_var_decl_id EQUALS yy_var_init  { VarDecl ($1, $3) }
  ;

yy_var_decl_id:
    ID            { $1 }
/* XXX: arrays */
  ;

yy_var_init:
    yy_exp        { $1 }
/* XXX: arrays */
  ;

yy_exp6:
    LPAREN yy_exp RPAREN   { $2 }
  | yy_name                { VarExp ($1) }
  | CONSTINT               { ConstInt ($1) }
  | CONSTBOOLEAN           { ConstBoolean ($1) }
  | CONSTSTRING            { ConstString ($1) }
  | LBRACE COLON yy_type RBRACE { ConstArray (NilExps,Some $3) }
  | LBRACE yy_arg_list RBRACE { ConstArray ($2,None) }
  | PRINTINT LPAREN yy_exp RPAREN  { PrintInt ($3) }
  | PRINTSTRING LPAREN yy_exp RPAREN { PrintString ($3) }
  | PRINTNEWLINE LPAREN RPAREN { PrintNewline }
  | INTOFSTRING LPAREN yy_exp RPAREN { IntOfString ($3) }
  | STRINGOFINT LPAREN yy_exp RPAREN { StringOfInt ($3) }
  | STDARGS { StdArgs }
  | NULL yy_name { Null ($2) }
  | SIZE LPAREN yy_exp RPAREN {ArraySize ($3)}
  ;

yy_exp5:
    yy_exp6                { $1 }
  | STDARGS LBRACKET yy_exp RBRACKET { Subscript(StdArgs,$3) }
  | ID LBRACKET yy_exp RBRACKET { Subscript(VarExp $1,$3) }
  | yy_exp5 DOT ID        { StructMember ($1, $3) }
  | NEWARRAY LPAREN yy_exp COMMA yy_exp RPAREN {NewArray ($3, $5)}
  | NEW ID LPAREN RPAREN   { NewStruct ($2, NilExps) }
  | NEW ID LPAREN yy_arg_list RPAREN   { NewStruct ($2, $4) }
  | ID LPAREN RPAREN     { FunCall ($1, NilExps) }
  | ID LPAREN yy_arg_list RPAREN     { FunCall ($1, $3) }
  ;

yy_exp4:
    yy_exp5                 { $1 }
  | yy_exp4 TIMES yy_exp4   { Intop (Popast.Itimes, $1, $3) }
  | yy_exp4 DIV yy_exp4     { Intop (Popast.Idiv, $1, $3) }
  | BANG yy_exp4            { Not ($2) }
  ;

yy_exp3:
    yy_exp4                  { $1 }
  | yy_exp3 PLUS yy_exp3     { Intop (Popast.Iplus, $1, $3) }
  | yy_exp3 MINUS yy_exp3    { Intop (Popast.Iminus, $1, $3) }
  | yy_exp3 EE yy_exp3       { Compare (Popast.IEq, $1, $3) }
  | yy_exp3 NE yy_exp3       { Compare (Popast.INe, $1, $3) }
  | yy_exp3 AMPERAMPER yy_exp3 { (Conditional($1,$3,ConstBoolean(false))) }
  | yy_exp3 PIPEPIPE yy_exp3 { (Conditional($1,ConstBoolean(true),$3)) }
  | yy_exp3 LESSTHAN yy_exp3 { Compare (Popast.ILt, $1, $3) }
  | yy_exp3 GREATERTHAN yy_exp3 { Compare (Popast.IGt, $1, $3) }
  | yy_exp3 LESSTHANEQ yy_exp3 { Compare (Popast.ILe, $1, $3) }
  | yy_exp3 GREATERTHANEQ yy_exp3 { Compare (Popast.IGe, $1, $3) }
  ;

yy_exp2:
    yy_exp3            { $1 }
  | yy_exp3 QUESTION yy_exp COLON yy_exp2  { Conditional ($1, $3, $5) }
  ;

yy_exp1:
    yy_exp2            { $1 }
    /* XXX this should be yy_exp1 EQUALS yy_exp1: */
  | ID EQUALS yy_exp1       { Assign ($1, $3) }
  | ID PLUSPLUS             { Assign ($1, Intop(Popast.Iplus,VarExp $1,ConstInt 1)) }
  | ID MINUSMINUS           { Assign ($1, Intop(Popast.Iminus,VarExp $1,ConstInt 1)) }
  | ID LBRACKET yy_exp RBRACKET EQUALS yy_exp1 {ArrayAssign(VarExp $1,$3,$6)}
  | yy_exp5 DOT ID EQUALS yy_exp1  { AssignStructMember ($1, $3, $5) }
  | ID PLUSEQUAL yy_exp3 { Assign($1,Intop(Popast.Iplus, VarExp $1,$3)) }
  | ID MINUSEQUAL yy_exp3 { Assign($1,Intop(Popast.Iminus, VarExp $1,$3)) }
  | ID TIMESEQUAL yy_exp3 { Assign($1,Intop(Popast.Itimes, VarExp $1,$3)) }
  | ID DIVEQUAL yy_exp3 { Assign($1,Intop(Popast.Idiv, VarExp $1,$3)) }
  ;


yy_exp:
    yy_exp1        { $1 }
  ;

yy_arg_list:
    yy_exp                    { ExpsLeft (NilExps, $1) }
  | yy_arg_list COMMA yy_exp    { ExpsLeft ($1, $3) }
  ;

yy_name:
    ID        { $1 }
  ;



