(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,                   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

let link = ref true;;
let set_no_link() = link := false;;
let assemble = ref true;;
let set_no_assemble() = (assemble := false; set_no_link());;
let executable = ref "a.out";;
let set_executable s = executable := s;;
let talfile = ref "";;
let set_talfile s = talfile := s;;
let objectfile = ref "";;
let set_objectfile s = objectfile := s;;
let set_no_optimize () = Popcompile.optimize_tal := false;;

let command s =
  Printf.printf "%s\n" s; flush stdout;
  Sys.command s;
  ()
;;

let objfiles = ref [];;

let compileFile filename =
  let fileBase = Filename.chop_extension filename in
  let modname = Filename.basename fileBase in
  let talfile = if (!talfile) = "" then fileBase ^ ".tal" else !talfile in
  let impfile = fileBase ^ "_i.tali" in
  let expfile = fileBase ^ "_e.tali" in
  let objectfile = 
      if (!objectfile) = "" then fileBase ^ ".obj" else !objectfile in
  Printf.printf "Compiling file: %s\n" fileBase; flush stdout;
  let fileChannel = open_in filename in
  let syntaxTree = Popparse.yy_top
    Poplex.token (Lexing.from_channel fileChannel) in
  let abstractTree = Popsyntax.syntaxToAst syntaxTree in
  let typedTree = Poptype.typecheck abstractTree in
  let (impint,expint,tal_imp) =
    Popcompile.compileToModule impfile expfile typedTree in
  try
    Talcomp.write_int (modname^"_i") impfile impint;
    Talcomp.write_int (modname^"_e") expfile expint;
    Talcomp.write_imp modname talfile tal_imp;
    command ("talc.exe "^talfile);
    if (!assemble) then begin
      Talcomp.asm true talfile (Some objectfile);
      objfiles := objectfile :: !objfiles
    end;
    ()
  with 
    Sys_error str ->
      print_string "System Error: "; print_string str; print_newline()
;;

let do_link () =
  if (!link) then
    Talcomp.link true (List.rev !objfiles) !executable;
;;

let main () =
  Arg.parse 
    [ "-r",Arg.String Talcomp.set_runtime,"runtime directory";
      "-c",Arg.Unit set_no_link,"do not link";
      "-o",Arg.String set_executable,"executable file name";
      "-A",Arg.Unit set_no_assemble,"do not assemble";
      "-t",Arg.String set_talfile,"tal file name";
      "-nopt",Arg.Unit set_no_optimize,"don't peephole optimize"
    ] 
    compileFile
    "popcorn: usage: <options> file ...";
  do_link ()
;;

Printexc.catch main () ;;
