(**********************************************************************)
(* (c) Greg Morrisett, Neal Glew, Chris Hawblitzel,                   *)
(*     June 1998, all rights reserved.                                *)
(**********************************************************************)

open Identifier;;
open Popast;;
open Poptype;;
open Tal;;

let string_list : (identifier * string) list ref = ref []
let add_string s = 
  let l = id_new "cstr" in
  begin
    string_list := (l,s) :: (!string_list);
    l
  end

type code_tree =
    InsList of instruction list
  | CodeLabel of identifier * con
  | CodeList of code_tree list

let codeTreeFoldLeft labelFunc insFunc =
  let rec doFold x t =
    match t with
      InsList [] -> x
    | InsList (ins::tl) -> doFold (insFunc x ins) (InsList tl)
    | CodeLabel (lab, labcon) -> (labelFunc x (lab, labcon))
    | CodeList [] -> x
    | CodeList (hd::tl) -> doFold (doFold x hd) (CodeList tl)
  in doFold

exception MissingStartLabel

(* convert a code_tree to a (label * con * (instruction list)) list *)
let flattenCodeTree (codeTree) =
  let reverseCodeList = codeTreeFoldLeft
    (fun buildList -> fun (lab, labState) ->
       (lab, labState, [])::buildList)
    (fun buildList -> fun ins ->
       match buildList with
         [] -> raise MissingStartLabel
       | (lab, labcon, insList)::tl -> (lab, labcon, ins::insList)::tl)
    []
    codeTree
  in
    List.rev (List.map (fun (lab, labcon, reverseInsList) ->
      (lab, labcon, List.rev reverseInsList)) reverseCodeList)

let negate_cc cc = 
  match cc with
    Eq -> NotEq
  | NotEq -> Eq
  | GreaterEq -> Less
  | Greater -> LessEq
  | LessEq -> Greater
  | Less -> GreaterEq
  | _ -> failwith "optimizer found unexpected condition code"

let optimize_tal = ref true;;

let optimize_code (l,ls,is) = 
  let clicks = ref 0 in
  let click () = clicks := (!clicks) + 1 in
  let rec optimize is = 
    match is with
      [] -> []
    | (Mov(Reg Ebx,(op1,[])))::(Cmp(Reg Ebx,Reg r))::rest ->
	(click(); optimize ((Cmp(op1,Reg r))::rest))
    | (Mov(Reg Eax,(Immed i,[])))::(Cmp(op,Reg Eax))::rest ->
	(click(); optimize ((Cmp(op,Immed i))::rest))
    | (ArithBin(Add,Reg Esp,Immed i))::(ArithBin(Add,Reg Esp,Immed j))::rest ->
	(click(); optimize((ArithBin(Add,Reg Esp,Immed (i+j)))::rest))
    | (Mov(Reg Eax,op))::(Mov(Reg Ebx,(Reg Eax,[])))::rest ->
	(click(); optimize((Mov(Reg Ebx,op))::rest))
    | (Mov(Reg Eax,op)::Push (Reg Eax,[])::rest) -> 
	(click(); optimize ((Push op):: rest))
    | (Push (r1,_ as op1))::(Mov(Reg Eax,(Prjr((Esp,c1),i),c2)))::
      (Pop(Reg r))::rest
      when (match r1 with Reg Eax | Prjr ((Eax,_),_) -> false | _ -> true) ->
	(click(); 
	 optimize(Mov(Reg Eax,(Prjr((Esp,c1),i-4),c2))::
		  (Mov(Reg r,op1))::rest))
    | (ArithBin(Add,_,Immed 0))::rest -> (click(); optimize rest)
    | (Mov (Reg Eax,(gop,cs1)))::(Mov(Reg Eax,(Reg Eax,cs2)))::rest ->
	(click(); optimize(Mov(Reg Eax,(gop,cs2@cs1))::rest))
    | ((Mov(Reg Ecx,(Tag 0,[Tosum c]))) as i)::
      (((Cmp(Reg Ebx,Reg Eax))::
       	(Setcc(cc,Reg Ecx))::
       	(Mov(Reg Eax,(Reg Ecx,[])))::
       	(Btagi(Eax,0,lc,cc2))::rest) as is') ->
	begin
	  match c.rcon with
	    Csum{sum_tags=[0;1];sum_vt=None} ->
	      click();
	      begin
	      	match cc2 with
	    	  Eq -> 
		    optimize((Cmp(Reg Ebx,Reg Eax))::
			     (Jcc(negate_cc cc,lc))::rest)
	      	| NotEq ->
		    optimize((Cmp(Reg Ebx,Reg Eax))::(Jcc(cc,lc))::rest)
	      	| _ -> failwith "optimize: strange boolean test"
	      end
	  | _ -> i::(optimize is')
	end
    | (Push (op1,[]))::(Mov (Reg Eax,op2))::(Pop (Reg Ebx))::
      (Cmp (Reg Ebx,Reg Eax))::rest when op1<>(Reg Eax) ->
	click();
	optimize ((Mov (Reg Ebx,op2))::(Cmp (op1,Reg Ebx))::rest)
    | (Retn _ as i)::_ -> [i]
    | (Jmp _ as i)::_ -> [i]
    | hd::tl -> hd::(optimize tl) in
  let optimize_loop is = 
    let is = optimize is in if (!clicks) > 0 then (clicks := 0; 
						   optimize is) else is
  in (l,ls,optimize_loop is)

let optimize s = if !optimize_tal then optimize_code s else s

let codeTreeToCode (codeTree) =
  let codeBlockList = List.map
    (fun (lab, labState, insList) ->
      (lab, labState, Array.of_list insList))
    (List.map optimize (flattenCodeTree codeTree))
  in
    Array.of_list codeBlockList

(*
 *  Register state at labels:
 *    All registers are spilled onto the stack at each label,
 *    so the only register with anything interesting is the
 *    stack pointer itself (all other registers are junk).
 *)
(*
 *  Booleans are represented as integers:
 *    zero indicates false
 *    non-zero indicates true
 *)
(*
 *  typedExpToCodeTree
 *
 *  Attributes:
 *    down:
 *      frame:
 *        frameSize (current frame size in words)
 *        stackState (a con value of the types on the stack frame)
 *            All words in stackState are set as initialized even if
 *            they aren't yet, because they will be initialized by the
 *            next label anyway.
 *      varMap (maps variable identifiers to locations in the frame)
 *    up:
 *      code_tree
 *)

type struct_def = {
    sd_lab : identifier;
    sd_option : struct_type;
    sd_def : decl list;
    sd_static : bool
  } 
;;

type env = {
    env_fun : (string * (tc_func_decl * identifier)) list;
    env_tids : (string * identifier) list;
    env_sdefs : (string * struct_def) list
  }
;;

let findStruct env id : struct_def =
  try
    List.assoc id env.env_sdefs
  with Not_found -> raise (NoSuchStruct id)

let findFun env id =
  try
    List.assoc id env.env_fun
  with Not_found -> raise (NoSuchFunction id)

let cprim pc = defcon(Cprim pc)
let csum sv = defcon(Csum sv)
let capp (c1,c2) = defcon(Capp(c1,c2))
let cvar v = defcon(Cvar v)
let ccons (c1,c2) = defcon(Ccons(c1,c2))
let cforall (v,k,c) = defcon(Cforall(v,k,c))
let ccode rs = defcon(Ccode rs)
let cempty = defcon(Cempty)
let cprod fs = defcon(Cprod fs)
let clam (v,k,c) = defcon(Clam(v,k,c))
let crec vkcs = defcon(Crec vkcs)
let clab l = defcon(Clab l)

let junkCon = cprim (PCjunk)
let intCon = cprim (PCbytes Byte4)
let booleanCon = csum {sum_tags = [0; 1]; sum_vt = None}
let arrayCon c = capp(cprim PCarray,c)
let stringCon = cprim (PCbytearray Byte1)
let structCon env id =
  try clab (List.assoc id env.env_tids)
  with Not_found -> raise (NoSuchStruct id)

let rec typeCon env tau = 
  match tau with
    IntType -> intCon
  | BooleanType -> booleanCon
  | VoidType -> junkCon
  | StringType -> stringCon
  | StructType id -> structCon env id
  | ArrayType c -> arrayCon (typeCon env c)

let addWordCon (c, (stackState, frameSize)) =
  (ccons(c, stackState), frameSize + 1)

let addInt (stackState, frameSize) =
  (ccons(intCon, stackState), frameSize + 1)

let addBoolean (stackState, frameSize) =
  (ccons(booleanCon, stackState), frameSize + 1)

let addString (stackState, frameSize) = 
  (ccons(stringCon, stackState), frameSize + 1)

let addType env (tau, frame) =
  match tau with
    VoidType -> frame
  | _ -> addWordCon (typeCon env tau,frame)

let addVar (id, tau, (stackState, frameSize), varMap) =
  let addWord () = (id, frameSize)::varMap in
    match tau with
      IntType -> addWord ()
    | BooleanType -> addWord ()
    | VoidType -> varMap
    | StringType -> addWord ()
    | StructType id -> addWord ()
    | ArrayType _ -> addWord ()

let varIndex (id, (stackState, frameSize), varMap) =
  let index = (List.assoc id varMap) in (frameSize - index - 1)

let emptyRegState = Dict.empty compare_regs

let callerStackVar = id_new "r"
let callerStackCvar = cvar (callerStackVar)

let exit_label = id_of_string "_exit"

let jumpCoercion = [Tapp callerStackCvar]
let fallthruCoercion = [callerStackCvar]
let funCallCoercion (stackState, frameSize) = [Tapp stackState]
let polyfunCallCoercion c (stackState, frameSize) = [Tapp stackState;Tapp c]

let labelState (stackState, frameSize) =
  cforall (callerStackVar, Kstack,
	   ccode (Dict.insert emptyRegState Esp 
		    (capp (cprim PCstackptr, stackState))))

let labelStateEax c (stackState, frameSize) =
  match c.rcon with
    Cprim(PCjunk) -> labelState (stackState,frameSize)
  | _ -> 
      cforall
  	(callerStackVar, Kstack,
	 ccode 
	   (Dict.insert 
    	      (Dict.insert 
		 emptyRegState Esp (capp (cprim PCstackptr, stackState)))
     	      Eax c))

let funLabelName (id) = ("func_" ^ id)

let popResult (gamma, tau, e) =
  let popWord () = InsList [ArithBin (Add,Reg Esp,Immed 4)] in
    match tau with
      IntType -> popWord ()
    | BooleanType -> popWord ()
    | VoidType -> CodeList []
    | StringType -> popWord ()
    | StructType id -> popWord ()
    | ArrayType _ -> popWord ()

let pushResult ((gamma, tau, e), r) =
  let pushWord () = InsList [Push (Reg r, [])] in
      match tau with
        IntType -> pushWord ()
      | BooleanType -> pushWord ()
      | VoidType -> CodeList []
      |	StringType -> pushWord ()
      | StructType id -> pushWord ()
      |	ArrayType _ -> pushWord ()

let wordsBelowRetAddr (argsRetAddrFrame, currentFrame) =
  let ((_, argsRetAddrFrameSize), (_, currentFrameSize)) =
    (argsRetAddrFrame, currentFrame)
  in (currentFrameSize - argsRetAddrFrameSize)

let retPopWords (_, argsRetAddrFrameSize) = (argsRetAddrFrameSize - 1)

let structMallocArg env fieldDecls =
  let conCapList =
    List.map
      (fun (Decl (id, tau)) -> (typeCon env tau, ReadWrite))
      fieldDecls in
  Mprod conCapList

let fieldOffset fieldDecls memberId =
  let rec matchField fieldList offset =
    match fieldList with
      [] -> raise (NoSuchField memberId)
    | (Decl (id, tau))::tl -> if id = memberId then offset
        else matchField tl (offset + 4)
  in matchField fieldDecls 0

let intCondition icmp =
  match icmp with
    IEq -> Eq
  | INe -> NotEq
  | IGe -> GreaterEq
  | IGt -> Greater
  | ILe -> LessEq
  | ILt -> Less

(* _inf_loop : All[a:Ts].{Esp:sptr a}: infinite loop to deal with exit problem.
 *)
let inf_loop_label = id_of_string "_inf_loop";;
let inf_loop_alpha = id_new "a";;
let inf_loop_con =
  cforall (inf_loop_alpha,Kstack,
	   ccode (Dict.insert emptyRegState Esp
		    (capp (cprim PCstackptr,cvar inf_loop_alpha))))
;;

(* _tal_main : {Esp: sptr se}:  calls popcorn code's main and then exits with
 * 0 as a value.
 *)
let tal_main_label = id_of_string "_tal_main";;
let tal_main_con = ccode (Dict.insert emptyRegState Esp (capp(cprim PCstackptr,
						      cempty)));;

(* _null_exn : All[a:Ts].{Esp: sptr a}: exits with 1 *)
let null_exn_label = id_of_string "_null_exn";;
let alpha = id_new "a";;
let null_exn_con =
  cforall (alpha,
	   Kstack,
	   ccode (Dict.insert
		    emptyRegState Esp (capp(cprim PCstackptr,cvar alpha))));;

let funBodyToCodeTree (env, argsRetAddrFrame) =
let rec typedExpToCodeTree ((gamma, tau, e), frame, varMap) =
  match e with
    TcIntop (Iplus, te1, te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and code2 = typedExpToCodeTree (te2, addInt frame, varMap) in
        CodeList 
	[ code1; 
	  InsList [Push (Reg Eax,[])];
	  code2;
          InsList
          [
            Pop (Reg Ebx);
            ArithBin (Add, Reg Eax, Reg Ebx)
          ]]
  | TcIntop (Itimes, te1, te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and code2 = typedExpToCodeTree (te2, addInt frame, varMap) in
        CodeList 
	[ code1; 
	  InsList [Push (Reg Eax,[])];
	  code2;
          InsList
          [
            Pop (Reg Ebx);
            ArithBin (Imul2, Reg Eax, Reg Ebx)
          ]]
  | TcIntop (Popast.Idiv, te1, te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and code2 = typedExpToCodeTree (te2, addInt frame, varMap) in
        CodeList
        [ code1;
	  InsList [Push (Reg Eax,[])];
	  code2;
          InsList
          [ Pop (Reg Ebx);
	    Xchg (Reg Eax,Ebx);
	    Conv Cdq;
            ArithMD (Tal.Div, Reg Ebx)
          ]]
  | TcIntop (Iminus, te1, te2) -> 
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and code2 = typedExpToCodeTree (te2, addInt frame, varMap) in
        CodeList 
	[ code1; 
	  InsList [Push (Reg Eax,[])];
	  code2;
          InsList
          [
	    Mov (Reg Ebx, (Reg Eax,[]));
            Pop (Reg Eax);
            ArithBin (Sub, Reg Eax, Reg Ebx)
          ]]
  | TcCompare (icmp, te1, te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and code2 = typedExpToCodeTree (te2, addInt frame, varMap) in
        CodeList 
	[ code1; 
	  InsList [Push (Reg Eax,[])];
	  code2;
          InsList
          [
            Pop (Reg Ebx);
            Mov (Reg Ecx, (Tag 0, [Tosum booleanCon]));
            Cmp (Reg Ebx, Reg Eax);
            Setcc (intCondition icmp, Reg Ecx);
	    Mov (Reg Eax,(Reg Ecx,[]))
          ]]
  | TcNot (te1) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
        CodeList [code1;
          InsList
          [
            Mov (Reg Ebx, (Tag 0, [Tosum booleanCon]));
            Cmp (Reg Eax, Reg Ebx);
            Setcc (Eq, Reg Ebx);
	    Xchg (Reg Eax,Ebx)
          ]]
  | TcIf (te1, te2, te3) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and code2 = typedExpToCodeTree (te2, frame, varMap)
      and code3 = typedExpToCodeTree (te3, frame, varMap)
      and falseLabel = id_new "ifFalse"
      and endLabel = id_new "ifEnd"
      in
        CodeList
        [
          code1;
          InsList
          [
              (* if false, jump: *)
            Btagi (Eax, 0, (falseLabel, jumpCoercion), Eq)
          ];
          code2;
          InsList
          [
            Jmp (Addr endLabel, jumpCoercion)
          ];
          CodeLabel (falseLabel, labelState frame);
          code3;
          InsList
          [
            Fallthru fallthruCoercion
          ];
          CodeLabel (endLabel, labelStateEax (typeCon env tau) frame)
        ]
  | TcWhile (te1, te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and code2 = typedExpToCodeTree (te2, frame, varMap)
      and loopBody = id_new "loopBody"
      and loopTest = id_new "loopTest"
      in
        CodeList
        [
          InsList [ Jmp (Addr loopTest, jumpCoercion) ]; (* goto loopTest *)
          CodeLabel (loopBody, labelState frame);        (* loopBody: *)
          code2;
          InsList [ Fallthru fallthruCoercion ] ;
          CodeLabel (loopTest, labelState frame);        (* loopTest: *)
          code1;
          InsList
          [
              (* if true goto loopBody *)
            Btagi (Eax, 0, (loopBody, jumpCoercion), NotEq)
          ]
        ]
  | TcCompound (te1, te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and code2 = typedExpToCodeTree (te2, frame, varMap) in
        CodeList
        [
          code1;
          code2
        ]
  | TcLet (TcVarDecl (id, tau, te1), te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap)
      and newVarMap = addVar (id, tau, frame, varMap) in
        let code2 = typedExpToCodeTree
          (te2, addType env (tau, frame), newVarMap)
        in
          CodeList
          [
            code1;  (* leave result (the new local variable) on the stack *)
    	    InsList [Push (Reg Eax,[])];
            code2;
            popResult te1 (* discard the local variable *)
          ]
  | TcVarExp (id) ->
      let pushWord () =
        let ind = varIndex (id, frame, varMap) in
          InsList
          [
	    Mov (Reg Eax,(Prjr((Esp, []), ind * 4),[]))
          ]
      in
      (
        match tau with
          IntType -> pushWord ()
        | BooleanType -> pushWord ()
        | VoidType -> CodeList []
	| StringType -> pushWord ()
        | StructType id -> pushWord ()
	| ArrayType _ -> pushWord ()
      ) 
  | TcAssign (id, te1) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
        let storeWord () =
          let ind = varIndex (id, frame, varMap) in
            InsList
            [
              (* move top value of stack into local variable *)
              (* leave the top stack value on the stack *)
              Mov (Prjr ((Esp, []), ind * 4), (Reg Eax, []))
            ]
        in
          CodeList
          [
            code1;
            match tau with
              IntType -> storeWord ()
            | BooleanType -> storeWord ()
            | VoidType -> CodeList []
	    | StringType -> storeWord ()
            | StructType id -> storeWord ()
	    | ArrayType _ -> storeWord ()
          ]
  | TcConstInt (i) ->
      InsList
      [
        Mov (Reg Eax, (Immed i, []))
      ]
  | TcConstBoolean (b) ->
      InsList
      [
        Mov (Reg Eax, (Tag (if b then 1 else 0), [Tosum booleanCon]))
      ]
  | TcConstString (s) ->
       InsList [Mov (Reg Eax, (Addr (add_string s),[]))]
  | TcConstVoid -> CodeList []
  | TcConstArray es ->
      let f (partialFrame, partialCode) (argGamma, argTau, argExp) =
	(addType env (argTau, partialFrame),
         (CodeList
	    [ typedExpToCodeTree
		((argGamma, argTau, argExp), partialFrame, varMap);
	      InsList [Push (Reg Eax,[])]
	    ])::partialCode) in
      let (newFrame, argPushCodeRev) = List.fold_left f (frame, []) es in
      let n = List.length es in
      let rec g n i partialCode =
	if i>4*n then
	  partialCode
	else
	  let partialCode =
	    (Pop (Reg Ebx))::
	    (Mov (Prjr ((Eax,[]),i),(Reg Ebx,[])))::
	    partialCode in
	  g n (i+4) partialCode in
      let stores = g n 4 [] in
      let ma =
	let t =
	  match tau with ArrayType t -> typeCon env t
	  | _ -> failwith "internal error" in
	let rec h n t ts = if n=0 then ts else h (n-1) t ((t,ReadWrite)::ts) in
	Mprod ((cprim (PCtag n),Read)::(h n t [])) in
      CodeList
        [ CodeList (List.rev argPushCodeRev);
          InsList [ Malloc (Eax,4*(n+1),ma);
		    Mov (Prjr ((Eax,[]),0),(Tag n,[])) ];
	  InsList stores;
	  InsList [ Coerce (Eax,[Toarray]) ]
        ]
  | TcPrintInt (te1) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
        CodeList
        [
          code1;
          InsList
	  [
            Push (Reg Eax,[]);
            Call (Addr (id_of_string "_print_int"), 
		  funCallCoercion frame);
            ArithBin(Add,Reg Esp,Immed 4)
          ] 
          (* popResult (te1, Eax) *)
        ]
  | TcPrintString (te1) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
        CodeList
	[ code1; 
	  InsList [ Push (Reg Eax,[]);
		    Call (Addr (id_of_string "_print_string"),
			  funCallCoercion frame);
		    ArithBin(Add,Reg Esp,Immed 4)
		  ] 
	] 
  | TcPrintNewline ->
      InsList [ Call (Addr (id_of_string "_print_newline"),
		      funCallCoercion frame)
	      ]	
  | TcIntOfString (te1) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
        CodeList
	[ code1; 
	  InsList [ Push (Reg Eax,[]);
		    Call (Addr (id_of_string "_int_of_string"),
			  funCallCoercion frame);
		    ArithBin(Add,Reg Esp,Immed 4)
		  ] 
	] 
  | TcStringOfInt (te1) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
        CodeList
	[ code1; 
	  InsList [ Push (Reg Eax,[]);
		    Call (Addr (id_of_string "_string_of_int"),
			  funCallCoercion frame);
		    ArithBin(Add,Reg Esp,Immed 4)
		  ] 
	] 
  | TcStdArgs ->
      InsList [ Mov(Reg Eax,(Addr (id_of_string "_args"),[])) ]
  | TcNewArray (te1,te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
      let code2 = typedExpToCodeTree (te2, addInt frame, varMap) in
      let (_,tau,_) = te2 in
      let c = typeCon env tau in
        CodeList
        [
          code1;
          InsList [ Push (Reg Eax,[])];
	  code2;
	  InsList 
          [ Pop (Reg Ebx);
            Push (Reg Eax,[]);
	    Push (Reg Ebx,[]);
            Call (Addr (id_of_string "_new_array"), 
		  polyfunCallCoercion c frame);
            ArithBin(Add,Reg Esp,Immed 8)
          ] 
        ]
  | TcFunCall (funId, argList) ->
      let (newFrame, argPushCodeRev) = List.fold_left
        (fun (partialFrame, partialCode) (argGamma, argTau, argExp) ->
          (
            addType env (argTau, partialFrame),
            (CodeList
	       [ typedExpToCodeTree
		   ((argGamma, argTau, argExp), partialFrame, varMap);
		 InsList [Push (Reg Eax,[])]
	       ] 
            )
              ::partialCode
          )
        )
        (frame, [])
        argList
      and (funDecl, funLabel) = (findFun env funId)
      in
        CodeList
        [
          CodeList (List.rev argPushCodeRev);
          InsList
          [
            Call (Addr funLabel, funCallCoercion frame)
          ]
        ]
  | TcReturn (te1) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
        CodeList
        [
          code1;
          InsList
          [
            (* pop off all stack data until we reach the return address: *)
            ArithBin (Add, Reg Esp, Immed
              (4 * wordsBelowRetAddr (argsRetAddrFrame, frame)));

            (* return, popping off the incoming arguments: *)
            Retn (Some (4 * (retPopWords argsRetAddrFrame)))
          ]
        ]
  | TcNull s ->
      let c = structCon env s in
      InsList [ Mov (Reg Eax,(Tag 0,[RollTosum c])) ]
  | TcNewStruct (structId, expList) ->
      let (newFrame, argPushCodeRev) = List.fold_left
        (fun (partialFrame, partialCode) (argGamma, argTau, argExp) ->
          (
            addType env (argTau, partialFrame),
            (CodeList
	       [ typedExpToCodeTree
		   ((argGamma, argTau, argExp), partialFrame, varMap);
		 InsList [Push (Reg Eax,[])]
	       ] 
            )
              ::partialCode
          )
        )
        (frame, [])
        expList
      and c = structCon env structId
      and sd = findStruct env structId
      and (fillCode, nBytes) = List.fold_left
        (fun (partialFillCode, nb) _ ->
          (
            (   (Pop (Reg Ebx))
              ::(Mov (Prjr ((Eax, []), nb), (Reg Ebx, [])))
              ::partialFillCode),
            nb + 4
          )
        )
        ([], 0)
        expList
      in
        CodeList
        [
          CodeList (List.rev argPushCodeRev);
          InsList
          [
            Malloc (Eax, nBytes, structMallocArg env sd.sd_def)
          ];
          InsList fillCode;
          (* If the struct is a ?, we must coerce it into the sum.  Then
	   * we must roll it up.
	   *)
          (match sd.sd_option with
	    NeverNull -> 
              InsList [ Mov(Reg Eax,(Reg Eax,[Roll c])) ]
	  | OptionNull ->
              InsList [ Mov(Reg Eax,(Reg Eax,[RollTosum c])) ])
        ]
  | TcStructMember (te1, memberId) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
        let (_, upTau1, _) = te1 in
        (match upTau1 with
          StructType structId ->
            let sd = findStruct env structId in
            let offset = fieldOffset sd.sd_def memberId in
	    CodeList
	      [ code1;
		InsList
	          (* check for null possibly *)
		  (match sd.sd_option with
		    NeverNull ->
		      [ Mov (Reg Eax, (Prjr ((Eax, [Unroll]), offset), []))
		      ]
		  | OptionNull ->
		      [ Mov(Reg Eax, (Reg Eax, [Unroll]));
			Btagi(Eax,0,(null_exn_label,funCallCoercion frame),Eq);
			Mov(Reg Eax, (Prjr ((Eax,[Fromsum]), offset), []))
		      ])
	      ]
        | _ -> raise (TypeError "internal bug"))
  | TcAssignStructMember (te1, memberId, te2) ->
      let code1 = typedExpToCodeTree (te1, frame, varMap) in
      let (_, upTau1, _) = te1 in
      (match upTau1 with
        StructType structId ->
          let sd = findStruct env structId in
          let offset = (fieldOffset sd.sd_def memberId)
          and code2 = typedExpToCodeTree
              (te2, addType env (upTau1, frame), varMap) in
	  CodeList
	    [ code1;
	      InsList [Push (Reg Eax,[])];
              code2;
              InsList
	        (* check for null possibly *)
	      	(match sd.sd_option with
		  NeverNull ->
                    [ Pop (Reg Ebx);
		      Mov (Prjr ((Ebx, [Unroll]), offset), (Reg Eax, []))
                    ]
	      	|	OptionNull ->
		    [ Pop (Reg Ebx);
		      Mov (Reg Ebx,(Reg Ebx,[Unroll]));
		      Btagi (Ebx,0,(null_exn_label,funCallCoercion frame),Eq);
		      Mov (Prjr ((Ebx, [Fromsum]), offset), (Reg Eax,[]))
		    ])
	    ] 
      | _ -> raise (TypeError "internal bug"))
  | TcSubscript(te1,te2) ->
      let code1 = typedExpToCodeTree (te1,frame,varMap) in
      let (_,argTau,_) = te1 in
      let code2 = 
        typedExpToCodeTree (te2,addType env (argTau,frame),varMap) in
      begin
	CodeList
	[ code1;
	  InsList
	  [ Push(Reg Eax,[])
	  ] ;
	  code2;
	  InsList
	  [ Pop(Reg Ecx);
	    Asub(Byte4,Eax,Reg Ecx,Eax)
	  ] 
        ]	
      end
  | TcArraySize te1 -> 
      let code = typedExpToCodeTree (te1,frame,varMap) in
      begin
	CodeList
	[ code;
	  InsList
	  [ Alen(Byte4,Eax,Reg Eax) ]
	] 
      end
  | TcArrayAssign(te1,te2,te3) -> 
      let code1 = typedExpToCodeTree (te1,frame,varMap) in
      let (_,tau1,_) = te1 in
      let frame2 = addType env (tau1,frame) in
      let code2 = typedExpToCodeTree (te2,frame2,varMap) in
      let frame3 = addInt frame2 in
      let code3 = typedExpToCodeTree (te3,frame3,varMap) in
      begin
	CodeList
	[ code1;
	  InsList [ Push(Reg Eax,[]) ];
	  code2;
	  InsList [ Push(Reg Eax,[]) ];
	  code3;
	  InsList 
	  [ Pop(Reg Ecx);
	    Pop(Reg Ebx);
	    Aupd(Byte4,Reg Ebx,Ecx,Eax)
	  ]
	] 
      end

in typedExpToCodeTree

let val_exports = ref [];;
let con_exports = ref [];;
let add_val_export l c = val_exports := (l,c) :: !val_exports;;
let add_con_export l k = con_exports := (l,k,AbsCon) :: !con_exports;;
let get_val_exports () =
  let ve = !val_exports in val_exports := []; Array.of_list (List.rev ve);;
let get_con_exports () =
  let ce = !con_exports in con_exports := []; Array.of_list (List.rev ce);;

let compileTopDecl (env, TcFunDecl (static, funId, (retTau, declList), te)) =
  let (_, funLabel) = findFun env funId in
  let returnRegState =
    Dict.inserts
      emptyRegState
      ((match retTau with
      	VoidType -> []
      | _ -> [(Eax, typeCon env retTau)]) @
       [(Esp, (capp (cprim PCstackptr, callerStackCvar)))]) in
  let (argsFrame, argsVarMap) =
    List.fold_left
      (fun (partialArgsFrame, partialArgsVarMap) (Decl (id, tau)) ->
      	(addType env (tau, partialArgsFrame)),
        addVar (id, tau, partialArgsFrame, partialArgsVarMap))
      ((callerStackCvar, 0), [])
      declList in
  let frame = addWordCon (ccode (returnRegState), argsFrame) in
  let varMap = argsVarMap in
  let ls = labelState frame in
  if static then add_val_export funLabel ls;
  CodeList
    [ CodeLabel (funLabel, labelState frame);
      funBodyToCodeTree
        (env, frame) (te, frame, varMap)
    ]
;;

let mainFunId = "main";;
let mainLabel = id_of_string "__main";;

let compileToCode (env, tcDeclList, has_main) =
  let codeList =
    List.map (fun tcTopDecl -> compileTopDecl (env, tcTopDecl)) tcDeclList in
  let codeTree = 
    CodeList 
      ((if has_main then
	[ CodeLabel (tal_main_label,tal_main_con);
	  InsList [ Call (Addr mainLabel,[Tapp cempty]);
		    Push (Immed 0,[]);
		    Call (Addr exit_label,[Tapp cempty]);
		    ArithBin (Add,Reg Esp,Immed 4);
		    Jmp (Addr inf_loop_label,[Tapp cempty]) ]
	]
      else
	[]) @
       [ CodeLabel (inf_loop_label,inf_loop_con);
	 InsList [ Jmp (Addr inf_loop_label,
			[Tapp (cvar inf_loop_alpha)]) ];
	 CodeLabel (null_exn_label,null_exn_con);
         InsList [ Push (Immed 1,[]);
		   Call (Addr exit_label,[Tapp (cvar alpha)]);
		   ArithBin (Add,Reg Esp,Immed 4);
                   Jmp (Addr inf_loop_label,[Tapp (cvar alpha)])
		 ] ] @
       codeList) in
  codeTreeToCode codeTree
;;

let string_list_to_data sl = 
  List.map
    (fun (l,s) ->
      (l,
       Some (cprim (PCbytearray Byte1)),
       ([Dtag (String.length s,[]); Dbytes s],[Toarray])))
    sl
;;

let compileExterns env externs =
  let cons = ref [] in
  let vals = ref [] in
  let f extern =
    match extern with
      TcExternFun (rt, id, ats) ->
  	let returnRegState =
	  Dict.inserts
	    emptyRegState
	    ((match rt with
      	      VoidType -> []
	    | _ -> [(Eax, typeCon env rt)]) @
	     [(Esp, (capp (cprim PCstackptr, callerStackCvar)))])
  	and argsFrame =
	  List.fold_left
	    (fun partialArgsFrame tau -> addType env (tau, partialArgsFrame))
	    (callerStackCvar, 0)
	    ats in
  	let frame = addWordCon (ccode (returnRegState), argsFrame) in
  	let ls = labelState frame in
	vals := (id_of_string ("__"^id), ls) :: !vals
    | TcExternStruct id ->
 	cons := (id_of_string id,K4byte,AbsCon) :: !cons in
  List.iter f externs;
  { int_includes = [||];
    int_cons = Array.of_list (List.rev !cons);
    int_vals = Array.of_list (List.rev !vals)
  }
;;

(* For a struct {t1,...,tn}:
 *   For a NeverNull, we define:
 *     l = *[|t1|^wi,...,|tn|^wi]
 *   For an OptionNull, we define:
 *     l = +[S(0),*[|t1|^wi,...,|tn|^wi]]
 *)

let mk_con_blks env =
  let aux (id, sd) =
    if sd.sd_static then add_con_export sd.sd_lab K4byte;
    let tuple_fields = 
      List.map
 	(fun (Decl (_,t)) -> (typeCon env t, ReadWrite, Init))
 	sd.sd_def in
    let body =
      match sd.sd_option with
	NeverNull -> cprod tuple_fields
      | OptionNull ->
	  csum {sum_tags=[0]; sum_vt = Some(Tuple tuple_fields)} in
    (sd.sd_lab, K4byte, body) in
  List.map aux env.env_sdefs
;;

let mk_env env =
  let conv_fun (id, funDecl) = (id, (funDecl, id_of_string ("__"^id))) in
  let funs = List.map conv_fun env.Poptype.env_fun in
  let tids =
    List.map (fun id -> (id, (id_of_string id))) env.Poptype.env_tids in 
  let conv_struct (id, (static, st, structDecl)) =
    let l = List.assoc id tids in
    (id, { sd_static=static;
	   sd_def=structDecl;
	   sd_option=st;
	   sd_lab=l }) in
  let sdefs = List.map conv_struct env.Poptype.env_sdefs in
  { env_fun=funs; env_tids=tids; env_sdefs=sdefs }
;;

let compileToModule impfile expfile (env, externs, tcDeclList) =
  let env = mk_env env in
  let has_main =
    try
      List.assoc mainFunId env.env_fun; true
    with Not_found -> false in
  let con_blks = mk_con_blks env in
  let cb = compileToCode (env, tcDeclList, has_main) in
  let sl = Array.of_list (string_list_to_data (!string_list)) in
  string_list := [];
  let imp =
    { imports = [|"tal.tali";"stdlib.tali";impfile|];
      exports = if has_main then [|"tal_prog.tali";expfile|] else [|expfile|];
      con_blocks = Array.of_list con_blks;
      code_blocks = cb;
      data_blocks = sl
    } in
  let impint = compileExterns env externs in
  let expint =
    { int_includes = [||];
      int_cons = get_con_exports ();
      int_vals = get_val_exports ()
    } in
  (impint,expint,imp)
;;
    
(* EOF: popcompile.ml *)
