
(* Simple, easy optimizations for IL
   After closure conversion and lifting, the IL is pretty easy to
   optimize in simple ways, being essentially SSA.

   Copy propagation is easy
   i.e. replacing "let x = e in e'" with "e[y/x]" - since e
   is either a constant integer/ function or is only used once, 
   making this substitution does not adversely affect anything.
   Must ensure that "e" is NOT a function; we don't want to undo the effect
   of lifting!
   
   Tricky thing: In general cannot substitute away something of type Int, 
   because it may in reality be a function application (say, applying Free)
   that returns an int, so it should nto be optimized away.
   So, we really should only propagate let bindings to:
   * variables
   * integer constants
   * pairs (linear)
   * packs (linear)
   * lets, letpairs, letunpacks of the above, recursively. (But we handle these
     instead by denesting lets, see denest).

   TODO:
   Although sometimes it is safe to propagate other things, we can't tell this
   without type information, which seems like another good reason to make
   type information available at this level (we have to anyway to translate to LTAL).
   

   Another easy optimization is aggressive pattern matching:
   if we see "let <x,y> = <e1,e2> in e'" then it's safe to 
   replace it with "e'[e1/x,e2/y]" for the same reason.
   Similarly for pack and unpack.

   Here, we don't have to worry about losing side-effectful things because 
   the variables x,y must be treated linearly.
*)

val copyprop : Il.exp -> Il.exp

(* Denesting: if we see a let in the argument to another let, 
   it's OK to lift the inside let out to before the outside let, 
   since variable names are unique... *)

val denest : Il.exp -> Il.exp


(* Dead code elimination.  If a variable is introduced by a "let"
   but never used,
   * and its initializer has no side effects, *
   then we can get rid of the binding and initialization altogether.
   Since linearly initialized expressions are always used, and
   we can't tell if arbitrary expressions (i.e., applications) of type
   int have side effects, we can really only eliminate values, like functions
   or arithmetic expressions.
*)

val deadcode : Il.exp -> Il.exp

(* Copy-free elimination.  Detect when a variable that was generated by
   a copy is freed.  Currently inefficient (quadratic) *)
val cfelim : Il.exp -> Il.exp

(* Function inlining:  Want to avoid as many explicit calls to copy, free, 
 * and app as possible, by inlining them whenever we have an explicit
 * call to a function label.  Obviously, we can't avoid the call when 
 * we call an arbitrary closure.  But by inlining and then
 * re-optimizing, ideally we can eventually figure these things out
 * for code that's static.  
 * This inlining is aggressive: we try to replace any function call
 * whose target is known.
 * The implementation is dependent on the variable names all being 
 * distinct and no functions being recursive.  Recursive
functions will have to be treated specially.
 *)

val inline : Il.exp -> Il.exp

