/***********************************************************************/
/*                                                                     */
/*                           Objective Caml                            */
/*                                                                     */
/*            Xavier Leroy, projet Cristal, INRIA Rocquencourt         */
/*                                                                     */
/*  Copyright 1996 Institut National de Recherche en Informatique et   */
/*  Automatique.  Distributed only by permission.                      */
/*                                                                     */
/***********************************************************************/

/* $Id: debugger.c,v 1.14 1997/09/02 12:53:57 xleroy Exp $ */

/* Interface with the debugger */

#include <string.h>

#include "config.h"
#include "debugger.h"
#include "fail.h"
#include "fix_code.h"
#include "instruct.h"
#include "intext.h"
#include "io.h"
#include "misc.h"
#include "mlvalues.h"
#include "stacks.h"
#include "sys.h"

int debugger_in_use = 0;
unsigned long event_count;

#if !defined(HAS_SOCKETS) || defined(_WIN32)

void debugger_init(void)
{
}

void debugger(enum event_kind event)
{
}

#else

#ifdef HAS_UNISTD
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

//_//TLS int sock_domain;         /* Socket domain for the debugger */
//_// exception since we cannot construct this in OCAMLDRV.h
TLS union {                  /* Socket address for the debugger */
  struct sockaddr s_gen;
  struct sockaddr_un s_unix;
  struct sockaddr_in s_inet;
} sock_addr[MAXTHREAD];
//_//TLS int sock_addr_len;       /* Length of sock_addr */

//_//TLS int dbg_socket = -1;     /* The socket connected to the debugger */
//_//TLS struct channel * dbg_in; /* Input channel on the socket */
//_//TLS struct channel * dbg_out;/* Output channel on the socket */

//TLS static int sock_domain;         /* Socket domain for the debugger */
//TLS static union {                  /* Socket address for the debugger */
//  struct sockaddr s_gen;
//  struct sockaddr_un s_unix;
//  struct sockaddr_in s_inet;
//} sock_addr;
//TLS static int sock_addr_len;       /* Length of sock_addr */

//TLS static int dbg_socket = -1;     /* The socket connected to the debugger */
//TLS static struct channel * dbg_in; /* Input channel on the socket */
//TLS static struct channel * dbg_out;/* Output channel on the socket */

static void open_connection(void)
{
int threadId = getThreadId((PVOID)getPeThread());
  dbg_socket = socket(tls[threadId].sock_domain, SOCK_STREAM, 0);
  if (tls[threadId].dbg_socket == -1 ||
      connect(tls[threadId].dbg_socket, &(sock_addr[threadId].s_gen), tls[threadId].sock_addr_len) == -1)
    fatal_error("cannot connect to debugger");
  tls[threadId].dbg_in = open_descriptor(tls[threadId].dbg_socket);
  tls[threadId].dbg_out = open_descriptor(tls[threadId].dbg_socket);
  if (!debugger_in_use) putword(tls[threadId].dbg_out, -1); /* first connection */
  putword(tls[threadId].dbg_out, getpid());
  flush(tls[threadId].dbg_out);
}

static void close_connection(void)
{
int threadId = getThreadId((PVOID)getPeThread());
  close_channel(tls[threadId].dbg_in);
  close_channel(tls[threadId].dbg_out);
  tls[threadId].dbg_socket = -1;              /* was closed by close_channel */
}

void debugger_init(void)
{
  char * address;
  char * port, * p;
  struct hostent * host;
  int n;
int threadId = getThreadId((PVOID)getPeThread());

  address = getenv("CAML_DEBUG_SOCKET");
  if (address == NULL) return;

  /* Parse the address */
  port = NULL;
  for (p = address; *p != 0; p++) {
    if (*p == ':') { *p = 0; port = p+1; break; }
  }
  if (port == NULL) {
    /* Unix domain */
    tls[threadId].sock_domain = PF_UNIX;
    sock_addr[threadId].s_unix.sun_family = AF_UNIX;
    strncpy(sock_addr[threadId].s_unix.sun_path, address,
            sizeof(sock_addr[threadId].s_unix.sun_path));
    tls[threadId].sock_addr_len = 
      ((char *)&(sock_addr[threadId].s_unix.sun_path) - (char *)&(sock_addr[threadId].s_unix))
        + strlen(address);
  } else {
    /* Internet domain */
    tls[threadId].sock_domain = PF_INET;
    for (p = (char *) &(sock_addr[threadId].s_inet), n = sizeof(sock_addr[threadId].s_inet);
         n > 0; n--) *p++ = 0;
    sock_addr[threadId].s_inet.sin_family = AF_INET;
    sock_addr[threadId].s_inet.sin_addr.s_addr = inet_addr(address);
    if (sock_addr[threadId].s_inet.sin_addr.s_addr == -1) {
      host = gethostbyname(address);
      if (host == NULL)
        fatal_error_arg("Unknown debugging host %s\n", address);
      bcopy(host->h_addr, &(sock_addr[threadId].s_inet.sin_addr), host->h_length);
    }
    sock_addr[threadId].s_inet.sin_port = htons(atoi(port));
    tls[threadId].sock_addr_len = sizeof(sock_addr[threadId].s_inet);
  }
  open_connection();
  debugger_in_use = 1;
  tls[threadId].trap_barrier = tls[threadId].stack_high;
}

static value getval(struct channel *chan)
{
  value res;
  if (really_getblock(chan, (char *) &res, sizeof(res)) == 0)
    raise_end_of_file(); /* Bad, but consistent with getword */
  return res;
}

static void putval(struct channel *chan, value val)
{
  really_putblock(chan, (char *) &val, sizeof(val));
}

static void safe_output_value(struct channel *chan, value val)
{
  struct longjmp_buffer raise_buf, * saved_external_raise;

  /* Catch exceptions raised by output_val */
  saved_external_raise = external_raise[threadId];
  if (sigsetjmp(raise_buf.buf, 1) == 0) {
    external_raise[threadId] = &raise_buf;
    output_val(chan, val, Val_unit);
  } else {
    /* Send wrong magic number, will cause input_value to fail */
    really_putblock(chan, "\000\000\000\000", 4);
  }
  external_raise[threadId] = saved_external_raise;
}

#define Pc(sp) ((code_t)(sp[0]))
#define Env(sp) (sp[1])
#define Extra_args(sp) (Long_val((sp[2])))
#define Locals(sp) (sp + 3)

void debugger(enum event_kind event)
{
int threadId = getThreadId((PVOID)getPeThread());
  int frame_number;
  value * frame;
  long i, pos;
  value val;

  if (tls[threadId].dbg_socket == -1) return;  /* Not connected to a debugger. */

  /* Reset current frame */
  frame_number = 0;
  frame = tls[threadId].extern_sp + 1;

  /* Report the event to the debugger */
  switch(event) {
  case PROGRAM_START:           /* Nothing to report */
    goto command_loop;
  case EVENT_COUNT:
    putch(tls[threadId].dbg_out, REP_EVENT);
    break;
  case BREAKPOINT:
    putch(tls[threadId].dbg_out, REP_BREAKPOINT);
    break;
  case PROGRAM_EXIT:
    putch(tls[threadId].dbg_out, REP_EXITED);
    break;
  case TRAP_BARRIER:
    putch(tls[threadId].dbg_out, REP_TRAP);
    break;
  case UNCAUGHT_EXC:
    putch(tls[threadId].dbg_out, REP_UNCAUGHT_EXC);
    break;
  }
  putword(tls[threadId].dbg_out, event_count);
  if (event == EVENT_COUNT || event == BREAKPOINT) {
    putword(tls[threadId].dbg_out, tls[threadId].stack_high - frame);
    putword(tls[threadId].dbg_out, (Pc(frame) - tls[threadId].start_code) * sizeof(opcode_t));
  } else {
    /* No PC and no stack frame associated with other events */
    putword(tls[threadId].dbg_out, 0);
    putword(tls[threadId].dbg_out, 0);
  }
  flush(tls[threadId].dbg_out);

 command_loop:
  
  /* Read and execute the commands sent by the debugger */
  while(1) {
    switch(getch(tls[threadId].dbg_in)) {
    case REQ_SET_EVENT:
      pos = getword(tls[threadId].dbg_in);
      Assert(pos >= 0 && pos < tls[threadId].code_size);
      set_instruction(tls[threadId].start_code + pos / sizeof(opcode_t), EVENT);
      break;
    case REQ_SET_BREAKPOINT:
      pos = getword(tls[threadId].dbg_in);
      Assert(pos >= 0 && pos < tls[threadId].code_size);
      set_instruction(tls[threadId].start_code + pos / sizeof(opcode_t), BREAK);
      break;
    case REQ_RESET_INSTR:
      pos = getword(tls[threadId].dbg_in);
      Assert(pos >= 0 && pos < tls[threadId].code_size);
      pos = pos / sizeof(opcode_t);
      set_instruction(tls[threadId].start_code + pos, tls[threadId].saved_code[pos]);
      break;
    case REQ_CHECKPOINT:
      i = fork();
      if (i == 0) {
        close_connection();     /* Close parent connection. */
        open_connection();      /* Open new connection with debugger */
      } else {
        putword(tls[threadId].dbg_out, i);
        flush(tls[threadId].dbg_out);
      }
      break;
    case REQ_GO:
      event_count = getword(tls[threadId].dbg_in);
      return;
    case REQ_STOP:
      mt_exit(0);
      break;
    case REQ_WAIT:
      wait(NULL);
      break;
    case REQ_INITIAL_FRAME:
      frame = tls[threadId].extern_sp + 1;
      /* Fall through */
    case REQ_GET_FRAME:
      putword(tls[threadId].dbg_out, tls[threadId].stack_high - frame);
      putword(tls[threadId].dbg_out, (Pc(frame) - tls[threadId].start_code) * sizeof(opcode_t));
      flush(tls[threadId].dbg_out);
      break;
    case REQ_SET_FRAME:
      i = getword(tls[threadId].dbg_in);
      frame = tls[threadId].stack_high - i;
      break;
    case REQ_UP_FRAME:
      i = getword(tls[threadId].dbg_in);
      if (frame + Extra_args(frame) + i + 3 >= tls[threadId].stack_high) {
        putword(tls[threadId].dbg_out, -1);
      } else {
        frame += Extra_args(frame) + i + 3;
        putword(tls[threadId].dbg_out, tls[threadId].stack_high - frame);
        putword(tls[threadId].dbg_out, (Pc(frame) - tls[threadId].start_code) * sizeof(opcode_t));
      }
      flush(tls[threadId].dbg_out);
      break;
    case REQ_SET_TRAP_BARRIER:
      i = getword(tls[threadId].dbg_in);
      tls[threadId].trap_barrier = tls[threadId].stack_high - i;
      break;
    case REQ_GET_LOCAL:
      i = getword(tls[threadId].dbg_in);
      putval(tls[threadId].dbg_out, Locals(frame)[i]);
      flush(tls[threadId].dbg_out);
      break;
    case REQ_GET_ENVIRONMENT:
      i = getword(tls[threadId].dbg_in);
      putval(tls[threadId].dbg_out, Field(Env(frame), i));
      flush(tls[threadId].dbg_out);
      break;
    case REQ_GET_GLOBAL:
      i = getword(tls[threadId].dbg_in);
      putval(tls[threadId].dbg_out, Field(tls[threadId].global_data, i));
      flush(tls[threadId].dbg_out);
      break;
    case REQ_GET_ACCU:
      putval(tls[threadId].dbg_out, *tls[threadId].extern_sp);
      flush(tls[threadId].dbg_out);
      break;
    case REQ_GET_HEADER:
      val = getval(tls[threadId].dbg_in);
      putword(tls[threadId].dbg_out, Hd_val(val));
      flush(tls[threadId].dbg_out);
      break;
    case REQ_GET_FIELD:
      val = getval(tls[threadId].dbg_in);
      i = getword(tls[threadId].dbg_in);
      putval(tls[threadId].dbg_out, Field(val, i));
      flush(tls[threadId].dbg_out);
      break;
    case REQ_MARSHAL_OBJ:
      val = getval(tls[threadId].dbg_in);
      safe_output_value(tls[threadId].dbg_out, val);
      flush(tls[threadId].dbg_out);
      break;
    case REQ_GET_CLOSURE_CODE:
      val = getval(tls[threadId].dbg_in);
      putword(tls[threadId].dbg_out, (Code_val(val) - tls[threadId].start_code) * sizeof(opcode_t));
      flush(tls[threadId].dbg_out);
      break;
    }
  }
}

#endif
