/***********************************************************************/
/*                                                                     */
/*                           Objective Caml                            */
/*                                                                     */
/*         Xavier Leroy and Damien Doligez, INRIA Rocquencourt         */
/*                                                                     */
/*  Copyright 1996 Institut National de Recherche en Informatique et   */
/*  Automatique.  Distributed only by permission.                      */
/*                                                                     */
/***********************************************************************/

/* $Id: startup.c,v 1.23 1997/12/09 13:53:13 vouillon Exp $ */

/* Start-up code */

#include "multithreads.h"
#include "ocamldrv.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include "config.h"
#ifdef HAS_UNISTD
#include <unistd.h>
#endif
#include "alloc.h"
#include "debugger.h"
#include "exec.h"
#include "fail.h"
#include "fix_code.h"
#include "gc_ctrl.h"
#include "interp.h"
#include "intext.h"
#include "io.h"
#include "memory.h"
#include "minor_gc.h"
#include "misc.h"
#include "mlvalues.h"
#include "prims.h"
#include "stacks.h"
#include "sys.h"

#ifndef O_BINARY
#define O_BINARY 0
#endif

#ifndef SEEK_END
#define SEEK_END 2
#endif

//_//TLS header_t atom_table[256];

// thread clean up memory usage
void thread_cleanup()
{
int threadId = getThreadId(getPeThread());

//	DbgPrint("\nthread_cleanup called!\n");
	// free start_code buffer
//	DbgPrint("start_code = %x\n", start_code);
//	DbgPrint("saved_code = %x\n", saved_code);
	if (tls[threadId].start_code) { stat_free(tls[threadId].start_code); tls[threadId].start_code = 0; }
	if (tls[threadId].saved_code) { stat_free(tls[threadId].saved_code); tls[threadId].saved_code = 0; }
	// free minor heap in minor_gc
//	DbgPrint("young_start = %x\n", young_start);
//	DbgPrint("ref_table = %x\n", ref_table);
//	DbgPrint("heap_start = %x\n", (((unsigned)heap_start) & 0xfffff000) - 0x1000 + 8);
//	DbgPrint("page_table = %x\n", page_table);
//	DbgPrint("gray_vals = %x\n", gray_vals);
//	DbgPrint("stack_low = %x\n", stack_low);

	// do not change this it will be automatically clean
//	if (young_start) { stat_free(young_start); young_start = 0; }
//	if (ref_table) { stat_free(ref_table); ref_table = 0; }

	// free global structure in roots.c
//	remove_global_root(NULL);
	// free stack
	if (tls[threadId].stack_low) { stat_free(tls[threadId].stack_low); tls[threadId].stack_low = 0; }

// free major heap in major_gc
	if (tls[threadId].gray_vals) { stat_free(tls[threadId].gray_vals); tls[threadId].gray_vals = 0; }
	if (tls[threadId].page_table) { stat_free(tls[threadId].page_table); tls[threadId].page_table = 0; }
	// this heap is cleant next cycle.
//	if (heap_start) { stat_free((int *)((((unsigned)heap_start) & 0xfffff000) - 0x1000 + 8)); }
}

/* Initialize the atom table */

static void init_atoms(void)
{
  int i;
int threadId = getThreadId(getPeThread());
  for(i = 0; i < 256; i++) tls[threadId].atom_table[i] = Make_header(0, i, White);
}

/* Read the trailer of a bytecode file */

static unsigned long read_size(char * ptr)
{
  unsigned char * p = (unsigned char *) ptr;

  return ((unsigned long) p[0] << 24) + ((unsigned long) p[1] << 16) +
         ((unsigned long) p[2] << 8) + p[3];
}

#define FILE_NOT_FOUND (-1)
#define TRUNCATED_FILE (-2)
#define BAD_MAGIC_NUM (-3)

static int read_trailer(
						IN OUT PVOID IoBuffer, 
						struct exec_trailer *trail
						)
{
	POCAML_BYTECODE_INFO pobi = (POCAML_BYTECODE_INFO) IoBuffer;
	char buffer[TRAILER_SIZE + 1];
	
	if (pobi->ByteCodeSize == 0) return TRUNCATED_FILE;
	
	memcpy (buffer, pobi->KByteCode + pobi->ByteCodeSize - TRAILER_SIZE, TRAILER_SIZE);
	
	trail->code_size = read_size(buffer);
	trail->prim_size = read_size(buffer + 4);
	trail->data_size = read_size(buffer + 8);
	trail->symbol_size = read_size(buffer + 12);
	trail->debug_size = read_size(buffer + 16);

//DbgPrint("trail->code_size = %d\ntrail->prim_size = %d\ntrail->data_size = %d\ntrail->symbol_size = %d\ntrail->debug_size = %d\n", 
//trail->code_size, trail->prim_size, trail->data_size, trail->symbol_size, trail->debug_size);	
	// magic number is "Caml1999x002"
	if (strncmp(buffer + 20, EXEC_MAGIC, 12) == 0)
		return 0;
	else
		return BAD_MAGIC_NUM; // -3 in case first arg is ocamlrun
}

/*
static int read_trailer(int fd, struct exec_trailer *trail)
{
  char buffer[TRAILER_SIZE];
  lseek(fd, (long) -TRAILER_SIZE, SEEK_END);
  if (read(fd, buffer, TRAILER_SIZE) < TRAILER_SIZE) return TRUNCATED_FILE;

  trail->code_size = read_size(buffer);
  trail->prim_size = read_size(buffer + 4);
  trail->data_size = read_size(buffer + 8);
  trail->symbol_size = read_size(buffer + 12);
  trail->debug_size = read_size(buffer + 16);

  // magic number is "Caml1999x002"
  if (strncmp(buffer + 20, EXEC_MAGIC, 12) == 0)
    return 0;
  else
    return BAD_MAGIC_NUM; // -3 in case first arg is ocamlrun
}
*/

static int attempt_open(char **name, struct exec_trailer *trail, int do_open_script)
{
  char * truename;
  int fd;
  int err;
  char buf [2];

  truename = searchpath(*name);
  if (truename == 0) truename = *name; else *name = truename;
  fd = open(truename, O_RDONLY | O_BINARY);
  if (fd == -1) return FILE_NOT_FOUND;
  if (!do_open_script){
    err = read (fd, buf, 2);
    if (err < 2) { close(fd); return TRUNCATED_FILE; }
    if (buf [0] == '#' && buf [1] == '!') { close(fd); return BAD_MAGIC_NUM; }
  }
//  err = read_trailer(fd, trail); // return (-3) if bad_magic_num : not ocaml byte code
  if (err != 0) { close(fd); return err; }
  return fd;
}

/* Check the primitives used by the bytecode file against the table of
   primitives linked in this interpreter */

static void check_primitives(void * pKByteCode, int prim_size)
{
  char * prims;
  char * p;
  int idx;
  prims = stat_alloc(prim_size);
	memcpy (prims, pKByteCode, prim_size);
//  if (read(fd, prims, prim_size) != prim_size) 
//  { stat_free(prims); fatal_error("Fatal error: cannot read primitive table\n"); }
  /* prims contains 0-terminated strings, concatenated. */
  for (p = prims, idx = 0;
       p < prims + prim_size;
       p = p + strlen(p) + 1, idx++) {
    if (names_of_cprim[idx] == NULL ||
        strcmp(p, names_of_cprim[idx]) != 0)
      { stat_free(prims); fatal_error_arg("Fatal error: this bytecode file cannot run on this bytecode interpreter\nMismatch on primitive `%s'\n", p); }
  }
  stat_free(prims);
}

/* Invocation of camlrun: 4 cases.

   1.  runtime + bytecode
       user types:  camlrun [options] bytecode args...
       arguments:  camlrun [options] bytecode args...

   2.  bytecode script
       user types:  bytecode args...
   2a  (kernel 1) arguments:  camlrun ./bytecode args...
   2b  (kernel 2) arguments:  bytecode bytecode args...

   3.  concatenated runtime and bytecode
       user types:  composite args...
       arguments:  composite args...

Algorithm:
  1-  If argument 0 is a valid byte-code file that does not start with #!,
      then we are in case 3 and we pass the same command line to the
      Caml Light program.
  2-  In all other cases, we parse the command line as:
        (whatever) [options] bytecode args...
      and we strip "(whatever) [options]" from the command line.

*/

/* Configuration parameters and flags */

//_//TLS unsigned long verbose_init = 0;
//_//TLS unsigned long percent_free_init = Percent_free_def;
//_//TLS unsigned long max_percent_free_init = Max_percent_free_def;
//_//TLS unsigned long minor_heap_init = Minor_heap_def;
//_//TLS unsigned long heap_chunk_init = Heap_chunk_def;
//_//TLS unsigned long heap_size_init = Init_heap_def;
//_//TLS unsigned long max_stack_init = Max_stack_def;
//TLS static unsigned long verbose_init = 0;
static unsigned long percent_free_init = Percent_free_def;
static unsigned long max_percent_free_init = Max_percent_free_def;
static unsigned long minor_heap_init = Minor_heap_def;
static unsigned long heap_chunk_init = Heap_chunk_def;
static unsigned long heap_size_init = Init_heap_def;
static unsigned long max_stack_init = Max_stack_def;
//_//extern int trace_flag;

/* Parse options on the command line */

static int parse_command_line(char **argv)
{
  int i;
int threadId = getThreadId(getPeThread());

  for(i = 1; argv[i] != NULL && argv[i][0] == '-'; i++) {
    switch(argv[i][1]) {
#ifdef DEBUG
    case 't':
      tls[threadId].trace_flag = 1;
      break;
#endif
    case 'v':
      tls[threadId].verbose_init = 1;
      break;
    default:
      fatal_error_arg("Unknown option %s.\n", argv[i]);
    }
  }
  return i;
}

/* Parse the CAMLRUNPARAM variable */
/* The option letter for each runtime option is the first letter of the
   last word of the ML name of the option (see [stdlib/gc.mli]).
   Except for l (maximum stack size) and h (initial heap size).
*/

static void scanmult (char *opt, long unsigned int *var)
{
#ifndef KAMEEL
  char mult = ' ';
  sscanf (opt, "=%lu%c", var, &mult);
  if (mult == 'k') *var = *var * 1024;
  if (mult == 'M') *var = *var * (1024 * 1024);
  if (mult == 'G') *var = *var * (1024 * 1024 * 1024);
#endif // KAMEEL
}

static void parse_camlrunparam(void)
{
#ifndef KAMEEL
  char *opt = getenv ("CAMLRUNPARAM");

  if (opt != NULL){
    while (*opt != '\0'){
      switch (*opt++){
      case 's': scanmult (opt, &minor_heap_init); break;
      case 'i': scanmult (opt, &heap_chunk_init); break;
      case 'h': scanmult (opt, &heap_size_init); break;
      case 'l': scanmult (opt, &max_stack_init); break;
      case 'o': scanmult (opt, &percent_free_init); break;
      case 'O': scanmult (opt, &max_percent_free_init); break;
      case 'v': scanmult (opt, &verbose_init); break;
      }
    }
  }
#endif // KAMEEL
}

extern void init_ieee_floats (void);

/* Main entry point when loading code from a file */





//void caml_main(char **argv)
int
caml_main(
    IN PVOID      IoBuffer
    )
{
	__try 
	{
		POCAML_BYTECODE_INFO pobi = (POCAML_BYTECODE_INFO) IoBuffer;
		int fd;
		struct exec_trailer trail;
		int pos;
		struct longjmp_buffer raise_buf;
		struct channel * chan;
		int threadId = getThreadId(getPeThread());

		//printf("argv[0]=%s, argv[1]=%s, argv[2]=%s, argv[3]=%s, argv[4]=%s\n", 
		//	   argv[0], argv[1], argv[2], argv[3], argv[4]);
		
		/* Machine-dependent initialization of the floating-point hardware
		so that it behaves as much as possible as specified in IEEE */
		init_ieee_floats();
		/* Set up a catch-all exception handler */
		if (sigsetjmp(raise_buf.buf, 1) == 0) {
			external_raise[threadId] = &raise_buf;
			/* Determine options and position of bytecode file */
#ifdef DEBUG
			tls[threadId].verbose_init = 1;
#endif
			parse_camlrunparam();
			pos = 0;
			
			switch(read_trailer(IoBuffer, &trail)) {
			case TRUNCATED_FILE:
				DbgPrint("Fatal error: the file is truncated.\n");
				break;
			case BAD_MAGIC_NUM:
				DbgPrint("Fatal error: the file is not a bytecode executable file\n");
				break;
			}

//DbgPrint("trail.code_size = %d\ntrail.prim_size = %d\ntrail.data_size = %d\ntrail.symbol_size = %d\ntrail.debug_size = %d\n", 
//trail.code_size, trail.prim_size, trail.data_size, trail.symbol_size, trail.debug_size);	
			/* Initialize the abstract machine */
			init_gc (minor_heap_init, heap_size_init, heap_chunk_init,
				percent_free_init, max_percent_free_init, tls[threadId].verbose_init);

			init_stack (max_stack_init);
			init_atoms();

			/* Initialize the interpreter */
			interprete(NULL, 0);
			/* Initialize the debugger, if needed */
			debugger_init();
			/* Load the code */
			load_code(pobi->KByteCode + pobi->ByteCodeSize - 
				(long) (TRAILER_SIZE + trail.code_size + trail.prim_size
				+ trail.data_size + trail.symbol_size
				+ trail.debug_size), trail.code_size);
//			lseek(fd, - (long) (TRAILER_SIZE + trail.code_size + trail.prim_size
//				+ trail.data_size + trail.symbol_size
//				+ trail.debug_size), SEEK_END);
//			load_code(fd, trail.code_size);
			/* Check the primitives */
			check_primitives(pobi->KByteCode + pobi->ByteCodeSize - 
				(long) (TRAILER_SIZE + trail.prim_size
				+ trail.data_size + trail.symbol_size
				+ trail.debug_size), trail.prim_size);

			/* Load the globals */
			chan = open_descriptor((int)(pobi->KByteCode + pobi->ByteCodeSize - 
				(long) (TRAILER_SIZE + trail.data_size + trail.symbol_size
				+ trail.debug_size)));
//			chan = open_descriptor(fd);
			tls[threadId].global_data = input_val(chan);
			close_channel(chan);

			/* Ensure that the globals are in the major heap. */
			oldify(tls[threadId].global_data, &tls[threadId].global_data);

			/* Record the command-line arguments */
//			sys_init(argv + pos);
			/* Execute the program */
			debugger(PROGRAM_START);
			// move to next module
			return trail.code_size;
//			interprete(start_code, trail.code_size);
//			thread_cleanup();
		} else {
			tls[threadId].extern_sp = &tls[threadId].exn_bucket; /* The debugger needs the exception value. */
			debugger(UNCAUGHT_EXC);
			fatal_uncaught_exception(tls[threadId].exn_bucket);
		}
	} // end __try
	__except (thread_cleanup(), EXCEPTION_CONTINUE_SEARCH) { }
	return 0;
}

value 
interpreter(
	IN int code_size
	)
{
int threadId = getThreadId(getPeThread());
	value result = interprete(tls[threadId].start_code, code_size);
	thread_cleanup();
	return result;
}

/* look like it was never used. / suwat ch.
/* Main entry point when code is linked in as initialized data */

void caml_startup_code(code_t code, asize_t code_size, char *data, char **argv)
{
  struct longjmp_buffer raise_buf;
int threadId = getThreadId(getPeThread());

  init_ieee_floats();
#ifdef DEBUG
  tls[threadId].verbose_init = 1;
#endif
  parse_camlrunparam();
  /* Set up a catch-all exception handler */
  if (sigsetjmp(raise_buf.buf, 1) == 0) {
    external_raise[threadId] = &raise_buf;
    /* Initialize the abstract machine */
    init_gc (minor_heap_init, heap_size_init, heap_chunk_init,
             percent_free_init, max_percent_free_init, tls[threadId].verbose_init);
    init_stack (max_stack_init);
    init_atoms();
    /* Initialize the interpreter */
    interprete(NULL, 0);
    /* Load the code */
    tls[threadId].start_code = code;
#ifdef THREADED_CODE
    thread_code(tls[threadId].start_code, code_size);
#endif
    /* Load the globals */
    tls[threadId].global_data = input_val_from_string((value)data, 0);
    /* Ensure that the globals are in the major heap. */
    oldify(tls[threadId].global_data, &tls[threadId].global_data);
    /* Run the code */
    sys_init(argv);
    interprete(tls[threadId].start_code, code_size);
  } else {
    fatal_uncaught_exception(tls[threadId].exn_bucket);
  }
}

void __cdecl caml_startup(void *x)
{
	return ;
}

  
