/***********************************************************************/
/*                                                                     */
/*                           Objective Caml                            */
/*                                                                     */
/*            Xavier Leroy, projet Cristal, INRIA Rocquencourt         */
/*                                                                     */
/*  Copyright 1996 Institut National de Recherche en Informatique et   */
/*  Automatique.  Distributed only by permission.                      */
/*                                                                     */
/***********************************************************************/

/* $Id: fix_code.c,v 1.17 1997/09/02 12:53:58 xleroy Exp $ */

/* Handling of blocks of bytecode (endianness switch, threading). */

#include "multithreads.h"
#include "config.h"
#include "debugger.h"
#include "fix_code.h"
#include "instruct.h"
#include "md5.h"
#include "memory.h"
#include "misc.h"
#include "mlvalues.h"
#include "reverse.h"
#ifdef HAS_UNISTD
#include <unistd.h>
#endif

//_//TLS code_t start_code;
//_//TLS asize_t code_size;
//_//TLS unsigned char * saved_code;
//_//TLS unsigned char code_md5[16];

/* Read the main bytecode block from a file */

void load_code(void * pKByteCode, asize_t len)
{
  int i;
  struct MD5Context ctx;
int threadId = getThreadId(getPeThread());

  tls[threadId].code_size = len;
  tls[threadId].start_code = (code_t) stat_alloc(tls[threadId].code_size);

  memcpy (tls[threadId].start_code, pKByteCode, tls[threadId].code_size);
//  if (read(fd, (char *) start_code, code_size) != code_size)
//    fatal_error("Fatal error: truncated bytecode file.\n");
  MD5Init(&ctx);
  MD5Update(&ctx, (unsigned char *) tls[threadId].start_code, tls[threadId].code_size);
  MD5Final(tls[threadId].code_md5, &ctx);
#ifdef ARCH_BIG_ENDIAN
  fixup_endianness(tls[threadId].start_code, tls[threadId].code_size);
#endif
  if (debugger_in_use) {
    len /= sizeof(opcode_t);
    tls[threadId].saved_code = (unsigned char *) stat_alloc(len);
    for (i = 0; i < len; i++) tls[threadId].saved_code[i] = tls[threadId].start_code[i];
  }
#ifdef THREADED_CODE
  /* Better to thread now than at the beginning of interprete(),
     since the debugger interface needs to perform SET_EVENT requests
     on the code. */
  thread_code(tls[threadId].start_code, tls[threadId].code_size);
#endif
}

/* This code is needed only if the processor is big endian */

#ifdef ARCH_BIG_ENDIAN

void fixup_endianness(code_t code, asize_t len)
{
  code_t p;
  len /= sizeof(opcode_t);
  for (p = code; p < code + len; p++) {
    Reverse_int32(p);
  }
}

#endif

/* This code is needed only if we're using threaded code */

#ifdef THREADED_CODE

//_//TLS char ** instr_table;
//_//TLS char * instr_base;

void thread_code (code_t code, asize_t len)
{
  code_t p;
  int l [STOP + 1];
  int i;
int threadId = getThreadId(getPeThread());
  
  for (i = 0; i <= STOP; i++) {
    l [i] = 0;
  }
  /* Instructions with one operand */
  l[PUSHACC] = l[ACC] = l[POP] = l[ASSIGN] =
  l[PUSHENVACC] = l[ENVACC] = l[PUSH_RETADDR] = l[APPLY] =
  l[APPTERM1] = l[APPTERM2] = l[APPTERM3] = l[RETURN] =
  l[GRAB] = l[PUSHGETGLOBAL] = l[GETGLOBAL] = l[SETGLOBAL] =
  l[PUSHATOM] = l[_ATOM] = l[MAKEBLOCK1] = l[MAKEBLOCK2] =
  l[MAKEBLOCK3] = l[GETFIELD] = l[SETFIELD] = l[DUMMY] =
  l[BRANCH] = l[BRANCHIF] = l[BRANCHIFNOT] = l[PUSHTRAP] =
  l[C_CALL1] = l[C_CALL2] = l[C_CALL3] = l[C_CALL4] = l[C_CALL5] =
  l[CONSTINT] = l[PUSHCONSTINT] = l[OFFSETINT] = l[OFFSETREF] = 1;

  /* Instructions with two operands */
  l[APPTERM] = l[CLOSURE] = l[CLOSUREREC] = l[PUSHGETGLOBALFIELD] =
  l[GETGLOBALFIELD] = l[MAKEBLOCK] = l[C_CALLN] = 2;

  len /= sizeof(opcode_t);
  for (p = code; p < code + len; /*nothing*/) {
    opcode_t instr = *p;
    if (instr < 0 || instr > STOP){
      fatal_error_arg ("Fatal error in fix_code: bad opcode (%lx)\n",
                       (char *)(long)instr);
    }
    *p++ = (opcode_t)(tls[threadId].instr_table[instr] - tls[threadId].instr_base);
    if (instr == SWITCH) {
      uint32 sizes = *p++;
      uint32 const_size = sizes & 0xFFFF;
      uint32 block_size = sizes >> 16;
      p += const_size + block_size;
    } else {
      p += l[instr];
    }
  }
  Assert(p == code + len);
}

#endif /* THREADED_CODE */

void set_instruction(code_t pos, opcode_t instr)
{
#ifdef THREADED_CODE
  *pos = (opcode_t)(tls[threadId].instr_table[instr] - tls[threadId].instr_base);
#else
  *pos = instr;
#endif
}

