/***********************************************************************/
/*                                                                     */
/*                           Objective Caml                            */
/*                                                                     */
/*            Xavier Leroy, projet Cristal, INRIA Rocquencourt         */
/*                                                                     */
/*  Copyright 1996 Institut National de Recherche en Informatique et   */
/*  Automatique.  Distributed only by permission.                      */
/*                                                                     */
/***********************************************************************/

/* $Id: extern.c,v 1.21 1997/12/11 20:53:06 xleroy Exp $ */

/* Structured output */

#include <string.h>
#include "alloc.h"
#include "fail.h"
#include "gc.h"
#include "intext.h"
#include "io.h"
#include "memory.h"
#include "misc.h"
#include "mlvalues.h"
#include "reverse.h"
#include "str.h"

#if defined(KAMEEL)
#define printf  DbgPrint
#endif
/* To keep track of sharing in externed objects */

typedef unsigned long byteoffset_t;

//_//struct extern_obj {
//_//  byteoffset_t ofs;
//_//  value obj;
//_//};

//_//TLS byteoffset_t initial_ofs = 1; /* Initial value of object offsets */
//_//TLS struct extern_obj * extern_table = NULL;
//_//TLS unsigned long extern_table_size;
//_//TLS byteoffset_t obj_counter;    /* Number of objects emitted so far */
//TLS static byteoffset_t initial_ofs = 1; /* Initial value of object offsets */
//TLS static struct extern_obj * extern_table = NULL;
//TLS static unsigned long extern_table_size;
//TLS static byteoffset_t obj_counter;    /* Number of objects emitted so far */

#ifdef ARCH_SIXTYFOUR
#define Hash(v) (((unsigned long) ((v) >> 3)) % tls[threadId].extern_table_size)
#else
#define Hash(v) (((unsigned long) ((v) >> 2)) % tls[threadId].extern_table_size)
#endif

/* Allocate a new extern table */
static void alloc_extern_table(void)
{
  asize_t i;
int threadId = getThreadId(getPeThread());

  tls[threadId].extern_table = (struct extern_obj *)
                 stat_alloc(tls[threadId].extern_table_size * sizeof(struct extern_obj));
  for (i = 0; i < tls[threadId].extern_table_size; i++) tls[threadId].extern_table[i].ofs = 0;
}

/* Grow the extern table */
static void resize_extern_table(void)
{
  asize_t oldsize;
  struct extern_obj * oldtable;
  value obj;
  byteoffset_t ofs;
  asize_t i, h;
int threadId = getThreadId(getPeThread());

  oldsize = tls[threadId].extern_table_size;
  oldtable = tls[threadId].extern_table;
  tls[threadId].extern_table_size = 2 * tls[threadId].extern_table_size;
  alloc_extern_table();
  for (i = 0; i < oldsize; i++) {
    ofs = oldtable[i].ofs;
    if (ofs >= tls[threadId].initial_ofs) {
      obj = oldtable[i].obj;
      h = Hash(obj);
      while (tls[threadId].extern_table[h].ofs >= tls[threadId].initial_ofs) {
        h++;
        if (h >= tls[threadId].extern_table_size) h = 0;
      }
      tls[threadId].extern_table[h].ofs = ofs;
      tls[threadId].extern_table[h].obj = obj;
    }
  }
  stat_free(oldtable);
}

/* Free the extern table. We keep it around for next call if
   it's still small (we did not grow it) and the initial offset
   does not risk running over next time. */
static void free_extern_table(void)
{
int threadId = getThreadId(getPeThread());
  if (tls[threadId].extern_table_size > INITIAL_EXTERN_TABLE_SIZE ||
      tls[threadId].initial_ofs >= INITIAL_OFFSET_MAX) {
    stat_free(tls[threadId].extern_table);
    tls[threadId].extern_table = NULL;
  }
}

/* To buffer the output */

//_//TLS char * extern_block, * extern_ptr, * extern_limit;
//_//TLS int extern_block_malloced;
//TLS static char * extern_block, * extern_ptr, * extern_limit;
//TLS static int extern_block_malloced;

static void alloc_extern_block(void)
{
int threadId = getThreadId(getPeThread());
  tls[threadId].extern_block = stat_alloc(INITIAL_EXTERN_BLOCK_SIZE);
  tls[threadId].extern_limit = tls[threadId].extern_block + INITIAL_EXTERN_BLOCK_SIZE;
  tls[threadId].extern_ptr = tls[threadId].extern_block;
  tls[threadId].extern_block_malloced = 1;
}

static void resize_extern_block(int required)
{
  long curr_pos, size, reqd_size;
int threadId = getThreadId(getPeThread());

  if (! tls[threadId].extern_block_malloced) {
    tls[threadId].initial_ofs += tls[threadId].obj_counter_e;
    free_extern_table();
    failwith("Marshal.to_buffer: buffer overflow");
  }
  curr_pos = tls[threadId].extern_ptr - tls[threadId].extern_block;
  size = tls[threadId].extern_limit - tls[threadId].extern_block;
  reqd_size = curr_pos + required;
  while (size <= reqd_size) size *= 2;
  tls[threadId].extern_block = stat_resize(tls[threadId].extern_block, size);
  tls[threadId].extern_limit = tls[threadId].extern_block + size;
  tls[threadId].extern_ptr = tls[threadId].extern_block + curr_pos;
}

/* Write characters, integers, and blocks in the output buffer */

#define Write(c) \
  if (tls[threadId].extern_ptr >= tls[threadId].extern_limit) resize_extern_block(1); \
  *tls[threadId].extern_ptr++ = (c)

static void writeblock(char *data, long int len)
{
int threadId = getThreadId(getPeThread());
  if (tls[threadId].extern_ptr + len > tls[threadId].extern_limit) resize_extern_block(len);
  bcopy(data, tls[threadId].extern_ptr, len);
  tls[threadId].extern_ptr += len;
}

static void writecode8(int code, long int val)
{
int threadId = getThreadId(getPeThread());
  if (tls[threadId].extern_ptr + 2 > tls[threadId].extern_limit) resize_extern_block(2);
  tls[threadId].extern_ptr[0] = code;
  tls[threadId].extern_ptr[1] = val;
  tls[threadId].extern_ptr += 2;
}

static void writecode16(int code, long int val)
{
int threadId = getThreadId(getPeThread());
  if (tls[threadId].extern_ptr + 3 > tls[threadId].extern_limit) resize_extern_block(3);
  tls[threadId].extern_ptr[0] = code;
  tls[threadId].extern_ptr[1] = val >> 8;
  tls[threadId].extern_ptr[2] = val;
  tls[threadId].extern_ptr += 3;
}

static void write32(long int val)
{
int threadId = getThreadId(getPeThread());
  if (tls[threadId].extern_ptr + 4 > tls[threadId].extern_limit) resize_extern_block(4);
  tls[threadId].extern_ptr[0] = val >> 24;
  tls[threadId].extern_ptr[1] = val >> 16;
  tls[threadId].extern_ptr[2] = val >> 8;
  tls[threadId].extern_ptr[3] = val;
  tls[threadId].extern_ptr += 4;
}

static void writecode32(int code, long int val)
{
int threadId = getThreadId(getPeThread());
  if (tls[threadId].extern_ptr + 5 > tls[threadId].extern_limit) resize_extern_block(5);
  tls[threadId].extern_ptr[0] = code;
  tls[threadId].extern_ptr[1] = val >> 24;
  tls[threadId].extern_ptr[2] = val >> 16;
  tls[threadId].extern_ptr[3] = val >> 8;
  tls[threadId].extern_ptr[4] = val;
  tls[threadId].extern_ptr += 5;
}

#ifdef ARCH_SIXTYFOUR
static void writecode64(int code, long val)
{
int threadId = getThreadId(getPeThread());
  int i;
  if (tls[threadId].extern_ptr + 9 > tls[threadId].extern_limit) resize_extern_block(9);
  *tls[threadId].extern_ptr++ = code;
  for (i = 64 - 8; i >= 0; i -= 8) *tls[threadId].extern_ptr++ = val >> i;
}
#endif

/* Marshal the given value in the output buffer */

//_//TLS unsigned long size_32;  /* Size in words of 32-bit block for struct. */
//_//TLS unsigned long size_64;  /* Size in words of 64-bit block for struct. */
//TLS static unsigned long size_32;  /* Size in words of 32-bit block for struct. */
//TLS static unsigned long size_64;  /* Size in words of 64-bit block for struct. */

//_//TLS int extern_ignore_sharing; /* Flag to ignore sharing */
//_//TLS int extern_closures;     /* Flag to allow externing code pointers */
//TLS static int extern_ignore_sharing; /* Flag to ignore sharing */
//TLS static int extern_closures;     /* Flag to allow externing code pointers */

static void extern_invalid_argument(char *msg)
{
int threadId = getThreadId(getPeThread());
  if (tls[threadId].extern_block_malloced) stat_free(tls[threadId].extern_block);
  tls[threadId].initial_ofs += tls[threadId].obj_counter_e;
  free_extern_table();
  invalid_argument(msg);
}

static void extern_rec(value v)
{
int threadId = getThreadId(getPeThread());
 tailcall:
  if (Is_long(v)) {
    long n = Long_val(v);
    if (n >= 0 && n < 0x40) {
      Write(PREFIX_SMALL_INT + n);
    } else if (n >= -(1 << 7) && n < (1 << 7)) {
      writecode8(CODE_INT8, n);
    } else if (n >= -(1 << 15) && n < (1 << 15)) {
      writecode16(CODE_INT16, n);
#ifdef ARCH_SIXTYFOUR
    } else if (n < -(1L << 31) || n >= (1L << 31)) {
      writecode64(CODE_INT64, n);
#endif
    } else
      writecode32(CODE_INT32, n);
    return;
  }

  if (Is_young(v) || Is_in_heap(v) || Is_atom(v)) {
    header_t hd = Hd_val(v);
    tag_t tag = Tag_hd(hd);
    mlsize_t sz = Wosize_hd(hd);
    asize_t h;
    /* Atoms are treated specially for two reasons: they are not allocated
       in the externed block, and they are automatically shared. */
    if (sz == 0) {
      if (tag < 16) {
        Write(PREFIX_SMALL_BLOCK + tag);
      } else {
        writecode32(CODE_BLOCK32, hd);
      }
      return;
    }
    /* Check if already seen */
    if (! tls[threadId].extern_ignore_sharing) {
      if (2 * tls[threadId].obj_counter_e >= tls[threadId].extern_table_size) resize_extern_table();
      h = Hash(v);
      while (tls[threadId].extern_table[h].ofs >= tls[threadId].initial_ofs) {
        if (tls[threadId].extern_table[h].obj == v) {
          byteoffset_t d = tls[threadId].obj_counter_e - (tls[threadId].extern_table[h].ofs - tls[threadId].initial_ofs);
          if (d < 0x100) {
            writecode8(CODE_SHARED8, d);
          } else if (d < 0x10000) {
            writecode16(CODE_SHARED16, d);
          } else {
            writecode32(CODE_SHARED32, d);
          }
          return;
        }
        h++;
        if (h >= tls[threadId].extern_table_size) h = 0;
      }
      /* Not seen yet. Record the object */
      tls[threadId].extern_table[h].ofs = tls[threadId].initial_ofs + tls[threadId].obj_counter_e;
      tls[threadId].extern_table[h].obj = v;
      tls[threadId].obj_counter_e++;
    }
    /* Output the contents of the object */
    switch(tag) {
    case String_tag: {
      mlsize_t len = string_length(v);
      if (len < 0x20) {
        Write(PREFIX_SMALL_STRING + len);
      } else if (len < 0x100) {
        writecode8(CODE_STRING8, len);
      } else {
        writecode32(CODE_STRING32, len);
      }
      writeblock(String_val(v), len);
      tls[threadId].size_32 += 1 + (len + 4) / 4;
      tls[threadId].size_64 += 1 + (len + 8) / 8;
      break;
    }
    case Double_tag: {
#ifdef KAMEEL
        invalid_argument("output_value: floating point not supported\n");
#else
      if (sizeof(double) != 8)
        extern_invalid_argument("output_value: non-standard floats");
      Write(CODE_DOUBLE_NATIVE);
      writeblock((char *) v, 8);
      tls[threadId].size_32 += 1 + 2;
      tls[threadId].size_64 += 1 + 1;
      break;
#endif /* KAMEEL */
    }
    case Double_array_tag: {
#ifdef KAMEEL
        invalid_argument("output_value: floating point not supported\n");
#else
      mlsize_t nfloats;
      if (sizeof(double) != 8)
        extern_invalid_argument("output_value: non-standard floats");
      nfloats = Wosize_val(v) / Double_wosize;
      if (nfloats < 0x100) {
        writecode8(CODE_DOUBLE_ARRAY8_NATIVE, nfloats);
      } else {
        writecode32(CODE_DOUBLE_ARRAY32_NATIVE, nfloats);
      }
      writeblock((char *) v, Bosize_val(v));
      tls[threadId].size_32 += 1 + nfloats * 2;
      tls[threadId].size_64 += 1 + nfloats;
      break;
#endif /* KAMEEL */
    }
    case Abstract_tag:
    case Final_tag:
      extern_invalid_argument("output_value: abstract value");
      break;
    case Infix_tag:
      writecode32(CODE_INFIXPOINTER, Infix_offset_hd(hd));
      extern_rec(v - Infix_offset_hd(hd));
      break;
    case Object_tag:
      extern_invalid_argument("output_value: object value");
      break;
    default: {
      mlsize_t i;
      if (tag < 16 && sz < 8) {
        Write(PREFIX_SMALL_BLOCK + tag + (sz << 4));
      } else {
        writecode32(CODE_BLOCK32, hd & ~Black);
      }
      tls[threadId].size_32 += 1 + sz;
      tls[threadId].size_64 += 1 + sz;
      for (i = 0; i < sz - 1; i++) extern_rec(Field(v, i)); 
      v = Field(v, i);
      goto tailcall;
      }
    }
    return;
  }

  if ((char *) v >= code_area_start && (char *) v < code_area_end) {
    if (!tls[threadId].extern_closures)
      extern_invalid_argument("output_value: functional value");
    writecode32(CODE_CODEPOINTER, (char *) v - code_area_start);
    writeblock((char *) code_checksum(), 16);
    return;
  }
  extern_invalid_argument("output_value: abstract value");
}

enum { NO_SHARING = 1, CLOSURES = 2 };
static int extern_flags[] = { NO_SHARING, CLOSURES };

static long extern_value(value v, value flags)
{

  long res_len;
  int fl;
int threadId = getThreadId(getPeThread());

  /* Parse flag list */
  fl = convert_flag_list(flags, extern_flags);
  tls[threadId].extern_ignore_sharing = fl & NO_SHARING;
  tls[threadId].extern_closures = fl & CLOSURES;
  /* Allocate hashtable of objects already seen, if needed */
  tls[threadId].extern_table_size = INITIAL_EXTERN_TABLE_SIZE;
  if (tls[threadId].extern_table == NULL) {
    alloc_extern_table();
    tls[threadId].initial_ofs = 1;
  }
  tls[threadId].obj_counter_e = 0;
  tls[threadId].size_32 = 0;
  tls[threadId].size_64 = 0;
  /* Write magic number */
  write32(Intext_magic_number);
  /* Set aside space for the sizes */
  tls[threadId].extern_ptr += 4*4;

  /* Marshal the object */
  extern_rec(v);

  /* Update initial offset for next call to extern_value(),
     if we decide to keep the table of shared objects. */
  tls[threadId].initial_ofs += tls[threadId].obj_counter_e;
  /* Free the table of shared objects (if needed) */
  free_extern_table();
  /* Write the sizes */
#ifdef ARCH_SIXTYFOUR
  if (tls[threadId].size_32 >= (1L << 32) || tls[threadId].size_64 >= (1L << 32)) {
    /* The object is so big its size cannot be written in the header.
       Besides, some of the block sizes or string lengths or shared offsets
       it contains may have overflowed the 32 bits used to write them. */
    failwith("output_value: object too big");
  }
#endif
  res_len = tls[threadId].extern_ptr - tls[threadId].extern_block;
  tls[threadId].extern_ptr = tls[threadId].extern_block + 4;
  write32(res_len - 5*4);
  write32(tls[threadId].obj_counter_e);
  write32(tls[threadId].size_32);
  write32(tls[threadId].size_64);

  /* Result is res_len bytes starting at extern_block */
  return res_len;
}

void output_val(struct channel *chan, value v, value flags)
{
  long len;
int threadId = getThreadId(getPeThread());
  alloc_extern_block();
  len = extern_value(v, flags);
  really_putblock(chan, tls[threadId].extern_block, len);
  stat_free(tls[threadId].extern_block);
}

value output_value(value vchan, value v, value flags) /* ML */
{
  struct channel * channel;
int threadId = getThreadId(getPeThread());
  channel = Channel(vchan);
  Begin_root(v)
    Lock(channel);
    output_val(channel, v, flags);
    Unlock(channel);
  End_roots();
  return Val_unit;
}

value output_value_to_string(value v, value flags) /* ML */
{
  long len;
  value res;
int threadId = getThreadId(getPeThread());
  alloc_extern_block();
  len = extern_value(v, flags);
  res = alloc_string(len);
  bcopy(tls[threadId].extern_block, String_val(res), len);
  stat_free(tls[threadId].extern_block);
  return res;
}

value output_value_to_buffer(value buf, value ofs, value len, value v, value flags) /* ML */
{
  long len_res;
int threadId = getThreadId(getPeThread());
  tls[threadId].extern_block = &Byte(buf, Long_val(ofs));
  tls[threadId].extern_limit = tls[threadId].extern_block + Long_val(len);
  tls[threadId].extern_ptr = tls[threadId].extern_block;
  tls[threadId].extern_block_malloced = 0;
  len_res = extern_value(v, flags);
  return Val_long(len_res);
}

