/************************
* Walker Program        *
* CS 672 Final Project  *
*                       *
* Allen Wang            *
* Matt Harren           *
************************/

import java.util.Vector;

import Program.*;
import java.io.*;

public class Walker implements Serializable
{
	int numVertices;
	public Vector vertices;
	Vector programs;
	boolean foot0hasMoved, foot2hasMoved;
	double fitness;
	
	//3-vertex constructor
	public Walker()
	{
		numVertices = 3;
		
		programs = new Vector(3);
		programs.addElement(RealExpr.generate(this, 0, 1.0));
		programs.addElement(RealExpr.generate(this, 1, 1.0));
		programs.addElement(RealExpr.generate(this, 2, 1.0));
		fitness = 0;
	}
	
	public void resetLocations()
	{
		vertices = new Vector(3);
		vertices.addElement(new Vertex(0, true, 200, 20));
		vertices.addElement(new Vertex(1, false, 260, 100));
		vertices.addElement(new Vertex(2, true, 320, 20));
		foot0hasMoved = foot2hasMoved = false;
		fitness = 0;
	}
	public void cleanLocations()
	{
		fitness = getTorsoX();
		vertices = null;
	}
	
	public Walker(Walker old)
	{
		numVertices = old.numVertices;
		programs = new Vector(numVertices);
		for (int i = 0; i < numVertices; i++)
		{
			programs.addElement( ((RealExpr)old.programs.elementAt(i)).copy(this) );
		}

		if(old.vertices != null)
		{
			vertices = new Vector(numVertices);
			for (int i = 0; i < numVertices; i++)
			{
				vertices.addElement( ((Vertex)old.vertices.elementAt(i)).copy()   );
			}
		}
		else
		{
			vertices = null;
			fitness = old.fitness;
		}
	}
	
	public final double getTorsoX()
	{
		if (vertices == null)
			return fitness;
		else
			return ((Vertex)vertices.elementAt(1)).getX();
	}
	
	public final int getNumVertices()
	{
		return numVertices;
	}
	public final double getXScaled(int i)
	{
		return ((Vertex)vertices.elementAt(i)).getX()/300;
	}
	public final double getYScaled(int i)
	{
		return ((Vertex)vertices.elementAt(i)).getY()/100;
	}
	public final double getVXScaled(int i)
	{
		return ((Vertex)vertices.elementAt(i)).getVX()/100;
	}
	public final double getVYScaled(int i)
	{
		return ((Vertex)vertices.elementAt(i)).getVY()/100;
	}
	public final boolean isFoot(int i) {
		return ((Vertex)vertices.elementAt(i)).isFoot();
	}
	
	
	public static final double PI_OVER_TWO = Math.PI/2.0;
	public static final double LENGTH = 100.0;
	public static final double HOP_VELOCITY = 10.0;
	public static final double GRAVITY = 2.0;
	public static final double HOP_THRESHOLD = 10.0;
	
	public static final int STEP_OK = 0;
	public static final int STEP_DEAD = 1;
	public static final int STEP_STUCK = 2;
	
	public int step(boolean verbose){
		int result = STEP_OK;
		Vertex foot0 = (Vertex)vertices.elementAt(0);
		Vertex torso = (Vertex)vertices.elementAt(1);
		Vertex foot2 = (Vertex)vertices.elementAt(2);
		double foot0result = ((RealExpr)programs.elementAt(0)).eval();
		double torsoresult = ((RealExpr)programs.elementAt(1)).eval()/10.0;
		double foot2result = ((RealExpr)programs.elementAt(2)).eval();

		boolean foot0hop = foot0result > HOP_THRESHOLD;
		boolean foot2hop = foot2result > HOP_THRESHOLD;
		foot0hasMoved |= foot0hop;
		foot2hasMoved |= foot2hop;

		if(foot0.onGround() && foot2.onGround())
		{	//The torso can't move.
			if(foot0hop && foot2hop)
			{	//both feet hop:
				foot0.vX = 0;
				foot2.vX = 0;
				foot0.vY = HOP_VELOCITY;
				foot2.vY = HOP_VELOCITY;
				foot0.Y = HOP_VELOCITY;
				foot2.Y = HOP_VELOCITY;
				
				torso.vY += HOP_VELOCITY;
				torso.Y += torso.vY;
				
				foot0.X += foot0.vX;
				foot2.X += foot2.vX;
				torso.X += torso.vX;
			}
			else if(!foot0hop && !foot2hop)
			{ 
				if(  (foot0.vX == 0) && (foot0.vY == 0)
				   &&(foot2.vX == 0) && (foot2.vY == 0)
				   &&(torso.vX == 0) && (torso.vY == 0))
				{
					result =  STEP_STUCK;
				}
				foot0.vX = 0;  foot0.vY = 0;
				foot2.vX = 0;  foot2.vY = 0;
				torso.vX = 0;  torso.vY = 0;
			}
			else 
			{	
				Vertex restingFoot = foot2;
				Vertex hoppingFoot = foot0;
				if (foot2hop){ 
					restingFoot = foot0;
					hoppingFoot = foot2;
				}
				double hoppingFootAngle = Math.atan2(torso.Y, torso.X-hoppingFoot.X);
				double restingFootAngle = Math.atan2(torso.Y, torso.X-restingFoot.X);
				double componentOfVelocityPushingTorso = HOP_VELOCITY *
					Math.cos(PI_OVER_TWO - hoppingFootAngle + restingFootAngle);
				double omega = componentOfVelocityPushingTorso/torso.distanceTo(restingFoot);
				
				double torsoToPivotX = torso.X-restingFoot.X;
				double torsoToPivotY = torso.Y-restingFoot.Y;
				torso.X = torsoToPivotX*Math.cos(omega)-torsoToPivotY*Math.sin(omega) + restingFoot.X;
				torso.Y = torsoToPivotX*Math.sin(omega)+torsoToPivotY*Math.cos(omega) + restingFoot.Y;
				torso.vX = HOP_VELOCITY * Math.cos(hoppingFootAngle);
				torso.vY = HOP_VELOCITY * Math.sin(hoppingFootAngle);
				
				double hoppingFoot_oldX = hoppingFoot.X;
				double hoppingFoot_oldY = hoppingFoot.Y;
				double hoppingToPivotX = hoppingFoot.X-restingFoot.X;
				double hoppingToPivotY = hoppingFoot.Y-restingFoot.Y;
				hoppingFoot.X = hoppingToPivotX*Math.cos(omega)-hoppingToPivotY*Math.sin(omega) + restingFoot.X;
				hoppingFoot.Y = hoppingToPivotX*Math.sin(omega)+hoppingToPivotY*Math.cos(omega) + restingFoot.Y;
				hoppingFoot.vX = hoppingFoot.X - hoppingFoot_oldX;
				hoppingFoot.vY = hoppingFoot.X - hoppingFoot_oldY;
				restingFoot.vX = 0;
				restingFoot.vY = 0;
			}
		}
		else if (foot0.onGround() || foot2.onGround())
		{
			Vertex restingFoot = foot2;
			Vertex hoppingFoot = foot0;
			boolean restingFootHopping = foot2hop;
			if (foot0.onGround()){ 
				restingFoot = foot0;
				hoppingFoot = foot2;
				restingFootHopping = foot0hop;
			}

			torso.vY -= GRAVITY;
			double restingFootAngle = Math.atan2(torso.Y, torso.X-restingFoot.X);
			double sine_restingFootAngle = Math.sin(restingFootAngle);
			double cos_restingFootAngle = Math.cos(restingFootAngle);
			double angularMovement = (-torso.vX * sine_restingFootAngle + torso.vY * cos_restingFootAngle)
									 / torso.distanceTo(restingFoot);
			restingFootAngle += angularMovement;
			sine_restingFootAngle = Math.sin(restingFootAngle);
			cos_restingFootAngle = Math.cos(restingFootAngle);
			torso.X = LENGTH*cos_restingFootAngle + restingFoot.X;
			torso.Y = LENGTH*sine_restingFootAngle + restingFoot.Y;
			torso.vX = -LENGTH*angularMovement*sine_restingFootAngle;
			torso.vY = LENGTH*angularMovement*cos_restingFootAngle;
			
			double distX = hoppingFoot.X-torso.X;
			double distY = hoppingFoot.Y-torso.Y;			
			double angleToHoppingFoot = normalize(Math.atan2(distY, distX) + torsoresult);
			hoppingFoot.X = LENGTH*Math.cos(angleToHoppingFoot) + torso.X;
			hoppingFoot.Y = LENGTH*Math.sin(angleToHoppingFoot) + torso.Y;
			hoppingFoot.vX = -LENGTH*normalize(torsoresult)*Math.sin(angleToHoppingFoot);
			hoppingFoot.vY = LENGTH*normalize(torsoresult)*Math.cos(angleToHoppingFoot);
			restingFoot.vX = 0;
			restingFoot.vY = 0;
		}
		else // both feet in air
		{
			//now move the feet "torsoresult" units farther apart
			double foot0Angle = Math.atan2(foot0.Y-torso.Y, foot0.X-torso.X);
			double foot2Angle = Math.atan2(foot2.Y-torso.Y, foot2.X-torso.X);
			double deltaAngle = normalize(torsoresult/2);
			if(foot2Angle > foot0Angle){
				foot0Angle -= deltaAngle;
				foot2Angle += deltaAngle;
			} else {
				foot0Angle += deltaAngle;
				foot2Angle -= deltaAngle;
			}
			
			torso.vY -= GRAVITY;
			torso.Y += torso.vY;
			torso.X += torso.vX;
			
			double approxMovement = LENGTH * deltaAngle;
			foot0.X = torso.X + LENGTH*Math.cos(foot0Angle);
			foot0.Y = torso.Y + LENGTH*Math.sin(foot0Angle);
			foot0.vX = approxMovement*Math.sin(foot0Angle);
			foot0.vY = approxMovement*Math.cos(foot0Angle);
			foot2.X = torso.X + LENGTH*Math.cos(foot2Angle);
			foot2.Y = torso.Y + LENGTH*Math.sin(foot2Angle);
			foot2.vX = approxMovement*Math.sin(foot2Angle);
			foot2.vY = approxMovement*Math.cos(foot2Angle);
		}
		foot0.checkBoundaries();
		torso.checkBoundaries();
		foot2.checkBoundaries();

		if(torso.onGround() || (torso.X <= 0))
			result = STEP_DEAD;

		return result;
	}
	
	public double normalize(double angle)
	{
		return (angle+Math.PI)%(2*Math.PI)-Math.PI;
	}
	
	public Walker copy()
	{
		return new Walker(this);
	}
	
	public void dumpToFile(String filename, int gen, int ps, int mg, double mp, double cp)
	{
		try {
			PrintWriter output = new PrintWriter(new FileOutputStream(filename));			
			((RealExpr)programs.elementAt(0)).dumpNode(output);
			 output.println();
			 output.println();
			((RealExpr)programs.elementAt(1)).dumpNode(output);
			 output.println();
			 output.println();
			((RealExpr)programs.elementAt(2)).dumpNode(output);
			 output.println();
			 output.println();
			output.println("====================");
			output.println("Created: " + (new java.util.Date()).toString());
			output.println("Generation:  " + gen);
			output.println("Pop size:    " + ps);
			output.println("Max gen:     " + mg);
			output.println("Mutate prob: " + mp);
			output.println("Cross prob:  " + cp);
			output.println();
			output.close();
		} 
		catch (IOException e)	{
			e.printStackTrace();
		}
	}
	
	public Walker(String filename)
	{
		numVertices = 3;
		resetLocations();
		
		programs = new Vector(3);
		try {
			Parser parser = new Parser(filename, this, 0);
			programs.addElement(parser.parseReal());
			parser.vertexNum = 1;
			programs.addElement(parser.parseReal());
			parser.vertexNum = 2;
			programs.addElement(parser.parseReal());
		} 
		catch (Exception e)	{
			e.printStackTrace();
			System.exit(-1);
		}
		programs.addElement(RealExpr.generate(this, 0, 1.0));
		programs.addElement(RealExpr.generate(this, 1, 1.0));
		programs.addElement(RealExpr.generate(this, 2, 1.0));
		fitness = 0;
	}
	

}