structure ToMP = struct

   open KAT
   
   fun openLib lib = (KAT.library := [] ; load [lib])

   val nameReplacements = [(#"*","_star"),
                           (#"+","_plus"),
                           (#"<","_leq"),
                           (#"=","_eq"),
                           (#".","_times"),
                           (#"1","_one"),
                           (#"0","_zero"),
                           (#"~","_not")]

   val toReduce = ["<elim","id+R","idemp+","id.L","id.R","annihL", "annihR","unwindL","unwindR", "<1", "compl.", "compl+","~1", "~0","idemp.","~~","abs+","abs."]
 

   fun termVars term = case term of
     (ACT_CON(x) | ACT_VAR(x)) => ["'"^x^" "]
   | (TST_VAR(_) | TST_CON(_)) => []
   | (PLUS(x) | TIMES(x)) => List.concat(map termVars x)
   | NOT(x) => termVars(x)
   | STAR(x) => termVars(x)
   | ZERO => []
   | ONE => []

   fun boolVars term = case term of
     (TST_CON(x) | TST_VAR(x)) => ["'"^x^" "]
   | (ACT_VAR(_) | ACT_CON(_)) => []
   | (PLUS(x) | TIMES(x)) => List.concat(map boolVars x)
   | NOT(x) => boolVars(x)
   | STAR(x) => boolVars(x)
   | ZERO => []
   | ONE => []

   fun getRightVar(eqn) = case eqn of
     (EQ(x,y) | LE(x,y)) => List.filter (fn t => not(Util.member t (termVars(x) @ boolVars(x)))) 
                            (termVars(y) @ boolVars(y))


   fun getPremVar (prems,conc) = let
     fun getVars prem = case prem of
       (EQ(x,y) | LE(x,y)) => termVars(x) @ termVars(y) @ boolVars(x) @ boolVars(y)
     val vars = List.concat (map getVars prems)
     val cvars = getVars(conc)
   in
     List.filter (fn x => not(Util.member x cvars)) vars
   end

   fun getTermVar (prems,conc) = let
     fun getVars prem = case prem of
       (EQ(x,y) | LE(x,y)) => termVars(x) @ termVars(y)
     val vars = List.concat (map getVars prems)
     val cvars = getVars(conc)
   in
     vars @ cvars
   end

   fun getBoolVar (prems,conc) = let
     fun getVars prem = case prem of
       (EQ(x,y) | LE(x,y)) => boolVars(x) @ boolVars(y)
     val vars = List.concat (map getVars prems)
     val cvars = getVars(conc)
   in
     vars @ cvars
   end

   fun getAllVar (prems,conc) = getBoolVar(prems,conc) @ getTermVar(prems,conc)

   fun makeMPT stm = case stm of
     (TST_VAR(x) | TST_CON(x) | ACT_CON(x) | ACT_VAR(x)) => "'"^x
   | PLUS([x]) => makeMPT(x)
   | PLUS(x::xs) => "("^makeMPT(x)^" + "^makeMPT(PLUS(xs))^")"
   | TIMES([x]) => makeMPT(x)
   | TIMES(x::xs) => "("^makeMPT(x)^" * "^makeMPT(TIMES(xs))^")"
   | NOT(x) => "(-"^"("^makeMPT(x)^"))"
   | STAR(x) => "(star{"^makeMPT(x)^"}"^")"
   | ZERO => "0"
   | ONE => "1"

   fun makeMP stm = "     "^(case stm of
     EQ(x,y) => makeMPT(x)^" <--> "^makeMPT(y)
   | LE(x,y) => makeMPT(x)^" + "^makeMPT(y)^" <--> "^makeMPT(y))

   fun makerwPMP stm = case stm of
     EQ(x,y) => "     ("^makeMPT(x)^" ~ "^makeMPT(y)^")"
   | LE(x,y) => "     ("^makeMPT(x)^" <= "^makeMPT(y)^")"

   fun makepremMP stm = "     sequent{ <H> >- "^
     (case stm of
       EQ(x,y) => makeMPT(x)^" ~ "^makeMPT(y)
     | LE(x,y) => makeMPT(x)^" <= "^makeMPT(y))^" }"

   fun makequants(prems,concs) = let 
     val bools = removeDuplicates(getBoolVar(prems,concs))
     val terms = removeDuplicates(getTermVar(prems,concs))
   in
      (foldr (fn (y,s) => s^"     ("^y^"in kleene) -->\n") "" terms)^
      (foldr (fn (y,s) => s^"     ("^y^"in bool) -->\n") "" bools)
   end

   fun makesquants(prems,concs) = let 
     val bools = removeDuplicates(getBoolVar(prems,concs))
     val terms = removeDuplicates(getTermVar(prems,concs))
   in
      (foldr (fn (y,s) => s^"     [wf] sequent{ <H> >- "^y^"in kleene} -->\n") "" terms)^
      (foldr (fn (y,s) => s^"     [wf] sequent{ <H> >- "^y^"in bool} -->\n") "" bools)
   end


   fun translateTheorem (thm:theorem) = let
     val theorem = thm
     val (prems,conclusion) = #1 (#2 (theorem))
     val proof = !(#3 (#2 (theorem)))
     val name = foldr(fn ((x,y),z) => (replaceCharWS x y z)) (#1 theorem) nameReplacements
     val name = implode(map Char.toLower (explode(name)))
     val premvar = removeDuplicates(getPremVar(prems,conclusion))
     val allvars = removeDuplicates(getAllVar(prems,conclusion))
     val quants = makequants(prems,conclusion)
     val squants = makesquants(prems,conclusion)
     val reduce = if (member (#1 theorem) toReduce) then " {|reduce|}" else ""
(*     val startstring = case (proof,prems) of
       (AXIOM _,[]) => "prim_rw "
     | (AXIOM _, _) => "prim "
     | (_,[]) => "interactive_rw "
     | (_,_) => "interactive "*)
   in
     case (proof,prems,conclusion) of
       (AXIOM _,[],EQ(x,y)) => let
          val lvars = removeDuplicates(getRightVar(conclusion))
          val rlvars = removeDuplicates(getRightVar(EQ(y,x)))
        in
          "prim "^name^" {| intro[] |}:\n"^squants^makepremMP(conclusion)^" = it\n\n"^
          "interactive_rw "^name^"_l "^(String.concat(lvars))^reduce^":\n"^quants^makeMP(conclusion)^"\n\n"^
          "interactive_rw "^name^"_r "^(String.concat(rlvars))^":\n"^quants^makeMP(EQ(y,x))^"\n\n"

        end

     | (AXIOM _,[],LE(x,y)) =>  let
          val lvars = removeDuplicates(getRightVar(conclusion))
        in
          "prim "^name^" {| intro[] |}:\n"^squants^makepremMP(conclusion)^" = it\n\n"
        end
   
     | (AXIOM _,_,EQ(x,y)) => let
          val lvars = removeDuplicates(getRightVar(conclusion))
          val rlvars = removeDuplicates(getRightVar(EQ(y,x)))
        in
          "prim "^name^" "^(String.concat(premvar))^":\n"^squants^
          (foldr (fn (y,s) => s^makepremMP(y)^" -->\n") "" prems)^makepremMP(conclusion)^" = it\n\n"^
          "interactive_rw "^name^"_l "^(String.concat(lvars @ premvar))^reduce^":\n"^quants^
          (foldr (fn (y,s) => s^makerwPMP(y)^" -->\n") "" prems)^makeMP(conclusion)^"\n\n"^
          "interactive_rw "^name^"_r "^(String.concat(rlvars @ premvar))^":\n"^quants^
          (foldr (fn (y,s) => s^makerwPMP(y)^" -->\n") "" prems)^makeMP(EQ(y,x))^"\n\n"

        end

     | (AXIOM _,_,LE(x,y)) =>  let
          val lvars = removeDuplicates(getRightVar(conclusion))
       in
          "prim "^name^" "^(String.concat(premvar))^":\n"^squants^
          (foldr (fn (y,s) => s^makepremMP(y)^" -->\n") "" prems)^makepremMP(conclusion)^" = it\n\n"
      end

    | (_,[],EQ(x,y)) =>let
          val lvars = removeDuplicates(getRightVar(conclusion))
          val rlvars = removeDuplicates(getRightVar(EQ(y,x)))
        in
          "interactive "^name^" "^(String.concat(premvar))^"{| intro[] |}:\n"^squants^makepremMP(conclusion)^"\n\n"^
          "interactive_rw "^name^"l_rw "^(String.concat(lvars))^reduce^":\n"^quants^makeMP(conclusion)^"\n\n"^
          "interactive_rw "^name^"r_rw "^(String.concat(rlvars))^":\n"^quants^makeMP(EQ(y,x))^"\n\n"
        end

     | (_,[],LE(x,y)) =>  let
          val lvars = removeDuplicates(getRightVar(conclusion))
        in
          "interactive "^name^" {| intro[] |}:\n"^squants^makepremMP(conclusion)^"\n\n"
        end
   
     | (_,_,EQ(x,y)) => let
          val lvars = removeDuplicates(getRightVar(conclusion))
          val rlvars = removeDuplicates(getRightVar(EQ(y,x)))
        in
          "interactive "^name^" "^(String.concat(premvar))^":\n"^squants^
          (foldr (fn (y,s) => s^makepremMP(y)^" -->\n") "" prems)^makepremMP(conclusion)^"\n\n"^
          "interactive_rw "^name^"l_rw "^(String.concat(lvars @ premvar))^reduce^":\n"^quants^
          (foldr (fn (y,s) => s^makerwPMP(y)^" -->\n") "" prems)^makeMP(conclusion)^"\n\n"^
          "interactive_rw "^name^"r_rw "^(String.concat(rlvars @ premvar))^":\n"^quants^
          (foldr (fn (y,s) => s^makerwPMP(y)^" -->\n") "" prems)^makeMP(EQ(y,x))^"\n\n"
        end

     | (_,_,LE(x,y)) =>  let
          val lvars = removeDuplicates(getRightVar(conclusion))
       in
          "interactive "^name^" "^(String.concat(premvar))^":\n"^squants^
          (foldr (fn (y,s) => s^makepremMP(y)^" -->\n") "" prems)^makepremMP(conclusion)^"\n\n"
      end


   end

   fun writeAllThms filename extends = let
     val outstreamli = TextIO.openOut (filename^".mli")
     val () = app (fn x => TextIO.outputSubstr(outstreamli,Substring.all("extends "^x^"\n"))) extends
     val () = TextIO.closeOut outstreamli
     val outstream = TextIO.openOut (filename^".ml")
     val () = app (fn x => TextIO.outputSubstr(outstream,Substring.all("extends "^x^"\n"))) extends
     val () = TextIO.outputSubstr(outstream,
              Substring.all ("\nopen Top_conversionals\nopen Base_select\nopen Dtactic\n\n"))
     val () = app (fn x => TextIO.outputSubstr(outstream,Substring.all (translateTheorem x))) (!library)
   in
     TextIO.closeOut outstream
   end



fun run () = let
val pref = "/Users/kamal/Desktop/MetaPRL/kat_"
val () = writeAllThms (pref^"ax") ["Kat_terms"]
val () = (openLib "std" ; writeAllThms (pref^"std") ["Kat_ax"])
val () = (openLib "bool" ; writeAllThms (pref^"bool") ["Kat_terms"])
(*val () = (openLib "Hoare" ; writeAllThms (pref^"hoare") ["Kat_std","Kat_bool"])*)
val () = (openLib "denest" ; writeAllThms (pref^"denest") ["Kat_std","Kat_bool"])
val () = (openLib "star" ; writeAllThms (pref^"star") ["Kat_std","Kat_bool"])
val () = (openLib "MSDriver" ; writeAllThms (pref^"MSDriver") ["Kat_std","Kat_bool"])
in () end

end