signature TERM = sig

  datatype term =
    TST_VAR of string
  | ACT_VAR of string
  | TST_CON of string
  | ACT_CON of string
  | PLUS of term list
  | TIMES of term list
  | NOT of term
  | STAR of term
  | ZERO
  | ONE

  type id = string
  type substitution = (id * term) list
  datatype eqn = EQ of term * term | LE of term * term
  type cond_eqn = eqn list * eqn

  (* Parse string to a term (equation, conditional equation)*)
  val parseTerm : string -> term option
  val parseEqn : string -> eqn option
  val parseCondEqn : string -> cond_eqn option

  (* Flatten PLUS and TIMES terms to a single list *)
  val flatten : term -> term

  (* Make all variables constants (variable) in a term*)
  val makeConstant : term -> term
  val makeVariable : term -> term

  (* Make all variables constants (variable) in an equation *)
  val makeConstantEqn : eqn -> eqn
  val makeVariableEqn : eqn -> eqn

  (* Make all variables constants (variable) in a conditional eqn *)
  val makeConstantCondEqn : cond_eqn -> cond_eqn
  val makeVariableCondEqn : cond_eqn -> cond_eqn

  (* Make all constants variables in a substitution *)
  val makeVariableSubst : substitution -> substitution

  (* Get all variables in a term (equation,conditional equation,substitution)*)
  val variables : term -> id list
  val eqnVariables : eqn -> id list
  val condEqnVariables : cond_eqn -> id list
  val substVariables : substitution -> id list

  (* Does the term contain only Booleans? *)
  val isTest : term -> bool

  (* Is the term a PLUS or TIMES? *)
  val headsymbol : term -> (term list -> term)

  (* Get term as string (printout of structure (for debugging),XML string) *)
  val toString : term -> string
  val termStructure : term -> string
  val termToXML : term -> string

  (* Get equation as string 
   * (printout of structure (for debugging),XML string) *)
  val eqnToString : eqn -> string
  val eqnStructure : eqn -> string
  val eqnToXML : eqn -> string

  (* Get conditional equation as string 
   * (printout of structure (for debugging),XML string) *)
  val condEqnToString : cond_eqn -> string
  val condEqnStructure : cond_eqn -> string
  val condEqnToXML : cond_eqn -> string

  (* Convert substitution list to XML *)
  val substToXML : substitution -> string

  (* Convert id to XML *)
  val idToXML : id -> string

  (* Convert term (eqn,eqn w/o &s) to LaTeX *)
  val toLaTeX : term -> string
  val eqnToLaTeX : eqn -> string
  val eqnToLaTeXNA : eqn -> string

  (* Get focus in an equantion as a string *)
  val focusSubterm :  (term * int list) -> int * int * int
  val focusInEqnToString : (eqn * int list -> string) 

  (* substitute a term into an equation at a given focus,
   * return new focus *)
  val focusedSubstInEqn : eqn * int list * term -> eqn * int list

  val moveToFocus : int list -> eqn -> string

  (* Get lefthand and righthand side of an equation *)
  val args : eqn -> term * term

  val opToString : eqn -> string

  (* Get term in equation at given focus (with length of + or ; term) *)
  val getFocusInEqn : eqn * int list -> term
  val getFocusInEqnWithContext : eqn * int list -> term * int

  (* CPS function to prompt user for terms that need filled in
   * Function is for CLI interface, update reference for other
   * interfaces.  Constructs list of all user-inputted terms.
   * When all holes filled in, pass to a continuation, which
   * will be the actual citation of a theorem                  *)
  val getMissing : (string list -> term list -> (term list -> unit) -> unit) ref

  (* Parse an identifier (equation) *)
  val parseId : string -> term
  val parseEqnTokenized : string list -> eqn option
end