structure Proof = struct

  open Util
  open Term
  open XML
  open SLang

(***********************************************
 * syntax
 ***********************************************)
  type fosubstitution = (C * term) list
  datatype proof =
    LAMBDAS of (id list) * proof
  | LAMBDAP of (id list) * proof
  | APPLYS of proof * substitution
  | APPLYF of proof * fosubstitution
  | APPLYP of proof * proof list
  | VAR of id
  | TASK of id * (cond_eqn * proof list * int list)
  | CONST of id
  | AXIOM of id
  | SUBST of int list * eqn * direction * proof * proof

  type task = id * (cond_eqn * proof list * int list)
  type  theorem = id * (cond_eqn * task option ref * proof ref)


(***********************************************
 * output
 ***********************************************)

  fun directionToString (d:direction) = case d of LEFT => "L" | RIGHT => "R"

  fun directionToXML (d:direction) = "<dir>" ^ directionToString d ^ "</dir>"


  fun fosubstToString (s:fosubstitution) : string =
    "[" ^ concat (mapAllButLast (fn x => x ^ " ")
     (map (fn(x,t) => FirstOrder.cTermToString x ^ "=" ^ toString t) s)) ^ "]"

  fun fosubstStructure (s:fosubstitution) : string =
    "[" ^ concat (mapAllButLast (fn x => x ^ " ") (map (fn(x,t) => FirstOrder.cTermToString x ^ "=" ^ termStructure t) s)) ^ "]"

  fun toString (p:proof) : string = let
    fun protect x =
      case x of
        TASK (_,(_,[],_)) => toString x
      | (APPLYF _ | APPLYS _ | APPLYP _ | TASK _ | SUBST _) =>
        "(" ^ toString x ^ ")"
      | _ => toString x
  in
    case p of
      (VAR x | CONST x| AXIOM x) => x
    | TASK (x,(_,q,_)) =>
        (case q of
          [] => x
        | [p] => x ^ " " ^ protect p
        | _  => x ^ " (" ^ concat(mapAllButLast (fn x => x^",") (map toString q)) ^ ")")
    | (LAMBDAS (s,p) | LAMBDAP (s,p))=>
        (case s of
          [] => protect p
        | _  => "\\" ^ concat(mapAllButLast (fn x => x^",") s) ^ "." ^ protect p)
    | APPLYS (p,s:substitution) =>
        (case s of
          [] => toString p
        | _ => toString p ^ " " ^ Unify.substToString s)

    | APPLYF (p,s:fosubstitution) =>
        (case s of
          [] => toString p
        | _ => toString p ^ " " ^ fosubstToString s)



    | APPLYP (p,q) =>
        (case q of
          [] => toString p
        | [x] => toString p ^ " " ^  protect x
        | _ => toString p ^ " (" ^ concat(mapAllButLast (fn x => x^",") (map toString q)) ^ ")")
    | SUBST (f,e,d,p,q) =>
        "subst [" ^ concat(mapAllButLast (fn x => x ^ ",") (map Int.toString f)) ^ "] ("
        ^ Term.eqnToString e ^ ") " ^ directionToString d ^ " " ^ protect p ^ " " ^ protect q
  end

  fun proofStructure (p:proof) : string =
    case p of
      VAR x => "VAR(" ^ x ^ ")"
    | CONST x => "CONST(" ^ x ^ ")"
    | AXIOM x => "AXIOM(" ^ x ^ ")"
    | TASK (x,(ce,q,_)) => "TASK(" ^ x ^ ":" ^ condEqnStructure ce ^
        ",[" ^ concat(mapAllButLast (fn x => x^",") (map proofStructure q)) ^ "])"
    | LAMBDAS (s,p) => "LAMBDAS([" ^ concat(mapAllButLast (fn x => x^",") s) ^ "]," ^ proofStructure p ^ ")"
    | LAMBDAP (s,p) => "LAMBDAP([" ^ concat(mapAllButLast (fn x => x^",") s) ^ "]," ^ proofStructure p ^ ")"
    | APPLYS (p,s:substitution) => "APPLYS(" ^ proofStructure p ^ "," ^ Unify.substStructure s ^ ")"
    | APPLYF (p,s:fosubstitution) => "APPLYS(" ^ proofStructure p ^ "," ^ fosubstStructure s ^ ")"
    | APPLYP (p,q) => "APPLYP(" ^ proofStructure p ^ ",[" ^
        concat(mapAllButLast (fn x => x^",") (map proofStructure q)) ^ "])"
    | SUBST (f,e,d,p,q) =>
        "SUBST([" ^ concat(mapAllButLast (fn x => x ^ ",") (map Int.toString f)) ^ "],"
        ^ Term.eqnStructure e ^ "," ^ directionToString d ^ "," ^ proofStructure p ^ "," ^ proofStructure q ^ ")"

  fun focusToXML (f:int list) : string =
    "<focus>" ^ concat(mapAllButLast (fn x => x ^ " ") (map Int.toString f)) ^ "</focus>"

  fun fosubstToXML (s:fosubstitution) : string =
    let fun toXML (x:C,t:term) = "<fsub><cterm>" ^ FirstOrder.cTermToString x ^ "</cterm>" ^ termToXML t ^ "</fsub>"
    in concat(map toXML s) end



  fun proofToXML (p:proof) : string = let
    val fixle : string -> string = String.map (fn c => if c = #"<" then #"@" else c)
    fun taskToXML ((id,(condEqn,proofs,focus)):task) : string =
      "<TASK>" ^ idToXML id ^ condEqnToXML condEqn ^
      focusToXML focus ^ concat(map toXML proofs) ^ "</TASK>"
    and toXML (p:proof) : string =
      case p of
        VAR x => "<VAR>" ^ fixle x ^ "</VAR>"
      | CONST x => "<CONST>" ^ fixle x ^ "</CONST>"
      | AXIOM x => "<AXIOM>" ^ fixle x ^ "</AXIOM>"
      | TASK t => taskToXML t
      | LAMBDAS (s,p) => "<LAMBDAS>" ^ concat(map idToXML s) ^ toXML p ^ "</LAMBDAS>"
      | LAMBDAP (s,p) => "<LAMBDAP>" ^ concat(map idToXML s) ^ toXML p ^ "</LAMBDAP>"
      | APPLYS (p,s:substitution) => "<APPLYS>" ^ toXML p ^ substToXML s ^ "</APPLYS>"
      | APPLYF (p,s:fosubstitution) => "<APPLYF>" ^ toXML p ^ fosubstToXML s ^ "</APPLYF>"
      | APPLYP (p,q) => "<APPLYP>" ^ toXML p ^ concat(map toXML q) ^ "</APPLYP>"
      | SUBST (f,e,d,p,q) => "<SUBST>" ^ focusToXML f ^ eqnToXML e ^
          directionToXML d ^ toXML p ^ toXML q ^ "</SUBST>"
  in
    "<proof>" ^ toXML p ^ "</proof>"
  end

(***********************************************
 * utilities
 ***********************************************)

  (* make deep copy *)
  fun copy (p:proof) : proof =
    case p of
      (VAR _ | CONST _ | AXIOM _) => p
    | TASK (id,(ce,t,f)) => TASK (id,(ce,map copy t,f))
    | LAMBDAS (s,q) => LAMBDAS (s,copy q)
    | LAMBDAP (s,q) => LAMBDAP (s,copy q)
    | APPLYS (q,s) => APPLYS (copy q,s)
    | APPLYF (q,s) => APPLYF (copy q,s)
    | APPLYP (q,r) => APPLYP (copy q,map copy r)
    | SUBST (f,e,d,q,r) => SUBST (f,e,d,copy q,copy r)

  (* get a list of tasks *)
  fun getTasks (proof:proof) : task list = let
    fun gt (p:proof) : task list =
      case p of
        TASK (task as (id,(ce,args,focus))) => task :: List.concat(map gt args)
      | (VAR _ | CONST _ | AXIOM _) => []
      | APPLYP (q,args) => gt q @ List.concat(map gt args)
      | (APPLYF (q,_) | APPLYS (q,_) | LAMBDAS (_,q) | LAMBDAP (_,q)) => gt q
      | SUBST (_,_,_,q,r) => gt q @ gt r
    in
      removeDuplicates(gt proof)
    end

  (* check for occurrence of a constant in a proof *)
  fun occurs (name:id) (p:proof) : bool =
    case p of
      CONST c => c = name
    | TASK (_,(_,args,_)) => List.exists (occurs name) args
    | (VAR _ | AXIOM _) => false
    | APPLYP (q,args) => occurs name q orelse List.exists (occurs name) args
    | (APPLYF (q,_) | APPLYS (q,_) | LAMBDAS (_,q) | LAMBDAP (_,q)) => occurs name q
    | SUBST (_,_,_,q,r) => occurs name q orelse occurs name r

  (* count occurrences of a constant in a proof *)
  fun occurrences (name:id) (p:proof) : int =
    case p of
      CONST c => if c = name then 1 else 0
    | TASK (t,(_,args,_)) => List.foldr (fn(x,y) => x + y) 0 (map (occurrences name) args)
    | (VAR _ | AXIOM _) => 0
    | APPLYP (q,args) => List.foldr (fn(x,y) => x + y) 0 (map (occurrences name) (q::args))
    | (APPLYF (q,_) | APPLYS (q,_) | LAMBDAS (_,q) | LAMBDAP (_,q)) => occurrences name q
    | SUBST (_,_,_,q,r) => (occurrences name q) + (occurrences name r)

  (* recursive substitution for a task variable *)
  fun replaceTask (substProof:proof) (taskId:id) (inProof:proof) : proof = let
    fun replace (p:proof) =
       case p of
         TASK (id,(ce,args,focus)) =>
         if id = taskId then substProof
         else TASK (id,(ce,map replace args,focus))
      | (VAR _ | CONST _ | AXIOM _) => p
      | APPLYP (q,[]) => replace q
      | APPLYP (q,args) => APPLYP (replace q, map replace args)
      | APPLYS (q,[]) => replace q
      | APPLYS (q,s) => APPLYS (replace q,s)
      | APPLYF (q,[]) => replace q
      | APPLYF (q,s) => APPLYF (replace q,s)
      | LAMBDAS (s,q) => LAMBDAS (s,replace q)
      | LAMBDAP (s,q) => LAMBDAP (s,replace q)
      | SUBST (c,e,d,q,r) => SUBST (c,e,d,replace q,replace r)
  in
    replace inProof
  end

  (* substitution of a proof q for a cited theorem in a proof p *)
  fun replaceTheorem (name:id) (q:proof) (p:proof) : proof = let
    fun replace (p:proof) =
      case p of
        CONST c => if c = name then q else p
      | TASK (t,(ce,args,focus)) => TASK (t,(ce,map replace args,focus))
      | (VAR _ | AXIOM _) => p
      | APPLYP (u,[]) => replace u
      | APPLYP (u,args) => APPLYP (replace u, map replace args)
      | APPLYS (p,[]) => replace p
      | APPLYS (p,s) => APPLYS (replace p,s)
      | APPLYF (p,[]) => replace p
      | APPLYF (p,s) => APPLYF (replace p,s)
      | LAMBDAS (s,p) => LAMBDAS (s,replace p)
      | LAMBDAP (s,p) => LAMBDAP (s,replace p)
      | SUBST (c,e,d,p,q) => SUBST (c,e,d,replace p,replace q)
  in
    replace p
 end

  (* same, but lazy computation of substituted proof *)
  fun replaceTheoremLazy (name:id) (q:unit -> proof) (p:proof) : proof = let
    fun replace (p:proof) =
      case p of
        CONST c => if c = name then q() else p
      | TASK (t,(ce,args,focus)) => TASK (t,(ce,map replace args,focus))
      | (VAR _ | AXIOM _) => p
      | APPLYP (u,[]) => replace u
      | APPLYP (u,args) => APPLYP (replace u, map replace args)
      | APPLYS (p,[]) => replace p
      | APPLYS (p,s) => APPLYS (replace p,s)
      | APPLYF (p,[]) => replace p
      | APPLYF (p,s) => APPLYF (replace p,s)
      | LAMBDAS (s,p) => LAMBDAS (s,replace p)
      | LAMBDAP (s,p) => LAMBDAP (s,replace p)
      | SUBST (c,e,d,p,q) => SUBST (c,e,d,replace p,replace q)
  in
    replace p
  end

(***********************************************
 * alpha-reduction
 **********************************************)

  (* substitution of a proof q for a cited theorem in a proof p *)
  fun replaceVar (name:id) (q:id) (p:proof) : proof = let
    fun replaceString x y z  = if x = z then y else z
    fun replace (p:proof) =
      case p of
        VAR c => if c = name then VAR(q) else p
      | TASK (t,(ce,args,focus)) => TASK (t,(ce,map replace args,focus))
      | (CONST _ | AXIOM _) => p
      | APPLYP (u,[]) => replace u
      | APPLYP (u,args) => APPLYP (replace u, map replace args)
      | APPLYS (p,[]) => replace p
      | APPLYS (p,s) => APPLYS (replace p,s)
      | APPLYF (p,[]) => replace p
      | APPLYF (p,s) => APPLYF (replace p,s)
      | LAMBDAS (s,p) => LAMBDAS (s,replace p)
      | LAMBDAP (s,p) => LAMBDAP (map (replaceString name q) s,replace p)
      | SUBST (c,e,d,p,q) => SUBST (c,e,d,replace p,replace q)
  in
    replace p
 end
fun getPs (p:proof) : id list =
      case p of
        LAMBDAP (ids,_) => ids
      | LAMBDAS (_,prf) => getPs(prf)
      | _ => []

fun alphaRedux(p:proof):proof = let
  val ps = getPs(p)
  fun makereplace(id,(proof,num)) =
    ((replaceVar id ("P"^Int.toString(num)) proof),num+1)
in
  #1(foldl makereplace (p,0) ps)
end


(***********************************************
 * normalize
 ***********************************************)

  (* check consistency -- variables in a lambda-binding are the
   * same as those in the substitution to which it is applied *)
  fun lambdaSConsis (v:id list) (s:substitution) : bool =
    length v = length s andalso List.all isSome (map (fn x => find x s) v)
  fun lambdaPConsis (v:id list) (p:proof list) : bool =
    length v = length p

  (* apply a substitution to a proof term *)
  fun applySubstToProof (s:substitution) (p:proof) : proof = let
    fun apply (p:proof) =
      case p of
        (VAR _ | CONST _ | AXIOM _) => p
      | TASK (t,(ce,args,_)) => let
          val ce = makeConstantCondEqn (Unify.applyToCondEqn s (makeVariableCondEqn ce))
          in TASK (t,(ce,map apply args,[]))
          end
      | APPLYP (u,[]) => apply u
      | APPLYP (u,args) => APPLYP (apply u, map apply args)
      | APPLYS (q,[]) => apply q
      | APPLYS (q,t) => APPLYS (apply q, Unify.compose t s)
      | APPLYF (q,[]) => apply q
      | APPLYF (q,t) => APPLYF (apply q, t)
      | LAMBDAS (t,q) => let
          val s = deleteAll t s (* don't subst in bound vars *)
          in
            case s of [] => p
            | _ => LAMBDAS (t,applySubstToProof s q)
          end
      | LAMBDAP (t,q) => LAMBDAP (t,apply q)
      | SUBST (c,e,d,q,r) => SUBST (c,e,d,apply q,apply r)
  in
    apply p
  end

  (* substitute proofs for proof variables *)
  fun substProofs (proofs:proof list) (vars:id list) (p:proof) : proof = let
    val s = ListPair.zip (vars,proofs)
    fun apply (s:(id * proof) list) (p:proof) =
      case p of
        VAR x => (case lookup x s of SOME y => y | _ => p)
      | (CONST _ | AXIOM _) => p
      | TASK (t,(ce,args,focus)) => TASK (t,(ce,map (apply s) args,focus))
      | APPLYP (u,[]) => apply s u
      | APPLYP (u,args) => APPLYP (apply s u, map (apply s) args)
      | APPLYS (q,[]) => apply s q
      | APPLYS (q,t) => APPLYS (apply s q,t)
      | APPLYF (q,[]) => apply s q
      | APPLYF (q,t) => APPLYF (apply s q,t)
      | LAMBDAS (t,q) => LAMBDAS (t,apply s q)
      | LAMBDAP (t,q) => let
          val s = deleteAll t s (* don't subst in bound vars *)
          in
            case s of [] => p
            | _ => LAMBDAP (t,apply s q)
          end
      | SUBST (c,e,d,q,r) => SUBST (c,e,d,apply s q,apply s r)
  in
    apply s p
  end

  (* reduce proof term to normal form *)
  fun normalize (p:proof) : proof = let
    fun existsRedux (p:proof) : bool =
      case p of
        (VAR _ | CONST _ | AXIOM _) => false
      | TASK (_,(_,args,_)) => List.exists existsRedux args
      | (APPLYP (LAMBDAP _,_) | APPLYS (LAMBDAS _,_)) => true
      | APPLYP (u,args) => existsRedux u orelse List.exists existsRedux args
      | (APPLYF (q,_) | APPLYS (q,_) | LAMBDAS (_,q) | LAMBDAP (_,q)) => existsRedux q
      | SUBST (_,_,_,q,r) => existsRedux q orelse existsRedux r
    fun reduceOne (p:proof) : proof =
      case p of
        (VAR _ | CONST _ | AXIOM _) => p
      | TASK (t,(ce,args,focus)) => TASK (t,(ce,map reduceOne args,focus))
      | APPLYP (LAMBDAP (t,q),args) =>
          if (lambdaPConsis t args) then substProofs args t q
          else raise Fail "system error: improper lambda binding"
      | APPLYP (u,[]) => reduceOne u
      | APPLYP (u,args) => APPLYP (reduceOne u, map reduceOne args)
      | APPLYS (LAMBDAS (t,q),s) =>
          if (lambdaSConsis t s) then applySubstToProof s q
          else raise Fail "system error: improper lambda binding"
      | APPLYS (q,[]) => reduceOne q
      | APPLYF (q,[]) => reduceOne q
      | APPLYS (q,t) => APPLYS (reduceOne q,t)
      | APPLYF (q,t) => APPLYF (reduceOne q,t)
      | LAMBDAS (t,q) => LAMBDAS (t,reduceOne q)
      | LAMBDAP (t,q) => LAMBDAP (t,reduceOne q)
      | SUBST (c,e,d,q,r) => SUBST (c,e,d,reduceOne q,reduceOne r)
  in
    if existsRedux p then normalize (reduceOne p)
    else p
  end
  handle Fail x => (println x; p)

(***********************************************
 * reset
 ***********************************************)

  (* create a new proof term for a newly published theorem *)
  fun reset (thm:cond_eqn,taskId:id) : proof = let
    val (pre,con) = thm
    val proofIds = map (fn x => newProofId()) pre
    val proofVars = map VAR proofIds
    val task = TASK (taskId,(makeConstantCondEqn thm,proofVars,[]))
    val inner =
      case proofIds of
        [] => task
      | _ => LAMBDAP (proofIds, task)
    val vars = condEqnVariables thm
  in
    case vars of
      [] => inner (* not too likely *)
    | _ => LAMBDAS (vars, inner)
  end

(***********************************************
 * cite
 ***********************************************)

  (* create proof term corresponding to a citation of a published theorem
   * p is the proof of the current theorem, q is the proof of the cited theorem,
   * s is the substitution, taskId is the old task, newTaskIds are the new tasks *)
  fun cite (p:proof, q:proof, s:substitution, taskId:id, newTasks:task list) =
    let
      val s = makeVariableSubst s
      val spec = case s of [] => q | _ => APPLYS (q,s)
      val newProof = case newTasks of
        [] => spec
      | _ => APPLYP (spec, map TASK newTasks)
    in
      replaceTask newProof taskId p
    end

  fun citeFocused (p:proof, q:proof, s:substitution, taskId:id, focus:int list,
                   dir:direction, specCon:eqn, newCurrentTask:task,
                   newTasks:task list) : proof = let
    val s = makeVariableSubst s
    val spec = case s of [] => q | _ => APPLYS (q,s)
    val applyNewTasks = case newTasks of
      [] => spec
    | _ => APPLYP (spec, map TASK newTasks)
    val newProof = SUBST(focus,makeVariableEqn(specCon),dir,applyNewTasks,TASK newCurrentTask)
  in
    replaceTask newProof taskId p
  end

(***********************************************
 * contains
 **********************************************)
fun contains(proof:proof,offLimits:string list): bool =
  case proof of
    (LAMBDAS(_,p1) | LAMBDAP(_,p1)) => contains(p1,offLimits)
  | (APPLYS(p1,_) | APPLYF(p1,_)) => contains(p1,offLimits)
  | APPLYP(p1,pl) => contains(p1,offLimits) orelse (foldl (fn (p,l) => contains(p,offLimits) orelse l) false pl)
  | VAR(id) => member id offLimits
  | CONST(id) => member id offLimits
  | AXIOM(_) => false
  | SUBST(_,_,_,p1,p2) => contains(p1,offLimits) orelse contains(p2,offLimits)
  | TASK(_,(_,pl,_)) => List.exists (fn p => contains(p,offLimits)) pl


fun allThms(proof:proof):id list =
  case proof of
    (LAMBDAS(_,p1) | LAMBDAP(_,p1)) => allThms(p1)
  | (APPLYS(p1,_) | APPLYF(p1,_)) => allThms(p1)
  | APPLYP(p1,pl) => allThms(p1) @ (foldl (fn (p,l) => allThms(p) @ l) [] pl)
  | VAR(id) => []
  | CONST(id) => if (let val letter = String.extract (id, 0, SOME 1)
                         val letters = String.extract (id, 0, SOME 2)
                         val number = Int.fromString(String.extract (id, 1, NONE))
                         val number = case number of SOME(x) => x | NONE => ~1
                     in (letter = "S" andalso number > 0 andalso number < 8) orelse letters = "FA" orelse letters = "Ar" end)
                  then [] else [id]
  | AXIOM(_) => []
  | SUBST(_,_,_,p1,p2) => allThms(p1) @ allThms(p2)
  | TASK(_,(_,pl,_)) => foldl (fn (p,l) => allThms(p) @ l) [] pl


(***********************************************
 * rename
 ***********************************************)

  fun rename (oldName:id,newName:id) (proof:proof ref) : unit =
    proof := replaceTheorem oldName (CONST newName) (!proof)

(***********************************************
 * forget a theorem
 ***********************************************)

  (* substitute proof terms for constant name of theorem being forgotten *)
  (* need new unique task ids in each copy *)
  fun forget (name:id, substProof:proof, inProof:proof) : proof = let
    fun renameTasks (p:proof) : proof =
      case p of
        TASK (_,(ce,args,focus)) =>
          TASK (newTaskId(),(ce,map renameTasks args,focus))
      | (VAR _ | CONST _ | AXIOM _) => p
      | APPLYP (q,[]) => renameTasks q
      | APPLYP (q,args) => APPLYP (renameTasks q, map renameTasks args)
      | APPLYS (q,[]) => renameTasks q
      | APPLYF (q,[]) => renameTasks q
      | APPLYS (q,s) => APPLYS (renameTasks q,s)
      | APPLYF (q,s) => APPLYF (renameTasks q,s)
      | LAMBDAS (s,q) => LAMBDAS (s,renameTasks q)
      | LAMBDAP (s,q) => LAMBDAP (s,renameTasks q)
      | SUBST (c,e,d,q,r) => SUBST (c,e,d,renameTasks q,renameTasks r)
  in
    replaceTheoremLazy name (fn _ => renameTasks substProof) inProof
  end

(***********************************************
 * read proof from XML representation
 ***********************************************)

  open XML

  fun proofFromXML (xml:xml) : proof = let
    fun fromXML (xml:xml) : proof =
      case xml of
        ELEMENT("VAR",_) => VAR (getString("VAR",xml))
      | ELEMENT("TASK",_) => let
          val (id,condEqn,proofs,focus) =
            case getContent("TASK",xml) of
              i::ce::f::pl => (i,ce,pl,f)
            | _ => raise Fail "could not parse task"
          val id = getString("id",id)
          val condEqnString = getString("condeqn",condEqn)
          val condEqn =
            case parseCondEqn condEqnString of
              SOME x => makeConstantCondEqn x
            | NONE => raise Fail ("could not parse task " ^ condEqnString)
          val proofs = map fromXML proofs
          val _ = (* sanity check *)
            if length (#1 condEqn) = length proofs then ()
            else raise Fail ("number of premises does not match number of proofs in task " ^ condEqnString)
          val focus = getString("focus",focus)
          val focus = map (valOf o Int.fromString) (String.tokens Char.isSpace focus)
        in TASK (id,(condEqn,proofs,focus))
        end
      | ELEMENT("LAMBDAS",_) => let
          val (boundVars,proof) =
            case rev (getContent("LAMBDAS",xml)) of
              p::bv => (rev bv,p)
            | _ => raise Fail "could not parse proof"
          val proof = fromXML proof
          val boundVars = map (fn x => getString("id",x)) boundVars
          in LAMBDAS (boundVars,proof)
          end
      | ELEMENT("LAMBDAP",_) => let
          val (boundVars,proof) =
            case rev (getContent("LAMBDAP",xml)) of
              p::bv => (rev bv,p)
            | _ => raise Fail "could not parse proof"
          val proof = fromXML proof
          val boundVars = map (fn x => getString("id",x)) boundVars
          in LAMBDAP (boundVars,proof)
          end
      | ELEMENT("APPLYS",_) => let
          val (proof,subst) =
            case getContent("APPLYS",xml) of
              p::s => (p,s)
            | _ => raise Fail "could not parse proof"
          val proof = fromXML proof
          fun subFromXML (sub:xml) = let
            val (id,term) = case getContent("sub",sub) of
              [i,t] => (i,t)
            | _ => raise Fail "could not parse proof"
            val id = getString("id",id)
            val term = getString("term",term)
            val term = (valOf(parseTerm term)
              handle Option => raise Fail "could not parse proof")
            in (id,term)
            end
          val subst : substitution = map subFromXML subst
          in APPLYS (proof,subst)
          end
      | ELEMENT("APPLYF",_) => let
          val (proof,subst) =
            case getContent("APPLYF",xml) of
              p::s => (p,s)
            | _ => raise Fail "could not parse proof"
          val proof = fromXML proof
          fun subFromXML (sub:xml) = let
            val (cterm,term) = case getContent("fsub",sub) of
              [i,t] => (i,t)
            | _ => raise Fail "could not parse proof"
            val id = getString("cterm",cterm)
            val id = Parser.parseLine(id)
            val term = getString("term",term)
            val term = (valOf(parseTerm term)
              handle Option => raise Fail "could not parse proof")
            in (id,term)
            end
          val subst : fosubstitution = map subFromXML subst
          in APPLYF (proof,subst)
          end
      | ELEMENT("APPLYP",_) => let
          val (f,args) =
            case getContent("APPLYP",xml) of
              f::a => (f,a)
            | _ => raise Fail "could not parse proof"
          val f = fromXML f
          val args = map fromXML args
          in APPLYP (f,args)
          end
      | ELEMENT("CONST",_) => CONST (getString("CONST",xml))
      | ELEMENT("AXIOM",_) => AXIOM (getString("AXIOM",xml))
      | ELEMENT("SUBST",_) => let
          val (focus,eqn,dir,p,q) =
            case getContent("SUBST",xml) of
              [f,e,d,p,q] => (f,e,d,p,q)
            | _ => raise Fail "could not parse proof"
          val focus = getString("focus",focus)
          val focus = map (valOf o Int.fromString) (String.tokens Char.isSpace focus)
            handle Option => raise Fail ("could not parse focus expression " ^ focus)
          val eqn = getString("eqn",eqn)
          val eqn =
            case parseEqn eqn of
              SOME e => e
            | _ => raise Fail ("could not parse equation " ^ eqn)
          val dir = getString("dir",dir)
          val dir =
            if dir = "L" then LEFT
            else if dir = "R" then RIGHT
            else raise Fail ("could not parse direction " ^ dir)
          val p = fromXML p
          val q = fromXML q
          in SUBST (focus,eqn,dir,p,q)
          end
      | ELEMENT(x,_) => raise Fail ("unknown xml element " ^ x)
      | _ => raise Fail "could not parse proof"
  in
    case getContent("proof",xml) of
      [x] => fromXML x
    | _ => raise Fail "could not parse proof"
  end

end