structure KAT_TK = struct

  open KAT
  open Unify
  open FOUnify
  open SmlTk



  (* IDs *)
  val mainwinID = newWinId()
  val libwinID = newWinId()
  val taskwinID = newWinId()
  val premwinID = newWinId()
  val suggwinID = newWinId()
  val tabbedwinID = newWinId()
  val reducewinID = newWinId()
  val prefWindowId = newWinId()
  val reducelistId = newWidgetId()
  val reduceEntryId = newWidgetId()
  val reducellistId = newWidgetId()
  val fowinID = newWinId()
  val entId = newWidgetId()
  val libEntId = newWidgetId()
  val rlnId = newWidgetId()
  val conclId = newWidgetId()
  val searchId = newWidgetId()
  val listId = mkWidgetId "libraryentry"
  val folistId = mkWidgetId "folibraryentry"
  val fotermlistId = mkWidgetId "fotermlist"
  val taskId = mkWidgetId "tasks"
  val premId = mkWidgetId "prementry"
  val foId = mkWidgetId "foentry"
  val foFileId1 = mkWidgetId "fofileentry1"
  val foFileId2 = mkWidgetId "fofileentry2"
  val fsuggId = mkWidgetId "fsugg"
  val usuggId = mkWidgetId "usugg"
  val mainBoxId = mkWidgetId "mainbox"
  val foBoxId1 = mkWidgetId "fobox1"
  val foBoxId2 = mkWidgetId "fobox2"
  val foNewTextId = mkWidgetId "fonewtext"
  val foNewPropId = mkWidgetId "fonewprop"
  val prefLibListId = newWidgetId();
  val prefAddListId = newWidgetId();
  val prefAddEntId = newWidgetId();

(* IDs for frames *)
  val tabframeID = newWidgetId()
  val libframeID = newWidgetId()
  val taskframeID = newWidgetId()
  val premframeID = newWidgetId()
  val suggframeID = newWidgetId()
  val reduceframeID = newWidgetId()

  val verbose = ref false

  (* Number of lines to display *)
  val dispLines = 24
  val dispWidth = 120


  (* Utility Functions *)

  (* Update focus printer *)
  fun  focusInEqnAsAnno(e,cf) = let
    val (lhs,rhs) = args e
    val s = eqnToString e
    val marks =
      case cf of
        [] => (Mark(2,0),Mark(2,size s))
      | 0::t => let val (i,j,_) = Term.focusSubterm(lhs,if t = [1] then [] else t)
               in (Mark(2,i),Mark(2,i+j)) end
      | 1::t => let
          val (i,j,_) = Term.focusSubterm(rhs,if t = [1] then [] else t)
          val rhsPos = size(toString lhs) + size(Term.opToString e) + 2
        in
          (Mark(2,i+rhsPos),Mark(2,i+rhsPos+j))
        end
      | _ => raise Fail "system error: focus does not exist"
   in
    AnnoText{len=NONE,str=("\n"^s),annotations=[TATag{annId=newAnnotationId(),bindings=[],
          configs=[Foreground(Blue)],
          marks=[marks]}]}
   end

  (* Updates to output functions *)
  val () = alert := UW.error
  val () = done := UW.info

  (* Update to resolveSubst function
   * Impossible to do citation in one step,
   * have to resolve ambiguity and then ask
   * user to cite again. 
   * Prompt user with dialog box listing all possible
   * unifications, ask for choice. choseSub keeps track
   * of the last ambiguity resolved          *)
  val choseSub = ref ~1
  exception Continue
  val () = resolveSubst := (fn s =>
     (if (!choseSub = ~1) then
     (let 
        val askWin = newWinId();
        (* Create a button for each possible response *)
        fun mkbuttons(n,responses) =
          case responses of
            nil => nil
          | (x::xs) => (Button{widId=newWidgetId(),packings=[Side Left],
                               configs=[Text x,
                       Command (fn () => (closeWindow askWin; choseSub := n))],
                               bindings=[]})::mkbuttons(n+1,xs)
        val z = List.tabulate(length s,Int.toString)

        (* Create message for box listing all possibilities *)
        val outstring = "specify desired bindings by number and then cite again:\n"^
        (foldl (fn ((n,x),s) => s^n^":"^substToString(x)^"\n") 
         "" (ListPair.zip(z,s)))

        val buttons = mkbuttons(0,z)
        val msg = Message{widId= newWidgetId(),
 		      packings= [Side Top, Fill Both],
                      configs= [Text outstring],
                      bindings= []}
        val frm = Frame{widId= newWidgetId(),
                     widgets= Pack buttons, 
                     packings= [Side Top, Fill X, Expand true], 
                     configs= [], bindings= []}
       in
        (* Open window w/buttons *)
        ((openWindow (mkWindow{winId    = askWin, 
  			config   =[WinTitle "Ambiguous Substitution",
  				   WinTransient NONE], 
                        widgets  = Pack [msg,frm], 
                        bindings = [],
			init     = noAction})); NONE)
	
  
     end)
     (* If the last citation resolved an ambiguity,
      * use it in this citation if an ambiguity exists *)
     else
       (let val subst = !choseSub
            val () = choseSub := ~1
        in
               SOME (List.nth(s,subst))
        end)))


  (* Update to printtext function
   * Put last x lines in main box on screen,
   * where x is the height of the text area
   * Cannot do more because auto-scrolling does
   * not work.  All output is still in string, just
   * not displayed                                  *)
  val boxString = ref ""
  val last40lines = ref ""
  val currentlength = ref 0
  val () = printtext := (fn x => let
    fun shrink(x) = if (String.size(x) > dispWidth andalso not(!verbose))
                    then (String.substring(x,0,(dispWidth-3) div 2)^"..."^
                          String.substring(x,(String.size(x)-(dispWidth-3) div 2),(dispWidth-3) div 2))
                    else if (String.size(x) > dispWidth)
                    then String.substring(x,0,dispWidth)^"\n"^
                         shrink(String.substring(x,dispWidth,String.size(x)-dispWidth))
                    else x
    val () = boxString := (!boxString)^x
    val allstrings = String.fields (fn x => x = #"\n") (!last40lines^(x))
    val newstring = String.fields (fn x => x = #"\n") x
    val listlength = List.length(allstrings)
    val lastfocus = List.length(allstrings) - 4
    val allstrings = if (listlength > 4 andalso List.length(newstring) > 2 andalso
                         (List.nth(allstrings,lastfocus) <> "") andalso
                        (List.all (fn x => (x = #"-" orelse x = #" " orelse x = #".")) 
                        (explode(List.nth(allstrings,lastfocus)))))
                     then (List.take(allstrings,lastfocus - 2) @ List.drop(allstrings,lastfocus + 1))
                     else allstrings
    val () = last40lines := foldr (fn (x,y) => (shrink(x))^"\n"^y) ("")
                                  (if listlength > dispLines 
                                   then (List.drop(allstrings,listlength-dispLines))
                                   else allstrings)
  in
    (replaceAnnoText mainBoxId (mkAT(!last40lines));
     currentlength := (!currentlength) + String.size(x) + 1)
  end)

  (* List of all closeable windows *)
  val closeableWins:(string*WinId*Window) list ref = ref nil


  (* Get the name of a theorem from a string *)
  fun getTheoremName (s) = let
    fun findColon(n) = if (String.sub(s,n) = #":") then n else findColon(n+1)
    handle Subscript => (~1)
    val length = findColon(0) - 1
  in
    if length > 0 then substring(s,1,length) else "TheoremDoesNotExist"
  end

  (* Set the library window to display all theorems
   * currently in the library.  Limit to ones matching
   * the search string, if it is specified            *)
  fun initialLibrary _ = let val () = clearText listId;
                             val () = outputstring := nil
                             val searchStr = readTextAll searchId
                             val () = viewLibrary([searchStr])
                         in
                             (app (insertTextEnd listId) (!outputstring); 
                              outputstring := nil)
                         end

  fun initialFOLibrary() = let val () = clearText folistId;
                             val () = outputstring := nil
                             val () = viewfoLibrary([])
                         in
                             (app (insertTextEnd folistId) (!outputstring); 
                              outputstring := nil)
                         end


  val currentWin = ref libframeID
  val currentWinS = ref "Library"

  (* Commands to fill in reduce window *)
  fun reduceWinFill() = let val () = clearText reducelistId;
                          val () = clearText reducellistId;
                          val redentries = (KAT.reduce["list"]; !outputstring)
                          val libentries = (#1 (ListPair.unzip (!KAT.library)))
                       in
                          (app (insertTextEnd reducelistId) (redentries); 
                           app (insertTextEnd reducellistId) (libentries); 
                           outputstring := nil)
                       end



  fun forAllCommandsNC () =  
   (
    (* Reset the alert function to prevent error messages
     * with commands done automatically *)

    alert := (fn x => ());
    if ((!currentWin = libframeID) orelse occursWin(libwinID)) then initialLibrary() else ();
    outputstring := nil;
    if occursWin(fowinID) then initialFOLibrary() else ();
    outputstring := nil;
    if ((!currentWin = premframeID) orelse occursWin(premwinID)) then (clearText premId;
    (* Display the premises *)
    premises();
    clearText premId;
    app (insertTextEnd premId) (!outputstring);
    outputstring := nil;
    conclusion();
    clearText conclId;
    app (insertTextEnd conclId) (!outputstring);
    outputstring := nil)
    else ();

    (* Display the tasks *)
    if ((!currentWin = taskframeID) orelse occursWin(taskwinID)) then
    (viewTasks();
    clearText taskId;
    app (fn y => insertTextEnd taskId 
        (mapFirstAndLast op^ 
        (String.fields (fn x => x = #"\n") y))) (!outputstring);
    outputstring := nil) else ();
    (* Display first-order equivalences *)
    viewFOTable();
    clearText foId;
    app (insertTextEnd foId) (!outputstring);
    outputstring := nil;
    doHeuristics();

    doSuggest();

    (* If there were suggested citations, display them *)
    if ((!currentWin = suggframeID) orelse occursWin(suggwinID)) then
    (clearText fsuggId;
    (if (List.length(!fsuggest) > 1) 
     then app (insertTextEnd fsuggId) (tl(!fsuggest)) 
     else ());
    clearText usuggId;
    (if (List.length(!usuggest) > 1)
     then app (insertTextEnd usuggId) (tl(!usuggest))
     else ())) else ();
    fsuggest := nil;
    usuggest := nil;
    (* Get Reductions List *)
    (if (!currentWin = reduceframeID) then reduceWinFill() else ());
    alert := UW.error)

  (* What to do after any command is run *)
  fun forAllCommands () =  
   (
    forAllCommandsNC();
    (* Print out the current task, if it exsists *)
    conclusion();
    (if focusing() then let
      val annotext = case !(getCurrentTask()) of
          SOME (_,((_,e),_,focus)) => focusInEqnAsAnno(e,focus)
        | NONE => mkAT("")
     in
        (replaceAnnoText mainBoxId (++ ((mkAT(!last40lines)),annotext)))
     end
    else if !outputstring <> nil 
          then         (replaceAnnoText mainBoxId (++ ((mkAT(!last40lines)),mkAT("\n"^(hd (rev(!outputstring)))))))
          else ())
   )










  (* Initialization function when prognam starts.
   * Set all radio buttons to left side,
   * make text area read-only,
   * Set debugging value, in case the user set this
   * in the config file                             *)
  fun initialize() = (readInitFile();
                    setVarValue "side" "l";
                    setVarValue "foside" "l";
                    setVarValue "redside" "l";
                    setVarValue "sside" "l";
                    setVarValue "mside" "l";
                    setVarValue "libWin" "1";
                    setVarValue "premWin" "1";
                    setVarValue "suggWin" "1";
                    setVarValue "taskWin" "1";
                    setVarValue "foWin" "1";
                    setVarValue "pside" "l";
                    setVarValue "verbose" "0";
                    setTextWidReadOnly mainBoxId true;

                    forAllCommands();
                    (if (!debugging) then setVarValue "debug" "1"
                     else setVarValue "debug" "0");
                    (if (!suggestOn) then setVarValue "suggest" "1"
                     else setVarValue "suggest" "0"))
                          


  (* Change ask2 function to display dialog box *)
  val () = ask2 := (fn (query:string, responses:string list) =>  fn cont => let
    val return:int option ref = ref (SOME ~1)
    val askWin = newWinId();
    fun mkbuttons(n,responses) =
      case responses of
        nil => nil
      | (x::xs) => (Button{widId=newWidgetId(),packings=[Side Left],
                           configs=[Text x,
                           Command (fn () => (closeWindow askWin;
                           return := SOME n;
                           cont (!return)(*;
                           forAllCommands()*) ))],
                           bindings=[]})::mkbuttons(n+1,xs)
     val buttons = mkbuttons(0,responses)
     val msg = Message{widId= newWidgetId(),
 		      packings= [Side Top, Fill Both],
                      configs= [Text query],
                      bindings= []}
     val frm = Frame{widId= newWidgetId(),
                     widgets= Pack buttons, 
                     packings= [Side Top, Fill X, Expand true], 
                    configs= [], bindings= []}
  in
     (openWindow (mkWindow{winId    = askWin, 
  			config   =[WinTitle "Clarify",
  				   WinTransient NONE], 
                        widgets  = Pack [msg,frm], 
                        bindings = [],
			init     = noAction}))
  end)


  (* Update getMissing function to display input dialog box for
   * user to fill in holes                                      *)
  val () = getMissing := (let
    fun getMiss(ses:id list) (missedterms:term list) (cont:term list -> unit) =
    (case ses of
      [] => (cont (missedterms) (*; forAllCommands()*))
    | (s::rest) => (let
        val myCont = (fn inputLine => (let
          val tokenizedInput = String.tokens Char.isSpace inputLine
          val _ = if tokenizedInput = [] 
                  then raise Fail "operation canceled"
                  else ()
          val term = parseTerm (concat (mapAllButLast (fn x => x^" ") 
                                tokenizedInput))
         in
          case term of
            SOME t => (getMiss rest ((makeConstant t)::missedterms) cont)
          | NONE => (println "invalid term, try again";
                     getMiss ses missedterms cont)
         end))
       in
         UW.enterLine{title="Specify term",
                     prompt=s^"=?",
                     default="",
                     width=50,
                     cc=myCont}
      end))
   in
     getMiss
   end)

val () = getMissingFO := 
  (fn list => fn cont =>
    (let fun myCont list = case list of
                           nil => cont ()
                         | (x::xs) => (UW.enterLine{title="Specify term",
                                          prompt=(aTermToString(Loc(x)))^"=?",
                                          default="",
                                          width=50,
                                          cc=(fn line => let 

                              val Math(parsed) = if line = "" then Math(Loc(x))
                                                 else FirstOrder.flatten(Parser.parseLine ("#"^line))
                              val () = unifyA(Loc(x),parsed)
                           in (myCont xs)
                           end)})
     in
         (myCont list)
     end))


  (* Actions for all buttons *)

  (* Commands with an input window *)
    fun conSave sel =
	 if sel <> "" then KAT.save([sel])
         else ()
    fun conSaveAll sel =
	 if sel <> "" then KAT.saveall([sel])
         else ()
    fun conSaveWork toSave sel = 
	 if sel <> "" then (let val outstream = TextIO.openOut(sel)
                                    in
                                        (TextIO.outputSubstr(outstream,Substring.all toSave);
                                         TextIO.closeOut outstream)
                                    end)
	   else ()
    fun conOutput sel =
	 if sel <> "" then KAT.outputLaTeX([sel])
         else ()

    fun conDump sel =
	 if sel <> "" then KAT.dumpToFile([sel])
         else ()

    fun conLoadWork window sel = 
	 if sel <> "" then (let val instream = TextIO.openIn(sel)
                                    in
                                         window := TextIO.inputAll(instream)
                                    end)
	   else ()


    fun conLoad contFunFun sel =
      if sel <> "" then (KAT.load([sel]); contFunFun())
      else ()

    fun conLoadDump contFunFun sel =
      if sel <> "" then (KAT.readDump([sel]); contFunFun())
      else ()


    (* Continuation and refs for parsing a program from a file
     * For some reason, using a filer in the continuation
     * doesn't work
     *)
    val fileList : string ref = ref ""

    fun conParse window textwin sel = 
     if sel <> "" then
       (let
         val instream = TextIO.openIn(sel)
         handle exn => raise Fail "No such file"
       in
         (SmlTk.insertTextEnd window (sel);
          SmlTk.replaceAnnoText textwin (SmlTk.mkAT(TextIO.inputAll(instream))))
       end)
    handle Fail x => (!alert) x
     else ()

    fun conGoto sel =
	 ((if sel <> "" then KAT.goto([sel])
         else KAT.goto []); forAllCommands())

(*    fun save () =  UW.enterLine{title="Save Theorem",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=conSave}
    fun saveAll () = UW.enterLine{title="Save All Theorems",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=conSaveAll}
    fun saveWork (toSave) = UW.enterLine{title="Save Work",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=(conSaveWork toSave)}
    fun loadWork (window) = UW.enterLine{title="Load Work",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=(conLoadWork window)}

    fun load (allFunFun) = UW.enterLine{title="Load Theorem(s)",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=(conLoad allFunFun)}

    fun parse window textwin = UW.enterLine{title="Load First-Order Term",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=(conParse window textwin)}
    fun goto () =  UW.enterLine{title="Find",
                     prompt="Term?",
                     default="",
                     width=50,
                     cc=conGoto}*)


  fun pushForget () = let val () = outputstring := nil
                          val mark = readCursor listId
                          val function = readText listId (mark, mark)
                          val funname = getTheoremName function
                          val () = forget([funname]);
                      in
                          forAllCommands()
                      end

  fun pushIgnore () = let val () = outputstring := nil
                          val mark = readCursor premId
                          val function = readText premId (mark, mark)
                          val funname = getTheoremName function
                          val () = ignore([funname]);
                      in
                          forAllCommands()
                      end

  fun pushSelect _ = let val () = outputstring := nil
                          val mark = readCursor listId
                          val function = readText listId (mark, mark)
                          val funname = getTheoremName function
                          val () = get [funname]
                          val () = viewCurrentTask()
                      in
                          forAllCommands()
                      end
  handle WIDGET _ => ()

  fun pushSelectTask () = let val () = outputstring := nil
                          val mark = readCursor taskId
                          val function = readText taskId (mark, mark)
                          val funname = getTheoremName function
                          val () = get [funname]
                          val () = viewCurrentTask()
                      in
                          forAllCommands()
                      end
  handle WIDGET _ => ()

  fun pushPublish _ = let val () = outputstring := nil
                           val newThm = readTextAll entId
                           val args = String.tokens (fn x => x = #" ") newThm
                           val () = publish args
                       in
                          ((clearText entId); forAllCommands())
                      end
  handle WIDGET _ => ()

  fun pushCut _ = let val () = outputstring := nil
                           val newThm = readTextAll entId
                           val args = String.tokens (fn x => x = #" ") newThm
                           val () = cut 0 args
                       in
                          ((clearText entId); forAllCommands())
                      end
  handle WIDGET _ => ()

  fun pushFocus () = let val () = outputstring := nil
                         val () = KAT.focus nil
                     in
                        forAllCommands()
                     end
  fun pushUnfocus () = let val () = outputstring := nil
                         val () = KAT.unfocus nil
                     in
                        forAllCommands()
                     end
  fun pushUp () = let val () = outputstring := nil
                         val () = KAT.up nil
                     in
                        forAllCommands()
                     end
  fun pushDown () = let val () = outputstring := nil
                         val () = KAT.down nil
                     in
                        forAllCommands()
                     end
  fun pushLeft () = let val () = outputstring := nil
                         val () = KAT.left nil
                     in
                        forAllCommands()
                     end
  fun pushRight () = let val () = outputstring := nil
                         val () = KAT.right nil
                     in
                        forAllCommands()
                     end
  fun pushCirc () = let val () = outputstring := nil
                     in
                        KAT.mCircCheck nil
                     end
  fun pushMove _ = let val () = outputstring := nil
                        val args = [(readVarValue "mside")]
                         val () = moveTerm(args);
                      in
                          forAllCommands()
                      end

  fun pushCite1 _ = let val () = outputstring := nil
                         val mark = readCursor listId
                         val function = readText listId (mark, mark)
                         val funname = getTheoremName function
                         val args = (funname::(if focusing()
                                               then ([(readVarValue "side")])
                                               else nil))
                      in
                          cite(args);forAllCommands()
                      end
  handle WIDGET _ => ()

  fun pushFOCite () = let val () = outputstring := nil
                         val mark = readCursor folistId
                         val function = readText folistId (mark, mark)
                         val funname = getTheoremName function
                         val args = (funname::(if focusing()
                                               then ([(readVarValue "foside")])
                                               else nil))
                      in
                          cite(args);forAllCommands()
                      end
  handle WIDGET _ => ()


  fun pushCite2 _ = let val () = outputstring := nil
                         val mark = readCursor premId
                         val function = readText premId (mark, mark)
                         val funname = getTheoremName function
                         val args = (funname::(if focusing()
                                               then ([(readVarValue "pside")])
                                               else nil))
                      in
                          use(args);forAllCommands()
                      end
  handle WIDGET _ => ()

  fun pushCite3 _ = let val () = outputstring := nil
                         val mark = readCursor fsuggId
                         val function = readText fsuggId (mark, mark)
                         val funname = getTheoremName function
                         val args = (funname::([(readVarValue "sside")]))
                      in
                         cite(args);forAllCommands()        
                      end
  handle WIDGET _ => ()

  fun pushCite4 _ = let val () = outputstring := nil
                         val mark = readCursor usuggId
                         val function = readText usuggId (mark, mark)
                         val funname = getTheoremName function
                         val () = unfocus nil;
                      in
                         cite([funname]);forAllCommands()
                      end
  handle WIDGET _ => ()

  fun pushNormalize () = let val () = outputstring := nil
                         val () = normalize([]);
                      in
                       forAllCommands()
                      end

  fun pushReduce _ = let val () = outputstring := nil
                          val () = KAT.reduce([]:string list);
                      in
                       forAllCommands()
                      end

  fun pushProof _ = let val () = outputstring := nil
                     in
                         (viewCurrentProof(nil); forAllCommands())
                     end

  fun pushOpen window textwin () = ((clearText window) ; 
                                   UW.enterLine{title="Load First-Order Term",
                                                prompt="File?",
                                                default="",
                                                width=50,
                                                cc=(conParse window textwin)})
  handle WIDGET _ => ()

(*  fun pushSave window textwin () = () *)





  (* Commands for menus *)
  fun stop _ = (closeWindow mainwinID;
                 closeWindow libwinID;
                 closeWindow taskwinID; 
                 closeWindow premwinID; 
                 closeWindow fowinID; 
                 closeWindow suggwinID;
		 KAT.quit())


  fun save _ = UW.enterLine{title="Save Theorem",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=conSave}

  fun output _ = UW.enterLine{title="Export Theorem",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=conOutput}

  fun dump _ = UW.enterLine{title="Dump Theorem",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=conDump}

  fun saveAll () = UW.enterLine{title="Save All Theorems",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=conSaveAll}
  fun load _ = UW.enterLine{title="Load Theorem(s)",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=(conLoad forAllCommands)}

  fun loadDump _ = UW.enterLine{title="Import commands",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=(conLoadDump forAllCommands)}
  fun newWork _ = let
                    val _ = boxString := ""
                    val _ = last40lines :=  ""
                    val _ = currentlength :=  0
                  in
                   (!printtext "")
                  end
  fun saveWork () = UW.enterLine{title="Save Work",
                     prompt="File?",
                     default="",
                     width=50,
                     cc=(conSaveWork (!boxString))}

  fun undoMenu _ = let val () = outputstring := nil
                         val () = undo([])
                      in
                       forAllCommands()
                      end
  fun redoMenu _ = let val () = outputstring := nil
                       val () = redo([])
                      in
                       forAllCommands()
                      end

  fun resetMenu() = let val () = outputstring := nil
                         val () = KAT.reset([])
                      in
                       forAllCommands()
                      end

  fun verifyMenu() = let val () = outputstring := nil
                         val () = KAT.verify([])
                      in
                       forAllCommands()
                      end

  fun deepverifyMenu() = let val () = outputstring := nil
                         val () = KAT.deepverify([])
                      in
                       forAllCommands()
                      end


  fun renameMenu() = let fun continue s =
                       let val () = outputstring := nil
                           val () = KAT.rename([s])
                       in
                         forAllCommands()
                      end
                     in
                       UW.enterLine({default="",
                                     width=14,
                                     prompt="New name:", 
                                     title="Rename",
                                     cc=continue})
                    end

  fun findMenu _ = UW.enterLine{title="Find",
                     prompt="Term?",
                     default="",
                     width=50,
                     cc=conGoto}
  
  fun findAgainMenu _ = conGoto ""



  fun defineMenu() = let fun continue s =
                       let val () = outputstring := nil
                           val () = KAT.define([s])
                       in
                         forAllCommands()
                      end
                     in
                       UW.enterLine({default="",
                                     width=14,
                                     prompt="New name:",
                                     title="Rename",
                                     cc=continue})
                     end


  fun changeHeuristics () = (app (fn (name,_) => heuristic([name,
                                                 case readVarValue(name) of
                                                    "0" => "off"
                                                  | "1" => "on"]))
                                                 (!heuristics);
                                                 forAllCommands())

  fun checkWindows () = (app (fn (x,y,z) =>
    case (readVarValue x) of
      "0" => if (occursWin(y)) then (closeWindow y) else ()
    | "1" => if (occursWin(y)) then ()
                               else (openWindow z))
    (!closeableWins); forAllCommands())

  fun changeDebug () = (debug [(if (readVarValue "debug") = "1" then "on" else "off")]; forAllCommands())

  fun changeRedLevel () = (reduce ("astbound"::(if (readVarValue "redlevel") = "1"
                                                then [(readVarValue "astbound")]
                                                else nil)); 
                           reduce ("timelimit"::(if (readVarValue "redtlevel") = "1"
                                                then [(readVarValue "timelimit")]
                                                else nil)); 
                           forAllCommands())

  fun changeVerbose () = ((verbose := let
                                        val verbval = (readVarValue "verbose")
                                      in
                                        if verbval = "1" then true else false
                                      end);
                          forAllCommands())

  fun getRedDepth() = case (!KAT.astBound) of
                        NONE => ((setVarValue "redlevel" "0");"")
                      | SOME(x) => ((setVarValue "redlevel" "1");Int.toString(x))
  fun getRedTime() = case (!KAT.timeLimit) of
                        NONE => ((setVarValue "redtlevel" "0");"")
                      | SOME(x) => ((setVarValue "redtlevel" "1");Real.toString(x))
  val numPrefLibs = ref 0
  val numPrefCommands = ref 0
  fun loadPrefs() =     let
      val instream = TextIO.openIn "../kat.ini"
        handle exn => (println "zz"; raise Fail "ww")
      val ()  = numPrefLibs := 0 
      val () = numPrefCommands := 0
      fun get() = Substring.all(TextIO.inputLine instream)
      fun getLines (s:string list list) : string list list =
        let val next = Substring.string (get())
            val input = String.tokens Char.isSpace next
        in if next = "" then rev s else getLines(input::s)
        end
      fun processLine (s) = if String.isPrefix "load" (hd(s))
                            then (numPrefLibs := !numPrefLibs + 1;
                                  insertTextEnd prefLibListId (hd(tl(s))))
                            else if (String.isPrefix "debug" (hd(s))  orelse
                                     String.isPrefix "reduce" (hd(s))  orelse
                                     String.isPrefix "heurist" (hd(s))  orelse
                                     String.isPrefix "suggest" (hd(s))  orelse
                                     String.isPrefix "verbos" (hd(s)))  then ()
                            else (numPrefCommands := !numPrefCommands +1;
                                  insertTextEnd prefAddListId 
                                 (foldr (fn (x,y) => x^" "^y) "" s))
      val initLines = getLines []
      val () = setVarValue "astbound" (getRedDepth())
      val () = setVarValue "timelimit" (getRedTime())
    in
      app processLine initLines
    end

  fun prefAddCommand() = (numPrefCommands := !numPrefCommands + 1;
                          insertTextEnd prefAddListId (readTextAll prefAddEntId))
  fun prefDelCommand() = let val mark = readCursor prefAddListId
                             val () = numPrefCommands := !numPrefCommands - 1
   in
    deleteText prefAddListId (mark,mark)
   end
   handle _ => numPrefCommands := !numPrefCommands + 1

  fun prefAddLibrary() = 
    let 
      val filename = if endsWith((readTextAll libEntId),".xml") 
                     then (readTextAll libEntId)
                     else ("../Lib/"^(readTextAll libEntId)^".xml")
      val instream = TextIO.openIn filename
      val () = TextIO.closeIn(instream)
      val () = numPrefLibs := !numPrefLibs + 1
     in
       insertTextEnd prefLibListId (readTextAll libEntId)
     end
     handle _ => (numPrefLibs := !numPrefLibs + 1;
                  (!alert) "No such library")

  fun prefDelLibrary() = let val mark = readCursor prefLibListId
                             val () = numPrefLibs := !numPrefLibs - 1
   in
    deleteText prefLibListId (mark,mark)
   end
   handle _ => numPrefLibs := !numPrefLibs + 1

  fun getTextItems l n = let
    fun getTextItems' c = if c < n
                         then (readText l (Mark(c,0),MarkToEnd(c)))::
                                (getTextItems' (c+1))
                         else []
  in
    getTextItems' 0
  end
  handle _ => []

  fun savePrefs() = let
    fun writePrefs() = let
      val outStream = TextIO.openOut("../kat.ini")
      fun put (s:string) = TextIO.outputSubstr(outStream,Substring.all s)
      fun putln (s:string) = put(s^"\n")
      fun onOrOff var = if (readVarValue var = "0") then "off" else "on"
      val miscCom = getTextItems prefAddListId (!numPrefCommands)
      val miscCom = map (replaceCharWS #"{" "") miscCom
      val miscCom = map (replaceCharWS #"}" "") miscCom
      val () = KAT.reduce["list"];
    in
      app (fn x => putln ("load "^x)) (getTextItems prefLibListId (!numPrefLibs));
      putln ("debug "^(onOrOff("debug")));
      putln ("suggest "^(onOrOff("suggest")));
(*      putln ("verbose "^(onOrOff("verbose")));*)
      app (fn (name,_) => putln ("heuristics "^name^" "^(onOrOff(name)))) (!heuristics);
      putln ("reduce astbound "^(if readVarValue "redlevel" = "0" then "" 
                              else (readVarValue "astbound")));
      putln ("reduce timelimit "^(if readVarValue "redtlevel" = "0" then "" 
                              else (readVarValue "timelimit")));
      app (fn x => putln ("reduce add "^x)) (!outputstring);
      app putln miscCom;
      TextIO.closeOut(outStream)
    end
  in
   (
    writePrefs();
    changeVerbose();
    changeDebug();  
    changeRedLevel();
    changeHeuristics();
    closeWindow prefWindowId
   )
  end

  fun upWrapper _ = (if ((readTextAll entId) = "") then pushUp() else ())
  fun downWrapper _ = (if ((readTextAll entId) = "") then (if focusing() 
                                                then pushDown() else pushFocus())
            else ())
  fun leftWrapper _ = (if ((readTextAll entId) = "") then pushLeft() else ())
  fun rightWrapper _ =  (if ((readTextAll entId) = "") then pushRight() else ())

 (* List of keyboard bindings *)
  val keyBinds = [
      BindEv(KeyPress("Return"),pushPublish), 
      BindEv(Ctrl(KeyPress("z")),undoMenu), 
      BindEv(Ctrl(KeyPress("y")),redoMenu), 
      BindEv(Ctrl(KeyPress("p")),pushProof),
      BindEv(Ctrl(KeyPress("c")),pushCut),
      BindEv(Ctrl(KeyPress("m")),pushMove),
      BindEv(Ctrl(KeyPress("f")),findMenu),
      BindEv(Ctrl(KeyPress("g")),findAgainMenu),
      BindEv(Ctrl(KeyPress("r")),pushReduce),
      BindEv(Ctrl(KeyPress("s")),save),
      BindEv(Ctrl(KeyPress("n")),newWork),
      BindEv(Ctrl(KeyPress("l")),load),
      BindEv(Ctrl(KeyPress("q")),stop),
      BindEv(KeyPress("Up"),upWrapper),
      BindEv(KeyPress("Down"),downWrapper),
      BindEv(KeyPress("Left"),leftWrapper),
      BindEv(KeyPress("Right"),rightWrapper)]

  fun setSideLWrapper _ = setVarValue "side" "l"
  fun setSideRWrapper _ = setVarValue "side" "r"

  fun setPSideLWrapper _ = setVarValue "pside" "l"
  fun setPSideRWrapper _ = setVarValue "pside" "r"

  fun setSSideLWrapper _ = setVarValue "sside" "l"
  fun setSSideRWrapper _ = setVarValue "sside" "r"

  fun cButWrap _ = case !currentWinS of
    "Library" => pushCite1()
  | "Premises" => pushCite2()
  | "Tasks" => ()
  | "Suggested Citations" => pushCite3()
  | "Reductions" => ()

  fun scButWrap _ = case !currentWinS of
    "Library" => ()
  | "Premises" => ()
  | "Tasks" => ()
  | "Suggested Citations" => pushCite4()
  | "Reductions" => ()

  fun sButWrap _ = case !currentWinS of
    "Library" => pushSelect()
  | "Premises" => ()
  | "Tasks" => pushSelectTask()
  | "Suggested Citations" => ()
  | "Reductions" => ()

  fun lButWrap _ = case !currentWinS of
    "Library" => setSideLWrapper()
  | "Premises" => setPSideLWrapper()
  | "Tasks" => ()
  | "Suggested Citations" => setSSideLWrapper()
  | "Reductions" => ()

  fun rButWrap _ = case !currentWinS of
    "Library" => setSideRWrapper()
  | "Premises" => setPSideRWrapper()
  | "Tasks" => ()
  | "Suggested Citations" => setSSideRWrapper()
  | "Reductions" => ()

  fun enterButWrap _ = case !currentWinS of
    "Library" => initialLibrary()
  | "Premises" => ()
  | "Tasks" => ()
  | "Suggested Citations" => ()
  | "Reductions" => ()

  val allKeyBinds = [
      BindEv(KeyPress("Return"),enterButWrap),
      BindEv(Ctrl(KeyPress("c")),cButWrap),  
      BindEv(Ctrl(KeyPress("s")),sButWrap),
      BindEv(Shift(Ctrl(KeyPress("c"))),scButWrap),
      BindEv(Ctrl(KeyPress("l")),lButWrap), 
      BindEv(Ctrl(KeyPress("r")),rButWrap),
      BindEv(Ctrl(KeyPress("q")),stop),
      BindEv(Double(ButtonPress(SOME(1))),cButWrap)]

  val allTabbedBindings:(WidId * Binding list) list ref = ref []
  val allTabbedFrames:(WidId * Widget) list ref = ref []
  val allTabbedWindowsID:(WidId * WinId) list ref = ref []
  val allTabbedText:(WidId * string) list ref = ref []
  val allTabbedWindows:(WidId * {winId : WinId,
 config : WinConfigure list,
 widgets : Widgets,
 bindings: Binding list,
 init : SimpleAction}) list ref = ref [];

  fun makeAWin thisWin  = let val newWindow = valOf(lookup (!currentWin) (!allTabbedWindows))
				val newWindowID = valOf(lookup (!currentWin) (!allTabbedWindowsID)) in
                  if List.length(!allTabbedFrames) > 1
                  then (if (thisWin = (!currentWin)) then delWidget (!currentWin) else ();
                        allTabbedFrames := List.filter (fn x => (#1 x) <> (thisWin)) (!allTabbedFrames);
                        if (not(occursWin(newWindowID))) then openWindow(mkWindow (newWindow)) else ();
                     if (thisWin = (!currentWin)) then (currentWin := #1(hd(!allTabbedFrames));
                     changeTitle tabbedwinID (mkTitle(valOf(lookup (!currentWin) (!allTabbedText))));
                     addWidget tabbedwinID tabframeID (valOf(lookup (!currentWin) (!allTabbedFrames)))) else ();
		     forAllCommandsNC()) else ()
                  end

  and switchTo newFrameID () = (*if newFrameID = (!currentWin) then makeAWin newFrameID
                               else*) 
                             (delWidget (!currentWin);
                              currentWin := newFrameID;
                              currentWinS := valOf(lookup newFrameID (!allTabbedText));
                              changeTitle tabbedwinID (mkTitle(valOf(lookup newFrameID (!allTabbedText))));
                              addWidget tabbedwinID tabframeID (valOf(lookup newFrameID (!allTabbedFrames))); 
                              forAllCommandsNC())


  (* Buttons for tabs *)
  val libTab = Button{widId=newWidgetId(),packings=[Side Left],
                          configs=[Text "Library", Command (switchTo libframeID)],
                          bindings=[]}
                                      

  val premTab = Button{widId=newWidgetId(),packings=[Side Left],
                          configs=[Text "Premises", Command (switchTo premframeID)],
                          bindings=[]}

  val taskTab = Button{widId=newWidgetId(),packings=[Side Left],
                          configs=[Text "Tasks", Command (switchTo taskframeID)],
                          bindings=[]}
                                      
  val suggTab = Button{widId=newWidgetId(),packings=[Side Left],
                          configs=[Text "Suggestions", Command (switchTo suggframeID)],
                          bindings=[]}

  val redTab = Button{widId=newWidgetId(),packings=[Side Left],
                          configs=[Text "Reductions", Command (switchTo reduceframeID)],
                          bindings=[]}

  val allTabs = Frame{widId=newWidgetId(),
                         widgets=Pack [libTab,premTab,taskTab,suggTab,redTab],
                         packings=[Side Top, Fill X],
                         configs=[],bindings=[]}



  (* Library Window *)
  val libLab = Label{widId=newWidgetId(), packings=[Side Top],
                configs=[Text "Library"], bindings=[]}

  val libList = Listbox{widId=listId, packings=[Side Top],
                configs=[Height 10,Width 40], bindings=[], scrolltype=RightScb}
  val citeButton = Button{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Cite",Command pushCite1],bindings=[]}
  val selectButton = Button{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Select", Command pushSelect],
                          bindings=[]}
  val forgetButton = Button{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Forget", Command pushForget],
                          bindings=[]}

  val leftRButton = Radiobutton{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Left", Variable "side", Value "l"],
                          bindings=[]}
  val rightRButton = Radiobutton{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Right", Variable "side", Value "r"],
                          bindings=[]}

  val searchWindow = Entry{widId=searchId,packings=[Side Top,Expand true],
                           configs=[Width 40],
                           bindings=[]}
  val searchButton = Button{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Search", Command initialLibrary],
                          bindings=[]}



  val librbutton = Frame{widId=newWidgetId(),
                         widgets=Pack [leftRButton,rightRButton],
                         packings=[Side Left, Fill X],
                         configs=[],bindings=[]}
  val libbbutton = Frame{widId=newWidgetId(),
                         widgets=Pack [selectButton,citeButton,forgetButton],
                         packings=[Side Left, Fill X],
                         configs=[],bindings=[]}
  val libbutton = Frame{widId=newWidgetId(),
                        widgets=Pack [libbbutton,librbutton],
                        packings=[Side Top],
                        configs=[],bindings=[]}


  val libFrame = Frame{widId = libframeID,
                   configs = [], packings=[Side Top],
                   widgets = Pack[libList,libbutton,searchWindow,searchButton],
                   bindings = []
                   }

  val libWindow = {winId = libwinID,
                   config = [WinTitle "Library"],
                   widgets = Pack[libFrame],
                   bindings = [],
                   init = noAction}

  (* Tasks Window *)
  val taskList = Listbox{widId=taskId,
                         packings=[Side Top],
                         configs=[Height 10,Width 40],
                         bindings=[],
                         scrolltype=RightScb}
  val tselectButton = Button{widId=newWidgetId(),
                             packings=[Side Top],
                             configs=[Text "Select", Command pushSelectTask],
                             bindings=[]}
  val taskWindow = {winId = taskwinID,
                   config = [WinTitle "Tasks"],
                   widgets = Pack [taskList,tselectButton],
                   bindings = [],
                   init = noAction}

  val taskFrame = Frame{widId = taskframeID,
                   configs = [],
                   packings = [Side Top],
                   widgets = Pack [taskList,tselectButton],
                   bindings = []}


  (* Premises Window *)
  val premLab = Label{widId=newWidgetId(),
                      packings=[Side Top],
                      configs=[Text "Premises"],
                      bindings=[]}
  val premList = Listbox{widId=premId,
                         packings=[Side Top],
                         configs=[Height 5,Width 40],
                         bindings=[],
                         scrolltype=RightScb}
  val useButton = Button{widId=newWidgetId(),
                         packings=[Side Top],
                         configs=[Text "Use",Command pushCite2],
                         bindings=[]}
  val ignoreButton = Button{widId=newWidgetId(),
                            packings=[Side Top],
                            configs=[Text "Ignore", Command pushIgnore],
                            bindings=[]}
  val leftPRButton = Radiobutton{widId=newWidgetId(),
                                 packings=[Side Top],
                                 configs=[Text "Left", Variable "pside",
                                          Value "l"],
                                 bindings=[]}
  val rightPRButton = Radiobutton{widId=newWidgetId(),
                                  packings=[Side Top],
                                  configs=[Text "Right", Variable "pside",
                                           Value "r"],
                                  bindings=[]}


  val premrbutton = Frame{widId=newWidgetId(),
                          widgets=Pack [leftPRButton,rightPRButton],
                          packings=[Side Left, Fill X],
                          configs=[],bindings=[]}
  val prembbutton = Frame{widId=newWidgetId(),
                          widgets=Pack [useButton,ignoreButton],
                          packings=[Side Left, Fill X],
                          configs=[],bindings=[]}
  val prembutton = Frame{widId=newWidgetId(),
                         widgets=Pack [prembbutton,premrbutton],
                         packings=[Side Top],
                         configs=[],bindings=[]}

  val premCLabel = Label{widId=newWidgetId(), packings=[Side Top,Expand false],
                configs=[Text "Conclusion:",Justify JLeft], bindings=[]}

  val premCBox = Entry{widId=conclId,
                          packings=[Side Top,Expand true],
                          configs=[Width 40],
                          bindings=[]}



  val premWindow = {winId = premwinID,
                   config = [WinTitle "Premises"],
                   widgets = Pack [premList,prembutton,premCLabel,premCBox],
                   bindings = [],
                   init = noAction}

  val premFrame = Frame{widId = premframeID,
                   configs = [],
                   widgets = Pack [premList,prembutton,premCLabel,premCBox],
                   packings = [Side Top],
                   bindings = []}



  val foLab = Label{widId=newWidgetId(),
                      packings=[Side Top],
                      configs=[Text "Translation Table"],
                      bindings=[]}
  

  (* First-Order Table Window *)

  fun parseFOLine () = let
    fun addBool (x,y) = insertTextEnd fotermlistId
                        (Util.replaceChar #"\n" #" "
                        (Term.toString(y)^" : "^FirstOrder.bTermToString(x)))
    fun addTerm (x,y) = insertTextEnd fotermlistId
                        (Util.replaceChar #"\n" #" " 
                        (Term.toString(y)^" : "^FirstOrder.cTermToString(x)))
    val newComm = readTextAll foNewTextId
    val transed = makeVariable(FirstOrder.slangCToKAT(flatten(Parser.parseLine newComm)))
    handle Continue => raise Fail "Unable to parse line."
  in
    clearText fotermlistId;
    clearText foNewTextId;
    app addBool (!FOUnify.boolList);
    app addTerm (!FOUnify.termList)
  end
  handle Fail x => (!alert) x

  fun parseFOFile () = let
    fun addBool (x,y) = insertTextEnd fotermlistId
                        (Util.replaceChar #"\n" #" "
                        (Term.toString(y)^" : "^FirstOrder.bTermToString(x)))
    fun addTerm (x,y) = insertTextEnd fotermlistId
                        (Util.replaceChar #"\n" #" " 
                        (Term.toString(y)^" : "^FirstOrder.cTermToString(x)))
    fun cont filename = let
      val transed = makeVariable(FirstOrder.slangCToKAT(flatten(Parser.parse filename)))
      handle Continue => raise Fail "Unable to parse file."
    in
      clearText fotermlistId;
      app addBool (!FOUnify.boolList);
      app addTerm (!FOUnify.termList)
    end
  in
    UW.enterLine{title="Load First-Order Term",
                 prompt="File?",
                 default="",
                 width=50,
                 cc=(cont)}
   end
  handle Fail x => (!alert) x

  fun resetFO() = (clearText fotermlistId;
                   clearText foNewPropId;
                  FirstOrder.reset())

  fun publishFO () = let val () = outputstring := nil
                           val newThm = readTextAll foNewPropId
                           val args = String.tokens (fn x => x = #" ") newThm
                           val () = KAT.publishFO args
                       in
                          ((clearText foNewPropId); forAllCommands())
                      end
  handle WIDGET _ => ()

  fun insertTerm _ = let 
    val mark = readCursor fotermlistId
    val function = readText fotermlistId (mark, mark)
    val funname = getTheoremName function
  in
    insertTextEnd foNewPropId funname
  end

  (* Label for following Entry *)
  val foNewTextLab = Label{widId=newWidgetId(), packings=[Side Left],
                configs=[Text "New Command:"], bindings=[]}

  (* Area to enter new first-order term *)
  val foNewText = Entry{widId=foNewTextId,
                        packings=[Side Left,Expand true],
                        configs=[Width 30],
                        bindings=[]}

  (* Parse new first-order term *)
  val foParseButton = Button{widId=newWidgetId(),
                            packings=[Side Left],
                            configs=[Text "Parse", Command parseFOLine],bindings=[]}

  (* Open first-order program *)
  val foOpenButton = Button{widId=newWidgetId(),
                            packings=[Side Left],
                            configs=[Text "Open", Command parseFOFile],bindings=[]}

  (* Clear first-order table *)
  val foClearButton = Button{widId=newWidgetId(),
                            packings=[Side Left],
                            configs=[Text "Reset", Command resetFO],bindings=[]}

  (* Above as a frame *)
  val foFrame1 = Frame{widId=newWidgetId(),
                          widgets=Pack [foNewTextLab,foNewText,
                                        foParseButton,foOpenButton,
                                        foClearButton],
                          packings=[Side Top, Fill X],
                          configs=[],bindings=[]}

  (* Table of terms *)
  val foTermsList = Listbox{widId=fotermlistId,
                            packings=[Side Top,Fill X],
                            configs=[Height 10,Width 40],
                            bindings=[BindEv(Double(ButtonPress(SOME(1))),
                                      insertTerm)],
                            scrolltype=RightScb}

  (* Label for following Entry *)
  val foNewPropLab = Label{widId=newWidgetId(), packings=[Side Left],
                configs=[Text "New Theorem:"], bindings=[]}

  (* New propositional line *)
  val foNewProp = Entry{widId=foNewPropId,
                        packings=[Side Left,Expand true,Fill X],
                        configs=[Width 40],
                        bindings=[]}

  (* Publish new theorem *)
  val foPubButton = Button{widId=newWidgetId(),
                           packings=[Side Left],
                           configs=[Text "Publish",Command publishFO],bindings=[]}

  val foFrame2 = Frame{widId=newWidgetId(),
                          widgets=Pack [foNewPropLab,foNewProp,foPubButton],
                          packings=[Side Top, Fill X],
                          configs=[],bindings=[]}



  val folibList = Listbox{widId=folistId, packings=[Side Top,Fill X,Expand true],
                configs=[Height 10,Width 40],
                bindings=[], scrolltype=RightScb}

  val foleftRButton = Radiobutton{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Left", Variable "foside", Value "l"],
                          bindings=[]}
  val forightRButton = Radiobutton{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Right", Variable "foside", Value "r"],
                          bindings=[]}

  val fociteButton = Button{widId=newWidgetId(),
                         packings=[Side Top],
                         configs=[Text "Cite",Command pushFOCite],
                         bindings=[]}

  val foWindow = {winId = fowinID,
                   config = [WinTitle "First-Order Terms"],
                   widgets = Pack [foFrame1,foTermsList,foFrame2,
                                   folibList,fociteButton,
                                   foleftRButton, forightRButton],
                   bindings = [      BindEv(Ctrl(KeyPress("q")),stop)],
                   init = initialFOLibrary}

  (* Suggestions Window *)
  val suggLab = Label{widId=newWidgetId(),
                      packings=[Side Top],
                      configs=[Text "Suggested Citations"],
                      bindings=[]}
  val fsuggLab = Label{widId=newWidgetId(),
                       packings=[Side Top],
                       configs=[Text "Focused"],
                       bindings=[]}
  val usuggLab = Label{widId=newWidgetId(),
                       packings=[Side Top],
                       configs=[Text "UnFocused"],
                       bindings=[]}
  

  val fsuggList = Listbox{widId=fsuggId,
                          packings=[Side Top],
                          configs=[Height 5,Width 40],
                          bindings=[],
                          scrolltype=RightScb}
  val usuggList = Listbox{widId=usuggId,
                          packings=[Side Top],
                          configs=[Height 5,Width 40],
                          bindings=[],
                          scrolltype=RightScb}
  val sciteButton = Button{widId=newWidgetId(),
                           packings=[Side Left],
                           configs=[Text "Cite", Command pushCite3],
                           bindings=[]}
  val leftSRButton = Radiobutton{widId=newWidgetId(),
                                 packings=[Side Top],
                                 configs=[Text "Left", Variable "sside",
                                          Value "l", Active true],
                                 bindings=[]}
  val rightSRButton = Radiobutton{widId=newWidgetId(),
                                  packings=[Side Top],
                                  configs=[Text "Right",
                                           Variable "sside", Value "r"],
                                  bindings=[]}
  val suciteButton = Button{widId=newWidgetId(),
                            packings=[Side Top],
                            configs=[Text "Cite",Command pushCite4],
                            bindings=[]}
  val suggrbutton = Frame{widId=newWidgetId(),
                          widgets=Pack [leftSRButton,rightSRButton],
                          packings=[Side Top, Fill X],
                          configs=[],bindings=[]}
  val suggbutton = Frame{widId=newWidgetId(),
                         widgets=Pack [sciteButton,suggrbutton],
                         packings=[Side Top],configs=[],
                         bindings=[]}

  val suggestWindow = {winId = suggwinID,
                       config = [WinTitle "Suggested Citations"],
                       widgets = Pack [fsuggLab,fsuggList,
                                       suggbutton,usuggLab,usuggList,
                                       suciteButton],
                       bindings = [],
                       init = noAction}

  val suggestFrame = Frame{widId = suggframeID,
                       configs = [],
                       widgets = Pack [fsuggLab,fsuggList,
                                       suggbutton,usuggLab,usuggList,
                                       suciteButton],
                       bindings = [],
                       packings=[Side Top]}


  (* Reduce Window *)
  fun pushRedAddField _ = let val mark = readCursor reducellistId
                          val thm = readText reducellistId (mark, mark)
                      in
			  insertTextEnd reduceEntryId (" "^thm^" "^(readVarValue "redside"))
                      end
  fun pushRedAdd _ =  let val thm = readTextAll reduceEntryId
                          val () = if thm <> ""
                                   then KAT.reduce ("add"::(String.tokens Char.isSpace thm))
                                   else ()
                      in
                          reduceWinFill()
                      end
  fun pushRedRem _ =  let val thm = readTextAll reduceEntryId
                          val () = if thm <> ""
                                   then KAT.reduce ("del"::(String.tokens Char.isSpace thm))
                                   else ()
                      in
                          reduceWinFill()
                      end

  fun pushRedEmpty _ =  let
                          val () = KAT.reduce (["empty"])
                      in
                          reduceWinFill()
                      end

  fun pushRedTry _ =  let val thm = readTextAll reduceEntryId
                          val () = if thm <> ""
                                   then KAT.reduce ("try"::(String.tokens Char.isSpace thm))
                                   else ()
                      in
                          reduceWinFill()
                      end

  fun getRedSide _ =  let val mark = readCursor reducelistId
                          val thm = readText reducelistId (mark, mark)
		          val () = clearText reduceEntryId
                      in
			 insertTextEnd reduceEntryId (String.extract(thm,1,SOME (String.size(thm)-2)))
                      end

  val redlibLab = Label{widId=newWidgetId(),
                      packings=[Side Top],
                      configs=[Text "Library"],
                      bindings=[]}
  val redLab = Label{widId=newWidgetId(),
                      packings=[Side Top],
                      configs=[Text "Reducers"],
                      bindings=[]}


  val redlibList = Listbox{widId=reducellistId, packings=[Side Top],
                configs=[Height 10,Width 20],
                bindings=[BindEv(Double(ButtonPress(SOME(1))),pushRedAddField)],
                scrolltype=RightScb}
  val redList = Listbox{widId=reducelistId, packings=[Side Top],
                configs=[Height 10,Width 20], 
                bindings=[BindEv(ButtonPress(SOME(1)),getRedSide),
                          BindEv(Double(ButtonPress(SOME(1))),pushRedRem)],
                scrolltype=RightScb}

  val redLibFrame = Frame{widId = newWidgetId(),
                       configs = [],
                       widgets = Pack [redlibLab,redlibList],
                       bindings = [],
                       packings=[Side Left]}

  val redFrame = Frame{widId = newWidgetId(),
                       configs = [],
                       widgets = Pack [redLab,redList],
                       bindings = [],
                       packings=[Side Left]}


  val redAddButton = Button{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "-->",Command pushRedAdd],bindings=[]}
  val redRemButton = Button{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "<--",Command pushRedRem],bindings=[]}
  val redleftButton = Radiobutton{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Left", Variable "redside", Value "l"],
                          bindings=[]}
  val redrightButton = Radiobutton{widId=newWidgetId(),packings=[Side Top],
                          configs=[Text "Right", Variable "redside", Value "r"],
                          bindings=[]}

  val reduceButton = Button{widId=newWidgetId(),
                          packings=[Side Left],
                          configs=[Text "Reduce", Command pushReduce],
                          bindings=[]}

  val redEmptyButton = Button{widId=newWidgetId(),packings=[Side Left],
                          configs=[Text "Empty",Command pushRedEmpty],bindings=[]}

  val redTryButton = Button{widId=newWidgetId(),packings=[Side Left],
                          configs=[Text "Try",Command pushRedTry],bindings=[]}

  val redButFrame = Frame{widId = newWidgetId(),
                       configs = [],
                       widgets = Pack [redAddButton,redRemButton,redleftButton,redrightButton],
                       bindings = [],
                       packings=[Side Left]}

  val reduceEntry = Entry{widId=reduceEntryId,packings=[Side Top,Expand true],
                           configs=[Width 40],
                           bindings=[]}
 
  val redButtonFrame = Frame{widId =newWidgetId(),
                             configs = [],
                             widgets = Pack [reduceButton,redTryButton,
                                             redEmptyButton],
                             bindings = [],
                             packings = [Side Top]}


  val redReduceFrame = Frame{widId = newWidgetId(),
                       configs = [],
                       widgets = Pack [redLibFrame,redButFrame,redFrame],
                       bindings = [],
                       packings=[Side Top]}

  val reduceWindow = {winId = reducewinID,
                   config = [WinTitle "Reductions"],
                   widgets = Pack [redReduceFrame,reduceEntry,redButtonFrame],
                   bindings = keyBinds,
                   init = reduceWinFill}

  val reduceFrame = Frame{widId = reduceframeID,
                   configs = [],
                   packings=[Side Top],
                   widgets = Pack [redReduceFrame,reduceEntry,redButtonFrame],
                   bindings = keyBinds}




 val tabbedFrame = Frame{widId = tabframeID,
                         configs=[],
                         widgets = Pack[libFrame], bindings=[],packings=[Side Top]}



  val tabbedWindow = {winId = tabbedwinID,
                       config = [WinTitle "Library"],
                       widgets = Pack [allTabs,tabbedFrame],
                       bindings = allKeyBinds,
                       init = initialize}
  val () = allTabbedFrames := [(libframeID,libFrame),(premframeID,premFrame),
                               (taskframeID,taskFrame),(suggframeID,suggestFrame),
                               (reduceframeID,reduceFrame)]

  val () = allTabbedWindows := [(libframeID,libWindow),(premframeID,premWindow),
                               (taskframeID,taskWindow),(suggframeID,suggestWindow),
                               (reduceframeID,reduceWindow)]

  val () = allTabbedText := [(libframeID,"Library"),(premframeID,"Premises"),
                               (taskframeID,"Tasks"),(suggframeID,"Suggested Citations"),
                               (reduceframeID,"Reductions")]

  val () = allTabbedWindowsID := [(libframeID,libwinID),(premframeID,premwinID),
                               (taskframeID,taskwinID),(suggframeID,suggwinID),
                               (reduceframeID,reducewinID)]


  (* Main Window *)
  (* Read-only Text Window *)
  val mainBox = TextWid{widId=mainBoxId,
                        packings=[Side Top,Expand true, Fill Both],
                        scrolltype=NoneScb,
                        configs=[Height (dispLines+3), Width dispWidth,
                                 Font (Typewriter([]))],
                        bindings=keyBinds,
                        annotext=mkAT("")}

  (* Input Window *)

  (* Label for following Entry *)
  val inputWindowLab = Label{widId=newWidgetId(), packings=[Side Left],
                configs=[Text "New Theorem:"], bindings=[]}
  val inputWindow = Entry{widId=entId,
                          packings=[Side Left,Expand true,Fill X],
                          configs=[Width 60],
                          bindings=keyBinds}

  val inputWindowF = Frame{widId=newWidgetId(),
                     widgets=Pack [inputWindowLab,inputWindow],
                     packings=[Side Top, Fill X],
                     configs=[],bindings=[]}

  val foList = Listbox{widId=foId,
                         packings=[Side Top,Expand true,Fill X],
                         configs=[Height 5,Width 60],
                         bindings=[],
                         scrolltype=RightScb}

  (* Buttons *)
  val pubButton = Button{widId=newWidgetId(),
                         packings=[Side Left,Expand false],
                         configs=[Text "Publish", Command pushPublish],
                         bindings=[]}
  val cutButton = Button{widId=newWidgetId(),
                         packings=[Side Left,Expand false],
                         configs=[Text "Cut", Command pushCut],
                         bindings=[]}
  val proofButton = Button{widId=newWidgetId(),
                           packings=[Side Left,Expand false],
                           configs=[Text "Proof", Command pushProof],
                           bindings=[]}
  val normButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand false],
                          configs=[Text "Normalize", Command pushNormalize],
                          bindings=[]}
  val moveButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand false],
                          configs=[Text "Move", Command pushMove],
                          bindings=[]}
  val leftMRButton = Radiobutton{widId=newWidgetId(),
                                 packings=[Side Left],
                                 configs=[Text "Left", Variable "mside",
                                          Value "l"],
                                 bindings=[]}
  val rightMRButton = Radiobutton{widId=newWidgetId(),
                                  packings=[Side Left],
                                  configs=[Text "Right", Variable "mside",
                                           Value "r"],
                                  bindings=[]}

  val focusButton = Button{widId=newWidgetId(),
                           packings=[Side Left,Expand true],
                           configs=[Text "Focus ",Command pushFocus],
                           bindings=[]}
  val unfocusButton = Button{widId=newWidgetId(),
                             packings=[Side Left,Expand true],
                             configs=[Text "Unfocus",Command pushUnfocus],
                             bindings=[]}
  val leftButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand true, Fill X],
                          configs=[Text "<",Command pushLeft],
                          bindings=[]}
  val rightButton = Button{widId=newWidgetId(),
                           packings=[Side Left,Expand true, Fill X],
                           configs=[Text ">",Command pushRight],
                           bindings=[]}
  val upButton = Button{widId=newWidgetId(),
                        packings=[Side Left,Expand true],
                        configs=[Text "^",Command pushUp],
                        bindings=[]}
  val downButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand true],
                          configs=[Text "v",Command pushDown],
                          bindings=[]}


  (* Frames *)
  val topnav = Frame{widId=newWidgetId(),
                     widgets=Pack [focusButton,upButton,unfocusButton],
                     packings=[Side Top, Fill X],
                     configs=[],bindings=[]}
  val bottomnav = Frame{widId=newWidgetId(),
                        widgets=Pack [leftButton,downButton,rightButton],
                        packings=[Side Top, Fill X],
                        configs=[],bindings=[]}
  val allnav = Frame{widId=newWidgetId(),
                     widgets=Pack [topnav,bottomnav],
                     packings=[Side Left, Fill X],
                     configs=[],bindings=[]}

  val commandsnav = Frame{widId=newWidgetId(),
                          widgets=Pack [pubButton,cutButton,
                                        normButton,proofButton,
                                        moveButton,leftMRButton,rightMRButton],
                          packings=[Side Top, Fill X,Expand true],
                          configs=[],bindings=[]}
  val inputFrame = Frame{widId=newWidgetId(),
                         widgets=Pack [inputWindowF,commandsnav],
                         packings=[Side Left, Fill X,Expand true],
                         configs=[],bindings=[]}

  val mainR2 = Frame{widId=newWidgetId(),
                     widgets=Pack [inputFrame,allnav],
                     packings=[Side Top, Fill X],
                     configs=[],bindings=[]}


  (* Preferences Window *)
  val prefLibLab = Label{widId=newWidgetId(),
                         packings=[Side Top],
                         configs=[Text "Loaded Libraries"],
                         bindings=[]}
  val prefLibList = Listbox{widId=prefLibListId,packings=[Side Top,Fill X],
                            configs=[Height 5,Width 40],
                            bindings=[],
                            scrolltype=RightScb}
  val prefLibAdd = Entry{widId=libEntId,
                          packings=[Side Left,Expand true,Fill X],
                          configs=[Width 40],
                          bindings=[]}
  val prefLibAddButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand true],
                          configs=[Text "Add Library",Command prefAddLibrary],
                          bindings=[]}
  val prefLibDelButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand true],
                          configs=[Text "Remove Library",Command prefDelLibrary],
                          bindings=[]}

  val prefLibFrame = Frame{widId=newWidgetId(),
                         packings=[Side Top,Fill X],
                         widgets=Pack[prefLibLab,prefLibList,
                                      prefLibAdd,prefLibAddButton,
                                      prefLibDelButton],
                         bindings=[],
                         configs=[]}


  val prefAddLab = Label{widId=newWidgetId(),
                         packings=[Side Top,Expand true],
                         configs=[Text "Additional Commands"],
                         bindings=[]}
  val prefAddList = Listbox{widId=prefAddListId,packings=[Side Top,Expand true,Fill X],
                            configs=[Height 5,Width 40],
                            bindings=[],
                            scrolltype=RightScb}
  val prefAddAdd = Entry{widId=prefAddEntId,
                          packings=[Side Left,Expand true,Fill X],
                          configs=[Width 40],
                          bindings=[]}
  val prefAddAddButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand true],
                          configs=[Text "Add Command",Command prefAddCommand],
                          bindings=[]}
  val prefAddDelButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand true],
                          configs=[Text "Remove Command",Command prefDelCommand],
                          bindings=[]}

  val prefAddFrame = Frame{widId=newWidgetId(),
                         packings=[Side Top,Fill X],
                         widgets=Pack[prefAddLab,prefAddList,prefAddAdd,
                                      prefAddAddButton,prefAddDelButton],
                         bindings=[],
                         configs=[]}


  val heuritems = map (fn (name,_) => 
                         Frame{widId=newWidgetId(),
                               packings=[Side Top,Fill X],
                               widgets= Pack[Checkbutton{widId=newWidgetId(),
                                       packings=[Side Left],
                                       configs=[Variable name],
                                       bindings=[]},Label{widId=newWidgetId(),
                                             packings=[Side Left],
                                             configs=[Text ("Heuristic: "^name)],
                                             bindings=[]}],
                                bindings=[],configs=[]})  (!heuristics)

  val debugLabel = Label{widId=newWidgetId(),
                         packings=[Side Left],
                         configs=[Text "Debug Mode"],
                         bindings=[]}
  val debugCheck = Checkbutton{widId=newWidgetId(),
                               packings=[Side Left],
                               configs=[Variable "debug"],
                               bindings=[]}
  val debugFrame = Frame{widId=newWidgetId(),
                         packings=[Side Top,Fill X],
                         widgets=Pack[debugCheck,debugLabel],
                         bindings=[],
                         configs=[]}

  val verboseLabel = Label{widId=newWidgetId(),
                         packings=[Side Left],
                         configs=[Text "Verbose Output"],
                         bindings=[]}
  val verboseCheck = Checkbutton{widId=newWidgetId(),
                               packings=[Side Left],
                               configs=[Variable "verbose"],
                               bindings=[]}
  val verboseFrame = Frame{widId=newWidgetId(),
                         packings=[Side Top,Fill X],
                         widgets=Pack[verboseCheck,verboseLabel],
                         bindings=[],
                         configs=[]}
  val suggestLabel = Label{widId=newWidgetId(),
                         packings=[Side Left],
                         configs=[Text "Auto Suggest"],
                         bindings=[]}
  val suggestCheck = Checkbutton{widId=newWidgetId(),
                               packings=[Side Left],
                               configs=[Variable "suggest"],
                               bindings=[]}
  val suggestFrame = Frame{widId=newWidgetId(),
                         packings=[Side Top,Fill X],
                         widgets=Pack[suggestCheck,suggestLabel],
                         bindings=[],
                         configs=[]}


  val redLevelLabel = Label{widId=newWidgetId(),
                         packings=[Side Left],
                         configs=[Text "Finite Reduction Search"],
                         bindings=[]}
  val redLevelCheck = Checkbutton{widId=newWidgetId(),
                               packings=[Side Left],
                               configs=[Variable "redlevel"],
                               bindings=[]}
  val redLevelNumLabel = Label{widId=newWidgetId(),
                         packings=[Side Left],
                         configs=[Text "Depth:"],
                         bindings=[]}
  val redLevelNumInput = ScaleWid{widId=rlnId,
                          packings=[Side Left,Fill X],
                          configs=[From 0.0,To 10.0,
                                   Resolution 1.0, Digits 2,ShowValue true,
                                   Orient Horizontal,Variable "astbound"],
                          bindings=[]}

  val redLevelFrame = Frame{widId=newWidgetId(),
                         packings=[Side Top,Fill X],
                         widgets=Pack[redLevelCheck,redLevelLabel,
                                      redLevelNumLabel,redLevelNumInput],
                         bindings=[],
                         configs=[]}
  val redTimeLabel = Label{widId=newWidgetId(),
                         packings=[Side Left],
                         configs=[Text "Finite Time Search"],
                         bindings=[]}
  val redTimeCheck = Checkbutton{widId=newWidgetId(),
                               packings=[Side Left],
                               configs=[Variable "redtlevel"],
                               bindings=[]}
  val redTimeNumLabel = Label{widId=newWidgetId(),
                         packings=[Side Left],
                         configs=[Text "Seconds:"],
                         bindings=[]}
  val redTimeNumInput = ScaleWid{widId=newWidgetId(),
                          packings=[Side Left,Fill X],
                          configs=[From 0.0,To 120.0,Length 300,
                                   Resolution 0.25, Digits 5,ShowValue true,
                                   Orient Horizontal,Variable "timelimit"],
                          bindings=[]}

  val redTimeFrame = Frame{widId=newWidgetId(),
                         packings=[Side Top,Fill X],
                         widgets=Pack[redTimeCheck,redTimeLabel,
                                      redTimeNumLabel,redTimeNumInput],
                         bindings=[],
                         configs=[]}
  val prefSaveButton = Button{widId=newWidgetId(),
                          packings=[Side Left,Expand true],
                          configs=[Text "Save",Command savePrefs],
                          bindings=[]}

  val preferencesWindow = {bindings=([]:Binding list),
                   winId = prefWindowId,
                   config=[WinTitle "Preferences"],
                   widgets=Pack ([prefLibFrame,verboseFrame,debugFrame,suggestFrame]@heuritems@[redLevelFrame,redTimeFrame,prefAddFrame,prefSaveButton]),

                   init=loadPrefs}




  (* Menu *)
  val fileMenu = Menubutton{widId=newWidgetId(),packings=[Side Left],
                  configs=[Text "File"],bindings=[],
                  mitems=[MCommand([Text "New WorkSpace",Command newWork]),
                          MCommand([Text "Save WorkSpace",Command saveWork]),
                          MCommand([Text "Load Library",Command load]),
                          MCommand([Text "Save Library",Command save]),
                          MCommand([Text "Save All",Command saveAll]),
                          MCommand([Text "Export to LaTeX",Command output]),
                          MCommand([Text "Dump Commands",Command dump]),
                          MCommand([Text "Load Commands",Command loadDump]),
                          MCommand([Text "Quit", Command stop])]}
  fun changePrefsMenu _ = openWindow (mkWindow preferencesWindow)
  val editMenu = Menubutton{widId=newWidgetId(),packings=[Side Left],
                  configs=[Text "Edit"],bindings=[],
                  mitems=[MCommand([Text "Undo",Command undoMenu]),
                          MCommand([Text "Redo",Command redoMenu]),
                          MCommand([Text "Find",Command findMenu]),
                          MCommand([Text "Find Again",Command findAgainMenu]),
                          MCommand([Text "Preferences",Command changePrefsMenu])]}

  val proofMenu = Menubutton{widId=newWidgetId(),packings=[Side Left],
                  configs=[Text "Proof"],bindings=[],
                  mitems=[MCommand([Text "Define Shortcut",Command defineMenu]),
                          MCommand([Text "Rename Theorem",Command renameMenu]),
                          MCommand([Text "Reset",Command resetMenu]),
                          MCommand([Text "Verify Proof",Command verifyMenu]),
                          MCommand([Text "Deep Verify Proof",Command deepverifyMenu])]}

  val commandsMenu = Menubutton{widId=newWidgetId(),packings=[Side Left],
                  configs=[Text "Commands"],bindings=[],
                  mitems=[MCommand([Text "Publish (Ret)",Command pushPublish]),
                          MCommand([Text "Cut (C-c)",Command pushCut]),
                          MCommand([Text "Display Proof (C-p)",Command pushProof]),
                          MCommand([Text "Move Current Term (C-m)",Command pushMove]),
                          MCommand([Text "Focus (Down)",Command pushFocus]),
                          MCommand([Text "Unfocus (Up)",Command pushUnfocus]),
                          MCommand([Text "Move Left (Left)",Command pushLeft]),
                          MCommand([Text "Move Right (Right)",Command pushRight]),
                          MCommand([Text "Move Up (Up)",Command pushUp]),
                          MCommand([Text "Move Down (Down)",Command pushDown]),
                          MCommand([Text "Check for Circularity",Command pushCirc])]}

  

(*  val prefsMenu = let 
    val menuitems = map (fn (name,_) => 
                         MCheckbutton([Text name,Variable name,
                                       Command changeHeuristics])) 
                    (!heuristics)
  in
    Menubutton{widId=newWidgetId(),packings=[Side Left],
               configs=[Text "Preferences"],bindings=[],
               mitems=[MCascade(menuitems,[Text "Heuristics"]),
                       MCascade([MRadiobutton([Text "On",Variable "debug",
                                               Value "1",
                                               Command changeDebug]),
                        MRadiobutton([Text "Off",Variable "debug",
                                      Value "0",
                                      Command changeDebug])],
                       [Text "Debug"]),
                       MCascade([MRadiobutton([Text "On",Variable "verbose",
                                               Value "on",
                                               Command changeVerbose]),
                       MRadiobutton([Text "Off",Variable "verbose",
                                     Value "off",Command changeVerbose])],
                       [Text "Verbose Output"])]}
  end*)

  
(*  val windowMenu = Menubutton{widId=newWidgetId(),packings=[Side Left],
                            configs=[Text "Window"],bindings=[],
                            mitems=[MCheckbutton([Text "Library", 
                                                  Variable "libWin",
                                                  Command checkWindows]),
                                    MCheckbutton([Text "Tasks", 
                                                  Variable "taskWin", 
                                                  Command checkWindows]),
                                    MCheckbutton([Text "Premises",
                                                  Variable "premWin",
                                                  Command checkWindows]),
                                    MCheckbutton([Text "First-Order Terms",
                                                  Variable "foWin",
                                                  Command checkWindows]),
                                    MCheckbutton([Text "Suggested Citations",
                                                  Variable "suggWin",
                                                   Command checkWindows])]}
*)
  val mainMenu = Frame{widId=newWidgetId(),
                       widgets=Pack [fileMenu,editMenu,commandsMenu,proofMenu
                                     ],
                       packings=[Side Top, Fill X],configs=[],bindings=[]}



  val mainWindow = {winId = mainwinID,
                   config = [WinTitle "KAT Theorem Prover"],
                   widgets = Pack [mainMenu,mainBox,foList,mainR2],
                   bindings = keyBinds,
                   init = noAction}

  val () = closeableWins := [("libWin",libwinID,mkWindow libWindow),
                             ("premWin",premwinID,mkWindow premWindow),
                             ("foWin",fowinID,mkWindow foWindow),
                             ("suggWin",suggwinID,mkWindow suggestWindow),
                             ("taskWin",taskwinID,mkWindow taskWindow)]

  fun go() = (SmlTk.init(); startTcl [mkWindow mainWindow,mkWindow tabbedWindow, mkWindow foWindow  (*,
                                        mkWindow libWindow,
                                        mkWindow premWindow,
                                        
                                        mkWindow suggestWindow,
                                        mkWindow taskWindow*)])

  (* call this to build the system *)
  fun build() =
    let fun main(s:string,t:string list) = (go(); OS.Process.success)
    in SMLofNJ.exportFn("../Bin/KAT_TK",main)
    end


end