structure KAT = struct

  open Util
  open Term
  open XML
  open Unify
  open Proof

  val debugging = ref false

  val outputstring:string list ref = ref nil
  val fsuggest:string list ref = ref nil
  val usuggest:string list ref = ref nil

  fun debug args =
    case args of
      ["on"] => debugging := true
    | ["off"] => debugging := false
    | _ => (!printtext) (if !debugging then "debug is on" else "debug is off")

 (* global continuations initialized to dummy values *)
  val errorCont = ref (SMLofNJ.Cont.isolate (fn() => ()))
  val exitCont = ref (SMLofNJ.Cont.isolate (fn() => ()))

  fun quit _ = SMLofNJ.Cont.throw (!exitCont) ()
  fun abort x = ((!alert) ("system error: " ^ x);
                  SMLofNJ.Cont.throw (!errorCont) ())

  exception Continue

(***********************************************
 * core environment
 ***********************************************)

  (* KA axioms *)
  val refEq = ("ref=","x = x")
  val sym = ("sym","x = y -> y = x")
  val transEq = ("trans=","x = y -> y = z -> x = z")
  val congPlus = ("cong+R","x = y -> x + z = y + z")
  val congTimesL = ("cong.L","y = z -> x;y = x;z")
  val congTimesR = ("cong.R","x = y -> x;z = y;z")
  val congStar = ("cong*","x = y -> x* = y*")
  val leIntro = ("<intro","x + y = y -> x < y")
  val leElim = ("<elim","x < y -> x + y = y")
  val commutPlus = ("commut+","x + y = y + x")
  val idPlusR = ("id+R","x + 0 = x")
  val idempPlus = ("idemp+","x + x = x")
  val idTimesL = ("id.L","1;x = x")
  val idTimesR = ("id.R","x;1 = x")
  val annihL = ("annihL","0;x = 0")
  val annihR = ("annihR","x;0 = 0")
  val distrL = ("distrL","x;(y + z) = x;y + x;z")
  val distrR = ("distrR","(x + y);z = x;z + y;z")
  val unwindL = ("unwindL","1 + x;x* = x*")
  val unwindR = ("unwindR","1 + x*;x = x*")
  val starL = ("*L","z y + x < z -> x y* < z")
  val starR = ("*R","x z + y < z -> x*y < z")

  val initLibrary =
    [refEq,sym,transEq,congPlus,congTimesL,congTimesR,congStar,leIntro,leElim,
     commutPlus,idPlusR,idempPlus,idTimesL,idTimesR,annihL,annihR,
     distrL,distrR,unwindL,unwindR,starL,starR]

(***********************************************
 * state
 ***********************************************)

  (* task = (name,(type,proofs of premises,focus)) *)
  type task = id * (cond_eqn * proof list * int list)

  (* theorem = (name,(statement,current task,proof)) *)
  type theorem = id * (cond_eqn * task option ref * proof ref)

  type transTable = (id * ((SLang.B * term) list * (SLang.C * term) list))

  val undostates:(theorem list * transTable list) list ref = ref []
  val redostates:(theorem list * transTable list) list ref = ref []

  val taskCache: theorem list ref = ref []


  (* library of proved and unproved theorems *)
  val library : theorem list ref = let
    val names = map #1 initLibrary
    val axioms = map (valOf o parseCondEqn o #2) initLibrary
      handle Option => abort "could not parse initial environment"
    fun createAxiom (name,axiom) = (name, (axiom, ref NONE, ref (AXIOM name)))
  in
    ref (ListPair.map createAxiom (names,axioms))
  end

  val foTable : transTable list ref = ref nil

  fun getCurrentTable() = hd (!foTable)
    handle Empty => abort "table is empty"

  fun getCurrentTheorem() : theorem = hd (!library)
    handle Empty => abort "library is empty"

  fun getCurrentProof() : proof ref = #3(#2(getCurrentTheorem()))

  fun getCurrentTask() : task option ref = #2(#2(getCurrentTheorem()))

  fun getCurrentTasks() : task list = Proof.getTasks (!(getCurrentProof()))

  fun newTaskId() = let val newID = Util.newTaskId()
                        val currentIds = map #1 (getCurrentTasks())
                    in
                        if List.exists (fn x => x = newID) currentIds
                        then newTaskId() else newID
  end

  fun getCurrentFocus() : int list =
    case !(getCurrentTask()) of SOME x => #3(#2 x) | _ => []

  fun resetCurrentTask _ = let
    val proof = getCurrentProof()
    val task = getCurrentTask()
    val tasks = Proof.getTasks(!proof)
    val newTask = case !task of
      SOME (id,_) => find id tasks
    | NONE => optionalHd tasks
  in
    task := newTask
  end

  fun firstUnify(eqn,lib) = case lib of
                              [] => NONE
                            | ((libID,(([],genCon), _, _))::rest) =>
                                 (case Unify.unifyEqn(genCon,eqn) of
                                                 NONE => firstUnify(eqn,rest)
                                               | SOME _ => SOME libID)
                            | (_::rest) => firstUnify(eqn, rest)

(***********************************************
 * undo
 ***********************************************)

  fun copyState() : theorem list = let
    fun copyTheorem ((id,(thm,ref task,ref proof)):theorem) : theorem =
      (id,(thm,ref task,ref (Proof.copy proof)))
  in
    map copyTheorem (!library)
  end

  fun undo _ =
    case !undostates of
      (x,y)::t =>
        (redostates := (copyState(),!foTable)::(!redostates);
         library := x;
         foTable := y;
         undostates := t)
    | [] => (!alert) "cannot undo"

  fun redo _ =
    case !redostates of
      (x,y)::t =>
        (undostates := (copyState(),!foTable)::(!undostates);
         library := x;
         foTable := y;
         redostates := t)
    | [] => (!alert) "cannot redo"

  fun saveState() =
    (undostates := (copyState(),!foTable)::(!undostates);
     redostates := [];
     (* garbage collect *)
     if length(!undostates) > 200
     then undostates := List.take(!undostates,100)
     else ())

(***********************************************
 * output
 ***********************************************)

  fun theoremToString((id,(ce,_,ref proof)):theorem) = let
    val n = length (Proof.getTasks proof)
    val status =
      case n of
        0 => (case proof of AXIOM _ => "  (axiom)" | _ => "  (proved)")
     | 1 => "  (1 task)"
      | _ => "  (" ^ (Int.toString n) ^ " tasks)"
  in
    id ^ ": " ^ (condEqnToString ce) ^ status
  end

  fun theoremStructure((id,(ce,_,ref proof)):theorem) = let
    val n = length (Proof.getTasks proof)
    val status =
      case n of
        0 => (case proof of AXIOM _ => "  (axiom)" | _ => "  (proved)")
      | 1 => "  (1 task)"
      | _ => "  (" ^ (Int.toString n) ^ " tasks)"
  in
    id ^ ": " ^ (condEqnStructure ce) ^ status
  end

  fun theoremToXML ((id,(thm,ref task,ref proof)):theorem) : string = let
    val _ = library := Util.moveToFront id (!library);
    val currentTask = case !(getCurrentTask()) of
      SOME (id,_) => "<task>" ^ id ^ "</task>"
    | NONE => "<task></task>"
  in
    "<lib>" ^ idToXML id ^ condEqnToXML thm ^ currentTask ^
              proofToXML proof ^
              FirstOrder.tableToXML(getCurrentTable())^"</lib>"
  end

(***********************************************
 * display info
 ************************************************)

  fun premises _ =
    case !(getCurrentTask()) of
      SOME (_,((premises as _::_,_),_,_)) => let
        fun listPremises(n:int,s:eqn list) =
          case s of
            e::t => (outputstring := (!outputstring @
                       [("A"^(Int.toString n)^": "^(eqnToString e))]);
                    listPremises(n+1,t))
          | [] => ()
        in
          listPremises(0,premises)
        end
    | SOME (_,(([],_),_,_)) => (!alert) "no premises"
    | NONE => (!alert) "no current task"

  fun conclusion _ =
    case !(getCurrentTask()) of
      SOME (_,((_,e),_,_)) => outputstring := (!outputstring) @ [eqnToString e]
    | NONE => ()

  fun taskToString (id,((ce,e),_,_)) = (premises();
(    id ^ ":" ^
    (concat (map (fn d => "\n     "^d) (map eqnToString ce)))
    ^ "\n     -------------------------\n     " ^ eqnToString e))

  fun taskStructure(id,((ce,e),_,_)) = (premises();
    (id ^ ":" ^
    concat (map (fn d => "\n     " ^ d) (map eqnStructure ce))
    ^ "\n     -------------------------\n     " ^ eqnStructure e))

  fun viewTasks _ = let
    val f = if !debugging then taskStructure else taskToString
    val taskList = getCurrentTasks()
    val returnstring = ref nil
    val () = outputstring := []
  in
    (case taskList of
      _::_ => app (fn x => returnstring := (!returnstring) @ [x])
              (map f taskList)
    | [] => ());
    outputstring := (!returnstring)
  end

  fun viewCurrentTask _ = let
    val f = if !debugging then taskStructure else taskToString
  in
    case !(getCurrentTask()) of
      SOME task => ((!printtext) ("\ncurrent task:\n"^(f task)))
    | NONE => (!printtext) ("\nno tasks")
  end

  fun viewLibrary args = let
    fun isSubString x (y as (name,thm)) =
      if String.size(name)=0 then false
      else ((String.isPrefix x name) orelse
           (isSubString x ((String.extract(name,1,NONE)),thm)))
  in
    app  (fn x => outputstring := (!outputstring) @ [x]) (map theoremToString
    (case args of
       nil => (!library)
     | [x] => (List.filter (isSubString x) (!library))
     | _ => raise Fail "Too many arguements"))
  end
  handle Fail x => (!alert) x


  fun viewfoLibrary args = let
    fun isSubString x (y as (name,thm,body)) =
      if String.size(name)=0 then false
      else ((String.isPrefix x name) orelse
           (isSubString x ((String.extract(name,1,NONE)),thm,body)))
  in
    app  (fn x => outputstring := (!outputstring) @ [x])
         (map (fn(x,y,_,_) => (x^": "^y))
    (case args of
       nil => (!FirstOrder.library)
(*     | [x] => (List.filter (isSubString x) (!FirstOrder.library))*)
     | _ => raise Fail "Too many arguements"))
  end
  handle Fail x => (!alert) x



  fun viewCurrentTheorem _ = let
    val f = if !debugging then theoremStructure else theoremToString
  in
    (!printtext) (f (getCurrentTheorem()))
  end

  fun viewCurrentProof _ = let
    val f = if !debugging then Proof.proofStructure else Proof.toString
  in
    (!printtext) (f (!(getCurrentProof())))
  end

(***********************************************
 * focus and navigation
 ***********************************************)

  fun focusing _ = not (null (getCurrentFocus()))

  fun setFocus (focus:int list) = let
    val task = getCurrentTask()
  in
    case !task of
      SOME (id,(ce,pl,_)) => let
        val proof = getCurrentProof()
        val newTask = (id,(ce,pl,focus))
      in
        saveState();
        task := SOME newTask;
        proof := Proof.replaceTask (TASK newTask) id (!proof)
      end
    | NONE => ()
  end

  fun displayCurrentFocus _ =
    case !(getCurrentTask()) of
      SOME (_,((_,e),_,focus)) => ("\n"^(focusInEqnToString(e,focus)))
    | NONE => ""

  fun changeFocus f cont = let
    val task = getCurrentTask()
    val proof = getCurrentProof()
  in
    case !task of
      SOME (id,((pre,con),pl,focus as x::_)) =>
        let
          val (focusTerm,n) = getFocusInEqnWithContext(con,focus)
          val newFocus = f(focusTerm,rev(focus),n)
          val newTask = (id,((pre,con),pl,newFocus))
        in
          saveState();
          task := SOME newTask;
          proof := Proof.replaceTask (TASK newTask) id (!proof);
          case cont of
            x::t =>
              (case lookupPrefix x [("up",up),("down",down),
                                    ("right",right),("left",left)] of
                SOME action => action t
              | NONE => raise Fail ("invalid direction " ^ x ^
                                    ", remainder of command ignored"))
          | [] => ()
        end
    | SOME (_,(_,_,[])) => (!alert) "not focusing"
    | NONE => (!alert) "no current task"
  end
  handle Fail x => (!alert) x

  and down cont =
    if focusing()
      then changeFocus (fn (s:term,cf:int list,n:int) =>
        case (s,cf) of
          ((PLUS x,[1,0]) | (TIMES x,[1,0])) => [0,0,length x - 1]
        | ((PLUS x,[1,1]) | (TIMES x,[1,1])) => [1,0,length x - 1]
        | ((PLUS x,1::t) | (TIMES x,1::t)) => rev(length x - 1::0::t)
        | ((PLUS x,m::t) | (TIMES x,m::t)) => rev(m-1::t)
        | ((NOT x,1::t) | (STAR x,1::t)) => rev(1::0::t)
        | _ => raise Fail "can't go down") cont
    else
      (setFocus[0,1];
       case cont of
         x::t =>
           (case lookupPrefix x [("up",up),("down",down),
                                 ("right",right),("left",left)] of
              SOME action => action t
            | NONE => raise Fail ("invalid direction " ^ x ^
                                  ", remainder of command ignored"))
       | [] => ())

  and up cont = changeFocus (fn (s:term,cf:int list,n:int) =>
    case (s,cf) of
      ((_,[1,0]) | (_,[1,1]))=> []
    | (_,m::j::t) =>
        (if (j > 0 andalso j + m < n) orelse
            (j = 0 andalso 1 + m < n) then rev(m+1::j::t)
         else rev(1::t))
    | _ => raise Fail "can't go up") cont

  and right cont = changeFocus (fn (s:term,cf:int list,n:int) =>
    case cf of
      m::j::t =>
        (if j + m < n then rev(m::j+1::t) else raise Fail "can't go right")
    | _ => raise Fail "can't go right") cont

  and left cont = changeFocus (fn (s:term,cf:int list,n:int) =>
    case cf of
      m::j::t =>
        (if j > 0 then rev(m::j-1::t) else raise Fail "can't go left")
    | _ => raise Fail "can't go left") cont

  fun unfocus _ = setFocus []
  fun focus cont =
    let val oldUndo = !undostates
      val _ = if focusing() then unfocus() else ()
      val _ = undostates:=oldUndo
    in
      down cont
    end


(***********************************************
 * Get all focii
***********************************************)
  fun getDown (s:term,cf:int list,n:int) =
    case (s,cf) of
      ((PLUS x,[1,0]) | (TIMES x,[1,0])) => [0,0,length x - 1]
    | ((PLUS x,[1,1]) | (TIMES x,[1,1])) => [1,0,length x - 1]
    | ((PLUS x,1::t) | (TIMES x,1::t)) => rev(length x - 1::0::t)
    | ((PLUS x,m::t) | (TIMES x,m::t)) => rev(m-1::t)
    | ((NOT x,1::t) | (STAR x,1::t)) => rev(1::0::t)
    | _ => []

  fun getUp (s:term,cf:int list,n:int) = (*unused*)
    case (s,cf) of
      ((_,[1,0]) | (_,[1,1]))=> []
    | (_,m::j::t) =>
        (if (j > 0 andalso j + m < n) orelse
            (j = 0 andalso 1 + m < n) then rev(m+1::j::t)
         else rev(1::t))
    | _ => []

  fun getLeft (s:term,cf:int list,n:int) = (*unused*)
    case cf of
      m::j::t =>
        (if j > 0 then rev(m::j-1::t) else [])
    | _ => []


  fun getRight (s:term,cf:int list,n:int) =
    case cf of
      m::j::t =>
        (if j + m < n then rev(m::j+1::t) else [])
    | _ => []

   val allfoc:int list list ref = ref []
   fun getAllLocalFocii _ = let
     val () = allfoc := []
     val task = getCurrentTask()
     val (curfocus,con) = (case !task of
                             SOME (id,((pre,con1),pl,focus1)) => (focus1,con1)
                           | NONE => raise Fail "Focus error")
     fun printfoc lst =
       case lst of
         [] => ""
       | [x] => Int.toString x
       | x::xs => (Int.toString x)^","^(printfoc  xs)
     fun allFocus (toproc:int list list) =
       case toproc of
         [] => allfoc := rev(!allfoc)
       | (afocus::rest) =>
           (allfoc := afocus::(!allfoc);
            let
              val (focusTerm,n) = getFocusInEqnWithContext(con,afocus)
              val nextLevel = getDown(focusTerm,rev(afocus),n)
              fun getRights newFocus =
                let
                  val (newfocusTerm,newN) =
                    getFocusInEqnWithContext(con,newFocus)
                  val n =
                    if (length curfocus) = (length newFocus)
                      then Int.min(hd(rev curfocus)+hd(tl(rev curfocus)),newN)
                    else newN
                in
                  case getRight(newfocusTerm,rev(newFocus),n) of
                    [] => []
                  | foc => if (member foc (toproc)) then getRights(foc)
                           else foc::getRights(foc)
                end
            in
              case nextLevel of
                [] => allFocus(rest)
              | _ =>  if (member nextLevel (toproc))
                        then allFocus(rest @ (getRights(nextLevel)))
                      else allFocus(rest @ (nextLevel::getRights(nextLevel)))
            end)
   in
     case curfocus of
       [] => (focus [""];
              let
                val curfocus = (case !task of
                                  SOME (_,(_,_,focus1)) => focus1
                                | NONE => raise Fail "Focus error")
              in
                allFocus([curfocus,(1::tl(curfocus))])
              end)
     | _ => allFocus [curfocus]
   end
 handle Fail x => (!alert) x

   fun getAllFocii _ =
     (unfocus(); getAllLocalFocii())

   fun compareFocii foc1 foc2 = case (foc1,foc2) of
     ([],[]) => false
   | ([],_) => true
   | (_,[]) => false
   | ((x::xs),(y::ys)) =>
       if x < y then true
       else if y < x then false
            else compareFocii xs ys

(***********************************************
 * command handlers
 ***********************************************)

(***********************************************
 * reset
 ***********************************************)

  fun reset args =
    case getCurrentTheorem() of
      (_,(thm,task,proof)) => let
        val taskId = newTaskId()
      in
        saveState();
        proof := Proof.reset(thm,taskId);
        task := SOME (hd(Proof.getTasks(!proof)))
      end
      handle Empty => abort "no tasks"

(***********************************************
 * publish
 ***********************************************)

  fun publish args = let
    val thm = valOf(parseCondEqn (String.concat (map (fn s => s^" ") args)))
      handle Option => raise Fail "invalid formula"
    fun uniqueLibId() = let val newid = newLibId()
                        in
                         case find newid (!library) of
                            NONE => newid
                          | SOME _ => uniqueLibId()
                        end
    val uniqueID = uniqueLibId()
    val newTheorem =
      (uniqueID,(thm, ref NONE, ref (CONST "")))
  in
    saveState();
    library := newTheorem::(!library);
    foTable := (uniqueID,([],[]))::(!foTable);
  (* Manually run reset to prevent a second
     saving of the state. *)
(    case getCurrentTheorem() of
      (_,(thm,task,proof)) => let
        val taskId = newTaskId()
      in
        proof := Proof.reset(thm,taskId);
        task := SOME (hd(Proof.getTasks(!proof)))
      end
      handle Empty => abort "no tasks");
    viewCurrentTheorem()
  end
  handle Fail x => (!alert) x

  fun publishFromSlang args = let
    val () = FirstOrder.reset();
    val (x,y) = (case args of
      []  => raise Fail "Not enough arguments"
    | [x] => raise Fail "Not enough arguments"
    | (x::y::z::_) => raise Fail "Too many arguements"
    | ([x,y]) => (x,y))

    val side1 = makeVariable(FirstOrder.slangCToKAT(FirstOrder.flatten(Parser.parse x)))
    handle Continue => raise Fail "Unable to parse program 1."
    val side2 = makeVariable(FirstOrder.slangCToKAT(FirstOrder.flatten(Parser.parse y)))
    handle Continue => raise Fail "Unable to parse program 2."
    val thm = ([],EQ(side1,side2))
    fun uniqueLibId() = let val newid = newLibId()
                        in
                         case find newid (!library) of
                            NONE => newid
                          | SOME _ => uniqueLibId()
                        end
    val uniqueID = uniqueLibId()
    val newTheorem:theorem =
      (uniqueID,(thm, ref NONE, ref (CONST "")))

  in
    saveState();
    library := newTheorem::(!library);
    foTable := (uniqueID,(FirstOrder.singleBools(),
                          FirstOrder.singleTerms()))::(!foTable);
  (* Manually run reset to prevent a second
     saving of the state. *)
    (    case getCurrentTheorem() of
      (_,(thm,task,proof)) => let
        val taskId = newTaskId()
      in
        proof := Proof.reset(thm,taskId);
        task := SOME (hd(Proof.getTasks(!proof)))
      end
      handle Empty => abort "no tasks");
    viewCurrentTheorem()
  end
  handle Fail x => (!alert) x


  fun publishFO args = let
    val thm = valOf(parseCondEqn (String.concat (map (fn s => s^" ") args)))
      handle Option => raise Fail "invalid formula"
    fun itemIsFO x = (Util.keymember x (!FirstOrder.boolList)) orelse
                     (Util.keymember x (!FirstOrder.termList))
    val allAccounted = List.all (itemIsFO) (map (makeConstant o parseId)
                       (Term.condEqnVariables thm))
    val () = if allAccounted then ()
             else raise Fail "Not all terms are from first-order commands."
    fun uniqueLibId() = let val newid = newLibId()
                        in
                         case find newid (!library) of
                            NONE => newid
                          | SOME _ => uniqueLibId()
                        end
    val uniqueID = uniqueLibId()
    val newTheorem:theorem =
      (uniqueID,(thm, ref NONE, ref (CONST "")))

  in
    saveState();
    library := newTheorem::(!library);
    foTable := (uniqueID,(FirstOrder.singleBools(),
                          FirstOrder.singleTerms()))::(!foTable);
  (* Manually run reset to prevent a second
     saving of the state. *)
    (    case getCurrentTheorem() of
      (_,(thm,task,proof)) => let
        val taskId = newTaskId()
      in
        proof := Proof.reset(thm,taskId);
        task := SOME (hd(Proof.getTasks(!proof)))
      end
      handle Empty => abort "no tasks");
    viewCurrentTheorem()
  end
  handle Fail x => (!alert) x


(***********************************************
 * get
 ***********************************************)

  fun get args = let
    val arg = case args of
        [y] => y
      | [] => raise Fail "no arguments"
      | _ => raise Fail "too many arguments"
    val task = getCurrentTask()
    val isTheorem = find arg (!library)
    val isTask = find arg (getCurrentTasks())
    fun getTheorem _ = (library := Util.moveToFront arg (!library);
                        foTable := Util.moveToFront arg (!foTable);
                        FirstOrder.boolList := (#1 (#2 (hd(!foTable))));
                        FirstOrder.termList := (#2 (#2 (hd(!foTable))));
			FirstOrder.boolIndex := List.length(!FirstOrder.boolList);
			FirstOrder.termIndex := List.length(!FirstOrder.termList);
                        viewCurrentTheorem())
    fun getTask _ = task := isTask
    val actions = [getTheorem,getTask]
  in
    saveState();
    case (isTheorem,isTask) of
      (SOME _,NONE) => getTheorem()
    | (NONE,SOME _) => getTask()
    | (SOME _,SOME _) =>
         (((!ask2) ("theorem or task? ",["theorem","task"]))
         (fn which => (case which of
           SOME n => List.nth (actions,n) ()
         | NONE => raise Fail "operation canceled")))
    | (NONE,NONE) => raise Fail ("theorem or task " ^ arg ^ " not found")
  end
  handle Fail x => (!alert) x

(***********************************************
 * cut
 ***********************************************)

  fun cut aNum args = let
    val eqn = parseEqnTokenized args
    val cut =
      case eqn of
        SOME e => makeConstantEqn e
      | NONE => raise Fail "operation canceled"
    val (_,(_,task,proof)) = getCurrentTheorem()
    val (taskId,((specPre,specCon),preProofs,focus))= valOf(!(getCurrentTask()))
      handle Option => raise Fail "no current task"
    val LAMBDAS(proofsubs,proofnot) = !proof
    val newvars =removeDuplicates((condEqnVariables(([],valOf(eqn))))@proofsubs)
    val useCutTaskId = newTaskId()
    val proveCutTaskId = if aNum<>0 then "T"^Int.toString(aNum) else newTaskId()
    val proveCut = (proveCutTaskId,((specPre,cut),preProofs,[]))
    val useCut = (useCutTaskId,((specPre@[cut],specCon),
                  preProofs@[TASK proveCut],focus))
  in
    saveState();
    task := SOME useCut;
    proof := LAMBDAS(newvars,Proof.replaceTask (TASK useCut) taskId (proofnot))
  end
  handle Fail x => !alert x


(***********************************************
 * define a symbol
 ***********************************************)

  (* define a new symbol as abbreviation of current focus *)
  (* add a new premise to current task *)
  fun define args = let
    val name =
      case args of
        [] => raise Fail "no symbol specified"
      | [x] => x
      | _ => raise Fail "too many arguments"
    val (_,(_,task,proof)) = getCurrentTheorem()
    val LAMBDAS(ids,proof2) = !proof
    val (taskId,(spec as (specPre,specCon),preProofs,focus)) =
      case !task of
        SOME x => x
      | NONE => raise Fail "no current task"
    val already = condEqnVariables (makeVariableCondEqn spec)
    val _ = if member name already
      then raise Fail ("symbol " ^ name ^ " is already in use")
      else ()
    val cf = Term.getFocusInEqn(specCon,focus)
      handle Fail x => raise Fail "no focus"
    val e =
      if Char.isUpper(String.sub(name,0)) then
        if isTest cf then EQ (TST_CON name,cf)
        else raise Fail "focus term is not a test"
      else EQ (ACT_CON name,cf)
    val newTaskId = newTaskId()
    val newTask:task = (newTaskId,((specPre@[e],specCon),preProofs@[
                        APPLYS (AXIOM "def",[(name,cf)])],focus))
  in
    saveState();
    proof:= Proof.replaceTask (TASK newTask) taskId (LAMBDAS(name::ids,proof2));
    task := SOME newTask
  end
  handle Fail x => (!alert) x | Parser.Continue => ()

(***********************************************
 * circCheck
 ***********************************************)
 fun circCheck() : bool = let
   fun checkproof(name,offLimits) = let
       val (_,_,thm) = valOf(lookup name (!library))
     in
       if(contains((!thm),offLimits)) then true
       else List.exists (fn p => (checkproof(p,(name::offLimits))))
                        (Proof.allThms(!thm))
     end

   val theProof = getCurrentProof()
   val (name,_) = getCurrentTheorem()
 in
   checkproof(name,[name])
 end

fun mCircCheck _ = (!alert) (if circCheck() then "A circularity exists!"
                             else "No circularities detected")

(***********************************************
 * cite
 ***********************************************)

  exception noTheorem of string * direction

  fun getDirection() : direction = let
    val inputLine = ((!print) "which side? "; TextIO.inputLine TextIO.stdIn)
    val tokenizedInput = String.tokens Char.isSpace inputLine
    val args = map (String.map Char.toUpper) tokenizedInput
  in
    case args of
      [d] =>
        if String.isPrefix "L" d andalso String.isPrefix d "LEFT" then LEFT
        else
          if String.isPrefix "R" d andalso String.isPrefix d "RIGHT" then RIGHT
          else (println "say left or right"; getDirection())
    | d::t => (println "say left or right"; getDirection())
    | _ => raise Fail "operation canceled"
  end

  fun direction(d:string) : direction = let
    val d = String.map Char.toUpper  d
  in
    if String.isPrefix "L" d andalso String.isPrefix d "LEFT" then LEFT
    else if String.isPrefix "R" d andalso String.isPrefix d "RIGHT" then RIGHT
    else raise Fail "unrecognized direction"
  end

  fun dirToString(dir) = case dir of
                           LEFT => "l"
                         | RIGHT => "r"


  (***********************************************
   * forget a published theorem
   ***********************************************)
  fun forget args = let
    val name = case args of
      [x] => x
    | _ => raise Fail "invalid name"
    val (_,_,ref substProof) =
      case lookup name (!library) of
        NONE => raise Fail (name ^ " does not exist")
      | SOME x => x
    val _ = case substProof of
      AXIOM _ => raise Fail "may not forget axioms"
    | _ => ()
    val _ = case !library of
      [x] => abort "forgetting last library entry"
    | _ => ()
    fun forgetInOne ((_,(_,task,inProof)):theorem) : unit =
      (inProof := Proof.forget(name,substProof,!inProof);
       let val tasks = getTasks (!inProof)
       in if isSome (!task) orelse null tasks then ()
          else task := SOME (hd tasks)
       end)
  in
    saveState();
    library := Util.delete name (!library);
    app forgetInOne (!library)
  end
  handle Fail x => (!alert) x

  (***********************************************
   * ignore a premise
   ***********************************************)
  fun ignore args = let
    val (_,(_,task,proof)) = getCurrentTheorem()
    val (taskId,((pre,con),preProofs,focus)) = valOf(!(getCurrentTask()))
      handle Option => raise Fail "no current task"
    val name =
      case args of
        [] => raise Fail "no premise specified"
      | [x] => x
      | _ => raise Fail "too many arguments"
    val index =
      if String.isPrefix "A" name then
        let val intName = Int.fromString(String.extract(name,1,NONE))
        in case intName of
          SOME j => j
        | NONE => raise Fail "no such premise"
        end
      else raise Fail "invalid premise"
    val newPre = (List.take(pre,index) @ List.drop(pre,index+1)
      handle Subscript => raise Fail "no such premise")
    val newPreProofs =(List.take(preProofs,index) @ List.drop(preProofs,index+1)
      handle Subscript => abort "no such premise")
    val newTaskId = newTaskId()
    val newTask = (newTaskId,((newPre,con),newPreProofs,focus))
  in
    saveState();
    task := SOME newTask;
    proof := Proof.replaceTask (TASK newTask) taskId (!proof)
  end
  handle Fail x => (!alert) x


  fun citeFO(name,dir) = let
    val laststates = (copyState(),!foTable)::(!undostates)
    val (_,_,conds,(lhs,rhs)) = valOf(List.find (fn (id,_,_,_) => id = name)
                        (!FirstOrder.library))
    handle Option => raise Fail "No such theorem"
    val tasks = getCurrentTasks()
    val task = getCurrentTask()
    val (taskId,((specPre,specCon),preProofs,focus)) =
      case !task of
        SOME x => x
      | NONE => raise Fail "no current task"
    val proof = getCurrentProof()

    val cf = Term.getFocusInEqn(specCon,focus)
    val transed = FirstOrder.flatten(valOf(FirstOrder.KATToSLang(cf)))
    handle Option => raise Fail "No translated term!"
    val allvars = FirstOrder.varsInC(transed)
    val (lhsu,rhsu) = (lhs,rhs) (*FirstOrder.uniquify((lhs,rhs),allvars)*)
    val cont = fn () => (let
    val () = FOUnify.subList := List.rev(!FOUnify.subList)
    val () = FOUnify.resolveSubs2()
    val subs = !FOUnify.subList
    val conditions = foldl (fn(f,res)=> (f(subs) andalso res)) true conds
    val () = if conditions then () else raise Fail "Conditions not met!"
    val newTerm = case dir of
                  LEFT => FirstOrder.subAll rhsu subs
                | RIGHT => FirstOrder.subAll lhsu subs

    val newTerm = FirstOrder.gVarToVar(newTerm)
    val newKATTerm1 = FirstOrder.slangCToKAT(newTerm)
    val newKATTerm = case dir of
                  LEFT => EQ(cf,newKATTerm1)
                | RIGHT => EQ(newKATTerm1,cf)
    fun indexOfA(y,list,n) = case list of
        [] => ~1
      | (x::xs) => if x = y then n else indexOfA(y,xs,n+1)
    val (id,(blist,clist)) = getCurrentTable();
    val () =foTable:= (id,(FirstOrder.singleBools(),FirstOrder.singleTerms()))::
                        (tl(!foTable))
    val () = cut ~1 (String.tokens Char.isSpace (Term.eqnToString(newKATTerm)));
    val () = use ["A"^Int.toString(List.length(specPre)), dirToString(dir)]
    val () = ignore ["A"^Int.toString(List.length(specPre))]
    val proof = getCurrentProof()
    val tasks = getCurrentTasks()
    val taskid = "T~1"
    val subList = case dir of
                   LEFT => FOUnify.makeSubList(FirstOrder.flatten(transed),lhs)@
                           FOUnify.makeSubList(FirstOrder.flatten(newTerm),rhs)
                 |  RIGHT => FOUnify.makeSubList(FirstOrder.flatten(transed),rhs)@
                             FOUnify.makeSubList(FirstOrder.flatten(newTerm),lhs)
    val () = proof := replaceTask (APPLYF(CONST(name),
               subList))
               (taskid) (!proof)
    val task = getCurrentTask()

    val (taskId,((specPre,specCon),preProofs,focus)) =
      case !task of
        SOME x => x
      | NONE => raise Fail "no current task"
  in
    (get [taskId]; undostates := laststates)
  end)
  in
    (case dir of
                 LEFT => FOUnify.unify(lhsu,transed) cont
               | RIGHT => FOUnify.unify(rhsu,transed) cont)
  end
  handle Fail x => (!alert) x

  and citeOrUse (args:string list, citing:bool) = let
    val tasks = getCurrentTasks()
    val task = getCurrentTask()
    val (taskId,((specPre,specCon),preProofs,focus)) =
      case !task of
        SOME x => x
      | NONE => raise Fail "no current task"
    val proof = getCurrentProof()
    val (name,dir) =
      case args of
        [] => raise Fail "no theorem or premise specified"
      | [x] => (x,if focusing() then getDirection() else LEFT)
      | [x,d] => if focusing() then (x,direction d)
                 else raise Fail "too many arguments cite"
      | _ => raise Fail "too many arguments cite"
    val _ = (!printtext) ("cite "^name)
    val (genPre,genCon,genProof) =
      if citing then
        let val ((genPre,genCon),_,_) = valOf(lookup name (!library))
        in (genPre,genCon,CONST name)
        end handle Option => raise noTheorem(name,dir)
      else if String.isPrefix "A" name then
        let val intName = Int.fromString(String.extract(name,1,NONE))
        in case intName of
          SOME j => (([],List.nth(specPre,j),List.nth(preProofs,j))
            handle Subscript => raise Fail "no such premise")
        | NONE => raise Fail "no such premise"
        end
      else raise Fail "invalid premise"
    val genCon =
      case (genCon,dir,focusing()) of
        (EQ(gen1,gen2),RIGHT,true) => EQ(gen2,gen1)
      | (LE _,_,true) => raise Fail "may not use inequality in focused citation"
      | _ => genCon
    val (gen1,gen2) = Term.args genCon
    fun getCTerms(lst,cterms) = case lst of
       nil => nil
     | (x::xs) => let val cterm = lookup (ACT_CON(x)) (revPairs(cterms))
       in case cterm of
            NONE => getCTerms(xs,cterms)
          | SOME(term) => (x,term)::getCTerms(xs,cterms)
       end

    fun getBTerms(lst,bterms) = case lst of
       nil => nil
     | (x::xs) => let val cterm = lookup (TST_CON(x)) (revPairs(bterms))
       in case cterm of
            NONE => getBTerms(xs,bterms)
          | SOME(term) => (x,term)::getBTerms(xs,bterms)
       end
    val foterms = lookup name (!foTable)
    val allVars =  condEqnVariables(genPre,genCon)
    val focterms = case foterms of
      NONE  => ([],[])
    | SOME(bterms,cterms) => (getCTerms(allVars,cterms),
                              getBTerms(allVars,bterms))

(*val () = app (fn (x,y) => ((!printtext) x ;
((!printtext) o FirstOrder.cTermToString) y)) (#1focterms)*)
    fun allSingles(subst:substitution) = List.all (fn (_,x) => case x of
     ((ACT_VAR(_)) | (ACT_CON(_))
      | (TST_VAR(_)) | (TST_CON(_))) => true
    | _ => false) subst
    fun compareCTerms subs (ctkat,cterm) = let
      val fromKAT = valOf(lookup ctkat subs)
      handle Option => raise Fail "First order terms do not unify."
      val foTerm = valOf(FirstOrder.findKATC(fromKAT))
      handle Option => raise Fail "First order terms do not unify."
(*
val () = app (fn (x,y)=>((!printtext) x; ((!printtext) o Term.toString) y)) subs
val () = (!printtext) (FirstOrder.cTermToString(cterm))
val () = (!printtext) (FirstOrder.cTermToString(foTerm))*)
    in
      FOUnify.unifyC(cterm,foTerm)
    end
    val oldLastTaskId = !nextTaskId
  in
    if focusing() then
      (* focused substitution *)
      let
        val cf = Term.getFocusInEqn(specCon,focus)
        val unif = Unify.unify(gen1,cf)
      in
        case unif of
          NONE => raise Fail "citation does not apply"
        | SOME s => let
            val sv = substVariables s
            val tv = removeDuplicates(List.concat(map eqnVariables genPre) @
                                      (variables gen2))
            val d = diff tv sv
            (* Continuation for after ambiguties and holes have been resolved *)
            val cont = (fn missing => let
              val s = ListPair.zip(d,missing) @ s
              val () = if ((fn (x,y) => not(List.null(x) andalso List.null(y)))
                           focterms) andalso not(allSingles(s))
                       then raise Fail "First order terms won't unify"
                       else ()
              val () = app (compareCTerms s) (#1(focterms))
              val newTaskCons = map (Unify.applyToEqn s) genPre
              val newTasks = foldl
                 (fn (x,lst) =>
                  ((case List.find (fn(_,(eqn,_,_))=> eqn=(specPre,x))
                      (getCurrentTasks()) of
                     NONE => newTaskId()
                   | SOME (id,(_,_,_))=>id),
                      ((specPre,x),preProofs,[]))::lst)
                        [] newTaskCons
              val newTaskIds = map #1 newTasks
              val (newSpecCon,newFocus) =
                Term.focusedSubstInEqn(specCon,focus,Unify.apply s gen2)
              val newCurrentTaskId =
                        (case List.find
                           (fn(_,(eqn,_,_))=>eqn=(specPre,newSpecCon))
                           (getCurrentTasks()) of
                           NONE => newTaskId()
                         | SOME (id,(_,_,_))=>id)
              val newCurrentTask =
                (newCurrentTaskId,((specPre,newSpecCon),preProofs,newFocus))
              val newProof =
                Proof.citeFocused (!proof,genProof,s,taskId,focus,
                                    dir,specCon,newCurrentTask,newTasks)

            in
              saveState();
              task := SOME newCurrentTask;
              proof := newProof;
              (if circCheck() then (undo(); raise Fail "Circularity in proof!")
               else ())
            end)
          in (!getMissing) d [] cont
          end
     end

    else
      (* unfocused substitution *)
      let
        val unif = Unify.unifyEqn(genCon,specCon)
      in
        case unif of
          NONE => raise Fail "citation does not apply"
        | SOME s => let
            val sv = substVariables s
            val tv = removeDuplicates(List.concat(map eqnVariables genPre))
            val d = diff tv sv
            (* Continuation for after ambiguties and holes have been resolved *)
            val cont = (fn missing => let
              val s = ListPair.zip(d,missing) @ s
              val () = if ((fn (x,y) => not(List.null(x) andalso List.null(y)))
                           focterms) andalso not(allSingles(s))
                       then raise Fail "First order terms won't unify"
                       else ()
              val () = app (compareCTerms s) (#1(focterms))
              val newTaskCons = map (Unify.applyToEqn s) genPre
              val newTasks =
                        map (fn x => (newTaskId(),((specPre,x),preProofs,[]))) newTaskCons
              val newTaskIds = map #1 newTasks
              val newProof = Proof.cite (!proof,genProof,s,taskId,newTasks)
              val newTask =
                case (newTasks,getTasks newProof) of
                  (x::_,_) => SOME x
                | ([],[x]) => SOME x
                | ([],x::xs) => SOME (hd(rev(xs)))
                | ([],[]) => NONE
            in
              saveState();
              task := newTask;
              proof := newProof;

              if circCheck() then (undo(); raise Fail "Circularity in proof!")
              else ();
              if newTasks = [] then (!done) "task completed" else ()
            end)
          in
            (!getMissing) d [] cont
          end
      end
  end
  handle Fail x => (!alert) x | noTheorem(name,dir) => citeFO(name,dir)

  and cite args = citeOrUse(args,true)
  and use args =  citeOrUse(args,false)

(***********************************************
 * rename a published theorem
 ***********************************************)

  fun rename args = let
    val (oldName,thm) = getCurrentTheorem()
    val (_,(btable,ctable)) = hd(!foTable)
    val name = case args of
      [x] => x
    | _ => raise Fail "invalid name"
    val _ = if isSome(lookup name (!library))
      then raise Fail "name already exists"
      else ()
  in
    saveState();
    foTable := (name,(btable,ctable))::tl(!foTable);
    library := (name,thm)::tl(!library);
    app (Proof.rename(oldName,name) o #3 o #2) (!library)
  end
  handle Fail x => (!alert) x

(***********************************************
 * normalize
 ***********************************************)

  fun normalize args = let
    val proof = getCurrentProof()
    val task = getCurrentTask()
  in
    saveState();
    proof := Proof.normalize(!proof);
    resetCurrentTask()
  end
  handle Fail x => (!alert) x


(***********************************************
 * verify
***********************************************)
val verlist:id list ref = ref []
val verflag = ref false
fun verify _ = let
  val direction = ref "L"
  val lastTask = ref []
  val laststates = ((copyState(),(!foTable)))::(!undostates)
(* get proof, renumbering proof terms to start at 0 *)
  val proof = alphaRedux(!(getCurrentProof()))
(* Suppress output *)
  val ptold = !printtext
  val alold = !alert
  val dnold = !done
  val gmold = !getMissing
  val rsold = !Unify.resolveSubst
(*  val () = printtext := (fn _ => ())*)
  val () = alert := (fn _ => ())
  val () = done := (fn _ => ())
  val () = reset()
(* Fun to compare final proofs *)
  fun compareProofs(p1,p2) = case (p1,p2) of
    (LAMBDAS(id1,p1),LAMBDAS(id2,p2)) => sameElements(id1,id2) andalso
                                        toString(p1) = toString(p2)

   | _ => ((!printtext) "HERE" ; false)
(* Fun to find index, used for finding prem in list *)
  fun indexOfA(y,list,n) = case list of
        [] => ~1
      | (x::xs) => if x = y then n else indexOfA(y,xs,n+1)
(* Get the number of the current task *)
  fun getTaskNum() = let
      val task = getCurrentTask();
      val (taskId,_) = (valOf(!task))
    in
      valOf(Int.fromString(String.extract(taskId,1,NONE)))
    end
(* Validate a proof *)
  fun valProof(prf) = case prf of
(* Skip over lambdas *)
    LAMBDAS(_,proof) => valProof(proof)
  | LAMBDAP(_,proof) => valProof(proof)
(* Apply a theorem *)
  | CONST(x) => (if (focusing()) then cite[x,!direction] else cite[x] ;
                 verlist := x::(!verlist))
(* Use a premise *)
  | VAR(x) => let
      val prem = "A"^Int.toString(valOf(Int.fromString(String.extract(x,1,NONE))))
    in
      if (focusing()) then use[prem,!direction] else use[prem]
    end
(* Citation that created new tasks *)
  | APPLYP(proof,plist) => let
      fun doPrem(prf,index) = (get["T"^Int.toString(index)] ;
                               valProof(prf) ;  index + 1)
      val () = valProof(proof)
      (* If focusing, an additional task exists in which the subst was made;
         ignore it *)
      val lastNum = getTaskNum()
      val wasfocused = focusing()
      val premNums = if (wasfocused) then lastNum + 1 else lastNum
    in
      (foldl doPrem (premNums) plist ;
       if wasfocused then get["T"^Int.toString(lastNum)] else ())
    end
(* Application of a substitution *)
  | APPLYF(proof,sub) => let
     fun makeSubList(subs) = case subs of
       [] => []
     | ((Boolean(_),_)::xs) => makeSubList(xs)
     | ((Assign(l,Loc(a)),term)::xs) => let
        val Assign(l2,a2) = valOf(FirstOrder.findKATC(makeConstant(term)))
       in
        ((l,Loc(l2))::(a,a2)::makeSubList(xs))
       end
     | ((x,term)::xs) => makeSubList(xs)
     val subList = makeSubList(sub)
     val () = FOUnify.getMissingFO :=
       (fn list => fn cont =>
         (let fun myCont list =
            case list of
              nil => cont ()
            | (x::xs) => (let
                      val () = FOUnify.unifyA(Loc(x),valOf(lookup x subList))
                          in (myCont xs)
                          end)
          in
              (myCont list)
          end))
      in
        valProof(proof)
      end
  | APPLYS(proof,sub) => let
    val sub = map (fn(x,y) => (x, makeConstant y)) sub
    val () = Unify.resolveSubst := (fn _ => SOME(sub))
    val () = getMissing := let
      fun getMiss(ses:id list) (missedterms:term list) (cont:term list -> unit) =
        case ses of
          []  => cont missedterms
        | (x::xs) => getMiss xs (makeConstant((valOf(lookup x sub)))::missedterms)
                     cont
      in getMiss
      end
    in
      case proof of
        AXIOM(x) => let
          val subasEQ = EQ(ACT_CON(#1(hd(sub))),makeConstant(#2(hd(sub))))
          val () = define[#1(hd(sub))]
          val task = getCurrentTask()
          val (taskId,((specPre,specCon),preProofs,oldfocus)) =
            case !task of
              SOME x => x
            | NONE => raise Fail "no current task"
         in
           use["A"^Int.toString(indexOfA(subasEQ,specPre,0)),!direction]
         end

      | _ => valProof(proof)
    end
(* Focused substitution *)
  | SUBST(focus,eqn,dir,p,q) => let
     val () = direction := directionToString(dir)
     val task = getCurrentTask()
     val (taskId,((specPre,specCon),preProofs,oldfocus)) =
       case !task of
         SOME x => x
       | NONE => raise Fail "no current task"
     val () = task := SOME(taskId,((specPre,specCon),preProofs,focus))
     val () = case p of
                 (* Need to cut *)
                SUBST(f2,e2,d2,p2,q2) => let
                  val () = cut 0 (String.tokens (fn x => x = (#" ")) (eqnToString(e2)))
                  val () = lastTask := (getTaskNum())::(!lastTask)
                  val largestID = let
                    val tasks = getCurrentTasks()
                  in
                     foldl (fn (x,y) =>
                       if (valOf(Int.fromString(String.extract((#1(x)),1,NONE)))) > y
                       then valOf(Int.fromString(String.extract((#1(x)),1,NONE)))
                       else y) 0 tasks
                  end
                  val () = get ["T"^Int.toString(largestID)]
                  val () = valProof(p)
                   val () = get[("T"^(Int.toString(hd(!lastTask))))]
                  val task = getCurrentTask()
                  val (taskId,((specPre,specCon),preProofs,oldfocus)) =
                  case !task of
                    SOME x => x
                  | NONE => raise Fail "no current task"
                  val () = task := SOME(taskId,((specPre,specCon),preProofs,focus))
                  val () = direction := directionToString(dir)
                  val () = lastTask := tl(!lastTask)
                in
                  use["A"^Int.toString(indexOfA(makeConstantEqn(e2),specPre,0)),(!direction)]
                end
               | _ => valProof(p)
     val () = unfocus()
    in
     valProof(q)
    end
   | _ => raise Fail "Problem with verify!"
in
   case proof of
    AXIOM(_) => ((!printtext) "Axiom, no proof necessary."; verflag := true; undo())
    | _ => (valProof(proof) ; printtext := ptold ; alert := alold ;
           done := dnold; getMissing := gmold;
           (getCurrentProof()) := alphaRedux(!(getCurrentProof()));
           undostates := laststates;
           if compareProofs(proof,!(getCurrentProof()))
            then (verflag := true ; (!printtext) "Proof verified.")
            else ((!printtext) "Proof cannot be verified." ; undo();
                  verlist := []; verflag := false))
end
handle _ => (verflag := false ; (!printtext) "Proof cannot be verified.")
val verified:id list ref = ref []
fun deepverify _ = let
  val thm = (#1(hd(!library)))
  val () = verlist := [thm]
  val () = verified := []
  fun doverify() = case (!verlist) of
    nil => ()
  | (x::xs) => let val () = verlist := tl(!verlist)
                   val wasVerified = member x (!verified)
               in
                 if wasVerified then doverify()
                 else ((!printtext) ("Verifying "^x);
                       get[x]; verified := x::(!verified); verify() ; doverify())
               end
in
  (doverify() ; get[thm])
end




(***********************************************
 * toLaTeX
***********************************************)
fun proofToLaTeX _ = let
  val direction = ref "L"
  val lastTask = ref []
  val laststates = ((copyState(),(!foTable)))::(!undostates)
(* get proof, renumbering proof terms to start at 0 *)
  val proof = alphaRedux(!(getCurrentProof()))
(* Suppress output *)
  val ptold = !printtext
  val alold = !alert
  val dnold = !done
  val gmold = !getMissing
  val rsold = !Unify.resolveSubst
(*  val () = printtext := (fn _ => ())*)
  val () = alert := (fn _ => ())
  val () = done := (fn _ => ())
  val () = reset()
  val task = getCurrentTask()
  val (taskId,((specPre,specCon),preProofs,oldfocus)) =
     case !task of
       SOME x => x
     | NONE => raise Fail "no current task"
  val output = ref "\\begin{theorem}\n"
  val assumecount = ref ~1;
  val () = if (not(null(specPre))) then output := (foldl (fn (x,y) => (assumecount := !assumecount + 1;
                                 y^"\\label{A"^(Int.toString(!assumecount))^"}\n"^(eqnToLaTeX x)
                                 ^"\\\\")) (!output^"If\n\\begin{eqnarray}\n") (rev(tl(rev(specPre))))) else ()
  val () = assumecount := !assumecount + 1
  val () =  if(not(null(specPre))) then output := (!output)^"\\label{A"^(Int.toString(!assumecount))^"}\n"^(eqnToLaTeX (hd(specPre)))
                                 ^"\n\\end{eqnarray}\nthen" else (output := (!output)^"\n")
  val () = output := (!output)^"\\["^(eqnToLaTeXNA specCon)^"\\]\n"
  val () = let val (_,(btable,ctable)) =  getCurrentTable()
           in
              if List.null(ctable) andalso List.null(btable) then ()
              else output := (!output)^"where\n\\begin{eqnarray*}\n"^
                             (foldr (fn ((x,y),z) => z^("\\mathsf{"^Term.toString(y))^"}&=&\\mathtt{"^(FirstOrder.cTermToString(x))^"}\\\\\n") "" ctable)^
                             (foldr (fn ((x,y),z) => z^(Term.toString(y))^"&=&"^(FirstOrder.bTermToString(x))^"\\\\\n")  "" btable)^
                             "\\end{eqnarray*}"
           end
  val () = output := (!output)^"\\end{theorem}"
(* Fun to compare final proofs *)
  fun compareProofs(p1,p2) = case (p1,p2) of
    (LAMBDAS(id1,p1),LAMBDAS(id2,p2)) => sameElements(id1,id2) andalso
                                        toString(p1) = toString(p2)
   | _ => false
(* Fun to find index, used for finding prem in list *)
  fun indexOfA(y,list,n) = case list of
        [] => ~1
      | (x::xs) => if x = y then n else indexOfA(y,xs,n+1)
(* Get the number of the current task *)
  fun getTaskNum() = let
      val task = getCurrentTask();
      val (taskId,_) = (valOf(!task))
    in
      valOf(Int.fromString(String.extract(taskId,1,NONE)))
    end
(* Validate a proof *)
 val currentTask = getTaskNum();
  fun valProof(prf) = case prf of
(* Skip over lambdas *)
    LAMBDAS(_,proof) => valProof(proof)
  | LAMBDAP(_,proof) => valProof(proof)
(* Apply a theorem *)
  | CONST(x) => (if (focusing()) then cite[x,!direction] else cite[x] ;
                 verlist := x::(!verlist) ; "By "^x^", ")
(* Use a premise *)
  | VAR(x) => let
      val prem = "A"^Int.toString(valOf(Int.fromString(String.extract(x,1,NONE))))
    in
      ((if (focusing()) then use[prem,!direction] else use[prem]) ; "By "^"(\\ref{"^prem^"}), ")
    end
(* Citation that created new tasks *)
  | APPLYP(proof,plist) => let
      val preProofSTR = ref []
      fun printPrem(prf,(out,index)) = let
        val () = get["T"^Int.toString(index)]
        val SOME (_,((_,e),_,_)) = !(getCurrentTask())
        val output = out^"\\label{T"^(Int.toString(index))^"}\n"^(eqnToLaTeX(e))^"\\\\\n"
      in
        (output,index + 1)
      end
      fun doPrem(prf,index) = (get["T"^Int.toString(index)] ;
                               preProofSTR := (valProof(prf)::(!preProofSTR)) ;  index + 1)
      val out = (valProof(proof)^"it suffices to show that\n\\begin{eqnarray}\n")
      (* If focusing, an additional task exists in which the subst was made;
         ignore it *)
      val lastNum = getTaskNum()
      val wasfocused = focusing()
      val premNums = if (wasfocused) then lastNum + 1 else lastNum
      val out = (#1((foldl printPrem (out,premNums) plist)))
      val out = String.extract(out,0,SOME (String.size(out)-3))^"\n\\end{eqnarray}\n"
      val _ = foldl doPrem (premNums) plist
      val _ = if wasfocused then get["T"^Int.toString(lastNum)] else ()
      fun makeProof(prf,(out,index)) = (out^"Consider (\\ref{T"^(Int.toString(index))^"}).  "^prf,index+1)
      val sofar = #1(foldr makeProof (out,premNums) (!preProofSTR))
    in
      if endsWith(sofar,"need.\n") then sofar else sofar^"we have what we need.\n"
    end
  | APPLYS(proof,sub) => let
    val sub = map (fn(x,y) => (x, makeConstant y)) sub
    val () = Unify.resolveSubst := (fn _ => SOME(sub))
    val () = getMissing := let
      fun getMiss(ses:id list) (missedterms:term list) (cont:term list -> unit) =
        case ses of
          []  => cont missedterms
        | (x::xs) => getMiss xs (makeConstant((valOf(lookup x sub)))::missedterms)
                     cont
      in getMiss
      end
    in
      case proof of
        AXIOM(x) => let
          val subasEQ = EQ(ACT_CON(#1(hd(sub))),makeConstant(#2(hd(sub))))
          val () = define[#1(hd(sub))]
          val task = getCurrentTask()
          val (taskId,((specPre,specCon),preProofs,oldfocus)) =
            case !task of
              SOME x => x
            | NONE => raise Fail "no current task"
         in
           use["A"^Int.toString(indexOfA(subasEQ,specPre,0)),!direction]; "We define $\\mathsf{"^(#1(hd(sub)))^"}$ to be $\\mathsf{"^toLaTeX(#2(hd(sub)))^"}$.  Therefore,"
         end

      | _ => valProof(proof)
    end
(* Application of a substitution *)
  | APPLYF(proof,sub) => let
     fun makeSubList(subs) = case subs of
       [] => []
     | ((Boolean(_),_)::xs) => makeSubList(xs)
     | ((Assign(l,Loc(a)),term)::xs) => let
        val Assign(l2,a2) = valOf(FirstOrder.findKATC(makeConstant(term)))
       in
        ((l,Loc(l2))::(a,a2)::makeSubList(xs))
       end
     | ((x,term)::xs) => makeSubList(xs)
     val subList = makeSubList(sub)
     val () = FOUnify.getMissingFO :=
       (fn list => fn cont =>
         (let fun myCont list =
            case list of
              nil => cont ()
            | (x::xs) => (let
                      val () = FOUnify.unifyA(Loc(x),valOf(lookup x subList))
                          in (myCont xs)
                          end)
          in
              (myCont list)
          end))
      in
        valProof(proof)
      end
(* Focused substitution *)
  | SUBST(focus,eqn,dir,p,q) => let
     val () = direction := directionToString(dir)
     val task = getCurrentTask()
     val (taskId,((specPre,specCon),preProofs,oldfocus)) =
       case !task of
         SOME x => x
       | NONE => raise Fail "no current task"
     val () = task := SOME(taskId,((specPre,specCon),preProofs,focus))
     val subProof = case p of
                 (* Need to cut *)
                SUBST(f2,e2,d2,p2,q2) => let
                  val () = cut 0 (String.tokens (fn x => x = (#" ")) (eqnToString(e2)))
                  val () = lastTask := (getTaskNum())::(!lastTask)
                  val largestID = let
                    val tasks = getCurrentTasks()
                  in
                     foldl (fn (x,y) =>
                       if (valOf(Int.fromString(String.extract((#1(x)),1,NONE)))) > y
                       then valOf(Int.fromString(String.extract((#1(x)),1,NONE)))
                       else y) 0 tasks
                  end
                  val () = get ["T"^Int.toString(largestID)]
                  val fromProof = valProof(p)
                   val () = get[("T"^(Int.toString(hd(!lastTask))))]
                  val task = getCurrentTask()
                  val (taskId,((specPre,specCon),preProofs,oldfocus)) =
                  case !task of
                    SOME x => x
                  | NONE => raise Fail "no current task"
                  val () = task := SOME(taskId,((specPre,specCon),preProofs,focus))
                  val () = direction := directionToString(dir)
                  val () = lastTask := tl(!lastTask)
                        val out = fromProof^"we know that\n\\begin{eqnarray}\n"
                        val oldside = case hd(focus) of
                            0 => #1(args(specCon))
                          | 1 => #2(args(specCon))
                          | _ => raise Fail "Problem with toLaTeX!"
                        val task = getCurrentTask()
                        val (_,((_,newspecCon),_,_)) =
                          case !task of
                            SOME x => x
                          | NONE => raise Fail "no current task"
                        val newside = case hd(focus) of
                            0 => #1(args(newspecCon))
                          | 1 => #2(args(newspecCon))
                          | _ => raise Fail "Problem with toLaTeX!"
                in
                  (use["A"^Int.toString(indexOfA(makeConstantEqn(e2),specPre,0)),(!direction)] ;
                   out^eqnToLaTeX(EQ(oldside,newside))^"\n\\end{eqnarray}" )
                end
               | _ => let
                        val theProof = valProof(p)
                        val out = theProof^"we know that\n\\begin{eqnarray*}\n"
                        val oldside = case hd(focus) of
                            0 => #1(args(specCon))
                          | 1 => #2(args(specCon))
                          | _ => raise Fail "Problem with toLaTeX!"
                        val task = getCurrentTask()
                        val (_,((_,newspecCon),_,_)) =
                          case !task of
                            SOME x => x
                          | NONE => raise Fail "no current task"
                        val newside = case hd(focus) of
                            0 => #1(args(newspecCon))
                          | 1 => #2(args(newspecCon))
                          | _ => raise Fail "Problem with toLaTeX!"
                      in
                        out^eqnToLaTeX(EQ(oldside,newside))^"\n\\end{eqnarray*}"
                      end
     val () = unfocus()
    in
     subProof^(valProof(q))
    end
   | _ => raise Fail "Problem with toLaTeX"
in
   case proof of
    AXIOM(_) => ((!printtext) "Axiom, no proof necessary."; verflag := true; undo(); "")
    | _ => (let val out = valProof(proof)
                val out = if String.sub(out,String.size(out)-2) = #","
                          then out^" the proof is complete. $\\Box$"
                          else out^"$\\Box$"
            in
           printtext := ptold ; alert := alold ;
           done := dnold; getMissing := gmold;
           (getCurrentProof()) := alphaRedux(!(getCurrentProof()));
           undostates := laststates;
           if compareProofs(proof,!(getCurrentProof()))
            then (verflag := true ; (!printtext) "Proof verified.")
            else ((!printtext) "Proof cannot be verified." ; undo();
                  verlist := []; verflag := false) ; (!output)^out
           end)
end

(***********************************************
 * dump
 **********************************************)
fun dump _ = let
  val direction = ref "L"
  val lastTask = ref []
  val laststates = ((copyState(),(!foTable)))::(!undostates)
(* get proof, renumbering proof terms to start at 0 *)
  val proof = alphaRedux(!(getCurrentProof()))
(* Suppress output *)
  val ptold = !printtext
  val alold = !alert
  val dnold = !done
  val gmold = !getMissing
  val rsold = !Unify.resolveSubst
(*  val () = printtext := (fn _ => ())*)
  val () = alert := (fn _ => ())
  val () = done := (fn _ => ())
  val () = reset()
  val task = getCurrentTask()
  val (taskId,((specPre,specCon),preProofs,oldfocus)) =
     case !task of
       SOME x => x
     | NONE => raise Fail "no current task"
  val output = ref ("pub "^(theoremToString (getCurrentTheorem())))
(* Fun to compare final proofs *)
  fun compareProofs(p1,p2) = case (p1,p2) of
    (LAMBDAS(id1,p1),LAMBDAS(id2,p2)) => sameElements(id1,id2) andalso
                                        toString(p1) = toString(p2)
   | _ => false
(* Fun to find index, used for finding prem in list *)
  fun indexOfA(y,list,n) = case list of
        [] => ~1
      | (x::xs) => if x = y then n else indexOfA(y,xs,n+1)
(* Get the number of the current task *)
  fun getTaskNum() = let
      val task = getCurrentTask();
      val (taskId,_) = (valOf(!task))
    in
      valOf(Int.fromString(String.extract(taskId,1,NONE)))
    end
(* Validate a proof *)
 val currentTask = getTaskNum();
  fun valProof(prf) = case prf of
(* Skip over lambdas *)
    LAMBDAS(_,proof) => valProof(proof)
  | LAMBDAP(_,proof) => valProof(proof)
(* Apply a theorem *)
  | CONST(x) => (if (focusing()) then cite[x,!direction] else cite[x] ;
                 verlist := x::(!verlist) ; "\ncite "^x^
                           (if focusing() then (" "^(!direction)) else ""))
(* Use a premise *)
  | VAR(x) => let
      val prem = "A"^Int.toString(valOf(Int.fromString(String.extract(x,1,NONE))))
    in
      ((if (focusing()) then use[prem,!direction] else use[prem]) ; "\nuse "^prem^(if focusing() then (" "^(!direction)) else ""))
    end
(* Citation that created new tasks *)
  | APPLYP(proof,plist) => let
      val preProofSTR = ref []
      fun printPrem(prf,(out:string,index)) = let
        val () = get["T"^Int.toString(index)]
        val SOME (_,((_,e),_,_)) = !(getCurrentTask())
      in
        (out,index + 1)
      end
      fun doPrem(prf,index) = (get["T"^Int.toString(index)] ;
                               output := (!output)^"\nget T"^(Int.toString(index));
                               preProofSTR := (valProof(prf)::(!preProofSTR)) ;  index + 1)
      val out = (valProof(proof))
      (* If focusing, an additional task exists in which the subst was made;
         ignore it *)
      val lastNum = getTaskNum()
      val wasfocused = focusing()
      val premNums = if (wasfocused) then lastNum + 1 else lastNum
      val out = (#1((foldl printPrem (out,premNums) plist)))
      val out = String.extract(out,0,SOME (String.size(out)-3))^"\n\\end{eqnarray}\n"
      val _ = foldl doPrem (premNums) plist
      val _ = if wasfocused then get["T"^Int.toString(lastNum)] else ()
      fun makeProof(prf,(out,index)) = (out^"\nget T"^(Int.toString(index)),index+1)
      val sofar = #1(foldr makeProof (out,premNums) (!preProofSTR))
    in
	sofar
    end
  | APPLYS(proof,sub) => let
    val sub = map (fn(x,y) => (x, makeConstant y)) sub
    val () = Unify.resolveSubst := (fn _ => SOME(sub))
    val () = getMissing := let
      fun getMiss(ses:id list) (missedterms:term list) (cont:term list -> unit) =
        case ses of
          []  => cont missedterms
        | (x::xs) => getMiss xs (makeConstant((valOf(lookup x sub)))::missedterms)
                     cont
      in getMiss
      end
    in
      case proof of
        AXIOM(x) => let
          val subasEQ = EQ(ACT_CON(#1(hd(sub))),makeConstant(#2(hd(sub))))
          val () = define[#1(hd(sub))]
          val task = getCurrentTask()
          val (taskId,((specPre,specCon),preProofs,oldfocus)) =
            case !task of
              SOME x => x
            | NONE => raise Fail "no current task"
         in
           use["A"^Int.toString(indexOfA(subasEQ,specPre,0)),!direction]; "\ndefine "^(#1(hd(sub)))^"\nuse A"^(Int.toString(indexOfA(subasEQ,specPre,0)))^" "^
             (!direction)
         end

      | _ => valProof(proof)
    end
(* Application of a substitution *)
  | APPLYF(proof,sub) => let
     fun makeSubList(subs) = case subs of
       [] => []
     | ((Boolean(_),_)::xs) => makeSubList(xs)
     | ((Assign(l,Loc(a)),term)::xs) => let
        val Assign(l2,a2) = valOf(FirstOrder.findKATC(makeConstant(term)))
       in
        ((l,Loc(l2))::(a,a2)::makeSubList(xs))
       end
     | ((x,term)::xs) => makeSubList(xs)
     val subList = makeSubList(sub)
     val () = FOUnify.getMissingFO :=
       (fn list => fn cont =>
         (let fun myCont list =
            case list of
              nil => cont ()
            | (x::xs) => (let
                      val () = FOUnify.unifyA(Loc(x),valOf(lookup x subList))
                          in (myCont xs)
                          end)
          in
              (myCont list)
          end))
      in
        valProof(proof)
      end
(* Focused substitution *)
  | SUBST(focus,eqn,dir,p,q) => let
     val () = direction := directionToString(dir)
     val task = getCurrentTask()
     val (taskId,((specPre,specCon),preProofs,oldfocus)) =
       case !task of
         SOME x => x
       | NONE => raise Fail "no current task"
     val () = task := SOME(taskId,((specPre,specCon),preProofs,focus))
     val outFocus = "\n"^(moveToFocus focus specCon)
     val subProof = case p of
                 (* Need to cut *)
                SUBST(f2,e2,d2,p2,q2) => let
                  val () = cut 0 (String.tokens (fn x => x = (#" ")) (eqnToString(e2)))
                  val () = lastTask := (getTaskNum())::(!lastTask)
                  val largestID = let
                    val tasks = getCurrentTasks()
                  in
                     foldl (fn (x,y) =>
                       if (valOf(Int.fromString(String.extract((#1(x)),1,NONE)))) > y
                       then valOf(Int.fromString(String.extract((#1(x)),1,NONE)))
                       else y) 0 tasks
                  end
                  val () = get ["T"^Int.toString(largestID)]
                  val fromProof = valProof(p)
                   val () = get[("T"^(Int.toString(hd(!lastTask))))]
                  val task = getCurrentTask()
                  val (taskId,((specPre,specCon),preProofs,oldfocus)) =
                  case !task of
                    SOME x => x
                  | NONE => raise Fail "no current task"
                  val () = task := SOME(taskId,((specPre,specCon),preProofs,focus))
                  val () = direction := directionToString(dir)
                  val () = lastTask := tl(!lastTask)
                  val out = "\nget T"^(Int.toString(largestID))^fromProof^
                            "\nget T"^(Int.toString(hd(!lastTask)))
                  val oldside = case hd(focus) of
                         0 => #1(args(specCon))
                       | 1 => #2(args(specCon))
                       | _ => raise Fail "Problem with toLaTeX!"
                  val task = getCurrentTask()
                  val (_,((_,newspecCon),_,_)) =
                          case !task of
                            SOME x => x
                          | NONE => raise Fail "no current task"
                  val newside = case hd(focus) of
                            0 => #1(args(newspecCon))
                          | 1 => #2(args(newspecCon))
                          | _ => raise Fail "Problem with toLaTeX!"
                in
                  (use["A"^Int.toString(indexOfA(makeConstantEqn(e2),specPre,0)),(!direction)] ;
                   outFocus^out^"\nuse A"^(Int.toString(indexOfA(makeConstantEqn(e2),specPre,0)))^" "^(!direction))
                end
               | _ => let
                        val theProof = valProof(p)
                        val out = theProof
                        val oldside = case hd(focus) of
                            0 => #1(args(specCon))
                          | 1 => #2(args(specCon))
                          | _ => raise Fail "Problem with toLaTeX!"
                        val task = getCurrentTask()
                        val (_,((_,newspecCon),_,_)) =
                          case !task of
                            SOME x => x
                          | NONE => raise Fail "no current task"
                        val newside = case hd(focus) of
                            0 => #1(args(newspecCon))
                          | 1 => #2(args(newspecCon))
                          | _ => raise Fail "Problem with toLaTeX!"
                      in
                         outFocus^out
                      end
     val () = unfocus()
    in
     subProof^"\nunfocus"^(valProof(q))
    end
   | _ => raise Fail "Problem with toLaTeX"
in
   case proof of
    AXIOM(_) => ((!printtext) "Axiom, no proof necessary."; verflag := true; undo(); "")
    | _ => (let val out = valProof(proof)
            in
           printtext := ptold ; alert := alold ;
           done := dnold; getMissing := gmold;
           (getCurrentProof()) := alphaRedux(!(getCurrentProof()));
           undostates := laststates;
           if compareProofs(proof,!(getCurrentProof()))
            then (verflag := true ; (!printtext) "Proof verified.")
            else ((!printtext) "Proof cannot be verified." ; undo();
                  verlist := []; verflag := false) ; (!output)^out
           end)
end

(***********************************************
 * moveTerm
 * commute a term as far to the specified
   direction as possible
 ***********************************************)
fun moveTerm args = let
  val laststates = ((copyState(),(!foTable)))::(!undostates)
  fun allButFirstAndLast list = case list of
                                  [] => raise Continue
                                | (x::t) => rev(tl(rev(t)))
  handle Empty => raise Continue
  (* Reverse terms for plus and times, leave others alone *)
  fun revTerms(t) =
    case t of
      PLUS [x,y] => PLUS [y,x]
    | TIMES [x,y] => TIMES [y,x]
    | _ => t

  (* Direction of move *)
  val dir =
    case args of
      (["L"] | ["l"]) => 0
    | (["R"] | ["r"]) => 1
    | _ => raise Fail "Need a direction"

  val (taskId,((_,con),_,focus)) = valOf(!(getCurrentTask()))

  (* Get term and term to right or left *)
  val (currterm,_) = if (foldl op+ dir (allButFirstAndLast(focus))) <> 0
                     then  getFocusInEqnWithContext(con,focus)
                     else raise Continue

  (* A term of length 2? *)
  val validTerm = let
    val (_,((_,upeq),_,focus)) = (up(nil); valOf(!(getCurrentTask())))
    val (upterm,_) = getFocusInEqnWithContext(upeq,focus)

  in
    (case upterm of
      (PLUS l | TIMES l) => (undo(nil);
                            (if (List.length(l) < 3 orelse (dir = 0)) then true
                             else raise Continue))
     | _ => false)
  end

  (* What do do after the move *)
  val enddirs =
    case dir of
      1 => let val _ = up(nil)
                     in
                         ["r"]
                     end
    | 0 => let val _ = left(["u"])
                     in
                         nil
                     end
    | _ => raise Fail "Invalid direction"
  val (taskId,((pre,con),preProofs,focus)) = valOf(!(getCurrentTask()))
  val (comterms,_) = getFocusInEqnWithContext(con,focus)
  val revterms = revTerms(comterms)
in
  (if validTerm
  then

    (* Communtitivity rule in premises? *)
    case index (EQ(comterms,revterms)) pre of
      SOME n => (use ["A" ^ Int.toString n,"l"] ; down(enddirs); moveTerm(args))
    | NONE =>
       (case index (EQ(revterms,comterms)) pre of
          SOME n => (use ["A" ^ Int.toString n,"r"] ;
                     down(enddirs);
                     moveTerm(args))

        (* If not, is it in the library? *)
        | NONE => (case firstUnify(EQ(comterms,revterms),(!library)) of
                     NONE => undo()
                   | SOME x => (cite([x,"l"]); down(enddirs); moveTerm(args))))
 else ((!alert) "Fail!")); undostates := laststates
end
handle Continue => ()
handle Fail x => (!alert) x

(* Get translation table from imperative statements to KAT *)
fun viewFOTable _ = let
  val (_,(btable,ctable)) = hd(!foTable)

in
  app(fn x => outputstring := !outputstring @ [(#2 x)])
     (Util.sortByKey(map (fn (y,x) =>
     (Term.toString(x),(Term.toString(x)^": "^FirstOrder.bTermToString(y))))
     btable));
  app(fn x => outputstring := !outputstring @ [(#2 x)])
     (Util.sortByKey(map (fn (y,x) =>
     (Term.toString(x),(Term.toString(x)^": "^FirstOrder.cTermToString(y))))
     ctable))

end
  handle Empty => ()


(***********************************************
 * heuristics
 ***********************************************)

  type heuristic = string * ((unit -> unit) * bool ref)


  val heuristics : heuristic list ref = ref []

  fun doHeuristics _ = let
    fun doHeuristic (_,(action,status)) =
      if !status then action() else ()
  in
    (app doHeuristic (!heuristics))
  end

  (* automatically use premise that matches conclusion *)
  and AutoPremises _ = let
    val (taskId,((pre,con),_,_)) = valOf(!(getCurrentTask()))
      handle Option => raise Continue
    val laststate = !undostates
    val premise =
      case index con pre of
        SOME n => "A" ^ Int.toString n
      | NONE => raise Continue
  in
    unfocus([]);
    use [premise];
    undostates := laststate;
    doHeuristics()
  end
  handle Continue => ()

  (* automatically use theorems with no premises that match conclusion *)
  and AutoTheorem _ = let
    val (_,((_,con),_,_)) = valOf(!(getCurrentTask()))
      handle Option => raise Continue
    val (thmID,_) = getCurrentTheorem()
      handle Option => raise Continue
    val laststate = !undostates
    fun unifies libEntries =
     case libEntries of
       [] => NONE
     | ((libID,(([],genCon), _, _))::rest) =>
         (case (String.compare(libID,thmID),(Unify.unifyEqn(genCon,con))) of
            ((EQUAL,_) | (_,NONE)) => unifies rest
          | (_,SOME _) => SOME (libID))
     | (_::rest) => unifies rest
      handle Option => raise Continue
    val premise =
      case unifies (!library) of
        SOME n => n
      | NONE => raise Continue
  in
    unfocus([]);
    cite [premise];
    undostates := laststate;
    doHeuristics()
  end
  handle Continue => ()

(* List of all unfocused citations that unify
 *
 * Value is stored in usuggest, for printing *)
and AutoSuggest _ = let
    val (_,((_,con),_,_)) = valOf(!(getCurrentTask()))
      handle Option => raise Continue
    val (thmID,_) = getCurrentTheorem()
      handle Option => raise Continue

    (* Unify if the theorem has a different name and unifies *)
    fun unifies libEntries =
     case libEntries of
       [] => ()
     | ((entry as (libID,((genPre::genPres,genCon), _, _)))::rest) =>
         (case (String.compare(libID,thmID),(Unify.unifyEqnNoRes(genCon,con))) of
            ((EQUAL,_) | (_,NONE)) => unifies rest
          | (_,SOME _) => (usuggest := ((!usuggest) @ [(theoremToString entry)]);
                           unifies rest))
     | ((entry as (libID,((nil,genCon), _, _)))::rest) =>
         if !(#2(#2(hd(!heuristics)))) then (unifies rest)
         else
         (case (String.compare(libID,thmID),(Unify.unifyEqnNoRes(genCon,con))) of
            ((EQUAL,_) | (_,NONE)) => unifies rest
          | (_,SOME _) => (usuggest := ((!usuggest) @ [(theoremToString entry)]);                   unifies rest))
      handle Option => raise Continue
  in
    usuggest := [("\nSuggested Citations\n-------------------")];
    unifies (!library);
    AutoFSuggest()
  end
  handle Continue => ()

(* List of all focused citations that unify
 *
 * Value is stored in fsuggest, for printing *)
and AutoFSuggest _ = let
    val (_,((_,con),_,focus)) = valOf(!(getCurrentTask()))
      handle Option => raise Continue
    val (thmID,_) = getCurrentTheorem()
      handle Option => raise Continue
    val cf = if focus <> nil then Term.getFocusInEqn(con,focus) else raise Continue
    fun unifies libEntries =
     case libEntries of
       [] => ()
     | ((entry as (libID,((_,EQ(s1,s2)), _, _)))::rest) =>
         (case (String.compare(libID,thmID)) of
            EQUAL => unifies rest
          | _ => (case (Unify.unifyNoRes(s1,cf),Unify.unifyNoRes(s2,cf)) of
                   ((SOME _,_) | (_, SOME _)) => ((fsuggest := (!fsuggest) @
                                                  [(theoremToString entry)]);
                                                   unifies rest)
                 | (NONE,NONE) => (unifies rest)))
     | (_::rest) => (unifies rest)
      handle Option => raise Continue
  in
    fsuggest := [("\nFocused Citations\n-------------------")];
    unifies (!library)
  end
  handle Continue => ()

(*  and AutoReduce _ = reduce[]*)


  val () = heuristics :=  [(* automatic use of theorems *)
                           ("theorems",(AutoTheorem,ref false)),
                           (* automatic use of premises *)
                           ("premises",(AutoPremises,ref false)),
                           (* automatic suggestion of theorem *)
                           ("suggest",(AutoSuggest,ref false))
                           (* automatic reduction of theorem
                            * Not a good idea yet; overrides focus *)
                           (*("reduce",(AutoReduce,ref false))*)]

  fun printHeuristic (name,(_,status)) =
    (!printtext) (name ^ (if !status then " on" else " off"))

  fun heuristic args =
    case args of
      [] => app printHeuristic (!heuristics)
    | ("all"::t) => app (fn (name,_) => (heuristic (name::t))) (!heuristics)
    | h::t => let
        val heur as (_,(_,status)) =
          case findPrefix h (!heuristics) of
            SOME heur => heur
          | NONE => raise Fail ("no such heuristic " ^ h)
      in
        case t of
          [] => printHeuristic heur
         | ["on"] => status := true
        | ["off"] => status := false
        | _ => raise Fail "invalid arguments"
      end
  handle Fail x => (!alert) x


(*********************************************
 * reduce
 *********************************************)
  datatype reducer =
    THM of string * string
  | OR of reducer list
  | SEQ of reducer list
  | AST of reducer
  | MATCH of term
  | ISEMPTY of reducer
  | NONEMPTY of reducer


  val astBound = ref(SOME 10)

  val timeLimit = ref(SOME 1000.0)

  fun termLstToRedLst lst =
    case lst of
      [] => []
    | (PLUS  x)::xs => (OR (termLstToRedLst x))::(termLstToRedLst xs)
    | (TIMES x)::xs => (SEQ(termLstToRedLst x))::(termLstToRedLst xs)
    | (STAR  x)::xs => (AST(hd(termLstToRedLst [x])))::(termLstToRedLst xs)
    | (TST_VAR x|ACT_VAR x|TST_CON x|ACT_CON x)::
        (TST_VAR y|ACT_VAR y|TST_CON y|ACT_CON y)::xs =>
        let
          val arg = parseTerm y
        in
          (case String.map Char.toLower x of
             ("not"|"isempty") =>
               ISEMPTY(hd(termLstToRedLst
                          [case arg of NONE=>raise Fail "Invalid term"
                                          | SOME r => r]))
           | ("nonempty"|"try"|"tobool") =>
               NONEMPTY(hd(termLstToRedLst
                           [case arg of NONE=>raise Fail "Invalid term"
                         | SOME r => r]))
           | "match" => MATCH (case arg of
                                 NONE=>raise Fail "Invalid term"
                               | SOME r => r)
           | "premises" =>
               let
                 val numPrems =
                   (case !(getCurrentTask()) of
                      SOME (_,((specPre,_),_,_)) => length specPre
                    | NONE => raise Fail "no current task")
                 val premList=List.tabulate
                   (numPrems,fn n => OR[THM("A"^Int.toString n,"l"),
                                        THM("A"^Int.toString n,"r")])
               in
                 foldl (fn(OR l1,OR l2)=>OR(l1 @ l2)) (OR[]) premList

               end
           | _  =>
               if(((String.isPrefix "A" x andalso
                    Int.fromString(String.extract(x,1,NONE))<>NONE))
                  orelse x="move"
                  orelse (case lookup x (!library) of
                            SOME(x as ((genPre,EQ(gen1,gen2)),_,_)) => true
                          | _ => ((!alert)("lookoup failed for "^x);false)))
                 then THM(x,y)
               else
                 raise Fail "Invalid reduction sequence")
             ::(termLstToRedLst xs)
        end
    | [(TST_VAR x|ACT_VAR x|TST_CON x|ACT_CON x)] =>
        if(((String.isPrefix "A" x andalso
             Int.fromString(String.extract(x,1,NONE))<>NONE))
           orelse (case lookup x (!library) of
                     SOME(x as ((genPre,EQ(gen1,gen2)),_,_)) => true
                   | _ => ((!alert)("lookoup failed for "^x);false)))
          then [OR[THM(x,"l"),THM(x,"r")]]
        else raise Fail "Invalid reduction sequence"
    | ZERO::xs => (OR[])::(termLstToRedLst xs)
    | ONE::xs => (SEQ[])::(termLstToRedLst xs)
    | (NOT x)::xs => (ISEMPTY(hd(termLstToRedLst [x])))::(termLstToRedLst xs)
    | _ => raise Fail "Invalid reduction sequence"


  fun RedToString r =
    case r of
      THM(x,dir)=> x^" "^dir
    | SEQ[] => "1"
    | OR[] => "0"
    | (SEQ[x]|OR[x]) => RedToString x
    | SEQ lst => "("^(foldr
                      (fn(x,y)=>(RedToString x)^(if y="" then "" else "; ")^y)
                      "" lst)^")"
    | OR lst => "("^(foldr
                     (fn(x,y)=>(RedToString x)^(if y="" then "" else " + ")^y)
                      "" lst)^")"
    | AST x => (case (RedToString x) of
                  "" => ""
                | str => str^"*")
    | MATCH x => "MATCH["^(Term.toString x)^"]"
    | ISEMPTY x => "ISEMPTY["^(RedToString x)^"]"
    | NONEMPTY x => "NONEMPTY["^(RedToString x)^"]"

  fun SimplifyReducer r =
    case r of
     SEQ lst => ((let
        val lst = map SimplifyReducer lst
        val lst = ((foldr
          (fn(SEQ xs,rs)=> xs@rs
            |(OR[],_) => raise Fail "zero"
            |(x,rs ) => x::rs)
          [] lst))
      in
        case lst of [x] => x | _ => SEQ lst
      end)handle Fail "zero" => OR[])
    | OR lst => let
        val lst = map SimplifyReducer lst
        val lst = foldr
          (fn(OR xs,rs) =>xs@rs
            | ((x as(MATCH _ |ISEMPTY _ | NONEMPTY _)) ,rs ) =>
               if List.exists (fn x => x=SEQ[]) lst then rs else x::rs
            | (x,rs) => x::rs)
          [] lst
        val lst = removeDuplicates lst
      in
        case lst of [x] => x | _ => OR lst
      end
    | AST x =>(case SimplifyReducer x of
                 (OR[]|SEQ[]) => SEQ[]
               | (x as(SEQ _ | OR _ | THM _))=> AST x
               | AST y => AST y
               | _ => x)
    | ISEMPTY x => ISEMPTY(SimplifyReducer x)
    | NONEMPTY x => NONEMPTY(SimplifyReducer x)
    | _ => r

  fun parseReducer str = let
    val red = parseTerm str
  in
    case red of
      NONE => raise Fail "Cannot parse reduction sequence"
    | SOME x => SimplifyReducer(hd(termLstToRedLst [x]))
  end

  fun strToMove s =
    case String.map Char.toLower s of
      "l" => left
    | "r" => right
    | "u" => up
    | "d" => down
    | _   => raise Fail "Invalid direction!"

  val reducers:reducer list ref = ref [](* List of all theorems used in reduce*)

  val checkTimer=ref(fn() => ())
  exception Done
  exception Timer

  fun copyThm(id,(e,t,p))=
    let
      val p2 = ref(Proof.copy(!p))
      val t2 = ref(!t)
    in
      (id,(e,t2,p2))
    end

  fun thmDone _ =
    let val oldLib = (copyThm(hd(!library)))::(tl(!library))
      val oldTaskId = !nextTaskId
      val numTasks = case getCurrentTasks() of
        [] => raise Done
      | lst => (unfocus();length lst)
    in
      if length (doHeuristics(); getCurrentTasks()) < numTasks
        then thmDone()
      else (nextTaskId:=oldTaskId; library:=oldLib)
    end

  fun tryThm(name,dir) =
    (if name <> "move" then
      let
        val citing = not(String.isPrefix "A" name andalso
          Int.fromString(String.extract(name,1,NONE))<>NONE)
        val (SOME premNum) = if citing then SOME ~1
                           else Int.fromString(String.extract(name,1,NONE))
        val (taskId,((specPre,specCon),preProofs,focus)) =
          case !(getCurrentTask()) of
            SOME x => x
          | NONE => raise Fail "no current task"
        val focusedterm =  Term.getFocusInEqn(specCon,focus)
        val (genPre,gen1,gen2) =
          if citing then
            case lookup name (!library) of
              SOME(x as ((genPre,EQ(gen1,gen2)),_,_)) => (genPre,gen1,gen2)
            | _ => raise Fail ("Invalid Reduction Command: "^name^" "^dir)
          else
            let val intName = Int.fromString(String.extract(name,1,NONE))
              val (genPre,genCon)=
                case intName of
                  SOME j => ([],List.nth(specPre,j)
                             handle x as (Done|Timer) => raise x
                             handle _ => raise Fail "no such premise")
                | NONE => raise Fail "no such premise"
            in
              case genCon of
                EQ(gen1,gen2) => (genPre,gen1,gen2)
              |_  => raise Fail "no such premise"
            end
      in
        case Unify.unify(if dir="l" then gen1 else gen2,focusedterm) of
          NONE => false
        | SOME s => let
            val sv = substVariables s
            val tv = removeDuplicates
              (List.concat(map eqnVariables genPre)
               @(variables (if dir="l" then gen2 else gen1)))
          in
            if (diff tv sv) <> [] then false
            else(citeOrUse([name,dir],citing);
                 thmDone();
                 !checkTimer();
                 true)
          end
      end
    else ((strToMove dir) [];true))
      handle Fail x => false

  fun tryThmWithAllUnifs thm = let
    val unifOld= !Unify.resolveSubst
    val numUnifs = ref 0
    val _ = Unify.resolveSubst:=
      (fn lst => (numUnifs:=length lst;
                  SOME(hd lst)))
    val result = if tryThm thm then [(copyThm(hd(!library)),!nextTaskId)]
                 else []
    fun tryWithUnifsFrom num passed =
      if num < 0 orelse num >= (!numUnifs) then passed
      else let
        val _ = Unify.resolveSubst:=
          (fn lst => SOME(List.nth(lst,num)))
        val result = if tryThm thm then [(copyThm(hd(!library)),!nextTaskId)]
                     else []
      in
        tryWithUnifsFrom (num+1) (result@passed)
      end
  in
    tryWithUnifsFrom 1 result
  end

  (* thmsEqual x y returns true if x and y are the same theorem
   * and their proofs have the same set of tasks*)

  fun printFocus lst = case lst of []=>"[]" | x::xs =>
    "["^(Int.toString x)^(foldl(fn(x,s)=> s^","^(Int.toString x))"" xs)^"]"

  fun tasksEqual (_,(e1,_,f1)) (_,(e2,_,f2)) =
    if(e1=e2 andalso f1=f2) then true
    else
      ((*!printtext((printFocus f1)^" != "^(printFocus f2));*)
       false)

  fun thmsEqual x y =
    let
      val oldLib = (copyThm(hd (!library)))::(tl(!library))
      val tasksX = (library:=[x];getCurrentTasks())
      val tasksY = (library:=[y];getCurrentTasks())
      val () = library:= oldLib
    in
      ((#1(#2 x))=(#1(#2 y)))andalso
      (List.all(fn t=>List.exists(fn s=>tasksEqual t s)tasksY)tasksX)andalso
      (List.all(fn t=>List.exists(fn s=>tasksEqual t s)tasksX)tasksY)
    end

  fun printThm x =
    let
      val oldLib = (copyThm(hd (!library)))::(tl(!library))
      val (_,((_,e),_,f)) = hd (library:=[x];getCurrentTasks())
      val () = library:= oldLib
    in
      (eqnToString e)^" "^(printFocus f)
    end

  (* get rid of duplicate theorems in given (theorem * taskNum) list *)
  fun simplifyThmList lst =
    case lst of
      [] => []
    | (x,m)::xs =>
        (foldl (fn((thm,m),(res, mf))=>
                  if thmsEqual thm res andalso m<mf then (thm,m) else (res, mf))
         (x,m) xs)::
        (simplifyThmList(List.filter(fn(y,_)=>not(thmsEqual x y ))xs))

  (* Returns list of theorems corresponding to the possible results
   * of applying the given reducer *)
  fun tryReducer red = let
 (*   val _ = !printtext ("tryReducer: "^ (RedToString red))*)
    val oldLib = (copyThm(hd (!library)))::(tl(!library))
    val oldTaskId = !nextTaskId
    val result =
      case red of
        THM x => simplifyThmList (tryThmWithAllUnifs x)
      | AST r => let
          val numLoopsLeft =
            case !astBound of
              NONE   => ~1
            | SOME n => n
          fun tryRedUntilCycle numLoopsLeft passed = let
            val lst = List.filter
              (fn (x,_)=>List.all(fn (y,_) =>not(thmsEqual x y)) passed)
              (tryReducer r)
            val numLoopsLeft = numLoopsLeft- (if(numLoopsLeft= ~1)then 0 else 1)
            val passed=simplifyThmList(lst@passed)
            val _ = !checkTimer()
          in
            if (numLoopsLeft>0 orelse numLoopsLeft= ~1) andalso lst<>[]
              then simplifyThmList(foldr(fn((thm,n),res)=>
                          (library:=(copyThm thm)::(tl(!library));
                           nextTaskId:=n;
                           (tryRedUntilCycle numLoopsLeft passed)@res)) [] lst)
            else
              ((*!printtext("ast done due to "
                        ^(if lst=[] then "no results" else "limit"));*)
              passed)
          end
          val lst =
            tryRedUntilCycle numLoopsLeft [(copyThm(hd (!library)),!nextTaskId)]
(*          val _ = !printtext("AST returning lst of len "
                           ^(Int.toString(length lst))^" containing:)
val _ = app (fn (x,_) => !printtext("    "^(printThm x))) lst*)
        in lst
        end
      | SEQ [] => [(copyThm(hd(!library)),!nextTaskId)]
      | SEQ(red::rs) => simplifyThmList
        (foldr (fn((thm,n),res)=>(library:=(copyThm thm)::(tl(!library));
                                  nextTaskId:=n;
                                  tryReducer(SEQ rs))@res) [] (tryReducer red))
      | OR lst => simplifyThmList
        (foldr(fn(red,result)=>(tryReducer red)@result) [] lst)
      | MATCH t =>
        let
          val (_,((_,specCon),_,focus)) =
            case !(getCurrentTask()) of
              SOME x => x
            | NONE => raise Fail "no current task"
          val focusedterm = Term.getFocusInEqn(specCon,focus)
         in
          if unify(t,focusedterm)<>NONE
            then [(copyThm(hd(!library)),!nextTaskId)]
          else []
        end
      | ISEMPTY red => if null (tryReducer red)
                         then [(copyThm(hd(!library)),!nextTaskId)]
                       else []
      | NONEMPTY red => if null (tryReducer red) then []
                        else [(copyThm(hd(!library)),!nextTaskId)]
    val _ = !checkTimer()
  in
    library:= oldLib;
    nextTaskId:=oldTaskId;
  (*    !alert ("returning result of length "^(Int.toString(List.length result))
            ^" from call to tryReducer: "^ (RedToString red)); *)
    result
  end

  fun betterThmState((y,n),(x,m)) = let
    val oldLib = (copyThm(hd (!library)))::(tl(!library))
    val tasksX = (library:=[x];getCurrentTasks())
    val tasksY = (library:=[y];getCurrentTasks())
    val _ = library:= oldLib
    fun numNodesInTerm t =
      case t of
        (PLUS lst | TIMES lst) =>
          (foldl (fn(trm,num)=>num+numNodesInTerm trm) 1 lst)
      | (NOT x | STAR x)       => 1 + numNodesInTerm x
      | _                      => 1
    fun numNodesInTask (_,((_,(LE(t1,t2) | EQ(t1,t2))),_,_)) =
      numNodesInTerm t1 + numNodesInTerm t2 + 1
    fun printTaskFocus(_,(_,_,f))=printFocus f
    val numY = foldl (fn(t,n)=>n+numNodesInTask t) 0 tasksY
    val numX = foldl (fn(t,n)=>n+numNodesInTask t) 0 tasksX
(*    val _ = !printtext("numY="^(Int.toString numY)^"numX="^(Int.toString numX)^
                   "n="^(Int.toString n)^"m="^(Int.toString m))
    val _ = !printtext((printTaskFocus(hd tasksX))^(printTaskFocus(hd tasksY)))*)
  in
    if numY < numX orelse (numY = numX andalso n < m)
      then (y,n)
    else (x,m)
  end

  fun tryReducers[] = false
    | tryReducers(r::rs)=
    case tryReducer r of
      [] => tryReducers rs
    | lst=>
        let val (thm,n) = foldl betterThmState (hd lst) lst
          val _ = library:=thm::(tl(!library))
          val _ = nextTaskId:=n;
        in
          true
        end

  fun tryReducerAtFocuses red focs =
    let val lst = foldl (fn(x,y) => x@y) []
      (map (fn f => (setFocus f;
                    (* !printtext("focus set to "^(printFocus f));*)
                     tryReducer red)) focs)
    in
      if null lst then false
      else
        let val (thm,n) =foldl betterThmState (hd lst) lst
          val _ = library:=thm::(tl(!library))
          val _ = nextTaskId:=n;
        in
          true
        end
    end

  fun tryAllRedsAtFocuses focuseslist =
    case focuseslist of
      [] => false
    | (newfocus::rest) => let
        val (taskId,((specPre,specCon),preProofs,focus)) =
          case !(getCurrentTask()) of
            SOME x => x
          | NONE => raise Fail "no current task"
        val _ = setFocus newfocus
        (*val _ = !alert("focus set to "^(printFocus newfocus))*)

      in
        if tryReducers(!reducers) then true
        else tryAllRedsAtFocuses rest
      end

  (* Calls the given function on each element in the list of all (local) focuses
   * with output turned off. *)
  fun redFun doReductions = let
      (* Turn off output and save state*)
      val ptold = !printtext
      val alold = !alert
      val dnold = !done
      val _ =
        ((printtext := ((fn _ => ())));
         (alert     := (fn _ => ()));
         (done      := (fn _ => ())))
      val lastResolve = ref (!Unify.resolveSubst)
      val lastState = (copyState(),!foTable)
      val oldUndo = !undostates
      val oldRedo = !redostates
      val oldFocus = getCurrentFocus()
      val oldTaskId = !nextTaskId
      (* Get all possible focuses within the current focus*)
      val () = getAllLocalFocii()
      val focuses = !allfoc
      fun resetState success =
        (if success
           then (unfocus();
                 undostates := lastState::oldUndo;
                 redostates := [])
         else (setFocus oldFocus;
               undostates := oldUndo;
               nextTaskId := oldTaskId;
               redostates := oldRedo);
           Unify.resolveSubst := !lastResolve;
           printtext := ptold; alert := alold; done := dnold)
      val success =
        (doReductions focuses
          handle Fail x => false
               | Timer => (resetState false; raise Timer)
               | Done =>  (resetState true; raise Done))
      val _ = resetState success
    in
      success
    end
  handle Fail x => (!alert x;false)

  fun reduceUntil done = let
    fun reduceUntilCycle passed =
      let
        val oldLib = (copyThm(hd(!library)))::(tl(!library))
        val oldTaskId = !nextTaskId
      in
        if done() then ()
        else if ((redFun tryAllRedsAtFocuses)
                 handle Timer => (library:=oldLib;
                                  nextTaskId := oldTaskId;
                                  false)
                      | Done => false)
               then
                 if List.all(fn x=> not(thmsEqual x (hd(!library)))) passed
                   then reduceUntilCycle ((copyThm(hd(!library)))::passed)
                 else (library:=oldLib; nextTaskId := oldTaskId)
             else ()
      end
  in
    reduceUntilCycle [copyThm(hd(!library))]
  end

  (* Verifies that the given reduction changes
   * the state of a theorem when applied *)
  fun isValid red = case red of
    THM (x,_) => x <> "move" andalso x <> "ref="
  | (SEQ lst | OR lst) =>
      (foldr(fn(red,result)=>(isValid red) orelse result) false lst)
  | (AST x) => isValid x
  | _ => false

  (* List all theorems used in reduce*)
  fun reduce ("list"::_) =
    outputstring := (!outputstring) @ (map RedToString (!reducers))

  (* Change bound on number of executions of command* *)
  | reduce(("astbound"|"astBound"|"AstBound"|"Astbound")::xs) = astBound :=
    (case Int.fromString (String.concat xs) of
         NONE => NONE
       | SOME n => SOME(Int.abs n))
  (* Change time bound on reductions *)
  | reduce(("timer"|"timeLimit"|"Timelimit"|"TimeLimit"|"timelimit")::xs) =
     timeLimit :=
     (case Real.fromString (String.concat xs) of
        NONE => NONE
      | SOME n => SOME(Real.abs n))
    (* Try to apply premises *)
  | reduce("premises"::xs) =
      let
        val numPrems =
          (case !(getCurrentTask()) of
             SOME (_,((specPre,_),_,_)) => length specPre
           | NONE => raise Fail "no current task")
      in
        if numPrems=0 then ()
        else let
          val premList=String.concat(List.tabulate
            (numPrems, fn n => "\"A"^Int.toString n^"\" r + "^
             "\"A"^Int.toString n^"\" l + "))
          val premList=(String.extract(premList,0,SOME(size premList-3)))
        in
            reduce ["try",premList]
        end
      end
  | reduce("until"::args) = let
      val timeLimit =
        case Real.fromString (String.concat args) of
          NONE => NONE
        | SOME n => SOME(Real.abs n+Time.toReal(Time.now()))
      fun done() =
        case timeLimit of
          NONE => false
        | SOME r => Time.toReal(Time.now()) > r
    in
      reduceUntil done
    end
  | reduce("for"::args) = let
      val times =
        case Int.fromString (String.concat args) of
          NONE => ref NONE
        | SOME n => ref (SOME(Int.abs n))
      fun done() =
        case !times of
          NONE => false
        | SOME r => (times:=SOME(r-1); r<1)
    in
      reduceUntil done
    end

    | reduce ["empty"] = reducers:=[]

  (* Remove/add a reducer to the reduction list or try the given reducer *)
  | reduce ((str as ("del"|"add"|"try"))::thms) =
      (let
        val red = parseReducer (String.concat (map (fn s => s^" ") thms))
      in
         case str of
           "del" => reducers := (List.filter (fn x => x <> red) (!reducers))
         | "add" => if (member red (!reducers))
                      then (!alert) "Reduction already in list."
                    else if isValid red then reducers := red::(!reducers)
                    else !alert("Reduction "^(RedToString red)^" is trivial")
         | "try" => (redFun(tryReducerAtFocuses red);())
      end handle Fail x => !alert x)

  (* Perform all reductions *)
  | reduce _ =  (let
      val endTime =
        case !timeLimit of
          NONE => 0.0
        | SOME n => (Real.abs n)+(Time.toReal(Time.now()))
      val _ = checkTimer :=
                   (case !timeLimit of
                      NONE => (fn _ => ())
                    | SOME _ =>
                     (fn () => if ((Time.toReal(Time.now())) < endTime)
                                 then ()
                               else (!print ("timer:  time now = "
                       ^(Real.toString(Time.toReal(Time.now())))^
              "time end = "^(Real.toString(endTime))^"\n");raise Timer)))
      val result = reduceUntil (fn _ => false)
      val _ = checkTimer:= (fn _ => ())
    in
      result
    end handle Fail x => !alert x)
    handle Fail x => !alert x


(*********************************************
 * Find
 *********************************************)
  val lastGoto = ref ""
  fun goto args = let
    val argstr =  String.concat (map (fn s => s^" ") args)
    val eqn = if argstr = ""
              then parseTerm (!lastGoto)
              else (lastGoto := argstr ;
                    parseTerm argstr)
    val ceqn =
      case eqn of
        SOME e => makeConstant e
      | NONE => raise Fail "operation canceled"
    val task = getCurrentTask()
    val (taskId,((specPre,specCon),preProofs,cfocus)) =
      case !task of
        SOME x => x
      | NONE => raise Fail "no current task"
    val () = getAllFocii()
    val focuses = !allfoc
    val focuses = Util.sort compareFocii focuses
    val (truefoc,falsefoc) = List.partition (fn x => compareFocii cfocus x) focuses
    val focuses = truefoc @ falsefoc
    fun termsMatch(focus) = ceqn = getFocusInEqn(specCon,focus)
    fun testForTerm(focuslist) = case focuslist of
        [] => (task := SOME(taskId,((specPre,specCon),preProofs,cfocus));
              (!alert) "No such term.")
      | (f::fs) => if termsMatch(f) then task := SOME(taskId,((specPre,specCon),preProofs,f))
                 else testForTerm(fs)
   in
     testForTerm(focuses)
   end


(***********************************************
 * save and load
 ***********************************************)

  type xml = XML.xml

  fun checkOverwrite (filename:string) : bool = let
    val instream = TextIO.openIn filename
      handle exn => raise Continue
  in
    TextIO.closeIn instream;
    false
  end
  handle Continue => true

  fun save args = let
    val filename = case args of
      [] => raise Fail "no file name specified"
    | [x] => if (isSubString ".xml" x) then x else "../Lib/" ^ x ^ ".xml"
    | _ => raise Fail "too many arguments"
    (* Continuation for after determining if file already exists (save to file)*)
    fun cont () = let
      val outstream = TextIO.openOut filename
      fun put (s:string) = TextIO.outputSubstr(outstream,Substring.all s)
      fun putln (s:string) = put(s^"\n")
      val thm = getCurrentTheorem()
    in
      putln "<kat>";
      putln(theoremToXML thm);
      putln "</kat>";
      TextIO.closeOut outstream
    end
  handle Fail x => (!alert) x
  in
    if checkOverwrite filename then cont()
    else ((!ask2) ("file " ^ filename ^ " exists; overwrite? [no,yes] ",["no","yes"])
                  (fn res => case res of SOME 1 => cont() | _ => ()))
  end
  handle Fail x => (!alert) x

  fun saveall args = let
    val filename = case args of
      [] => raise Fail "no file name specified"
    | [x] => if (isSubString ".xml" x) then x else "../Lib/" ^ x ^ ".xml"
    | _ => raise Fail "too many arguments"
    (* Continuation for after determining if file already exists (save to file)*)
    fun cont () = let
      val outstream = TextIO.openOut filename
      fun put (s:string) = TextIO.outputSubstr(outstream,Substring.all s)
      fun putln (s:string) = put(s^"\n")
      val thm = getCurrentTheorem()
    fun notAxiom ((_,(_,_,ref proof)):theorem) =
      case proof of AXIOM _ => false | _ => true
    in
      putln "<kat>";
      map (putln o theoremToXML) (List.filter notAxiom (!library));
      putln "</kat>";
      TextIO.closeOut outstream
    end
  handle Fail x => (!alert) x
  in
    if checkOverwrite filename then cont()
    else ((!ask2) ("file " ^ filename ^ " exists; overwrite? [no,yes] ",["no","yes"])
                  (fn res => case res of SOME 1 => cont() | _ => ()))
  end
  handle Fail x => (!alert) x

  fun load args = let
    val filename = case args of
      [] => let
        val dirList = OS.FileSys.openDir "../Lib"
          handle exn => raise Fail "cannot find library directory"
        fun printFiles() =
          case OS.FileSys.readDir dirList of
            "" => raise Continue
          | next => (println next; printFiles())
        in
          printFiles()
        end
    | [x] => if (isSubString ".xml" x) then x else "../Lib/" ^ x ^ ".xml"
    | _ => raise Fail "too many arguments"
    val instream = TextIO.openIn filename
      handle exn => raise Fail ("file " ^ filename ^ " not found")
    fun get() = Substring.all(TextIO.inputLine instream)
    fun getLines (s:string list) : string list =
      let val next = Substring.string (get())
      in if next <> "" then getLines(next::s) else rev s
      end

    fun getTheorem (xml:xml)  = let
      val (id,condEqn,currentTask,proof,table) =
        case XML.getContent("lib",xml) of
          [i,c,t,p] => (i,c,t,p,ELEMENT("table",[]))
        | [i,c,t,p,ta] => (i,c,t,p,ta)
        | _ => raise Fail "could not parse theorem"
      val id = XML.getString("id",id)
      val currentTask = XML.getString("task",currentTask)
      val condEqn = XML.getString("condeqn",condEqn)
      val thm =
        case parseCondEqn condEqn of
          SOME x => x
        | NONE => raise Fail ("could not parse theorem " ^ condEqn)
      val proof = Proof.proofFromXML proof
      val (blist,clist) = case table of
                            ELEMENT("table",[]) => ([],[])
                          | _ => FirstOrder.tableFromXML(table)
      val task =
        case currentTask of
          "" => if null (Proof.getTasks proof) then NONE
                else raise Fail "no current task"
        | _ => SOME (valOf (find currentTask (Proof.getTasks proof)))
                 handle Option => raise Fail ("no such task " ^ currentTask)
    in
      ((id,(thm,ref task,ref proof)),(blist,clist))
    end

    val replacing : bool ref = ref false

    fun addtheorem (newtheoremlist) (theorems) : unit =
         case newtheoremlist of
           [] => (let val newLib = map valOf (List.filter isSome theorems)
                  in
                    saveState();
                    library := (if length newLib <= 1
                                then newLib @ !library else !library @ newLib);
                    foTable := Util.moveToFront (#1(getCurrentTheorem())) (!foTable);
                    FirstOrder.boolList := (#1 (#2 (hd(!foTable))));
                    FirstOrder.termList := (#2 (#2 (hd(!foTable))));
		    FirstOrder.boolIndex := List.length(!FirstOrder.boolList);
		    FirstOrder.termIndex := List.length(!FirstOrder.termList)
                  end)
         | ((newtheorem as (id,(newThm,_,_)),tables))::rest => (let
             val thm = find id (!library)
           in
             case thm of
             NONE => ((foTable := (id,tables)::(!foTable));
                       addtheorem (rest) ((SOME newtheorem)::theorems))
           | SOME (_,(_,_,ref proof))  =>
               (case proof of
                  AXIOM _ => ((!alert) ("cannot overwrite axiom " ^ id);
                              addtheorem (rest) theorems)
                | _ => (if !replacing
                        then addtheorem (rest) ((SOME newtheorem)::theorems)
                        else ((!ask2)
                              ("theorem " ^ id ^
                               " exists; overwrite? [yes,no,all,stop] ",
                               ["yes","no","all","stop"])
                             (* Continuation, what to do based on answer *)
                             (fn response =>
                                case response of
                                  SOME 0 => (forget [id];
                                             foTable := (id,tables)::(!foTable);
                                             addtheorem (rest) (
                                             (SOME newtheorem)::theorems))
                                | SOME 1 => addtheorem (rest) (theorems)
                                | SOME 2 => (replacing := true;
                                             forget [id];
                                             foTable := (id,tables)::(!foTable);
                                             addtheorem (rest) (
                                             (SOME newtheorem)::theorems))
                                | _ => raise Fail "operation halted")))
           ) end)


    in
    let
       val input = concat(getLines [])
       val inputString = concat(String.tokens (fn c =>
                             Char.isSpace c andalso c <> #" ") input)
       val xml = XML.parse inputString
    in
       addtheorem (map (getTheorem) (XML.getContent("kat",xml))) []
    end
end
handle Fail x => (!alert)  x | Continue => () (*| exn => (!alert) "corrupt file"*)

fun outputLaTeX args = let
    val filename = case args of
      [] => raise Fail "no file name specified"
    | [x] => if (isSubString ".tex" x) then x else "../LaTeX/" ^ x ^ ".tex"
    | _ => raise Fail "too many arguments"
    val outstream = TextIO.openOut filename
      handle exn => raise Fail ("file " ^ filename ^ " not found")
    fun put (s:string) = TextIO.outputSubstr(outstream,Substring.all s)
    fun putln (s:string) = put(s^"\n")
    val () = putln "\\input{header}\n\\begin{document}"
    val proof = proofToLaTeX()
    val tokenized = String.tokens (fn x => x = #"\n") proof
   in
     (put proof ; put "\n\\end{document}" ; TextIO.closeOut outstream)
   end

fun dumpToFile args = let
    val filename = case args of
      [] => raise Fail "no file name specified"
    | [x] => if (isSubString ".kat" x) then x else "../" ^ x ^ ".kat"
    | _ => raise Fail "too many arguments"
    val outstream = TextIO.openOut filename
      handle exn => raise Fail ("file " ^ filename ^ " not found")
    fun put (s:string) = TextIO.outputSubstr(outstream,Substring.all s)
    fun putln (s:string) = put(s^"\n")
    val proof = dump()
   in
     (put proof ; TextIO.closeOut outstream)
   end


  val commands =
    [("left",left),
     ("right",right),
     ("up",up),
     ("down",down),
     ("library",viewLibrary),
     ("folibrary",viewfoLibrary),
     ("theorem",viewCurrentTheorem),
     ("tasks",viewTasks),
     ("get",get),
     ("circ",mCircCheck),
     ("define",define),
     ("publish",publish),
     ("parse",publishFromSlang),
     ("table",viewFOTable),
     ("forget",forget),
     ("ignore",ignore),
     ("focus",focus),
     ("unfocus",unfocus),
     ("cite",cite),
     ("use",use),
     ("quit",quit),
     ("premises",premises),
     ("cut",cut 0),
     ("normalize",normalize),
     ("move",moveTerm),
     ("undo",undo),
     ("redo",redo),
     ("save",save),
     ("saveall",saveall),
     ("export",outputLaTeX),
     ("dump",dumpToFile),
     ("load",load),
     ("rename",rename),
     ("verify",verify),
     ("verifyall",deepverify),
     ("reset",reset),
     ("proof",viewCurrentProof),
     ("heuristics",heuristic),
     ("reduce",reduce),
     ("find",goto),
     ("debug",debug)]

  fun help() =
   (println "Available commands are:"; app println (map #1 commands))

  (* process a command *)
  fun doCommand input =
    case input of
      cmd::args =>
        (case findPrefix cmd commands of
           SOME (_,action) => action args
         | NONE => help())
    | [] => ()

  fun readInitFile _ : unit =
    let
      val instream = TextIO.openIn "../kat.ini"
        handle exn => (println "zz"; raise Fail "ww")
      fun get() = Substring.all(TextIO.inputLine instream)
      fun getLines (s:string list list) : string list list =
        let val next = Substring.string (get())
            val input = String.tokens Char.isSpace next
        in if next = "" then rev s else getLines(input::s)
        end
      val initLines = getLines []
    in
      app doCommand initLines
    end
    handle Fail x => println x | exn => () (* no init file *)

end
