(* Module for handling slang -> KAT conversion *)
structure FirstOrder = struct

open Util
open Term
open SLang


(* First-Order Library *)
val library:(string * string * ((loc * C) list -> bool) list
                      * (C * C)) list ref = ref nil

(* Unique ids for new terms and booleans *)
val termIDs = ["a","b","c","d","e","f","g","h","i","j","k","l","m","n",
               "o","p","q","r","s","t","u","v","w","x","y","z","aa","bb","cc"]
val boolIDs = ["A","B","C","D","E","F","G","H","I","J","K","L","M","N",
               "O","P","Q","R","S","T","U","V","W","X","Y","Z"]
val termIndex = ref ~1
val boolIndex = ref ~1

(* Get the next identifier for a term and boolean *)
fun newTerm () = let val () = termIndex := !termIndex + 1
                 in List.nth(termIDs,!termIndex)
                 end

fun newBool () = let val () = boolIndex := !boolIndex + 1
                 in List.nth(boolIDs,!boolIndex)
                 end

(* Lists of terms and booleans already assigned a KAT term *)
val termList:(C * term) list ref = ref nil
val boolList:(B * term) list ref = ref nil

(* Reset terms and booleans lists *)
fun reset() = let val () = termIndex := ~1
                  val () = boolIndex := ~1
                  val () = termList := nil
                  val () = boolList := nil
              in
                  ()                             
              end

(* Find KAT term corresponding to boolean *)
fun findBool(bool) = case 
                      (List.find (fn (term,kat) => term = bool) (!boolList)) of
                   SOME(term,kat) => SOME(kat)
                 | NONE => NONE

(* Find KAT term corresponding to command *)
fun findCom(com) = case (List.find (fn (term,kat) => term = com) (!termList)) of
                   SOME(term,kat) => SOME(kat)
                 | NONE => NONE

(* Find command corresponding to KAT term *)
fun findKATC(katt) = case (List.find (fn (term,kat) => katt = kat) (!termList)) of
                   SOME(term,kat) => SOME(term)
                 | NONE => NONE

(* Find boolean corresponding to KAT term *)
fun findKATB(katt) = case (List.find (fn (term,kat) => katt = kat) (!boolList)) of
                   SOME(term,kat) => SOME(term)
                 | NONE => NONE


(* Test for a location in arithmetic expressions *)
  fun varInA(loc,a) =
   case a of
     N(_) => false
   | Loc(x) => loc = x
   | (Plus(a1,a2) | Minus(a1,a2) 
    | Times(a1,a2) | Div(a1,a2)
    | Mod(a1,a2)) => varInA(loc,a1) orelse varInA(loc,a2)
   | RepA(a1,a2,a3) => (varInA(loc,a1) orelse varInA(loc,a2) orelse varInA(loc,a3))
   | AFCall(s,args) => List.exists (fn x => (varInA(loc,x))) args
(* Test for a location in boolean expressions *)
  fun varInB(loc,b) = 
   case b of
     (True | False) => false
   | (Equals(a1,a2) | LessEq(a1,a2) 
    | GreEq(a1,a2) | Less(a1,a2)
    | Gre(a1,a2)) => varInA(loc,a1) orelse varInA(loc,a2)
   | (Not(b1)) => varInB(loc,b1)
   | (And(b1,b2) | Or(b1,b2)) => varInB(loc,b1) orelse varInB(loc,b2)
(* Test for a location in commands *)
  fun varInC(loc,c) =
   case c of
     Assign(x,a) => x = loc orelse varInA(loc,a)
   | Seq(c1) => List.exists (fn x => varInC(loc,x)) c1
   | Cond(b,c1,c2) => varInB(loc,b) orelse varInC(loc,c1) orelse varInC(loc,c2)
   | While(b,c) => varInB(loc,b) orelse varInC(loc,c)
   | Boolean(b) => varInB(loc,b)
   | Math(a) => varInA(loc,a)
   | Phi => false
   | Pre => false
   | One => false
   | FCall(s,args) => List.exists (fn x => (varInA(loc,x))) args
   | Rep(a1,a2,a3) => (varInC(loc,a1) orelse varInA(loc,a2) orelse varInA(loc,a3))
   | _ => raise Fail "Error with varInC!"


(* Substitute the expression a for x in term *)

  fun subForXInA(x,a as Math(a'),term) = if term = x then a' else
  (case term of
    N(n) => N(n)
  | Loc(Arr(s,[ind as RepA(y,xa,sa)])) => (case (x,a') of 
       (Loc(Arr(sg,[yg])),Loc(Arr(s',[y']))) => if y = yg andalso sg = s
                            then Loc(Arr(s',[RepA(y',xa,sa)]))
                            else Loc(Arr(s,[RepA(subForXInA(x,a,y),subForXInA(x,a,xa),subForXInA(x,a,sa))]))
     | _ => Loc(Arr(s,[subForXInA(x,a,ind)])))
  | Loc(Arr(s,a1)) =>  Loc(Arr(s,map (fn(ax) => subForXInA(x,a,ax)) a1))
  | Loc(ASub(y,b,c)) => let val Loc(y') = subForXInA(x,a,Loc(y))
                           val Loc(b') = subForXInA(x,a,Loc(b))
                       in
                         Loc(ASub(y',b',subForXInA(x,a,c)))
                       end
  | Loc(_) => term
  | Plus(a1,a2) => Plus(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | Minus(a1,a2) => Minus(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | Times(a1,a2) => Times(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | Div(a1,a2) => Div(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | Mod(a1,a2) => Mod(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | RepA(a1,a2,a3) => RepA(subForXInA(x,a,a1),
                           subForXInA(x,a,a2),
                           subForXInA(x,a,a3))
  | AFCall(s,args) => AFCall(s,List.map (fn y => subForXInA(x,a,y)) args))
  | subForXInA(x,a,term) = term

  fun subForXInB(x,a as (Math(_) | Boolean(_)),term) = case term of
    (True | False) => term
  | Equals(a1,a2) => Equals(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | LessEq(a1,a2) => LessEq(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | GreEq(a1,a2) => GreEq(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | Less(a1,a2) => Less(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | Gre(a1,a2) => Gre(subForXInA(x,a,a1),subForXInA(x,a,a2))
  | Not(b) => Not(subForXInB(x,a,b))
  | And(a1,a2) => And(subForXInB(x,a,a1),subForXInB(x,a,a2))
  | Or(a1,a2) => Or(subForXInB(x,a,a1),subForXInB(x,a,a2))

  fun subForXInC(x,a,term) = case term of
    Assign(loc,a1) => let val Loc(newLoc) = subForXInA(x,a,Loc(loc))
                      in
                          Assign(newLoc,subForXInA(x,a,a1))
                      end
  | Math(aexp) => Math(subForXInA(x,a,aexp))
  | Boolean(bexp) => Boolean(subForXInB(x,a,bexp))
  | Cond(b,c1,c2) => Cond(subForXInB(x,a,b),subForXInC(x,a,c1),
                          subForXInC(x,a,c2))
  | While(b,c1) => While(subForXInB(x,a,b),subForXInC(x,a,c1))
  | Seq(c1) => Seq(map (fn c => subForXInC(x,a,c)) c1)
  | Phi => if x = Loc(LPhi) then a else Phi
  | Pre => if x = Loc(LPre) then a else Pre
  | One => One
  | FCall(s,args) => FCall(s,List.map (fn y => subForXInA(x,a,y)) args)
  | Rep(a1,a2,a3) => let val a1' = if x <> a2 then subForXInC(x,a,a1)
                                   else a1
                         val a2' = subForXInA(x,a,a2)
                         val a3' = subForXInA(x,a,a3)
                     in Rep(a1',a2',a3')
                     end
  | _ => raise Fail "Error with subForXInC!"

  fun uniquify((lhs,rhs),vars) = let
    val myrand = Random.rand(1,Real.round(Math.sqrt(Time.toReal(Time.now()))))
    fun replace(x,(myl,myr)) = let
      val nextrand = Random.randNat(myrand)
    in
      (subForXInC(Loc(x),Math(Loc(Var("x"^Int.toString(nextrand)))),myl),
       subForXInC(Loc(x),Math(Loc(Var("x"^Int.toString(nextrand)))),myr))
    end
  in
    foldl replace (lhs,rhs) vars
  end
          


(* Extact single-variable terms *)
fun singleTerms() = List.filter (fn x => 
                      case x of
                        (_,ACT_CON(_)) => true
                      | _ => false) (!termList)
fun singleBools() = List.filter (fn x => 
                      case x of
                        (_,TST_CON(_)) => true 
                      | _ => false) (!boolList)


(* Remove assumptions from a list that are copies
 * All assumptions are of the form p = q, so no
 * need to worry about <= *)
fun removeCopy(EQ(term1,term2),list) =
  if List.exists (fn x => x = EQ(term2,term1) orelse x = EQ(term1,term2)) list 
  then list
  else EQ(term1,term2)::list





(* Convert KAT terms to commands *)
  fun KATToSLang(katterm) = let
    fun translate(katterm) = (case katterm of
      TST_CON(x) => Boolean(valOf(findKATB(katterm)))
    | ACT_CON(x) => valOf(findKATC(katterm))
    | NOT(x) => let val Boolean(btrans) = translate(x)
                in Boolean(Not(btrans))
                end
    | PLUS[TIMES[A,a],TIMES[NOT B,b]] =>
        if A = B then let
           val Boolean(booltrans) = translate(A)
           handle exn => raise Fail "Not a boolean!"
        in
           Cond(booltrans,translate(a),translate(b))
        end
        else raise Fail "Translation not possible"
    | TIMES [(STAR(TIMES[A,a])),NOT B] =>
        if A = B then let
           val Boolean(booltrans) = translate(A)
           handle exn => raise Fail "Not a boolean!"
        in
           While(booltrans,translate(a))
        end
        else raise Fail "Translation not possible"
    | TIMES ((STAR(TIMES[A,a]))::NOT B::xs) =>
        if A = B then let
           val Boolean(booltrans) = translate(A)
           handle exn => raise Fail "Not a boolean!"
        in
           Seq([While(booltrans,translate(a)),valOf(KATToSLang(TIMES(xs)))])
        end
        else raise Fail "Translation not possible"
    | TIMES [x] => translate(x)
    | TIMES (x::xs) => Seq([translate(x),translate(TIMES(xs))])
    | ONE => Boolean(True)
    | ZERO => Boolean(False)
    | _ => raise Fail "Translation not possible")
   in SOME(translate(katterm))
   end
   handle  Option => ((println (Term.termStructure(katterm))); NONE)
   handle Fail x => NONE | Option => ((println (Term.toString(katterm))); NONE)
  

  fun subAll term list = case list of
    [] => term
  | ((subterm,rep)::rest) => subAll (subForXInC(Loc(subterm),rep,term)) rest



  fun hold (c as Assign(x as Var(xs),s),kat1,
                         d as Assign(y as Var(ys),t),kat2) = NONE


  (* Check to see if a boolean check is unnecessary after an assignment,
   * i.e., the assignment assigns the variable to the value against which
   * it is being tested
   *)      
  fun booleanAssumest (c as Assign(x as Var(_),N(a)),ct,Boolean(b),bt) =
   (case b of
     (Equals(Loc(x1),N(a1)) | Equals(N(a1),Loc(x1))) => 
       if x = x1 andalso a = a1
       then SOME(EQ(TIMES [ct,bt],ct))
       else NONE
   | (LessEq(Loc(x1),N(a1)) | GreEq(N(a1),Loc(x1))) => 
       if x = x1 andalso a <= a1
       then SOME(EQ(TIMES [ct,bt],ct))
       else NONE
   | (GreEq(Loc(x1),N(a1)) | LessEq(N(a1),Loc(x1))) =>
       if x = x1 andalso a >= a1
       then SOME(EQ(TIMES [ct,bt],ct))
       else NONE
   | (Less(Loc(x1),N(a1)) | Gre(N(a1),Loc(x1))) => 
       if x = x1 andalso a < a1
       then SOME(EQ(TIMES [ct,bt],ct))
       else NONE
   | (Gre(Loc(x1),N(a1)) | Less(N(a1),Loc(x1))) => 
       if x = x1 andalso a > a1
       then SOME(EQ(TIMES [ct,bt],ct))
       else NONE
   | Not(b1) => (case booleanAssumest(c,ct,Boolean(b1),bt) of
                   SOME _ => NONE
                 | NONE   => SOME(EQ(TIMES [ct,bt],ct)))
   | Or(b1,b2) => (case (booleanAssumest(c,ct,Boolean(b1),bt),
                         booleanAssumest(c,ct,Boolean(b2),bt)) of
                   (NONE,NONE) => NONE
                 | _   => SOME(EQ(TIMES [ct,bt],ct)))
   | _ => NONE)
 | booleanAssumest _ = NONE

  (* Check to see if a boolean check is unnecessary after an assignment,
   * i.e., the assignment assigns the variable to a value that makes the
   * test false
   *)      
  fun booleanAssumesf (c as Assign(x as Var(_),N(a)),ct,Boolean(b),bt) =
   (case b of
     (Equals(Loc(x1),N(a1)) | Equals(N(a1),Loc(x1))) =>
       if x = x1 andalso a <> a1
       then SOME(EQ(TIMES [ct,NOT(bt)],ct))
       else NONE
   | (LessEq(Loc(x1),N(a1)) | GreEq(N(a1),Loc(x1))) => 
       if x = x1 andalso a > a1
       then SOME(EQ(TIMES [ct,NOT(bt)],ct))
       else NONE
   | (GreEq(Loc(x1),N(a1)) | LessEq(N(a1),Loc(x1))) => 
       if x = x1 andalso a < a1
       then SOME(EQ(TIMES [ct,NOT(bt)],ct))
       else NONE
   | (Less(Loc(x1),N(a1)) | Gre(N(a1),Loc(x1))) =>
       if x = x1 andalso a >= a1
       then SOME(EQ(TIMES [ct,NOT(bt)],ct))
       else NONE
   | (Gre(Loc(x1),N(a1)) | Less(N(a1),Loc(x1))) => 
       if x = x1 andalso a <= a1
       then SOME(EQ(TIMES [ct,NOT(bt)],ct))
       else NONE
   | Not(b1) => (case booleanAssumesf(c,ct,Boolean(b1),bt) of
                   SOME _ => SOME(EQ(TIMES [ct,bt],ct))
                 | NONE   => NONE)
   | Or(b1,b2) => (case (booleanAssumesf(c,ct,Boolean(b1),bt),
                         booleanAssumesf(c,ct,Boolean(b2),bt)) of
                   (SOME _,SOME _) => SOME(EQ(TIMES [ct,NOT(bt)],ct))
                 | _   => NONE)
   | _ => NONE)
 | booleanAssumesf _ = NONE

  (* Get commuting with boolean conditions *)
  fun booleanCommute(c as Assign(x,_),ct,Boolean(b),bt) = 
    if(not(varInB(x,b)))
    then SOME([EQ(TIMES [ct,bt],TIMES [bt,ct]),
               EQ(TIMES [ct,NOT(bt)],TIMES [NOT(bt),ct])])
    else NONE
  | booleanCommute _ = NONE

fun makeListString args = case args of
      nil => ""
    |  [x] => aTermToString x
    | (x::xs) => (aTermToString x)^","^makeListString(xs)

(* Convert commands to strings *)
and aTermToString(aterm) =
  case aterm of
    N(n) => Int.toString(n)
  | Loc(Var(loc)) => loc
  | Loc(LPhi) => "phi"
  | Loc(LPre) => "pre"
  | Loc(Arr(loc,ind)) => loc^(String.concat(map (fn(ind) => "["^aTermToString(ind)^"]") ind))
  | Plus(a1,a2) => "("^aTermToString(a1)^" + "^aTermToString(a2)^")"
  | Minus(a1,a2) => "("^aTermToString(a1)^" - "^aTermToString(a2)^")"
  | Times(a1,a2) => "("^aTermToString(a1)^" * "^aTermToString(a2)^")"
  | Div(a1,a2) => "("^aTermToString(a1)^" / "^aTermToString(a2)^")"
  | Mod(a1,a2) => "("^aTermToString(a1)^" % "^aTermToString(a2)^")"
  | AFCall(s,args) => s^"("^(makeListString(args))^")"
  | RepA(a1,a2,a3) => "& "^aTermToString(a1)^" & "^aTermToString(a2)^"//"
                    ^aTermToString(a3)^" & "
  | _ => ""

fun bTermToString(bterm) = case bterm of
  True => "true"
| False => "false"
| Equals(a1,a2) => aTermToString(a1)^" = "^ aTermToString(a2)
| LessEq(a1,a2) => aTermToString(a1)^ " <= "^ aTermToString(a2)
| GreEq(a1,a2) => aTermToString(a1)^ " >= "^ aTermToString(a2)
| Less(a1,a2) => aTermToString(a1)^ " < "^ aTermToString(a2)
| Gre(a1,a2) => aTermToString(a1)^ " > "^ aTermToString(a2)
| Not(b) => "!"^ bTermToString(b)
| And(b1,b2) => "("^bTermToString(b1)^ ") && ("^ bTermToString(b2)^")"
| Or(b1,b2) => "("^bTermToString(b1)^ ") || ("^ bTermToString(b2)^")"

fun cTermToString(cterm) = case cterm of
  Assign(Var(loc),a) => loc^" := "^aTermToString(a)
| Assign(Arr(loc,ind),a) => loc^(String.concat(map (fn(ind) => "["^aTermToString(ind)^"]") ind))^" := "^aTermToString(a)
| Seq(c1) => foldr (fn (x,y) => cTermToString(x)^";\n"^y) "" c1
| Cond(b,c1,c2) => "if ("^bTermToString(b)^") then\n{\n"^cTermToString(c1)
                   ^"\n}\nelse\n{\n"^cTermToString(c2)^"\n}"
| While(b,c) => "while ("^bTermToString(b)^")\n{\n"^cTermToString(c)^"\n}" 
| Boolean(b) => "$"^bTermToString(b)
| Math(a) => "#"^aTermToString(a)
| Phi => "phi"
| Pre => "pre"
| One => "1"
| FCall(s,args) => s^"("^(makeListString(args))^")"
| Rep(a1,a2,a3) => "& "^cTermToString((a1))^" & "^aTermToString((a2))^"//"
                    ^aTermToString((a3))^" & "
| FunDef(name,vars,body) => name^"("^(makeListString(vars))^")"^"{"^(cTermToString(body))^"}"
| _ => raise Fail "Error with cTermToString!"


(* Convert booleans to KAT terms *)
fun slangBToKAT (bool:B) =
    case (List.find (fn (term,kat) => term = bool) (!boolList)) of
      SOME(term,kat) => kat
    | NONE => (case bool of
                 True => ONE
               | False => ZERO
               | Or(b1,b2) => let 
                                val b1KAT = slangBToKAT(b1)
                                val b2KAT = slangBToKAT(b2)
                                val nextID = flatten(PLUS [b1KAT,b2KAT])
                                val () = boolList := (bool,nextID)::(!boolList)
                              in
                                nextID
                              end
               | And(b1,b2) => let 
                                val b1KAT = slangBToKAT(b1)
                                val b2KAT = slangBToKAT(b2)
                                val nextID = flatten(TIMES [b1KAT,b2KAT])
                                val () = boolList := (bool,nextID)::(!boolList)
                              in
                                nextID
                              end
               | Not(b) => NOT(slangBToKAT(b))
               | _ => let
                         val nextID = TST_CON(newBool())
                         val () = boolList := (bool,nextID)::(!boolList)
                       in
                         nextID
                       end)
       

(* Convert SLang Commands to KAT terms *)
fun slangCToKAT (com:C) =
  (* First see if the term has already been converted *)
  case (List.find (fn (term,kat) => term = com) (!termList)) of
    SOME (term,kat) => kat
  | NONE =>  (case com of
               Assign(loc,exp) => (let
                 val nextID = ACT_CON(newTerm())
                 val () = termList := (com,nextID)::(!termList)
               in
                 nextID
               end)
             | FCall(f1) => (let
                 val nextID = ACT_CON(newTerm())
                 val () = termList := (com,nextID)::(!termList)
               in
                 nextID
               end)
             | Boolean(b) => slangBToKAT(b)
             | Seq(c1) => (let
                 val nextID = flatten(TIMES (map slangCToKAT c1))
                 val () = termList := (com,nextID)::(!termList)
               in
                 nextID
               end)
             | Cond(b,c1,c2) => (let
                 val nextIDb = slangBToKAT(b)
                 val nextIDc1 = slangCToKAT(c1)
                 val nextIDc2 = slangCToKAT(c2)
                 val nextID = PLUS [TIMES [nextIDb,nextIDc1],
                                    TIMES [NOT(nextIDb),nextIDc2]]
                 val () = termList := (com,nextID)::(!termList)
               in
                 nextID
               end)
             | While(b,c) => (let
                 val nextIDb = slangBToKAT(b)
                 val nextIDc = slangCToKAT(c)
                 val nextID = TIMES [STAR(TIMES [nextIDb,nextIDc]),NOT(nextIDb)]
                 val () = termList := (com,nextID)::(!termList)
               in
                 nextID
               end)
             | One => ONE
             | Rep(c,a1,a2) => slangCToKAT(subForXInC(a1,Math(a2),c))
             | FunDef(name,vars,body) => (case vars of
                 nil => let
                   val asstring = name^"() = "
                                      ^(cTermToString(body))
                   val axiom = (FCall(name,[]),
                                body)
                   val () = library := ("FA1",asstring,[],axiom)::(!library)
                  in
                   ONE
                  end
               | [Loc(Var(x))] => let 
                   val asstring = name^"("^(makeListString(vars))^") = "
                                      ^(cTermToString(body))
                   val axiom = (FCall(name,[Loc(Var(x))]),
                                Rep(body,Loc(GVar(x)),Loc(GVar("x"))))
                   val () = library := ("FA1",asstring,[],axiom)::(!library)
                  in
                   ONE
                  end
               | morevars => let
                   fun makeVarList(vars) = case vars of
                      [] => []
                   | (Loc(x)::xs) => x::(makeVarList(xs))
                   fun makeRight(vars) = case vars of
                       [Loc(Var(x))] => Rep(body,Loc(GVar(x)),Loc(GVar(x)))
                     | (Loc(Var(x))::xs) => Rep(makeRight(xs),Loc(GVar(x)),Loc(GVar(x)))
                   val asstring = name^"("^(makeListString(vars))^") = "
                                      ^(cTermToString(body))
                   val axiom = (FCall(name,morevars),makeRight(morevars))
                   val () = library := ("FA1",asstring,[],axiom)::(!library)
                  in
                     ONE
                  end)
             | _ => raise Fail "Should not be translating this!"
)


fun varsInA(aterm) = case aterm of
  N(_) => []
| Loc(Var(x)) => [Var(x)]
| Loc(Arr(A,x)) => List.concat(map varsInA x)
| (Plus(a1,a2) | Minus(a1,a2) |
   Times(a1,a2) | Div(a1,a2) |
   Mod(a1,a2)) => varsInA(a1) @ varsInA(a2)
| AFCall(s,args) => List.concat(map varsInA args)
| RepA(a1,a2,a3) => varsInA(a1) @ varsInA(a2) @ varsInA(a3)
| _ => raise Fail "Error with varsInA!"

fun varsInB(bterm) = case bterm of
  (True | False) => []
| (Equals(a1,a2) | LessEq(a1,a2) |
   GreEq(a1,a2) | Less(a1,a2) |
   Gre(a1,a2)) => varsInA(a1) @ varsInA(a2)
| Not(b) => varsInB(b)
| (And(b1,b2) | Or(b1,b2)) => varsInB(b1) @ varsInB(b2)

fun varsInC(cterm) = case cterm of
  Assign(x,a) => varsInA(Loc(x)) @ (varsInA(a))
| Seq(c1) => List.concat (map varsInC c1)
| Cond(b,c1,c2) => varsInB(b) @ varsInC(c1) @ varsInC(c2)
| While(b,c) => varsInB(b) @ varsInC(c)
| FCall(s,args) => List.concat(map varsInA args)
| (Phi | One | Pre)  => []
| Boolean(b) => varsInB(b)
| Math(a) => varsInA(a)
| Rep(c,a1,a2) => varsInC(c) @ varsInA(a1) @ varsInA(a2)
| FunDef(name,args,body) => (List.concat(map varsInA args))@(varsInC(body))




(* Functions for testing conditions for matching a first-order axiom *)

(* x and y in assignments are different *)
fun unique(subs) =
    (lookup (Var("x")) subs) <> (lookup (Var("y")) subs)


(* y is not free in s *)
fun yNotFree(subs:(SLang.loc * SLang.C) list) = let
   val yv = lookup (Var("y")) subs
   val sv = lookup (Var("s")) subs
   val (y,s) = case (yv,sv) of
     (SOME(yval),SOME(sval)) => (yval,sval)
   | _ => raise Fail "Substitution not found"
   val y = case y of 
      Math(Loc(yloc)) => yloc
    | _ => raise Fail "y not found"
 in
    not(varInC(y,s))
 end

(* x is not free in s *)
fun xNotFree(subs:(SLang.loc * SLang.C) list) = let
   val xv = lookup (Var("x")) subs
   val sv = lookup (Var("s")) subs
   val (x,s) = case (xv,sv) of
     (SOME(xval),SOME(sval)) => (xval,sval)
   | _ => raise Fail "Substitution not found"
   val x = case x of 
      Math(Loc(xloc)) => xloc
    | _ => raise Fail "x not found"
 in
    not(varInC(x,s))
 end

(*fun xNotFreeS1(Seq(c as Assign(x as Var(xs),s),
                         d as Assign(y as Var(ys),t)),RIGHT) = not(varInA(y,s))

| xNotFreeS1 _ = true*)

(* Simplify a term *)
(*fun simple(Seq(c as Assign(x as Arr(name1,xs),s),
                         d as Assign(y as Arr(name2,ys),t)),_) =
let fun isSimple(a) =
  case a of
    Loc(Arr(_,_)) => false
  | (Plus(a1,a2) | Minus(a1,a2) | 
     Times(a1,a2) | Div(a1,a2) | 
     Mod(a1,a2)) => isSimple(a1) andalso isSimple(a2)
  | _ => true
in
  List.all isSimple xs andalso List.all isSimple ys
end*)

(*fun noAssign(Seq(c as Assign(x as Var(xs),s),
                 d as Assign(y as Arr(name2,ys),t)),_) = 
  (List.all (fn var => (x <> var)) (List.concat (map varsInA ys)))
| noAssign(Seq(d as Assign(y as Arr(name2,ys),t),
               c as Assign(x as Var(xs),s)),_) = 
  (List.all (fn var => (x <> var)) (List.concat (map varsInA ys)))

| noAssign(Seq(d as Assign(y as Arr(name2,ys),t),
               c as Assign(x as Arr(name3,xs),s)),_) = true

| noAssign(d as Assign(y as Arr(name2,ys),t),_) = true
| noAssign(Seq(Boolean(_),d as Assign(y as Arr(name2,ys),t)),_) = true
| noAssign(_,_) = true*)

(* Array functions *)
fun arr(a) = case a of
  N(_) => []
| Loc(Arr(A,ind)) => (Arr(A,ind))::(List.concat (map arr ind))
| Loc(_) => []
| (Plus(a1,a2) | Minus(a1,a2) | Times(a1,a2)
  | Div(a1,a2) | Mod(a1,a2)) => arr(a1)@(arr(a2))
| AFCall(A,ind) => (List.concat (map arr ind))

fun arrb(b) = case b of
  (True | False) => []
| (Equals(a1,a2) | LessEq(a1,a2) | Less(a1,a2)
  | GreEq(a1,a2) | Gre(a1,a2)) => arr(a1)@(arr(a2))
| Not(b1) => arrb(b1)
| (And(b1,b2) | Or(b1,b2)) => arrb(b1)@(arrb(b2))

fun arrs(term,Math(Loc(Arr(A,ind))),s) = 
  let
    val sarr = arr(s)
    fun getAArr term =
       case term of 
         Math(a) => arr(a)
       | Boolean(b) => arrb(b)
       | Seq(nil) => []
       | Seq(x::xs) => getAArr(x)@(getAArr(Seq(xs)))
       | _ => raise Fail "Error with arrs"
    val aarr = getAArr(term)
    fun getSubArr term = case term of
         Math(a) => arr(subForXInA(Loc(Arr(A,ind)),Math(s),a))
       | Boolean(a) => arrb(subForXInB(Loc(Arr(A,ind)),Math(s),a))
       | Seq(nil) => []
       | Seq(x::xs) => getSubArr(x)@(getSubArr(Seq(xs)))
    val subarr = getSubArr(term)
    val anoti = List.filter (fn x => x <> Arr(A,ind)) aarr
    val aands = List.filter (fn x => member x anoti) sarr
    val snotint = List.filter (fn x => not(member x aands)) sarr
  in
    List.filter (fn x => not(member x snotint)) subarr
  end


fun inEqualArrs(c) = case c of
  Boolean(Not(Equals(a1,a2))) => [(a1,a2)]
| Seq(c1) => List.concat (map inEqualArrs c1)
| _ => []

fun equalArrs(c) = case c of
  Boolean(Equals(a1,a2)) => [(a1,a2)]
| Seq(c1) => List.concat (map equalArrs c1)
| _ => []

fun aaxiom1(subs) = let
  val Ai = lookup (Arr("A",[Loc(Var("x"))])) subs
  val i = case Ai of
    SOME(Math(Loc(Arr(_,[x])))) => x
  | _ => raise Fail "Error with preconditions"

  val Aj = lookup (Arr("A",[Loc(Var("y"))])) subs
  val (A,j) = case Aj of
    SOME(Math(Loc(Arr(Ar,[y])))) => (Ar,y)
   |  _ => raise Fail "Error with preconditions"
  val s = lookup (Var("s")) subs
  val s = case s of
    SOME(Math(s')) => s'
  | _ => raise Fail "Error with preconditions"
  val t = lookup (Var("t")) subs
  val t = case t of
    SOME(t') => t'
  | NONE => raise Fail "Error with preconditions"
  val precond = lookup LPre subs
  val notEqual = case precond of
    SOME(pre) => inEqualArrs(pre)
  | NONE => raise Fail "Error with preconditions"
  val tset = arrs(t,valOf(Ai),s)
  val jset = arrs(Math(j),valOf(Ai),s)
  val arrayS = arr(s)
  val arrayI = arr(i)
  val j' = subForXInA(Loc(Arr(A,[i])),Math(s),j)
  fun inNE i j = member (i,j) notEqual orelse member (j,i) notEqual
  fun allinNE i lst = List.all (fn (Arr(_,[x])) => inNE i x) lst
in
  (inNE i j) andalso allinNE i tset andalso allinNE i jset andalso
  allinNE j' arrayS andalso allinNE j' arrayI
end

fun aaxiom2(subs) = let
  val Ai = lookup (Arr("A",[Loc(Var("x"))])) subs
  val i = case Ai of
    SOME(Math(Loc(Arr(_,[x])))) => x
  | _ => raise Fail "Error with preconditions"

  val Aj = lookup (Arr("A",[Loc(Var("y"))])) subs
  val (A,j) = case Aj of
    SOME(Math(Loc(Arr(Ar,[y])))) => (Ar,y)
   |  _ => raise Fail "Error with preconditions"
  val s = lookup (Var("s")) subs
  val s = case s of
    SOME(Math(s')) => s'
  | _ => raise Fail "Error with preconditions"
  val t = lookup (Var("t")) subs
  val t = case t of
    SOME(t') => t'
  | NONE => raise Fail "Error with preconditions"
  val precond = lookup LPre subs
  val notEqual = case precond of
    SOME(pre) => inEqualArrs(pre)
  | NONE => raise Fail "Error with preconditions"
  val tset = arrs(t,valOf(Ai),s)
  val jset = arrs(Math(j),valOf(Ai),s)
  val arrayS = arr(s)
  val arrayI = arr(i)
  fun inNE i j = member (i,j) notEqual orelse member (j,i) notEqual
  fun allinNE i lst = List.all (fn (Arr(_,[x])) => inNE i x) lst
in
  (inNE i j) andalso allinNE i tset andalso allinNE i jset andalso
  allinNE i arrayS andalso allinNE i arrayI
end

fun aaxiom3(subs) = let
  val Ai = lookup (Arr("A",[Loc(Var("x"))])) subs
  val i = case Ai of
    SOME(Math(Loc(Arr(_,[x])))) => x
  | _ => raise Fail "Error with preconditions"

  val Aj = lookup (Arr("A",[Loc(Var("y"))])) subs
  val (A,j) = case Aj of
    SOME(Math(Loc(Arr(Ar,[y])))) => (Ar,y)
   |  _ => raise Fail "Error with preconditions"
  val s = lookup (Var("s")) subs
  val s = case s of
    SOME(Math(s')) => s'
  | _ => raise Fail "Error with preconditions"
  val t = lookup (Var("t")) subs
  val t = case t of
    SOME(t') => t'
  | NONE => raise Fail "Error with preconditions"
  val precond = lookup LPre subs
  val (equal,notEqual) = case precond of
    SOME(pre) => (equalArrs(pre),inEqualArrs(pre))
  | NONE => raise Fail "Error with preconditions"
  val tset = arrs(t,valOf(Ai),s)
  val jset = arrs(Math(j),valOf(Ai),s)
  val arrayS = arr(s)
  val arrayI = arr(i)
  fun inE i j = member (i,j) equal orelse member (j,i) equal
  fun inNE i j = member (i,j) notEqual orelse member (j,i) notEqual
  fun allinNE i lst = List.all (fn (Arr(_,[x])) => inNE i x) lst
in
  (inE i j orelse i = j) andalso allinNE i tset andalso allinNE i jset
end

fun aaxiom4(subs) = let
  val Ai = lookup (Arr("A",[Loc(Var("x"))])) subs
  val i = case Ai of
    SOME(Math(Loc(Arr(_,[x])))) => x
  | _ => raise Fail "Error with preconditions"
  val s = lookup (Var("s")) subs
  val s = case s of
    SOME(Math(s')) => s'
  | _ => raise Fail "Error with preconditions"
  val t = lookup LPhi subs
  val t = case t of
    SOME(t') => t'
  | NONE => raise Fail "Error with preconditions"
  val precond = lookup LPre subs
  val precond = case precond of
    SOME(pre') => pre'
  | NONE => raise Fail "Error with preconditions"
  val notEqual = inEqualArrs(precond)
  val tset = arrs(t,valOf(Ai),s)
  fun inNE i j = member (i,j) notEqual orelse member (j,i) notEqual
  fun allinNE i lst = List.all (fn (Arr(_,[x])) => inNE i x) lst
in
  allinNE i tset
end

fun aaxiom5(subs) = let
  val Aj = lookup (Arr("A",[Loc(Var("y"))])) subs
  val (A,j) = case Aj of
    SOME(Math(Loc(Arr(Ar,[y])))) => (Ar,y)
   |  _ => raise Fail "Error with preconditions"
  val x = lookup (Var("x")) subs
  val x = case x of
    SOME(Math(Loc(GVar(x')))) => x'
   |  _ => raise Fail "Error with preconditions"
  val s = lookup (Var("s")) subs
  val s = case s of
    SOME(Math(s')) => s'
  | _ => raise Fail "Error with preconditions"
  val arrayS = arr(s)
  val j' = subForXInA(Loc(Var(x)),Math(s),j)
  val precond = lookup LPre subs
  val notEqual = case precond of
    SOME(pre) => inEqualArrs(pre)
  | NONE => raise Fail "Error with preconditions"
  fun inNE i j = member (i,j) notEqual orelse member (j,i) notEqual
  fun allinNE i lst = List.all (fn (Arr(_,[x])) => inNE i x) lst
in
  allinNE j' arrayS
end

fun aaxiom6(subs) = let
  val Ai = lookup (Arr("A",[Loc(Var("x"))])) subs
  val i = case Ai of
    SOME(Math(Loc(Arr(_,[x])))) => x
  | _ => raise Fail "Error with preconditions"

  val y = lookup (Var("y")) subs
  val y = case y of
    SOME(Math(Loc(y'))) => y'
   |  _ => raise Fail "Error with preconditions"
  val s = lookup (Var("s")) subs
  val s = case s of
    SOME(Math(s')) => s'
  | _ => raise Fail "Error with preconditions"
  val t = lookup (Var("t")) subs
  val t = case t of
    SOME(t') => t'
  | NONE => raise Fail "Error with preconditions"
  val precond = lookup LPre subs
  val notEqual = case precond of
    SOME(pre) => inEqualArrs(pre)
  | NONE => raise Fail "Error with preconditions"
  val tset = arrs(t,valOf(Ai),s)
  fun inNE i j = member (i,j) notEqual orelse member (j,i) notEqual
  fun allinNE i lst = List.all (fn (Arr(_,[x])) => inNE i x) lst
in
  (allinNE i tset) andalso not(varInA(y,s)) andalso not(varInA(y,i))
end

fun aaxiom8(subs) = let
  val Ai = lookup (Arr("A",[Loc(Var("x"))])) subs
  val i = case Ai of
    SOME(Math(Loc(Arr(_,[x])))) => x
  | _ => raise Fail "Error with preconditions"
  val s = lookup (Var("s")) subs
  val s = case s of
    SOME(Math(s')) => s'
  | _ => raise Fail "Error with preconditions"
  val t = lookup (Var("t")) subs
  val t = case t of
    SOME(t') => t'
  | NONE => raise Fail "Error with preconditions"
  val precond = lookup LPre subs
  val notEqual = case precond of
    SOME(pre) => inEqualArrs(pre)
  | NONE => raise Fail "Error with preconditions"
  val tset = arrs(t,valOf(Ai),s)
  val arrayS = arr(s)
  val arrayI = arr(i)
  fun inNE i j = member (i,j) notEqual orelse member (j,i) notEqual
  fun allinNE i lst = List.all (fn (Arr(_,[x])) => inNE i x) lst
in
  allinNE i tset andalso allinNE i arrayS andalso allinNE i arrayI
end

(* Library of first order axioms *)
val () = library := [("S1","x := s ; y := t == y := t[x/s] ; x := s",
                       [unique,yNotFree],(Seq([Assign (Var "x",Loc (Var "s")),
                        Assign (Var "y",Loc (Var "t"))]),
                        Seq([Assign (Var "y",RepA(Loc(Var("t")),
                        Loc(Var("x")),Loc(Var("s")))),
                        Assign (Var "x",Loc (Var "s"))]))),
                     ("S2","x := s ; y := t == x := s; y := t[x/s]",
                       [unique,xNotFree],(Seq([Assign (Var "x",Loc (Var "s")),
                        Assign (Var "y",Loc (Var "t"))]),
                        Seq([Assign (Var "x",Loc (Var "s")), 
                            Assign (Var "y",RepA(Loc(Var("t")),
                            Loc(Var("x")),Loc(Var("s"))))]))),

                     ("S3","x := s ; x := t == x := t[x/s]",
                      [],
                      (Seq([Assign (Var "x",Loc (Var "s")),
                       Assign (Var "x",Loc (Var "t"))]),
                       Assign (Var "x",RepA(Loc(Var("t")),
                       Loc(Var("x")),Loc(Var("s")))))),
          ("S4","x := a;x = a == x := a",[],
           (Seq([Assign (Var "x",Loc (Var "a")),
            Boolean(Equals(Loc(Var "x"),Loc(Var("a"))))]),
            Assign (Var "x",Loc (Var "a")))),

          ("S5","s = t; x := s == s = t; x := t",[],
           (Seq([Boolean(Equals(Loc(Var "s"),Loc(Var("t")))),
                Assign (Var "x",Loc (Var "s"))]),
            Seq([Boolean(Equals(Loc(Var "s"),Loc(Var("t")))),
                Assign (Var "x",Loc (Var "t"))]))),
          ("S6","x := x == 1",[],(Assign(Var "x",Loc(Var "x")),One)),
          ("S7","phi[x/t]; x := t == x := t; phi",[],
           (Seq([Rep(Phi,Loc(Var("x")),Loc(Var("t"))),Assign (Var "x",Loc (Var "t"))]),
            Seq([Assign (Var "x",Loc (Var "t")),Phi]))),

("Arr1","PRE ; A[x] := s ; A[y] := t == PRE ; A[y] := t[A[x]/s] ; A[x] := s",
                       [aaxiom1],(Seq([Pre,Assign (Arr("A",[Loc( Var "x")]),Loc (Var "s")),
                        Assign (Arr("A",[Loc(Var("y"))]),Loc (Var "t"))]),
                        Seq([Pre,Assign (Arr("A",[RepA(Loc(Var("y")),Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s")))]),RepA(Loc(Var("t")),
                        Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s")))),
                        Assign (Arr("A",[Loc(Var("x"))]),Loc (Var "s"))]))),
("Arr2","PRE ; A[x] := s ; A[y] := t == PRE ; A[x] := s ; A[y] := t[A[x]/s]",
                       [aaxiom2],(Seq([Pre,Assign (Arr("A",[Loc( Var "x")]),Loc (Var "s")),
                        Assign (Arr("A",[Loc(Var("y"))]),Loc (Var "t"))]),
                        Seq([Pre,Assign (Arr("A",[Loc(Var("x"))]),Loc (Var "s")),Assign (Arr("A",[RepA(Loc(Var("y")),Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s")))]),RepA(Loc(Var("t")),
                        Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s"))))]))),

("Arr3","PRE ; A[x] := s ; A[y] := t == PRE ; A[y] := t[A[x]/s]",
                       [aaxiom3],(Seq([Pre,Assign (Arr("A",[Loc( Var "x")]),Loc (Var "s")),
                        Assign (Arr("A",[Loc(Var("y"))]),Loc (Var "t"))]),
                        Seq([Pre,Assign(Arr("A",[RepA(Loc(Var("y")),Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s")))]),RepA(Loc(Var("t")),
                        Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s"))))]
                        ))),

("Arr4","PRE; Phi[A[x]/s] ; A[x] := s == PRE ; A[x] := s ; Phi",
                       [aaxiom4],(Seq([Pre,Rep(Phi,Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s"))),Assign (Arr("A",[Loc( Var "x")]),Loc (Var "s"))]),Seq([Pre,Assign (Arr("A",[Loc( Var "x")]),Loc (Var "s")),Phi]))),

("Arr5","Phi ; x := s ; A[y] := t == Phi ; A[y] := t[x/s] ; x := s",
                       [aaxiom5],(Seq([Pre,Assign (Var "x",Loc (Var "s")),
                        Assign (Arr("A",[Loc(Var("y"))]),Loc (Var "t"))]),
                        Seq([Pre,Assign (Arr("A",[RepA(Loc(Var("y")),Loc(Var("x")),Loc(Var("s")))]),RepA(Loc(Var("t")),
                        Loc(Var("x")),Loc(Var("s")))),
                        Assign (Var("x"),Loc (Var "s"))]))),

("Arr6","Phi ; A[x] := s ; y := t == Phi ; y := t[A[x]/s] ; A[x] := s",
                       [aaxiom6],(Seq([Pre,Assign (Arr("A",[Loc( Var "x")]),Loc (Var "s")),
                        Assign (Var("y"),Loc (Var "t"))]),
                        Seq([Pre,Assign (Var("y"),RepA(Loc(Var("t")),
                        Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s")))),
                        Assign (Arr("A",[Loc(Var("x"))]),Loc (Var "s"))]))),

("Arr7","x := s ; A[y] := t == x:= s ; A[y] := t[x/s]",
                       [xNotFree],(Seq([Assign (Var "x",Loc (Var "s")),
                        Assign (Arr("A",[Loc(Var("y"))]),Loc (Var "t"))]),
                        Seq([Assign (Var("x"),Loc (Var "s")),Assign (Arr("A",[RepA(Loc(Var("y")),Loc(Var("x")),Loc(Var("s")))]),RepA(Loc(Var("t")),
                        Loc(Var("x")),Loc(Var("s"))))]))),

("Arr8","Phi ; A[x] := s ; y := t == Phi ; A[x] := s ; y := t[A[x]/s]",
                       [aaxiom8],(Seq([Pre,Assign (Arr("A",[Loc( Var "x")]),Loc (Var "s")),
                        Assign (Var("y"),Loc (Var "t"))]),
                        Seq([Pre,Assign (Arr("A",[Loc(Var("x"))]),Loc (Var "s")),Assign (Var("y"),RepA(Loc(Var("t")),Loc(Arr("A",[Loc(Var("x"))])),Loc(Var("s"))))])))
]

(* Convert ground variables to variables *)
fun gVarToVarA(term) = case term of
  N(n) => N(n)
| Loc(GVar(x)) => Loc(Var(x))
| Loc(Arr(s,x)) => Loc(Arr(s,map gVarToVarA x))
| Loc(ASub(y,x,s)) => subForXInA(gVarToVarA(Loc(x)),Math(gVarToVarA(s)),gVarToVarA(Loc(y)))
| Loc(x) => Loc(x)
| Plus(a1,a2) => Plus(gVarToVarA(a1),gVarToVarA(a2))
| Minus(a1,a2) => Minus(gVarToVarA(a1),gVarToVarA(a2))
| Times(a1,a2) => Times(gVarToVarA(a1),gVarToVarA(a2))
| Div(a1,a2) => Div(gVarToVarA(a1),gVarToVarA(a2))
| Mod(a1,a2) => Mod(gVarToVarA(a1),gVarToVarA(a2))
| AFCall(s,args) => AFCall(s,map gVarToVarA args)
(* Take care of an expression of the from s[x/t] here, too *)
| RepA(a1,a2,a3) => subForXInA(gVarToVarA(a2),Math(gVarToVarA(a3)),
                      gVarToVarA(a1))


fun gVarToVarB(term) = case term of
  (True | False) => term
| Equals(a1,a2) => Equals(gVarToVarA(a1),gVarToVarA(a2))
| LessEq(a1,a2) => LessEq(gVarToVarA(a1),gVarToVarA(a2))
| GreEq(a1,a2) => GreEq(gVarToVarA(a1),gVarToVarA(a2))
| Less(a1,a2) => Less(gVarToVarA(a1),gVarToVarA(a2))
| Gre(a1,a2) => Gre(gVarToVarA(a1),gVarToVarA(a2))
| And(b1,b2) => And(gVarToVarB(b1),gVarToVarB(b2))
| Or(b1,b2) => Or(gVarToVarB(b1),gVarToVarB(b2))
| Not(b) => Not(gVarToVarB(b))

fun gVarToVar(term) = case term of
  Assign(GVar(x),a) => Assign(Var(x),gVarToVarA(a))
| Assign(x,a) => let val Loc(x') = gVarToVarA(Loc(x))
                 in  Assign(x',gVarToVarA(a)) end
| Seq(c1) => Seq(map gVarToVar c1)
| Cond(b,c1,c2) => Cond(gVarToVarB(b),gVarToVar(c1),gVarToVar(c2))
| While(b,c) => While(gVarToVarB(b),gVarToVar(c))
| Boolean(b) => Boolean(gVarToVarB(b))
| Math(a) => Math(gVarToVarA(a))
| Phi => Phi
| Pre => Pre
| One => One
| FCall(s,args) => FCall(s,map gVarToVarA args)
| Rep(a1,a2,a3) => subForXInC(gVarToVarA(a2),Math(gVarToVarA(a3)),gVarToVar(a1))
| _ => raise Fail "Error with varsInC!"


(* Convert variable to ground variable *)
fun varToGVarA(term) = case term of
  N(n) => N(n)
| Loc(Var(x)) => Loc(GVar(x))
| Loc(Arr(s,x)) => Loc(Arr(s,map varToGVarA x))
| Loc(x) => Loc(x)
| Plus(a1,a2) => Plus(varToGVarA(a1),varToGVarA(a2))
| Minus(a1,a2) => Minus(varToGVarA(a1),varToGVarA(a2))
| Times(a1,a2) => Times(varToGVarA(a1),varToGVarA(a2))
| Div(a1,a2) => Div(varToGVarA(a1),varToGVarA(a2))
| Mod(a1,a2) => Mod(varToGVarA(a1),varToGVarA(a2))
| AFCall(s,args) => AFCall(s,map varToGVarA args)
| _ => raise Fail "Should not have this in the translation!"


fun varToGVarB(term) = case term of
  (True | False) => term
| Equals(a1,a2) => Equals(varToGVarA(a1),varToGVarA(a2))
| LessEq(a1,a2) => LessEq(varToGVarA(a1),varToGVarA(a2))
| GreEq(a1,a2) => GreEq(varToGVarA(a1),varToGVarA(a2))
| Less(a1,a2) => Less(varToGVarA(a1),varToGVarA(a2))
| Gre(a1,a2) => Gre(varToGVarA(a1),varToGVarA(a2))
| And(b1,b2) => And(varToGVarB(b1),varToGVarB(b2))
| Or(b1,b2) => Or(varToGVarB(b1),varToGVarB(b2))
| Not(b) => Not(varToGVarB(b))

fun varToGVar(term) = case term of
  Assign(Var(x),a) => Assign(GVar(x),varToGVarA(a))
| Assign(x,a) => let val Loc(x') = varToGVarA(Loc(x))
                 in  Assign(x',varToGVarA(a)) end
| Seq(c1) => Seq(map varToGVar c1)
| Cond(b,c1,c2) => Cond(varToGVarB(b),varToGVar(c1),varToGVar(c2))
| While(b,c) => While(varToGVarB(b),varToGVar(c))
| Boolean(b) => Boolean(varToGVarB(b))
| Math(a) => Math(varToGVarA(a))
| Phi => Phi
| Pre => Pre
| One => One
| FCall(s,args) => FCall(s,map varToGVarA args)
| Rep(a,b,c) => Rep(a,b,c)
| _ => raise Fail "Error with varToGVar"

fun flatten(term) =  case term of
  Seq(x) => let
     val y = map flatten x
     val z = List.concat(map (fn u => case u of Seq v => v | _ => [u]) y)
   in
     case z of [z'] => z' | _ => Seq(z)
   end
| Cond(b,c1,c2) => Cond(b,flatten(c1),flatten(c2))
| While(b,c) => While(b,flatten(c))
| Rep(a,b,c) => Rep(flatten(a),b,c)
| _ => term




(* Saving/loading from XML file *)
open XML

fun tableToXML (id,(blist,clist)) = let
  fun bToXML(b,kat) = Substring.all("<bterm><slang>$"^bTermToString(b)^"</slang>"^
                      termToXML(kat)^"</bterm>")
  fun cToXML(c,kat) = Substring.all("<cterm><slang>"^cTermToString(c)^"</slang>"^
                      termToXML(kat)^"</cterm>")
in
  "<table><bterms>"^(Substring.concat(map bToXML (blist)))^"</bterms><cterms>"^
                         (Substring.concat(map cToXML (clist)))^
  "</cterms></table>"
end

fun tableFromXML (xml) = let
  val xml1::xml2::_ = getContent("table",xml)
  fun parseBTerm (bxml) = let
    val bxml1::bxml2::_ = getContent("bterm",bxml)
    val Boolean(slang) = flatten(Parser.parseLine(XML.getString("slang",bxml1)))
    val kat = makeConstant(valOf(parseTerm(XML.getString("term",bxml2))))
  in
    (slang,kat)
  end
  fun parseCTerm (cxml) = let
    val cxml1::cxml2::_ = getContent("cterm",cxml)
    val slang = flatten(Parser.parseLine(XML.getString("slang",cxml1)))
    val kat = makeConstant(valOf(parseTerm(XML.getString("term",cxml2))))
  in
    (slang,kat)
  end
(*  fun fromXML (xml:xml):((B * term) list * (C * term) list) =
    case xml of
      ELEMENT("bterm",bterms) => parseBTerm bterms
    | ELEMENT("cterm",cterms) => parseCTerm cterms*)
in
  (map parseBTerm ((getContent("bterms",xml1))),
  map parseCTerm (getContent("cterms",xml2)))
end


end