/*******************************************************************************
 *  mincut-cluster -- program that uses minimum cut algorithm to cluster a     *
 *                    graph and varies a parameter "alpha" to obtain a         *
 *                    hierarchical clustering. This method is described in     *
 *                    [Tar99]                                                  *
 *                                                                             *
 *  Author: Ramiro Rodriguez       Dec 2003                                    *
 *          Jonathan Pereses       Dec 2003                                    *
 *          Ariful Alam Gani       Dec 2003                                    *
 *                                                                             *
 *  Usage: mincut-cluster <edges file> <ids file> [output file]                *
 *         If no output file is specified we output to standard outout         *
 *                                                                             *
 *  Input: Edges file - the first line contains the number of nodes in the     *
 *                      graph and the number of edges in the graph. After that *
 *                      each line represents an edge where the first number is *
 *                      the from node and the second number is the to node.    *
 *                      The node numbers are expected as consecutive integers  *
 *                      from 0 to the number of nodes minus one.               *
 *                                                                             *
 *                      EXAMPLE:                                               *
 *                                                                             *
 *                      3 4                                                    *
 *                      0 1                                                    *
 *                      2 3                                                    *
 *                      3 1                                                    *
 *                      1 0                                                    *
 *         Ids file - The first line is the number of nodes in the graph that  *
 *                    actually correspond to something we wish to cluster.     *
 *                    Each consecutive line is a pair of numbers, the first    *
 *                    number is the id in the graph (between 0 and n-1). The   *
 *                    second number is the true id of the node.                *
 *                                                                             *
 *  Output: The output is given as a list of communities. after # we list the  *
 *          alpha value at which these communities occur. After the alpha we   *
 *          list all the merges that occured for that alpha.                   *
 ******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "cluster.h"
#include "mincut.h"
#include "iolib.h"
#include "tree.h"

/* Global variables */
bool treedebug = false;
bool treeclustercalls = 0;
extern bool clusdebug;
int calls = 0;
int truetotal;
FILE *output;

/* Loads the graph and calls the recursive procedure to find all the splits
   in the tree by varying alpha.
*/
int main(int args, char* argv[]){
  char *edgesfile;
  char *idsfile;
  char *ofilename;
  int numnodes, commLow, commHigh;
  FILE* file;
  Graph_t Graph;
  numtype alphaD, alphaL, alphaH;

  if (args != 4) {
    printf("usage: <edges file> <ids file> <output file>\n");
    exit(1);
  }
  //fprintf(stderr,"MinCut Hirarchical Clustering\n");
  edgesfile = argv[1];
  idsfile = argv[2];
  ofilename = argv[3];

  file = fopen(edgesfile,"r");
  fscanf(file,"%d",&numnodes);
  fclose(file);

  file = fopen(edgesfile,"r");
  fscanf(file,"%d",&commLow);
  fclose(file);

  file = fopen(idsfile,"r");
  fscanf(file,"%d",&truetotal);
  fclose(file);

  output = fopen(ofilename,"w");
  alphaL = numnodes*numnodes*2;
  alphaH = numnodes*2;


  // read in graph from disk
  Graph = parseInputFile(edgesfile,idsfile,alphaL,alphaL);
  Graph->sink = Graph->n-1;
  //printf(stderr,"sink %d\n",Graph->sink);
  //printf(stderr,"Read graph\n");
  //PrintGraph(Graph);

  // cluster with alpha value of 1
  //commLow = cluster(Graph,alphaL);
  printf("clustered with alpha = 1, Number of communities %d\n",commLow);
  //writeClusters(Graph,output,alphaL);
  writeAll(Graph,output,alphaL);



  // cluster with alpha value of 1/(2n^2)
  commHigh = cluster(Graph,alphaH);
  writeClusters(Graph,output,alphaH,truetotal);
  printf("clustered with alpha = 1/n, Number of communities %d\n",commHigh);


  // re-read graph because above clustering changed it
  Graph = parseInputFile(edgesfile,idsfile,alphaL,alphaL);
  Graph->sink = Graph->n-1;

  //PrintGraph(Graph);
  //exit(0);

  // call recursive tree clustering routine
  treeCluster(alphaH,commHigh,alphaL,commLow,Graph);

  printf("clustering complete\n");
  fclose(output);

  return 0;
}

/*
 */
int treeCluster(numtype alphaHigh, int commHigh, numtype alphaLow, int commLow, Graph_t Graph){
  numtype alphaMid;
  int commMid, nextCommsize;
  Graph_t G;


  treeclustercalls++;
  //printf(stderr,"ENTERING TREE CLUSTER\n");
  // compute middle alpha numerator hence clustering the graph with this as the alpha weights
  // will like having alpha be (alphaHigh / alphaD + alphaLow / alphaD ) / 2 or the midpoint
  // between the alpha values if the change in alpha is too small for a merge to occur
  // return since we found all the merges
  alphaMid = (alphaHigh + alphaLow) / 2;
  //fprintf(stderr,"about to cluster %d %d\n",alphaMid);
  //printf(stderr,"alpha -> High %d%d  Mid %d%d  Low %d%d\n",alphaHigh,alphaMid,alphaLow);


  if (alphaMid == alphaHigh)
    return 0;


  // Run clustering for one value of alpha on this graph
  //checkNumTrue(Graph);
  //checkOtherNode(Graph);
  //if (alphaMid == 91262846){
    //fprintf(stderr,"debugging\n");
    //clusdebug = true;
  //}
  commMid = cluster(Graph,alphaMid);
  printf("clustered with alpha = %d %d\n",alphaMid);
  //printf(stderr,"comm -> High %d  Mid %d  Low %d\n",commHigh,commMid,commLow);
  //printf(stderr,"clustered the middle with %d communities\n",commMid);
  //if (treeclustercalls == 105){
  //if (alphaMid == 2048){
    //printComm(Graph,4,30);
    //exit(0);
  //}
  /*
  if (treedebug){
    calls++;
    if (calls == 9){
      PrintGraph(Graph);
      printComm(Graph,3,20);
      exit(0);
    }
  }
  */

  //checkNumTrue(Graph);
  //checkOtherNode(Graph);
  //fprintf(stderr,"about to write to comms\n");
  // if there was a merge write it to disk
  // we only have to check in Graph to see the different
  // changes if there were any
  if ( (commMid != commLow) && (commMid != commHigh)){
    //printf(stderr,"there were changed writing them\n");
    writeClusters(Graph,output,alphaMid,truetotal);
  }

  //fprintf(stderr,"about to do small graph stuff\n");
  // create small graphs and recusively call this
  // function
  int temp = 0;
  //checkNumTrue(Graph);
  //checkOtherNode(Graph);
  if ( (alphaLow -  alphaHigh > 1) && (commMid != commLow) ){
    //printf(stderr,"clustering below alphaMid\n");
    while ((G = nextGraph(Graph,&nextCommsize)) != NULL){
      temp++;
      //printf(stderr,"FSDFSDFSDF %d clustering some small graph\n",temp);
      //PrintGraph(G);
      //exit(0);
      //printf(stderr,"cluster %d\n",temp);
      //if (G->n > 3){
        //PrintGraph(G);
        //printf(stderr,"---------------------------------------------------------\n");
        //exit(0);
      //}
      //fprintf(stderr,"small graph next\n");
      //checkNumTrue(G);
      //checkOtherNode(G);
      treeCluster(alphaMid,1,alphaLow,nextCommsize,G);
      //printf("what a great ride\n");
      //exit(0);
    }
    //printf(stderr,"calling exit\n");
    //exit(0);
  }

  //exit(0);
  //fprintf(stderr,"about to do compress graph stuff\n");
  //checkNumTrue(Graph);
  //checkOtherNode(Graph);
  if ( (alphaMid - alphaHigh > 1) && (commMid != commHigh) ){
    //printf(stderr,"for some reason we are here\n");
    //treedebug = true;
    // Compact Graph (the larger graph is deleted
    //printf(stderr,"about to compress graph with alpha %d %d\n",alphaMid);
    //exit(0);
    //PrintGraph(Graph);
    //fprintf(stderr,"compressing graph %d %d\n",alphaMid);
    G = compressGraph(Graph);
    //fprintf(stderr,"performing some checks\n");
    //checkNumTrue(G);
    //fprintf(stderr,"performing other checks\n");
    //checkOtherNode(G);
    //printf(stderr,"compressing graph\n");
    //PrintGraph(G);
    //exit(0);
    //fprintf(stderr,"compressed graph %d %d\n",alphaMid);
    treeCluster(alphaHigh,commHigh,alphaMid,commMid,G);
    return 0;
  }

  // free Graph
  //printf(stderr,"about to free graph\n");
  freeGraph(Graph);
  //printf(stderr,0"graph free\n");
}
