/*******************************************************************************
 *  mincut -- Finds the minimun cut of a given graph for use in doing mincut   *
 *            clustering. We implemented the preflow-push algorithm with the   *
 *            following optimizations:                                         *
 *                                                                             *
 *            1. Gap Realabeling                                               *
 *            2. Freezing nodes at height n                                    *
 *            3. Relabel one above next higest node                            *
 *                                                                             *
 *  Author: Ramiro Rodriguez       Dec 2003                                    *
 *          Jonathan Pereses       Dec 2003                                    *
 *          Ariful Alam Gani       Dec 2003                                    *
 *                                                                             *
 *  Usage: mincut <edges file> <ids file> <sink> <source> [output file]        *
 *         If no output file is specified we output to standard outoup         *
 *                                                                             *
 *  Input: Edges file - the first line contains the number of nodes in the     *
 *                      graph and the number of edges in the graph. After that *
 *                      each line represents an edge where the first number is *
 *                      the from node and the second number is the to node.    *
 *                      The node numbers are expected as consecutive integers  *
 *                      from 0 to the number of nodes minus one.               *
 *                                                                             *
 *                      EXAMPLE:                                               *
 *                                                                             *
 *                      3 4                                                    *
 *                      0 1                                                    *
 *                      2 3                                                    *
 *                      3 1                                                    *
 *                      1 0                                                    *
 *         Ids file - The first line is the number of nodes in the graph that  *
 *                    actually correspond to something we wish to cluster.     *
 *                    Each consecutive line is a pair of numbers, the first    *
 *                    number is the id in the graph (between 0 and n-1). The   *
 *                    second number is the true id of the node.                *
 *                                                                             *
 *  Output: The output is given as a list of the elements in the minimum cut   *
 ******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "mincut.h"
#include "iolib.h"

bool pushdebug = false;
bool reladebug = false;
bool debug = false;

void printExcess(Graph_t G, int max);
void printHeight(Graph_t G, int max);

#ifdef MINCUT
int main(int args, char* argv[]){
  char *edgesfile;
  char *idsfile;
  int numnodes, commLow, commHigh;
  FILE* file;
  Graph_t G;
  numtype alpha;
  int temp;
  if (args != 4) {
    printf("usage: <edges file> <ids file> <alpha numerator>\n");
    printf("the alpha denominator is always the n^2 where n is"
           "always the number of nodes\n");
    exit(1);
  }

  edgesfile = argv[1];
  idsfile = argv[2];
  temp = atoi(argv[3]);
  alpha = temp;

  file = fopen(edgesfile,"r");
  fscanf(file,"%d",&numnodes);
  fclose(file);

  // import the graph into memory
  G = parseInputFile(edgesfile,idsfile,alpha,numnodes*numnodes);
  printf("alpha = %d %d\n",alpha);
  //PrintGraph(G);
  G->sourceQueue = 0;
  G->sink = G->n - 1;
  // run mincut
  MinCut(G);
}
#endif


/*
  initGraph - Given a graph initializes all the datastructures necessary to
              run the preflow-push algorithm.

  Input:  Graph_t G - Graph to initialize datastructures for

  Output: void
 */
void initGraph(Graph_t G){
  Node_t NodeArray = G->nodes;
  Edge_t e;
  int i, prev;
  Height_t tmpH;
  AvailNode_t tempNode;

  // Initialize heights to zero and flows to zero
  for (i = 0; i < G->n; i++) {
    NodeArray[i].excess = 0;
    NodeArray[i].height = 0;
    e = NodeArray[i].firstEdge;
    while (e != NULL) {
      e->flow = 0;
      e = e->nextEdge;
    }
  }

  // Initialize heights data structure
  // place the source node at height n
  // place all other nodes at height 0
  tmpH = new Height;
  G->height = new Height;
  tmpH->above = NULL;
  G->height->above = tmpH;
  tmpH->below = G->height;
  G->height->below = NULL;

  // place the source at height n
  tmpH->nodes = G->sourceQueue;
  tmpH->height = G->n;
  G->heightn = tmpH;
  NodeArray[G->sourceQueue].prev = -1;
  NodeArray[G->sourceQueue].next = -1;
  NodeArray[G->sourceQueue].height = G->n;

  // place everything else at height 0
  G->height->height = 0;
  prev = -1;
  for(i = 0; i < G->n; i++){
    if (i != G->sourceQueue){
      if (prev == -1)
        G->height->nodes = i;
      else
        NodeArray[prev].next = i;
      NodeArray[i].prev = prev;
      prev = i;
    }
  }
  NodeArray[prev].next = -1;

  // Initialize Excess Queue
  G->excess = new AvailHeight;
  G->excess->height = 0;
  G->excess->nextAvailHeight = NULL;
  G->excess->heightAvailNodes = NULL;

  // push flow from the source
  // to all the nodes it can reach
  e = NodeArray[G->sourceQueue].firstEdge;
  while (e != NULL){
    // push flow
    e->flow = e->capacity;
    NodeArray[e->toNode].excess = e->capacity;
    // add to excess data structure if the node is
    // not the sink
    if (e->toNode == G->sink){
      e = e->nextEdge;
      continue;
    }
    tempNode = new AvailNode;
    if (G->excess->heightAvailNodes != NULL)
      G->excess->heightAvailNodes->prevAvailNode = tempNode;
    tempNode->nextAvailNode = G->excess->heightAvailNodes;
    tempNode->prevAvailNode = NULL;
    tempNode->nodeNum = e->toNode;
    G->excess->heightAvailNodes = tempNode;
    e = e->nextEdge;
  }
}



/*
  MinCut - Given a graph and a specified source and sink return the first
           element in the mincut. The mincut can then access as a link list

  Input:  Graph_t G - Graph to find the mincut of

  Output: first element in the mincut
 */
int MinCut(Graph_t G){
  Edge_t e;
  Node_t fromNode;
  Node_t NodeArray;
  int numIterations = 0;
  //int from;
  // Initialize Graph by creating all the queues
  // and saturating all edges from the source
  initGraph(G);
  //printHeight(G,100);
  //printExcess(G,100);
  //PrintGraph(G);

  NodeArray = G->nodes;
  //main loop
  while (G->excess != NULL && G->excess->heightAvailNodes != NULL) {
    //if (numIterations % 1000 == 0){
    //if (numIterations > 120110){
    if (debug){
      printf("\n\n");
      printf("------- Iteration %d -------\n",numIterations);
      printHeight(G,10);
      printExcess(G,10);
    }
    numIterations++;
    if (debug && (numIterations == 120120)){
      //printHeight(G,10);
      //printExcess(G,10);
      exit(0);
    }
    // Pick the next available node with excess
    fromNode = &NodeArray[G->excess->heightAvailNodes->nodeNum];
    if (pushdebug){
      printf("v = %d\n",G->excess->heightAvailNodes->nodeNum);
    }
    // For that node, until it loses all of its excess
    // do a push and add the new nodes with excess
    // to the proper height queue
    e = fromNode->firstEdge;
    while (e != NULL) {
      //actually do the push
      Push(G,fromNode, e);
      e = e->nextEdge;
      if (fromNode->excess == 0)
        e = NULL;
    }
    if (pushdebug)
      printf("push completed successfully\n");
    // If that node still has excess after all pushed then relabel it
    // Else remove it from the height queue structure
    if (reladebug)
      printf("the excess on the node is #\n");
      //printf("the excess on the node is %d\n",fromNode->excess);
    if (fromNode->excess > 0) {
      if (reladebug){
        printf("\n");
        printf("Relabeling %d\n",G->excess->heightAvailNodes->nodeNum);
      }
      Relabel(G,fromNode);
      if (reladebug) {
        //printHeight(G,100);
        //printExcess(G,100);
      }
    }
    else {
      AvailNode_t tmpNode;
      if (debug)
        printf("taking node off the excess queue\n");
      // No excess, so remove this node from the queue structure entirely
      tmpNode = G->excess->heightAvailNodes;
      G->excess->heightAvailNodes = G->excess->heightAvailNodes->nextAvailNode;
      delete tmpNode;
      // If this was the last node from this particular height, delete this height entry
      if (G->excess->heightAvailNodes == NULL) {
        AvailHeight_t tempHeight;
        tempHeight = G->excess;
        G->excess = G->excess->nextAvailHeight;
        delete tempHeight;
      }
      //heightsWithNodes must always be the height of the next thing with excess
      //(although it may have both nextHWN and prevHWN not null at the time)
      //this will always be exactly one below unless we just pushed to the source
      while ((G->excess != NULL) && (G->excess->height != G->height->height)) {
        G->height = G->height->below;
      }
    }
  }

  // give the source excess so that it is included in cut
  G->nodes[G->sourceQueue].excess = 1;

  //return G->heightn->nodes;
  return makeMinCut(G);
}


void printDFS_QUEUE(char* queue_name, Graph_t G, int start){
  int itr = start;

  printf("%s: ",queue_name);
  while(itr != -1){
    printf("%d, ",itr);
    itr = G->nodes[itr].next;
  }
  printf("\n");
}

int makeMinCut(Graph_t G){
  int ExcessNodes = -1;
  int MincutNodes = -1;
  int Dfs_Queue = -1;
  int itr; // generic iterator
  int temp;
  Edge_t dfs_itr; // depth first search iterator



  // make all visited false
  for (int i = 0; i < G->n-1; i++){
    G->nodes[i].visited = false;
  }


  // add all nodes with excess at height n to a queue
  itr = G->heightn->nodes;
  //printf("Nodes with excess: ");
  while (itr != -1){
    temp = (G->nodes)[itr].next;

    // if excess is larger than 0 place in excess queue
    if ((G->nodes)[itr].excess > 0){
      if (ExcessNodes != -1)
        (G->nodes)[ExcessNodes].prev = itr;
      (G->nodes)[itr].next = ExcessNodes;
      (G->nodes)[itr].prev = -1;
      ExcessNodes = itr;
      (G->nodes)[itr].visited = true;
      //printf("%d , ",itr);
    }
    itr = temp;
  }
  //printf("\n");





  // for each node in this queue perform depth first search
  Dfs_Queue = ExcessNodes;
  while (Dfs_Queue != -1){

    // remove from DFS Queue
    if ((G->nodes)[Dfs_Queue].next != -1)
      (G->nodes)[(G->nodes)[Dfs_Queue].next].prev = -1;
    temp = Dfs_Queue;
    Dfs_Queue = (G->nodes)[Dfs_Queue].next;

    // place in MincutNodes queue
    if (MincutNodes != -1)
      (G->nodes)[MincutNodes].prev = temp;
    (G->nodes)[temp].next = MincutNodes;
    (G->nodes)[temp].prev = -1;
    MincutNodes = temp;

    // add the nodes with excess to the DFS Queue
    dfs_itr = (G->nodes)[temp].firstEdge;
    while (dfs_itr != NULL){
      // if flow is smaller than the capacity along the edge
      // then there is a path along the residual graph
      // so we add the node to dfs queue
      if ( (dfs_itr->flow < dfs_itr->capacity) && (!((G->nodes)[dfs_itr->toNode].visited)) ){
      //if (dfs_itr->flow < dfs_itr->capacity){
        if (Dfs_Queue != -1)
          (G->nodes)[Dfs_Queue].prev = dfs_itr->toNode;
        (G->nodes)[dfs_itr->toNode].next = Dfs_Queue;
        (G->nodes)[dfs_itr->toNode].prev = -1;
        Dfs_Queue = dfs_itr->toNode;
        (G->nodes)[dfs_itr->toNode].visited = true;
      }
      dfs_itr = dfs_itr->nextEdge;
    }

  }
  return MincutNodes;
}

/*
  Relabel - Given a graph and a node we relabel the node

  Input:  Graph_t G - Graph to initialize datastructures for

  Output: void
 */
void Relabel(Graph_t G, Node_t node) {
  Node_t NodeArray = G->nodes;

  // Relabel node in G->height
  // if node is only node at this height
  if ( (node->next == -1) && (node->prev == -1)) {
    if ( G->height->above == NULL) {
      Height_t tmpH;
      tmpH = new Height;
      tmpH->height = G->height->height + 1;
      tmpH->nodes = -1;
      tmpH->above = NULL;
      tmpH->below = G->height;
      G->height->above = tmpH;
      //fprintf(stderr,"Error: node will keep being relabled upwards\n");
      //exit(0);
    }
    // remove the height from its current place
    if ( G->height->above != NULL )
      G->height->above->below = G->height->below;
    if ( G->height->below != NULL)
      G->height->below->above = G->height->above;
    // if there is height there add node
    if ( G->height->above->height == G->n){
      Height_t tmpH;
      int tmp;
      tmp = G->height->nodes;
      tmpH = G->height;
      node->next = G->height->above->nodes;
      node->prev = -1;
      NodeArray[node->next].prev = tmp;
      G->height->above->nodes = tmp;
      G->height = G->height->above;
      delete tmpH;
    }
    else if ( (G->height->above->above != NULL) &&
              (G->height->above->above->height == G->height->above->height+1)){
      Height_t tmpH;
      int tmp;
      tmp = G->height->nodes;
      tmpH = G->height;
      node->next = G->height->above->above->nodes;
      node->prev = -1;
      NodeArray[node->next].prev = tmp;
      G->height->above->above->nodes = tmp;
      G->height = G->height->above->above;
      delete tmpH;
    }
    // if there is no height there change height
    else {
      if (G->height->above->above != NULL)
        G->height->above->above->below = G->height;
      G->height->above = G->height->above->above;
      G->height->below->above->above = G->height;
      G->height->below = G->height->below->above;
      G->height->height = G->height->below->height + 1;
    }
  }
  else {
    Height_t tmpH;
    int nodenum;
    if (node->prev != -1)
      nodenum = NodeArray[node->prev].next;
    else
      nodenum = NodeArray[node->next].prev;
    if ( G->height->above == NULL ){
      Height_t tmpH;
      tmpH = new Height;
      tmpH->height = G->height->height + 1;
      tmpH->nodes = -1;
      tmpH->above = NULL;
      tmpH->below = G->height;
      G->height->above = tmpH;
      //fprintf(stderr,"Error: node will keep being relabled upwards\n");
      //PrintGraph(G);
      //exit(0);
    }
    // remove node from the height queue
    if (node->prev != -1) 
      NodeArray[node->prev].next = node->next;
    else
      G->height->nodes = node->next;
    if (node->next != -1)
      NodeArray[node->next].prev = node->prev;

    //if (debug)
    //  printf("-->we took our node off the queue yay!\n");
    //if (debug)
    //  printHeight(G,10);
    // if height structure does not exist there create it
    if ( G->height->above->height != G->height->height+1 ) {
      tmpH = new Height;
      tmpH->height = G->height->height + 1;
      tmpH->above = G->height->above;
      tmpH->below = G->height;
      tmpH->nodes = -1;
      G->height->above->below = tmpH;
      G->height->above = tmpH;
      //if(debug)
      //  printf("-->our thing did not exist\n");
    }
    G->height = G->height->above;
    node->prev = -1;
    node->next = G->height->nodes;
    if (G->height->nodes != -1)
      NodeArray[G->height->nodes].prev = nodenum;
    G->height->nodes = nodenum;
  }
  node->height = G->height->height;




  /*   Relabel node in the excess Queue */
  // Relabel node in G->excess
  if (G->excess->heightAvailNodes->nextAvailNode == NULL) {
    G->excess->height = node->height;
  }
  else {
    AvailHeight_t newHeight;
    // Otherwise, must create a new height node above this height
    newHeight = new AvailHeight;
    newHeight->height = node->height;
    // New height's first node should point to current height's first node (the node we raise)
    newHeight->heightAvailNodes = G->excess->heightAvailNodes;
    // Current height's first node should point to the current height's second node
    G->excess->heightAvailNodes = G->excess->heightAvailNodes->nextAvailNode;
    // First node in new height is the only item, so it should point to NULL
    newHeight->heightAvailNodes->nextAvailNode = NULL;
    // New height item should point to the next height right below it
    newHeight->nextAvailHeight = G->excess;
    // Height queue iterator should point to the highest current height
    G->excess = newHeight;
  }





  /* Perform Gap Relabeling */
  // check to see if there is a gap
  if (G->excess->height >= G->n) {
    G->excess = G->excess->nextAvailHeight;
    // bring node down

    if (G->excess != NULL)
      while(G->height->height != G->excess->height)
        G->height = G->height->below;
    return;
  }
  Height_t tmpH;
  tmpH = G->height;
  // detect gap
  while ( ( tmpH->height != 0) && (tmpH->height == tmpH->below->height + 1)){
    tmpH = tmpH->below;
  }

  if (tmpH->height != 0) {
    //printHeight(G,10);
    //printExcess(G,10);
    //printf("WE HAVE GAP RELABELING\n");
    int tmpNode;
    Height_t heightItr;
    Height_t trash;
    int stopheight;
    heightItr = G->height;
    stopheight = tmpH->below->height;
    G->height = G->heightn;
    //printf("WE HAVE GAP AT %d\n",stopheight);
    // go to the top of the gap
    while ( heightItr->above->height != G->n )
      heightItr = heightItr->above;
    while (heightItr->height != stopheight){
      // find the last node
      tmpNode = heightItr->nodes;
      while(NodeArray[tmpNode].next != -1){
        NodeArray[tmpNode].height = G->n;
        tmpNode = NodeArray[tmpNode].next;
      }
      NodeArray[tmpNode].height = G->n;

      // add this link list to height n
      NodeArray[tmpNode].next = G->heightn->nodes;
      NodeArray[G->heightn->nodes].prev = tmpNode;
      G->heightn->nodes = heightItr->nodes;
      heightItr->above->below = heightItr->below;
      heightItr->below->above = heightItr->above;
      trash = heightItr;
      heightItr = heightItr->below;
      delete trash;
    }


    
    // relabel nodes with excess
    AvailHeight_t tmp;
    AvailHeight_t trashH;
    AvailNode_t tmpNodeE;
    stopheight = tmpH->height;
    G->excess->height = G->n;
    tmp = G->excess->nextAvailHeight;
    while ( (tmp != NULL) && (tmp->height >= stopheight)){

      // get last node with excess
      tmpNodeE = tmp->heightAvailNodes;
      while(tmpNodeE->nextAvailNode != NULL){
        tmpNodeE = tmpNodeE->nextAvailNode;
      }
      // add to height n
      tmpNodeE->nextAvailNode = G->excess->heightAvailNodes;
      G->excess->heightAvailNodes = tmp->heightAvailNodes;
      trashH = tmp;
      tmp = tmp->nextAvailHeight;
      delete trashH;
    }
    //debug = true;
  }
  // freeze nodes at height n

  if (debug)
    printf("---> %d \n",G->excess->height);
  if (G->excess->height >= G->n){
    //printf("\n\ndoing this thing\n");
    G->excess = G->excess->nextAvailHeight;
    if (G->excess != NULL)
      while(G->height->height != G->excess->height)
        G->height = G->height->below;
  }
}

// gap relabel 
void DoGapRelabelEnd(Graph_t G) {
  /*
	//check if we have the gap relabeling situation at the end
	
	//heightN always points to the HWNqueue at height n.

	int lastHeight = heightN->height;
	int gapAt;
        
	tmpHeight = heightN;
	while (lastHeight > 0) {
		tmpHeight = tmpHeight->prevHWN;

		if (tmpHeight->height != lastHeight - 1) {
            //if this is the case - do a gap relabeling
			
			//gap at this height
			//relabel every node with height < n and > h 
			gapAt = lastHeight - 1;

			//
			// Perform same operations on HeightWithNodes structure
			//
			HWN_t startHWN, tmpHWN, HWNtoDelete;
			startHWN = heightN;
			tmpHWN = heightN->prevHWN;

			// StartHWN points to the HWN at height N
			startHWN = startHWN->nextHWN;
			
			AvailNode_t tmpNode;

			// Increment number of nodes at height N, delete each intermediate HWN
			while (tmpHWN->height > gapAt) {
				startHWN->numNodes += tmpHWN->numNodes;
				
				// Insert nodes from tmpHWN to height N (startHWN)
				tmpNode = tmpHWN->nodes;
				while (tmpNode->nextAvailNode != NULL) {
					NodeArray[tmpNode->nodeNum].height = n;
					tmpNode = tmpNode->nextAvailNode;
				}

				NodeArray[tmpNode->nodeNum].height = n;
				tmpNode->nextAvailNode = startHWN->nodes;
				startHWN->nodes->prevAvailNode = tmpNode;
				startHWN->nodes = tmpHWN->nodes;
				startHWN->nodes->prevAvailNode = NULL;

				HWNtoDelete = tmpHWN;
				tmpHWN = tmpHWN->prevHWN;
				delete HWNtoDelete;                 
			}

			// HWN at height N should point to HWN below gapAt
			startHWN->prevHWN = tmpHWN;

			// HWN below gapAt should point to HWN at height N
			tmpHWN->nextHWN = startHWN;

			heightsWithNodes = startHWN;

			break;
		}
		
		lastHeight = tmpHeight->height;
	}
  */
}


/* pushes flow from node v along edge e if possible otherwise
   it does nothing
*/
void Push(Graph_t G, Node_t v, Edge_t e) {
  numtype min;
  Node_t w;
  Node_t NodeArray = G->nodes;
  
  min = (v->excess > (e->capacity - e->flow)) ? e->capacity - e->flow : v->excess;
  w = &NodeArray[e->toNode];
  if (pushdebug){
    //printf("  push: v -> %d (%d) ",e->toNode,min);
    printf("  push: v -> %d (#) ",e->toNode);
  }
  if ( (min > 0) && (v->height - w->height == 1) ) {

    if (pushdebug)
      printf(" occured\n");

    e->flow += min;
    v->excess -=min;
    w->excess += min;
    e->oppositeEdge->flow -= min;
    // if the node had no excess and it is not
    // the source or the sink add it to the excess
    // data structure
    if ( (w->excess == min) && (e->toNode != G->sourceQueue) && (e->toNode != G->sink) ) {
      AvailNode_t newNode;
      AvailHeight_t excess;

      newNode = new AvailNode;
      newNode->nodeNum = e->toNode;
      excess = G->excess;
      // no availheight structure at height of w then make it
      if ((excess->nextAvailHeight == NULL) || (excess->nextAvailHeight->height != (v->height - 1))) {
        AvailHeight_t newHeight;
        newHeight = new AvailHeight;
        newHeight->height = w->height;
        newHeight->heightAvailNodes = newNode;
        newHeight->nextAvailHeight = excess->nextAvailHeight;
        excess->nextAvailHeight = newHeight;
        newNode->nextAvailNode = NULL;
      }
      else {
        newNode->nextAvailNode = excess->nextAvailHeight->heightAvailNodes;
        excess->nextAvailHeight->heightAvailNodes = newNode;
      }
    }
  }
  else {
    if (pushdebug)
      printf(" \n");
  }
}


/*        DEBUGGING FUNCTIONS       */


/*
  initGraph - Given a graph initializes all the datastructures necessary to
              run the preflow-push algorithm.

  Input:  Graph_t G - Graph to initialize datastructures for

  Output: void
 */
void printExcess(Graph_t G, int max){
  AvailHeight_t h;
  AvailNode_t node;
  int k;
  h = G->excess;
  printf("   ** Excess Data Structure **\n");
  while (h != NULL){
    printf("Height %d: ",h->height);
    node = h->heightAvailNodes;
    k = 0;
    while((node != NULL) && (k <= max)){
      printf("%d ",node->nodeNum);
      node = node->nextAvailNode;
      k++;
    }
    h = h->nextAvailHeight;
    printf("\n");
  }
}

/*
  initGraph - Given a graph initializes all the datastructures necessary to
              run the preflow-push algorithm.

  Input:  Graph_t G - Graph to initialize datastructures for

  Output: void
 */
void printHeight(Graph_t G, int max){
  Height_t h;
  int node, k;
  Node_t NodeArray = G->nodes;
  h = G->height;
  printf("   ** Heights Data Structure **\n");
  while (h->above != NULL)
    h = h->above;
  
  while (h != NULL){
    printf("Height %d: ",h->height);
    node = h->nodes;
    if (h->nodes == -1)
      printf("no nodes ");
    k = 0;
    while((node != -1) && (k <= max)){
      printf("%d ",node);
      node = NodeArray[node].next;
      k++;
    }
    printf("\n");
    h = h->below;
  }
}
