/*******************************************************************************
 *  cluster-single -- Clusters a given graph for using the minimum cut method  *
 *                    graph and varies a parameter "alpha" to obtain a         *
 *                    hierarchical clustering as described by [Flake02].       *
 *                                                                             *
 *  Author: Ramiro Rodriguez       Dec 2003                                    *
 *          Jonathan Pereses       Dec 2003                                    *
 *          Ariful Alam Gani       Dec 2003                                    *
 *                                                                             *
 *  Usage: cluster <edges file> <ids file> <alpha> [output file]               *
 *         If no output file is specified we output to standard outoup         *
 *                                                                             *
 *  Input: Edges file - the first line contains the number of nodes in the     *
 *                      graph and the number of edges in the graph. After that *
 *                      each line represents an edge where the first number is *
 *                      the from node and the second number is the to node.    *
 *                      The node numbers are expected as consecutive integers  *
 *                      from 0 to the number of nodes minus one.               *
 *                                                                             *
 *                      EXAMPLE:                                               *
 *                                                                             *
 *                      3 4                                                    *
 *                      0 1                                                    *
 *                      2 3                                                    *
 *                      3 1                                                    *
 *                      1 0                                                    *
 *         Ids file - The first line is the number of nodes in the graph that  *
 *                    actually correspond to something we wish to cluster.     *
 *                    Each consecutive line is a pair of numbers, the first    *
 *                    number is the id in the graph (between 0 and n-1). The   *
 *                    second number is the true id of the node.                *
 *                                                                             *
 *  Output: The output is given as a list of communities. after # we list the  *
 *          alpha value at which these communities occur. After the alpha we   *
 *          list all the merges that occured for that alpha.                   *
 ******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "mincut.h"
#include "iolib.h"
#include "cluster.h"

//#define DEBUG

// something to think about
bool clusdebug = false;
int clustercalls = 0;
bool clusterVerbose = true;
bool compressdebug = false;
extern bool debug;

int checkOtherNode(Graph_t G){
  int i, other;
  Node_t NodeArray;

  NodeArray = G->nodes;
  other = 0;
  for (i = 0; i < G->n-1; i++){
    if (!NodeArray[i].toCluster){
      fprintf(stderr,"%d -->\n",i);
      other++;
    }
  }

  if (other > 1){
    fprintf(stderr,"there is some problem here solve it %d %d\n",other,G->n);
    exit(0);
  }

  return 0;
}

int checkNumTrue(Graph_t G){
  int i, n;
  iid_t idItr;
  Node_t NodeArray = G->nodes;
  for (i = 0; i < G->n-1; i++){
    // check if the number of true is greater than zero
    // or the ids pointer points to something
    if ((NodeArray[i].ids != NULL) || (NodeArray[i].numtrue > 0)){
      n = 0;
      // count the number of things
      idItr = NodeArray[i].ids;
      while (idItr != NULL){
        n ++;
        idItr = idItr->nextId;
      }
      if (n != NodeArray[i].numtrue){
        printf("THE NODE IS %d\n",i);
        printf("WE DONT HAVE A GRAPH THAT HAS THAT STUFF %d %d numnodes %d\n",NodeArray[i].numtrue,n,G->n);
        exit(0);
      }
    }
  }
  return 0;
}

void printComm(Graph_t G, int node, int max){
  int itr, k;
  Node_t NodeArray = G->nodes;
  printf("--- Community %d [%d] ---\n",node,NodeArray[node].commsize);
  itr = node;
  k = 0;
  while ( (itr != -1) && (k <= max)){
    printf("%d, ",itr);
    k++;
    itr = NodeArray[itr].nextComm;
  }
  printf("\n--- END Community --- \n");
}

void printMinCut(Graph_t G, int first, int max){
  int next, k;
  Node_t NodeArray = G->nodes;

  printf("  --- MINCUT %d ---\n",first);
  next = first;
  k = 0;
  while ( (next != -1) && (k <= max)){
    printf("%d, ",next);
    next = NodeArray[next].next;
    k++;
  }
  printf("\n--- DONE --\n");
}

void writeAll(Graph_t G, FILE *file, numtype alpha){
  int i;
  iid_t idItr;
  Node_t NodeArray = G->nodes;

  fprintf(file,"# alphaMid %ld\n",alpha);
  for (i = 0; i < G->n-1; i++){
    if (NodeArray[i].ids != NULL){
      fprintf(file,"%d ",NodeArray[i].numtrue);
      idItr = NodeArray[i].ids;
      while (idItr != NULL){
        fprintf(file,"%d ",idItr->item);
        idItr = idItr->nextId;
      }
      fprintf(file,"\n");
    }
  }

  fprintf(stderr,"ending up\n");
}

/* writes to the given file object a line for each community in the
   given graph.
 */
//void writeClusters(Graph_t G, FILE *file, numtype alpha){
void writeClusters(Graph_t G, FILE *file, numtype alpha,int truetotal){
  Node_t NodeArray;
  int i, itr, numnodes, n, total;
  iid_t idItr;
  bool started = false;

  NodeArray = G->nodes;
  // for each community give it a name (<main node, num nodes>)
  // print a line containing all the nodes in that community
  total = 0;
  for (i = 0; i < G->n-1; i++){
    if ( (!NodeArray[i].deleted) && (NodeArray[i].commsize > 1)){
    //if (!NodeArray[i].deleted){
      numnodes = 0;
      itr = i;
      while(itr != -1){
        numnodes += NodeArray[itr].numtrue;
        itr = NodeArray[itr].nextComm;
      }
      total += numnodes;
      if (numnodes == 0)
        continue;

      // print at the beggining the alpha value
      if (!started){
        fprintf(file,"# alphaMid %d %d\n",alpha);
        started = true;
      }
      n = 0;
      fprintf(file,"%d ",numnodes);
      for (itr = i; itr != -1; itr = NodeArray[itr].nextComm){
        idItr = NodeArray[itr].ids;
        while(idItr != NULL){
          n++;
          fprintf(file,"%d ",idItr->item);
          idItr = idItr->nextId;
        }
      }
      if (n != numnodes){
        fprintf(stderr,"WE HAVE A BIG PRBLEM NUMTRUE IS %d NUMBER SAYS %d alpha: %d %d\n",n,numnodes,alpha);
        exit(0);
      }
      fprintf(file,"\n");
    }
  }
  //printf("Total True: %d\n",total);
  //if (total != truetotal){
  //fprintf(stderr,"we lost some communities somewhere [total: %d, truetotal %d]\n",total,truetotal);
  //exit(0);
  //}

  return;
  /*

  int i, itr, truec, truen, fakec, faken;
  bool hastruec;
  FILE *file;
  char filename[128];
  char truename[128];
  char fakename[128];
  int above;
  int maxcommsize = 0;
  int mincommsize = 70;

  NodeArray = G->nodes;
  printf("writing changes...\n");
  return;
  truec = truen = 0;
  fakec = faken = 0;
  above = 0;
  for (i = 0; i < G->n-1; i++){
    // if the node is a main node
    // then write all the nodes in
    // it to a file where the name is
    // the name of the node
    if (!NodeArray[i].deleted){
      if (clusterVerbose)
        printf("----- COMMUNITY %d [%d] -----\n",i,NodeArray[i].commsize);
      // go thru list and write everything that is not -1
      itr = i;
      hastruec = false;
      int job = 0;
      int truesize = 0;
      while (itr != -1){
        if (NodeArray[itr].id != -1){
          if (!hastruec){
            hastruec = true;
            truec++;
          }
          sprintf(truename, "%d", NodeArray[itr].id);
          sprintf(fakename, "%d", itr);
          if (clusterVerbose)
            printf("%s %s\n",fakename,truename);
          truen++;
          truesize++;
        }
        itr = NodeArray[itr].nextComm;
        faken++;
        job++;
      }
      //if (hastruec)
        //fclose(file);
      fakec++;
      if (clusterVerbose)
        printf("--- END COMMUNITY [%d][%d] ----\n\n",job,truesize);
      if (truesize >= mincommsize)
        above++;
      if (truesize >= maxcommsize)
        maxcommsize = truesize;
    }
  }
  if (clusterVerbose){
    printf("TRUE-> communities: %d\titems: %d\n",truec,truen);
    printf("TRUE-> communities: %d\titems: %d\n",fakec,faken);
    printf("number above %d %d\n",mincommsize,above);
    printf("largest community is size %d\n",maxcommsize);
  }
  */
}

Graph_t nextGraph(Graph_t G, int *numComms){
  Graph_t newGraph;
  Node_t NodeArray, node;
  int i, k, itr;
  Edge_t bedge, fedge, e, eItr;
  int *map;

  //printf("Next CLUSTER: %d\n",G->nextCluster);
  //printf("SOMEWHERE HERE THERE IS AN ERROR\n");
  if (G->nextCluster == -1)
    return NULL;

  *numComms = G->nodes[G->nextCluster].commsize;
  NodeArray = G->nodes;
  map = new int[G->n];
  //printf("G->n = %d\n",G->n);
  for(i = 0; i < G->n-1; i++){
    map[i] = NodeArray[G->nextCluster].commsize;
    //printf("map[%d] = %d\n",map[i],NodeArray[G->nextCluster].commsize);
  }
  map[G->n-1] = NodeArray[G->nextCluster].commsize+1;

  // allocate the new graph
  newGraph = new Graph;

  // initialize all the parameters
  // the size of the graph is the size of the community
  // plus the sink and a node that represents all the
  // other nodes in the graph
  newGraph->n = NodeArray[G->nextCluster].commsize + 2;
  newGraph->nodes = new Node[newGraph->n];
  newGraph->sink = newGraph->n-1;


  // initialize nodes
  // each node is initialized with
  // the original id in the graph G
  // we contract into one node all the nodes
  // not in the community and we place it at
  // the end of the node array in the graph
  // before the sink node
  k = 0;
  itr = G->nextCluster;
  while(itr != -1){
    // initialize node
    node = &(newGraph->nodes[k]);
    /* Copy IDS list from larger graph */
    node->ids = NodeArray[itr].ids;
    node->lastid = NodeArray[itr].lastid;
    node->numtrue = NodeArray[itr].numtrue;
    /* End Copy IDS List */
    node->numnodes = NodeArray[itr].numnodes;
    node->toCluster = NodeArray[itr].toCluster;


    map[itr] = k;
    k++;
    itr = NodeArray[itr].nextComm;
  }
  newGraph->nodes[newGraph->n-2].ids = NULL;
  newGraph->nodes[newGraph->n-2].lastid = NULL;
  newGraph->nodes[newGraph->n-2].numtrue = 0;
  int temp = 0;
  for(i = 0; i < G->n-1; i++)
    temp += G->nodes[i].numnodes;
  newGraph->nodes[newGraph->n-2].numnodes = temp - NodeArray[G->nextCluster].commsize;
  if (newGraph->nodes[newGraph->n-2].numnodes <= 0){
    fprintf(stderr,"WE HAVE SOME STUPID SHIT GOING ON\n");
    fprintf(stderr,"temp %d commsize %d\n",temp,NodeArray[G->nextCluster].commsize);
    fprintf(stderr,"G->nextCluster %d , G->n %d\n",G->nextCluster,G->n);
  }
  newGraph->nodes[newGraph->n-2].toCluster = false;

  //printf("&&&&&&&&&&&\n");
  //printf("newGraph->nodes[newGraph->n-2].numnodes = %d\n",newGraph->nodes[newGraph->n-2].numnodes);
  //printf("&&&&&&&&&&&\n");

  // make edges from all the nodes to the
  // sink
  newGraph->nodes[newGraph->n-1].firstEdge = NULL;
  for (i = 0; i < newGraph->n-1;i++){
    fedge = new Edge;
    bedge = new Edge;

    // add fowards edge
    fedge->capacity = newGraph->nodes[i].numnodes;
    fedge->toNode = newGraph->n-1;
    fedge->oppositeEdge = bedge;
    fedge->nextEdge = NULL;
    fedge->prevEdge = NULL;
    newGraph->nodes[i].firstEdge = fedge;

    // add backwards edge
    bedge->capacity = newGraph->nodes[i].numnodes;
    bedge->toNode = i;
    bedge->oppositeEdge = fedge;
    bedge->nextEdge = newGraph->nodes[newGraph->n-1].firstEdge;
    bedge->prevEdge = NULL;;
    if (newGraph->nodes[newGraph->n-1].firstEdge != NULL)
      newGraph->nodes[newGraph->n-1].firstEdge->prevEdge = bedge;
    newGraph->nodes[newGraph->n-1].firstEdge = bedge;
  }

  //printComm(G,G->nextCluster,10);
  //printf("nextComm size %d\n",NodeArray[G->nextCluster].commsize);
  //printf("-------- MAPPING TABLE -------\n");
  //printf("| true value  | new graph value |\n");
  //printf("________________________________\n");
  //for (int j = 0; j < G->n-1;j++)
  //  printf("|\t%d\t|\t%d\t|\n",j,map[j]);
  //printf("-----------------------\n");


  // copy edges from large graph to this small graph
  int otherNode = NodeArray[G->nextCluster].commsize;
  itr = G->nextCluster;
  while(itr != -1){
    e = NodeArray[itr].firstEdge;
    while(e != NULL){

      //printf("checking if edge [%d,%d] is added yet\n",itr,e->toNode);
      // check if edge has already been added
      eItr = newGraph->nodes[map[itr]].firstEdge;
      while(( eItr != NULL) && (eItr->toNode != map[e->toNode]))
        eItr = eItr->nextEdge;


      // edge has not been added so we add the forwards and
      // backwards edges
      if (eItr == NULL) {
        fedge = new Edge;
        bedge = new Edge;

        // add forward edge
        fedge->capacity = e->capacity;
        fedge->toNode = map[e->toNode];
        fedge->oppositeEdge = bedge;
        fedge->nextEdge = newGraph->nodes[map[itr]].firstEdge;
        fedge->prevEdge = NULL;
        newGraph->nodes[map[itr]].firstEdge->prevEdge = fedge;
        newGraph->nodes[map[itr]].firstEdge = fedge;

        // add backwards edge
        bedge->capacity = e->capacity;
        bedge->toNode = map[itr];
        bedge->oppositeEdge = fedge;
        bedge->nextEdge = newGraph->nodes[map[e->toNode]].firstEdge;
        bedge->prevEdge = NULL;
        newGraph->nodes[map[e->toNode]].firstEdge->prevEdge = bedge;
        newGraph->nodes[map[e->toNode]].firstEdge = bedge;
      }
      // else if it has been added and it is to the node reperesenting
      // all other nodes not in the cluster ten we add increase the
      // capacity of the edge
      else if (map[e->toNode] == otherNode){
        eItr->capacity += e->capacity;
        eItr->oppositeEdge->capacity += e->capacity;
      }

      e = e->nextEdge;
    }
    itr = NodeArray[itr].nextComm;
  }


  // set the next cluster
  for (i = G->nextCluster + 1; i < G->n-1; i++)
    if ( (!NodeArray[i].deleted) && (NodeArray[i].toCluster) && (NodeArray[i].commsize > 1)){
      //printf("encountered %d %d\n",i,NodeArray[i].commsize);
      G->nextCluster = i;
      break;
    }

  if (i == G->n-1)
    G->nextCluster = -1;

  //PrintGraph(newGraph);
  //printf("AFTER NEXT CLUSTER %d size %d new size %d commsize of sink-1 %d\n",G->nextCluster,G->n,newGraph->n,newGraph->nodes[newGraph->n-2].commsize);

  delete map;

  //if (clustercalls == 4){
  //  printf("WE GOT WORK TODO\n");
  //  PrintGraph(newGraph);
  //  printf("what the hell\n");
  //  exit(0);
  //}

  return newGraph;
}

/* compress all the nodes in each cluster into one node
 */
Graph_t compressGraph(Graph_t G){
  int i, numclusters;
  Node_t NodeArray;
  Graph_t newGraph;
  Edge_t fedge, bedge, e, eItr, eToDel;
  int *map;

  //printf("entering compress graph\n");
  NodeArray = G->nodes;
  // 1. find the number of clusters
  numclusters = 0;
  for(i = 0; i < G->n-1;i++)
    if (!NodeArray[i].deleted)
      numclusters++;
  if (numclusters == G->n-1)
    return G;
  //fprintf(stderr,"obtained number of clusters\n");

  // 2. initialize the graph
  newGraph = new Graph;
  newGraph->n = numclusters+1;
  newGraph->nodes = new Node[newGraph->n];
  newGraph->sourceQueue = -1;
  newGraph->sink = newGraph->n-1;
  newGraph->excess = NULL;
  newGraph->height = NULL;
  newGraph->heightn = NULL;
  newGraph->nextCluster = -1;
  //fprintf(stderr,"initialized the graph\n");



  // 3. create the mapping from nodes in one graph to nodes in
  //    the other graph
  int k = 0, itr = 0;
  int numcomms = 0;
  int numnodes;
  map = new int[G->n];
  for (i = 0; i < G->n-1; i++){
    if ( (!NodeArray[i].deleted) && (NodeArray[i].toCluster)){
      numcomms++;
      itr = i;
      numnodes = 0;
      /* Add the first list of IDS to the new node */
      //newGraph->nodes[k].ids = NULL;
      newGraph->nodes[k].ids = NodeArray[itr].ids;
      newGraph->nodes[k].lastid = NULL;
      newGraph->nodes[k].numtrue = 0;
      /* END */
      //if (k == 3054)
      //compressdebug = true;
      //else
      //compressdebug = false;
      //if (compressdebug){
      //fprintf(stderr,"error loading k = %d\n",k);
      //}
      while(itr != -1){
        map[itr] = k;
        numnodes += NodeArray[itr].numnodes;
        /* Add the list of IDS to the new node if it has true IDS*/
        if (NodeArray[itr].numtrue > 0){
          // set the number of nodes new node represents
          //if (compressdebug){
          //fprintf(stderr,"itr: %d numtrue: %d %d\n",itr,NodeArray[itr].numtrue,k);
          //fprintf(stderr,"itr.ids %d itr.lastid %d\n",NodeArray[itr].ids,NodeArray[itr].lastid);
          //}
          if (newGraph->nodes[k].ids == NULL){
            newGraph->nodes[k].ids = NodeArray[itr].ids;
            //printf("what a shame\n");
            //exit(0);
          }
          if (newGraph->nodes[k].lastid != NULL)
            newGraph->nodes[k].lastid->nextId = NodeArray[itr].ids;
          newGraph->nodes[k].lastid = NodeArray[itr].lastid;
          newGraph->nodes[k].numtrue += NodeArray[itr].numtrue;
        }
        /* End of adding IDS to new node */
        itr = NodeArray[itr].nextComm;
      }
      if (newGraph->nodes[k].lastid != NULL)
        newGraph->nodes[k].lastid->nextId = NULL;
      newGraph->nodes[k].numnodes = numnodes;
      newGraph->nodes[k].toCluster = true;
      k++;
    }
    // node is a group node
    else if (!NodeArray[i].toCluster){
      if (NodeArray[i].commsize != 1){
        fprintf(stderr,"what the hell happened the group node is not one\n");
        exit(0);
      }
      numcomms++;
      map[i] = k;
      newGraph->nodes[k].numnodes = NodeArray[i].numnodes;
      newGraph->nodes[k].toCluster = false;
      newGraph->nodes[k].ids = NULL;
      newGraph->nodes[k].lastid = NULL;
      newGraph->nodes[k].numtrue = 0;
      k++;
    }
  }
  map[G->n-1] = numcomms;
  //fprintf(stderr,"created the mapping\n");

  // 4. add edges from the sink to everynode
  newGraph->nodes[newGraph->n-1].firstEdge = NULL;
  for (i = 0; i < newGraph->n-1;i++){
    fedge = new Edge;
    bedge = new Edge;

    // add fowards edge
    fedge->capacity = newGraph->nodes[i].numnodes;
    fedge->toNode = newGraph->n-1;
    fedge->oppositeEdge = bedge;
    fedge->nextEdge = NULL;
    fedge->prevEdge = NULL;
    newGraph->nodes[i].firstEdge = fedge;

    // add backwards edge
    bedge->capacity = newGraph->nodes[i].numnodes;
    bedge->toNode = i;
    bedge->oppositeEdge = fedge;
    bedge->nextEdge = newGraph->nodes[newGraph->n-1].firstEdge;
    bedge->prevEdge = NULL;;
    if (newGraph->nodes[newGraph->n-1].firstEdge != NULL)
      newGraph->nodes[newGraph->n-1].firstEdge->prevEdge = bedge;
    newGraph->nodes[newGraph->n-1].firstEdge = bedge;
  }
  //fprintf(stderr,"added edges from sink to everyone else\n");

  // 5. add the edges to the graph
  for(i = 0; i < G->n-1; i++){
    //printf("Adding Edges from node %d\n",i);
    e = NodeArray[i].firstEdge;
    while(e != NULL){
      // if the edge is between two different clusters
      // then add it else don't add it
      if (map[i] != map[e->toNode]){
        // check if the edge alread exists
        //printf("checking if edge [%d,%d] is added yet\n",itr,e->toNode);
        eItr = newGraph->nodes[map[i]].firstEdge;
        while(( eItr != NULL) && (eItr->toNode != map[e->toNode]))
          eItr = eItr->nextEdge;

        // if edge has not been added eItr will be NULL
        // if it has not been added we add the edge
        // with capacity 0
        if (eItr == NULL){
          // allocate edges
          fedge = new Edge;
          bedge = new Edge;

          // add forward edge
          fedge->capacity = 0;
          fedge->toNode = map[e->toNode];
          fedge->oppositeEdge = bedge;
          fedge->nextEdge = newGraph->nodes[map[i]].firstEdge;
          fedge->prevEdge = NULL;
          newGraph->nodes[map[i]].firstEdge->prevEdge = fedge;
          newGraph->nodes[map[i]].firstEdge = fedge;

          // add backwards edge
          bedge->capacity = 0;
          bedge->toNode = map[i];
          bedge->oppositeEdge = fedge;
          bedge->nextEdge = newGraph->nodes[map[e->toNode]].firstEdge;
          bedge->prevEdge = NULL;
          newGraph->nodes[map[e->toNode]].firstEdge->prevEdge = bedge;
          newGraph->nodes[map[e->toNode]].firstEdge = bedge;

          // set the edge finder to the new edge
          eItr = fedge;
        }

        // the edge must exist now so we simply add the capacity
        eItr->capacity += e->capacity;
        eItr->oppositeEdge->capacity += e->capacity;
      }

      // delete the edge so that we will not
      // increase the capacity when we come accross the opposite edge
      eToDel = e;
      e = e->nextEdge;

      // take eToDel from edge list
      if (eToDel->nextEdge != NULL)
        eToDel->nextEdge->prevEdge = eToDel->prevEdge;
      if (eToDel->prevEdge != NULL)
        eToDel->prevEdge->nextEdge = eToDel->nextEdge;
      else
        NodeArray[i].firstEdge = eToDel->nextEdge;

      // take out eToDel->oppositeEdge from edge list
      if (eToDel->oppositeEdge->nextEdge != NULL)
        eToDel->oppositeEdge->nextEdge->prevEdge = eToDel->oppositeEdge->prevEdge;
      if (eToDel->oppositeEdge->prevEdge != NULL)
        eToDel->oppositeEdge->prevEdge->nextEdge = eToDel->oppositeEdge->nextEdge;
      else
        NodeArray[eToDel->toNode].firstEdge = eToDel->oppositeEdge->nextEdge;

      delete eToDel->oppositeEdge;
      delete eToDel;
    }
  }
  //fprintf(stderr,"added edges from the bigger graph to the smaller graph\n");

  // free the graph
  // free the nodes array
  delete G->nodes;
  delete G;

  return newGraph;
}

/* free all the memory associated with the
   given graph
*/
int freeGraph(Graph_t G){
  int i;
  Node_t NodeArray;
  Edge_t e, dedge;


  NodeArray = G->nodes;
  // 1. free all the edges
  for (i = 0; i < G->n; i++){
    e = NodeArray[i].firstEdge;
    while(e != NULL){
      dedge = e;
      e = e->nextEdge;
      delete dedge;
    }
  }

  // 2. free the node array
  delete NodeArray;

  // 3. delete the graph
  delete G;

  return 0;
}







int checkNumComms(Graph_t G, int node){
  Node_t NodeArray;
  int elements, itr;

  NodeArray = G->nodes;
  itr = node;
  elements = 0;
  while (itr != -1){
    elements++;
    itr = NodeArray[itr].nextComm;
  }
  if (elements == NodeArray[node].commsize)
    return 1;
  else {
    printf("community %d thinks %d actually %d\n",node,NodeArray[node].commsize,elements);
    printComm(G,node,20);
    return 0;
  }
}

#ifdef SINGLECLUSTER
int main(int args, char* argv[]){
  FILE *file, *logfile;
  Graph_t G;
  int n, communities;
  numtype alpha;
  numtype alphaD;
  char *edges;
  char *ids;
  int truetotal;

  if (args != 4){
    printf("usage: cluster <edges file> <ids file> <alpha value>\n");
    exit(0);
  }
  edges = argv[1];
  ids = argv[2];
  alpha = atoi(argv[3]);

  file = fopen(edges,"r");
  fscanf(file,"%d",&n);
  fclose(file);

  file = fopen(ids,"r");
  fscanf(file,"%d",&truetotal);
  fclose(file);

  printf("\nclustering for alpha %d %d (n = %d) \n",alpha,n);
  printf("true total: %d\n",truetotal);
  alphaD = 2*n*n;
  printf("alpha Denominator = %d %d\n",alphaD);

  // import the graph into memory
  G = parseInputFile(edges,ids,2*n*n,2*n*n);
  G->sink = G->n -1;

  communities = cluster(G,alpha);

  logfile = fopen("cluster.log","w");
  printf("Number of communities: %d\n",communities);
  writeClusters(G,logfile,alpha,truetotal);
  fclose(logfile);

  //G = compressGraph(G);
  //PrintGraph(G);
  /*
  printf("NEXT CLUSTER = %d\n",G->nextCluster);
  Graph_t smallG;
  while ((smallG = nextGraph(G)) != NULL){
    PrintGraph(smallG);
    printf("NEXT CLUSTER = %d\n",G->nextCluster);
  }

  for(int i = 0; i < G->n-1; i++){
    if (!G->nodes[i].deleted){
      if (checkNumComms(G,i) != 1){
        printf("still have bug\n");
        exit(0);
      }
    }
  }
  */
}
#endif



/* Cluster the graph for one value of alpha and return
   the number of clusters found. The graph is modified
   so that we can create smaller graphs from it
*/
int cluster(Graph_t G, numtype alpha){
  int i, comm, nextComm, communities;
  int prevNextComm, source, subCommItr, itr;
  Edge_t e;
  Node_t NodeArray;

#ifdef DEBUG
  int numIterations;
#endif

  NodeArray = G->nodes;

  // Initialize the correct alpha value for the
  // alpha edges
  e = G->nodes[G->sink].firstEdge;
  while(e != NULL){
    e->capacity = NodeArray[e->toNode].numnodes * alpha;
    e->oppositeEdge->capacity = NodeArray[e->toNode].numnodes * alpha;
    e = e->nextEdge;
  }

  // set up the source queue start
  for (i = 0; i < G->n-1; i++) {
    G->nodes[i].deleted = false;
    G->nodes[i].nextComm = -1;
    G->nodes[i].parent = i;
    G->nodes[i].commsize = 1;

    if (!G->nodes[i].toCluster){
      G->nodes[i-1].nextNode = -1;
      continue;
    }
    G->nodes[i].nextNode = i+1;
    G->nodes[i].prevNode = i-1;
  }
  G->sourceQueue = 0;
  G->nodes[G->n-2].nextNode = -1;

  //fprintf(stderr,"()()()()()()()()()()()()()()()()()()()()()()()()()()()()()(\n");
  //PrintGraph(G);
  //fprintf(stderr,"()()()()()()()()()()()()()()()()()()()()()()()()()()()()()(\n");
#ifdef DEBUG
  if (clusdebug)
    fprintf(stderr,"initialization complete\n");
  numIterations = 0;
#endif
  //if (clustercalls == 12){
    //PrintGraph(G);
    //fprintf(stderr,"TOO MANY CALLS\n");
    //exit(0);
  //}
  clustercalls++;
  // Main mincut loop - here we pick a node that we have not picked
  // before and that is not already in a community as the source
  // 1. find its mincut
  // 2. add all the nodes in the mincut to this community
  while (G->sourceQueue != -1 ){

#ifdef DEBUG
    numIterations++;
    if (clusdebug)
      fprintf(stderr,"MinCut from node %d starting (sink %d).\n",G->sourceQueue,G->sink);
#endif

    /* 1. Find its MinCut
     */
    comm = MinCut(G);


#ifdef DEBUG
    if (clusdebug){
      fprintf(stderr,"MinCut from node %d complete.\n",G->sourceQueue);
      //PrintGraph(G);
      printMinCut(G,comm,30);
    }
    if (clusdebug)
      fprintf(stderr,"Merging starting\n");
#endif
    //if (clustercalls == 5)
    //exit(0);
    /* 2. add all the nodes in the mincut to this community
       we have to check for subcommunities that we have already placed in a cluster
       for all those nodes in the subcommunity we have to place in the bigger
       cluster now
     */


    // make sure we start with something other than the source
    if (G->sourceQueue == comm) {
      nextComm = G->nodes[comm].next;
    }
    else{
      nextComm = comm;
    }
    // set the source to point to the first element in the mincut
    NodeArray[G->sourceQueue].nextComm = nextComm;
    prevNextComm = G->sourceQueue;
    source = G->sourceQueue;

#ifdef DEBUG
    if (clusdebug)
    fprintf(stderr,"BEGGINNING OF MERGE: nextComm (%d) source (%d) preveNextComm (%d) next (%d) \n",
           nextComm,source,prevNextComm,NodeArray[nextComm].next);
    int itrs = 0;
#endif

    // if the mincut only contains the source
    // we update G->sourceQueue
    if (nextComm == -1){
      if (NodeArray[source].nextNode != -1)
        NodeArray[NodeArray[source].nextNode].prevNode = -1;
      G->sourceQueue = NodeArray[source].nextNode;
    }


    // this loop looks at everything in the mincut to place into the
    // community of the source
    while (nextComm != -1){

#ifdef DEBUG
      if (clusdebug){
        fprintf(stderr,"hook %d %d sink: %d\n",nextComm,G->n,G->sink);
        fprintf(stderr,"nextComm %d, parent: %d, nextMyComm %d ",nextComm,NodeArray[nextComm].parent,NodeArray[nextComm].nextComm);
        if (G->n > 353){

        }
      }
      itrs++;
#endif
      
      //fprintf(stderr,"Removing from source queue start %d\n",NodeArray[nextComm].toCluster);
      // Remove nextComm from source queue
      // if it hasn't been deleted already
      if (NodeArray[nextComm].nextNode != -1)
        NodeArray[NodeArray[nextComm].nextNode].prevNode = NodeArray[nextComm].prevNode;
      //fprintf(stderr,"Removing from source queue start 1 %d\n",NodeArray[nextComm].prevNode);
      if (NodeArray[nextComm].prevNode != -1)
        NodeArray[NodeArray[nextComm].prevNode].nextNode = NodeArray[nextComm].nextNode;
      //fprintf(stderr,"Removing from source queue start 2\n");
      if (G->sourceQueue == nextComm)
        G->sourceQueue = NodeArray[nextComm].nextNode;
      //fprintf(stderr,"Removing from source queue start 3\n");
      NodeArray[nextComm].nextNode = -1;
      //fprintf(stderr,"Removing from source queue start 4\n");
      NodeArray[nextComm].prevNode = -1;
      //fprintf(stderr,"Removed from source queue start 5\n");


      // nextComm is not the source so we add this to the cluster
      if (nextComm != source){

        // if node's parent is itself and there is nothing
        // in the nodes nextComm queue then it must be a
        // node by itself and we only have to merge this one node
        if ( (NodeArray[nextComm].parent == nextComm) && (NodeArray[nextComm].nextComm == -1) ){

#ifdef DEBUG
          if (clusdebug)
            fprintf(stderr,"lone node\n");
#endif

          // increase the size of the cluster by one
          NodeArray[source].commsize++;
          // put the node inside the cluster of the source
          NodeArray[nextComm].commsize = 0;
          NodeArray[nextComm].parent = source;
          NodeArray[nextComm].deleted = true;
          NodeArray[nextComm].nextComm = NodeArray[nextComm].next;

          // Remove from mincut queue
          if (NodeArray[nextComm].next != -1)
            NodeArray[NodeArray[nextComm].next].prev = NodeArray[nextComm].prev;
          if (NodeArray[nextComm].prev != -1)
            NodeArray[NodeArray[nextComm].prev].next = NodeArray[itr].next;
          else {
            comm = NodeArray[nextComm].next;
            prevNextComm = nextComm;
            nextComm = NodeArray[nextComm].next;
          }


          //prevNextComm = nextComm;
          //nextComm = NodeArray[nextComm].next;
        }
        // if the node's parent is not itself or there are nodes in its
        // nextComm queue then we have a cluster inside our cluster
        // generated by the source, hence we need to add everything in
        // this cluster to the cluster of the source
        else {

#ifdef DEBUG
          if (clusdebug)
            fprintf(stderr,"subcluster node\n");
#endif

          // obtain the begginning of the cluster
          subCommItr = NodeArray[nextComm].parent;

          // fix the previous node in the community
          NodeArray[prevNextComm].nextComm = subCommItr;
          // increase the size of the source cluster
          NodeArray[source].commsize += NodeArray[subCommItr].commsize;
          // delete the node representing the subcluster
          NodeArray[subCommItr].deleted = true;

#ifdef DEBUG
          if (clusdebug)
            fprintf(stderr,"SUBCLUSTER [%d] size: %d\n",subCommItr,NodeArray[subCommItr].commsize);
#endif
          // make the subclusters main node community size of zero
          NodeArray[subCommItr].commsize = 0;

          // Iterate throught all the elements of the sub community to
          // delete them from the mincut queue and to find the last element
          // so that we can make that element point to the next item in the mincut
          // queue
          while (NodeArray[subCommItr].nextComm !=  -1){

            // 1. delete from mincut queue
            itr = comm;
            while(itr != subCommItr)
              itr = NodeArray[itr].next;
            if (NodeArray[itr].next != -1)
              NodeArray[NodeArray[itr].next].prev = NodeArray[itr].prev;
            if (NodeArray[itr].prev != -1)
              NodeArray[NodeArray[itr].prev].next = NodeArray[itr].next;
            else {
              comm = NodeArray[itr].next;
              nextComm = NodeArray[itr].next;
            }

            // 2. change the parent of the node
            NodeArray[subCommItr].parent = source;

            // 3. get next item in subcluster
            subCommItr = NodeArray[subCommItr].nextComm;
          }
          // delete last item in subcommunity from mincut queue
          itr = comm;
          while(itr != subCommItr)
            itr = NodeArray[itr].next;
          if (NodeArray[itr].next != -1)
            NodeArray[NodeArray[itr].next].prev = NodeArray[itr].prev;
          if (NodeArray[itr].prev != -1)
            NodeArray[NodeArray[itr].prev].next = NodeArray[itr].next;
          else {
            comm = NodeArray[itr].next;
            nextComm = NodeArray[itr].next;
          }

          NodeArray[subCommItr].nextComm = nextComm;

          prevNextComm = subCommItr;
        } // end we have a sub community
      }
      // nextComm is the source so we skip the node and change the
      // pointer of the previous item
      else {

#ifdef DEBUG
        if (clusdebug)
          fprintf(stderr,"we found the source\n");
#endif

        NodeArray[prevNextComm].nextComm = NodeArray[nextComm].next;
        if (NodeArray[nextComm].next != -1)
          NodeArray[NodeArray[nextComm].next].prev = -1;
        nextComm = NodeArray[nextComm].next;
      }
#ifdef DEBUG
      if (clusdebug)
        fprintf(stderr,"G->source QUEUE %d\n",G->sourceQueue);
#endif
    } // end of while merge
    //#ifdef DEBUG
    //if (clusdebug)
    //fprintf(stderr,"merging complete\n");
    if (checkNumComms(G,source) != 1){
      fprintf(stderr,"error\n");
      exit(0);
    }
    //#endif

  }
#ifdef DEBUG
  if (clusdebug)
    fprintf(stderr,"we have clustered wow!!\n");
#endif
  int things = 0;
  // count the number of communities
  communities = 0;
  for (i = 0; i < G->n-1; i++){
    if ( (!NodeArray[i].deleted) && (NodeArray[i].toCluster) )
      communities++;
    things += NodeArray[i].commsize;
  }

  // obtain the next cluster parent
  for (i = 0; i < G->n-1; i++)
    if ( (!NodeArray[i].deleted) && (NodeArray[i].commsize > 1)){
      G->nextCluster = i;
      break;
    }
  if (i == G->n-1)
    G->nextCluster = -1;
#ifdef DEBUG
  if (clusdebug)
    fprintf(stderr,"clustered %d things communities %d\n",things,communities);
#endif

  return communities;
}
